;;;; Major mode for displaying and editing SyMP options.

(require 'symp-common)

(defvar symp-options-debug nil
  "The list of function names that can be debugged.  It's value is set
automatically when the SyMP server starts.")

(defun symp-options-debug-table ()
  "Build the table (a-list) of debug symbols from `symp-options-debug'
variable, used internally for completing reads"
  (mapcar '(lambda (el) (cons el nil)) symp-options-debug))

(defvar symp-options-buffer nil
  "The buffer for displaying and editing SyMP options.  This variable is buffer local,
and is normally stored in the SyMP server buffer.")

(defvar symp-options nil
  "The assoc. list of current SyMP options.  You shouldn't modify this
variable other than by `symp-options-update' function.")

(make-variable-buffer-local 'symp-options-buffer)
(make-variable-buffer-local 'symp-options)

(defun symp-options-find-buffer ()
  "Find the options buffer for the current SyMP session, or create one
if not found."
  (let ((old-buffer (current-buffer))
	(status symp-server-status)
	(process symp-server-process)
	(symp-buffer symp-server-process-buffer)
	(buffer nil))
    (if symp-buffer
	;; there is a symp server buffer, go there
	(progn
	  (set-buffer symp-buffer)
	  (if (and symp-options-buffer
		   (bufferp symp-options-buffer)
		   (buffer-live-p symp-options-buffer))
	      ;; the error buffer already exists, just take it
	      (setq buffer symp-options-buffer)
	    ;; no error buffer, create a new one
	    (progn
	      (setq buffer (generate-new-buffer "*SyMP Options*"))
	      ;; register it in the server buffer
	      (setq symp-options-buffer buffer)
	      ;; and set up the error buffer properly
	      (set-buffer buffer)
	      (setq symp-server-process process)
	      (setq symp-server-status status)
	      (setq symp-server-process-buffer symp-buffer)
	      (symp-options-mode)))
	  (set-buffer old-buffer))
      ;; if there is no symp buffer, give up, leave `buffer' nil.
      )
    buffer))

(defun symp-options-get-value (key &optional options)
  "Find the current value of an option KEY.  If not found, return NIL."
  (let ((value (assoc key (if options options (symp-options)))))
    (if value (cdr value) nil)))

(defun symp-options-update-one (options pair)
  "Update the a-list OPTIONS with new (KEY . VALUE).  Add the pair if
it's not present."
  (if (null options) (list pair)
    (let* ((curr-pair (car options))
	   (tail (cdr options)))
      (if (equal (car pair) (car curr-pair)) (cons pair tail)
	(cons curr-pair (symp-options-update-one tail pair))))))

(defun symp-options-update (options)
  "Update `symp-options' in the symp-server buffer with the new values
listed in OPTIONS, and leave other values unchanged."
  ;; First, find the main symp buffer and go there
  (let ((old-buffer (current-buffer))
	(symp-buffer symp-server-process-buffer))
    (if symp-buffer
	(progn
	  (set-buffer symp-buffer)
	  (while options
	    (if (equal "debug" (caar options))
		(symp-server-update-args (cdar options)))
	    (setq symp-options 
		  (symp-options-update-one symp-options (car options)))
	    (setq options (cdr options)))
	  (set-buffer old-buffer)))))

(defun symp-options ()
  "Returns the a-list of currently cached SyMP options."
  (let ((old-buffer (current-buffer))
	(symp-buffer symp-server-process-buffer)
	(options nil))
    (if symp-buffer 
	(progn
	  (set-buffer symp-buffer)
	  (setq options symp-options)
	  (set-buffer old-buffer)))
    options))

(defun symp-options-display (options &optional quiet)
  "Updates the SyMP Options buffer with the OPTIONS given in the
argument (not cached ones), and displays the buffer, unless the second
optional QUIET argument is non-nil.  If there is only one option to
display, and it's value is shorter than 80 characters, it will also be
displayed in the minibuffer, and the Options buffer will not be
brought up if it is not visible.  In any event, the Options buffer
will be created and/or updated."
  (let ((old-buffer (current-buffer))
	(display-string "")
	(options-buffer (symp-options-find-buffer))
	;; Current point position in options-buffer
	(pos 0))
    (if options-buffer
	(progn
	  (set-buffer options-buffer)
	  (setq pos (point))
	  (erase-buffer)
	  (while options
	    (setq display-string (format "%S\n" (car options)))
	    (insert display-string)
	    (setq options (cdr options)))
	  (goto-char pos)
	  (set-buffer old-buffer)
	  (if (and (= 1 (length options))
		   (< (length display-string) 80))
	      (message display-string)
	    (if (not quiet) (display-buffer options-buffer t))))
      (symp-message "Warning: no Options buffer found.  Options cannot be displayed"))))

(defun symp-options-edit (key value)
  "Edit SyMP options and submit the changes to the server.  Currently,
you need to know the option's name, and it will auto-fill the current
value as the default which you can edit."
  (interactive
   (let ((key nil)
	 (value nil)
	 (options (symp-options))
	 (old-value nil))
     (if symp-debug
	 (symp-debug-display (format "\nsymp-options-edit: options = %S\n" options)))
     (setq key (completing-read "Option name to change: " options nil nil))
     (setq old-value (symp-options-get-value key options))
     (if symp-debug
	 (symp-debug-display (format "\nsymp-options-edit: old-value = %S\n" old-value)))
     (setq value (read (read-from-minibuffer "Value: " (format "%S" old-value))))
     (list key value)))
  ;; WARNING! These functions will not be loaded with "require"
  ;; Send the requested update
  (symp-server-send-expr (list 'options (list key '= value)))
  ;; Then request the new options to update our cache
  (symp-server-send-expr '(options)))

(defun symp-options-debug-edit (name &optional remove)
  "Add a new NAME to the `debug' option's list, unless the REMOVE
argument is non-nil, in which case it's, of course, removed."
  (let* ((debug-value (symp-options-get-value "debug")))
    (if (not (stringp name))
	(error "Elements of `debug' option must be strings"))
    (if remove
	(setq debug-value (delete name debug-value))
      (add-to-list 'debug-value name))
    (symp-options-edit "debug" debug-value)))

(defun symp-options-debug-add (name)
  "Add a new NAME to the `debug' option's list."
  (interactive
   ;; For now, allow the user to enter a new function that we do not know about
   (list (completing-read "Function to debug: " 
			  (symp-options-debug-table) nil nil)))
  (symp-options-debug-edit name))

(defun symp-options-debug-remove (name)
  "Remove a NAME from the `debug' option's list."
  (interactive
   (let ((debug-value-table (mapcar  '(lambda (el) (cons el nil))
				     (symp-options-get-value "debug"))))
     (list (completing-read "Stop debugging function: " 
			    debug-value-table nil t))))
  (symp-options-debug-edit name t))

(defun symp-options-request-and-display ()
  "Requests options values from the server that will be displayed when
received."
  (interactive)
  ;; WARNING! This function will not be loaded with "require"
  (symp-server-send-expr '(displayoptions)))

(defun symp-options-mode ()
  "SyMP major mode for displaying and editing SyMP options.

At this point, only displaying is implemented.  Edit options with
\\[symp-options-edit] command."
  (interactive)
  ;;; Define the major mode
  (setq major-mode 'symp-options-mode)
  ;; Watch the status of the inferior SYMP process
  (setq mode-line-process 'symp-server-status)  
  (setq mode-name "SyMP Options")
  (symp-server-register-buffer))

(provide 'symp-options-mode)
