;;;; Major mode for a SyMP message buffer and the set of functions to
;;;; send and display messages.

(require 'symp-common)

(defun symp-message-buffer ()
  "Find or create a new buffer for displaying SyMP messages.  This
function should never return nil."
    (let ((symp-buffer symp-server-process-buffer)
	  (process symp-server-process)
	  (status symp-server-status)
	  (old-buffer (current-buffer))
	(message-buffer nil))
    (if symp-buffer
	(let ((buffer-list nil))
	  ;; Enter the main server buffer and search in its buffer list
	  (set-buffer symp-buffer)
	  (setq buffer-list symp-server-buffer-list)
	  (while (and (not message-buffer) buffer-list)
	    (if (buffer-live-p (car buffer-list))
		(progn
		  (set-buffer (car buffer-list))
		  (if (eq major-mode 'symp-message-mode)
		      (setq message-buffer (car buffer-list))))
	      ;; the buffer is dead, remove it from the list
	      (progn
		(set-buffer symp-buffer)
		(setq symp-server-buffer-list
		      (delete (car buffer-list) symp-server-buffer-list))))
	    (setq buffer-list (cdr buffer-list)))
	  ;; if no message buffer, create one
	  (if (not message-buffer)
	      (progn
		(setq message-buffer (generate-new-buffer "SyMP Message"))
		(set-buffer message-buffer)
		(setq symp-server-process-buffer symp-buffer)
		(setq symp-server-status status)
		(setq symp-server-process process)
		(symp-message-mode))))
      ;; There is no master symp buffer.  This is odd, so we get the
      ;; global default message buffer.
      (progn
	(setq message-buffer (get-buffer-create "SyMP Message Default"))
	(set-buffer message-buffer)
	(setq symp-server-process-buffer nil)
	(setq symp-server-process nil)
	(symp-message-mode)))
    ;; Restore the original buffer and return
    (set-buffer old-buffer)
    message-buffer))

(defun symp-message (string &rest args)
  "Displays STRING and all the extra arguments as a SyMP message.  The
format of the string and argumets is the same as for `format' and
`message'.  The string is inserted into the SyMP message buffer and
echoed in the minibuffer."
  (let ((msg (apply 'format (cons string args)))
	(old-buffer (current-buffer))
	(msg-buffer (symp-message-buffer))
	(old-point nil))
    (if msg-buffer
	(progn
	  (set-buffer msg-buffer)
	  (setq old-point (if (eobp) nil (point)))
	  (goto-char (point-max))
	  (if old-point 
	      (message "will restore the point")
	    (message "will NOT restore the point"))
	  (insert msg)(insert "\n")
	  (if old-point
	      (goto-char old-point)
	    (goto-char (point-max)))
	  (set-buffer old-buffer)))
    (message msg)))

(defun symp-message-mode ()
  "Major mode for displaying verbose messages of SyMP server."
  (interactive)
  ;; Define the major mode
  (setq major-mode 'symp-message-mode)
  (setq mode-name "SyMP Message")
  ;; (use-local-map symp-message-mode-map)

  ;; Watch the status of the inferior SYMP process
  (setq mode-line-process 'symp-server-status)
  ;; Register the buffer with the server
  (symp-server-register-buffer))

(provide 'symp-message-mode)
