;;;; Major mode for SyMP error messages.

(require 'symp-common)

(defvar symp-error-buffer nil
  "The buffer for reporting an error.  This variable is buffer local,
and is normally stored in the SyMP server buffer.")

(defun symp-error-find-buffer ()
  "Find the error buffer for the current SyMP session, or create one
if not found."
  (let ((old-buffer (current-buffer))
	(status symp-server-status)
	(process symp-server-process)
	(symp-buffer symp-server-process-buffer)
	(buffer nil))
    (if symp-buffer
	;; there is a symp server buffer, go there
	(progn
	  (set-buffer symp-buffer)
	  (if (and symp-error-buffer
		   (bufferp symp-error-buffer)
		   (buffer-live-p symp-error-buffer))
	      ;; the error buffer already exists, just take it
	      (setq buffer symp-error-buffer)
	    ;; no error buffer, create a new one
	    (progn
	      (setq buffer (generate-new-buffer "*SyMP Error*"))
	      ;; register it in the server buffer
	      (setq symp-error-buffer buffer)
	      ;; and set up the error buffer properly
	      (set-buffer buffer)
	      (setq symp-server-process process)
	      (setq symp-server-status status)
	      (setq symp-server-process-buffer symp-buffer)
	      (symp-error-mode)))
	  (set-buffer old-buffer))
      ;; if there is no symp buffer, give up, leave `buffer' nil.
      )
    buffer))

(defun symp-error-erase-buffer ()
  "Erases the SyMP error buffer for the current SyMP session and hides
it if it's displayed."
  (let ((old-buffer (current-buffer))
	(err-buffer (symp-error-find-buffer))
	(start-point 0)
	(windows nil))
    (if err-buffer
	(progn
	  (set-buffer err-buffer)
	  (erase-buffer)
	  ;; Search for all the windows that display the buffer
	  (walk-windows 
	   '(lambda (window) 
	      (if (equal err-buffer (window-buffer window))
		  (add-to-list 'windows window)))
	   'no t)
	  ;; Delete all the collected windows
	  (while windows
	    (delete-window (car windows))
	    (setq windows (cdr windows)))
	  (set-buffer old-buffer)))))

(defun symp-error-display-message (message &optional erase)
  "Pop up the error buffer and display the MESSAGE string in it.  If
the optional ERASE is non-nil, erase the buffer before displaying the
message."
  (let ((old-buffer (current-buffer))
	(err-buffer (symp-error-find-buffer))
	(start-point 0))
    (if err-buffer
	(progn
	  ;; put the message into the error buffer
	  (set-buffer err-buffer)
	  (if erase (erase-buffer))
	  (goto-char (point-max))
	  (setq start-point (point))
	  (insert "\n")
	  (insert message)
	  (goto-char start-point)
	  ;; go back and display the error buffer in another window
	  (set-buffer old-buffer)
	  (display-buffer err-buffer t)
	  (message "SyMP Error!")
	  (beep))
      ;; Ouch, no error buffer!  So do the best we can...
      (message message))))

(defun symp-error-mode ()
  "SyMP major mode for error reporting."
  (interactive)
  ;;; Define the major mode
  (setq major-mode 'symp-error-mode)
  ;; Watch the status of the inferior SYMP process
  (setq mode-line-process 'symp-server-status)  
  (setq mode-name "SyMP Error")
  ;; Register the buffer with the server
  (symp-server-register-buffer))

(provide 'symp-error-mode)
