;;;; Major mode for SyMP debugging output

(require 'symp-common)

(defun symp-debug-display (message)
  "Display MESSAGE in a special debugging buffer for the current SyMP
process."
  (let ((old-buffer (current-buffer))
	(debug-buffer (symp-debug-find-buffer)))
    (if debug-buffer
	(let ((old-point nil)
	      (restore-point t))
	  (set-buffer debug-buffer)
	  (setq old-point (point))
	  (goto-char (point-max))
	  (setq restore-point (not (= (point) old-point)))
	  (insert message)
	  (if restore-point
	      (goto-char old-point)
	    (goto-char (point-max)))
	  (set-buffer old-buffer))
      ;; No debug buffer found (which is a bug), just send a message instead
      (message message))))

(defun symp-debug-find-buffer ()
  "Find a debug buffer for the current SyMP server, or create a new
one if it doesn't exist.  This function should never return nil."
  (let ((symp-buffer symp-server-process-buffer)
	(process symp-server-process)
	(status symp-server-status)
	(old-buffer (current-buffer))
	(debug-buffer nil))
    (if symp-buffer
	(let ((buffer-list nil))
	  ;; Enter the main server buffer and search in its buffer list
	  (set-buffer symp-buffer)
	  (setq buffer-list symp-server-buffer-list)
	  (while (and (not debug-buffer) buffer-list)
	    (if (buffer-live-p (car buffer-list))
		(progn
		  (set-buffer (car buffer-list))
		  (if (eq major-mode 'symp-debug-mode)
		      (setq debug-buffer (car buffer-list))))
	      ;; the buffer is dead, remove it from the list
	      (progn
		(set-buffer symp-buffer)
		(setq symp-server-buffer-list
		      (delete (car buffer-list) symp-server-buffer-list))))
	    (setq buffer-list (cdr buffer-list)))
	  ;; if no debug buffer, create one
	  (if (not debug-buffer)
	      (progn
		(setq debug-buffer (generate-new-buffer "SyMP Debug"))
		(set-buffer debug-buffer)
		(setq symp-server-process-buffer symp-buffer)
		(setq symp-server-status status)
		(setq symp-server-process process)
		(symp-debug-mode))))
      ;; There is no master symp buffer.  This is odd, so we get the
      ;; global default debug buffer.
      (progn
	(setq debug-buffer (get-buffer-create "SyMP Debug Default"))
	(set-buffer debug-buffer)
	(setq symp-server-process-buffer nil)
	(setq symp-server-process nil)
	(symp-debug-mode)))
    ;; Restore the original buffer and return
    (set-buffer old-buffer)
    debug-buffer))

(defun symp-debug-mode ()
  "Major mode for displaying debugging output of SyMP server."
  (interactive)
  ;; Define the major mode
  (setq major-mode 'symp-debug-mode)
  (setq mode-name "SyMP Debug")
  ;; (use-local-map symp-debug-mode-map)

  ;; Watch the status of the inferior SYMP process
  (setq mode-line-process 'symp-server-status)
  ;; Debug output is often chatty, this will keep emacs more sane
  (buffer-disable-undo)
  ;; Register the buffer with the server
  (symp-server-register-buffer))

(provide 'symp-debug-mode)
