(defvar symp-debug t
  "If non-nil, print the debugging info in the SyMP Debug buffer.")

(defvar symp-font-lock-mode-on t
  "If not nil, turn the fontification on.")

(defun symp-list-to-alist (lst)
  "Converts a list of elements into an alist for `completing-read'
function."
  (let ((res nil))
    (while lst
      (setq res (cons (cons (car lst) nil) res))
      (setq lst (cdr lst)))
    res))

(defun symp-keyword-match (keyword)
  "Convert a string into a regexp matching any capitalization of that string."
;  "Convert a string into a regexp matching that string as a separate word."
  (let ((regexp "")
	(index 0)
	(len (length keyword)))
    (while (< index len)
      (let ((c (aref keyword index)))
	(setq regexp
	      (concat regexp (format "[%c%c]" (downcase c) (upcase c))))
	(setq index (+ index 1))))
;    (format "\\b%s\\b" keyword)
    (format "\\b%s\\b" regexp)
    ))

(defvar symp-font-lock-separator-face 'symp-font-lock-separator-face)
(defvar font-lock-preprocessor-face 'font-lock-preprocessor-face)
(defvar font-lock-variable-name-face 'font-lock-variable-name-face)

(if (facep 'font-lock-preprocessor-face) ()
    (progn
      (make-face 'font-lock-preprocessor-face)
      (set-face-foreground 'font-lock-preprocessor-face "green4")))

(if (facep 'font-lock-variable-name-face) ()
    (progn
      (make-face 'font-lock-variable-name-face)
      (set-face-foreground 'font-lock-variable-name-face "deeppink")))

(if (facep 'symp-font-lock-separator-face) ()
    (progn
      (make-face 'symp-font-lock-separator-face)
      (set-face-foreground 'symp-font-lock-separator-face "indianred")))

(defconst symp-spec-types '("default")
  "List of values allowed for SPEC-TYPE in `symp-typecheck', which
identifies the proof system to use.")

(defvar symp-spec-type "default"
  "Specifies the default proof system for the current SyMP buffer.")
(make-variable-buffer-local 'symp-spec-type)

(defun symp-typecheck (&optional spec-type)
  "Typecheck the current file.  The input language is determined by
SPEC-TYPE argument, the major mode of the current buffer, or left for
the tool to decide, in that order."

  (interactive 
;   (list (completing-read "Select proof system: "
;			  (symp-list-to-alist symp-spec-types)
;			  nil t symp-spec-type))
)
;   (if (or (null spec-type)
; 	  (equal spec-type ""))
;       (setq spec-type "default"))
  (setq symp-spec-type spec-type)
  ;; more cases will be added for future prover types
  (cond ((eq major-mode 'symp-mode)
	 (if (not spec-type) (setq spec-type "default")))
	((eq major-mode 'symp-athena-mode)
	 (if (not spec-type) (setq spec-type "athena")))
	(t (error "Buffer is not in any known SyMP modes")))
  (let ((file nil))
    (save-buffer)
    ;; Erase all the previous errors
    (symp-error-erase-buffer)
    ;; set the file here, since `save-buffer' may ask the user for
    ;; a file name if the buffer was not associated with a file
    (setq file (buffer-file-name))
    ;; ...and double check that we HAVE a file name
    (if file
	(progn
	  ;; Restart the server if needed
	  (if (not (and symp-server-process
			(eq (process-status symp-server-process) 'run)))
	      (symp-server-start))
	  ;; Send the command for typechecking
	  (symp-server-send-expr 
	   (if spec-type
	       (list 'typecheck file spec-type)
	     (list 'typecheck file))))
      (error "Buffer is not associated with any file"))))

(defun read-list-from-minibuffer (prompt)
  "Read several user inputs from minibuffer until an empty string is
read, and returns them in a list."
  (let ((inputs nil)
	(str nil))
    (setq str (read-from-minibuffer prompt))
    (while (not (string= str ""))
      (setq inputs (cons str inputs))
      (setq str (read-from-minibuffer prompt)))
    (reverse inputs)))

(defvar symp-prove-previous-module ""
  "Previously chosen module name in which a prover was invoked for a theorem.") 
(defvar symp-prove-previous-theorem ""
  "Previously chosen theorem name for which a prover was invoked.") 

(defun symp-prove (module theorem &optional name)
  "Sets the default MODULE and starts the prover for the THEOREM.  If
the optional NAME is given, assign this theorem the NAME.  Otherwise
the prover decides on the name itself."
  (interactive 
   (let ((need-typecheck nil))
     ;; Restart the server and typecheck file if needed
     (if (not (and symp-server-process
		   (eq (process-status symp-server-process) 'run)))
	 (progn
	   (symp-server-start)
	   (setq need-typecheck t)))
     (if need-typecheck (call-interactively 'symp-typecheck))
     (list (read-from-minibuffer "Context module: " symp-prove-previous-module)
	   (read-from-minibuffer "Theorem: " symp-prove-previous-theorem))))
  (setq symp-prove-previous-module module
	symp-prove-previous-theorem theorem)
  (symp-server-send-expr
   (append (list 'provetheorem
		 (if (equal module "") theorem (cons theorem module)))
	   (if name (list name) nil))))

(defun symp-server-stop ()
  "Ask the server to exit.  Don't try to kill it if it resists."
  (interactive)
  (symp-server-send-expr 'exit))

(defun symp-version ()
  "Ask the server to print its version."
  (interactive)
  (symp-server-send-expr 'version))

;;;; Now define the keymap

(defconst symp-mode-map nil  "SYMP keymap")

(if symp-mode-map ()
  (progn
    (setq symp-mode-map (make-sparse-keymap))
    (define-key symp-mode-map [delete] 'backward-delete-char-untabify)
    (define-key symp-mode-map [backspace] 'backward-delete-char-untabify)
    (define-key symp-mode-map "\C-c\C-s"  'symp-server-start)
    (define-key symp-mode-map "\C-c\C-c"  'symp-server-interrupt)
    (define-key symp-mode-map "\C-c\C-e"  'symp-server-exit)
    (define-key symp-mode-map "\C-c\C-t"  'symp-typecheck)
    (define-key symp-mode-map "\C-c\C-p"  'symp-prove)
    (define-key symp-mode-map "\C-c\C-o"  'symp-options-request-and-display)
    (define-key symp-mode-map "\C-co"  'symp-options-edit)
    (define-key symp-mode-map "\C-c\C-d"  'symp-options-debug-add)
    (define-key symp-mode-map "\C-cd"  'symp-options-debug-remove)
    (define-key symp-mode-map "\C-c;"  'comment-region)
;    (define-key symp-mode-map "\t"  'symp-indent-line)
    ))

(provide 'symp-common)
