(* Typechecking without overloading of identifiers *)

functor TypecheckFun(structure Hash: HASH
		     structure Evaluate: EVALUATE): TYPECHECK =
  struct
    structure Hash = Hash
    structure Evaluate = Evaluate
    structure TopClosureStruct
	= TopClosureFun(structure ParseTreeStruct = Evaluate.ParserDefault.ParseTreeStruct)

    open Str
    open ListOps
    open Pos
    open Evaluate
    open ParserDefault
    open Interface
    open TimingStat
    open TopClosureStruct
    open ParseTreeStruct
    open Hash
    open Options

    (* Some vars for statistics *)
    val tHashMaxLength = ref 0

    fun typeCheckStat2string() =
	("tHash max length = "^(Int.toString(!tHashMaxLength))^"\n"
	 ^(stat2string()))

    (* Reset all the statistical variables *)
    fun resetTypeCheckStat() = (resetStat(); tHashMaxLength:=0)

    (* Different judgements for typechecking expressions, declarations,
     expanding types, etc. *)
    datatype Judgement =
	(* HasType(C,E,T): Expression E has type T in context C *)
	HasType of ParseTree * ParseTree * ParseTree
      (* Decl(C,D,declList): Declaration D in context C provides 
       * the declList of new objects *)
      | Decl of ParseTree * ParseTree * (ParseTree list)
      (* ModuleJudgement(C, M, Mclosure): Module M in context C has a closure Mclosure *)
      | ModuleJudgement of ParseTree * ParseTree * ParseTree
	(* ThmJudgement(C,Thm,Thm_tc'ed) *)
      | ThmJudgement of ParseTree * ParseTree * ParseTree
      | ProgramJudgement of ParseTree

    fun newTvar() = TvarN(newNumber())

    fun TvarNum(TvarN n) = n
      | TvarNum x = raise SympBug("TvarNum: not a TvarN: "^(pt2stringDebug x))

    (* Returns the list of type variable names (string list) *)
    fun collectVars tp =
	let fun insertList lst x =
	       if List.exists (fn y => ptEq(y,x)) lst then lst
	       else x::lst
	    fun loop hash (Tvar(p,n)) = insertList hash (Tvar(p,n))
	      | loop hash (TvarN n) = insertList hash (TvarN n)
	      | loop hash (TypeInst(_,lst,tp)) =
	           loop (looplist hash lst) tp
	      | loop hash (FunType(_,x,y)) = loop (loop hash x) y
	      | loop hash (TupleType(_,lst)) = looplist hash lst
	      | loop hash (ArrayType(_,x,y)) = loop (loop hash x) y
	      | loop hash (RecordType(_,lst)) = looplist hash lst
	      | loop hash (RecordField{Type=tp,...}) = loop hash tp
	      | loop hash (TypedVars(_,_,opt)) =
		   (case opt of
			NONE => hash
		      | SOME(x) => loop hash x)
	      | loop hash (EnumType(_,lst)) = looplist hash lst
	      | loop hash (TypeConstr{Type=tp,...}) = loop hash tp
	      | loop hash (Of(_,_,x)) = loop hash x
	      | loop hash _ = hash
	    and looplist hash [] = hash
	      | looplist hash (hd::tl) = looplist (loop hash hd) tl
	in loop [] tp
	end

    (* Converts the list of type variables to the list of strings,
     * removing non-polymorphic variables *)
    fun collectVarsPoly tp =
	let fun ff (Tvar(_,n)) = SOME n
	      | ff _ = NONE
	in
	    List.mapPartial ff (collectVars tp)
	end

    (* Converts the list of type variables to the list of numbers,
     * removing polymorphic variables *)
    fun collectVarsN tp =
	let fun ff (TvarN n) = SOME n
	      | ff _ = NONE
	in
	    List.mapPartial ff (collectVars tp)
	end

    (* Instantiates an incompletely specified type according to the
       substitution hash.  That is, replaces all the non-poly vars
       repeatedly until nothing changes.

       This function is called ***way*** too often and must be
       **really** fast.  It is acceptably fast already, but any
       further enhancements wouldn't hurt.*)

    fun substTypeCommon evalVar hash tp =
	(* "Evaluates" the type variable in the "hash" environment *)
	let val timer = startTimer "substTypeCommon"
	    fun evalVar1 x = evalVar eval hash x
	    (* "Evaluate" the type *)
	    and eval tp =
		 (case tp of
		      Tvar(p,name) => evalVar1(Tvar(p,name))
		    | TvarN n => evalVar1(TvarN n)
		    | BoolType _ => tp
		    | NatType _ => tp
		    | IntType _ => tp
		    | Id _ => tp (* For recursive references in datatypes *)
		    | Uid _ => tp (* References to the base type in datatype constructors *)
		    | TypeClosure _ => tp
		    | PType _ => tp
		    | StaticFormalType _ => tp
		    | FunType(p,t1,t2) => FunType(p,eval t1,eval t2)
		    | ArrayType(p,t1,t2) => ArrayType(p,eval t1,eval t2)
		    | TupleType(p,lst) => TupleType(p,evallist lst)
		    | RecordType(p,lst) => RecordType(p,evalRecList lst)
		    | EnumType(p,lst) => EnumType(p,evallist lst)
		    | TypeInst(p,lst,tp) => TypeInst(p,evallist lst,tp)
		    | RecordField{name=n,Type=tp} => 
			  RecordField{name=n,Type=eval tp}
		    | TypedVars(p,lst,SOME(t)) => TypedVars(p,lst,SOME(eval t))
		    | Group(p,tp) => eval tp
		    | Type(_,tp) => eval tp
		    | Datatype(_,tp) => eval tp
		    | TypedVars(_,_,NONE) =>
			  raise SympBug("substType/eval: untyped field "
					^"names in a record type: "
					^(pt2stringDebug tp))
		    | Of(p,t1,t2) => Of(p,eval t1,eval t2)
		    | TypeConstr{name=n,uname=u,Type=tp} => 
			TypeConstr{name=n,uname=u,Type=eval tp}
		    | _ => raise SympBug("substType/eval: "
					 ^"unknown type "
					 ^(pt2stringDebug tp)))
	    and evallist lst = map eval lst
	    and ff (Tvar(p1,v1)) = [Tvar(p1,v1)]
	      | ff (TvarN n) = [TvarN n]
	      | ff (RecordType(_,lst)) = lst
	      | ff x = raise SympBug("substType/evalRecList: bad value of "
				     ^"partial record var: "
				     ^(pt2stringDebug x))
	    and evalRecList [] = []
	      | evalRecList [Tvar x] = ff(evalVar1(Tvar x))
	      | evalRecList [TvarN x] = ff(evalVar1(TvarN x))
	      | evalRecList (hd::tl) = (eval hd)::(evalRecList tl)
	in (eval tp) before (updateStat timer)
	end

    (* Substitute polymorphic variables.  The hash maps strings (Tvar's names)
     * to types *)
    fun substTypePoly hash tp =
	let fun evalVar eval hash (Tvar(p,v)) =
	         (case findHash(hash,v) of
		      NONE => Tvar(p,v)
		    | SOME(tp) => eval tp)
	      | evalVar eval _ x = x
	in 
	    substTypeCommon evalVar hash tp
	end

    (* Substitutes non-polymorphic (numerical) type variables *)
    fun substType hash tp =
	let fun evalVar eval hash (TvarN n) =
	         (case findHash(hash,n) of
		      NONE => TvarN n
		    | SOME(tp) => eval tp)
	      | evalVar eval _ x = x
	in 
	    substTypeCommon evalVar hash tp
	end

    fun instantiateType tp parms args =
	let (* val _ = print("instantiateType("^(pt2string tp)
			  ^",\n  ["^(ptlist2str "," parms)
			  ^"],\n  ["^(ptlist2str "," args)^"])[\n") *)
	    fun loop [] [] hash = hash
	      | loop ((Tvar(_,n))::parms) (tp::args) hash = 
	           insertHash(hash,n,tp)
	      | loop (x::parms) (_::_) _ =
		   raise SympBug("instantiateType: wrong type parameter: "
				 ^(pt2stringDebug x))
	      | loop p a _ = raise SympBug("instantiateType: "
					   ^"wrong number of arguments: "
					   ^(pt2stringDebug tp)^"\nexpects:\n  "
					   ^(Int.toString(List.length p))
					   ^"\nbut received:\n  "
					   ^(Int.toString(List.length a)))
	    val hash = loop parms args (makeHashDefault(op =, fn x=>x))
	    val res = substTypePoly hash tp
	in (* (print("instantiateType => "^(pt2string res)^"]\n")); *)
	    res
	end

    (* Generates a substitution hash for the first type that will make its
     * polymorphic variables disjoint from the second type.  Note, that
     * the non-polymorphic variables do NOT change. *)
    fun alphaHash(tp1,tp2) =
	let val hash = makeHashDefault(op =,fn x => x)
	    val vars1 = collectVarsPoly tp1
	    val vars2 = collectVarsPoly tp2
	    val common = List.filter(fn x => List.exists(fn y => x = y) vars2)
		                    vars1
	    fun foldfun(str,hash) =
		   insertHashDestructive(hash,str,newTvar())
	in List.foldr foldfun hash common
	end

    (* Generates a new hash that would replace each _polymorphic_ variable
     * in the type by a new _non-polymorphic_ variable. *)
    fun newVarsHash tp =
	let val hash = makeHashDefault(op =,fn x => x)
	    val vars = collectVarsPoly tp
	    fun foldfun(str,hash) =
		   insertHashDestructive(hash,str,newTvar())
	in List.foldr foldfun hash vars
	end

    (* Generates a new hash that would replace each _polymorphic_ variable
     * in the type by a new _non-polymorphic_ variable, except for vars
     * in `polyhash'. *)
    fun newVarsHashExcept polyhash tp =
	let val hash = makeHashDefault(op =,fn x => x)
	    val vars = List.filter(fn n=>not(isSome(findHash(polyhash,n))))
		                  (collectVarsPoly tp)
	    fun foldfun(str,hash) =
		   insertHashDestructive(hash,str,newTvar())
	in List.foldr foldfun hash vars
	end

    (* Replace all vars in the type by 'a, 'b, ... to make it look nicer *)
    fun abcVarsHash tp =
	let val hash = makeHashDefault(op =,fn x => x)
	    val Na = Char.ord(#"a")
	    val N = ref(0)
	    fun N2str n = ((Char.toString(chr(Na+(n mod 26))))
			   ^(case n div 26 of
				 0 => ""
			       | x => Int.toString x))
	    val vars = collectVarsPoly tp
	    fun foldfun(str,hash) =
		   insertHash(hash,str,Tvar(dp,(N2str(!N) before N := (!N)+1)))
	in List.foldr foldfun hash vars
	end

    (* Compute hash to replace non-polymorphic variables numbered 
     * >= n with unique polymorphic variables *)
    fun polyVarsHash n tp =
	let fun ff m = if m>=n then SOME m else NONE
	    fun foldfun(m,hash) = insertHash(hash,m,Tvar(dp,"Y!"^(Int.toString m)))
	    val vars = List.mapPartial ff (collectVarsN tp)
	    val hash = makeHashDefault(op =,Int.toString)
	in List.foldr foldfun hash vars
	end

    (* Return tp1 with disjoint polymorphic vars from tp2.
     * It is probably not needed anymore. *)
    fun alphaSubst(tp1,tp2) = substTypePoly(alphaHash(tp1,tp2)) tp1

    (* Replaces all the polymorphic variables by new non-polymorphic ones
     * in the type tp *)
    fun newVarsSubst tp = substTypePoly(newVarsHash tp) tp

    (* Replaces all the polymorphic variables (except for those in `polyhash')
     * by new non-polymorphic ones in the type tp *)
    fun newVarsSubstExcept polyhash tp = substTypePoly(newVarsHashExcept polyhash tp) tp

    (* Replaces all polymorphic variables by alphabetical ones *)
    fun abcVarsSubst tp = substTypePoly(abcVarsHash tp) tp

    (* Replaces all the non-polymorphic variables >= n with new 
     * polymorphic ones *)
    fun polyVarsSubst n tp = substType(polyVarsHash n tp) tp

    (* Different context search modes *)
    datatype SearchBits = 
	ValuesOnly of bool (* whether to include state vars or not *)
      (* Same as (ValuesOnly false), only excluding formal parameters of functions *)
      | StaticValuesOnly
      | TypeConstrOnly
      | TypesOnly
      | ModulesOnly
      | StateVarOnly
      | TheoremsOnly

    datatype SearchName = UserName | UName

    datatype SearchMode = 
	Global of SearchName * SearchBits
      | Local of SearchName * SearchBits
      | ModuleOnly of SearchName * SearchBits

    fun GetSearchBits (Global(_,x)) = x
      | GetSearchBits (Local(_,x)) = x
      | GetSearchBits (ModuleOnly(_,x)) = x

    fun GetSearchName (Global(x,_)) = x
      | GetSearchName (Local(x,_)) = x
      | GetSearchName (ModuleOnly(x,_)) = x

    fun bits2str (ValuesOnly true) = "ValuesOnly(true)"
      | bits2str (ValuesOnly false) = "ValuesOnly(false)"
      | bits2str StaticValuesOnly = "StaticValuesOnly"
      | bits2str TypeConstrOnly = "TypeConstrOnly"
      | bits2str TypesOnly = "TypesOnly"
      | bits2str ModulesOnly = "ModulesOnly"
      | bits2str StateVarOnly = "StateVarOnly"
      | bits2str TheoremsOnly = "TheoremsOnly"

    fun SearchName2str UserName = "UserName"
      | SearchName2str UName = "UName"

    fun SearchMode2str mode =
	let fun pair2str(name, bits) = "("^(SearchName2str name)^", "^(bits2str bits)^")"
	    fun loop(Global pair) = "Global"^(pair2str pair)
	      | loop(Local pair) = "Local"^(pair2str pair)
	      | loop(ModuleOnly pair) = "ModuleOnly"^(pair2str pair)
	in
	    loop mode
	end

    (* Finds the list of objects that have this name in the list 
     * provided.  It is supposed to be called on a closure, so
     * it's better be a list of "Object" values and the like. *)
    fun findName bits searchname (name,lst) = 
	let fun isTypeConstr(TypeConstr _) = true
	      | isTypeConstr _ = false
	    fun isModule (ModuleClosure _) = true
	      | isModule _ = false
	    fun isType (TypeClosure _) = true
	      | isType (PType _) = true
	      | isType (StaticFormalType _) = true
	      | isType _ = false
	    fun isStateVar (StateVar _) = true
	      | isStateVar _ = false
	    fun isConst _ (Object _) = true
	      | isConst s (StateVar _) = s 
	      | isConst _ (FunClosure _) = true
	      | isConst _ (TypeConstr _) = true
	      | isConst _ (RecordField _) = true
	      | isConst _ (QuantifiedVar _) = true
	      | isConst _ (StaticFormalConst _) = true
	      | isConst _ (PatternFormal _) = true
	      | isConst s (DynPatternFormal _) = s
	      | isConst _ (SkolemConst _) = true
	      (* Let's hope it doesn't point to itself... *)
	      (*  | isConst (SharedRef x) = not tconly andalso isConst(!x) *)
	      | isConst _ x = false
	    fun isTheorem (Theorem _) = true
	      | isTheorem _ = false
	    fun isX ff x =
		if ff x then
		    (case (case searchname of
			       UserName => GetObjectName 
			     | UName => GetObjectUName) x of
			 NONE => false
		       | SOME n => ptEq(n,name))
		else false 
	in (case bits of
		ModulesOnly => List.filter (isX isModule) lst
	      | TypeConstrOnly => List.filter (isX isTypeConstr) lst
	      | TypesOnly => List.filter (isX isType) lst
	      | StateVarOnly => List.filter (isX isStateVar) lst
	      (* Allow state variables and formal parameters of functions here,
	         we'll catch them later and report an error if they are not allowed. *)
	      | ValuesOnly s => List.filter (isX (isConst true)) lst
	      | StaticValuesOnly => List.filter (isX (isConst true)) lst
	      | TheoremsOnly => List.filter (isX isTheorem) lst)
	end

    (* Finds the list of objects with a given name in a given context.
     * "strict" means "this context proper", don't go to parent contexts. *)
    fun findinContextCommon mode context name =
	let val bits = GetSearchBits mode
	    val _ = pushFunStackLazy("findinContextCommon",
				     fn()=>"mode = "^(SearchMode2str mode)
				     ^", name = "^(pt2string name))
	    val debug = lazyVerbDebug (getOptions()) "findinContextCommonDebug"
	    val _ = debug(fn()=>("\ncontext = \n"^(pt2string context)
				 ^"\nend of context\n"))
(* 	    val _ = debug(debug "\ncontext = \n"; ptPrintDebug debug context;
			  "\nend of context\n") *)
	    val searchname = GetSearchName mode
	    fun doFind(lst, pcl) =
		(findName bits searchname(name,lst))
		@(case mode of
		      Local _ => []
		    | _ => findinContextCommon mode pcl name)
	    fun split(ModuleClosure{closure=cl,
				    Sig=ModuleSig{statparams=statOpt, dynParam=dynOpt},
				    parent=pcl, ...}) =
		let val params = (case dynOpt of
				      NONE => []
				    | SOME dyn => [dyn])
		                @(case statOpt of
				      NONE => []
				    | SOME lst => lst)
		in doFind(cl@params,pcl)
		end
	      (* | split(ModuleParamClosure{closure=cl, parent=pcl}) =
		   (findName bits searchname(name,cl))
		   @(case mode of
			 Global bits =>
			     findinContextCommon(Global bits)
			 (* If StateVar's from outside shouldn't be visible,
			    change the search mode when leaving the module's scope.
			    But it looks like they should... *)
				      (* (Global(case bits of
					   ValuesOnly s => ValuesOnly false
					 | x => x)) *)
			       pcl name
		       | _ => []) *)
	      | split(FunClosure{formals=fp, parent=pcl, ...}) = doFind([fp], pcl)
	      | split(LetClosure{locals=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(LetAsstClosure{locals=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(ChoiceClosure{names=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(ChoiceAsstClosure{names=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(ChooseAsstClosure{names=lstOpt, parent=pcl,...}) =
		  doFind(case lstOpt of NONE => [] | SOME lst => lst, pcl)
	      | split(ForeachAsstClosure{names=lst, parent=pcl,...}) = doFind(lst, pcl)
	      | split(ChooseClosure{names=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(ForallClosure{names=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(ExistsClosure{names=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(SyncClosure{names=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(AsyncClosure{names=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(TypeClosure{params=lst,parent=pcl,...}) = doFind(lst, pcl)
	      | split(SequentClosure{names=lst, parent=pcl,...}) = doFind(lst, pcl)
	      | split(RecordExpr(p,lst)) = (findName bits searchname(name,lst))
	      | split(TopClosure lst) = (findName bits searchname(name,lst))
	      | split x = raise SympBug("findinContextCommon: bad closure: "
					^(pt2stringDebug x))
	    val res = split context
	    val _ = popFunStackLazy("findinContextCommon", fn()=>"["^(ptlist2str ", " res)^"]")
	in res
	end

    (* Find the first constant with a given name.
       If `state' is true, allow state variables along with constants. *)
    fun findNameCommon mode context name = 
	let val timer = startTimer "findinContextCommon"
	    val obj = (case findinContextCommon mode context name of
			   [] => NONE
			 | hd::_ => SOME hd)
	    val _ = updateStat timer
	    val bits = GetSearchBits mode
	in (case bits of
		ValuesOnly false =>
		    (case obj of
			 SOME(StateVar _) => raise TypeError
			     ((pos2string(pos name))
			      ^": state variables are not allowed in this context: "
			      ^(pt2string name))
		       | _ => obj)
	      | StaticValuesOnly => 
		    (case obj of
			 SOME(StateVar _) => raise TypeError
			     ((pos2string(pos name))
			      ^": state variables are not allowed in this context: "
			      ^(pt2string name))
		       | SOME(PatternFormal _) => raise TypeError
			     ((pos2string(pos name))
			      ^": formal parameters are not allowed in this context: "
			      ^(pt2string name))
		       | _ => obj)
	      | _ => obj)
	end

    fun findNameGlobal bits = findNameCommon (Global(UserName,bits))

    fun findNameLocal bits = findNameCommon (Local(UserName,bits))

    fun findUNameGlobal bits = findNameCommon (Global(UName,bits))

    fun findUNameLocal bits = findNameCommon (Local(UName,bits))

    (* Functions to export *)
    val findNameInContext = findNameGlobal (ValuesOnly false)
    val findNameInLocalContext = findNameLocal (ValuesOnly false)

    (* Find the first module with a given name *)
    fun findModule context name = 
	  findNameGlobal ModulesOnly context name

    fun findModuleUname context name = 
	  findUNameGlobal ModulesOnly context name

    (* Find a state variable with a given name.
       Here we should not go beyond the scope of the currect module. *)
    fun findStateVar context name = 
	  findNameCommon (ModuleOnly(UserName,StateVarOnly)) context name

    fun findStateVarUname context name = 
	  findNameCommon (ModuleOnly(UName,StateVarOnly)) context name

    (* Find the definition of a type.  
     * Types have no overloading (i.e. no two types can be named with 
     * the same name) and they simply shadow previous declarations. *)
    fun findTypeName context name = findNameGlobal TypesOnly context name
    fun findTypeNameLocal context name = findNameLocal TypesOnly context name

    fun findTypeUName context name = findUNameGlobal TypesOnly context name
    fun findTypeUNameLocal context name = findUNameLocal TypesOnly context name

    (* Merge two type variable's hashes, so h2(h1('a)) = h('a), where
       h is the merged hash.  These functions must be as efficient as
       possible, since hashes can grow pretty rapidly and they are
       merged frequently.  We use the destructive insert, avoid
       concatenation of lists, and do not insert if the variable is
       already in the hash.  Also, since `substType' computes the
       transitive closure of substitution (replaces repeatedly until
       no vars can be replaced), we run it and insert the final result
       for each variable, thus doing something similar to the
       union-find optimization. *)

    fun mergeHash h2 h1 =
	let val timer = startTimer "mergeHash"
	    val l1=List.map(fn(n,_)=>n)(hash2any(fn x=>x) (fn x=>x) h1)
	    val l2=List.map(fn(n,_)=>n)(hash2any(fn x=>x) (fn x=>x) h2)
	    val count = ref 0
	    fun update h n = 
		(case findHash(h,n) of
		     SOME _ => h
		   | NONE => (count := (!count)+1;
			      insertHashDestructive(h,n,substType h2 (substType h1 (TvarN n)))))
	    fun loop [] h = h
	      | loop (n::tl) h = loop tl (update h n)
	in (loop l2 (loop l1 (makeHashDefault(op =,Int.toString))))
	    before (if (!count)>(!tHashMaxLength) then tHashMaxLength:=(!count)
		    else ();updateStat timer)
	end

    fun mergeHashPoly h2 h1 =
	let val timer = startTimer "mergeHashPoly"
	    val l1=List.map(fn(n,_)=>n)(hash2any(fn x=>x) (fn x=>x) h1)
	    val l2=List.map(fn(n,_)=>n)(hash2any(fn x=>x) (fn x=>x) h2)
	    fun update h n = 
		(case findHash(h,n) of
		     SOME _ => h
		   | NONE => insertHashDestructive
			 (h,n,substTypePoly h2 (substTypePoly h1 (Tvar(dp,n)))))
	    fun loop [] h = h
	      | loop (n::tl) h = loop tl (update h n)
	in (loop l2 (loop l1 (makeHashDefault(op =,fn x=>x))))
	    before (updateStat timer)
	end

(*      fun mergeHash h2 h1 = *)
(*  	let val l1=hash2any(fn x=>x) (fn x=>x) h1 *)
(*  	    val l2=hash2any(fn x=>x) (fn x=>x) h2 *)
(*  	    val l = List.filter(fn (x,_)=>not(List.exists(fn (y,_)=>x=y) *)
(*  					      l1)) l2 *)
(*  	    fun loop [] h = h *)
(*  	      | loop ((n,v)::tl) h = loop tl (insertHash(h,n,v)) *)
(*  	in loop l h1 *)
(*  	end *)
		
    (* Replaces all non-poly vars from<=n<to with poly ones in the hash *)
    fun polyVarsSubstHash from to hash =
	let fun dohash acc n =
	      if n >= to then acc
	      else
		  let val tp = substType hash (TvarN n)
		  in dohash (insertHash(acc,n,polyVarsSubst from tp)) (n+1)
		  end
	in dohash hash from
	end

    (* Create a string of types for error messages *)
    fun type2str h tp = pt2string (substType h tp)

    (* Printing the entire hash (for debugging) *)
    fun hash2str sep h =
	let val vars = hash2any(fn x=>TvarN x) (fn x=>x) h
	    val strs = List.map(fn(v,tp)=>(pt2string v)^"="
				^(pt2string (substType h tp))) vars
	    fun ff [] = ""
	      | ff [x] = x
	      | ff (hd::tl) = hd^sep^(ff tl)
	in "["^(ff strs)^"]"
	end

    fun polyhash2str sep h =
	let val vars = hash2any(fn x=>Tvar(dp,x)) (fn x=>x) h
	    val strs = List.map(fn(v,tp)=>(pt2string v)^"="
				^(pt2string (substTypePoly h tp))) vars
	    fun ff [] = ""
	      | ff [x] = x
	      | ff (hd::tl) = hd^sep^(ff tl)
	in "["^(ff strs)^"]"
	end

    (* Implements h |- e >> h' judgement: given a phash and
       expression, find any new poly vars in e that will be exported
       and add them to phash. *)

    fun newVarsExprHash phash expr =
	let fun looplist phash [] = phash
	      | looplist phash (e::lst) = loop (looplist phash lst) e
	    and loop phash (TypedExpr(_,e,tp)) =
		  let val h = newVarsHash(substTypePoly phash tp)
		  in mergeHashPoly h phash
		  end
	      | loop phash (IfExpr(_,lst,e)) = looplist phash (e::lst)
	      | loop phash (CondExpr(_,e1,e2)) = loop (loop phash e1) e2
	      | loop phash (LetExpr(_,_,e)) = loop phash e
	      | loop phash (CaseExpr(_,e,_)) = loop phash e
	      | loop phash (WithExpr(_,e,lst)) = looplist phash (e::lst)
	      | loop phash (WithAsst(_,_,e)) = loop phash e
	      | loop phash (RecordExpr(_,lst)) = looplist phash lst
	      | loop phash (RecordAsst(_,_,e)) = loop phash e
	      | loop phash (TupleExpr(_,lst)) = looplist phash lst
	      | loop phash (NondetExpr(_,lst)) = looplist phash lst
	      | loop phash (Dot(_,e,_)) = loop phash e
	      | loop phash (Appl(_,e1,e2)) = loop (loop phash e1) e2
	      | loop phash (TypedPattern(_,e,pat)) =
		  let val h = newVarsHash(substTypePoly phash pat)
		  in mergeHashPoly h phash
		  end
	      | loop phash (ApplPattern(_,_,pat)) = loop phash pat
	      | loop phash (RecordPattern(_,lst)) = looplist phash lst
	      | loop phash (TuplePattern(_,lst)) = looplist phash lst
	      | loop phash (ChoicePattern(_,lst)) = looplist phash lst
	      | loop phash (ModuleInst(_,_,lst)) = looplist phash lst
	      | loop phash _ = phash
	in loop phash expr
	end

    (* Extract static and dynamic declarations from the module's signature *)
    fun getStatDecls (ModuleSig{statparams=stats,...}) = stats
    fun getDynDecls (ModuleSig{dynParam=dyn,...}) = dyn

    (* Instantiates all object definitions with the final types,
       according to the `hash'.

       Assumptions: all instances of `Objects' are wrapped into
       `ObjectInst', and instances of modules - in `ModuleInst'.
       Therefore, if we encounter an `Object' or a similar structure,
       it must be its original definition in some closure list, so we
       update its parent closure with the current one. *)

    fun finalizeExpr options context hash e =
	let val timer = startTimer "finalizeExpr"
	    val _ = lazyVerbDebug options ""
		       (fn()=>("\nfinalizeExpr(...)\n"))
	    val _ = pushFunStackLazy("finalizeExpr", fn()=>pt2string e)
	    (* Searches for the same type of object in the 
	       context and returns it *)
	    (* fun resolveName context obj =
		let fun getBits (TypeConstr _) = TypeConstrOnly
		      | getBits (ModuleClosure _) = ModulesOnly
		      | getBits (TypeClosure _) = TypesOnly
		      | getBits (PType _) = TypesOnly
		      | getBits (StateVar _) = StateVarOnly
		      | getBits _ = ValuesOnly false
		in (case Option.mapPartial(fn n=>findUNameGlobal(getBits obj) context n)
			                  (GetObjectUName obj) of
			SOME x => x
		      | NONE => raise SympBug
			    ("finalizeExpr/resolveName: object didn't resolve:\n  "
			     ^(pt2stringDebug obj)
			     ^"\n\nSearch bits = "^(bits2str (getBits obj))
			     ^"\nContext:\n  "^(pt2stringDebug context)))
		end *)
	    fun subst tp = substType hash tp
	    fun recur context x = ptTransformParent loop context x
	    and loop context x =
		let val _ = pushFunStackLazy("finalizeExpr/loop", fn()=>pt2string x)
		    val res = loop0 context x
		    val _ = popFunStackLazy("finalizeExpr/loop", fn()=>pt2string res)
		in res
		end
	    and doStateId context id = loop context id
	    and loop0 context (QuantifiedVar{name=name,
					     uname=uname,
					     Type=tp}) =
		  QuantifiedVar{name=name,
				uname=uname,
				Type=subst tp}
	      | loop0 context (Object{name=name,
				      uname=uname,
				      Type=t,
				      def=def}) =
		  Object{name=name,
			 uname=uname,
			 Type=subst t,
			 def=loop context def}
	      | loop0 context (StateVar{name=name,
					uname=uname,
					Type=t,
					id=id}) =
		  StateVar{name=name,
			   uname=uname,
			   Type=subst t,
			   id=doStateId context id}
	      | loop0 context (TypeConstr{name=name,
					  uname=uname,
					  Type=t}) =
		  TypeConstr{name=name,
			     uname=uname,
			     Type=subst t}
	      | loop0 context (PatternFormal{name=name,
					     uname=uname,
					     Type= t,
					     extract=eff}) =
		  PatternFormal{name=name,
				uname=uname,
				Type=subst t,
				extract=eff}
	      | loop0 context (DynPatternFormal{name=name,
						uname=uname,
						Type= t,
						value=v,
						extract=eff}) =
		  DynPatternFormal{name=name,
				   uname=uname,
				   Type=subst t,
				   value=Option.map(loop context) v,
				   extract=eff}
	      | loop0 context (RecordField{name=n,
					   Type=tp}) =
		  RecordField{name=n,
			      Type=subst tp}
	      | loop0 context (TypedExpr(p,e,tp)) =
		  TypedExpr(p,loop context e, subst tp)
	      | loop0 context (TypedPattern(p,e,tp)) =
		  TypedPattern(p,loop context e, subst tp)
	      | loop0 context (TypedVars(p,vars,tp)) =
		  TypedVars(p,vars,(case tp of
					NONE => NONE
				      | SOME t => SOME(subst t)))
	      | loop0 context (StaticFormalConst{name=name, uname=uname, Type=tp, value=v}) =
		  StaticFormalConst{name=name, uname=uname, Type=subst tp,
				    value=Option.map(loop context) v}
	      (* | loop0 context (m as ModuleClosure{parent=pc, ...}) = recur pc m *)
	      (* Object instances: we did name resolution, but it
	         cannot be done for names like M.x where M is a
	         module.  So, leave objects alone.  And when the types
	         are instantiated with `ff', the object will be
	         identical to the one in parent closure anyway. *)
	      | loop0 context (oInst as (ObjectInst{obj=obj,subst=ff})) =
		  let val newobj = obj (*  resolveName context obj *)
		  in oInst (* ObjectInst{obj=newobj, subst=ff} *)
		  end
	      (* Module instances: wrap into `ObjectInst' and forget it. *)
	      | loop0 context (ModuleInst(p,m,stats)) =
		  ObjectInst{obj=ModuleInst(p,m,stats),
			     subst=substType hash}
	      | loop0 context e = recur context e
	    val res = (loop context e) before (updateStat timer)
	    val _ = popFunStack("finalizeExpr", "exited normally")
	in 
	    res
	end

    (* Global hash for all named objects, indexed by uname.
       Update it destructively. *)
    val objectHashRef = ref(makeHashDefault(ptEq,pt2string): (ParseTree, ParseTree) Hash)

    fun resetObjectHash() = (objectHashRef := makeHashDefault(ptEq,pt2string))

    fun typeCheck (options: options) judgement = 
	let val verb = verb options
	    val lazyVerbDebug = lazyVerbDebug options
	    (* The object lookup function that will be returned after typechecking *)
	    fun findObject uname = findHash(!objectHashRef, uname)
	    (* Generate a new `findObject' function for an external use *)
	    fun makeFindObject () = 
		let val objHash = !objectHashRef
		in 
		    fn uname => findHash(objHash,uname)
		end
	    val evalExpr = evaluateExpr options findObject
	    (* Hash for newly added objects into the master objectHash.
	       We need this later to strip ObjectInst from them. *)
	    val newObjHash = makeHashDefault(ptEq,pt2string)
	    fun addObject(uname,obj) = 
		(insertHashDestructive(newObjHash,uname,obj);
		 insertHashDestructive(!objectHashRef,uname,obj))
	    (* Same as `addObject', only doesn't register the object in the `newObjHash' *)
	    fun replaceObject(uname,obj) = insertHashDestructive(!objectHashRef,uname,obj)
	    (* Replace all uid's in the parse tree by their objects when possible *)
	    fun expandUids (uid as Uid _) =
		(case findObject uid of
		     SOME x => x
		   | NONE => uid)
	      | expandUids pt = ptTransform expandUids pt		     
	    (* Global hash for type variables *)
	    val tHash = ref(makeHashDefault(op =,Int.toString))
	    (* Instantiates type with both poly and non-poly hash *)
	    fun subst phash hash tp = substType hash (substTypePoly phash tp)
	    (* Same, only tHash is a reference.  Use it to encapsulate 
	       substitution hashes into the ObjectInst closure. 
	       The currect reference to tHash is "captured", and later being 
	       updated to the final one automatically, so no need to go 
	       through the program again just to instantiate the types. *)
	    fun makeSubst phash tHash tp = substType(!tHash)(substTypePoly phash tp)

	    (* Convert a list of typed vars in binding declarations
	       (like quantifiers) into the list of new
	       QuantifiedVar's.  The new vars are added to the global
	       object hash. *)

	    fun typedVars2QVars vars =
		let fun newQV(name,tp) = 
		        let val uid = Uid(newName())
			    val qv = QuantifiedVar{name=name,uname=uid,Type=tp}
			in (addObject(uid,qv); qv)
			end
		    fun typed2vars [] _ = []
		      | typed2vars (v::tl) (SOME tp) =
			  (newQV(v,tp))::(typed2vars tl (SOME tp))
		      | typed2vars (v::tl) NONE =
			  (newQV(v,newTvar()))::(typed2vars tl NONE)
		    fun convertVars [] = []
		      | convertVars ((TypedVars(_,lst,tp))::tl) =
			  (typed2vars lst tp)@(convertVars tl)
		      | convertVars (name::tl) = 
			  (typed2vars [name] NONE)@(convertVars tl)
		in convertVars vars
		end
	    (* Expand all the named types and instantiate parametric types.
	     Since types may involve expressions in module instances, it
	     is mutually recursive with everything else.
	     Returns new type. *)
	    fun expandType context tp =
		let val debug = lazyVerbDebug "expandType"
		    val _ = pushFunStackLazy("expandType", 
					     fn()=>((pt2string tp)
						    ^", context = "^(pt2string context)))
		    fun loopName mode context name args =
			let val tdef = findNameCommon mode context name
			    val _ = debug(fn()=>"  expandType/loopName: "^(pt2string name)
					  ^" to "
					  ^(case tdef of
						SOME t => (pt2string t)
					      | NONE => "NONE")^"\n")
			in 
			 (case tdef of
			      (* Assumption: TypeClosure is already expanded 
			       * sans parameters *)
			      SOME(tp as TypeClosure{params=parms,def=d,
						     recursive=r,parent=pc,...}) => 
			       if (List.length parms) = (List.length args) then
				   if r then TypeInst(dp,args,tp)
				   else instantiateType d parms args
			       else raise TypeError
				   ((pos2string(pos name))
				    ^": type "^(pt2string name)^" requires "
				    ^(Int.toString (List.length parms))
				    ^"parameters, "
				    ^(Int.toString (List.length args))
				    ^" provided")
					(* All the other types are not expanded *)
			    | SOME x => x
			    | NONE => raise TypeError
				((pos2string(pos name))^": Can't find type name: "
				 ^(pt2string name)))
			end
		    (* The main recursive loop.  Takes the current
		       context, the type expression, and the type
		       arguments (already expanded).
		       Returns the expanded type. *)
		    fun loop context x args =
			let val _ = debug(fn()=>"  expandType/loop("^(pt2string x)
				      ^", ["^(ptlist2str "," args)^"])[\n")
			    val res = loop' context x args
			    val _ = debug(fn()=>"  expandType/loop => "^(pt2string res)^"]\n")
			in res
			end
		    and loop' context (Id(p,name)) args =
			  loopName (Global(UserName,TypesOnly)) context (Id(p,name)) args
		      | loop' context (StaticFormalType{value=SOME t, ...}) [] =
			  loop context t []
		      | loop' context (StaticFormalType{uname=un,...}) [] =
			  (case findObject un of
			       SOME t => t
			     | NONE => loopName (Global(UName,TypesOnly)) context un [])
		      | loop' c (Dot(p,m,n)) args = 
			let val phash = makeHashDefault(op =,fn x=>x)
			    val moduleCl = typeCheckModuleExpr true phash c (m, NONE)
			in (case moduleCl of
				ModuleClosure _ =>
				    loopName (Local(UserName,TypesOnly)) moduleCl n args
			      | x => raise TypeError
				 ((pos2string p)
				  ^": this module expression doesn't define any visible names:\n  "
				  ^(pt2string(Dot(p,m,n)))))
			end
		      | loop' c (TypeInst(p,targs,tp)) [] = 
			loop c tp (List.map (fn x=>loop c x []) targs)
		      (* Assumption: TypeClosure is already expanded 
		       * sans parameters - NO LONGER TRUE for non-recursive types *)
		      | loop' c (TypeClosure{name=name,uname=uname,params=params,def=def,
					    recursive=r,parent=pc}) args =
			if (List.length args) = (List.length params) then
			    if r then TypeInst(dp,args,
					       TypeClosure{name=name,
							   uname=uname,
							   params=params,
							   def=def,
							   recursive=r,
							   parent=pc})
			    else loop c (instantiateType def params args) []
			else raise TypeError
			    (" type "^(pt2string name)^" requires "
			     ^(Int.toString (List.length params))
			     ^" parameters, "
			     ^(Int.toString (List.length args))
			     ^" provided (debug info: expandType)")
		      | loop' c (FunType(p,x1,x2)) [] = 
			  FunType(p, loop c x1 [], loop c x2 [])
		      | loop' c (ArrayType(p,x1,x2)) [] =
			  ArrayType(p,loop c x1 [],loop c x2 [])
		      | loop' c (TupleType(p,lst)) [] = 
			  TupleType(p,List.map(fn x => loop c x []) lst)
		      | loop' c (RecordType(p,lst)) [] =
			  let fun conv (RecordField{name=n, Type=t}) =
			            [RecordField{name=n, Type=loop c t []}]
				| conv (TypedVars(p,vars,SOME(t))) =
				    let val tp = loop c t []
				    in map(fn n => RecordField{name=n,Type=tp}) vars
				    end
				| conv (TypedVars(p,vars,NONE)) =
				    raise TypeError
					((pos2string p)
					 ^": untyped fields in a record:\n"
					 ^(ptlist2str ", " vars))
				| conv x = raise SympBug
					("expandType: wrong record field: "
					 ^(pt2stringDebug x))
			  in RecordType(p,List.foldr(op @) [] (map conv lst))
			  end
		      | loop' c (EnumType(p,lst)) [] = 
			  let fun conv (Of(p1,n,t)) = Of(p1,n,loop c t [])
				| conv (t as Id _) = t
				| conv (TypeConstr{name=name,uname=uname,Type=tp}) =
			            TypeConstr{name=name,uname=uname,
					       Type=loop c tp []}
				| conv x = raise SympBug
			             ("expandType/loop/conv: "
				      ^"wrong EnumType element: "
				      ^(pt2stringDebug x))
			  in EnumType(p,List.map conv lst)
			  end
		      | loop' c (Group(p,x)) args = loop c x args
		      | loop' c (Type(_,x)) args = loop c x args
		      | loop' c (Datatype(_,x)) args = loop c x args
		      | loop' c (PType(_,x)) args = loop c x args
		      | loop' _ x [] = x
		      | loop' _ x args = raise SympBug
			  ("expandType: stray args for type "
			   ^(pt2stringDebug x)^"\n"
			   ^"Args: "^(ptlist2strDebug ", " args))
		    val res = loop context tp []
		    val _ = popFunStackLazy("expandType", fn()=>pt2string res)
		in res
		end

    (* Instantiate the module with static and dynamic parameters.
       `context' must be a `ModuleParamClosure' with actual
       parameters, and `m' must be a `ModuleClosure'.  

       Assumption: `m' is already typechecked against the actual
       parameters, and all parameters match.  Also, all parameters are
       `final', that is, fully expanded and ready to go to the final
       module expression `as is'.

       Returns updated `m'. *)

	    and instantiateModule options context (m, newSig) = 
		let val funName = "instantiateModule"
		    val debug = lazyVerbDebug funName
		    val _ = pushFunStackLazy(funName,
					     fn()=>((pt2stringDebug m)
						    ^", newSig="^(ModuleSig2string newSig)
						    ^"\n  in context\n    "
						    ^(pt2stringDebug context)))

		    (* We need to keep all the substitution contexts for the modules above
		       us, we need them to instantiate stateVar `id's properly.

		       The Cxt datatype keeps the subst. context for the current module
                       body, the name of the module, and the parent module's Cxt if there is
                       one. *)

		    datatype Cxt = Mcxt of {name: ParseTree,
					    (* Current context *)
					    cxt: ParseTree list,
					    (* Original context for the current module *)
					    origCxt: ParseTree list,
					    parent: Cxt option}
		    fun sameSig(ModuleSig{statparams=s1, dynParam=d1},
				ModuleSig{statparams=s2, dynParam=d2}) =
			let fun eqOpt _ (NONE, NONE) = true
			      | eqOpt eq (SOME x, SOME y) = eq(x,y)
			      | eqOpt _ _ = false
			    fun eqList eq (l1, l2) =
				(case zipOpt(l1,l2) of
				     NONE => false
				   | SOME pairs => List.all eq pairs)
			    fun eqOptUname(x,y) = eqOpt ptEq (GetObjectUName x, GetObjectUName y)
			in eqOpt(eqList eqOptUname)(s1,s2)
			    andalso
			    eqOpt eqOptUname (d1, d2)
			end
		    val mName = (case GetObjectName m of
				     SOME n => n
				   | NONE => raise SympError
				       ("instantiateModule: unnamed module:\n  "
					^(pt2string m)))
		    val ModuleSig{statparams=statOpt, dynParam=dynOpt} = newSig
		    val cxtList = (case dynOpt of
				       NONE => []
				     | SOME dyn => [dyn])
			         @(case statOpt of
				       NONE => []
				     | SOME lst => lst)
		    val _ = debug(fn()=>"\ninstantiateModule: cxtList = ["
				  ^(ptlist2str ", " cxtList)^"]\n")
		    val cxt = Mcxt{name=mName,
				   cxt=cxtList,
				   origCxt=cxtList,
				   parent=NONE}
		    (* Find replacement for the `obj' in the `lst' *)
		    fun resolveParamLst lst obj =
			let val funName = "instantiateModule/resolveParam"
			    val _ = pushFunStackLazy(funName,
						     fn()=>(pt2stringDebug obj)
						     ^" in context = ["
						     ^(ptlist2str ", " lst)^"]")
			    fun getBits (PType _) = TypesOnly
			      | getBits (StaticFormalType _) = TypesOnly
			      | getBits _ = ValuesOnly true
			    fun getObjectDef(Object{def=def,...}) = def
			      | getObjectDef(StaticFormalConst{value=SOME def, ...}) = def
			      | getObjectDef(x as StaticFormalConst _) = x
			      | getObjectDef(DynPatternFormal{value=SOME def, ...}) = def
			      | getObjectDef(x as DynPatternFormal _) = x
			      | getObjectDef x = raise SympBug
				  ("instantiateModule/resolveParam: not an Object: "
				   ^(pt2stringDebug x))
			    val bits = getBits obj
			    fun findNameOpt name =
				  (case findName bits UName (name,lst) of
				       (x::_) => SOME(case bits of
							  TypesOnly => expandType context x
							| _ => getObjectDef x)
				     | [] => NONE)
			    val res = Option.mapPartial findNameOpt (GetObjectUName obj)
			    val _ = popFunStackLazy(funName,
						    fn()=>(case res of
							       NONE => "NONE"
							     | SOME x => (pt2string x)))
			in res
			end
		    fun resolveParam (Mcxt{cxt=lst,...}) obj = resolveParamLst lst obj
		    (* Descends through the list of declarations that may depend on each
		       other, and instantiates them preserving the scoping.
		       Returns the new list of declarations and the new instantiation context. *)
		    fun descend ff (Mcxt{name=n,cxt=[],origCxt=ocxt,parent=p}) lst acc =
			  ((List.rev lst)@acc,
			   Mcxt{name=n,cxt=[],origCxt=ocxt,parent=p})
		      | descend ff cxt [] acc = (acc, cxt)
		      | descend ff cxt (x::lst) acc =
			(* If `x' is type parameter, don't do anything to it *)
			  let val newX=(case x of
					    StaticFormalType _ => x
					  | _ => loop ff cxt x)
			      val newCxt = cxt (* removeObject x cxt *)
			  in descend ff newCxt lst (newX::acc)
			  end
		    and recur ff cxt x = ptTransform (loop ff cxt) x
		    and loop ff cxt x =
			let fun pr(x as ModuleClosure _)  = pt2stringDebug x
			      | pr x = pt2string x
			    val _ = pushFunStackLazy("instantiateModule/loop", fn()=>pr x)
			    val res = loop0 ff cxt x
			    val _ = popFunStackLazy("instantiateModule/loop", fn()=>pr res)
			in res
			end
		    (* These objects we instantiate *)
		    and loop0 ff cxt (e as StaticFormalType{name=name,uname=un,...}) = 
			  (case resolveParam cxt e of
			       NONE => e
			     | SOME x => x)
		      (* | loop0 ff cxt (ObjectInst{obj=e as StaticFormalConst{name=n,uname=un,
									    Type=tp, value=v},
						 subst=ff1}) =
			  (case resolveParam cxt e of
			       NONE => ObjectInst{obj=StaticFormalConst
						   {name=n,uname=un, Type=loop (ff o ff1) cxt tp
						    value=Option.map(loop (ff o ff1) cxt) v},
						   subst=ff1}
			     | SOME x => x) *)
		      (* Instantiate id string of StateVar *)
		      (* | loop0 ff cxt (ObjectInst{obj=e as StateVar x, subst=ff1}) =
			  ObjectInst{obj=loop (ff o ff1) cxt e, subst=ff1} *)
		      (* And this might be a dynamic parameter *)
		      | loop0 ff cxt (e as DynPatternFormal{name=n,uname=un,Type=tp,
							    extract=eff,value=v}) =
			  (case resolveParam cxt e of
			       NONE => let val newtp = loop ff cxt (substType (!tHash) (ff tp))
				       in DynPatternFormal{name=n,uname=un,extract=eff,
							   Type=newtp,
							   value=Option.map(loop ff cxt) v}
				       end
			     | SOME x => x)
		      | loop0 ff cxt (e as StaticFormalConst{name=n,uname=un,Type=tp,value=v}) =
			  (case resolveParam cxt e of
			       NONE => let val newtp = loop ff cxt (substType (!tHash) (ff tp))
				       in StaticFormalConst{name=n,uname=un,
							   Type=newtp,
							   value=Option.map(loop ff cxt) v}
				       end
			     | SOME x => x)
		      | loop0 ff cxt (ObjectInst{obj=obj,subst=ff1}) =
			  ObjectInst{obj=loop (ff o ff1) cxt obj, subst=ff1}
		      (* Now deal with incomplete types: instantiate them all *)
		      | loop0 ff cxt (TypedVars(p,vars,to)) =
			  TypedVars(p,vars,Option.map(fn tp=>loop ff cxt 
						        (substType (!tHash) (ff tp))) to)
		      | loop0 ff cxt (TypedExpr(p,e,tp)) =
			  TypedExpr(p,loop ff cxt e,loop ff cxt (substType (!tHash) (ff tp)))
		      | loop0 ff cxt (TypedPattern(p,e,tp)) =
			  TypedPattern(p,loop ff cxt e,loop ff cxt (substType (!tHash) (ff tp)))
		      | loop0 ff cxt (QuantifiedVar{name=n,uname=un,Type=tp}) =
			  QuantifiedVar{name=n,uname=un,
					Type=loop ff cxt (substType (!tHash) (ff tp))}
		      | loop0 ff cxt (StateVar{name=n,uname=un,id=id,Type=tp}) =
			  StateVar{name=n,uname=un,id=loop ff cxt id,
				   Type=loop ff cxt (substType (!tHash) (ff tp))}
		      | loop0 ff cxt (Object{name=n,uname=un,def=def,Type=tp}) =
			  Object{name=n,uname=un,def=loop ff cxt def,
				 Type=loop ff cxt (substType (!tHash) (ff tp))}
		      | loop0 ff cxt (PatternFormal{name=n,uname=un,Type=tp,extract=eff}) =
			  PatternFormal{name=n,uname=un,extract=eff,
					Type=loop ff cxt (substType (!tHash) (ff tp))}
		      | loop0 ff cxt (TypeConstr{name=n,uname=un,Type=tp}) =
			  TypeConstr{name=n,uname=un,Type=loop ff cxt (substType (!tHash) (ff tp))}
		      | loop0 ff cxt (RecordField{name=n,Type=tp}) =
			  RecordField{name=n,Type=loop ff cxt (substType (!tHash) (ff tp))}
		      (* Replace the module's signature with the instantiated one *)
		      | loop0 ff cxt (m as ModuleClosure{name=name,
							 uname=uname,
							 Sig=Sig,
							 closure=cl,
							 def=def,
							 parent=pc}) =
			  if sameSig(Sig, newSig) then
			      ModuleClosure{name=name,
					    uname=uname,
					    Sig=newSig,
					    closure=List.map (loop ff cxt) cl,
					    def=Option.map (loop ff cxt) def,
					    parent=pc}
			  else recur ff cxt m
		      | loop0 ff cxt x = recur ff cxt x
		    val res = 
			(case loop (fn x=>x) cxt m of
			     ModuleClosure{name=name,
					   uname=un,
					   Sig=Sig,
					   closure=cl,
					   def=def, ...} =>
			         ModuleClosure{name=name,
					       uname=un,
					       Sig=Sig,
					       closure=cl,
					       def=def,
					       parent=context}
			   | x => raise SympBug
				 ("instantiateModule: not ModuleClosure:\n  "
				  ^(pt2stringDebug x)))
		    val _ = popFunStackLazy(funName,
					    fn()=>((pt2string res)^"\n context = "
						   ^(pt2string context)))
		in res
		end

	    (* Returns SOME(new substitution hash) or NONE, if mismatch.
	     * Unified variables must be non-polymorphic (all polymorphic ones have
	     * to be instantiated beforehand, otherwise they are matched verbatim). 
	     * Types have to be the actual type expressions, not "Objects",
	     * except for recursive types, and those are unified verbatim.
	     * Also all definitions of (parameterized) named types must be 
	     * instantiated, or they will not be expanded.

	     When one variable is unified with another, the one with a larger
	     index gets assigned the smaller one.  This is to ensure proper
	     selection of poly vars in polymorphic functions.
	     
	     Some optimization for speed: use destructive hash insert. *)

	    and unifyTypes (tp1,tp2) =
		(* Hash the variables' substitutions in a 'hash'.
		 * Variables are hashed by numbers (int).
		 * Important: variables that are _not_ replaced must not
		 * be in the hash. *)
		let val timer = startTimer "unifyTypes"
		    val hash = ref(makeHashDefault(op =,Int.toString))
		    (* Takes the two types and actually unifies them.
		     * Returns bool (success or failure).
		     * Assumption: the arguments are already instantiated by
		     *  "substType". *)
		    fun loop t1 t2 =
			let val debug = lazyVerbDebug "unifyTypes"
			    val _ = pushFunStackLazy
				        ("unifyTypes",
					 fn()=>((pt2stringDebug t1)^", "^(pt2stringDebug t2)))
			    val res = loop0 t1 t2
			    fun printBool true = "true"
			      | printBool false = "false"
			    val _ = popFunStackLazy("unifyTypes", fn ()=>printBool res)
			in res
			end
		    and loop0(TvarN n1) (TvarN n2) =
			if n1 = n2 then true 
			else if n1 > n2 then 
			    (hash := insertHashDestructive(!hash,n1,TvarN n2); true)
			     else (hash := insertHashDestructive(!hash,n2,TvarN n1); true)
		      | loop0(TvarN n1) tp = 
			if List.exists(fn n=>n=n1)(collectVarsN tp) then false
			else (hash := insertHashDestructive(!hash,n1,tp); true)
		      | loop0 tp (TvarN n2) = 
			if List.exists(fn n=>n=n2)(collectVarsN tp) then false
			else (hash := insertHashDestructive(!hash,n2,tp); true)
		      (* Polymorphic vars are compared verbatim *)
		      | loop0 (Tvar(_,n1))(Tvar(_,n2)) = n1 = n2
		      | loop0(BoolType _) (BoolType _) = true
		      | loop0 (NatType _) (NatType _) = true
		      | loop0 (IntType _) (IntType _) = true
		      | loop0(RangeType(_,x1,y1))(RangeType(_,x2,y2)) = 
			 ptEq(x1,x2) andalso ptEq(y1,y2)
		      (* For now, match only exactly the same types, do not unify,
		       * say, range with int.  Subtyping should be implemented later,
		       * this is a temporary requirement to preserve soundness *)
		   (* | loop0 (NatType _) (IntType _) = true
		      | loop0 (NatType _) (RangeType(_,_,_)) = true
		      | loop0 (IntType _) (NatType _) = true
		      | loop0 (IntType _) (RangeType(_,_,_)) = true
		      | loop0(RangeType(_,_,_)) (NatType _) = true
		      | loop0(RangeType(_,_,_)) (IntType _) = true *)
		      (* Now the type constructors *)
		      | loop0(FunType(_,x1,x2))(FunType(_,y1,y2)) =
			 (loop(substType(!hash) x1)(substType(!hash) y1))
			 andalso (loop(substType(!hash) x2)(substType(!hash) y2))
		      | loop0(TupleType(_,lst1))(TupleType(_,lst2)) =
			 (looplist lst1 lst2)
		      (* Translate ArrayType into FunType *)
		      | loop0(ArrayType(p,x1,x2)) tp = loop(FunType(p,x1,x2)) tp
		      | loop0 tp (ArrayType(p,x1,x2)) = loop tp (FunType(p,x1,x2))
		      (* Records and datatypes may have fields and constructs
		       *  out of order, handle them with special care *)
		      | loop0(RecordType(_,lst1))(RecordType(_,lst2)) =
			 matchRecord lst1 lst2
		      | loop0(EnumType(_,lst1))(EnumType(_,lst2)) =
			 matchEnum lst1 lst2
		      (* Instantiated types and datatypes are compared for 
		       * verbatim equivalence.
		       * Assumption: TypeClosure contains definitions that do not
		       * depend on its parent closure (that is, they are expanded) *)
		      | loop0(TypeInst(_,lst1,name1))(TypeInst(_,lst2,name2)) =
			 (eqUNames(name1, name2)) andalso (looplist lst1 lst2)
		      | loop0(TypeClosure{name=n1,uname=un1,params=a1,...})
			 (TypeClosure{name=n2,uname=un2,params=a2,...}) =
			 ((case a1 of
			       [] => ()
			     | _ => raise TypeError
				   ("Datatype "^(pt2string n1)
				    ^" requires parameters:\n  ("
				    ^(ptlist2str "," a1)^")"));
			       (case a2 of
				    [] => ()
				  | _ => raise TypeError
					("Datatype "^(pt2string n2)
					 ^" requires parameters:\n  ("
					 ^(ptlist2str "," a2)^")"));
				    ptEq(un1,un2))
		      (* Now TypeClosures might not always be expanded,
		         expand them here if no params and non-recursive *)
		      | loop0(TypeClosure{def=tp1,params=[],recursive=false,...}) tp2 =
			 loop tp1 tp2
		      | loop0 tp1 (TypeClosure{def=tp2,params=[],recursive=false,...}) =
			 loop tp1 tp2
		      | loop0(PType(_,name1))(PType(_,name2)) = ptEq(name1,name2)
		      | loop0(StaticFormalType{uname=un1,...})(StaticFormalType{uname=un2,...}) =
			 ptEq(un1,un2)
			 (* Type Constructors may have Uid instead of the base type;
			    we may have to expand it *)
		      | loop0(u1 as Uid _) (u2 as Uid _) = ptEq(u1,u2)
		      | loop0(u1 as Uid _) tp = 
			 (case findObject u1 of
			      SOME t1 =>
				loop (expandType (TopClosure [])
				      (substType(!hash)(substTypePoly (newVarsHash t1) t1))) tp
				(* loop (substType(!hash)(substTypePoly (newVarsHash t1) t1)) tp *)
			    | NONE => false)
		      | loop0 tp (u2 as Uid _) = 
			 (case findObject u2 of
			      SOME t2 =>
				loop tp (expandType (TopClosure [])
					 (substType(!hash)(substTypePoly (newVarsHash t2) t2)))
				(* loop tp (substType(!hash)(substTypePoly (newVarsHash t2) t2)) *)
			    | NONE => false)
		      | loop0 _ _ = false
		    and looplist [] [] = true
		      | looplist(hd1::tl1)(hd2::tl2) =
			(loop(substType(!hash) hd1)(substType(!hash) hd2))
			andalso looplist tl1 tl2
		      | looplist _ _ = false
	    (* Make sure all the fields in one list appear in the other 
	     * with the right types, and vice versa.
	     * Each list must be the list of TypedVars or RecordField
	     * objects. 
	     * Partial records also contain a type variable field at the end
	     * that matches any list of fields.  These extra otherwise 
	     * unmatched fields are collected into lists for each of the
	     * partial records, and the end variable is updated accordingly. *)
		    and matchRecord lst1 lst2 =
			let fun matchOne [] name tp = NONE
		      (* If a field name is a poly var, then we search for a matching poly
		         var at the end of the record. *)
			      | matchOne [Tvar(_,n1)] (Tvar(_,n2)) _ = 
			         if n1 = n2 then SOME [] else NONE
			      | matchOne [Tvar(_,n1)] _ _ = NONE
			      | matchOne (_::lst) (Tvar x) tp = matchOne lst (Tvar x) tp
			      | matchOne [TvarN _] name tp =
				 SOME [RecordField{name=name,Type=tp}]
			      | matchOne ((TypedVars(_,vars,SOME(t)))::tl) name tp =
				 if List.exists(fn x =>ptEq(x,name)) vars then
				     (case loop(substType(!hash) tp)(substType(!hash) t)
					  of true => SOME []
					| false => NONE)
				 else matchOne tl name tp
			      | matchOne ((TypedVars(p,vars,NONE))::_) _ _ =
				 raise SympBug((pos2string p)
					       ^"Untyped fields in a record type: "
					       ^(pt2string(TypedVars(p,vars,NONE))))
			      | matchOne ((RecordField{name=n,
						       Type=t,...})::tl) name tp =
				 if ptEq(name,n) then
				     (case loop(substType(!hash) tp)(substType(!hash) t)
					  of true => SOME []
					| false => NONE)
				 else matchOne tl name tp
			      | matchOne (x::_) _ _ =
				 raise SympBug("unifyTypes/matchRecord/matchOne: "
					       ^"Found a bad record field: "
					       ^(pt2stringDebug x))
			    (* Check that a list of names with a type appear in
			     * the list of record fields *)
			    fun sublist [] _ _ acc = SOME acc
			      | sublist(hd::tl) lst tp acc =
				(case (matchOne lst hd tp) of
				     NONE => NONE
				   | SOME l => sublist tl lst tp (l@acc))
			    (* Check that one record is a subrecord of the other, and collect all
			     the fields that matched only against the end variable in `acc'. *)
			    fun subrecord [] lst acc = SOME acc
			      | subrecord [TvarN _] lst acc = SOME acc
			      (* A poly var has to match the other's record poly var, and no
			       other fields should match that variable *)
			      | subrecord [Tvar x] lst acc = 
				(case matchOne lst (Tvar x) (Tvar x) of
				     NONE => NONE
				   | SOME l => (case acc of
						    [] => SOME []
						  | _ => NONE))
			      | subrecord((TypedVars(_,vars,SOME(t)))::tl) lst acc =
				(case (sublist vars lst t []) of
				     NONE => NONE
				   | SOME l => subrecord tl lst (l@acc))
			      | subrecord ((TypedVars(p,vars,NONE))::_) _ _ =
				raise SympBug((pos2string p)
					      ^"Untyped fields in a record type: "
					      ^(pt2string(TypedVars(p,vars,NONE))))
			      | subrecord((RecordField{name=n,
						       Type=t,...})::tl) lst acc =
				(case (matchOne lst n t) of
				     NONE => NONE
				   | SOME l => subrecord tl lst (l@acc))
			      | subrecord(x::_) _ _ =
				raise SympBug("unifyTypes/matchRecord/subrecord: "
					      ^"Found a bad record field: "
					      ^(pt2stringDebug x))
			    (* Check whether the record is partial *)
			    fun PartialVar [TvarN n] = SOME n
			      | PartialVar [] = NONE
			      | PartialVar (_::tl) = PartialVar tl
			    (* Terribly inefficient, but can we do better?.. *)
			    val L2 = subrecord lst1 lst2 []
			    val L1 = subrecord lst2 lst1 []
			in
			    if (isSome L1) andalso (isSome L2) then
			      (case (PartialVar lst1, PartialVar lst2) of
				   (NONE, NONE) => true
				 | (SOME n1, NONE) =>
				       (hash:=insertHashDestructive(!hash,n1,
								    RecordType(dp,valOf L1));
					true)
				 | (NONE, SOME n2) =>
				       (hash:=insertHashDestructive(!hash,n2,
								    RecordType(dp,valOf L2));
					true)
				 | (SOME n1, SOME n2) =>
				     let val V=newTvar()
				     in 
					 (hash:=insertHashDestructive(!hash,n1,
					      RecordType(dp,(valOf L1)@[V]));
					  hash:=insertHashDestructive(!hash,n2,
					      RecordType(dp,(valOf L2)@[V]));
					  true)
				     end)
			    else false
			end
		    (* And the same mess with Datatypes *)
		    and matchEnum lst1 lst2 =
			let fun matchOne [] name tp = false
			      | matchOne ((Id(p,n))::tl) name NONE =
			         ptEq(Id(p,n),name) orelse matchOne tl name NONE
			      (* Here tp = SOME(_) *)
			      | matchOne ((Id(p,n))::tl) name tp = 
				 if ptEq(Id(p,n),name) then false
				 else matchOne tl name tp
			      | matchOne ((Of(_,n,t))::tl) name tp =
				 if ptEq(n,name) then
				     case tp of
					 NONE => false
				       | SOME x => 
					     loop(substType(!hash) t)
					     (substType(!hash) x)
				 else matchOne tl name tp
			      | matchOne ((TypeConstr{name=n,
						      Type=t,...})::tl) name tp =
				 if ptEq(n,name) then
				     case (t,tp) of
					 (EnumType _,NONE) => true
				       | (uid as Uid _, NONE) => true
				       | (FunType(_,x1,_),SOME(x2)) => 
					     loop(substType(!hash) x1)
					         (substType(!hash) x2)
				       | _ => false
				 else matchOne tl name tp
			      | matchOne (x::_) _ _ =
				 raise SympBug("unifyTypes/matchEnum/matchOne: "
					       ^"Found a bad type constructor: "
					       ^(pt2stringDebug x))
			    (* Check that one list is a sublist of the other *)
			    fun sublist [] lst = true
			      | sublist ((Id(p,n))::tl) lst =
				(matchOne lst (Id(p,n)) NONE)
				andalso (sublist tl lst)
			      | sublist ((Of(_,n,t))::tl) lst =
				(matchOne lst n (SOME t))
				andalso (sublist tl lst)
			      | sublist ((TypeConstr{name=n,
						     Type=t,...})::tl) lst =
				(case t of
				     EnumType _ => (matchOne lst n NONE)
				   | FunType(_,tp,_) => (matchOne lst n (SOME tp))
				   | uid as Uid _ => 
				       (case findObject uid of
					    SOME tp => (matchOne lst n NONE)
					  | NONE => false)
				   | x => raise SympBug
					 ("unifyTypes/matchEnum/sublist: "
					  ^"Found a bad type of a type "
					  ^"constructor: "^(pt2stringDebug x)))
				     andalso (sublist tl lst)
			      | sublist (x::_) _ =
				 raise SympBug("unifyTypes/matchEnum/sublist: "
					       ^"Found a bad type constructor: "
					       ^(pt2stringDebug x))
			in (sublist lst1 lst2) andalso (sublist lst2 lst1)
			end
		in 
		    (if loop tp1 tp2 then SOME(!hash) else NONE)
			 before (updateStat timer)
		end

	    (* Unify the two types for the hash and return SOME(hash)
	       with updated hash or NONE.  The types must be expanded
	       and instantiated with the latest hashes. *)

	    and coerce phash hash (t1,t2) =
		let val timer = startTimer "coerce"
		    (* Print both pure and instantiated types
		       - they must be the same *)
		    fun argsFun() = 
			  ((pt2stringDebug t1)
			   ^"=<"^(pt2string (substType hash t1))^">,\n       "
			   ^(pt2string t2)^"=<"^(pt2string (substType hash t2))^">")
		    val _ = pushFunStackLazy("coerce", argsFun)
		    val res = (case unifyTypes (t1,t2) of
				   NONE => NONE
				 | SOME(h) => SOME(mergeHash h hash))
		    fun resFun() = 
			(case res of
			     NONE => "No Match"
			   | SOME(h) => ((pt2string t1)
					 ^"=<"^(pt2string(substType h t1))^">,\n          "
					 ^(pt2string t2)^"=<"^(pt2string(substType h t2))^">"))
		in (popFunStackLazy("coerce", resFun);
		    updateStat timer;
		    res)
		end

(**********************************************************************)

            and tcObject bits context (obj, tp) =
		let fun argsFun() = (bits2str bits)^", "^(pt2string(TypedExpr(dp, obj, tp)))
		    val _ = pushFunStackLazy("tcObject", argsFun)
		    val _ = (case (bits,obj) of
				 (ValuesOnly false,StateVar _) => raise TypeError
				   ((pos2string(pos obj))
				    ^": State variable is not allowed in constant expression:\n  "
				    ^(pt2string obj))
			       | _ => ())
		    (* Assumption: object type is already expanded,
		       we only instantiate it with the current tHash *)
		    val objtp0 = (case GetObjectType obj of
				      SOME tp => tp
				    | NONE => raise SympBug
					("tcName: object doesn't have a type:\n  "
					 ^(pt2stringDebug obj)))
		    val objtp = substType (!tHash) objtp0
		    val _ = lazyVerbDebug "tcName"
			      (fn()=>("  tcName: "
				      ^(pt2string(TypedExpr(dp,Id(dp,"objtp"),objtp)))
				      ^"\n"))
		    (* Instantiate polymorphic vars.  Note that there
		       are no user's polymorphic vars left in the
		       object, so we do not need outer phash.
		       Also, if the type is a Uid, we want to expand it here to see
		       if we need to instantiate poly vars.  *)
		    val objtp = (case objtp of
				     Uid _ => (case findObject objtp of
						   SOME t => t
						 | NONE => raise SympBug
						    ("tcName: No type object with such Uid: "
						     ^(pt2string objtp)))
				   | _ => objtp)
		    val phash1 = newVarsHash objtp
		    val tp2 = expandType context (substTypePoly phash1 objtp)
		    val _ = lazyVerbDebug "tcName"
			      (fn()=>("  tcName: "
				      ^(pt2string(TypedExpr(dp,Id(dp,"tp2"), tp2)))
				      ^"\n"))
		    (* And unify the types *)
		    val newhash = (case coerce phash1 (!tHash) (tp2,tp) of
				       NONE => raise TypeError
					   ((pos2string (pos obj))
					    ^": Type mismatch: "^(pt2string obj)
					    ^" has type\n  "
					    ^(pt2string objtp)
					    ^"\nExpected:\n  "
					    ^(pt2string tp))
				     | SOME h => (tHash:=h; h))
		    val newtp = substType newhash tp
		    val _= (lazyVerbDebug "tcNameDebug" (fn()=>("  newhash("^(pt2string tp)
							   ^")=<"
							   ^(pt2string newtp)
							   ^">\n  phash=["
							   ^(polyhash2str "," phash1)
							   ^"]\n\n")))
		in
		    (ObjectInst{obj=obj,subst=makeSubst phash1 tHash},
		     newtp)
		    before
		    (popFunStackLazy("tcObject", fn()=>pt2string(TypedExpr(dp,obj,newtp))))
		end

    (* Typecheck name - an Id, unary/binary operator, whatever happens
       to be the name.  `bits' tells whether to include state vars
       into expression and/or whether to look only for type constructors.
       * Assumptions: The type 'tp' *is* expanded and
         substituted with the latest phash and tHash. *)
	    and tcName bits context (name,tp) =
		let val timer = startTimer "tcName"
		    fun argsFun() = (bits2str bits)^", "^(pt2string(TypedExpr(dp, name, tp)))
		    (* ^", <"^(pt2string(substType (!tHash) tp))^">" *)
		    val _ = pushFunStackLazy("tcName", argsFun)
		    val _ = lazyVerbDebug "tcNameDebug"
			(fn()=>"Context:\n "^(pt2stringDebug context)^"\n")
		    val ff = (case bits of
				  StateVarOnly => findStateVar
				| ValuesOnly _ => findNameGlobal(ValuesOnly true)
				| _ => findNameGlobal bits)
		    val objStr = (case bits of
				      StateVarOnly => "state variable"
				    | TypeConstrOnly => "type constructor"
				    | ValuesOnly true => "constant or variable"
				    | ValuesOnly false => "constant"
				    | _ => "object")
		    val obj = (case ff context name of
				   NONE => raise TypeError
				       ((pos2string (pos name))
					^": Can't find "^objStr
					^": "^(pt2string name))
				 | SOME obj => obj)
		    val _ = lazyVerbDebug "tcNameDebug"
			     (fn()=>"tcName: found "^(pt2string name)
			      ^": "^(pt2stringDebug obj)^"\n")
		    val (objInst, newtp) = tcObject bits context (obj, tp)
		in
		    (* Attention: other functions rely on this format: typeCheckPattern *)
		    (objInst, newtp)
		     before 
		     (popFunStackLazy("tcName", fn()=>pt2string(TypedExpr(dp,obj,newtp)));
		      updateStat timer)
		end

	    and tcTypeConstr x = tcName TypeConstrOnly x
	    and tcStateVar x = tc StateVarOnly x

	    (* The recursive type checker proper.
	     * Returns a pair (<expression with variables typed>, type).
	     * Assumptions: The type 'tp' *is* expanded and
               substituted with the latest tHash. *)

	    and tc bits phash context (e,tp) = 
		let fun argsFun() = 
		     ((pt2string(TypedExpr(dp,e,tp)))
		      ^",\n context:\n\n"^(pt2string context))
		    (* ^",  tHash=["^(hash2str "," (!tHash))
		     ^"]\n  phash=["^(polyhash2str "," phash)^"]" *)
		    val _ = pushFunStackLazy("tc", argsFun)
		    val _ = lazyVerbDebug "tcDebug" (fn()=>"tc("^(pt2stringDebug e)^","
						^(pt2stringDebug tp)^"),[\n  tHash=["
						^(hash2str "," (!tHash))
						^"]\n  phash=["
						^(polyhash2str "," phash)
						^"]\n\n")
		    val (ee,tt) = tc0 bits phash context (e,tp)
		    fun resFun() = (pt2string(TypedExpr(dp,ee,tt)))
		    (* ^",\n  tHash=["^(hash2str "," (!tHash))
		     ^"]\n  phash=["^(polyhash2str "," phash)^"]" *)
		    val _ = popFunStackLazy("tc", resFun)
		    val _ = lazyVerbDebug "tcDebug" (fn()=>"tc("^(pt2stringDebug e)^","
						^(pt2stringDebug tp)^") => ("
						^(pt2stringDebug ee)^","
						^(pt2stringDebug tt)^"),\n  tHash=["
						^(hash2str "," (!tHash))
						^"]\n  phash=["
						^(polyhash2str "," phash)
						^"] ]\n\n")
		in (ee,tt)
		end
	    and tc0 bits phash context (Dot(p,e,name),tp) =
		let val _ = 
		      if not(isName name) then
			  raise SympBug((pos2string p)^": not a name after Dot:\n  "
					^(pt2stringDebug name))
		      else ()
		    (* Partial record type for the field *)
		    val prType = RecordType(dp,[RecordField{name=name,
							    Type=tp},
						newTvar()])
		    (* Try to typecheck `e' as a module first.  If the user actually meant
		       a record that ambiguously define the same field, then prefer module
		       variant, since the interpretation as a record can be forced with pattern
		       matching. *)
		    val hash_bak = !tHash
		    val moduleRes = 
			 (case bits of
			      StateVarOnly => NONE
			    | _ => SOME(typeCheckModuleExpr true phash context (e, NONE)))
			  handle TypeError _ => (tHash := hash_bak;
						 lazyVerbDebug "typeCheckModuleExpr"
						  (fn()=>"\ntypeCheckModuleExpr aborted\n");
						 NONE)
		    val moduleRes2 =
			(case moduleRes of
			     SOME(ModuleClosure x) =>
				 SOME(tc bits phash (ModuleClosure x) (name,tp))
			   | _ => NONE)
		    (*  val hash_bak = !tHash *)

		    (* If the module interpretation fails, try record *)
		    val recordRes = 
			  (case moduleRes of
			       SOME _ => NONE
			     | NONE => SOME(tc bits phash context (e,prType)))
			  (* handle TypeError _ => (tHash := hash_bak;
						 lazyVerbDebug "tc" (fn()=>"\ntc aborted\n");
						 NONE) *)
		    val res =
			(case (recordRes,moduleRes2) of
			     (NONE,NONE) => raise TypeError
				 ((pos2string p)
				  ^": The expression is neither module nor record that defines "
				  ^(pt2string name)^":\n  "
				  ^(pt2string e))
			   | (NONE,SOME x) => x
			   | (SOME (re,t),NONE) => (ExtractRecord(name,re),substType(!tHash) tp)
			   | (SOME x,SOME y) => raise TypeError
				 ((pos2string p)
				  ^": Ambiguous expression, both module and record define "
				  ^(pt2string name)^" in:\n  "
				  ^(pt2string e)))
		in
		    res
		end
	      (* Function application *)
	      | tc0 bits phash context (Appl(p,f,x),tp) =
		  let val tx = newTvar()
		      val (ff,t1,t2) = (case tc bits phash context (f,FunType(dp,tx,tp)) of
					  (ff,FunType(_,t1,t2)) => (ff,t1,t2)
					| (ff,ArrayType(_,t1,t2)) => (ff,t1,t2)
					| (ff,t) => raise SympBug
					      ("tc("^(pt2stringDebug(Appl(p,f,x)))
					       ^": not a function type: "
					       ^(pt2stringDebug t)))
		      val paramBits =
			  (case bits of
			       StateVarOnly => ValuesOnly true
			     | _ => bits)
		      val (xx,_) = tc paramBits phash context (x,t1)
		  in (Appl(p,ff,xx),substType (!tHash) t2)
		  end
	      | tc0 bits phash context (TypedExpr(p,e,tp1), tp2) =
		 let val t1 = substType (!tHash) (substTypePoly phash (expandType context tp1))
		 in (case coerce phash (!tHash) (t1,tp2) of
			 NONE => raise TypeError
			     ((pos2string p)
			      ^": type mismatch in coercion.\n  Context type: "
			      ^(pt2string t1)^"  Coerced to: "^(pt2string tp2)^"\n")
		       | SOME h => (tHash := h; tc bits phash context (e,substType h t1)))
		 end

	      (* All the other cases are not allowed when bits = StateVarOnly *)
	      | tc0 bits phash context (Number(p,n),tp) =
		 let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(Number(p,n))))
				| _ => ())
		 in (case coerce phash (!tHash) (IntType dp, tp) of
			 NONE => raise TypeError
			     ((pos2string p)
			      ^": type mismatch for "^(pt2string(Number(p,n)))
			      ^": int\n    Context type: "^(pt2string tp)^"\n")
		       | SOME h => (tHash := h; (Number(p,n),IntType dp)))
		 end

	      | tc0 bits phash context (ModuleInst(p,m,parms), tp) =
		 raise TypeError((pos2string p)^": module cannot be an expression:\n  "
				 ^(pt2string(ModuleInst(p,m,parms))))

	      | tc0 bits phash context (TupleExpr(p,lst),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(TupleExpr(p,lst))))
				| _ => ())
		     val tlist = List.map(fn _=>newTvar()) lst
		      val t0 = TupleType(dp,tlist)
		      val newhash = 
			  (case coerce phash (!tHash) (t0,tp) of
			       NONE => raise TypeError
					((pos2string p)
					 ^": Type mismatch in tuple:\n  "
					 ^(pt2string(TupleExpr(p,lst)))
					 ^"\n Expected types:\n  "
					 ^(pt2string(substType(!tHash) tp)))
			     | SOME x => (tHash := x; x))
		      fun loop [] [] = []
			| loop (e::elst) (t::tlst) =
			    (tc bits phash context (e,substType (!tHash) t))::(loop elst tlst)
			| loop _ _ = raise SympBug
			    ("tc("^(pt2stringDebug(TupleExpr(p,lst)))
			     ^": number of entries don't match")
		      val reslist = loop lst tlist
		  in (TupleExpr(p,List.map(fn (x,_) => x) reslist),
		      TupleType(dp,List.map(fn (_,y) => substType (!tHash) y) reslist))
		  end
	      | tc0 bits phash context (IfExpr(p,lst,last),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(IfExpr(p,lst,last))))
				| _ => ())
		      fun ff (CondExpr(p1,c,e)) =
		        let val (cc,_) = tc bits phash context (c,BoolType dp)
			    val (ee,_) = tc bits phash context (e,substType (!tHash) tp)
			in CondExpr(p1,cc,ee)
			end
			| ff x = raise SympBug("tc: IfExpr has wrong case: "
					       ^(pt2stringDebug x))
		      val lst1 = List.map ff lst
		      val (last1,tp1) = tc bits phash context (last,substType (!tHash) tp)
		  in (IfExpr(p,lst1,last1),tp1)
		  end
	      | tc0 bits phash context (LetExpr(p,decls,body),tp) =
		  (* first, typecheck every declaration, accumulating the 
		   * list of decl's and creating a temp. context for
		   * later ones.  Then typecheck the body and create the
		   * final LetClosure. *)
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(LetExpr(p,decls,body))))
				| _ => ())
		      fun loop clist [] = clist
			| loop clist (d::tl) =
		            let val cxt = LetClosure{locals=clist,
						     parent=context,
						     body=Fake}
				val lst = typeCheckDecl bits phash cxt d
			    in
				loop (lst@clist) tl
			    end
		      val locals = loop [] decls
		      val (bd,tp1) = tc bits phash (LetClosure{locals=locals,
							       parent=context,
							       body=Fake})
			               (body,substType (!tHash) tp)
		  in
		      (LetClosure{locals=locals,
				  parent=context,
				  body=bd},
		       tp1)
		  end
	      | tc0 bits phash context (CaseExpr(p,expr,lst),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(CaseExpr(p,expr,lst))))
				| _ => ())
		      val (sel,stp) = tc bits phash context (expr,newTvar())
		      (* The case selector is given a unique name in
		         the case statement and is referenced to only by its
			 name in the choices. *)
		      val unameSel =
			  let val u = Uid(newName())
			      val pf = PatternFormal{name=u,
					   uname=u,
					   Type=stp,
					   extract=(fn x=>x)}
			  in (addObject(u,pf); pf)
			  end
		      fun ff (ChoiceExpr(p,pat,e)) =
			  let val ph = newVarsExprHash(newVarsExprHash phash pat) e
			      fun convPatName(PatternFormal{name=name,
							    uname=un,
							    Type=tp,
							    extract=ff}) =
			            Object{name=name,
					   uname=un,
					   Type=substTypePoly ph tp,
					   def=ff unameSel}
				| convPatName x = raise SympBug
				    ("typeCheckExpr/CaseExpr: "
				     ^"not PatternFormal :"
				     ^(pt2stringDebug x))
		       (* Don't instantiate stp with ph: new type vars don't appear in it *)
			      val (newpat,plist,ptp) = 
				typeCheckPattern ph context (pat,substType (!tHash) stp)
			      val plist = List.map convPatName plist
			      val cl = ChoiceClosure{pattern=newpat,
						     uname=unameSel,
						     names=plist,
						     body=Fake,
						     parent=context}
			      val (ee,tp1) = tc bits ph cl (e,substType (!tHash) tp)
			  in ChoiceClosure{pattern=newpat,
					   uname=unameSel,
					   names=plist,
					   body=ee,
					   parent=context}
			  end
			| ff x = raise SympBug("typeCheckExpr/CaseExpr: "
					       ^"not a ChoiceClosure: "
					       ^(pt2stringDebug x))
		      val reslst = List.map ff lst
		  in (CaseExpr(p,sel,reslst), substType (!tHash) tp)
		  end
	      | tc0 bits phash context (Fn(p,lst),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(Fn(p,lst))))
				| _ => ())
		      (* Convert fn expression to fn with single
		         argument and case statement, unless it's
		         already in this form. *)
		      val (argvar,argtp,body0) = 
			    (case lst of
				 [ChoiceExpr(_,Id(p,name),expr)] =>
				     (Id(p,name),newTvar(),expr)
			       | lst => 
				     let val (argvar,argtp) = (Id(dp,newName()),newTvar())
				     in (argvar,argtp,
					 CaseExpr(p,TypedExpr(dp,argvar,argtp),lst))
				     end)
		      val restp = newTvar()
		      (* Do coercion here (instead of later) to catch quick 
		       * context errors when context type doesn't match
		       * 'a -> 'b. *)
		      val newhash = coerce phash (!tHash) (FunType(dp,argtp,restp), tp)
		      val _ = (case newhash of
				   NONE => raise TypeError
				       ((pos2string p)
					^": Type mismatch in `fn' expression: "
					^(pt2string(Fn(p,lst)))
					^"\nDeduced type:\n  "
					^(pt2string (FunType(dp,argtp,restp)))
					^"\nExpected types:\n  "
					^(type2str (!tHash) tp))
				 | SOME x => (tHash := x; x))
		      val argobjUid = Uid(newName())
		      val argobj=PatternFormal{name=argvar,
					       uname=argobjUid,
					       Type=substType (!tHash) argtp,
					       extract=(fn x=>x)}
		      val _ = addObject(argobjUid,argobj)
		      val cl = FunClosure{name=NONE,
					  formals=argobj,
					  parent=context,
					  body=Fake}
		      val (body,btp) = tc bits phash cl (body0, substType (!tHash) restp)
		  in (FunClosure{name=NONE,
				 formals=argobj,
				 parent=context,
				 body=body},
		      FunType(dp,substType (!tHash) argtp,btp))
		  end
	      | tc0 bits phash context (RecordExpr(p,fields),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(RecordExpr(p,fields))))
				| _ => ())
		      fun getFieldName (RecordAsst(_,name,_)) = name
			| getFieldName x =
		            raise SympBug("typeCheckExpr/getFieldName"
					  ^": wrong Record field: "
					  ^(pt2stringDebug x))
		      fun getFieldExpr (RecordAsst(_,_,e)) = e
			| getFieldExpr x =
			    raise SympBug("typeCheckExpr/getFieldExpr"
					  ^": wrong Record field: "
					  ^(pt2stringDebug x))
		      val fieldNames = (List.map getFieldName fields)
		      val types = List.map(fn n=>newTvar()) fieldNames
		      fun makeTypeFields [] [] = []
			| makeTypeFields (n::names) (t::types) =
			    (RecordField{name=n,Type=t})
			    ::(makeTypeFields names types)
			| makeTypeFields _ _ =
			    raise SympBug("typeCheckExpr/makeTypeFields: "
					  ^"This can't happen")
		      val tp1 = RecordType(dp, makeTypeFields fieldNames types)
			(* We don't care about reordering fields, 
			 * it's taken care by unification *)
		      val _ = (case coerce phash (!tHash) (tp1,tp) of
				     NONE => raise TypeError
				       ((pos2string p)
					^": Type mismatch in record:\n  "
					^"deduced type: "
					^(pt2string tp1)^"\n  "
					^"has to be:\n  "
					^(pt2string(substType (!tHash) tp)))
				   | SOME x => tHash:=x)
		      fun loop [] [] = []
			| loop (f::fields) (t::types) =
			    let val (e,_) = 
				  tc bits phash context (getFieldExpr f, substType (!tHash) t)
			    in RecordAsst(pos f, getFieldName f, e)
			    end
			   ::(loop fields types)
			| loop _ _ = raise SympBug("typeCheckExpr/RecordExpr: "
						   ^"This can't happen")
		      val newfields = loop fields types
		  in (RecordExpr(p,newfields), substType (!tHash) tp1)
		  end
	      | tc0 bits phash context (NondetExpr(p,lst), tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(NondetExpr(p,lst))))
				| _ => ())
		      fun loop ([],tp) = ([],tp)
			| loop ((e::tl),tp) =
			    let val (x,t) = tc bits phash context (e, tp)
				val (lst,t1) = loop (tl,t)
			    in (x::lst,t1)
			    end
		      val (newlst,newtp) = loop (lst,tp)
		  in (NondetExpr(p,newlst), newtp)
		  end
	      | tc0 bits phash context (WithExpr(p,expr,lst),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(WithExpr(p,expr,lst))))
				| _ => ())
		      val (e,tp1) = tc bits phash context (expr,tp)
		      val ff = (fn (WithAsst(_,name,_)) => (name,newTvar())
		                 | x => raise SympBug
				   ("TypeCheckExpr/WithExpr: not a WithAsst: "
				    ^(pt2stringDebug x)))
		      val assts = List.map ff lst
		      val asstTypes = List.map(fn (_,t) => t) assts
		      val asstFields = List.map(fn (n,t)=>RecordField{name=n,Type=t}) assts
		      (* Construct partial record from assignments *)
		      val asstRecord = RecordType(dp,asstFields@[newTvar()])
		      val newhash = 
			  (case coerce phash (!tHash) (tp1,asstRecord) of
			       NONE => raise TypeError
				   ((pos2string p)
				    ^": Type mismatch in WITH statement: "
				    ^(pt2string expr)
				    ^"\nDeduced type:\n  "^(pt2string tp1)
				    ^"\nExpected type:\n  "^(pt2string asstRecord))
			     | SOME x => (tHash := x; x))
		      (* Typecheck the field assignments *)
		      fun loop [] [] = []
			| loop ((WithAsst(p,name,e))::tl) (t::types) =
			   let val (e1,_) = tc bits phash context (e,substType (!tHash) t)
			   in (WithAsst(p,name,e1))::(loop tl types)
			   end
			| loop (x::_) (_::_) = raise SympBug
			   ("TypeCheckExpr/WithExpr/loop: not a WithAsst: "
			    ^(pt2stringDebug x))
			| loop _ _ = raise SympBug
			   ("TypeCheckExpr/WithExpr/loop: this can't happen")
		  in (WithExpr(p,e,loop lst asstTypes), substType (!tHash) tp1)
		  end
	      | tc0 bits phash context (ChooseExpr(p,pat,expr),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string(ChooseExpr(p,pat,expr))))
				| _ => ())
		      val ph = newVarsExprHash(newVarsExprHash phash pat) expr
		      val (newpat,plist,ptp) =
			typeCheckPattern ph context (pat,substTypePoly ph tp)
		      val tmpCC = ChooseClosure{pattern=newpat,
						Type=ptp,
						names=plist,
						body=Fake,
						parent=context}
		      val (e,_) = tc bits ph tmpCC (expr,BoolType dp)
		      val newptp = substType (!tHash) ptp
		  in (ChooseClosure{pattern=newpat,
				    Type=newptp,
				    names=plist,
				    body=e,
				    parent=context},
		      newptp)
		  end
	      | tc0 bits phash context (x as Forall(p,vars,expr),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string x))
				| _ => ())
		      (* fun newQV(name,tp) = 
			  let val uid = Uid(newName())
			      val qv = QuantifiedVar{name=name,uname=uid,Type=tp}
			  in (addObject(uid,qv); qv)
			  end
		      fun typed2vars [] _ = []
			| typed2vars (v::tl) (SOME tp) =
		            (newQV(v,tp))::(typed2vars tl (SOME tp))
			| typed2vars (v::tl) NONE =
		            (newQV(v,newTvar()))::(typed2vars tl NONE)
		      fun convertVars [] = []
			| convertVars ((TypedVars(_,lst,tp))::tl) =
			    (typed2vars lst tp)@(convertVars tl)
			| convertVars (name::tl) = 
			    (typed2vars [name] NONE)@(convertVars tl) *)
		      (* First, make sure the context type is boolean *)
		      val _ = (case coerce phash (!tHash) (tp,BoolType dp) of
				   NONE => raise TypeError
				       ((pos2string p)
					^": Type mismatch: FORALL has type "
					^"bool, but the expected type is:\n  "
					^(pt2string(substType (!tHash) tp)))
				 | SOME x => (tHash := x))
		      (* Compute new phash *)
		      fun newVarsList ph [] = ph
			| newVarsList ph (e::lst) = newVarsExprHash(newVarsList ph lst) e
		      val ph = newVarsList phash (expr::vars)
		      (* Convert variables into QuantifiedVar{...} list *)
		      (* val newvars = convertVars vars *)
		      val newvars = typedVars2QVars vars
		      val tmpCC = ForallClosure{names=newvars,
						body=Fake,
						parent=context}
		      val (e,_) = tc bits ph tmpCC (expr,BoolType dp)
		  in (ForallClosure{names=newvars,
				    body=e,
				    parent=context},
		      BoolType dp)
		  end
	      | tc0 bits phash context (x as Exists(p,vars,expr),tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string x))
				| _ => ())
		      (* fun newQV(name,tp) = 
			  let val uid = Uid(newName())
			      val qv = QuantifiedVar{name=name,uname=uid,Type=tp}
			  in (addObject(uid,qv); qv)
			  end
		      fun typed2vars [] _ = []
			| typed2vars (v::tl) (SOME tp) =
		            (newQV(v,tp))::(typed2vars tl (SOME tp))
			| typed2vars (v::tl) NONE =
		            (newQV(v,newTvar()))::(typed2vars tl NONE)
		      fun convertVars [] = []
			| convertVars ((TypedVars(_,lst,tp))::tl) =
			    (typed2vars lst tp)@(convertVars tl)
			| convertVars (name::tl) = 
			    (typed2vars [name] NONE)@(convertVars tl) *)
		      (* First, make sure the context type is boolean *)
		      val _ = (case coerce phash (!tHash) (tp,BoolType dp) of
				   NONE => raise TypeError
				       ((pos2string p)
					^": Type mismatch: EXISTS has type "
					^"bool, but the expected type is:\n  "
					^(pt2string(substType (!tHash) tp)))
				 | SOME x => (tHash := x))
		      (* Compute new phash *)
		      fun newVarsList ph [] = ph
			| newVarsList ph (e::lst) = newVarsExprHash(newVarsList ph lst) e
		      val ph = newVarsList phash (expr::vars)
		      (* Convert variables into QuantifiedVar{...} list *)
		      (* val newvars = convertVars vars *)
		      val newvars = typedVars2QVars vars
		      val tmpCC = ExistsClosure{names=newvars,
						body=Fake,
						parent=context}
		      val (e,_) = tc bits ph tmpCC (expr,BoolType dp)
		  in (ExistsClosure{names=newvars,
				    body=e,
				    parent=context},
		      BoolType dp)
		  end
	      | tc0 bits phash context (x as Mu(p, v, expr), tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string x))
				| _ => ())
		      fun newQV(name,tp) = 
			  let val uid = Uid(newName())
			      val qv = QuantifiedVar{name=name,uname=uid,Type=tp}
			  in (addObject(uid,qv); qv)
			  end
		      (* First, make sure the context type is boolean *)
		      val _ = (case coerce phash (!tHash) (tp,BoolType dp) of
				   NONE => raise TypeError
				       ((pos2string p)
					^": Type mismatch: MU has type "
					^"bool, but the expected type is:\n  "
					^(pt2string(substType (!tHash) tp)))
				 | SOME x => (tHash := x))
		      (* Compute new phash *)
		      fun newVarsList ph [] = ph
			| newVarsList ph (e::lst) = newVarsExprHash(newVarsList ph lst) e
		      val ph = newVarsList phash [v,expr]
		      (* Convert the variable into QuantifiedVar{...} list *)
		      val qv = newQV(v, BoolType dp)
		      val tmpCC = MuClosure{name=qv,
					    body=Fake,
					    parent=context}
		      val (e,_) = tc bits ph tmpCC (expr,BoolType dp)
		  in (MuClosure{name=qv,
				body=e,
				parent=context},
		      BoolType dp)
		  end
	      | tc0 bits phash context (x as Nu(p, v, expr), tp) =
		  let val _ = (case bits of
				  StateVarOnly => raise TypeError
				      ((pos2string p)^": not a state variable: "
				       ^(pt2string x))
				| _ => ())
		      fun newQV(name,tp) = 
			  let val uid = Uid(newName())
			      val qv = QuantifiedVar{name=name,uname=uid,Type=tp}
			  in (addObject(uid,qv); qv)
			  end
		      (* First, make sure the context type is boolean *)
		      val _ = (case coerce phash (!tHash) (tp,BoolType dp) of
				   NONE => raise TypeError
				       ((pos2string p)
					^": Type mismatch: NU has type "
					^"bool, but the expected type is:\n  "
					^(pt2string(substType (!tHash) tp)))
				 | SOME x => (tHash := x))
		      (* Compute new phash *)
		      fun newVarsList ph [] = ph
			| newVarsList ph (e::lst) = newVarsExprHash(newVarsList ph lst) e
		      val ph = newVarsList phash [v,expr]
		      (* Convert the variable into QuantifiedVar{...} list *)
		      val qv = newQV(v, BoolType dp)
		      val tmpCC = NuClosure{name=qv,
					    body=Fake,
					    parent=context}
		      val (e,_) = tc bits ph tmpCC (expr,BoolType dp)
		  in (NuClosure{name=qv,
				body=e,
				parent=context},
		      BoolType dp)
		  end
	      | tc0 bits phash context (e,tp) = 
		  if isName e then tcName bits context (e,tp)
		  else if isObject e then tcObject bits context (e, tp)
		  else raise SympBug("typeCheckExpr: This expression is "
				     ^"not implemented: "
				     ^(pt2stringDebug e))

    (* Determines the type of the pattern and builds the list of 
       formal parameters in the form of PatternFormal{...}.

       Returns a triple (pattern, paramList, type), where `pattern' is
       almost the same as the argument, only names of type
       constructors are replaced by TypeConstr.

       Assumption: `tp' is expanded and instantiated (as in `tc'). *)

            and typeCheckPattern phash context (pattern,tp) =
		let val lazyVerbDebug = lazyVerbDebug "typeCheckPattern"
		    val _ = pushFunStackLazy("typeCheckPattern",
					     fn()=>(pt2string pattern)^", "^(pt2string tp))
		    fun loop ff (pat,tp) =
			let val _ = pushFunStackLazy("tcPattern/loop",
						     fn()=>((pt2string pat)^", "^(pt2string tp)))
			    val (newpat,plist,t) = loop0 ff (pat,tp)
			    fun resFun() = "["^(ptlist2str "," plist)^"], "^(pt2string t)
			    val _ = popFunStackLazy("tcPattern/loop", resFun)
			in (newpat,plist,t)
			end
						  
		    and loop0 ff (Underscore p, tp) = (Underscore p, [], tp)
		      | loop0 ff (TypedPattern(p,pat,tp1),tp2) =
			let val t1 = substType (!tHash) 
			              (substTypePoly phash (expandType context tp1))
			    val _ = (case coerce phash (!tHash) (t1,tp2) of
					 NONE => raise TypeError
					     ((pos2string p)
					      ^": Type mismatch in pattern type coercion:\n"
					      ^"Deduced type:\n  "
					      ^(pt2string tp2)^"\n  coerced to: "
					      ^(pt2string t1))
				       | SOME x => tHash:=x)
			in loop ff (pat,substType (!tHash) t1)
			end
		      | loop0 ff (TuplePattern(p,lst),tp) =
			let val tplst = List.map(fn x => newTvar()) lst
			    val tp1 = TupleType(dp,tplst)
			    val _ = (case coerce phash (!tHash) (tp1,tp) of
					 NONE => raise TypeError
					     ((pos2string p)
					      ^": Type mismatch in pattern:\n  "
					      ^"deduced type:\n  "
					      ^(pt2string tp1)^"\n"
					      ^"Expected types:\n  "
					      ^(pt2string tp))
				       | SOME x => tHash:=x)
			    fun count _ [] [] = []
			      | count n (pp::ptl) (tp::ttl) =
				let val (newpat,lst,tp) =
				       loop (fn x => ExtractTuple(n,ff x))
				            (pp,substType (!tHash) tp)
				    val reslst = count (n+1) ptl ttl
				in (newpat,lst,tp)::reslst
				end
			      | count n l1 l2 = raise SympBug
				("typeCheckPattern/count (tuple)"
				 ^": wrong list lengths")
			    val reslst = count 0 lst tplst
			in
			    (TuplePattern(p,List.map(fn (pat,_,_) =>pat) reslst),
			     (List.foldr(fn ((_,lst1,_),(lst2)) => lst1@lst2)
			                [] reslst),
			     TupleType(dp,(List.map(fn (_,_,tp) => substType (!tHash) tp) reslst)))
			end
		      | loop0 ff (RecordPattern(p,lst),tp) =
			let fun getField (RecordAsst(_,name,_)) = SOME name
			      | getField (Ellipsis p) = SOME(newTvar())
			      | getField (Tvar _) = NONE
			      | getField (TvarN _) = NONE
			      | getField (TypedVars(_,[name],_)) = SOME name
			      | getField (RecordField{name=name,...}) = SOME name
			      | getField x = raise SympBug("typeCheckPattern/getField"
							   ^": wrong Record field: "
							   ^(pt2stringDebug x))
			    fun getType (TypedVars(_,_,SOME tp)) = SOME tp
			      | getType (RecordField{Type=tp,...}) = SOME tp
			      | getType (Tvar _) = NONE
			      | getType (TvarN _) = NONE
			      | getType x = raise SympBug("typeCheckPattern/getType"
							  ^": wrong Record field: "
							  ^(pt2stringDebug x))
			    fun getPat (RecordAsst(_,_,pat)) = pat
			      | getPat x = raise SympBug("typeCheckPattern/getPat"
							 ^": wrong Record field: "
							 ^(pt2stringDebug x))
			    fun makeField (Tvar(p,v)) = Tvar(p,v)
			      | makeField (TvarN n) = TvarN n
			      | makeField name = RecordField{name=name,
							     Type=newTvar()}
			    val fields = (List.mapPartial getField lst)
			    val tp1 = RecordType(dp,List.map makeField fields)
			    val _ = (case coerce phash (!tHash) (tp1,tp) of
					 NONE => raise TypeError
					     ((pos2string p)
					      ^": Type mismatch in pattern:\n  "
					      ^"Deduced types:\n  "
					      ^(pt2string tp1)^"\n"
					      ^"Expected types:\n  "
					      ^(pt2string tp))
				       | SOME x => tHash:=x)
			    val tp2 = substType (!tHash) tp1
			    val _ = lazyVerbDebug
				(fn()=>"  tp="^(pt2string tp)
				 ^",\n  tp1="^(pt2string tp1)^"\n")
			    (* The fields might be reordered now.
			     * Recompute the names and types,
			     * but add only those that occur in the pattern. *)
			    fun isNameInPattern name =
				List.exists(fn (Tvar _) => false
		                             | n => ptEq(name,n)) fields
			    fun isInPattern f =
				(case getField f of
				     NONE => false
				   | SOME n => isNameInPattern n)
			    (* Leave only fields relevant to the pattern *)
			    val PatternLst = 
				(case tp2 of
				     RecordType(_,lst1) =>
					 List.filter isInPattern lst1
				   | x => raise SympBug("typeCheckPattern "
							^"(PatternLst): "
							^"tp1 is not record: "
							^(pt2stringDebug x)))
			    val fields = List.mapPartial getField PatternLst
			    val types = List.mapPartial getType PatternLst
			    fun getPatByName f lst =
				(case List.find(fn x => (case getField x of
							     NONE => false
							   | SOME f1 => ptEq(f,f1)))
				               lst of
				     SOME(y) => getPat(y)
				   | NONE => raise SympBug
					 ("typeCheckPattern/getPatByName: "
					  ^"no such field: "
					  ^(pt2stringDebug f)^" in "
					  ^(pt2stringDebug(RecordType(dp,lst)))))
			    val patlst = List.map(fn x => getPatByName x lst) fields
			    (* Now we have field, types, and patlst all in the same order *)
			    fun count (pp::ptl) (nn::ntl) (tp::ttl) =
				let val (newpat,lst,tp) =
				      loop(fn x => ExtractRecord(nn,ff x))
				          (pp,substType (!tHash) tp)
				    val reslst = count ptl ntl ttl
				in (RecordAsst(dp,nn,newpat),lst,tp)::reslst
				end
			      | count [] [] [] = []
			      | count _ l1 l2 = raise SympBug
				("typeCheckPattern/count (record)"
				 ^": wrong list lengths")
			    val reslst = count patlst fields types
			in
			    (RecordPattern(p,List.map(fn(pat,_,_)=>pat) reslst),
			     (List.foldr(fn ((_,lst1,_),(lst2)) => lst1@lst2) [] reslst),
			     substType (!tHash) tp2)
			end
		      | loop0 ff (ApplPattern(p,c,pat),tp) = 
			let val ptp = newTvar()
			    val (c1,ctp) = tcTypeConstr context (c,FunType(dp,ptp,tp))
			    val (ptp1,tp1) = (case ctp of
						  FunType(_,t1,t2) => (t1,t2)
						| ArrayType(_,t1,t2) => (t1,t2)
						| x => raise SympBug
						      ("typeCheckPattern/Applpattern: "
						       ^"not FunType: "
						       ^(pt2stringDebug x)))
			    val (subpat,lst,ptp2) = loop(fn x=>(ExtractAppl(c1,ff x))) (pat,ptp1)
			in (ApplPattern(p,c1,subpat), lst, substType (!tHash) tp1)
			end
		      | loop0 ff (pat as (Number(p,n)),tp) = 
			(case coerce phash (!tHash) (tp,IntType dp) of
			     NONE => raise TypeError
				 ((pos2string p)
				  ^"Type mismatch in pattern "
				  ^(pt2string(Number(p,n)))
				  ^": int, selector type is one of:\n  "
				  ^(pt2string tp))
			   | SOME x => (tHash:=x;(pat, [],IntType dp)))
		      | loop0 ff (pat as (True p),tp) = 
			(case coerce phash (!tHash) (tp,BoolType dp) of
			     NONE => raise TypeError
				 ((pos2string p)
				  ^"Type mismatch in pattern "
				  ^(pt2string(True p))
				  ^": bool, selector type is one of:\n  "
				  ^(pt2string tp))
			   | SOME x => (tHash:=x; (pat, [],BoolType dp)))
		      | loop0 ff (pat as (False p),tp) = 
			(case coerce phash (!tHash) (tp,BoolType dp) of
			     NONE => raise TypeError
				 ((pos2string p)
				  ^"Type mismatch in pattern "
				  ^(pt2string(False p))
				  ^": bool, selector type is one of:\n  "
				  ^(pt2string tp))
			   | SOME x => (tHash:=x; (pat, [],BoolType dp)))
		      | loop0 ff (ChoicePattern(p,lst),tp) =
			raise SympBug("Sorry, choice patterns are "
				      ^"not implemented yet.")
		      | loop0 ff (AsPattern(p,id,pat),tp) =
			let val uid = Uid(newName())
			    val pf = PatternFormal{name=id,
						   uname=uid,
						   Type=tp,
						   extract=ff}
			    val _ = addObject(uid,pf)
			    val (subpat,lst,ptp) = loop ff (pat,tp)
			in (AsPattern(p,id,subpat),pf::lst,ptp)
			end
		      (* The last case is when a pattern is a name, in
		         a broad sense.  One case is that the name is
		         a datatype constructor, and if it is the
		         case, it takes precedence.  Otherwise it is
		         treated as a bound variable. *)

		      | loop0 ff (name,tp) =
			if isName name then
			   let val uid = Uid(newName())
			       val pf = PatternFormal{name=name,
						      uname=uid,
						      Type=tp,
						      extract=ff}
			       val _ = addObject(uid,pf)
			       (* Backup the tHash, we need to restore
			          it if the search for a type constructor fails *)
			       val hash = !tHash
			       (* Remember, tcName returnes ObjectInst!! *)
			       val (c,tt) = tcName(ValuesOnly false) context (name,tp)
				             handle TypeError _ => (Fake,Fake)
			   in (case c of
				   ObjectInst{obj=TypeConstr _,...} => (c,[],tt)
				 | _ => (tHash := hash; (pf,[pf],tp)))
			   end 
			else raise SympBug("typeCheckPattern/loop: "
					   ^"not a pattern: "
					   ^(pt2stringDebug name))
		    val res as (newpat,plist,t) = loop (fn x => x) (pattern,tp)
		    fun resFun() = "["^(ptlist2str "," plist)^"], "^(pt2string t)
		    val _ = popFunStackLazy("typeCheckPattern", resFun)
		in
		    res
		end

	    (* Returns a new ModuleClosure or other module expression.
	       If `requireParams' is true, require that parameterized
	       modules be fully instantiated. 

	       `statsOpt' is an optional list of formal static
	       parameters carried over from the named module
	       declaration somewhere above. *)

	    and typeCheckModuleExpr requireParams phash context (m, msigOpt) =
		let val funName="typeCheckModuleExpr"
		    val debug = lazyVerbDebug "typeCheckModuleExprDebug"
		    val _ = pushFunStackLazy(funName, fn()=>pt2string m)
		    (* val Sig = ModuleSig{statparams=NONE} *)
		    (* Finds a module closure by name *)
		    fun tcModuleName phash context name =
			let val _ = pushFunStackLazy("tcModuleName", fn()=>pt2string name)
			    val res =
				if isName name then 
				    (case findNameGlobal ModulesOnly context name of
					 SOME x => x
				       | NONE => raise TypeError
					     ((pos2string(pos name))
					      ^": can't find module named "
					      ^(pt2string name)))
				else raise TypeError
				    ((pos2string(pos name))
				     ^": not a module expression:\n  "
				     ^(pt2string name))
			    val _ = popFunStackLazy
				("tcModuleName",
				 fn()=>(case res of
					    ModuleClosure{name=SOME name,...} =>
						"ModuleClosure{"^(pt2string name)^"}"
					  | ModuleClosure{name=NONE,...} =>
						"ModuleClosure{<unnamed>}"
					  | x => "NOT module closure: "^(pt2string x)))
			in res
			end
		    (* Check if the module's static params are already instantiated *)
		    fun areStaticInstantiated(ModuleClosure{Sig=ModuleSig{statparams=stats,...},
							    ...}) =
			let fun ff (StaticFormalConst{value=SOME _,...}) = true
			      | ff _ = false
			in
			    case stats of
				NONE => true
			      | SOME lst => List.all ff lst
			end
		      | areStaticInstantiated x = raise SympBug
			 ("areStaticInstantiated: not a ModuleClosure:\n  "
			  ^(pt2stringDebug x))
		    (* Check if the module's dynamic params are already instantiated *)
		    fun areDynamicInstantiated(ModuleClosure{Sig=ModuleSig{dynParam=dyn,...},
							     ...}) =
			let fun ff (DynPatternFormal{value=SOME _,...}) = true
			      | ff _ = false
			in
			    case dyn of
				NONE => true
			      | SOME x => ff x
			end
		      | areDynamicInstantiated x = raise SympBug
			 ("areDynamicInstantiated: not a ModuleClosure:\n  "
			  ^(pt2stringDebug x))
		    (* Check if the module is fully instantiated *)
		    fun isInstantiated m =
			 areStaticInstantiated m andalso areDynamicInstantiated m
		    (* Typechecks module instance (with or without parameters),
		       and returns a pair (ModuleClosure _, statParam list),
		       where the module closure is not yet instantiated with parameters.
		       Important: it *must* be a ModuleClosure, nothing else. *)
		    fun tcModuleInst phash context (m as ModuleInst(p,name,stats)) =
			let (* Resolve the module name and get the pieces *)
			    val _ = pushFunStackLazy("tcModuleInst", fn()=>(pt2string m))
			    val mc = tcModuleName phash context name
			    val Sig =
			      (case mc of
				   ModuleClosure{Sig=s,...} => s
				 | x => raise SympBug
				     ("typeCheckModuleExpr/tcModuleInst: not ModuleClosure:\n  "
				      ^(pt2stringDebug x)))
			    val statDecls = 
				  (case getStatDecls Sig of
				       SOME lst => lst
				     | NONE => raise TypeError
					   ((pos2string p)
					    ^": module doesn't need static parameters in:\n  "
					    ^(pt2string(ModuleInst(p,name,stats)))))
			    (* Typecheck static parameters *)
			    val slen = List.length stats
			    val dlen = List.length statDecls
			    val _ = if slen <> dlen then
				      raise TypeError
					  ((pos2string p)
					   ^": wrong number of static parameters: "
					   ^(Int.toString slen)^" provided, "
					   ^(Int.toString dlen)^" required in\n  "
					   ^(pt2string m))
				    else ()
			    fun mpc lst = LetClosure{locals=lst,
						     body=Fake,
						     parent=context}
			    fun descend [] [] acc = acc
			      | descend (sp::plst) (sd::dlst) acc =
				 let val sp1 = typeCheckStatParam phash (mpc acc) (sp,sd)
				 in descend plst dlst (sp1::acc)
				 end
			      | descend _ _ _ = raise SympBug
				 ("typeCheckModuleExpr/tcModuleInst: this can't happen")
			    val newStats = descend stats statDecls []
			    val _ = popFunStackLazy("tcModuleInst",
						    fn()=>((pt2string mc)^", ["
							   ^(ptlist2str "," newStats)^"]"))
			in (mc, SOME newStats)
			end
		      | tcModuleInst phash context m =
			let val _ = pushFunStackLazy("tcModuleInst", fn()=>(pt2string m))
			    val mc = tcModuleName phash context m
			    val (Sig, name) =
			      (case mc of
				   ModuleClosure{Sig=s, name=name,...} => (s, name)
				 | x => raise SympBug
				     ("typeCheckModuleExpr/tcModuleInst: not ModuleClosure:\n  "
				      ^(pt2stringDebug x)))
			    val statDeclOpt = getStatDecls Sig
			    val requireParams = requireParams andalso 
				  (case name of
				       NONE => true
				     | SOME n => not(ptEq(n, Self dp)))
				  andalso not(areStaticInstantiated m)
			    val _ =
				 (case statDeclOpt of
				      NONE => ()
				    | SOME lst => 
					  if requireParams then raise TypeError
					      ("Module `"^(pt2string m)
					       ^"' requires static parameters:\n  ["
					       ^(ptlist2str ", " lst)^"]")
					  else ())
			    fun stats2str NONE = "<none>"
			      | stats2str (SOME l) = "["^(ptlist2str "," l)^"]"
			    val _ = popFunStackLazy("tcModuleInst",
						    fn()=>((pt2string mc)^", "
							   ^(stats2str statDeclOpt)))
			in (mc, statDeclOpt)
			end
		    (* Convert variables into QuantifiedVar{...} list *)
		    fun newQV(name,tp) = 
			let val uid = Uid(newName())
			    val qv = QuantifiedVar{name=name,uname=uid,Type=tp}
			in (addObject(uid,qv); qv)
			end
		    fun vars2qv [] _ = []
		      | vars2qv (v::vars) ot =
			(newQV(v,(case ot of
				      NONE => newTvar()
				    | SOME t => t)))
			::(vars2qv vars ot)
		    fun typed2qv ph [] = []
		      | typed2qv ph ((TypedVars(p,vars,tp))::lst) =
			let val ot = Option.map
			    (fn t=>(substType (!tHash) 
				    (substTypePoly ph 
				     (expandType context t)))) tp
			in (vars2qv vars ot)@(typed2qv ph lst)
			end
		      | typed2qv ph (v::lst) = (vars2qv [v] NONE)@(typed2qv ph lst)
		    (* The main recursive procedure.
		       The main arguments: (moduleExpr, newModuleSig) *)
		    fun loop phash context (x, msig) =
			let fun args() = (pt2string x)^", "^(ModuleSig2string msig)
			    val _ = pushFunStackLazy("tcModuleExpr", args)
			    val res = loop0 phash context (x, msig)
			    val _ = popFunStackLazy("tcModuleExpr", fn()=>pt2string res)
			in res
			end
		    and loop0 phash context (BeginEnd(p,lst), msig) =
		        let val uname = Uid(newName())
			    fun mClosure name lst = 
			          ModuleClosure{name=name,
						uname=uname,
						Sig=msig,
						closure=lst,
						def=NONE,
						parent=context}
			    fun tmpClosure lst =
				  LetClosure{locals=lst,
					     body=Fake,
					     parent=context}
			    fun descend [] acc = acc
			      | descend (d::lst) acc =
				let val cl = tmpClosure((mClosure (SOME(Self dp)) acc)::acc)
				    val dlst = typeCheckDecl(ValuesOnly false) phash cl d
				in descend lst (dlst@acc)
				end
			    val res = mClosure NONE (descend lst [])
			in (addObject(uname,res); res)
			end
		      | loop0 phash context (Sync2(p,m1,m2), msig) =
			let val mc1 = loop phash context (m1, msig)
			    val mc2 = loop phash context (m2, msig)
			in Sync2(p,mc1,mc2)
			end
		      | loop0 phash context (Async2(p,m1,m2), msig) =
			let val mc1 = loop phash context (m1, msig)
			    val mc2 = loop phash context (m2, msig)
			in Async2(p,mc1,mc2)
			end
		      | loop0 phash context (Sync(p,vars,m), msig) =
			let (* Compute new phash *)
			    fun newVarsList ph [] = ph
			      | newVarsList ph (e::lst) = newVarsExprHash(newVarsList ph lst) e
			    val ph = newVarsList phash (m::vars)
			    val qvars = typed2qv ph vars
			    fun cl m = SyncClosure{names=qvars,
						   body=m,
						   parent=context}
			in cl (loop ph (cl Fake) (m, msig))
			end
		      | loop0 phash context (Async(p,vars,m), msig) =
			let (* Compute new phash *)
			    fun newVarsList ph [] = ph
			      | newVarsList ph (e::lst) = newVarsExprHash(newVarsList ph lst) e
			    val ph = newVarsList phash (m::vars)
			    val qvars = typed2qv ph vars
			    fun cl m = AsyncClosure{names=qvars,
						    body=m,
						    parent=context}
			in cl (loop ph (cl Fake) (m, msig))
			end
		      | loop0 phash context (me as Appl(p,m,dyn), msig) =
			let val (mc,statsOpt) = tcModuleInst phash context m
			    val stats = (case statsOpt of
					     NONE => []
					   | SOME l => l)
			    val (name,uname,Sig,cl,def,pc) = 
			      (case mc of
				   ModuleClosure{name=name,
						 uname=uname,
						 Sig=Sig,
						 closure=cl,
						 def=def,
						 parent=pc,...} => (name,uname,Sig,cl,def,pc)
				 | x => raise SympBug
				     ("typeCheckModuleExpr/Appl: not ModuleClosure:\n  "
				      ^(pt2stringDebug x)))
			    fun getDyn(ModuleSig{dynParam=SOME
						 (dpf as DynPatternFormal{name=dn,
									  Type=tp,...}),
						 ...}) = (dpf, dn,tp)
			      | getDyn _ = raise TypeError
				     ((pos2string p)
				      ^": module does not require dynamic parameter in:\n  "
				      ^(pt2string me))
			    val (dpf, dn, dyntp0) = getDyn Sig
			    val _ = if areDynamicInstantiated mc then
				     raise TypeError
					 ((pos2string p)
					  ^": module's dynamic parameters are already "
					  ^"instantiated in:\n  "
					  ^(pt2string me))
				    else ()
			    fun mpc lst = LetClosure{locals=lst,
						     body=Fake,
						     parent=context}
			    val staticMPC = mpc stats
			    val dyntp1 = (substType (!tHash) 
					  (substTypePoly phash
					   (expandType staticMPC dyntp0)))
			    val (dyn1,dyntp) = tc (ValuesOnly true) phash staticMPC (dyn,dyntp1)
			    (* Extract all PatternFormal decl's from
			       the original MPC and instantiate them
			       with dyn1 *)
			    fun instPatternFormal(DynPatternFormal{name=n,
								   uname=un,
								   Type=t,
								   extract=ff, ...}) =
				  SOME(DynPatternFormal{name=n,
							uname=un,
							Type=(substType (!tHash) 
							      (substTypePoly phash
							       (expandType staticMPC t))),
							extract=ff,
							value=SOME(ff dyn1)})
			      | instPatternFormal _ = NONE
			    val dynObj = instPatternFormal dpf
			    val newSig = ModuleSig{statparams=statsOpt, dynParam=dynObj}
			    val newMC = ModuleClosure{name=name,
						      uname=uname,
						      Sig=newSig,
						      closure=cl,
						      def=def,
						      parent=context}
			in instantiateModule options context (newMC, newSig)
			end
		      (* This must be a module instance without dynamic parameters *)
		      | loop0 phash context (m, msig) =
			let val (mc,statsOpt) = tcModuleInst phash context m
			    val requireParams = 
				 requireParams andalso not(areDynamicInstantiated mc)
				 andalso
				 (case GetObjectName m of
				      NONE => true
				    | SOME n => not(ptEq(n, Self dp)))
			    val _ = if requireParams then raise TypeError
				     ((pos2string(pos m))
				      ^": module requires dynamic parameter:\n  "
				      ^(pt2string m))
				    else ()
			    val ModuleSig{dynParam=dynOpt,...} = msig
			    val newSig = ModuleSig{dynParam=dynOpt, statparams=statsOpt}
			in 
			    if isInstantiated mc then mc
			    else instantiateModule options context (mc, newSig)
			end
		    val msig = (case msigOpt of
				    NONE => ModuleSig{statparams=NONE, dynParam=NONE}
				  | SOME s => s)
		    val res = loop phash context (m, msig)
		    val _ = debug(fn()=>(debug(fn()=>"\ntypeCheckModuleExpr[Debug] = \n");
					 ptPrintDebug(verbDebug(getOptions())
						      "typeCheckModuleExprDebug") res;
					 ""))
		    val _ = popFunStackLazy(funName, fn()=>(pt2string res))
		in res
		end
	    and typeCheckStatDecl context (PType(p,tp)) = 
		let val uid = Uid(newName())
		    val sft = StaticFormalType{name=tp,uname=uid, value=NONE}
		in (addObject(uid,sft); [sft])
		end
	      | typeCheckStatDecl context (TypedVars(p,vars,tp)) =
		let val tp1 = (case tp of
				   NONE => NONE
				 | SOME t => 
				     let val t0 = expandType context t
				     in (case collectVarsPoly t0 of
					     [] => SOME t0
					   | _ => raise TypeError
						 ((pos2string p)
						  ^": type variables are not allowed in "
						  ^"module's static parameters:\n  "
						  ^(pt2string t0)
						  ^"\nIn declaration:\n  "
						  ^(pt2string(TypedVars(p,vars,tp)))))
				     end)
		    fun newSFC(name,tp) = 
			let val uid = Uid(newName())
			    val sfc = StaticFormalConst{name=name,
							uname=uid,
							Type=tp, value=NONE}
			in addObject(uid,sfc); sfc
			end
		    fun ff v = (case tp1 of
				    NONE => newSFC(v,newTvar())
				  | SOME t => newSFC(v,t))
		in List.map ff vars
		end
	      | typeCheckStatDecl _ x = raise SympBug
		  ("typeCheckStatDecl: bad static declaration: "
		   ^(pt2stringDebug x))

	    and typeCheckStatParam phash context (x, y) =
		let val funName = "typeCheckStatParam"
		    fun loop(tp, x as StaticFormalType{name=n,uname=un,...}) =
			let val _ = pushFunStackLazy(funName, fn()=>(pt2stringDebug tp)
					     ^", "^(pt2string x)^" (* type *)")
			    fun extractType(Type(_,t)) = t
			      | extractType(Datatype(_,t)) = t
			      | extractType(BoolType p) = BoolType p
			      | extractType(IntType p) = IntType p
			      | extractType(NatType p) = NatType p
			      | extractType x = raise TypeError
				((pos2string(pos tp))
				 ^": type instance required:\n  Expression: "
				 ^(pt2string tp)^"\n  Static formal parameter: type "
				 ^(pt2string n))
			    val tp0 = expandType context (extractType tp)
			    val tp1 = substType(!tHash)(substTypePoly phash tp0)
			    val res = StaticFormalType{name=n,
						       uname=un,
						       value=SOME tp1}
			    val _ = popFunStackLazy(funName, fn()=>pt2stringDebug res)
			in res
			end
		      | loop (e, x as StaticFormalConst{name=n,uname=un,Type=t,...}) =
			let val _ = pushFunStackLazy(funName,fn()=>(pt2string e)
						     ^", "^(pt2string x)
						     ^" (* constant *)\n")
			    fun isType (Type _) = true
			      | isType (Datatype _) = true
			      | isType (BoolType _) = true
			      | isType (IntType _) = true
			      | isType (NatType _) = true
			      | isType _ = false
			    val _ = if isType e then
				raise TypeError
				    ((pos2string(pos e))
				     ^": type instantiated where constant required:\n"
				     ^"  Actual parameter: "
				     ^(pt2string e)^"\n  Static formal parameter: "
				     ^(pt2string n)^": "^(pt2string t))
				    else ()
			    val tp = substType(!tHash)(substTypePoly phash (expandType context t))
			    val (e1,tp1) = tc (ValuesOnly false) phash context (e,tp)
			    val res = StaticFormalConst{name=n,uname=un,Type=tp1,value=SOME e1}
			    val _ = popFunStackLazy(funName, fn()=>pt2stringDebug res)
			in res
			end
		      | loop (e,x) = raise SympBug
			("typeCheckStatParam: not a static formal parameter: "
			 ^(pt2stringDebug x))
		in
		    loop (x, y)
		end

            and typeCheckDecl bits phash context (Theorem(p,name,body)) =
		  typeCheckTheorem phash context (Theorem(p,name,body))
	      | typeCheckDecl bits phash context (TypeDecl(p,name,parms,def)) =
		  typeCheckType context (TypeDecl(p,name,parms,def))
	      | typeCheckDecl bits phash context (DatatypeDecl(p,name,parms,def)) =
		  typeCheckType context (DatatypeDecl(p,name,parms,def))
	      | typeCheckDecl bits phash context (Var(p,lst)) =
		  raise SympBug((pos2string p)
				^": VAR declarations are not implemented,"
				^" and probably will be depricated.")
	      | typeCheckDecl bits phash context (StateVarDecl(p,lst)) =
		  let val _ = lazyVerbDebug "" (fn()=>("\ntypeCheckDecl("
						       ^(pt2string(StateVarDecl(p,lst)))
						       ^")\n"))
		      val StateVarId = 
			  let fun contextList c =
			           (case parentClosure c of
					NONE => [c]
				      | SOME pc => c::(contextList pc))
			    fun getModuleInfo (ModuleClosure{uname=n,
							     Sig=Sig,...}) =
				   SOME(n, getDynDecls Sig, getStatDecls Sig)
			      | getModuleInfo _ = NONE
			    val ModInfoList = List.mapPartial getModuleInfo (contextList context)
			    fun wrapStat (t as StaticFormalType _) = t
			      | wrapStat x = ObjectInst{obj=x,subst=fn x=>x}
			    fun wrapStats lst = List.map wrapStat lst
			    fun doModInst(m,SOME(dn),SOME stats) = 
				   Appl(dp,ModuleInst(dp,m,wrapStats stats),
					ObjectInst{obj=dn,subst=fn x=>x})
			      | doModInst(m,NONE,SOME stats) = ModuleInst(dp,m,wrapStats stats)
			      | doModInst(m,SOME dn,NONE) = 
				   Appl(dp,m,ObjectInst{obj=dn,subst=fn x=>x})
			      | doModInst(m,NONE,NONE) = m
			    fun foldDot [] = raise SympBug
				   ("typeCheckDecl/StateVarDecl: "
				    ^"no named module above state vars: \n"
				    ^(pt2stringDebug(StateVarDecl(p,lst))))
			      | foldDot [x] = doModInst x
			      | foldDot ((n,dn,stats)::lst) =
				   doModInst(Dot(dp,foldDot lst,n), dn, stats)
			  in foldDot ModInfoList
			  end
		      fun newSV(name,tp) =
			  if isSome(findNameLocal StateVarOnly context name) then
			      raise TypeError
				  ((pos2string(pos name))
				   ^": State variable has already been "
				   ^"declared in this module:\n  "
				   ^(pt2string name))
			  else
			  let 
			      val uid = Uid(newName())
			      val sv = StateVar{name=name,
						uname=uid,
						Type=(case tp of
							  NONE => newTvar()
							| SOME t => t),
						id=StateVarId}
			  in addObject(uid,sv); sv
			  end
		      fun vars2statevar [] tp = []
			| vars2statevar (v::lst) tp =
			    (newSV(v,tp))::(vars2statevar lst tp)
		      fun typed2statevar [] = []
			| typed2statevar ((TypedVars(_,vars,tp))::lst) =
			    (vars2statevar vars
			     (case tp of
				  NONE => NONE
				| SOME t => SOME(substType(!tHash)
						 (substTypePoly phash (expandType context t)))))
			    @(typed2statevar lst)
			| typed2statevar (x::lst) = 
			    if isName x then 
				(vars2statevar [x] NONE)@(typed2statevar lst)
			    else raise SympBug
			    ("typeCheckDecl/StateVarDecl/typed2statevar: "
			     ^"not a TypedVars:\n  "
			     ^(pt2stringDebug x))
		  in typed2statevar lst
		  end
	      | typeCheckDecl bits phash context (Val(p,pat,expr)) =
		  let val _ = lazyVerbDebug "" (fn()=>("\ntypeCheckDecl("
						       ^(pt2string(Val(p,pat,expr)))
						       ^")\n"))
		      val ph = newVarsExprHash(newVarsExprHash phash pat) expr 
		      val (ee,tp) = tc bits ph context (expr,newTvar())
		      val newexpr = evalExpr ee
		      val (_,plist,ptp) = typeCheckPattern ph context (pat,tp)
		      val _ = lazyVerbDebug "typeCheckDecl"
		                            (fn () => 
					     ("TC: typeCheckDecl = ("
					      ^(pt2string pat)^": "
					      ^(pt2string ptp)^" = "
					      ^(pt2string ee)^": "
					      ^(type2str (!tHash) tp)^")\n"))
		      fun extractDef e (PatternFormal{name=name,
						      uname=un,
						      Type=vtp,
						      extract=ff}) =
			  Object{name=name,
				 uname=un,
				 Type=substType (!tHash) vtp,
				 def=evalExpr(ff e)}
			| extractDef _ x = raise SympBug("typeCheckDecl/val: "
							 ^"not PatternFormal: "
							 ^(pt2stringDebug x))
		  in
		      (List.map(extractDef newexpr) plist)
		  end
	      | typeCheckDecl bits phash context (FunCases(p,funcases)) =
		  let (* First, linearize the recursion, for 
		       * "fun f x1 y1 z1 = e1
		       *    | f x2 y2 z2 = e2 ..."
		       * construct:
		       * "fn f => fn x => fn y => fn z =>
		       *    case (x,y,z) of
		       *        (x1,y1,z1) => e1
		       *      | (x2,y2,z2) => e2 ..." *)
		      val _ = lazyVerbDebug "" (fn()=>("\ntypeCheckDecl("
						       ^(pt2string(FunCases(p,funcases)))
						       ^")\n"))
		      val _ = lazyVerbDebug "typeCheckDecl" (fn()=>"[\n")
		      val MinPolyNumber = nextNumber() (* for generating polymorphic type vars *)
		      fun getInfo(FunCase(_,name,params,_,_)::_) = (name, params)
			| getInfo(x::_) = 
			  raise SympBug("typeCheckDecl: not FunCase in "
					^" FunCases at "
					^(pos2string(pos x))^": "
					^(pt2stringDebug x))
			| getInfo [] =
			      raise SympBug("typeCheckDecl: empty FunCases at "
					    ^(pos2string p))
		      (* Check the name and # of args,
		       *  and convert to a "case"'s case *)
		      fun doCase (nm,n) (FunCase(p,name,params,restp,def)) =
			  if not(ptEq(nm,name)) then
			      raise TypeError((pos2string p)
					      ^": function name mismatch: "
					      ^(pt2string name)
					      ^", but expecting "
					      ^(pt2string nm))
			  else if n<>(List.length params) then
			      raise TypeError((pos2string p)
					      ^": wrong number of parameters: "
					      ^(Int.toString(List.length params))
					      ^" instead of "
					      ^(Int.toString n)^" in "
					      ^(pt2string(FunCase(p,name,params,
								  restp,def))))
			  else
			    ChoiceExpr(p, (case params of
					       [x] => x
					     | _ => TuplePattern(dp,params)),
				       def)
			| doCase (nm,_) x = raise SympBug
			    ("typeCheckDecl/doCase: not FunCase: "
			     ^(pt2stringDebug x)^" in function "^(pt2stringDebug nm))
		      fun foldCurry(name,res) = Fn(dp,[ChoiceExpr(dp,name,res)])
		      val (fname,params) = getInfo funcases
		      val n = List.length params
		      (* Create list of new names for parameters *)
		      val parmNames = List.map(fn _ => Id(dp,newName())) params
		      val caseExpr = 
			  CaseExpr(dp,
				   (case parmNames of
					[] => raise SympBug
					    ("typeCheckDecl/FunCases: no parameters!")
				      | [x] => x
				      | lst => TupleExpr(dp,lst)),
				   List.map(doCase(fname,n)) funcases)
		      val expr = List.foldr foldCurry caseExpr (fname::parmNames)
		      val auxtp = FunType(dp,newTvar(),newTvar())
		      val etype = FunType(dp,auxtp,auxtp)
		      (* The typechecked "flattened" recur. func, with 
		       * the type that the user should see *)
		      val (ee,tp) = 
			  (case tc bits phash context (expr,etype) of
			       (e,FunType(_,tp,_)) => (e,tp)
			     | (e,ArrayType(_,tp,_)) => (e,tp)
			     | (e,x) => raise SympBug("typeCheckDecl/fun: "
						      ^"not FunType: "
						      ^(pt2stringDebug x)))
		      val newexpr = evalExpr ee
		      val _ = tHash := polyVarsSubstHash MinPolyNumber (nextNumber()) (!tHash)
		      val uid = Uid(newName())
		      val res = Object{name=fname,
				       uname=uid,
				       Type=substType (!tHash) tp,
				       def=RecurFun(newexpr)}
		      val _ = addObject(uid,res)
		      val _ = lazyVerbDebug "typeCheckDecl"
			       (fn()=>("\ntypeCheckDecl("
				       ^(pt2string(FunCases(p,funcases)))
				       ^") =\n  "^(pt2string res)
				       ^"]\n"))
		  in [res]
		  end
	      | typeCheckDecl bits phash context (Module(p,{name=name,
						 statparams=stats,
						 dynparams=dyns,
						 body=body})) =
		  let val _ = lazyVerbDebug "" 
		         (fn()=>("\ntypeCheckDecl("
				 ^(pt2string(Module(p,{name=name,
						       statparams=stats,
						       dynparams=dyns,
						       body=Quote(dp,"<skipping>")})))
				 ^")\n"))

		      (* To preserve type safety and mis-aliasing of state variables, we
		         prohibit declaration of modules with the same name on the same
		         level. *)
		      val _ = (case findNameLocal ModulesOnly context name of
				   NONE => ()
				 | SOME _ => raise TypeError
				     ((pos2string p)
				      ^": Modules with the same name cannot be defined "
				      ^"on the same level:\n"
				      ^(pt2string(Module(p,{name=name,
							    statparams=stats,
							    dynparams=dyns,
							    body=Quote(dp,"<skipping>")})))))

		      (* First, typecheck the headers, static and dynamic parameters, and
		         create a local closure with these definitions. *)

		      val phash0 = makeHashDefault(op =,fn x => x)
		      fun spClosure lst = LetClosure{locals=lst,body=Fake,parent=context}
		      fun statDescend [] acc = acc
			| statDescend (sp::lst) acc =
			    statDescend lst ((typeCheckStatDecl (spClosure acc) sp)@acc)
		      val statParams = (case stats of
					    NONE => []
					  | SOME lst => statDescend lst [])
		      val statParamsOpt = (case stats of
					       NONE => NONE
					     | SOME _ => SOME(List.rev statParams))
		      val (_,dynParams,dyntp) = 
			    (case dyns of
				 NONE => (Fake,[],Fake)
			       | SOME pat => 
				   typeCheckPattern phash0 (spClosure statParams)(pat,newTvar()))
		      val dynParams = List.map(fn (PatternFormal{name=name,
								 uname=uname,
								 Type=tp,
								 extract=eff}) =>
					       DynPatternFormal{name=name,
								uname=uname,
								Type=tp,
								value=NONE,
								extract=eff}
		                                | x => x) dynParams
		      (* Name the dynamic parameter *)
		      val dynName = Option.map(fn x=>newName()) dyns
		      val dynFormal = 
			   Option.map(fn n => 
				      let val res =
					    DynPatternFormal{name=Id(dp,n),
							     uname=Uid n, (* it's new anyway *)
							     Type=dyntp,
							     value=NONE,
							     extract=(fn x=>x)}
				      in addObject(Uid n, res); res end) dynName
		      val dynNamePF = (case dynFormal of
					   NONE => []
					 | SOME n => [n])
		      (* val spContext = spClosure(dynNamePF@dynParams@statParams) *)
		      val Sig = ModuleSig{statparams=statParamsOpt,
					  dynParam=dynFormal}
		      val uname = Uid(newName())
		      fun mClosure me =
			  let val (lst,def) =
			       (case me of
				    ModuleClosure{closure=lst,
						  def=def,...} => (lst,def)
				  | _ => ([],SOME me))
			  in ModuleClosure{name=SOME name,
						     uname=uname,
						     Sig=Sig,
						     closure=lst,
						     def=def,
						     parent=context}
			  end
		      (* Now typecheck the body of the module.  It is
		         important to do it in `ModuleClosure' context
		         so that state variables get the correct id string *)
		      val cl = typeCheckModuleExpr true phash0 (mClosure Fake) (body, SOME Sig)
		      val mc = mClosure cl
		  in addObject(uname,mc); [mc]
		  end
	      | typeCheckDecl bits phash context (Open(p,name)) =
		  raise SympBug((pos2string p)
				^": sorry, `open' declaration is "
				^"not implemented yet")
	      | typeCheckDecl bits phash context (Export(p,name)) =
		  raise SympBug((pos2string p)
				^": sorry, `export' declaration is "
				^"not implemented yet")
              | typeCheckDecl bits phash context (a as Nop x) =
		  typeCheckAsst phash context a
	      | typeCheckDecl bits phash context (a as NormalAsst x) =
		  typeCheckAsst phash context a
	      | typeCheckDecl bits phash context (a as InitAsst x) =
		  typeCheckAsst phash context a
	      | typeCheckDecl bits phash context (a as NextAsst x) =
		  typeCheckAsst phash context a
	      | typeCheckDecl bits phash context (a as IfAsst x) =
		  typeCheckAsst phash context a
	      | typeCheckDecl bits phash context (a as LetAsst x) =
		  typeCheckAsst phash context a
	      | typeCheckDecl bits phash context (a as CaseAsst x) =
		  typeCheckAsst phash context a
	      | typeCheckDecl bits phash context (a as ChooseAsst _) =
		  typeCheckAsst phash context a
	      | typeCheckDecl bits phash context (a as LabeledAsst _) =
		  typeCheckAsst phash context a
	      (* Included files *)
	      | typeCheckDecl bits phash context (Included(_,_,prog)) =
		  typeCheckProgram context prog
	      | typeCheckDecl bits phash _ x =
		  raise SympBug("typeCheckDecl: wrong declaration:\n  "
				^(pt2stringDebug x))

            and typeCheckAsst phash context asst =
		let val _ = lazyVerbDebug "" (fn()=>("typeCheckAsst("
						     ^(pt2string asst)
						     ^")\n"))
		    fun looplist phash context lst =
			  List.map(fn asst => loop phash context asst) lst
		    and loop phash context (Nop p) = Nop p
		      | loop phash context (NormalAsst(p,var,expr)) = 
			 let val (v,tv) = tc StateVarOnly phash context (var,newTvar())
			     val (e,tp) = tc (ValuesOnly true) phash context (expr,tv)
			     val ee = evalExpr e
			 in NormalAsst(p,v,ee)
			 end
		      | loop phash context (InitAsst(p,var,expr)) = 
			 let val (v,tv) = tc StateVarOnly phash context (var,newTvar())
			     val (e,tp) = tc (ValuesOnly true) phash context (expr,tv)
			     val ee = evalExpr e
			 in InitAsst(p,v,ee)
			 end
		      | loop phash context (NextAsst(p,var,expr)) = 
			 let val (v,tv) = tc StateVarOnly phash context (var,newTvar())
			     val (e,tp) = tc (ValuesOnly true) phash context (expr,tv)
			     val ee = evalExpr e
			 in NextAsst(p,v,ee)
			 end
		      | loop phash context (IfAsst(p,lst1,lst2)) =
			 let val condlst = looplist phash context lst1
			     val elselst = looplist phash context lst2
			 in IfAsst(p,condlst,elselst)
			 end
		      | loop phash context (CondAsst(p,e,lst)) =
			 let val (e1,_) = tc (ValuesOnly true) phash context (e,BoolType dp)
			     val assts = looplist phash context lst
			     val ee1 = evalExpr e1
			 in CondAsst(p,ee1,assts)
			 end
		      | loop phash context (LetAsst(p,defs,assts)) =
			 let fun c defs assts = LetAsstClosure{locals=defs,
							       body=assts,
							       parent=context}
			     fun descend [] acc = acc
			       | descend (d::lst) acc =
				  let val decls=typeCheckDecl(ValuesOnly true) phash(c acc []) d
			          in descend lst (decls@acc)
				  end
			     val locals = descend defs []
			     val cxt = c locals []
			 in c locals (looplist phash cxt assts)
			 end
		      | loop phash context (CaseAsst(p,expr,assts)) =
			 let val (sel,stp) = tc (ValuesOnly true) phash context (expr,newTvar())
			     val newsel = evalExpr sel
			     fun newPhash ph [] = ph
			       | newPhash ph (e::lst) = newPhash(newVarsExprHash ph e) lst
			     (* The case selector is given a unique name in
			      the case statement and is referenced to only by its
			      name in the choices. *)
			     val unameSel = 
				 let val u = Uid(newName())
				     val pf = PatternFormal{name=u,
							    uname=u,
							    Type=stp,
							    extract=(fn x=>x)}
				 in addObject(u,pf); pf
				 end
			     fun doChoice (ChoiceAsst(p,pat,assts)) =
			          let val ph = newPhash phash (pat::assts)
				      fun convPatName(PatternFormal{name=name,
								    uname=un,
								    Type=tp,
								    extract=ff}) =
					    Object{name=name,
						   uname=un,
						   Type=substTypePoly ph tp,
						   def=ff unameSel}
					| convPatName x = raise SympBug
					    ("typeCheckAsst/CaseExpr: not PatternFormal :"
					     ^(pt2stringDebug x))
		       (* Don't instantiate stp with ph: new type vars don't appear in it *)
				      val (newpat,plist,ptp) =
					typeCheckPattern ph context (pat,substType (!tHash) stp)
				      val plist = List.map convPatName plist
				      fun cxt assts = 
					    ChoiceAsstClosure{pattern=newpat,
							      uname=unameSel,
							      names=plist,
							      body=assts,
							      parent=context}
				  in cxt(looplist ph (cxt []) assts)
				  end
			       | doChoice x = raise SympBug
				  ("typeCheckAsst/CaseAsst: not ChoiceAsst: "
				   ^(pt2stringDebug x))
			 in CaseAsst(p,newsel,List.map doChoice assts)
			 end
		      | loop phash context (ChooseAsst(p,namesOpt,lst)) =
			 let val varsOpt = Option.map typedVars2QVars namesOpt
			     fun qvar2param(QuantifiedVar{name=name, uname=u, Type=tp}) =
				    (PatternFormal{name=name, uname=u, Type=tp,
						   extract=fn x=>x},
				     tp)
			       | qvar2param x = raise SympBug
				    ("typeCheckAsst/ChooseAsst: not QuantifiedVar: "
				   ^(pt2stringDebug x))
			     fun vars2label vars =
				 let val pairs = List.map qvar2param vars
				     val types = List.map #2 pairs
				     val params = TupleExpr(dp, List.map #1 pairs)
				     val tp = FunType(dp, TupleType(dp, types), BoolType dp)
				 in ()
				 end
			     (* Compute new phash *)
			     fun newVarsList ph [] = ph
			       | newVarsList ph (e::lst) = newVarsExprHash(newVarsList ph lst) e
			     val phash = (case varsOpt of
					      SOME vars => newVarsList phash vars
					    | NONE => phash)
			     fun doChoice cxt (ChoiceAsst(p,expr,assts)) =
			          let val (e,_) = tc (ValuesOnly true) phash cxt
				                     (expr,BoolType dp)
				      val ee = evalExpr e
				      val newassts = looplist phash cxt assts
				  in ChoiceAsst(p,ee,newassts)
				  end
			       | doChoice _ x = raise SympBug
				  ("typeCheckAsst/ChooseAsst: not ChoiceAsst: "
				   ^(pt2stringDebug x))
			     fun cl choices = 
				 ChooseAsstClosure{names=varsOpt,
						   choices=choices,
						   parent=context}
			 in 
			     cl(List.map(doChoice(cl [])) lst)
			 end
		      | loop phash context (ForeachAsst(p,names,lst)) =
			 let val vars = typedVars2QVars names
			     (* Compute new phash *)
			     fun newVarsList ph [] = ph
			       | newVarsList ph (e::lst) = newVarsExprHash(newVarsList ph lst) e
			     val phash = newVarsList phash vars
			     fun cl assts = 
				 ForeachAsstClosure{names=vars,
						    assts=assts,
						    parent=context}
			 in 
			     cl(looplist phash (cl []) lst)
			 end
		      | loop phash context (LabeledAsst(p, label, a)) =
			 LabeledAsst(p, label, loop phash context a)
		      | loop _ _ x = raise SympBug
			 ("typeCheckAsst: not an assignment:\n  "
			  ^(pt2stringDebug x))
		in [loop phash context asst]
		end

            and typeCheckTheorem phash context (Theorem(p,name,body)) =
	        [Theorem(p,name,typeCheckTheoremExpr phash context body)]
	      | typeCheckTheorem phash context x = raise SympBug
		  ("typeCheckTheorem: not a theorem declaration:\n  "
		   ^(pt2stringDebug x))

	    and typeCheckTheoremExpr phash context body =
		let fun tcThmName context name =
		        let val _ = if isName name then ()
				    else raise SympBug
					("typeCheckTheoremExpr/tcThmName: not a name: "
					 ^(pt2string name))
                            val thm = (case findNameLocal TheoremsOnly context name of
					   SOME x => x
					 | NONE => raise TypeError
					       ((pos2string(pos name))
						^": can't find theorem named "
						^(pt2string name)))
			in thm
			end
		    fun tcBody (Models(p,me,e)) =
		         let val module = typeCheckModuleExpr true phash context (me, NONE)
			     val (e1,_) = tc (ValuesOnly true) phash context (e,BoolType dp)
			     val ee = evalExpr e1
			 in Models(p,module,ee)
			 end
		      | tcBody (Refines(p,m1,m2)) =
			 let val mod1 = typeCheckModuleExpr true phash context (m1, NONE)
			     val mod2 = typeCheckModuleExpr true phash context (m2, NONE)
			 in Refines(p,mod1,mod2)
			 end
		      | tcBody (thm as Id _) = tcThmName context thm
		      | tcBody (Dot(p,me,n)) =
			  let val phash = makeHashDefault(op =,fn x=>x)
			      val cl = typeCheckModuleExpr true phash context (me, NONE)
			      val _ = (case cl of
					   ModuleClosure _ => ()
					 | x => raise SympBug
					       ("typeCheckTheoremExpr/Dot: not ModuleClosure:\n "
						^(pt2stringDebug x)))
			  in tcThmName cl n
			  end
		      | tcBody x = raise TypeError
			 ((pos2string(pos x))^"bad theorem expression:\n  "
			  ^(pt2stringDebug x))
		in tcBody body
		end

            and typeCheckType context decl =
		let val _ = lazyVerbDebug "" (fn()=>("typeCheckType("
						     ^(pt2string decl)
						     ^")\n"))
		    val phash = makeHashDefault(op =,fn x=>x)
		    val (p,name,params,def,recursive) =
			(case decl of
			     TypeDecl(p,name,params,def) => (p,name,params,def,false)
			   | DatatypeDecl(p,name,params,def) => (p,name,params,def,true)
			   | x => raise SympBug("typeCheckType: not a type declaration: "
						^(pt2stringDebug x)))
		    (* Keep track of whether we actually encountered a
		       recursive reference in the type def *)
		    val recursiveRef = ref false
		    val TheTypeUname = Uid(newName())
		    (* First of all, check that all parameters are different *)
		    fun checkParams [] = ()
		      | checkParams ((Tvar(p,name)::tl)) =
			if List.exists(fn x => ptEq(Tvar(p,name),x)) tl then
			    raise TypeError((pos2string p)
					    ^": repeated type parameter: "
					    ^(pt2string(Tvar(p,name))))
			else checkParams tl
		      | checkParams (x::_) =
			    raise SympBug("typeCheckType/checkParams: not Tvar: "
					  ^(pt2stringDebug x))
		    val _ = checkParams params
		    (* Hash to map type constructors' names to their TypeConstr objects *)
		    val tcHash = makeHashDefault(ptEq,pt2string) 
		    (* The Uid used for any circular reference to the base type *)
		    val TheTypeInstUid = 
			(case params of
			     [] => TheTypeUname
			   | _ => Uid(newName()))

	    (* Recursively process the type definition.  If the type
	     is recursive, make sure the recursive instances are
	     positive.  We do not have to check for zero-depth
	     recursion (like, in datatype T = T), since this type will
	     have the only element `undefined', and it's ok. *)

	    (* Assumptions/observations: any named type must be either
	     the datatype being defined, or an externally defined
	     type, since named types cannot be defined within a type
	     expression.  Similarly, any type variable must be in the
	     list of parameters, since it cannot be introduced
	     otherwise, and we do not allow free type variables in
	     named type declarations. *)

		    fun tcTypeName global positive context (Id(p,n)) = 
			if recursive andalso ptEq(Id(p,n),name) then
			    (recursiveRef := true;
			     if positive then TheTypeUname
			     else raise TypeError
				((pos2string p)
				 ^": negative occurence of recursive reference "
				 ^"in a datatype: "
				 ^(pt2string(Id(p,n)))))
			else (case if global then findTypeName context (Id(p,n))
				   else findTypeNameLocal context (Id(p,n)) of
				  SOME x => (* expandType context *) x
				| NONE => raise TypeError
				      ((pos2string p)
				       ^": Can't find named type: "^n))
		      | tcTypeName _ _ _ x = raise SympBug
			    ("typeCheckType/tcTypeName: not a name: "
			     ^(pt2stringDebug x))
		    fun tcEnumType positive (enumtp as EnumType(p,lst), tpUid) =
			let val tpUidRef = ref(tpUid)
			    fun newTC(name,tp) = 
			          (case findHash(tcHash,name) of
				       (* The constructor is brand new *)
				       NONE => 
					 let val uid = Uid(newName())
					     val tpUid = (case !tpUidRef of
							      SOME x => x
							    | NONE => 
								let val uid = Uid(newName())
								in tpUidRef := SOME uid;
								    uid
								end)
					     val obj = TypeConstr{name=name,
								  uname=uid,
								  Type=(case tp of
									    NONE => tpUid
									  | SOME t => 
									     FunType(dp,t,tpUid))}
					 in insertHashDestructive(tcHash,name,obj);
					     addObject(uid,obj);
					     obj
					 end
				       (* The constructor occurred before *)
				     | SOME(obj as TypeConstr{Type=t1,...}) => obj
					 (* Check that the type is right *)
					 before 
					 let val baseTp = 
					         (case !tpUidRef of
						      SOME uid => uid
						    | NONE => enumtp)
					     val correctTp =
					         (case tp of
						      NONE => baseTp
						    | SOME t => FunType(dp,t,baseTp))
					     val baseUid =
						 (case t1 of
						      FunType(dp,_,uid as Uid _) => uid
						    | uid as Uid _ => uid
						    | x => raise SympBug
						       ("typeCheckType/loop/EnumType: "
							^"base type is not Uid: "
							^(pt2string x)))
					 in 
					     (* if the type matches, update tpUidRef *)
					    if isSome(unifyTypes (t1,correctTp)) then
						(case !tpUidRef of
						     SOME _ => ()
						   | NONE => tpUidRef := SOME(baseUid))
					    else raise TypeError
						((pos2string(pos name))
						 ^": conflicting occurence of "
						 ^"type constructor:\n  "
						 ^(pt2string(TypedExpr(dp,name,
								       expandUids correctTp)))
						 ^"\nPrevious occurence:\n  "
						 ^(pt2string(TypedExpr(dp,name,expandUids t1))))
					 end
				     | SOME x => raise SympBug
					 ("typeCheckType/loop/EnumType: "
					  ^"not TypeConstr in the tcHash: "
					  ^(pt2string x)))
			    fun ff(Of(p,n,t)) = newTC(n,SOME(loop positive t))
			      | ff n = newTC(n,NONE)
			    val etp = EnumType(p,List.map ff lst)
			in 
			    etp before 
			    ((Option.map(fn uid=>
					 (case findObject uid of 
					      NONE => (addObject(uid,etp); ())
					    | _ => ())) (!tpUidRef));
			     ())
			end
		      | tcEnumType _ x = raise SympBug
			("typeCheckType/tcEnumType: this can't happen")

		    and loop positive x = 
			let val debug = lazyVerbDebug "typeCheckType"
			    fun argsFun() = ((if positive then "positive" else "negative")
					     ^", "^(pt2string x))
			    val _ = pushFunStackLazy("typeCheckType/loop", argsFun)
			    val res = loop' positive x
			    val _ = popFunStackLazy("typeCheckType/loop", fn()=>pt2string res)
			in res
			end
		    and loop' positive (Id(p,n)) = tcTypeName true positive context (Id(p,n))
		      | loop' positive (Dot(p,me,n)) =
			  let val cl = typeCheckModuleExpr true phash context (me, NONE)
			      val _ = (case cl of
					   ModuleClosure _ => ()
					 | x => raise SympBug
					       ("typeCheckType/Dot: not ModuleClosure:\n "
						^(pt2stringDebug x)))
			  in tcTypeName false positive cl n
			  end
		      | loop' _ (Tvar(p,n)) =
			if List.exists(fn x=>ptEq(x,Tvar(p,n))) params 
			    then Tvar(p,n)
			else raise TypeError
			    ((pos2string p)
			     ^": unbound type variable: "
			     ^(pt2string (Tvar(p,n))))
		      | loop' _ (BoolType p) = BoolType p
		      | loop' _ (NatType p) = NatType p
		      | loop' _ (IntType p) = IntType p
		      | loop' _ (RangeType(p,e1,e2)) = 
			  let val phash = makeHashDefault(op =,fn x=>x)
			      val (ne1,_) = tc (ValuesOnly false) phash context (e1,IntType dp)
			      val (ne2,_) = tc (ValuesOnly false) phash context (e2,IntType dp)
			      val (ee1,ee2) = (evalExpr ne1, evalExpr ne2)
			  in RangeType(p,ee1,ee2)
			  end
		      | loop' positive (FunType(p,t1,t2)) =
			  FunType(p,loop false t1, loop positive t2)
		      | loop' positive (ArrayType(p,t1,t2)) =
			  ArrayType(p,loop false t1, loop positive t2)
		      | loop' positive (TupleType(p,lst)) =
			  TupleType(p,List.map(loop positive) lst)
		      | loop' positive (RecordType(p,lst)) =
			let fun toRF((RecordField{name=n,Type=t})::lst) =
			           (RecordField{name=n,Type=loop positive t})
				   ::(toRF lst)
			      | toRF((TypedVars(p,vars,SOME t))::lst) =
				   let val newtp = loop positive t
				   in (List.map(fn x=>RecordField{name=x, Type=newtp}) vars)
				       @(toRF lst)
				   end
			      | toRF((TypedVars(_,lst,NONE))::_) =
				   raise TypeError
				       ((pos2string p)
					^": Untyped fields in a record type: "
					^(ptlist2str ", " lst))
			      | toRF (x::_) = raise SympBug
				   ("typeCheckType/Record: wrong record field: "
				    ^(pt2stringDebug x))
			      | toRF [] = []
			in
			    RecordType(p,toRF lst)
			end
		      | loop' positive (enumtp as EnumType(p,lst)) =
			  tcEnumType positive (enumtp, NONE)
		      | loop' positive (TypeInst(p,args,tp)) =
			let val tt = loop positive tp
			    val args1 = List.map(loop positive) args
			in
			    (case tt of
				 Uid _ => TypeInst(p, args1, tt)
			       | TypeClosure{recursive=r,params=params,def=d,name=name,...} =>
				     if (List.length args1) = (List.length params) then
					 if r then TypeInst(p,args1,tt)
					 else instantiateType d params args1
				     else raise TypeError
					 (" type "^(pt2string name)^" requires "
					  ^(Int.toString (List.length params))
					  ^"parameters, "
					  ^(Int.toString (List.length args))
					  ^" provided (debug info: typeCheckType)")
			       | x => raise TypeError
					 ((pos2string(pos x))
					  ^": parameters given to the type that doesn't "
					  ^"require parameters:\n  "
					  ^(pt2string x)))
			end
		      | loop' positive (Group(_,x)) = loop positive x
		      | loop' _ x = raise SympBug("typeCheckType/loop: not a type: "
						 ^(pt2stringDebug x))

	            (* The final type closure *)
	            fun makeTheType tp =
			   TypeClosure{name=name,
				       uname=TheTypeUname,
				       params=params,
				       def=tp,
				       recursive = !recursiveRef,
				       parent=context}
		    (* If the type has enumeration type on top, give it
		     * a special consideration *)
		    val TheType = makeTheType
			 (case def of
			      EnumType _ => tcEnumType true (def, SOME TheTypeInstUid)
			    | _ => loop true def)
		    (* val TheTypeInst = expandType context *)
		    val TheTypeInst = 
			(case params of
			     [] => TheType
			   | _ =>  TypeInst(dp,params,TheType))
		    val _ = addObject(TheTypeInstUid, TheTypeInst)
		    val constrList = List.map #2 (hash2any(fn x=>x)(fn x=>x) tcHash)
		in (* Return both the type closure and the list of construtors *)
		    TheType::constrList
		    (*  constrList@[makeTheType(EnumType(dp,constrList))] *)
		end

            (* The type checking of a file: takes the list of previous
	       declaration on the same level and adds new declarations
	       from the `Program' to them. *)

            and typeCheckProgram context (Program(p,decls)) =
		let val n = nextNumber()
		    fun makeCxt lst = LetClosure{locals=lst,
						 parent=context,
						 body=Fake}
		    fun loop acc [] = acc
		      | loop acc (d::decls) =
			let val phash = makeHashDefault(op =,fn x=>x)
			    val lst = typeCheckDecl (ValuesOnly false) phash (makeCxt acc) d
			in
			    loop (lst@acc) decls
			end
		    val res = loop [] decls
		in ((*  tHash := polyVarsSubstHash n (nextNumber()) (!tHash); *)
		    res)
		end
	      | typeCheckProgram _ x = 
		raise SympBug("typeCheckProgram: not a program:\n"
			      ^(pt2stringDebug x))


	    (* The typechecking of judgement *)
            fun typeCheckJudgement(HasType(c,e,tp)) =
		let val n = nextNumber()
		    val _ = tHash := makeHashDefault(op =,Int.toString)
		    val phash = makeHashDefault(op =,fn x=>x)
		    (* Replace all poly type vars with non-poly *)
		    val tp' = newVarsSubst tp
		    val (e1,tp1) = tc (ValuesOnly true) phash c (e,tp')
		    (* val e2 = resolveNames c e1 *)
		    val _ = tHash := polyVarsSubstHash n (nextNumber()) (!tHash)
		    val tp2 = substType (!tHash) tp1
		    val e2 = finalizeExpr options c (!tHash) e1
		in HasType(c, e2, tp2)
		end
	      (* The initial declList is ignored.
	       * The returned declaration is of no importance,
	       * the declList is the result. *)
	      | typeCheckJudgement(Decl(c,d,_)) =
		let val n = nextNumber()
		    val _ = tHash := makeHashDefault(op =,Int.toString)
		    val phash = makeHashDefault(op =,fn x=>x)
		    val lst = typeCheckDecl (ValuesOnly false) phash c d
		in Decl(c,d, List.map (finalizeExpr options c (!tHash)) lst)
		end
	      | typeCheckJudgement(ModuleJudgement(c,m,_)) =
		let val _ = tHash := makeHashDefault(op =,Int.toString)
		    val phash = makeHashDefault(op =,fn x=>x)
		in ModuleJudgement(c,m, typeCheckModuleExpr false phash c (m, NONE))
		end
	      | typeCheckJudgement(ThmJudgement(c,thm,_)) =
		let val _ = tHash := makeHashDefault(op =,Int.toString)
		    val phash = makeHashDefault(op =,fn x=>x)
		in ThmJudgement(c,thm,typeCheckTheoremExpr phash c thm)
		end
	      | typeCheckJudgement(ProgramJudgement p) = 
		let val timer=startTimer "typeCheckProgram"
		    val _ = resetNumber()
		    val _ = resetObjectHash()
		    val n = nextNumber()
		    val _ = tHash := makeHashDefault(op =,Int.toString)
		    val tclist = 
		    (case topclosure of
			 TopClosure lst => lst
		       | x => raise SympBug("typeCheckProgram: topclosure is bad: "
					    ^(pt2stringDebug x)))
		    val cl = TopClosure((typeCheckProgram topclosure p)@tclist)
		    val cl1 = finalizeExpr options topclosure (!tHash) cl
		    val res = ProgramJudgement(cl1)
		    val _ = updateStat timer
		in res
		end

	    fun stripObjectInstJudgement (HasType(c,e,t)) =
		 HasType(c,stripObjectInst e, stripObjectInst t)
	      | stripObjectInstJudgement (Decl(c,d,lst)) =
		 Decl(c,d,List.map stripObjectInst lst)
	      | stripObjectInstJudgement (ModuleJudgement(c,m,cl)) =
		 ModuleJudgement(c, m, stripObjectInst cl)
	      | stripObjectInstJudgement (ThmJudgement(c,t,t')) =
		 ThmJudgement(c, t, stripObjectInst t')
	      | stripObjectInstJudgement (ProgramJudgement p) =
		 ProgramJudgement(stripObjectInst p)

	    (* Strip ObjectInts from all new objects in the master hash *)
	    fun stripObjectInstHash() =
		let val unames = List.map #1 (hash2any(fn x=>x)(fn x=>x) newObjHash)
		    fun strip uname =
			(case findObject uname of
			     NONE => ()
			   | SOME obj => (replaceObject(uname, stripObjectInst obj); ()))
		in
		    List.app strip unames
		end

	    (* Figure out what to do with alphabetical renaming of types *)
	    (* val abchash = abcVarsHash restp *)
	    val newJudgement = stripObjectInstJudgement(typeCheckJudgement judgement)
	    val _ = stripObjectInstHash()
	    val findObject = makeFindObject()
	in
	    (newJudgement, findObject)

	end

    fun typeCheckExpr options context (expr,tp) =
	  (case typeCheck options (HasType(context,expr,tp)) of
	       (HasType(_,e,tp), ff) => (e,tp,ff)
	     | x => raise SympBug("typeCheckExpr: this can't happen"))

    fun typeCheckDecl options context decl =
	  (case typeCheck options (Decl(context,decl,[])) of
	       (Decl(_,_,lst), ff) => (lst, ff)
	     | x => raise SympBug("typeCheckDecl: this can't happen"))

    fun typeCheckModuleExpr options context module =
	  (case typeCheck options (ModuleJudgement(context, module, Fake)) of
	       (ModuleJudgement(_,_,cl), ff) => (cl, ff)
	     | x => raise SympBug("typeCheckModuleExpr: this can't happen"))

    fun typeCheckTheorem options context thm =
	let val (tmp,ff) = 
	     (case typeCheck options (ThmJudgement(context,thm,Fake)) of
		  (ThmJudgement(_,_,tcthm), ff) => (tcthm, ff)
		| x => raise SympBug("typeCheckTheorem: this can't happen"))
	in (case tmp of
		Theorem _ => (tmp,ff)
	      | x => (Theorem(dp,Id(dp,newName()),x), ff))
	end

    fun typeCheckProgram options program =
	  (case typeCheck options (ProgramJudgement program) of
	       (ProgramJudgement p, ff) => (p, ff)
	     | x => raise SympBug("typeCheckProgram: this can't happen"))

  end
