functor TransGenFun(structure TransPrep: TRANS_PREP): TRANS_GEN =

  struct
    structure TransPrep = TransPrep

    open TransPrep
    open TransCommon

    open ListOps
    open Hash
    open Pos
    open Evaluate
    open Abstract
    open ParserDefault
    open ParseTreeStruct
    open Interface
    open Options

    fun isBoolFun (Iff _) = true
      | isBoolFun (And _) = true
      | isBoolFun (Or _) = true
      | isBoolFun (Implies _) = true
      | isBoolFun (Eq _) = true
      | isBoolFun (NotEq _) = true
      | isBoolFun (Not _) = true
      | isBoolFun (Ag _) = true
      | isBoolFun (Af _) = true
      | isBoolFun (Eg _) = true
      | isBoolFun (Ef _) = true
      | isBoolFun (Ax _) = true
      | isBoolFun (Ex _) = true
      | isBoolFun (Au _) = true
      | isBoolFun (Eu _) = true
      | isBoolFun (Ar _) = true
      | isBoolFun (Er _) = true
      | isBoolFun (Lt _) = true
      | isBoolFun (Gt _) = true
      | isBoolFun (Le _) = true
      | isBoolFun (Ge _) = true
      | isBoolFun _ = false

    (* Replace all the built-in functions in expression `e' with their
       abstract interpretatons and evaluate the result expression.
       Implements the `omega(e)' operation from the hacker's
       manual. *)

    fun absEval options  findObject abs e =
	let fun findAbsInt(f,tp) =
	        let fun findAbsList [] = NONE
		      | findAbsList (tp::lst) =
		          (case findAbsInt(f,tp) of
			       SOME ai => SOME ai
			     | NONE => findAbsList lst)
		    fun extrField (RecordField{Type=t,...}) = t
		      | extrField x = raise SympBug
			  ("absEval: not a RecordField: "^(pt2string x))
		    fun extrArg (TypeConstr{Type=tp,...}) =
			  (case tp of
			       FunType(_,tp,_) => SOME tp
			     | _ => NONE)
		      | extrArg x =raise SympBug
			  ("absEval: not a TypeConstr: "^(pt2string x))
		(* Alert: This is not type safe, strictly speaking.
		   Perhaps, later we need to match the types before returning a hit. *)
		in (case getTypeAbstraction options abs tp of
			SOME ta => getAbsInt options ta f
		      | NONE =>
			    (case tp of
				 TupleType(_,lst) => findAbsList lst
			       | FunType(_,t1,t2) => findAbsList[t1,t2]
			       | ArrayType(_,t1,t2) => findAbsList[t1,t2]
			       | RecordType(_,lst) =>
				  findAbsList(List.map extrField lst)
			       | EnumType(_,lst) => 
				  findAbsList(List.mapPartial extrArg lst)
			       | _ => NONE))
		end
	    (* fun replace (e as Builtin{name=f, Type=tp}) =
		  (case findAbsInt(f,tp) of
		       SOME af => af
		     | NONE => e)
	      | replace e = ptTransform replace e *)
	    (* This abstraction is not correct.  Disable it for now. *)
	    fun replace e = e
	in evaluateExpr options findObject (replace e)
	end

    fun tryAbstract options abs tp =
	(case abstract options abs tp of
	     SOME ta => getAbsType ta
	   | NONE => tp)

    (* Sequent `(delta <=~ gamma) => d'.  Find an index for an array
       state var `gamma' that is used in `delta' (the first arg).  If
       `delta' doesn't have `gamma' at all, return NONE.  Assumption:
       `delta' is primitive. *)

    fun findArrayIndex (Appl(_,delta,d), gamma) =
	  if svSimilar(delta,gamma) then SOME(d)
	  else findArrayIndex(delta,gamma)
      | findArrayIndex (ExtractRecord(_,delta), gamma) = findArrayIndex(delta,gamma)
      | findArrayIndex (ExtractTuple(_,delta), gamma) = findArrayIndex(delta,gamma)
      | findArrayIndex (ExtractAppl(_,delta), gamma) = findArrayIndex(delta,gamma)
      | findArrayIndex (ExtractIndex delta, gamma) = findArrayIndex(delta, gamma)
      (* delta is atomic *)
      | findArrayIndex (_,_) = NONE

    (* Sequent `V |- gamma => D'.  Split V into subsets that are parts
       of (gamma d) for each `d'.  The set D is a set of pairs (d, V_d). *)

    fun splitByArrayIndex ([],_) = []
      | splitByArrayIndex (delta::V, gamma) =
	let val D = splitByArrayIndex(V, gamma)
	    fun insert((d1,Vd)::lst, d, delta) =
		  if ptEq(d1,d) then (d,delta::Vd)::lst
		  else (d1,Vd)::(insert(lst, d, delta))
	      | insert([], d, delta) = [(d,[delta])]
	in 
	    (case findArrayIndex(vtName delta, gamma) of
		 NONE => D
	       | SOME d => insert(D,d,delta))
	end

    val true3Val = Builtin{name=True dp, Type=BoolType dp}
    val false3Val = Builtin{name=False dp, Type=BoolType dp}
    val trueVal = True2
    val falseVal = False2
    (* 2-valued propositional connectives *)
    val notVal = Builtin{name=Not dp, Type=FunType(dp,BoolType2, BoolType2)}
    val andVal = Builtin{name=And dp, 
			 Type=FunType(dp,TupleType(dp,[BoolType2, BoolType2]),
				      BoolType2)}
    val orVal = Builtin{name=Or dp, 
			Type=FunType(dp,TupleType(dp,[BoolType2, BoolType2]),
				     BoolType2)}
    val impliesVal = Builtin{name=Implies dp, 
			     Type=FunType(dp,TupleType(dp,[BoolType2, BoolType2]),
					  BoolType2)}
    fun negFormula f = Appl(dp, notVal, f)
    fun impliesFormula(f,g) = Appl(dp, impliesVal, TupleExpr(dp,[f,g]))
    fun foldOp (Op,default) lst =
	let fun foldFun(f1,f2) = Appl(dp,Op,TupleExpr(dp,[f1,f2]))
	    fun doFold [] = default
	      | doFold (f::lst) = List.foldr foldFun f lst
	in doFold(List.rev lst)
	end
    val foldAnd = foldOp(andVal, trueVal)
    val foldOr = foldOp(orVal, falseVal)
    (* Formula for `if c then tcase else fcase endif' *)
    fun caseFormula(c, tcase, fcase) =
	foldAnd[impliesFormula(c,tcase),impliesFormula(negFormula c, fcase)]
    (* Create a formula `e1 = e2' of type BoolType2 *)
    fun eqFormula findObject (e1,e2) = 
	let val tp = getExprType findObject e1
	in Appl(dp,Builtin{name=Eq dp,
			   Type=FunType(dp,TupleType(dp,[tp,tp]),BoolType2)},
		TupleExpr(dp,[e1,e2]))
	end
    (* Substitute `v' for `e' in a ParseTree `x'. 
       Checks for name shadowing in FunClosure and ChoiceClosure. *)
    fun subst(v,e) (body as FunClosure{formals=f,...}) = 
	if eqUNames(v,f) then body else substRec(v,e) body
      | subst(v,e) (body as ChoiceClosure{uname=u,...}) = 
	    if eqUNames(v,u) then body else substRec(v,e) body
      | subst(v,e) body =
		if eqUNames(v,body) then e else substRec(v,e) body
    and substRec(v,e) x = ptTransform (subst (v, e)) x
    fun foldCase ([],Fbot) = Fbot
      | foldCase ((c,f)::pairs, Fbot) =
	caseFormula(c,f,foldCase(pairs,Fbot))
    (* Create an `undefined' value of type `tp' *)
    fun bot tp = Builtin{name=Undefined dp, Type=tp}

    (* Check for a state var expression *)
    fun isStateVar (StateVar _) = true
      | isStateVar (DynPatternFormal _) = true
      | isStateVar (ExtractTuple(_,e)) = isStateVar e
      | isStateVar (ExtractRecord(_,e)) = isStateVar e
      | isStateVar (ExtractAppl(_,e)) = isStateVar e
      | isStateVar (ExtractIndex(e)) = isStateVar e
      | isStateVar (Appl(_,e,_)) = isStateVar e
      | isStateVar _ = false

    (* Checks if the expression is a constant literal (an expression
       that has a concrete interpreted value) *)

    fun isLiteral (Builtin{name=Anyvalue _, ...}) = false
      | isLiteral (Builtin _) = true
      | isLiteral (FunClosure _) = true
      | isLiteral (Number _) = true
      | isLiteral (TypeConstr _) = true
      | isLiteral (ExtractTuple(_,e)) = isLiteral e
      | isLiteral (ExtractRecord(_,e)) = isLiteral e
      | isLiteral (ExtractAppl(_,e)) = isLiteral e
      | isLiteral (ExtractIndex(e)) = isLiteral e
      | isLiteral (Object{def=e,...}) = isLiteral e
      | isLiteral (Appl(_,e,e')) = isLiteral e andalso isLiteral e'
      | isLiteral _ = false
    (* Check if the expression is atomic (a state var or a constant with only constant
       expressions in the array indices) *)
    fun isAtomic (StateVar _) = true
      | isAtomic (Next _) = true
      | isAtomic (DynPatternFormal _) = true
      | isAtomic (ExtractTuple(_,e)) = isAtomic e
      | isAtomic (ExtractRecord(_,e)) = isAtomic e
      | isAtomic (ExtractAppl(_,e)) = isAtomic e
      | isAtomic (ExtractIndex(e)) = isAtomic e
      | isAtomic (Appl(_,e,e')) = isAtomic e andalso isConst e'
      | isAtomic e = isConst e

    fun isBuiltin (Builtin _) = true
      | isBuiltin (Object{def=e,...}) = isBuiltin e
      | isBuiltin (ObjectInst{obj=e,...}) = isBuiltin e
      | isBuiltin _ = false

    fun getBuiltinName (Builtin{name=n,...}) = SOME n
      | getBuiltinName (Object{def=e,...}) = getBuiltinName e
      | getBuiltinName (ObjectInst{obj=e,...}) = getBuiltinName e
      | getBuiltinName _ = NONE

    fun getBuiltinCore (e as Builtin _) = SOME e
      | getBuiltinCore (Object{def=e,...}) = getBuiltinCore e
      | getBuiltinCore (ObjectInst{obj=e,...}) = getBuiltinCore e
      | getBuiltinCore _ = NONE

    (* Wraps `e' into an extractor *)
    fun wrapExtr(ExtractAppl(c,_), e) = ExtractAppl(c,e)
      | wrapExtr(ExtractTuple(n,_), e) = ExtractTuple(n,e)
      | wrapExtr(ExtractRecord(f,_), e) = ExtractRecord(f,e)
      | wrapExtr(ExtractIndex _, e) = ExtractIndex e
      | wrapExtr(Appl(p,_,d), e) = Appl(p,e,d)
      | wrapExtr(x,_) = raise SympBug
	("wrapExtr: not an extractor: "^(pt2string x))
    fun wrapExtrs lst expr = List.foldr wrapExtr expr (List.rev lst)

    fun extractIndex options findObject pvars e =
	(* if isStateVar e then 
	  (case getPrimitiveIndexVar pvars e of
		   SOME x => x
		 | NONE => raise SympBug
		       ("generateTrans/extractIndex: state var doesn't have index var:\n  "
			^(pt2string e)) )
	else *) evaluateExpr options findObject (ExtractIndex e)

    (* Compute all the vectors of values for variables in `names' *)
    fun allVarValues options findObject abs lim names =
	let val tps = List.map(fn x=>tryAbstract options abs (getExprType findObject x)) names
	    val typeSize = 
		(case getTypeSize options (TupleType(dp,tps)) of
		     FiniteSize n => if n <= lim then FiniteSize n 
				     else InfiniteSize
		   | InfiniteSize => InfiniteSize)
	    val _ = (case typeSize of
			 InfiniteSize => raise TransError
			     ("Quantifier expansion will be too large for the type:\n  "
			      ^(pt2string (TupleType(dp,tps))))
		       | _ => ())
	    fun getTpVals tp = 
		(case getTypeValues options lim tp of
		     SOME lst => lst
		   (* This, actually, shouldn't happen already *)
		   | NONE => raise TransError
			 ("type of a quantified variable is too large "
			  ^"to list all its values:\n  "
			  ^(pt2string tp)
			  ^"\nin expressions\n  "^(ptlist2str "\n  " names)))
	    val ds = comblist(List.map getTpVals tps)
	in ds
	end
	    
	    
    (* Return the list of expressions e_d obtained by substituting all
       possible combinations of values for `names' into `e'.  Each new
       expression is statically evaluated. *)

    fun substVarValues options findObject abs lim (names, e) =
	let val evalExpr = evaluateExpr options findObject
	    val ds = allVarValues options findObject abs lim names
	    fun doOne dvec =
		List.foldr(fn((v,d),e) => evalExpr(subst(v,d) e)) e (zip(names, dvec))
	in List.map doOne ds
	end

    fun substAVT options findObject pvars lim (tree, old, new) =
	let fun substExpr e = evaluateExpr options findObject (subst(old, new) e)
	    fun substVar(NormalVar v) = NormalVar(substExpr v)
	      | substVar(NextVar v) = NextVar(substExpr v)
	      | substVar(InitVar v) = InitVar(substExpr v)
	    fun substVars{init=initV, next=nextV, norm=normV} =
		let val initV = List.map substVar initV
		    val nextV = List.map substVar nextV
		    val normV = List.map substVar normV
		in {init=initV, next=nextV, norm=normV}
		end			  
	    fun loop(t as NormalAsstTree(vars, v, e)) =
		NormalAsstTree(substVars vars, substExpr v, substExpr e)
	      | loop(t as NextAsstTree(vars, v, e)) =
		NextAsstTree(substVars vars, substExpr v, substExpr e)
	      | loop(t as InitAsstTree(vars, v, e)) = 
		InitAsstTree(substVars vars, substExpr v, substExpr e)
	      | loop(NopAsstTree vars) = NopAsstTree(substVars vars)
	      | loop(ListAsstTree(vars, lst)) =
		ListAsstTree(substVars vars, List.map loop lst)
	      | loop(LetAsstTree(vars, decls, t)) =
		LetAsstTree(substVars vars, decls, loop t)
	      | loop(CaseAsstTree(vars, sel, pairs)) =
		let fun ff(p, t) = (substExpr p, loop t)
		in CaseAsstTree(substVars vars, substExpr sel, List.map ff pairs)
		end
	      | loop(IfAsstTree(vars, pairs, t)) =
		let fun ff(c, t) = (substExpr c, loop t)
		in IfAsstTree(substVars vars, List.map ff pairs, loop t)
		end
	      | loop(ChooseAsstTree(vars, paramsOpt, pairs)) =
		let fun ff(c, t) = (substExpr c, loop t)
		in ChooseAsstTree(substVars vars, paramsOpt, List.map ff pairs)
		end
	      | loop(ForeachAsstTree(vars, params, t)) =
		ForeachAsstTree(substVars vars, params, loop t)
	      | loop(LabeledAsstTree(vars, label, t)) =
		LabeledAsstTree(substVars vars, label, loop t)
	    val tree = loop tree

	in tree
	end

    (* Same as above, only do it for pairs (expr, asstTree) *)

    fun substVarValuesPairs options findObject abs pvars lim names pairs =
	let val funName = "substVarValuesPairs"
	    fun pairs2str pairs = "[\n  "
	        ^(strlist2str "\n  " (List.map(fn (c,t)=>
					       (pt2string c)^" => "^(avt2str t)) pairs))
		^"]"
	    fun argsFn() =
		"names = ["
		^(ptlist2str ", " names)^"],\n   pairs = "
		^(pairs2str pairs)
	    val _ = pushFunStackLazy(funName, argsFn)
	    val evalExpr = evaluateExpr options findObject
	    val substAVT = substAVT options findObject pvars lim
	    fun doExpr(e, dvec) =
		List.foldr(fn((v,d),e) => evalExpr(subst(v,d) e)) e (zip(names, dvec))
	    fun doAVT(t, dvec) = List.foldr(fn((v,d),t) => substAVT(t,v,d)) t (zip(names, dvec))
	    fun doPair [] (_, acc) = acc
	      | doPair(dvec::ds) (pair as (e1, t1), acc) =
		  doPair ds (pair, (doExpr(e1, dvec), doAVT(t1, dvec))::acc)
	    val ds = allVarValues options findObject abs lim names
	    val res = List.foldr (doPair ds) [] pairs
	    val _ = popFunStackLazy(funName, fn()=> pairs2str res)
	in 
	    res
	end

    (* Lift nondeterminism to the top level, return the
       list of expressions without nondet. choice.
       Sequent e ^~> (e_1 | ... | e_n) *)
    fun lift options findObject lim e =
	let val debug = lazyVerbDebug options
	    val lift = lift options findObject lim
	    fun lift' (Appl(p,e1,e2)) =
		let val lst1 = lift e1
		    val lst2 = lift e2
		    fun loop ([],_) = []
		      | loop (h::tl,[]) = loop(tl,lst2)
		      | loop (lst1 as (h1::tl1), h2::tl2) = (Appl(p,h1,h2))::(loop(lst1,tl2))
		in loop(lst1,lst2)
		end
	      | lift' (NondetExpr(_,lst)) = List.foldr(op @) [] (List.map lift lst)
	      | lift' (RecordExpr(p,lst)) =
		let val pairs = List.map(fn RecordAsst(_,f,e)=> (f,e) | x => raise SympBug
					 ("generateTrans/lift/RecordExpr: not RecordAsst: "
					  ^(pt2string x))) lst
		    val fields = List.map(fn (f,_)=>f) pairs
		    val exprs = List.map(fn (_,e)=>e) pairs
		    val combs = comblist(List.map lift exprs)
		    fun doComb lst = 
			RecordExpr(p, List.map(fn (f,e)=> RecordAsst(dp,f,e)) (zip(fields,lst)))
		in List.map doComb combs
		end
	      | lift' (IfExpr(p,conds,last)) =
		let val pairs = List.map(fn CondExpr(_,c,e)=> (c,e) | x => raise SympBug
					 ("generateTrans/lift/IfExpr: not CondExpr: "
					  ^(pt2string x))) conds
		    val linearlist = List.foldr(fn((c,e),lst)=>c::e::lst) [last] pairs
		    val combs = comblist(List.map lift linearlist)
		    fun fold acc (c::e::lst) = fold ((CondExpr(dp,c,e))::acc) lst
		      | fold acc [e] = IfExpr(p, List.rev acc, e)
		      | fold _ [] = raise SympBug
			  ("generateTrans/lift/IfExpr: no `else' part in the list")
		in List.map(fold []) combs
		end
	      | lift' (CaseExpr(p,sel,lst)) =
		let val pairs = List.map(fn ChoiceExpr(_,pat,e)=> (pat,e) | x => raise SympBug
					 ("generateTrans/lift/CaseExpr: not ChoiceExpr: "
					  ^(pt2string x))) lst
		    val cases = List.map(fn (_,e)=>e) pairs
		    val patterns = List.map(fn (p,_)=>p) pairs
		    val combs = comblist(List.map lift (sel::cases))
		    fun fold (sel::lst) = 
			  CaseExpr(p,sel,List.map(fn (p,e)=>ChoiceExpr(dp,p,e))(zip(patterns,lst)))
		      | fold [] = raise SympBug
			  ("generateTrans/lift/CaseExpr: empty list")
		in List.map fold combs
		end
	      | lift' (LetClosure{body=body,...}) = lift body
	      | lift' (FunClosure{name=name,
				 formals=arg,
				 parent=p,
				 body=body}) =
		  List.map(fn x => FunClosure{name=name,
					      formals=arg,
					      parent=p,
					      body=x})
		          (lift body)
	      | lift' (e as ChooseClosure _) = raise TransError
		  ("Can't use PICK in conditions or as an array index:\n  "
		   ^(pt2string e))
	      | lift' (Builtin{name=Anyvalue _,Type=tp}) =
		  let val lst = (case getTypeValues options lim tp of
				     SOME lst => lst
				   | NONE => raise TransError
				      ("type of `anyvalue' is too large: "
				       ^(pt2string tp)))
		  in lst
		  end
	      | lift' (Object{def=e,...}) = lift e

	      (* Extractors.  Assume that the expression is already evaluated, so the
	         best we can do is to lift the expression, apply the extractors, and
	         remove duplicates *)

	      | lift' (ExtractTuple(n,e)) =
		  let val es = lift e
		      val hash = makeHashDefault(ptEq, pt2string)
		      fun checkDup x = 
			  (case findHash(hash, x) of
			       SOME _ => NONE
			     | NONE => (insertHash(hash,x,()); SOME x))
		      fun wrap x = evaluateExpr options findObject (ExtractTuple(n,x))
		      fun removeDups lst = List.mapPartial checkDup lst
		  in removeDups(List.map wrap es)
		  end
	      | lift' (ExtractRecord(f,e)) =
		  let val es = lift e
		      val hash = makeHashDefault(ptEq, pt2string)
		      fun checkDup x = 
			  (case findHash(hash, x) of
			       SOME _ => NONE
			     | NONE => (insertHash(hash,x,()); SOME x))
		      fun wrap x = evaluateExpr options findObject (ExtractRecord(f,x))
		      fun removeDups lst = List.mapPartial checkDup lst
		  in removeDups(List.map wrap es)
		  end
	      | lift' (ExtractAppl(c,e)) =
		  let val es = lift e
		      val hash = makeHashDefault(ptEq, pt2string)
		      fun checkDup x = 
			  (case findHash(hash, x) of
			       SOME _ => NONE
			     | NONE => (insertHash(hash,x,()); SOME x))
		      fun wrap x = evaluateExpr options findObject (ExtractAppl(c,x))
		      fun removeDups lst = List.mapPartial checkDup lst
		  in removeDups(List.map wrap es)
		  end
	      | lift' (ExtractIndex e) =
		  let val es = lift e
		      val hash = makeHashDefault(ptEq, pt2string)
		      fun checkDup x = 
			  (case findHash(hash, x) of
			       SOME _ => NONE
			     | NONE => (insertHash(hash,x,()); SOME x))
		      fun wrap x = evaluateExpr options findObject (ExtractIndex x)
		      fun removeDups lst = List.mapPartial checkDup lst
		  in removeDups(List.map wrap es)
		  end
	      | lift' x = [x]
	    val _ = debug "lift" (fn()=>"\nlift("^(pt2string e)^")[\n")
	    val res = if isAtomic e then [e] else lift' e
	    val _ = debug "lift" (fn()=>"\nlift => ["^(ptlist2str ", " res)^"]]\n")
	in res
	end
		
    (* Translate pattern matching.
       Sequent matches(e, P) ~> F *)
    and matches options findObject allVars abs pvars lim (expr, pat) = 
	let val debug = lazyVerbDebug options
	    val funName = "TransGen.matches"
	    val _ = pushFunStackLazy(funName, (fn()=>(pt2string expr)^", "^(pt2string pat)))
	    (* val builtinFun = builtinFun options findObject allVars abs pvars lim *)
	    val eqFun = eqFun options findObject allVars abs pvars lim 
	    val evalExpr = evaluateExpr options findObject
	    val matches = matches options findObject allVars abs pvars lim 
	    fun matches' (_, Underscore _) = trueVal
	      | matches' (_, PatternFormal _) = trueVal
	      | matches' (e, AsPattern(_,_,pat)) = matches(e, pat)
	      | matches' (e, TypedPattern(_,pat,_)) = matches(e, pat)
	      | matches' (e, TuplePattern(_,lst)) = 
		 let val es = List.map(fn n=>evalExpr (ExtractTuple(n,e)))
		                      (List.tabulate(List.length lst, fn i=>i))
		 in foldAnd(List.map matches (zip(es,lst)))
		 end
	      | matches' (e, RecordPattern(_,lst)) =
		 let fun doField (RecordAsst(_, f, pat)) = 
		          matches(evalExpr (ExtractRecord(f,e)), pat)
		       | doField x = raise SympBug
			  ("generateTrans/matches: not a RecordAsst: "
			   ^(pt2string x))
		 in foldAnd(List.map doField lst)
		 end
	      | matches' (e, ApplPattern(_, c as TypeConstr _, pat)) =
		 (let val eI = (extractIndex options findObject pvars e)
		     val ec = evalExpr (ExtractAppl(c,e))
		     val C = eqFun(eI,c)
		 in foldAnd[C, matches(ec, pat)]
		 end handle EvalError _ => falseVal)
	      (* `d' must be a contstant or TypeConstr *)
	      | matches' (e, d) = eqFun(e,d)
	    val _ = debug "TransGen.matchesDebug" 
		     (fn()=>"\nexpr = "^(pt2stringDebug expr)^"\npat = "^(pt2stringDebug pat)^"\n")
	    (* First check whether it can be computed statically *)
	    val seluname = Uid(newName())
	    val ee = evalExpr expr
	    val matchexpr = 
		if isConst ee andalso isConst pat then
		    eqFormula findObject (ee,pat)
		else CaseExpr(dp,expr,
			      [ChoiceClosure{pattern=pat,
					     uname=seluname,
					     names=[],
					     body=true3Val,
					     parent=Fake},
			       ChoiceClosure{pattern=Underscore dp,
					     uname=seluname,
					     names=[],
					     body=false3Val,
					     parent=Fake}])
	    val matchRes = evalExpr matchexpr
	    val F = if isConst matchRes then matchRes else evalExpr(matches'(expr, pat))
	    val _ = debug "TransGen.matchesDebug"
		     (fn()=>"\nTransGen.matches: result = "^(pt2stringDebug F)^"\n")
	    val _ = popFunStackLazy(funName,(fn()=> pt2string F))
	in F
	end

    (* Special handling of boolean operators in `e1' (connectives and quantifiers) *) 

    and eqBuiltin options findObject allVars abs pvars lim (e1, e2) =
	let val funName = "eqBuiltin"
	    val debug = lazyVerbDebug (getOptions()) funName
	    val _ = pushFunStackLazy(funName, fn()=>(pt2string e1)^", "^(pt2string e2))
	    val evalExpr = evaluateExpr options findObject
	    val eqFun = eqFun options findObject allVars abs pvars lim
	    val boolBot = bot(BoolType dp)
	    val eqBuiltinBool = eqBuiltinBool options findObject allVars abs pvars lim
	    (* FIXME: Check if `Eq' is abstracted first (?) *)
	    (* fun doEq args =
		let val (a1, a2) = (evalExpr(ExtractTuple(0,args)),
				    evalExpr(ExtractTuple(1,args)))
		    val Ft = eqFun(true3Val, e2)
		    val Ff = eqFun(false3Val, e2)
		    val C = eqFun(a1,a2)
		    val conj1 = impliesFormula(C, Ft)
		    val conj2 = impliesFormula(negFormula C, Ff)
		in foldAnd[conj1, conj2]
		end *)
		
	    val res =
		case e1 of
		    ForallClosure _ => eqBuiltinBool(e1, e2)
		  | ExistsClosure _ => eqBuiltinBool(e1, e2)
		  | Appl(_, Builtin{name=Eq _, ...}, args) =>
			let val (a1, a2) = (evalExpr(ExtractTuple(0,args)),
					    evalExpr(ExtractTuple(1,args)))
			    val Ft = eqFun(true3Val, e2)
			    val Ff = eqFun(false3Val, e2)
			    val C = eqFun(a1,a2)
			    val conj1 = impliesFormula(C, Ft)
			    val conj2 = impliesFormula(negFormula C, Ff)
			in foldAnd[conj1, conj2]
			end
		  | Appl(_, Builtin{name=NotEq _, ...}, args) =>
			let val (a1, a2) = (evalExpr(ExtractTuple(0,args)),
					    evalExpr(ExtractTuple(1,args)))
			    val Ft = eqFun(true3Val, e2)
			    val Ff = eqFun(false3Val, e2)
			    val C = eqFun(a1,a2)
			    val conj1 = impliesFormula(C, Ff)
			    val conj2 = impliesFormula(negFormula C, Ft)
			in foldAnd[conj1, conj2]
			end
		  | Appl(_, Builtin{name=Op, ...}, _) =>
			if isBoolFun Op then eqBuiltinBool(e1, e2)
			else raise SympBug
			    ("TransGen/eqBuiltin: sorry, this builtin function is"
			     ^" not implemented yet:\n  "
			     ^(pt2stringDebug e1))
		  | _ => raise SympBug
			    ("TransGen/eqBuiltin: not a builtin function:\n  "
			     ^(pt2stringDebug e1))
	    val _ = popFunStackLazy(funName, fn()=>(pt2string res))
	in
	    res
	end
    and eqBuiltinBool options findObject allVars abs pvars lim (e1, e2) =
	let val funName = "eqBuiltinBool"
	    val debug = lazyVerbDebug (getOptions()) funName
	    val evalExpr = evaluateExpr options findObject
	    val _ = pushFunStackLazy(funName, fn()=>(pt2string e1)^", "^(pt2string e2))
	    val eqFun = eqFun options findObject allVars abs pvars lim
	    val boolBot = bot(BoolType dp)
	    val Ft = eqFun (true3Val, e2)
	    val Ff = eqFun (false3Val, e2)
	    val Fbot = eqFun (boolBot, e2)
	    fun doImplies (e,e') =
		let val (a1,a2) = (evalExpr(ExtractTuple(0,e')),
				   evalExpr(ExtractTuple(1,e')))
		    val (C1t, C2t) = (eqFun(a1,true3Val), eqFun(a2, true3Val))
		    val (C1f, C2f) = (eqFun(a1,false3Val), eqFun(a2, false3Val))
		    val (C1bot, C2bot) = (eqFun(a1,boolBot), eqFun(a2, boolBot))
		    val conj1 = impliesFormula(foldAnd[C1t,C2f], Ff)
		    val conj2 = impliesFormula(foldOr[foldAnd[C1t,C2bot],foldAnd[C1bot,C2f]], Fbot)
		    val conj3 = impliesFormula(foldOr[C1f,C2t], Ft)
		in
		    foldAnd[conj1,conj2,conj3]
		end
	    fun doUnaryTemporal(name, e) =
		let val Ct = Appl(dp, Builtin{name=name, Type=FunType(dp,BoolType2, BoolType2)},
				  eqFun(e, true3Val))
		in (* eqFormula findObject (Ct, Ft) *)
		    Ct
		end
	    fun doBinaryTemporal(name, e) =
		let val a1 = evalExpr(ExtractTuple(0,e))
		    val a2 = evalExpr(ExtractTuple(1,e))
		    val Ct = Appl(dp, Builtin{name=name,
					      Type=FunType(dp,TupleType(dp, [BoolType2,BoolType2]),
							   BoolType2)},
				  TupleExpr(dp, [eqFun(a1, true3Val), eqFun(a2, true3Val)]))
		in (* eqFormula findObject (Ct, Ft) *)
		    Ct
		end
	    (* Compute the lists C^t, C^f, and C^bot for `e' and variables `names' *)
	    fun doQuant(names, e) = 
		let val eds = substVarValues options findObject abs lim (names, e)
		    val Ct = List.map(fn e=>eqFun(e,true3Val)) eds
		    val Cf = List.map(fn e=>eqFun(e,false3Val)) eds
		    val Cbot = List.map(fn e=>eqFun(e,boolBot)) eds
		in (Ct, Cf, Cbot)
		end
	    fun loop(ForallClosure{names=names, body=e,...}) =
		let val (Ct, Cf, Cbot) = doQuant(names, e)
		    val conj1 = impliesFormula(foldAnd Ct, Ft)
		    val tmp2 = foldAnd(List.map(fn(e1,e2)=>foldOr[e1,e2])(zip(Ct,Cbot)))
		    val conj2 = impliesFormula(foldAnd[foldOr Cbot, tmp2], Fbot)
		    val conj3 = impliesFormula(foldOr Cf, Ff)
		in
		    foldAnd[conj1, conj2, conj3]
		end
	      | loop(ExistsClosure{names=names, body=e,...}) =
		let val (Ct, Cf, Cbot) = doQuant(names, e)
		    val conj1 = impliesFormula(foldAnd Cf, Ff)
		    val tmp2 = foldAnd(List.map(fn(e1,e2)=>foldOr[e1,e2])(zip(Cf,Cbot)))
		    val conj2 = impliesFormula(foldAnd[foldOr Cbot, tmp2], Fbot)
		    val conj3 = impliesFormula(foldOr Ct, Ft)
		in
		    foldAnd[conj1, conj2, conj3]
		end
	      (* Though we should never see them here, just for completeness... *)
	      | loop(True _) = Ft
	      | loop(False _) = Ff
	      | loop(Undefined _) = Fbot
	      (* Boolean propositional connectives *)
	      | loop(e as Appl(_, Builtin{name=And _,...}, e')) =
		let val (a1,a2) = (evalExpr(ExtractTuple(0,e')),
				   evalExpr(ExtractTuple(1,e')))
		    val (C1t, C2t) = (eqFun(a1,true3Val), eqFun(a2, true3Val))
		    val (C1f, C2f) = (eqFun(a1,false3Val), eqFun(a2, false3Val))
		    val (C1bot, C2bot) = (eqFun(a1,boolBot), eqFun(a2, boolBot))
		    val conj1 = impliesFormula(foldAnd[C1t,C2t], Ft)
		    val conj2 = impliesFormula(foldOr[foldAnd[C1t,C2bot],foldAnd[C1bot,C2t]], Fbot)
		    val conj3 = impliesFormula(foldOr[C1f,C2f], Ff)
		in
		    foldAnd[conj1,conj2,conj3]
		end
	      | loop(e as Appl(_, Builtin{name=Or _,...}, e')) =
		let val (a1,a2) = (evalExpr(ExtractTuple(0,e')),
				   evalExpr(ExtractTuple(1,e')))
		    val (C1t, C2t) = (eqFun(a1,true3Val), eqFun(a2, true3Val))
		    val (C1f, C2f) = (eqFun(a1,false3Val), eqFun(a2, false3Val))
		    val (C1bot, C2bot) = (eqFun(a1,boolBot), eqFun(a2, boolBot))
		    val conj1 = impliesFormula(foldAnd[C1f,C2f], Ff)
		    val conj2 = impliesFormula(foldOr[foldAnd[C1f,C2bot],foldAnd[C1bot,C2f]], Fbot)
		    val conj3 = impliesFormula(foldOr[C1t,C2t], Ft)
		in
		    foldAnd[conj1,conj2,conj3]
		end
	      | loop(e as Appl(_, Builtin{name=Implies _,...}, e')) = doImplies(e,e')
	      | loop(e as Appl(_, Builtin{name=Arrow _,...}, e')) = doImplies(e,e')
	      | loop(e as Appl(_, Builtin{name=Darrow _,...}, e')) = doImplies(e,e')
	      | loop(e as Appl(_, Builtin{name=Iff _,...}, e')) =
		let val (a1,a2) = (evalExpr(ExtractTuple(0,e')),
				   evalExpr(ExtractTuple(1,e')))
		in 
		    eqFun(a1,a2)
		end
	      | loop(e as Appl(_, Builtin{name=Not _,...}, e')) =
		let val Ct = eqFun(e', true3Val)
		    val Cf = eqFun(e', false3Val)
		    val Cbot = eqFun(e', boolBot)
		    val conj1 = impliesFormula(Ct, Ff)
		    val conj2 = impliesFormula(Cf, Ft)
		    val conj3 = impliesFormula(Cbot, Fbot)
		in
		    foldAnd[conj1,conj2,conj3]
		end
	      (* Temporal operators *)
	      | loop(e as Appl(_, Builtin{name=name as Ag _,...}, e')) = doUnaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Af _,...}, e')) = doUnaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Eg _,...}, e')) = doUnaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Ef _,...}, e')) = doUnaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Ax _,...}, e')) = doUnaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Ex _,...}, e')) = doUnaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Au _,...}, e')) = doBinaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Eu _,...}, e')) = doBinaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Ar _,...}, e')) = doBinaryTemporal(name, e')
	      | loop(e as Appl(_, Builtin{name=name as Er _,...}, e')) = doBinaryTemporal(name, e')
	      | loop e = raise SympBug
		  ("generateTrans/eqBuiltin: sorry, this expression is not implemented yet:\n  "
		   ^(pt2stringDebug e))
	    val res = loop e1
	    val _ = popFunStackLazy(funName, fn()=>(pt2string res))
	in
	    res
	end

    (* Implements the sequent e1=e2 ~> F *)
    and eqFun options findObject allVars abs pvars lim (e1,e2) =
	let val funName = "eqFun"
	    val _ = pushFunStackLazy(funName, fn()=>(pt2string e1)^", "^(pt2string e2))
	    val eqFun = eqFun options findObject allVars abs pvars lim
	    (* val boolExpr = boolExpr options findObject allVars abs pvars lim *)
	    val evalExpr = evaluateExpr options findObject
	    val eqBuiltin = eqBuiltin options findObject allVars abs pvars lim
	    (* val builtinFun = builtinFun options findObject allVars abs pvars lim *)
	    val extractIndex = extractIndex options findObject pvars
	    val absEval = absEval options findObject
	    val lift = lift options findObject lim
	    val matches = matches options findObject allVars abs pvars lim 
	    val debug = lazyVerbDebug options "eqFun"
	    val tp = getExprType findObject e1
	    (* Translating expressions of primitive types.
	       Sequent (Sigma; e1) = e2 ~>_=^prim F.

	       Assumption: atomic2 must agree with (isAtomic e2). *)
	    fun eqPrim atomic2 (elist, e1, e2) = 
		let val funName = "eqPrim"
		    val debug = lazyVerbDebug options funName
		    val eqPrim = eqPrim atomic2
		    val _ = pushFunStackLazy(funName,
					     fn()=>("["^(ptlist2str ", " elist)^"],\n        "
						    ^(pt2string e1)^", "^(pt2string e2)
						    ^(if atomic2 then " <atomic>" else "")))
		    fun bool2str true = "true"
		      | bool2str false = "false"
		    val _ =
			if (isAtomic e2) <> atomic2 then
			    raise SympBug
				("eqPrim: invariant violated: (isAtomic e2) # atomic2: "
				 ^(bool2str (isAtomic e2))^" <> "^(bool2str atomic2))
			else ()
		    val res =
		 if isAtomic e1 then
		     if atomic2 then
			 wrapPrim true (elist, e1, e2)
		     else wrapPrim false (elist, e1, e2)
		 else
		  (case (elist, e1) of
		       (* Collecting the extractors *)
		       (l, ExtractTuple(n,e)) => eqPrim((ExtractTuple(n,Fake))::l, e, e2)
		     | (l, ExtractRecord(f,e)) => eqPrim((ExtractRecord(f,Fake))::l, e, e2)
		     | (l, ExtractIndex e) => eqPrim((ExtractIndex Fake)::l, e, e2)
		     | (l, ExtractAppl(c,e)) => eqPrim((ExtractAppl(c,Fake))::l, e, e2)
		     (* Application can mean many things, consider all cases *)
		     | ((ExtractAppl(c',_))::l, Appl(p, c as TypeConstr _, e')) =>
			 if eqUNames(c',c) then eqPrim(l,e',e2)
			 else raise TransError
			   ((pos2string p)
			    ^": type constructor in extraction does not "
			    ^"match one in expression:\n  Extracting: "
			    ^(pt2string c')
			    ^"\n  Expression: "^(pt2string e1))
		     | ((ExtractIndex _)::l, Appl(p, c as TypeConstr _, _)) => eqPrim(l,c,e2)
		     | (x::l, Appl(p, c as TypeConstr _, _)) => raise SympBug
			 ("eqFun/eqPrim: wrong extractor for TypeConstr application: "
			  ^(pt2string x)
			  ^"\n  Expression: "^(pt2string e1))
		     | ([], Appl(p, c as TypeConstr _, _)) => raise SympBug
			 ("eqFun/eqPrim: no extractor for TypeConstr application:\n  "
			  ^(pt2string e1))
		     | (l, Appl(p, e, e')) => eqPrim((Appl(p,Fake,e'))::l, e, e2)
		     (* Wrap back some extractor while we can *)
		     | ((ExtractTuple(n,_))::l, TupleExpr(_,lst)) => 
			 eqPrim(l, (case extractTuple n e1 of
					SOME e => e
				      | NONE => raise SympBug
					  ("eqFun: Tuple extraction failed for "
					   ^(pt2string e1))), e2)
		     | (x::_, TupleExpr _) => raise SympBug
			 ("eqFun/eqPrim: not a tuple extractor for a tuple: "^(pt2string x)
			  ^"\n  Expression: "^(pt2string e1))
		     | ([], TupleExpr _) => raise SympBug
			 ("eqFun/eqPrim: no extractor for a tuple: "
			  ^(pt2string e1))
		     | ((ExtractRecord(f,_))::l, RecordExpr(_,lst)) => 
			 eqPrim(l, (case extractRecord f e1 of
					SOME e => e
				      | NONE => raise SympBug
					  ("eqFun: Record extraction failed for "
					   ^(pt2string e1))), e2)
		     | (x::_, RecordExpr _) => raise SympBug
			 ("eqFun/eqPrim: not a record extractor for a record: "^(pt2string x)
			  ^"\n  Expression: "^(pt2string e1))
		     | ([], RecordExpr _) => raise SympBug
			 ("eqFun/eqPrim: no extractor for a record: "
			  ^(pt2string e1))
		     | ((ExtractRecord(f,_))::l, WithExpr(_, e1', lst)) => 
			 (case extractRecord f e1 of
					SOME e => eqPrim(l, e, e2)
		     (* If `f' is not in `lst', the `with' is irrelevant, take it off *)
				      | NONE => eqPrim(elist, e1', e2))
		     | (x::_, WithExpr _) => raise SympBug
			 ("eqFun/eqPrim: not a record extractor for a `with': "^(pt2string x)
			  ^"\n  Expression: "^(pt2string e1))
		     | ([], WithExpr _) => raise SympBug
			 ("eqFun/eqPrim: no extractor for a `with': "
			  ^(pt2string e1))
		     | ((Appl(p,_,e'))::l, FunClosure _) => eqPrim(l, evalExpr(Appl(p,e1,e')), e2)
		     | (x::_, FunClosure _) => raise SympBug
			 ("eqFun/eqPrim: not an Appl extractor for a `fn': "^(pt2string x)
			  ^"\n  Expression: "^(pt2string e1))
		     | ([], FunClosure _) => raise SympBug
			 ("eqFun/eqPrim: no extractor for a `fn': "
			  ^(pt2string e1))
		(* Now the compound statements *)
		     | (l, Object{def=e,...}) => eqPrim(l,e,e2)
		     | (l, NondetExpr(_,lst)) => foldOr(List.map(fn e=>eqPrim(l,e,e2)) lst)
		     | (l, Builtin{name=Anyvalue _,...}) => 
			 let val F = eqPrim([],bot(tp),e2)
			 in negFormula F
			 end
		     | (l, LetClosure{body=e,...}) => eqPrim(l, e, e2)
		     | (l, IfExpr(p,[],last)) => eqPrim(l, last, e2)
		     | (l, IfExpr(p,cnd::conds,last)) => 
			 let val F2 = eqPrim (l, IfExpr(p,conds,last), e2)
			     val (c,e) = (case cnd of
					      CondExpr(_,c,e) => (c,e)
					    | x => raise SympBug
						  ("eqFun/eqPrim: not a CondExpr: "
						   ^(pt2string x)))
			     val F1 = eqPrim(l, e, e2)
			     val Fbot = eqPrim([], bot(tp), e2)
			     val clist = List.map evalExpr (lift(absEval abs c))
			     val Cpairs = List.map(fn c=>(eqFun(c,true3Val),
							  eqFun(c, bot(BoolType dp)))) clist
			     fun doPair(c,cb) = caseFormula(c,F1,caseFormula(cb,Fbot,F2))
			 in foldOr(List.map doPair Cpairs)
			 end
		     | (l, CaseExpr(p,sel,cases)) =>
			 let val Fbot = eqPrim([], bot(tp), e2)
			     val selType = getExprType findObject sel
			     val sels = List.map evalExpr (lift(absEval abs sel))
			     fun doChoice sel (ChoiceClosure{pattern=pat, uname=u, body=e,...}) =
				 let val Fij = eqPrim(l, subst(u,sel) e, e2)
				     val Cij = matches(sel, pat)
				 in (Cij, Fij)
				 end
			       | doChoice _ x = raise SympBug
				 ("eqFun/eqPrim/doChoice: not a ChoiceClosure:\n "
				  ^(pt2string x))
			     fun doSel sel =
				 let val Cbot = eqFun(sel, bot selType)
				 in foldCase((Cbot, Fbot)::(List.map (doChoice sel) cases), Fbot)
				 end
			 in foldOr(List.map doSel sels)
			 end
		     | ([], ForallClosure _) => eqBuiltin(e1,e2)
		     | ([], ExistsClosure _) => eqBuiltin(e1,e2)
		     | (l, ForallClosure _) => raise SympBug
			 ("eqFun/eqPrim: extractor list is not empty for Forall:\n "
			  ^(pt2string e1)^"\n  Extractors: ["
			  ^(ptlist2str ", " l)^"]\n")
		     | (l, ExistsClosure _) => raise SympBug
			 ("eqFun/eqPrim: extractor list is not empty for Exists:\n "
			  ^(pt2string e1)^"\n  Extractors: ["
			  ^(ptlist2str ", " l)^"]\n")
		     | ([], True _) => eqPrim ([], true3Val,e2)
		     | ([], False _) => eqPrim ([], false3Val,e2)
		     | (l, True _) => raise SympBug
			 ("eqFun/eqPrim: extractor list is not empty for True:\n "
			  ^(pt2string e1)^"\n  Extractors: ["
			  ^(ptlist2str ", " l)^"]\n")
		     | (l, False _) => raise SympBug
			 ("eqFun/eqPrim: extractor list is not empty for False:\n "
			  ^(pt2string e1)^"\n  Extractors: ["
			  ^(ptlist2str ", " l)^"]\n")
		     | _ => raise SympBug
		       ("eqFun/eqPrim: sorry, this expression is not implemented yet: "
			^(pt2stringDebug e1)))
		    val _ = popFunStackLazy(funName, fn()=>(pt2string res))
		in res
		end (* end `eqPrim' *)
	    and wrapPrim atomic2 (elist,e1,e2) = 
		let val funName = "wrapPrim"
		    val debug = lazyVerbDebug options funName
		    val _ = pushFunStackLazy
			      (funName,
			       fn()=>("["^(ptlist2str ", " elist)^"],\n        "
				      ^(pt2stringDebug e1)^", "^(pt2stringDebug e2)
				      ^(if atomic2 then " <atomic>" else "")))
		    fun finish(e1,e2) =
		         if atomic2 then eqFormula findObject (e1, e2)
			 else eqPrim true ([], e2, e1)
		    fun doAppl(p, e', e, rest) =
			(* TODO: check if e is a state var, and construct D out of
			   `allVars' instead of `getTypeValues'. *)
			let val tp = getExprType findObject e'
			    val D = 
				if isConst e' then [e']
				else 
				 (case getTypeValues options lim tp of
				      SOME lst => lst
				    | NONE => raise TransError
					  ("array index type is too large "
					   ^"to list all its values:\n  "
					   ^(pt2string tp)
					   ^"\nin expression\n  "
					   ^(pt2string(Appl(p,e,e')))))
			    fun doOne d = 
				let val ed = evalExpr(Appl(p,e,d))
				    val Cd = eqFun(e',d)
				    val Fd = loop rest ed
				in impliesFormula(Cd,Fd)
				end
			    val Cbot = eqFun(e', bot tp)
			    val Fbot = finish(bot(getExprType findObject e2),e2)
			in foldAnd((impliesFormula(Cbot,Fbot))::(List.map doOne D))
			end
		    and loop [] e = finish(e,e2)
		      | loop ((ExtractAppl(c,_))::rest) e = loop rest (ExtractAppl(c,e))
		      | loop ((ExtractTuple(n,_))::rest) e = loop rest (ExtractTuple(n,e))
		      | loop ((ExtractRecord(f,_))::rest) e = loop rest (ExtractRecord(f,e))
		      | loop ((ExtractIndex _)::rest) e = loop rest (ExtractIndex e)
			(* We do not have built-in function that return non-primitive types,
			   so only a single extractor Appl can be here *)
		      | loop [Appl(p,_,e')] (f as Builtin{name=n, Type=tp}) =
			  eqBuiltin(evalExpr(Appl(p,f,e')), e2)
			  (* if isBoolFun n then builtinFun(f,e')
			  else raise SympBug
			    ("eqFun/wrapPrim: sorry, this built-in function is"
			     ^" not implemented yet: "^(pt2string f)) *)
		      | loop ((Appl(p,_,e'))::rest) e =
			(case getBuiltinCore e of
			     SOME f => eqBuiltin(evalExpr(Appl(p,f,e')), e2)
			   | NONE => doAppl(p, e', e, rest))
		      | loop (x::rest) _ = raise SympBug
			    ("wrapPrim: not an extractor: "^(pt2string x))
		    val res = loop elist e1
		    val _ = popFunStackLazy(funName,fn()=>(pt2string res))
 		in
		    res
		end
	    (* Wrap eqPrim into this function to guarantee its invariant *)
	    val eqPrim = fn(l, e1,e2) => eqPrim (isAtomic e2) (l, e1,e2)

		(* let val tp = getExprType findObject e1
		    val funtp = FunType(dp,TupleType(dp,[tp,tp]), BoolType dp)
		in Appl(dp, Builtin{name=Eq dp, Type=funtp}, TupleExpr(dp,[e1,e2]))
		end *)
	    fun loop tp =
		let val funName = "eqFun/loop"
		    val _ = pushFunStackLazy(funName, fn()=> pt2string tp)
		    val res = loop' tp
		    val _ = popFunStackLazy(funName, fn()=> pt2string res)
		in res
		end
	    and loop'(TupleType(_,lst)) =
		  let val nlist = List.tabulate(List.length lst, fn i=>i)
		      fun don n =
			  let val ei1 = evalExpr (ExtractTuple(n,e1))
			      val ei2 = evalExpr (ExtractTuple(n,e2))
			  in eqFun(ei1, ei2)
			  end
		  in foldAnd(List.map don nlist)
		  end
	      | loop'(RecordType(_,lst)) =
		  let fun doField (RecordField{name=f,...}) =
		          let val ef1 = evalExpr(ExtractRecord(f,e1))
			      val ef2 = evalExpr(ExtractRecord(f,e2))
			  in eqFun(ef1,ef2)
			  end
			| doField x = raise SympBug
			   ("eqFun/doField: not RecordField: "^(pt2string x))
		  in foldAnd(List.map doField lst)
		  end
	      | loop'(EnumType(_,lst)) =
		  let val eI1 = extractIndex e1
		      val eI2 = extractIndex e2
		      fun doC (c as Uid _) =
			   (case findObject c of
				SOME c1 => doC c1
			      | NONE => raise SympBug
				  ("eqFun/doC: no object for the Uid: "^(pt2string c)))
			| doC (c as TypeConstr{Type=tp,...}) =
			  let val C1 = eqPrim([],c,eI1)
			      val C2 = eqPrim([],c,eI2)
			      val lst = 
				  (case tp of
				       FunType _ => 
					   let val ei1 = evalExpr(ExtractAppl(c,e1))
					       val ei2 = evalExpr(ExtractAppl(c,e2))
					       val F = eqFun(ei1, ei2)
					   in [C1,C2,F]
					   end
				     | _ => [C1,C2])
			  in foldAnd lst
			  end
			| doC c = raise SympBug("eqFun/doC: not a TypeConstr/Uid: "^(pt2string c))
		  in foldOr(List.map doC lst)
		  end
	      | loop'(AbstractType _) =
		  let val funtp = FunType(dp,TupleType(dp,[tp,tp]), BoolType dp) 
		  in Appl(dp, Builtin{name=Eq dp, Type=funtp}, TupleExpr(dp,[e1,e2]))
		  end
	      | loop'(FunType(_,t1,t2)) =
		  (* TODO: check if e1 is a state var, and construct D out of
		     `allVars' instead of `getTypeValues'. *)
		  let val D = (case getTypeValues options lim t1 of
				   SOME lst => lst
				 | NONE => raise TransError
				       ("array index type is too large "
					^"to list all its values:\n  "
					^(pt2string t1)
					^"\nin expression\n  "
					^(pt2string(Appl(dp,Eq dp,TupleExpr(dp,[e1,e2]))))))
		      fun doOne d = 
			  let val ed1 = evalExpr(Appl(dp,e1,d))
			      val ed2 = evalExpr(Appl(dp,e2,d))
			  in eqFun(ed1,ed2)
			  end
		  in foldAnd(List.map doOne D)
		  end
	      | loop'(TypeClosure{recursive=false, def=t,...}) = loop t
	      | loop'(tp as TypeClosure{recursive=true, def=t,...}) = raise TransError
		  ("Cannot translate an (infinite) recursive type:\n  "
		   ^(pt2string tp)^" = "^(pt2string t))
	      | loop'(StaticFormalType{value=SOME t, ...}) = loop t
	      | loop'(t as StaticFormalType{value=NONE, ...}) = raise TransError
		  ("Cannot translate an uninterpreted type: "
		   ^(pt2string t))
	      | loop' (t as Uid _) =
		  (case findObject t of
		       SOME tp => loop tp
		     | NONE => raise SympBug
			   ("eqFun/loop: Uid type is not in the object hash: "^(pt2string t)))
	      (* The type is atomic *)
	      | loop' x = eqPrim([],e1,e2)
	    val res = evalExpr (loop tp)
	    val _ = popFunStackLazy(funName, fn()=>(pt2string res))
	in res
	end

    (* Generate a formula representing the transition relation given
       by the AsstVarsTree. *)

    (* Assumptions: all the variables in AsstVars are primitive, all
       the irrelevant vars are removed by the COI reduction (they
       don't hurt if finite, but we don't check the cone here). *)

    fun generateTrans options findObject abs pvars lim tree =
	let val debug = lazyVerbDebug options
	    val debugStr = lazyVerbDebugStr options
	    (* Compute the list of all relevant variables,
	       we'll need them in boolExpr, etc. *)
	    val {init=initvars, norm=normvars, next=nextvars} = getAsstVars tree
	    val allVars = List.map vtName (normvars@nextvars)
	    val lift = lift options findObject lim
	    (* val boolExpr = boolExpr options findObject allVars abs pvars lim *)
	    val eqFun = eqFun options findObject allVars abs pvars lim 
	    val matches = matches options findObject allVars abs pvars lim 
	    val evalExpr = evaluateExpr options findObject
	    val absEval = absEval options findObject
	    val eqFormula = eqFormula findObject
	    (* Keep the set of all vars around *)
	    val allVars = getAsstVars tree
	    (* All vars in the list get assigned Undefined *)
	    fun AssignUndef(vars,flag) =
		let fun doAsst v =
			let val vname = vtName v
			    val tp = getExprType findObject vname
			in asstGen flag ([v], vname, bot tp)
			end
		in foldAnd(List.map doAsst vars)
		end
	    (* All (vars: AsstVars) get assigned Undefined *)
	    and AssignUndefAll{norm=norm,next=next,init=init} =
		(AssignUndef(norm,NormalFlag),
		 AssignUndef(next,NextFlag),
		 AssignUndef(init,InitFlag))

	    (* Walk over the AsstVarsTree, recurse into the compound assignments.
	       Sequent N; L; I |- A ~> W; F; T *) 
	    and walkTree avt = 
		let val funName = "walkTree"
		    val _ = pushFunStackLazy(funName, fn()=>(avt2str avt))
		    val (normF, nextF, initF) = walkTree0 avt
		    val (normF, nextF, initF) = 
			   (evalExpr normF,
			    evalExpr nextF,
			    evalExpr initF)
		    val _ = popFunStackLazy(funName,
					    fn()=>("\n  INIT\n    "
						   ^(pt2string initF)
						   ^"\n  INVAR\n    "
						   ^(pt2string normF)
						   ^"\n  TRANS\n    "
						   ^(pt2string nextF)))
		in (normF, nextF, initF)
		end
	    and walkTree0 (NopAsstTree vars) =
		let val {norm=norm,next=next,init=init} = vars
		    fun doAsst flag v =
			let val vname = vtName v
			    val expr = (case flag of
					    NextFlag => vname
					  | _ => bot(getExprType findObject vname))
			in asstGen flag ([v], vname, expr)
			end
		    val normF = foldAnd(List.map(doAsst NormalFlag) norm)
		    val nextF = foldAnd(List.map(doAsst NextFlag) next)
		    val initF = foldAnd(List.map(doAsst InitFlag) init)
		in (normF, nextF, initF)
		end
	      | walkTree0 (NormalAsstTree(avars,stvar,expr)) =
		let val vars = (case avars of (* Sanity check *)
				    {norm=norm,next=[],init=[]} => norm
				  | {next=lst, init=[], ...} => raise SympBug
					("generateTrans/walkTree/normalAsst: "
					 ^"next vars non-empty:\n  "
					 ^(vtlist2str ", " lst))
				  | {init=lst, ...} => raise SympBug
				        ("generateTrans/walkTree/normalAsst: "
					 ^"init vars non-empty:\n  "
					 ^(vtlist2str ", " lst)))
		    val e = evalExpr expr
		    val F = asstGen NormalFlag (vars, stvar, e)
		in (F, trueVal, trueVal)
		end
	      | walkTree0 (NextAsstTree(avars,stvar,expr)) =
		let val vars = (case avars of (* Sanity check *)
				    {norm=[],next=next,init=[]} => next
				  | {norm=lst, init=[], ...} => raise SympBug
					("generateTrans/walkTree/nextAsst: "
					 ^"normal vars non-empty:\n  "
					 ^(vtlist2str ", " lst))
				  | {init=lst, ...} => raise SympBug
				        ("generateTrans/walkTree/nextAsst: "
					 ^"init vars non-empty:\n  "
					 ^(vtlist2str ", " lst)))
		    val e = evalExpr expr
		    val F = asstGen NextFlag (vars, stvar, e)
		in ( trueVal, F, trueVal)
		end
	      | walkTree0 (InitAsstTree(avars, stvar, expr)) =
		let val vars = (case avars of (* Sanity check *)
				    {norm=[],next=[],init=init} => init
				  | {next=lst, norm=[], ...} => raise SympBug
					("generateTrans/walkTree/initAsst: "
					 ^"next vars non-empty:\n  "
					 ^(vtlist2str ", " lst))
				  | {norm=lst, ...} => raise SympBug
				        ("generateTrans/walkTree/initAsst: "
					 ^"norm vars non-empty:\n  "
					 ^(vtlist2str ", " lst)))
		    val e = evalExpr expr
		    val F = asstGen InitFlag (vars, stvar, e)
		in (trueVal, trueVal, F)
		end
	      | walkTree0 (ListAsstTree(vars, tlist)) =
		let val parts = List.map walkTree tlist
		in (foldAnd(List.map(fn(f,_,_)=>f) parts),
		    foldAnd(List.map(fn(_,f,_)=>f) parts),
		    foldAnd(List.map(fn(_,_,f)=>f) parts))
		end		    
	      | walkTree0 (LetAsstTree(vars, defs, t)) = walkTree t
	      | walkTree0 (LabeledAsstTree(vars, label, t)) = walkTree t
	      | walkTree0 (IfAsstTree(vars, pairs, elsePart)) =
		let val cbot = bot(BoolType dp)
		    fun doCond v c =

			let val clist = List.map evalExpr (lift(absEval abs c))
			in List.map(fn c=> (eqFun(c,v), eqFun(c, cbot))) clist
			end
		    val Fpairs = List.map (fn(p,t) => (doCond true3Val p, walkTree t)) pairs
		    val last = walkTree elsePart
		    val (normF, nextF, initF) = AssignUndefAll vars
		    (* Recurse into the conditions: loop(parts,true_conds,undef_conds) *)
		    fun loop([]) = last
		      | loop((cl, (W,F,T))::rest) =
			let val (W2,F2,T2) = loop rest
			in (foldOr(List.map(fn (c,cb)=>
					    caseFormula(c,W,caseFormula(cb,normF,W2))) cl),
			    foldOr(List.map(fn (c,cb)=>
					    caseFormula(c,F,caseFormula(cb,nextF,F2))) cl),
			    foldOr(List.map(fn (c,cb)=>
					    caseFormula(c,T,caseFormula(cb,initF,T2))) cl))
			end
		in loop Fpairs
		end
	      | walkTree0 (CaseAsstTree(vars, sel, pairs)) =
		let val Fbot as (normF, nextF, initF) = AssignUndefAll vars
		    val selType = getExprType findObject sel
		    (* List of deterministic selectors *)
		    val selList = List.map evalExpr (lift(absEval abs sel))
		    (* Do one choice in the case statement *)
		    fun doPair sel (ChoiceAsstClosure arg, t) =
			let val {uname=u, pattern=pat, ...} = arg
			    val triple = walkTree(substAsstTree(u,sel) t)
			    val C = matches(sel, pat)
			in (C, triple)
			end
		      | doPair sel (x, _) = raise SympBug
			  ("generateTrans/walkTree/CaseAsst: not a ChoiceAsstClosure: "
			   ^(pt2string x))
		    fun doSel sel =
			let val botPair = (matches(sel, bot selType), Fbot)
			    val Fpairs = botPair::(List.map (doPair sel) pairs)
			    val normPairs = List.map(fn(C,(F,_,_))=>(C,F)) Fpairs
			    val nextPairs = List.map(fn(C,(_,F,_))=>(C,F)) Fpairs
			    val initPairs = List.map(fn(C,(_,_,F))=>(C,F)) Fpairs
			in
			    (foldCase(normPairs, normF),
			     foldCase(nextPairs, nextF),
			     foldCase(initPairs, initF))
			end
		    val triples = List.map doSel selList
		    val W = foldOr(List.map(fn(f,_,_)=>f) triples)
		    val F = foldOr(List.map(fn(_,f,_)=>f) triples)
		    val T = foldOr(List.map(fn(_,_,f)=>f) triples)
		in (W,F,T)
		end
	      | walkTree0 (ChooseAsstTree(vars, paramsOpt, pairs)) =
		let val (W0,F0,T0) = walkTree(NopAsstTree vars)
		    val debug = debug "walkTree"
		    fun doPair (p, t) =
			let val C = eqFun(p, true3Val)
			    val F = walkTree t
			in (C, F)
			end
		    fun getPairs NONE pairs = pairs
		      | getPairs (SOME params) pairs =
			  substVarValuesPairs options findObject abs pvars lim params pairs
		    fun part2str(c,(W,F,T)) = 
			((pt2string c)
			 ^" => (norm = "^(pt2string W)
			 ^", next = "^(pt2string F)
			 ^", init = "^(pt2string T)^")")
		    fun pairs2strFn pairs () =
			"\nwalkTree: pairs = [\n  "
			^(strlist2str "\n  " (List.map(fn (c,t)=>
						       (pt2string c)^" => "^(avt2str t)) pairs))
			^"]\n"
		    fun parts2strFn parts () = 
			 "\nwalkTree: parts = [\n  "
			 ^(strlist2str "\n  " (List.map part2str parts))
			 ^"]\n"
		    fun ChooseFold(pairs, F0) =
			let fun isTruePair(_, Builtin{name=True _,...}) = true
			      | isTruePair(_, True2) = true
			      | isTruePair _ = false
			    val matchList = List.map(fn(c,_)=>c) pairs
			in
			    if List.all isTruePair pairs then
				trueVal
			    else
				foldOr[foldOr(List.map(fn(e,f)=>foldAnd[e,f]) pairs),
				       foldAnd[foldAnd(List.map negFormula matchList), F0]]
			end
		    fun doPairs pairs =
			let val _ = debug(pairs2strFn pairs)
			    val parts = List.map doPair pairs
			    val _ = debug(parts2strFn parts)
			in
			    (ChooseFold(List.map(fn(g, (f,_,_))=>(g, f)) parts,W0),
			     ChooseFold(List.map(fn(g, (_,f,_))=>(g, f)) parts,F0),
			     ChooseFold(List.map(fn(g, (_,_,f))=>(g, f)) parts,T0))
			end
		    fun rebuild tree = 
			let val {init=init, next=next, norm=norm} = getAsstVars tree
			    val stateVars = union ptEq 
				    (List.map(List.map vtName) [init, next, norm])
			    val opt = rebuildAsstOptionsDefault
			    val (res, newCone, newPvars) = 
				rebuildAsstVars options findObject pvars lim
				                (SOME stateVars, tree, opt)
			in res
			end

		    val newPairs = getPairs paramsOpt pairs
		in 
		    case paramsOpt of
			SOME _ => walkTree(rebuild(ChooseAsstTree(vars, NONE, newPairs)))
		      | NONE => doPairs newPairs
		end
	      | walkTree0 (ForeachAsstTree(vars, params, tree)) = raise SympBug
		  ("TransGen/walkTree: sorry, FOREACH assignment is not implemented yet")

	    (* Translate the atomic assignments.
	       Sequent V |- g := e ~> F *)
	    and asstGen (flag: VarTypeFlag)
		        (vars: VarType list, stvar: ParseTree, expr: ParseTree) = 
		(* Strip ObjectInst from `stvar' and `expr', so we have type constructors
		   "naked" for easier comparison. *)
		let val funName = "asstGen"
		    val _ = pushFunStackLazy(funName, 
		      fn()=>((vtlist2str "," vars)^" |- "
			     ^(vt2str(vtWrap flag stvar))^" := "^(pt2string expr)))
		    val _ = debug "asstGenDebug" 
		      (fn()=>"debug: asstGen("^(vtlist2str "," vars)^" |- "
		       ^(vt2str(vtWrap flag stvar))^" := "^(pt2stringDebug expr)^")\n")
		    val F = evalExpr(asstGen1 flag (vars, [],
						    stripObjectInst stvar,
						    stripObjectInst expr))
		    val _ = debug "asstGenDebug" (fn()=>"debug: asstGen => "
						  ^(pt2stringDebug F)^"\n")
		    val _ = popFunStackLazy(funName, fn()=>(pt2string F))
		in F
		end
	    (* First stage: recurse into `e', push down extractors.
	       Sequent V; S |- g << e *)
	    (* First, "extraction introduction" rules *)
	    and asstGen1 flag (vars, extractList, stvar, expr) =
		let fun argsFn() = 
		         ((vtlist2str "," vars)^" ; "
			  ^(ptlist2str ", " extractList)^" |- "
			  ^(vt2str(vtWrap flag stvar))^" << "^(pt2string expr))
		    val _ = pushFunStackLazy("asstGen1", argsFn)		      
		    val _ = debug "asstGen1Debug" 
			(fn()=>"asstGen1("^(vtlist2str "," vars)^" ; "
			 ^(ptlist2strDebug ", " extractList)^" |- "
			 ^(vt2str(vtWrap flag stvar))^" << "^(pt2stringDebug expr)^")\n")
		    val F = evalExpr(asstGen1' flag (vars, extractList, stvar, expr))
		    val _ = debug "asstGen1Debug" (fn()=>"asstGen1 => "^(pt2stringDebug F)^"\n")
		    val _ = popFunStackLazy("asstGen1", fn()=>pt2string F)
		in F
		end
	    and asstGen1' flag (vars, extractList, stvar, ExtractRecord(f,expr)) =
		  asstGen1 flag (vars, (ExtractRecord(f,Fake))::extractList, stvar, expr)
	      | asstGen1' flag (vars, extractList, stvar, ExtractTuple(n,expr)) =
		  asstGen1 flag (vars, (ExtractTuple(n,Fake))::extractList, stvar, expr)
	      | asstGen1' flag (vars, extractList, stvar, ExtractAppl(c,expr)) =
		  asstGen1 flag (vars, (ExtractAppl(c,Fake))::extractList, stvar, expr)
	      | asstGen1' flag (vars, extractList, stvar, ExtractIndex expr) =
		  asstGen1 flag (vars, (ExtractIndex Fake)::extractList, stvar, expr)
	      (* "Extraction elimination rules" *)
	      | asstGen1' flag (vars, (ExtractRecord(f,_))::extractList, stvar, RecordExpr(_,lst)) =
		  let fun findField ((RecordAsst(_,name,e))::lst) = 
		           if ptEq(f,name) then e else (findField lst)
			| findField (x::_) = raise SympBug
			   ("generateTrans/asstGen1: not RecordAsst: "^(pt2string x))
			| findField [] = raise SympBug
			   ("generateTrans/asstGen1: no such field in a record: "
			    ^(pt2string f))
		  in asstGen1 flag (vars, extractList, stvar, findField lst)
		  end
	      | asstGen1' flag (vars, (ExtractTuple(n,_))::extractList, stvar,
			       e as TupleExpr(_,lst)) =
		  if (List.length lst) > n then
		      asstGen1 flag (vars, extractList, stvar, List.nth(lst,n))
		  else raise SympBug
		      ("generateTrans/asstGen1: tuple is shorter than extractor: "
		       ^(pt2string e))
	      (* Treat application carefully, it matches several cases *)
	      | asstGen1' flag (vars, elst, stvar, Appl(p, e1, e2)) =
		  (case (elst, e1) of
		       (* ExtractAppl Elim. rule *)
		       ((ExtractAppl(c,_))::lst, TypeConstr{name=name,...}) =>
			   if ptEq(c,name) then
			       asstGen1 flag (vars, lst, stvar, e2)
			   else  raise TransError
			       ("generateTrans/asstGen1: type constructor "
				^(pt2string name)
				^", but trying to extract "^(pt2string c))
		     | ((ExtractIndex _)::lst, c as TypeConstr _) =>
			   asstGen1 flag (vars, lst, stvar, c)

		     (* Other Appl rules: just leave application alone.  The cases on
		        argument's values will be split at the later stage (boolExpr) if
		        necessary. *)

		     | _ => asstGen1 flag (vars, (Appl(p,Fake,e2))::elst, stvar, e1))
	      (* Appl Elim. rule *)
	      | asstGen1' flag (vars, (Appl(p, _, d))::lst, stvar, e as FunClosure _) =
		  asstGen1 flag (vars, lst, stvar, evalExpr (Appl(p,e,d)))
	      | asstGen1' flag (vars, lst, stvar, NondetExpr(_,elist)) =
		  foldOr(List.map(fn e=>asstGen1 flag (vars, lst, stvar, e)) elist)
	      | asstGen1' flag (vars, lst, stvar, LetClosure{body=e,...}) =
		  asstGen1 flag (vars, lst, stvar, e)
	      | asstGen1' flag (vars, lst, stvar, IfExpr(_,[],last)) =
		  asstGen1 flag (vars, lst, stvar, last)
	      | asstGen1' flag (vars, lst, stvar, IfExpr(p,cnd::conds,last)) =
		  let val F2 = asstGen1 flag (vars, lst, stvar, IfExpr(p,conds,last))
		      val (c,e) = (case cnd of
				       CondExpr(_,c,e) => (c,e)
				     | x => raise SympBug
					("generateTrans/asstGen1: not a CondExpr: "
					 ^(pt2string x)))
		      val F1 = asstGen1 flag (vars, lst, stvar, e)
		      val Fbot = asstGen1 flag (vars, [], stvar, bot(getExprType findObject stvar))
		      val clist = List.map(evalExpr)(lift(absEval abs c))
		      val Cpairs = List.map(fn c=>(eqFun(c,true3Val),
						   eqFun(c, bot(BoolType dp)))) clist
		  in foldOr(List.map(fn(c,cb)=>caseFormula(c,F1,caseFormula(cb,Fbot,F2))) Cpairs)
		  end
	      | asstGen1' flag (vars, lst, stvar, CaseExpr(p,sel,cases)) =
		  let val Fbot = asstGen1 flag (vars, [], stvar, bot(getExprType findObject stvar))
		      val selType = getExprType findObject sel
		      val sels = List.map(evalExpr)(lift(absEval abs sel))
		      fun doChoice sel (ChoiceClosure{pattern=pat, uname=u, body=e,...}) =
			  let val Fij = asstGen1 flag (vars, lst, stvar, subst(u,sel) e)
			      val Cij = matches(sel, pat)
			  in (Cij, Fij)
			  end
			| doChoice _ x = raise SympBug
			   ("generateTrans/asstGen1/doChoice: not a ChoiceClosure:\n "
			    ^(pt2string x))
		      fun doSel sel =
			  let val Cbot = eqFun(sel, bot selType)
			  in foldCase((Cbot, Fbot)::(List.map (doChoice sel) cases), Fbot)
			  end
		  in foldOr(List.map doSel sels)
		  end
	      | asstGen1' flag (vars, _, stvar, ChooseClosure{names=names, body=e,...}) =
		  raise SympBug("generateTrans/asstGen1: Sorry, `pick' is not implemented yet.")
	      | asstGen1' flag (vars, lst, stvar, expr) =
		  let (* Check whether state var expr. has non-constant expressions in it *)
		      fun hasNonval (StateVar _) = false
			| hasNonval (DynPatternFormal _) = false
			| hasNonval (ExtractTuple(_,v)) = hasNonval v
			| hasNonval (ExtractRecord(_,v)) = hasNonval v
			| hasNonval (ExtractAppl(_,v)) = hasNonval v
			| hasNonval (ExtractIndex v) = hasNonval v
			| hasNonval (Appl(_,v,e)) =
			    not (isConst e) orelse hasNonval v
			| hasNonval x = raise SympBug
			    ("generateTrans/asstGen1/hasNonval: not a state var expression:\n  "
			     ^(pt2string x))
		      fun defaultRule newExpr =
			  if hasNonval stvar then (* default_2 *)
			      asstGen2 flag (vars, [], stvar, newExpr)
			  else (* default_1 *)
			      asstGen4 flag (vars, stvar, newExpr)
		  in 
		      (* case (lst,expr) of
			   ([], e as ForallClosure{names=names, body=e, ...}) =>
			       let val eds = doQuant evalExpr (names, e)
				   fun recur e = asstGen1 flag (vars, lst, stvar, e)
				   val Fds = List.map recur eds
			       in
				   foldAnd Fds
			       end
			 | (_, ForallClosure _) => raise SympBug
			       ("generateTrans/asstGen1: Extractor list is not empty for `forall'")
			 | ([],e as ExistsClosure _) => defaultRule (boolExpr e)
			 | (_, ExistsClosure _) => raise SympBug
			       ("generateTrans/asstGen1: Extractor list is not empty for `exists'")
			 (* Default rules *)
			 |  (_, _) => *)
			    let val newExpr = List.foldr wrapExtr expr (List.rev lst)
			    in 
				defaultRule newExpr
			    end
		  end

	    (* Second stage: recurse into `gamma' and collect all the extractors 
	       Sequent V; S |- _gamma g << e *)
	    and asstGen2 flag (vars, extractList, stvar, expr) =
		let fun argsFn() = 
		         ((vtlist2str "," vars)^" ; "
			  ^(ptlist2str ", " extractList)^" |- "
			  ^(vt2str(vtWrap flag stvar))^" << "^(pt2string expr))
		    val _ = pushFunStackLazy("asstGen2", argsFn)
		    val F = evalExpr(asstGen2' flag (vars, extractList, stvar, expr))
		    val _ = popFunStackLazy("asstGen2", fn()=>pt2string F)
		in F
		end
	    and asstGen2' flag (vars, elist, ExtractRecord(f,g), expr) =
		  asstGen2 flag (vars, (ExtractRecord(f,Fake))::elist, g, expr)
	      | asstGen2' flag (vars, elist, ExtractTuple(n,g), expr) =
		  asstGen2 flag (vars, (ExtractTuple(n,Fake))::elist, g, expr)
	      | asstGen2' flag (vars, elist, ExtractAppl(c,g), expr) =
		  asstGen2 flag (vars, (ExtractAppl(c,Fake))::elist, g, expr)
	      | asstGen2' flag (vars, elist, ExtractIndex g, expr) =
		  asstGen2 flag (vars, (ExtractIndex Fake)::elist, g, expr)
	      | asstGen2' flag (vars, elist, Appl(p,g,e), expr) =
		  asstGen2 flag (vars, (Appl(p,Fake,e))::elist, g, expr)
	      (* g is atomic *)
	      | asstGen2' flag args = asstGen3 flag args

	    (* Wrap back the extractors from stage 2 and create
	       multiple assignments for dynamic indices.
	       Sequent V; S |- _delta g << e *)
	    and asstGen3 flag (vars, extractList, stvar, expr) =
		let fun argsFn() =
		         ((vtlist2str "," vars)^" ; "
			  ^(ptlist2str ", " extractList)^" |- "
			  ^(vt2str(vtWrap flag stvar))^" << "^(pt2string expr))
		    val _ = pushFunStackLazy("asstGen3", argsFn)
		    val F = evalExpr(asstGen3' flag (vars, extractList, stvar, expr))
		    val _ = popFunStackLazy("asstGen3", fn()=>pt2string F)
		in F
		end
	    and asstGen3' flag (vars, [], g, expr) = asstGen4 flag (vars, g, expr)
	      | asstGen3' flag (vars, (ExtractRecord(f,_))::elist, g, expr) =
		  asstGen3 flag (vars, elist, ExtractRecord(f,g), expr)
	      | asstGen3' flag (vars, (ExtractTuple(n,_))::elist, g, expr) =
		  asstGen3 flag (vars, elist, ExtractTuple(n,g), expr)
	      | asstGen3' flag (vars, (ExtractAppl(c,_))::elist, g, expr) =
		  asstGen3 flag (vars, elist, ExtractAppl(c,g), expr)
	      | asstGen3' flag (vars, (ExtractIndex _)::elist, g, expr) =
		  asstGen3 flag (vars, elist, ExtractIndex g, expr)
	      | asstGen3' flag (vars, (Appl(p,_,e))::elist, g, expr) =
		  let val Dpairs = splitByArrayIndex(vars, g)
		      val Bot = bot(getExprType findObject expr)
		      val es = lift e
		      fun doPair e (d,V)= 
			  let val Cd = eqFun(e,d)
			      val gd = Appl(p,g,d)
			      val Fd = asstGen3 flag (V, elist, gd, expr)
			      val Ud = 
				(case flag of
				     NextFlag => asstGen4 flag (V, gd, gd)
				   | _ => asstGen3 flag (V, elist, gd, Bot))
			  in (Cd, Fd, Ud)
			  end
		      fun doE e =
			  let val triples = List.map (doPair e) Dpairs
			      type Triple = ParseTree * ParseTree * ParseTree
			      fun makeCase(Cd,Fd,Ulist: Triple list) =
				   impliesFormula(Cd,foldAnd(Fd::(List.map #3 Ulist)))
			      fun loop old [] = raise SympBug
				   ("generateTrans/asstGen3: state var is not in vars: "
				    ^(pt2string g))
				| loop old [(Cd,Fd,Ud)] = [makeCase(Cd, Fd, old)]
				| loop old ((x as (Cd,Fd,Ud))::new) = 
				   (makeCase(Cd, Fd, old@new))::(loop (x::old) new)
			  in foldAnd(loop [] triples)
			  end
		  in foldOr(List.map doE es)
		  end
	      | asstGen3' flag (vars, x::_, _, _) = raise SympBug
		  ("generateTrans/asstGen3: bad extractor in the list: "
		   ^(pt2string x))

	    (* Expand the type of `stvar' if it is not primitive, and
	       proceed with stage 1 again.  Assumption: `stvar' does
	       not have expressions as indices in it.
	       Sequent V |- _gamma g := e ~> F *)
	    and asstGen4 flag (vars, stvar, expr) =
		let fun argsFn() = 
		          ((vtlist2str "," vars)^" |-_g "
			   ^(vt2str(vtWrap flag stvar))^" := "^(pt2string expr))
		    val _ = pushFunStackLazy("asstGen4", argsFn)
		    val F = evalExpr(asstGen4' flag (vars, stvar, expr))
		    val _ = popFunStackLazy("asstGen4", fn()=>pt2string F)
		in F
		end
	    and asstGen4' flag (vars, stvar, expr) =
		let val tp = getExprType findObject stvar
		    fun wrapFlagPt NormalFlag v = v
		      | wrapFlagPt NextFlag v = Next v
		      | wrapFlagPt InitFlag v = v
		    fun doPair(d,V) =
			let val ed = evalExpr (Appl(dp,expr,d))
			in asstGen1 flag (V, [], Appl(dp,stvar,d), ed)
			end
		    fun doFun() = foldAnd(List.map doPair (splitByArrayIndex(vars, stvar)))
		    fun doRecField(RecordField{name=f,...}) =
			let val ef = evalExpr (ExtractRecord(f,expr))
			    val gf = ExtractRecord(f,stvar)
			    val Vf = List.filter(fn x=>svPartOf(vtName x,gf)) vars
			in asstGen1 flag (Vf, [], gf, ef)
			end
		      | doRecField x = raise SympBug
			 ("generateTrans/asstGen4: not a RecordField: "
			  ^(pt2string x))
		    fun doRecord lst = foldAnd(List.map doRecField lst)
		    fun doTupleEl(n,e) =
			let val en = evalExpr (ExtractTuple(n,e))
			    val gn = ExtractTuple(n,stvar)
			    val Vn = List.filter(fn x=>svPartOf(vtName x,gn)) vars
			in asstGen1 flag (Vn, [], gn, en)
			end
		    fun doTuple lst = foldAnd(List.map doTupleEl 
					      (zip(List.tabulate(List.length lst, fn x=>x),
						   lst)))
		    (* For EnumType, an `expr' may be of the form (C e') for a particular
		       type constructor C, and evaluateExpr may raise an exception in
		       cases other than C.  When this happens, simply ignore those C
		       (return NONE) and do not include this case in the final
		       formula. *)
		    fun doEnumEl expr index (c as TypeConstr{Type=tp,...}) =
			let val I = eqFun(wrapFlagPt flag index, c)
			    val pat = (case tp of
					   FunType _ => ApplPattern(dp,c,Underscore dp)
					 | _ => c)
			    val C = matches(expr, pat)
			in (case tp of
				FunType _ => 
				  let val ec = evalExpr (ExtractAppl(c,expr))
				      val gc = ExtractAppl(c,stvar)
				      val V = List.filter(fn x=>svPartOf(vtName x,gc)) vars
				      val F = asstGen1 flag (V, [], gc, ec)
				  in SOME(impliesFormula(C, foldAnd[I, F]))
				  end
			      | _ => SOME(impliesFormula(C, I)))
			    handle EvalError _ => NONE
			end
		      | doEnumEl _ _ x = raise SympBug
			 ("generateTrans/asstGen4: not a TypeConstr: "
			  ^(pt2stringDebug x))
		    fun doEnum lst =
			let val index = (case getPrimitiveIndexVar pvars stvar of
					     SOME x => x
					   | NONE => raise SympBug
					      ("generateTrans/asstGen4: no index var for "
					       ^(pt2stringDebug stvar)))
			    val Fbot = eqFun(wrapFlagPt flag index,
					     bot(getExprType findObject index))
			    val es = List.map(evalExpr)(lift(absEval abs expr))
			    fun doE e =
				let val Cbot = eqFun(e,bot(getExprType findObject expr))
				in foldAnd((impliesFormula(Cbot,Fbot))
					   ::(List.mapPartial (doEnumEl e index) lst))
				end
			in foldOr(List.map doE es)
			end
		    fun splitType(FunType _) = doFun()
		      | splitType (ArrayType _) = doFun()
		      | splitType (RecordType(_,lst)) = doRecord lst
		      | splitType (TupleType(_,lst)) = doTuple lst
		      | splitType (EnumType(_,lst)) = doEnum lst
		      | splitType (StaticFormalType{value=SOME t,...}) = splitType t
		      | splitType (t as StaticFormalType{value=NONE, ...}) = raise TransError
			  ("Cannot translate an assignment with an uninterpreted type: "
			   ^(pt2string t))
		      | splitType (TypeClosure{recursive=false, def=t,...}) = splitType t
		      | splitType (tp as TypeClosure{recursive=true, def=t,...}) = raise TransError
			  ("Cannot translate an assignment with an (infinite) recursive type:\n  "
			   ^(pt2string tp)^" = "^(pt2string t))
		      | splitType (t as Uid _) =
			  (case findObject t of
			       SOME tp => splitType tp
			     | NONE => raise SympBug
				   ("generateTrans/genAsst4: Uid type is not in object hash: "
				    ^(pt2string t)))
		      (* Type is primitive *)
		      | splitType _ = eqFun(wrapFlagPt flag stvar, expr)
		in (case getTypeAbstraction options abs tp of
			(* Type is abstracted, thus,  stvar is primitive. Wrap up. *)
			SOME _ => eqFun(wrapFlagPt flag stvar, expr)
		      (* Split the type *)
		      | NONE => splitType tp)
		end
	    val (normF, nextF, initF) = walkTree tree
	    fun checkTrue (f as Builtin{name=True _, ...}) = 
		  Appl(dp,Builtin{name=Eq dp,
				  Type=FunType(dp,TupleType(dp,[BoolType dp,BoolType dp]),
					       BoolType dp)},
		       TupleExpr(dp,[f,f]))
	      | checkTrue f = f
	in (checkTrue normF, checkTrue nextF, checkTrue initF)
	end

    (* Translate a theorem into a primitive boolean formula.
       Currently, only theorems of the form `self |= e' are handled *)
    fun generateTheorem options findObject vars abs pvars lim (Theorem(_,_,Models(_, _, e))) =
	let val evalExpr = evaluateExpr options findObject
	in 
	    eqFun options findObject vars abs pvars lim (evalExpr e, true3Val)
	end
      | generateTheorem _ _ _ _ _ _ x = raise SympBug
	  ("generateTheorem: sorry, this theorem statement is not implemented yet:\n  "
	   ^(pt2string x))

    (* Translate `e = true' into a primitive 2-valued boolean formula *)
    fun boolExpr options findObject vars abs pvars lim e =
	let val evalExpr = evaluateExpr options findObject
	in 
	    eqFun options findObject vars abs pvars lim (evalExpr e, true3Val)
	end

    (* Gerenate a formula for keeping all the values of `next'
       variables unchanged in the next state.  Only one formula is
       generated and returned. *)

    fun generateStallRel options findObject abs trans =
	let val { limit=lim,...} = options
	    fun loop (TransAtomic(AtomicModel model)) =
		let val { assts=assts,
			  cone=cone,
			  pvars=pvars, ... } = model
		    val generateTrans = generateTrans options findObject abs pvars lim
		    val vars as {next=nextvars, ...} = getAsstVars assts
		    (* Generate assignments `next(x) := x' for all `next' vars *)
		    fun doVars trees ((v as NextVar n)::vars) =
			  doVars ((NextAsstTree({norm=[], next=[v], init=[]}, n, n))::trees) vars
		      | doVars trees [] = ListAsstTree(vars, trees)
		      | doVars _ (v::_) = raise SympBug
			  ("TransGen/generateStallRel: not a `next' var: "
			   ^(vt2str v))
		    val newAssts = doVars [] nextvars
		    val (_, nextTr, _) = generateTrans newAssts
		in
		    nextTr
		end
	      | loop (TransSync2(tr1,tr2)) = foldAnd[loop tr1, loop tr2]
	      | loop (TransAsync2(tr1, tr2)) = foldAnd[loop tr1, loop tr2]
	      | loop  _ = raise SympBug
		  ("TransGen/generateStallRel: sorry, closed form composition is not implemented yet")
	in
	    loop trans
	end
	
    (* Generate transition relation for a composition of models represented by TransRel *)
    fun generateTransRel options findObject abs trans =
	let val { limit=lim,...} = options
	    val generateTransRel = generateTransRel options findObject abs
	    val generateStallRel = generateStallRel options findObject abs
	    fun loop (TransAtomic(AtomicModel model)) =
		 let val { assts=assts,
			   cone=cone,
			   pvars=pvars, ... } = model
		 in 
		     generateTrans options findObject abs pvars lim assts
		 end
	      (* Synchronous parallel composition: symply conjunct all the formulas *)
	      | loop (TransSync2(tr1, tr2)) = 
		 let val (norm1, next1, init1) = generateTransRel tr1
		     val (norm2, next2, init2) = generateTransRel tr2
		 in
		     (foldAnd[norm1, norm2],
		      foldAnd[next1, next2],
		      foldAnd[init1, init2])
		 end

	      (* Asynchronous composition is trickier.  First, we
	         generate an extra transition relation tr{1,2}' for
	         stalling each module (all latched vars remain the
	         same), and then put them together as follows:

                 (tr1 & tr2') | (tr2 & tr1')

		 That is, no selector variable is generated.  The
		 invariants and initial states are conjuncted as in
		 the synchronous case (??? Is this a reasonable
		 semantics ???) *)

	      | loop (TransAsync2(tr1, tr2)) =
		 let val (norm1, next1, init1) = generateTransRel tr1
		     val (norm2, next2, init2) = generateTransRel tr2
		     val stall1 = generateStallRel tr1
		     val stall2 = generateStallRel tr2
		 in
		     (foldAnd[norm1, norm2],
		      foldOr[foldAnd[next1, stall2], foldAnd[stall2, next1]],
		      foldAnd[init1, init2])
		 end
		      
              | loop _ = raise SympBug
		 ("TransGen/generateTransRel: sorry, closed form parallel composition is not implemented yet")
	in
	    loop trans
	end

    fun generateModelTrans options (model: Model) =
	let val { trans=trans,
		  findObject=findObject,
		  abs=abs,... } = model
	in 
	    generateTransRel options findObject abs trans
	end

  end
