functor TransCommonFun(structure Hash: HASH
		       structure Abstract: ABSTRACT
		       structure Evaluate: EVALUATE
		       sharing Abstract.ParserDefault = Evaluate.ParserDefault): TRANS_COMMON =
  struct
    structure Hash = Hash
    structure Abstract = Abstract
(*     structure Interface = Abstract.Interface
    structure Evaluate = EvaluateFun(structure Interface = Interface) *)
    structure Evaluate = Evaluate

    open Str
    open ListOps
    open Hash
    open Abstract
    open Pos
    open Evaluate
    open ParserDefault
    open Interface
    open Options
    open ParseTreeStruct

    datatype VarType =
	NormalVar of ParseTree
      | NextVar of ParseTree
      | InitVar of ParseTree

    datatype ConeVar =
	PVar of ParseTree
      | DelayedVar of ParseTree (* Delayed computation of cone, DelayedVar(expr) *)

    (* Type for a flag that tells what type of assigmnent we are
       dealing with.  *)
    datatype VarTypeFlag = NormalFlag | NextFlag | InitFlag
    datatype ConeVarFlag = DelayedFlag | PVarFlag

    type Cone = (VarType, ConeVar list) Hash

    (* correspondence b/w SyMP state vars and primitive vars in the translation.
       Implemented as a hash mapping a state var to its list of primitive vars. *)
    type PrimitiveVars = {vars: (ParseTree, ParseTree list) Hash.Hash,
			  indexVars: (ParseTree, ParseTree) Hash.Hash}

    type AsstVars = {norm: VarType list,
		     next: VarType list,
		     init: VarType list}

    (* Representation of sequent N;L;I |- A1;...;An *)
    (* type AsstSeq = AsstVars * (ParseTree list) *)

    datatype AsstVarsTree =
        (* vars |- varName := expr *)
	NormalAsstTree of AsstVars * ParseTree * ParseTree
      | NextAsstTree of AsstVars * ParseTree * ParseTree
      | InitAsstTree of AsstVars * ParseTree * ParseTree
      | NopAsstTree of AsstVars
      | ListAsstTree of AsstVars * (AsstVarsTree list)
        (* vars |- let [defs] in <asst> end *)
      | LetAsstTree of AsstVars * (ParseTree list) * AsstVarsTree
        (* vars |- case selExpr of [(pat1, asst1), ..., (patN, asstN)] endcase *)
      | CaseAsstTree of AsstVars * ParseTree * ((ParseTree * AsstVarsTree) list)
        (* vars |- if c1 then asst1 elsif c2 then asst2 .... else asstN *)
      | IfAsstTree of AsstVars * ((ParseTree * AsstVarsTree) list) * AsstVarsTree
      | ChooseAsstTree of AsstVars * (ParseTree list option) * ((ParseTree * AsstVarsTree) list)
      | ForeachAsstTree of AsstVars * (ParseTree list) * AsstVarsTree
      | LabeledAsstTree of AsstVars * ParseTree * AsstVarsTree

    (* Transition relation of a leaf module (not a composition of any other modules) *)
    datatype AtomicModel =
	AtomicModel of { name: ParseTree option,
			 uname: ParseTree,
			 assts: AsstVarsTree,
			 cone: Cone,
			 pvars: PrimitiveVars,
			 (* We might not need this list, as we have the master list *)
			 (* stateVars: ParseTree list *)
			 (* Abstraction modules nested into the current module.
			    These modules may have further abstractions inside them. *)
			 absModules: AtomicModel list
		       }

    (* Transition relation for the model representaiton *)
    datatype TransRel = 
	(* Parallel composition of two modules *)
	TransSync2 of TransRel * TransRel
      | TransAsync2 of TransRel * TransRel
	(* Parallel composition over the parameters' ranges *)
      | TransSync of { names: ParseTree list,
		       body: TransRel,
		       parent: ParseTree }
      | TransAsync of { names: ParseTree list,
		        body: TransRel,
			parent: ParseTree }
      | TransAtomic of AtomicModel

    (* Representation of a model in the sequent *)
    type Model = { trans: TransRel,
		   findObject: ParseTree -> ParseTree option,
		   (* List of StateVar objects *)
		   stateVars: ParseTree list,
		   (* The united set of primitive variables *)
		   pvars: PrimitiveVars,
		   abs: Abstraction }

    (* Check if the expression is a constant for the purposes of COI reduction *)
    fun isConst(StaticFormalConst _) = true
      | isConst(SkolemConst _) = true
      | isConst x = isValue x

    (* Check if l1 is a subset of l2 w.r.t. the equality `eq' *)
    fun subset eq (l1,l2) = List.all(fn x=>List.exists (fn y=>eq(x,y)) l2) l1
    fun eqset eq (l1,l2) = subset eq (l1,l2) andalso subset eq (l2,l1)

    fun vt2str(NormalVar pt) = "NormalVar("^(pt2string pt)^")"
      | vt2str(NextVar pt) = "NextVar("^(pt2string pt)^")"
      | vt2str(InitVar pt) = "InitVar("^(pt2string pt)^")"

    fun vt2strDebug(NormalVar pt) = "NormalVar("^(pt2stringDebug pt)^")"
      | vt2strDebug(NextVar pt) = "NextVar("^(pt2stringDebug pt)^")"
      | vt2strDebug(InitVar pt) = "InitVar("^(pt2stringDebug pt)^")"

    fun vtlist2str sep [] = ""
      | vtlist2str sep [hd] = vt2str hd
      | vtlist2str sep (hd::tl) = (vt2str hd)^sep^(vtlist2str sep tl)

    fun vtEq(NormalVar v1, NormalVar v2) = ptEq(v1,v2)
      | vtEq(NextVar v1, NextVar v2) = ptEq(v1,v2)
      | vtEq(InitVar v1, InitVar v2) = ptEq(v1,v2)
      | vtEq _ = false

    fun vtName(NormalVar n) = n
      | vtName(NextVar n) = n
      | vtName(InitVar n) = n

    fun vtFlag (NormalVar _) = NormalFlag
      | vtFlag (NextVar _) = NextFlag
      | vtFlag (InitVar _) = InitFlag

    fun vtWrap NormalFlag e = NormalVar e
      | vtWrap NextFlag e = NextVar e
      | vtWrap InitFlag e = InitVar e

    fun vtEqName(v1,v2) = ptEq(vtName v1, vtName v2)

    fun cvEq(PVar n1, PVar n2) = ptEq(n1,n2)
      | cvEq(DelayedVar v1, DelayedVar v2) = ptEq(v1,v2)
      | cvEq _ = false

    fun cvName(PVar n) = SOME n
      | cvName _ = NONE

    fun cvValue(PVar v) = v
      | cvValue(DelayedVar v) = v

    fun cvFlag(PVar _) = PVarFlag
      | cvFlag(DelayedVar _) = DelayedFlag

    fun cvWrap DelayedFlag e = DelayedVar e
      | cvWrap PVarFlag v = PVar v

    (* Check whether two state vars are similar upto non-value
       expressions as array indices. Sequent g1 ~= g2. *)
    fun svSimilar (g1 as StateVar _, g2 as StateVar _) = eqUNames(g1,g2)
      | svSimilar (g1 as DynPatternFormal _, g2 as DynPatternFormal _) = eqUNames(g1,g2)
      | svSimilar (ExtractRecord(f1,g1), ExtractRecord(f2,g2)) =
	 eqUNames(f1,f2) andalso svSimilar(g1,g2)
      | svSimilar (ExtractTuple(n1,g1), ExtractTuple(n2,g2)) =
	 n1=n2 andalso svSimilar(g1,g2)
      | svSimilar (ExtractAppl(c1,g1), ExtractAppl(c2,g2)) =
	 eqUNames(c1,c2) andalso svSimilar(g1,g2)
      | svSimilar (ExtractIndex g1, ExtractIndex g2) = svSimilar(g1, g2)
      | svSimilar (Appl(_,g1,e1), Appl(_,g2,e2)) =
	 if isValue e1 andalso isValue e2 then ptEq(e1,e2) andalso svSimilar(g1,g2)
	 else svSimilar(g1,g2)
      | svSimilar (g1,g2) = false

    (* Check whether g1 is a sub-var of g2.  Sequent g1 <=~ g2. *)
    fun svPartOf(g1,g2) =
	if svSimilar(g1,g2) then true
	else (case g1 of
		  ExtractRecord(_,g1') => svPartOf(g1', g2)
		| ExtractTuple(_,g1') => svPartOf(g1', g2)
		| ExtractAppl(_,g1') => svPartOf(g1', g2)
		| ExtractIndex g1' => svPartOf(g1', g2)
		| Appl(_,g1',_) => svPartOf(g1', g2)
		| _ => false)

    fun makeCone () = makeHashDefault(vtEq, vt2str): Cone
    fun copyCone (cone: Cone) = copyHash cone
    fun addConeDestructive(cone: Cone, v, lst) = insertHashDestructive(cone,v,lst)
    fun addCone(cone: Cone, v, lst) = addConeDestructive(copyCone cone,v,lst)

    local fun op * (x,y) = Conc(x,y)
    in 
	fun asstSeq2Str ({norm = norm, next = next, init = init}, ptlist) =
	    Str((vtlist2str "," norm) ^ ";")
	    *Str((vtlist2str "," next) ^ ";")
	    *Str((vtlist2str "," init) ^ " |- ")
	    *(Strlist2Str ", " (List.map pt2str ptlist))

	fun asstSeq2str seq = Str2string(asstSeq2Str seq)

	fun cv2str(PVar n) = "PVar("^(pt2string n)^")"
	  | cv2str(DelayedVar pt) = "DelayedVar("^(pt2string pt)^")"

	fun cv2strDebug(PVar n) = "PVar("^(pt2stringDebug n)^")"
	  | cv2strDebug(DelayedVar pt) = "DelayedVar("^(pt2stringDebug pt)^")"

	fun cvlist2Str sep lst = Strlist2Str sep (List.map(fn x=>Str(cv2str x)) lst)
	fun cvlist2str sep lst = Str2string(cvlist2Str sep lst)

	fun AsstVars2Str{norm=norm, next=next, init=init} =
	    (Str("["^(vtlist2str "," norm)^"; ")
	     *Str((vtlist2str "," next)^"; ")
	     *Str((vtlist2str "," init)^"]"))

	fun avt2Str (NormalAsstTree(vars, name, e)) = 
	     (AsstVars2Str vars)*(Str " |- ")*(pt2str name)*(Str " := ")*(pt2str e)
	  | avt2Str (NextAsstTree(vars, name, e)) = 
	     (AsstVars2Str vars)*(Str " |- next(")*(pt2str name)*(Str ") := ")*(pt2str e)
	  | avt2Str (InitAsstTree(vars, name, e)) =
	     (AsstVars2Str vars)*(Str " |- init(")*(pt2str name)*(Str ") := ")*(pt2str e)
	  | avt2Str (NopAsstTree vars) = (AsstVars2Str vars)*(Str " |- nop")
	  | avt2Str (ListAsstTree (vars, lst)) = 
	      (AsstVars2Str vars)*(Str " |- (\n")
	      *(Strlist2Str ";\n" (List.map avt2Str lst))
	      *(Str "\n)")
	  | avt2Str (CaseAsstTree (vars, sel, pairs)) = 
	      (AsstVars2Str vars)*Str(" |- case ")*(pt2str sel)*(Str " of\n      ")
	      (* The " => " is printed in the pt2str from ChoiceAsstClosure *)
	      *(Strlist2Str "\n    | " (List.map (avtpair2Str "    ") pairs))
	      *(Str "\nendcase")
	  | avt2Str (IfAsstTree (vars, pairs, last)) = 
	      (AsstVars2Str vars)*Str(" |- if ")
	      *(Strlist2Str "\nelsif" (List.map (avtpair2Str " then ") pairs))
	      *(Str "\nelse ")*(avt2Str last)*(Str "\nendif")
	  | avt2Str (ChooseAsstTree (vars, paramsOpt, pairs)) = 
	      (AsstVars2Str vars)*(Str " |- choose ")
	      *(option2Str (Str "")
		(Option.map (fn l=>(Strlist2Str ", " (List.map pt2str l))*(Str ":")) paramsOpt))
	      *(Str "\n    ")
	      *(Strlist2Str "\n  | " (List.map (avtpair2Str " => ") pairs))
	      *(Str "\nendchoose")
	  | avt2Str (ForeachAsstTree(vars, params, t)) =
	      (AsstVars2Str vars)*(Str " |- foreach ")
	      *(Strlist2Str ", " (List.map pt2str params))*(Str ":\n  ")
	      *(avt2Str t)*(Str "\nendforeach")
	  | avt2Str (LabeledAsstTree(vars, l, t)) =
	      (AsstVars2Str vars)*(Str " |- label ")*(pt2str l)
	      *(Str "\n  ")*(avt2Str t)
	  | avt2Str (LetAsstTree (vars, defs, t)) = 
	      (AsstVars2Str vars)*(Str " |- let ")
	      *(Strlist2Str "\n    " (List.map pt2str defs))
	      *(Str "\nin\n")*(avt2Str t)*(Str "\nend")
	and avtpair2Str sep (p,t) = (pt2str p)*(Str sep)*(avt2Str t)

	fun avt2str avt = Str2string(avt2Str avt)
    end

    val unionC = union cvEq
    val unionV = union vtEqName
    val unionPt = union ptEq
    fun unionAV [] = {norm = [], next = [], init = []}
      | unionAV [av] = av
      | unionAV ({norm = norm, next = next, init = init} :: t) =
      let
	val {norm = normt, next = nextt, init = initt} = unionAV t
      in
	{norm = unionV [norm, normt], next = unionV [next, nextt],
	 init = unionV [init, initt]}
      end

    (* Get parts of the assignment tree *)
    fun getAsstVars(NormalAsstTree(vars,_,_)) = vars
      | getAsstVars(NextAsstTree(vars,_,_)) = vars
      | getAsstVars(InitAsstTree(vars,_,_)) = vars
      | getAsstVars(NopAsstTree vars) = vars
      | getAsstVars(ListAsstTree(vars,_)) = vars
      | getAsstVars(LetAsstTree(vars,_,_)) = vars
      | getAsstVars(CaseAsstTree(vars,_,_)) = vars
      | getAsstVars(IfAsstTree(vars,_,_)) = vars
      | getAsstVars(ChooseAsstTree(vars,_,_)) = vars
      | getAsstVars(ForeachAsstTree(vars,_,_)) = vars
      | getAsstVars(LabeledAsstTree(vars,_,_)) = vars

    (* Get parts of the assignment tree *)
    fun setAsstVars vars (NormalAsstTree(_,v,a)) = NormalAsstTree(vars, v, a)
      | setAsstVars vars (NextAsstTree(_,v,a)) = NextAsstTree(vars, v, a)
      | setAsstVars vars (InitAsstTree(_,v,a)) = InitAsstTree(vars, v, a)
      | setAsstVars vars (NopAsstTree _) = NopAsstTree vars
      | setAsstVars vars (ListAsstTree(_, lst)) = ListAsstTree(vars, lst)
      | setAsstVars vars (LetAsstTree(_,defs,a)) = LetAsstTree(vars, defs, a)
      | setAsstVars vars (CaseAsstTree(_,sel,pairs)) = CaseAsstTree(vars, sel, pairs)
      | setAsstVars vars (IfAsstTree(_,pairs, a)) = IfAsstTree(vars, pairs, a)
      | setAsstVars vars (ChooseAsstTree(_,params,pairs)) = ChooseAsstTree(vars, params, pairs)
      | setAsstVars vars (ForeachAsstTree(_,params,a)) = ForeachAsstTree(vars, params, a)
      | setAsstVars vars (LabeledAsstTree(_,l,a)) = LabeledAsstTree(vars, l, a)

    (* Remove vars in `V' from `U' and return the result *)
    fun removeVars(U,V) =
	let fun rem(l1,l2) = List.filter(fn x=> not(List.exists(fn y=> vtEq(x,y)) l2)) l1
	    val {norm=normU, next=nextU, init=initU} = U
	    val {norm=normV, next=nextV, init=initV} = V
	in {norm=rem(normU, normV),
	    next=rem(nextU, nextV),
	    init=rem(initU, initV)}
	end
    (* Balance the vars in the conditional branches. *)
    fun balanceAsstTree tree =
	let val emptyU = {norm=[], next=[], init=[]}: AsstVars
	    fun isEmpty({norm=[], next=[], init=[]}: AsstVars) = true
	      | isEmpty _ = false
	    fun V2str{norm=norm, next=next, init=init} =
		("["^(vtlist2str "," norm)^"; "
		 ^(vtlist2str "," next)^"; "
		 ^(vtlist2str "," init)^"]")
	    fun doPair V (p,t) =
		let val (V', t') = loop(V, t)
		    val newt = 
			 if isEmpty V' then t'
			 else ListAsstTree(V, [t', NopAsstTree V'])
		in
		    (p, newt)
		end
	    (* loop returns a pair (U', tree'), where U' = U - vars(tree) *)
	    and loop (U, tree as NormalAsstTree(V,_,_)) = (removeVars(U,V), tree)
	      | loop (U, tree as NextAsstTree(V,_,_)) = (removeVars(U,V), tree)
	      | loop (U, tree as InitAsstTree(V,_,_)) = (removeVars(U,V), tree)
	      (* Adjust `nop' to assign all the variables in `U' *)
	      | loop (U, NopAsstTree _) = (emptyU, NopAsstTree U)
	      | loop (U, ListAsstTree(V, lst)) =
		  (removeVars(U,V), ListAsstTree(V, loopList(V, lst)))
	      | loop (U, t as LetAsstTree(V, defs, tree)) =
		  let val (V', tree') = loop(V, tree)
		  in if isEmpty V' then
		      (removeVars(U,V), LetAsstTree(V, defs, tree'))
		     else raise SympBug
		       ("TransPrep/balance: LetAsst didn't use these vars:\n  "
			^(V2str V')^"\nin the following assignment:\n  "
			^(avt2str t))
		  end
	      | loop (U, CaseAsstTree(V, sel, pairs)) =
		  (removeVars(U, V), CaseAsstTree(V, sel, List.map (doPair V) pairs))
	      | loop (U, IfAsstTree(V, pairs, last)) =
		  let val (_, last') = doPair V (Fake, last)
		  in 
		      (removeVars(U, V),
		       IfAsstTree(V, List.map (doPair V) pairs, last'))
		  end
	      | loop (U, ChooseAsstTree(V,params,pairs)) =
		  (removeVars(U, V), ChooseAsstTree(V, params, List.map (doPair V) pairs))
	      | loop (U, ForeachAsstTree(V,params,t)) =
		  let val (_, t') = loop(V, t)
		  in (removeVars(U,V), ForeachAsstTree(V, params, t'))
		  end
	      | loop (U, t as LabeledAsstTree(V, label, tree)) =
		  let val (V', tree') = loop(V, tree)
		  in if isEmpty V' then
		      (removeVars(U,V), LabeledAsstTree(V, label, tree'))
		     else raise SympBug
		       ("TransPrep/balance: LabelAsst didn't use these vars:\n  "
			^(V2str V')^"\nin the following assignment:\n  "
			^(avt2str t))
		  end
	    and loopList(V, []) = 
		  if isEmpty V then []
		  else [NopAsstTree V]
	      | loopList(V, a::lst) =
		  let val (V', a') = loop(V, a)
		  in 
		      a'::(loopList(V', lst))
		  end
	    val (_, tree') = loop(getAsstVars tree, tree)
	in
	    tree'
	end

    (* Traverse the tree, collect all the vars that are *actually*
       assigned in the tree, and balance the tree.

       Assumes the leaves have correct sets of vars. *)

    fun recomputeAsstVars tree =
	let val emptyVars = {norm=[], next=[], init=[]}
	    fun collect (t as NormalAsstTree _) = t
	      | collect (t as NextAsstTree _) = t
	      | collect (t as InitAsstTree _) = t
	      | collect (NopAsstTree _) = NopAsstTree emptyVars
	      | collect (ListAsstTree(_, lst)) =
		let val newList = List.map collect lst
		    val vars = unionAV(List.map getAsstVars newList)
		in ListAsstTree(vars, newList)
		end
	      | collect (LetAsstTree(_, defs, t)) =
		let val newT = collect t
		in LetAsstTree(getAsstVars newT, defs, newT)
		end
	      | collect (CaseAsstTree(_, sel, pairs)) =
		let val newPairs = List.map (fn(pat,t)=>(pat, collect t)) pairs
		    val vars = unionAV(List.map(fn(_, t)=> getAsstVars t) newPairs)
		in CaseAsstTree(vars, sel, newPairs)
		end
	      | collect (IfAsstTree(_, pairs, t)) =
		let val newPairs = List.map (fn(c,t)=>(c, collect t)) pairs
		    val newT = collect t
		    val vars = unionAV((getAsstVars t)
				       ::(List.map(fn(_, t)=> getAsstVars t) newPairs))
		in IfAsstTree(vars, pairs, t)
		end
	      | collect (ChooseAsstTree(_, paramsOpt, pairs)) =
		let val newPairs = List.map (fn(c,t)=>(c, collect t)) pairs
		    val vars = unionAV(List.map(fn(_, t)=> getAsstVars t) newPairs)
		in ChooseAsstTree(vars, paramsOpt, newPairs)
		end
	      | collect (ForeachAsstTree(_, params, t)) =
		let val newT = collect t
		in ForeachAsstTree(getAsstVars newT, params, newT)
		end
	      | collect (LabeledAsstTree(_, label, t)) =
		let val newT = collect t
		in LabeledAsstTree(getAsstVars newT, label, newT)
		end
	    val tree' = collect tree
	in 
	    balanceAsstTree tree'
	end

    (* Do variable substitution in RHS expressions over all the asst tree *)
    fun substAsstTree (v,expr) tree =
	let fun substRec(v,e) x = ptTransform (subst (v, e)) x
	    and subst(v,e) (body as FunClosure{formals=f,...}) = 
	         if eqUNames(v,f) then body else substRec(v,e) body
	      | subst(v,e) (body as ChoiceClosure{uname=u,...}) = 
	         if eqUNames(v,u) then body else substRec(v,e) body
	      | subst(v,e) body =
		 if eqUNames(v,body) then e else substRec(v,e) body
	    fun substList(v,e) lst = List.map(subst(v,e)) lst
	    fun substPairs(v,e) lst = List.map(fn(p,t)=> (subst(v,e) p, loop(v,e) t)) lst
	    and loop (v,e) (NormalAsstTree(vars,name,expr)) = 
		  NormalAsstTree(vars, subst(v,e) name, subst(v,e) expr)
	      | loop (v,e) (NextAsstTree(vars,name,expr)) = 
		  NextAsstTree(vars, subst(v,e) name, subst(v,e) expr)
	      | loop (v,e) (InitAsstTree(vars,name,expr)) = 
		  InitAsstTree(vars, subst(v,e) name, subst(v,e) expr)
	      | loop (v,e) (t as NopAsstTree _) = t
	      | loop (v,e) (ListAsstTree(vars,tlist)) =
		  ListAsstTree(vars, looplist(v,e) tlist)
	      | loop (v,e) (LetAsstTree(vars,lst,t)) =
		  LetAsstTree(vars, substList(v,e) lst, loop(v,e) t)
	      | loop (v,e) (CaseAsstTree(vars,sel,pairs)) =
		  CaseAsstTree(vars, subst(v,e) sel, substPairs(v,e) pairs)
	      | loop (v,e) (IfAsstTree(vars,pairs,last)) =
		  IfAsstTree(vars, substPairs(v,e) pairs, loop(v,e) last)
	      | loop (v,e) (ChooseAsstTree(vars, paramsOpt, pairs)) =
		  ChooseAsstTree(vars, Option.map (substList(v,e)) paramsOpt,
				 substPairs(v,e) pairs)
	      | loop (v,e) (ForeachAsstTree(vars, params, t)) =
		  ForeachAsstTree(vars, substList(v,e) params, loop(v,e) t)
	      | loop (v,e) (LabeledAsstTree(vars, label, t)) =
		  LabeledAsstTree(vars, label, loop(v,e) t)
	    and looplist(v,e) tlist = List.map(loop(v,e)) tlist
	in loop(v,expr) tree
	end

    fun makePrimitiveVars () = {vars=makeHashDefault(ptEq, pt2string),
				indexVars=makeHashDefault(ptEq, pt2string)}: PrimitiveVars
    fun copyPrimitiveVars ({vars=vars,indexVars=iv}: PrimitiveVars) =
	{vars=copyHash vars, indexVars=copyHash iv}

    (* Get the first layer of `primitive vars' of s *)
    fun getPrimitiveVars1 ({vars=pvars,...}: PrimitiveVars) s = findHash(pvars,s)

    (* Return the list of all primitive vars for a state var `s', or NONE
       if `s' is not in the database. *)
    fun getPrimitiveVars pvars s =
	let fun getList [] = []
	      | getList (h::t) = 
	         (case getPrimitiveVars pvars h of
		      NONE => [h]
		    | SOME l => l)::(getList t)
	in Option.map(fn l => List.foldr(op @) [] (getList l))
	             (getPrimitiveVars1 pvars s)
	end

    fun addPrimitiveVarsDestructive (pvars as {vars=vars,...}: PrimitiveVars) s lst =
	 (insertHashDestructive(vars,s,lst); pvars)
    fun addPrimitiveVars pvars s lst =
	 addPrimitiveVarsDestructive(copyPrimitiveVars pvars) s lst
	  
    fun getPrimitiveIndexVar ({indexVars=pvars,...}: PrimitiveVars) s = findHash(pvars,s)
    fun addPrimitiveIndexVarDestructive (pvars as {indexVars=vars,...}: PrimitiveVars) s iv =
	 (insertHashDestructive(vars,s,iv); pvars)
    fun addPrimitiveIndexVar pvars s iv =
	 addPrimitiveIndexVarDestructive(copyPrimitiveVars pvars) s iv

    (* Merge the two sets of primitive vars into a new one (no side effects) *)
    fun mergePrimitiveVars(pvars1,pvars2) =
	let val {vars=vars1, indexVars=ivars1} = pvars1
	    val {vars=vars2, indexVars=ivars2} = pvars2
	    val pvars as {vars=vars, indexVars=ivars} = makePrimitiveVars()
	    val varList = (hash2any(fn x=>x)(fn x=>x) vars1)
		         @(hash2any(fn x=>x)(fn x=>x) vars2)
	    val ivarList = (hash2any(fn x=>x)(fn x=>x) ivars1)
		          @(hash2any(fn x=>x)(fn x=>x) ivars2)
	    fun addHash hash (key,v) =
		(case findHash(hash,key) of
		     SOME _ => ()
		   | NONE => (insertHashDestructive(hash,key,v); ()))
	    val _ = List.app (addHash vars) varList
	    val _ = List.app (addHash ivars) ivarList
	in 
	    {vars=vars, indexVars=ivars}: PrimitiveVars
	end


     (* Split types of a StateVar `e', sequent `V |-_2 delta'.  
        SIDE EFFECT: When `addIndex' is true, may add index vars to
        `pvars' as a side effect.

	It returns a ConeVar list, but the types of DelayedVar's may
	not be consistent with `t'. *)

    fun splitEType options findObject addIndex pvars lim (e, t) = 
	let val funName = "splitEType"
	    val _ = pushFunStackLazy(funName, fn()=>pt2string(TypedExpr(dp, e, t)))
	    val recur = splitEType options findObject addIndex pvars lim
	    val eval = evaluateExpr options findObject
	    val res =
	  (case t of
	      TupleType (_, tlist) => 
		let
		    (* List of pairs (ET(n,e), Type_n) for all n *)
		    val pair = (List.map
				(fn n => (eval(ExtractTuple (n, e)), List.nth (tlist, n)))
				(List.tabulate (List.length tlist, fn i => i)))
		in unionC(List.map recur pair)
		end
	    | RecordType (_, rlist) => 
		let
		    (* List of pairs (ER(f,e), tp_f) for all fields `f' *)
		    val pair = (List.map
				(fn RecordField{name=n,Type=tp} => (ExtractRecord (n, e), tp)
			      | x => raise SympBug
				 ("splitEType: not RecordField in RecordType: "^(pt2string x)))
				rlist)
		in unionC(List.map recur pair)
		end
	    | FunType (_, t1, t2) => 
		let
		    val dvals = getTypeValues options lim t1
		in
		    (case dvals of
			 SOME lv => 
			     let val ens = List.map (fn d => eval(Appl (dp, e, d))) lv
				 val Ns = List.map (fn en => recur (en, t2)) ens
			     in
				 unionC Ns
			     end
		       (* Since `e' is a state var that can't be split further,
			it is considered "primitive" *)
		       | NONE => [DelayedVar e])
		end
	    | ArrayType (_, t1, t2) => 
		let
		    val dvals = getTypeValues options lim t1
		in
		    (case dvals of
			 SOME lv => 
			     let val ens = List.map (fn d => eval(Appl (dp, e, d))) lv
				 val Ns = List.map (fn en => recur (en, t2)) ens
			     in
				 unionC Ns
			     end
		       (* Since `e' is a state var that can't be split further,
			it is considered "primitive" *)
		       | NONE => [DelayedVar e])
		end
	    | EnumType(_,lst) => 
		let val index =
		    (case getPrimitiveIndexVar pvars e of
			 SOME x => x
		       (* If it doesn't have an index var yet, add it 
			  DESTRUCTIVELY *)
		       | NONE => 
			   if addIndex then
			       let val x = ExtractIndex e
				   (* val x = StateVar {name = un, uname = un, 
						     Type = AbstractType lst,
						     id = un} *)
			       in 
				   addPrimitiveIndexVarDestructive pvars e x;
				   x
			       end
			   else raise SympBug
			       ("splitEType:  state var doesn't have an index var: "
			       ^(pt2string e)))
		    (* Get a function computing a pair (ExtractAppl(c,e), tp) for `e',
		       if the TC has an argument. *)
		    fun extract (c as TypeConstr{Type=tp,...}) =
			(case tp of
			     FunType(_,t,_) => SOME(eval(ExtractAppl(c, e)), t)
			   | _ => NONE)
		      | extract c = raise SympBug
			     ("splitEType: not a TypeConstr:\n  "^(pt2string c))
		    val pairs = List.mapPartial extract lst
		in (PVar index)::(unionC(List.map recur pairs))
		end
	    | StaticFormalType{value=SOME t, ...} => recur(e, t)
	    (* Base type, just return what we already have. *)
	    | _ => [PVar e])
	    val _ = popFunStackLazy(funName, fn()=>"["^(cvlist2str ", " res)^"]")
	in
	    res
	end

    (* Sequent ``V |- e'', but leaving DelayedVar's delayed.
       SIDE EFFECTS: may add index variables to `pvars' if `addIndex' is true. *)
    fun varsFromDelayed options findObject addIndex pvars lim expr =
	let val funName = "varsFromDelayed"
	    val _ = pushFunStackLazy(funName, fn()=>pt2string expr)
	    (* Wraps `e' into an extractor *)
	    fun wrapExtr(ExtractAppl(c,_), e) = ExtractAppl(c,e)
	      | wrapExtr(ExtractTuple(n,_), e) = ExtractTuple(n,e)
	      | wrapExtr(ExtractRecord(f,_), e) = ExtractRecord(f,e)
	      | wrapExtr(ExtractIndex _, e) = ExtractIndex e
	      | wrapExtr(Appl(p,_,d), e) = Appl(p,e,d)
	      | wrapExtr(x,_) = raise SympBug
		  ("varsFrom/wrapExtr: not an extractor: "^(pt2string x))
	    fun wrapExtrs lst e = List.foldr wrapExtr e (List.rev lst)
	    (* Sequent ``Sigma; V |-_1 e'' *)
	    fun loop extrList expr =
	        let val funName = "varsFromDelayed/loop"
		    val debug = lazyVerbDebug options funName
		    val _ = pushFunStackLazy(funName,
					     fn()=>"["^(ptlist2str ", " extrList)
					     ^"], "^(pt2string expr))
		    val res = loop0 extrList expr
		    val _ = popFunStackLazy(funName, fn()=> "["^(cvlist2str ",\n  " res)^"]")
		in res
		end
	    and loop0 lst (x as StateVar _) =
		 let val newx = wrapExtrs lst x
		 (* Since x is a StateVar, we know that any DelayedVar
		    in the result of `splitEType' is a state var, and
		    we can safely re-wrap it into PVar. *)
		 in splitEType options findObject addIndex pvars lim
		       (newx, getExprType findObject newx)
		 end
	      | loop0 lst (x as DynPatternFormal _) =
		 let val newx = wrapExtrs lst x
		 in splitEType options findObject addIndex pvars lim
		     (newx, getExprType findObject newx)
		 end
	      | loop0 lst (NondetExpr (_, ndlist)) = 
		 unionC(List.map (loop lst) ndlist)
	      | loop0 ((ExtractRecord(f,_))::lst) (x as WithExpr _) = 
		 (* Rely on evaluator's ability to eliminate WithExpr.
		    Either the field is extracted from `with', or `with' is
		    irrelevant and is removed.  So the progress is guaranteed. *)
		 loop lst (evaluateExpr options findObject (ExtractRecord(f,x)))
	      | loop0 (ex::_) (x as WithExpr _) = raise SympBug
		 ("varsFrom: not ExtractRecord for WithExpr:\n "
		  ^(pt2string ex)^" applied to "^(pt2string x))
	      | loop0 ((ExtractIndex _)::lst) (x as Appl(_, TypeConstr _, _)) =
		 loop lst (evaluateExpr options findObject (ExtractIndex x))
	      | loop0 [] (WithExpr (_, record, wlist)) = 
		  unionC((loop [] record) :: (List.map (loop []) wlist))
	      | loop0 ((ExtractTuple(n, _))::lst) (x as TupleExpr (_, tlist)) = 
		  loop lst ((List.nth(tlist,n)) handle Subscript => raise SympBug
			    ("varsFrom: ExtractTuple out of range: "
			     ^(pt2string(ExtractTuple(n,x)))))
	      | loop0 (ex::lst) (x as TupleExpr _) = raise SympBug
		  ("varsFrom: not ExtractTuple for TupleExpr:\n "
		   ^(pt2string ex)^" applied to "^(pt2string x))
	      | loop0 [] (x as TupleExpr (_, tlist)) = unionC(List.map(loop []) tlist)
	      (* Unwrap extractors after trying to apply those from the list first *)
	      | loop0 lst (ExtractRecord (f, r)) = loop ((ExtractRecord(f,Fake))::lst) r
	      | loop0 lst (ExtractTuple (n, v)) = loop ((ExtractTuple(n, Fake))::lst) v
	      | loop0 lst (ExtractAppl (c, v)) = loop ((ExtractAppl(c, Fake))::lst) v
	      | loop0 lst (ExtractIndex v) = loop((ExtractIndex Fake)::lst) v
	      | loop0 ((ExtractRecord(f,_))::lst) (x as RecordExpr (_, rlist)) = 
		 loop lst (evaluateExpr options findObject (ExtractRecord(f,x)))
	      | loop0 (ex::_) (x as RecordExpr _) = raise SympBug
		 ("varsFrom: not ExtractRecord for RecordExpr:\n "
		  ^(pt2string ex)^" applied to "^(pt2string x))
	      | loop0 [] (RecordExpr (_, rlist)) = 
		  unionC(List.map (loop []) rlist)
	      | loop0 lst (RecordAsst (_, _, v)) = loop lst v
	      | loop0 lst (WithAsst (_, _, v)) = loop lst v
	      | loop0 lst (IfExpr (_, conds, else_exp)) = 
		  unionC((loop lst else_exp) :: (List.map (loop lst) conds))
	      | loop0 lst (CondExpr (_, cond, e)) = unionC[loop [] cond, loop lst e]
	      | loop0 lst (CaseExpr (_, e, clist)) = 
		  unionC((loop [] e) :: (List.map (loop lst) clist))
	      (* A choice in a `case' statement *)
	      | loop0 lst (ChoiceClosure {body=exp, ...}) = loop lst exp
	      (* `pick' operator *)
	      | loop0 lst (ChooseClosure _) = raise SympBug
		  ("varsFrom: Sorry, `pick' is not implemented yet.")
	      | loop0 lst (LetClosure {body=body,...}) = loop lst body
	      | loop0 ((ExtractAppl(ec,_))::lst) (Appl(_,c as TypeConstr _, v)) =
		  if eqUNames(ec,c) then loop lst v else []
	      | loop0 lst (Appl(p,x,y)) = 
		  if isConst y then loop((Appl(p,Fake,y))::lst) x
		  else unionC [loop [] x, loop [] y]
	      | loop0 ((w as Appl _)::lst) (e as FunClosure _) =
		  loop lst (evaluateExpr options findObject (wrapExtr(w, e)))
	      | loop0 lst (FunClosure {body=body,...}) = loop [] body
	      | loop0 lst (RecurFun r) = loop [] r
	      | loop0 lst (Forall(_,_,e)) = loop [] e
	      | loop0 lst (Exists(_,_,e)) = loop [] e
	      | loop0 lst (ForallClosure{body=e,...}) = loop [] e
	      | loop0 lst (ExistsClosure{body=e,...}) = loop [] e
	      | loop0 lst (Theorem(_,_,e)) = loop [] e
	      | loop0 lst (Models(_,_,e)) = loop [] e
	      | loop0 lst (Object {def=def,...}) = loop lst def
	      | loop0 lst (x as ObjectInst _) = loop lst (stripObjectInst x)
	      (* Otherwise it is a contstant *)
	      | loop0 lst _ = []
	    val res = loop [] expr
	    val _ = popFunStackLazy(funName, fn()=> "["^(cvlist2str ",\n  " res)^"]")
	in res 
	end

    (* Sequent ``V |- e''.
       SIDE EFFECTS: may add index variables to `pvars'. *)
    fun varsFrom options findObject pvars lim expr =
	  List.map(fn x=>PVar(cvValue x))(varsFromDelayed options findObject true pvars lim expr)

    fun getVarConeDirect(cone, v) = findHash(cone, v)

    (* sequent `C(v) -> V'.  To guarantee that `v' is "as primitive as
       possible", we split its type with `splitEType'. *)
    fun getVarCone1Delayed options findObject pvars lim cone v =
	let val funName = "getVarCone1Delayed"
	    val debug = lazyVerbDebug options funName
	    val _ = pushFunStackLazy(funName, fn()=>(vt2str v))
	    val (flag, vName) = (vtFlag v, vtName v)
	    (* val vType = getExprType findObject vName *)
	    datatype SplitResult = Success of ConeVar list | Failure of ConeVar
	    fun splitres2str (Success lst) = "Success["^(cvlist2str "," lst)^"]"
	      | splitres2str (Failure cv) = "Failure("^(cv2str cv)^")"
	    (* Try to split a var `e' into *)
	    fun split e =
		let val funName = "getVarCone1Delayed/split"
		    val _ = pushFunStackLazy(funName, fn()=> pt2string e)
		    val newE = evaluateExpr options findObject e 
		    val vars = varsFromDelayed options findObject false pvars lim newE
		    val res = 
			if List.exists(fn DelayedVar _ => true | _ => false) vars then
			    Failure(DelayedVar newE)
			else Success vars
		    val _ = popFunStackLazy(funName, fn()=>splitres2str res)
		in res
		end
	    (* The core lookup procedure, sequent `C(v) -> V' *)
	    fun lookup v =
		let val _ = debug(fn()=>"  getVarCone1Delayed/lookup("^(pt2string v)^")\n")
		    fun subcase extrFun gamma =
		      let val ve = lookup gamma
			  val vs = List.filter(fn PVar _ => true | _ => false) ve
			  val es = List.map cvValue 
			             (List.filter(fn DelayedVar _ => true | _ => false) ve)
			  val es' = List.map (fn e => (case split (extrFun e) of
							   Success l => l
							 | Failure cv => [cv])) es
		      in unionC(vs::es')
		      end
		in (case getVarConeDirect(cone,vtWrap flag v) of
			SOME l => l
		      | NONE => 
			  (case v of
			       Appl(p,g,d) => subcase (fn e => Appl(p,e,d))(g)
			     | ExtractRecord(f,g) => subcase (fn e => ExtractRecord(f,e))(g)
			     | ExtractTuple(i,g) => subcase (fn e => ExtractTuple(i,g))(g)
			     | ExtractAppl(c,g) => subcase (fn e => ExtractAppl(c,g))(g)
			     | ExtractIndex g => subcase (fn e => ExtractIndex g)(g)
			     | _ => []))
		end
	    val res =
		 (case split vName of
		      Success l => List.foldr(op @) [] (List.map (lookup o cvValue) l)
		    | Failure _ => [DelayedVar vName])
	in res before (popFunStackLazy(funName, fn()=>cvlist2str ", " res))
	end
    (* Sequent `C(v) => V'.  To guarantee that `v' is "as primitive as
       possible", we split its type with `splitEType'. *)

    fun getVarCone1 options findObject pvars lim (cone: Cone) v =
	let val funName = "getVarCone1"
	    val debug = lazyVerbDebug options funName
	    val _ = pushFunStackLazy(funName, fn()=>vt2str v)
	    val ve = getVarCone1Delayed options findObject pvars lim cone v
	    val vs = List.filter(fn PVar _ => true | _ => false) ve
	    val es = List.map cvValue 
		       (List.filter(fn DelayedVar _ => true | _ => false) ve)
	    val Vs = List.foldr(op @) [] (List.map(varsFrom options findObject pvars lim) es)
	    val res = vs@Vs
	in res before (popFunStackLazy(funName, fn()=>cvlist2str ", " res))
	end
			       
    (* Get the COI (list of vars that influence `v') upto level n.
       When n=0, returns emply hash.  n=1 gets the immediate vars (one step), etc.
       Get the full cone if n<0.
       When `includeNext' is false, ignore NextVar variables (useful
       for checking circular assts). *)
    fun getVarConeHashCommon includeNext getVarCone options findObject 
                             pvars lim n (cone: Cone) v = 
	(* `hash' is the current set of cone vars. *)
	let val funName = "getVarConeHashCommon"
	    val hash = makeHashDefault(cvEq,cv2str)
	    val debug = lazyVerbDebug options funName
	    val _ = pushFunStackLazy(funName,
				     fn()=>"includeNext="
				     ^(if includeNext then "Yes, " else "No, ")
				     ^(Int.toString n)^","^(vt2str v))
	    (* Functions for manipulating wild cards *)
	    (* wildCard: ParseTree -> ParseTree; takes a state var expression and 
	       computes its wild card.  It *must* be a state var expr. *)
	    fun wildCard (ExtractTuple(n,s)) = ExtractTuple(n,wildCard s)
	      | wildCard (ExtractRecord(f,s)) = ExtractRecord(f,wildCard s)
	      | wildCard (ExtractAppl(c,s)) = ExtractAppl(c,wildCard s)
	      | wildCard (ExtractIndex s) = ExtractIndex(wildCard s)
	      | wildCard (Appl(p,s,d)) =
		  let val newd = 
		       (if isValue d then
			    case getTypeSize options (getExprType findObject d) of
				FiniteSize m => if m > lim then Fake else d
			      | _ => Fake
			else Fake)
		  in Appl(p, wildCard s, newd)
		  end
	      | wildCard (s as StateVar _) = s
	      | wildCard (s as DynPatternFormal _) = s
	      | wildCard x = raise SympBug
		  ("getVarConeHashCommon/wildCard: not a state var expression:\n  "
		   ^(pt2string x))
	    (* Find the topmost array with a wild card index, if there is one.
	       If none are found, just return the original argument. *)
	    fun topmostArray wc =
		let fun hasWildcard (ExtractTuple(_,s)) = hasWildcard s
		      | hasWildcard (ExtractRecord(_,s)) = hasWildcard s
		      | hasWildcard (ExtractAppl(_,s)) = hasWildcard s
		      | hasWildcard (ExtractIndex s) = hasWildcard s
		      | hasWildcard (Appl(_,_,Fake)) = true
		      | hasWildcard (Appl(_,s,_)) = hasWildcard s
		      | hasWildcard _ = false
		    fun loop (x as (ExtractTuple(n,s))) =
			  if hasWildcard s then loop s else x
		      | loop (x as (ExtractRecord(f,s))) =
			  if hasWildcard s then loop s else x
		      | loop (x as (ExtractAppl(c,s))) =
			  if hasWildcard s then loop s else x
		      | loop (x as (ExtractIndex s)) =
			  if hasWildcard s then loop s else x
		      | loop (x as (Appl(_,s,_))) =
			  if hasWildcard x then loop s else x
		      | loop x = x
		in loop wc
		end
	    (* Find a potential endless loop in the cone lookup and return
	       the topmost array for `v' if cycle is found.  Otherwise return NONE. *)
	    fun checkCycle cycleHash v =
		let val funName = "getVarConeHashCommon/checkCycle"
		    val wc = wildCard v
		    val _ = pushFunStackLazy(funName, fn()=>(pt2string wc))
		    val res = (case findHash(cycleHash, wc) of
				   NONE => NONE
				 | SOME n => if n >= lim then SOME(topmostArray wc)
					     else NONE)
		in (popFunStackLazy(funName, fn()=>(case res of
							NONE => "NONE\n"
						      | SOME v => ("SOME("^(pt2string v)^")\n")));
		    res)
		end
	    fun addToCycleHash(cycleHash, v) =
		let val wc = wildCard v
		in (case findHash(cycleHash, wc) of
			NONE => insertHash(cycleHash, wc, 0)
		      | SOME n => insertHash(cycleHash, wc, n+1))
		end
	    fun loop cycleHash m cv =
	         let val funName = "getVarConeHashCommon/loop"
		     val _ = pushFunStackLazy(funName,
					      fn()=>(Int.toString m)^","^(cv2str cv))
		     fun notInHash cv = not(isSome(findHash(hash, cv)))
		     val filter = List.filter notInHash
		     fun nextList v = filter(getVarCone options findObject pvars lim cone v)
		     val (fringe, newCycleHash) =
		      if m = n then ([], cycleHash)
		      else
			(case cv of
			     DelayedVar e => 
			       (* let val vs = List.map cvValue 
				             (filter(varsFrom options pvars lim e))
				   val Vs = ((if includeNext then
						  List.map(fn n=>nextList(NextVar n)) vs
					      else [])
					     @(List.map(fn n=>nextList(NormalVar n)) vs)
					     @(List.map(fn n=>nextList(InitVar n)) vs))
			       in (List.app(fn v=>(insertHashDestructive(hash, PVar v, ());())) vs;
				   (unionC Vs, cycleHash))
			       end *)
				 (insertHashDestructive(hash, cv, ());
				  ([], cycleHash))
			   | PVar n =>
			      (case checkCycle cycleHash n of
				    SOME new => (insertHashDestructive(hash, DelayedVar new, ());
						 ([], cycleHash))
				  | NONE => 
				     let val lst =
					 if notInHash cv then
					     (if includeNext then nextList(NextVar n) else [])
						  @(nextList(NormalVar n))
						  @(nextList(InitVar n))
					 else []
				     in (insertHashDestructive(hash,cv,()));
					(lst, addToCycleHash(cycleHash, n))
				     end))
		     val _ = List.app(loop newCycleHash (m+1)) fringe
		     val _ = popFunStack(funName, "")
		 in ()
		 end
	    (* Hash of state vars' wild cards that has been looked up.
	       If a state var with the same wild card is about to be
	       looked up second time, add that topmost array into the
	       result.  Otherwise we may get into an endless loop for
	       infinite arrays. *)
	    val cycleHash = makeHashDefault(ptEq,pt2string)
	    val _ = List.app(loop cycleHash 0) (getVarCone options findObject pvars lim cone v)
	in hash
	    before (popFunStackLazy
		    (funName,
		     fn()=>(strlist2str ", " 
			    (List.map(fn(x,_)=>cv2str x)(hash2any(fn x=>x)(fn x=>x) hash)))))
	end

    val getVarConeHash = getVarConeHashCommon true getVarCone1
    val getVarConeHashDelayed = getVarConeHashCommon true getVarCone1Delayed

    val getVarConeHashNoNext = getVarConeHashCommon false getVarCone1
    val getVarConeHashDelayedNoNext = getVarConeHashCommon false getVarCone1Delayed

    fun getVarConeN options findObject pvars lim n cone v =
	 List.map(fn(x,_)=>x)
	  (hash2any(fn x=>x)(fn x=>x)(getVarConeHash options findObject pvars lim n cone v))

    fun getVarConeNnoNext options findObject pvars lim n cone v =
	 List.map(fn(x,_)=>x)
	  (hash2any(fn x=>x)(fn x=>x)(getVarConeHashNoNext options findObject pvars lim n cone v))

    fun getVarCone options findObject pvars lim = getVarConeN options findObject pvars lim (~1)
    fun getVarConeNoNext options findObject pvars lim =
	  getVarConeNnoNext options findObject pvars lim (~1)

    (* Check whether v is in the cone of v1.  Return true, false, or NONE if not known. *)
    fun inConeOfCommon includeNext options findObject pvars lim cone v1 v = 
	let val funName = "inConeOfCommon"
	    val _ = pushFunStackLazy(funName, fn()=>(vt2str v1)^", "^(cv2str v))
	    (* Make sure to remove any stray ObjectInst that might have sneaked in *)
	    val v1 = vtWrap(vtFlag v1)(stripObjectInst(vtName v1))
	    val v = cvWrap(cvFlag v)(stripObjectInst(cvValue v))
	    (* Now compute the cone and do the rest *)
	    val coneHash = (getVarConeHashCommon includeNext getVarCone1Delayed
			      options findObject pvars lim (~1) cone v1)
	    (* Primitive vars of `v' *)
	    val primvars = (case v of
				PVar n => (case getPrimitiveVars pvars n of
					       SOME vs => vs
					     | NONE => [])
			      | DelayedVar _ => [])
	    fun checkVar v = 
	      (case findHash(coneHash, v) of
		   SOME _ => SOME(true)
		 | NONE => 
		    let val coneList = List.map(fn(x,_)=>x)(hash2any(fn x=>x)(fn x=>x)(coneHash))
			fun checkOne _ (DelayedVar _) = false
			  | checkOne v (PVar v') = svPartOf(v,v') orelse svPartOf(v',v)
		    in (* If there is a delayed cone in the list, we don't know *)
			(case v of
			     PVar vName => 
			       if List.exists (checkOne vName) coneList then SOME(true)
			       else if List.exists(fn DelayedVar _ => true | _ => false) coneList
			       then NONE
			       else SOME(false)
			     | DelayedVar _ => NONE)
		    end)
	    val resList = List.mapPartial checkVar (v::(List.map(cvWrap PVarFlag) primvars))
	    val hasTrue = List.exists (fn x=>x) resList
	    val res = (case resList of
			   [] => NONE
			 | _ => SOME hasTrue)
	in 
	    (popFunStackLazy(funName, fn()=>(case res of
						    SOME true => "SOME true"
						  | SOME false => "SOME false"
						  | NONE => "NONE"));
	     res)
	end

    val inConeOf = inConeOfCommon true
    val inConeOfNoNext = inConeOfCommon false

    (* Return set-difference of l1 - l2 w.r.t. equality `eq' *)
    fun listDiff eq (l1,l2) =
	List.filter(fn x=>not(List.exists(fn y=>eq(x,y)) l2)) l1
    fun addPrim pvars (stvar, vars) = 
	let val oldvars = (case getPrimitiveVars1 pvars stvar of
			       NONE => []
			     | SOME lst => lst)
	    val newvars = listDiff ptEq (vars, oldvars)
	in addPrimitiveVarsDestructive pvars stvar (newvars@oldvars)
	end
    fun addToCone options findObject pvars lim cone (stvar, vars) = 
	(* Strip that ObjectInst, it bloodily confuses the hashes! *)
	let val funName = "addToCone"
	    val stvar = vtWrap (vtFlag stvar) (stripObjectInst(vtName stvar))
	    val conev = (case getVarConeDirect(cone, stvar) of
			     NONE => []
			   | SOME l => l)
	    val newconev = unionC [conev, vars]
	    val _ = lazyVerbDebug options funName
		(fn()=>"\naddToCone("^(vt2str stvar)^", ["^(cvlist2str "," vars)
		 ^"])\nold cone("^(vt2str stvar)^") = ["^(cvlist2str "," conev)
		 ^"]\nnew cone("^(vt2str stvar)^") = ["^(cvlist2str "," newconev)^"]\n")
	in
	    (addConeDestructive (cone, stvar, newconev);
	     (* Check for circularity EXCEPT for the next state assts *)
	     (case (stvar, inConeOfNoNext options findObject pvars lim cone stvar (PVar (vtName stvar))) of
		  (NextVar _, _) => ()
		| (_, SOME true) => raise TransError ("Circular assignment of: "
						 ^ (vt2str stvar))
		| _ => ()))
	end

    fun addToCones options findObject pvars lim cone ({norm = norm, next = next, init = init}, varsE) =
	(List.app (fn stvar => addToCone options findObject pvars lim cone (stvar, varsE)) norm;
	 List.app (fn stvar => addToCone options findObject pvars lim cone (stvar, varsE)) next;
	 List.app (fn stvar => addToCone options findObject pvars lim cone (stvar, varsE)) init)
	
    fun updateConeDestructive options findObject lim pvars cone (var,expr) =
	(* Strip that ObjectInst, it bloodily confuses the hashes! *)
	let val funName = "updateConeDestructive"
	    val var = vtWrap (vtFlag var) (stripObjectInst(vtName var))
	    val _ = pushFunStackLazy(funName, fn ()=>(vt2str var)^", "^(pt2string expr))
	    fun recur x = updateConeDestructive options findObject lim pvars cone x
	    val vName = vtName var
	    val flag = vtFlag var
	    val vtp = getExprType findObject vName
	    val res =
	     (case vtp of
		TupleType(_,lst) =>
		  let val l = List.length lst
		      fun subCase 0 = ()
			| subCase n = 
			  let val e = (*  evaluateExpr options findObject *) (ExtractTuple(n-1,expr))
			      val g = vtWrap flag (ExtractTuple(n-1,vName))
			  in recur (g,e); subCase(n-1)
			  end
		  in subCase l; cone
		  end
	      | RecordType(_,lst) => 
		  let fun getField (RecordField{name=f,Type=tp}) = (f,tp)
			| getField x = raise SympBug
		           ("updateConeDestructive: not a RecordField: "
			    ^(pt2string x))
		      val pairs = List.map getField lst
		      fun subCase (f,e) = 
			  let val e = (*  evaluateExpr options findObject *) (ExtractRecord(f,expr))
			      val g = vtWrap flag (ExtractRecord(f,vName))
			  in recur(g,e)
			  end
		  in List.map subCase pairs; cone
		  end
	      | EnumType(_,lst) => 
		  let val index =
		       (case getPrimitiveIndexVar pvars vName of
			     SOME x => x
			   | NONE => raise SympBug
			      ("updateConeDestructive: state var doesn't have an index var: "
			       ^(pt2string vName)))
		      val V = varsFrom options findObject pvars lim expr
		      val _ = addToCone options findObject pvars lim cone (vtWrap flag index, V)
		      fun extractor (a as TypeConstr{Type=tp,...}) =
			  (case tp of
			       FunType _ => SOME(fn x=> ExtractAppl(a, x))
			     | _ => NONE)
			| extractor a = raise SympBug
			       ("updateConeDestructive: not a TypeConstr:\n  "
				^(pt2string a))
		      val extractors = List.mapPartial extractor lst
		      fun applyExtr f = 
			  let val e = evaluateExpr options findObject (f expr)
			      val gamma = vtWrap flag (f vName)
			  in SOME(gamma,e)
			  end
		          (* If the evaluation raised an exception, the `expr' is a different 
			     constructor than f tried to extract, thus we can skip it *)
		          handle EvalError _ => NONE
		      val _ = List.map recur (List.mapPartial applyExtr extractors)
		  in cone
		  end
	      | FunType(_,t1,t2) => 
		  (case getTypeValues options lim t1 of
		       NONE => (addToCone options findObject pvars lim cone (var, [DelayedVar expr]);
				cone)
		     | SOME vals => 
			 let fun ff v = (vtWrap flag (Appl(dp,vName,v)),
					 evaluateExpr options findObject (Appl(dp,expr,v)))
			 in (List.map ff vals; cone)
			 end)
	      (* For one of the base types, collect all the vars (no delayed expr) and
	         add them to the cone *)
	      | _ => let val V = varsFrom options findObject pvars lim expr
		     in (addToCone options findObject pvars lim cone (var, V); cone)
		     end)
	    val _ = popFunStackLazy(funName, fn ()=>"finish")
	in res
	end

    (* SIDE EFFECT: may add index vars to `pvars' *)
    fun updateCone options findObject lim pvars cone pair =
	 updateConeDestructive options findObject lim pvars (copyCone cone) pair

    local fun op * (x,y) = Conc(x,y)
    in 
	fun varCone2Str (var,lst) =
	      Str("cone ("^(vt2str var) ^ ") = ")
	      (* *(cvlist2Str ", " (case findHash(cone, var) of
				     NONE => []
				   | SOME l => l)) *)
  	      *(cvlist2Str ", " lst)
	fun varCone2str (var,lst) = Str2string(varCone2Str (var,lst))

	fun cone2Str cone =
	    let val lst = hash2any(fn x=>x)(fn x=>x) cone
	    in (Str "  ")*(Strlist2Str "\n  " (List.map varCone2Str lst))
	    end
	fun cone2str cone = Str2string (cone2Str cone)

	fun prim2Str prim var =
	    let
		val oy = (getPrimitiveVars1 prim var)
		val y = 
		    case oy of
			SOME pv => pv
		      | NONE => []
	    in
		(Str ("primvars ("))*(pt2str var)*(Str ") =\n    ")
		*(Strlist2Str ",\n    " (List.map pt2str y))
	    end

	fun primIndex2Str prim var =
	    let
		val oy = (getPrimitiveIndexVar prim var)
		val y = 
		    case oy of
			SOME pv => [pv]
		      | NONE => []
	    in
		(Str ("indexvar ("))*(pt2str var)*(Str ") = ")
		*(Strlist2Str ",\n    " (List.map pt2str y))
	    end

	fun prim2str prim var = Str2string(prim2Str prim var)

	fun primVars2StrCommon debug (prim as {vars=vars,indexVars=iv}) =
	    let val varlst = hash2any(fn x=>x)(fn x=>x) vars
		val ivlst = hash2any(fn x=>x)(fn x=>x) iv
		val Pt2Str = if debug then pt2strDebug else pt2str
		fun prim2Str(v,lst) = 
		    (Str ("primvars ("))*(Pt2Str v)*(Str ") =\n    ")
		    *(Strlist2Str ",\n    " (List.map Pt2Str lst))
		fun index2Str(v,i) =
		    (Str ("indexvar ("))*(Pt2Str v)*(Str ") = ")*(Pt2Str i)
	    in  (Str "  ")*(Strlist2Str "\n  " (List.map prim2Str varlst))
		*(Str "\n  ")
		*(Strlist2Str "\n  " (List.map index2Str ivlst))
	    end
	val primVars2Str = primVars2StrCommon false
	val primVars2StrDebug = primVars2StrCommon true
	fun primVars2str prim = Str2string(primVars2Str prim)
	fun primVars2strDebug prim = Str2string(primVars2StrDebug prim)
    end

    (* Substitute `arg' for `param' in `tp'. *)
    fun substituteType(arg, param) tp =
	let fun recur x = ptTransform loop x
	    and loop (t as TypeClosure _) = t
	      | loop x = 
		  if ptEq(x, param) then arg
		  else recur x
	in loop tp
	end

    fun instantiateType([],[]) tp = tp
      | instantiateType(arg::alist, param::plist) tp =
	  instantiateType(alist, plist)(substituteType(arg,param) tp)
      | instantiateType(_, _) tp = raise SympBug
	  ("trans/instantiateType: different number of arguments and parameters for\n  "
	   ^(pt2string tp))

    fun atomicModelEq(AtomicModel{name=n1, uname=u1,
				  assts=assts1,cone=cone1,
				  pvars=pvars1, absModules=ams1},
		      AtomicModel{name=n2, uname=u2,
				  assts=assts2,cone=cone2,
				  pvars=pvars2, absModules=ams2}) =
	(* Also need to compare cones and pvars somehow *)
	(case (n1,n2) of
	     (SOME n1, SOME n2) => ptEq(n1,n2) 
	   | (NONE, NONE) => true
	   | _ => false)
	andalso ptEq(u1,u2) andalso eqset atomicModelEq (ams1,ams2)
		      

    fun transEq(TransSync2(t11,t12), TransSync2(t21,t22)) =
	  transEq(t11,t21) andalso transEq(t12,t22)
      | transEq(TransAsync2(t11,t12), TransAsync2(t21,t22)) =
	  transEq(t11,t21) andalso transEq(t12,t22)
      | transEq(TransSync{names=n1,body=b1,...}, TransSync{names=n2,body=b2,...}) =
	  eqset ptEq (n1,n2) andalso transEq(b1,b2)
      | transEq(TransAsync{names=n1,body=b1,...}, TransAsync{names=n2,body=b2,...}) =
	  eqset ptEq (n1,n2) andalso transEq(b1,b2)
      | transEq(TransAtomic m1, TransAtomic m2) = atomicModelEq(m1,m2)
      | transEq _ = false

    fun modelEq({trans=t1,stateVars=vars1,abs=abs1,...}: Model,
		{trans=t2,stateVars=vars2,abs=abs2,...}: Model) =
	transEq(t1,t2) andalso
	(case Option.map (fn l=>List.all ptEq l) (zipOpt(vars1,vars2)) of
	     SOME true => true
	   | _ => false)
        (* Compare the abstractions *)

    fun model2str(model: Model) =
	let val { trans=trans, ...} = model
	    (* To be finished *)
	    fun trans2str(TransAtomic(AtomicModel{assts=assts, pvars=pvars, ...})) =
		("-- Assignments\n\n"
		 ^(avt2str assts)
		 (* ^"\n\n--Primitive variables:\n\n"*)
		 ^"\n\n")
	      | trans2str _ = "<trans_rel>"
	in 
	    "Model{\n   TRANSITION RELATION:\n     "
	    ^(trans2str trans)
	    ^"\n     }"
	end

  end
