(* Internal signature for parts of the trans.{sig,sml} implementation.
   Do not use it anywhere else. *)

signature TRANS_COMMON =
  sig
    structure Abstract: ABSTRACT
    structure Evaluate: EVALUATE
    sharing Abstract.ParserDefault = Evaluate.ParserDefault
    type Abstraction = Abstract.Abstraction
    structure Hash: HASH
    type ParseTree = Abstract.ParserDefault.ParseTreeStruct.ParseTree
    type Str = Str.Str
    type options = Abstract.ParserDefault.Interface.Options.options

    datatype VarType =	NormalVar of ParseTree
      | NextVar of ParseTree
      | InitVar of ParseTree

    datatype ConeVar =
	PVar of ParseTree
      | DelayedVar of ParseTree (* Delayed computation of cone, DelayedVar(expr) *)

    (* Type for a flag that tells what type of assigmnent we are
       dealing with.  *)
    datatype VarTypeFlag = NormalFlag | NextFlag | InitFlag

    (* Cone of influence.
       Here we define the type explicitly,
       but this signature will only be used within Trans. *)
    type Cone

    (* correspondence b/w SyMP state vars and more primitive vars in the
       translation, a tree hierarchy down to the truely primitive vars.
       Implemented as a hash mapping a state var to its list of primitive vars. *)
    type PrimitiveVars

    type AsstVars = {norm: VarType list,
		     next: VarType list,
		     init: VarType list}

    (* Representation of sequent C;N;L;I |- A1;...;An *)
    (* type AsstSeq = AsstVars * (ParseTree list) *)

    datatype AsstVarsTree =
        (* vars |- varName := expr *)
	NormalAsstTree of AsstVars * ParseTree * ParseTree
      | NextAsstTree of AsstVars * ParseTree * ParseTree
      | InitAsstTree of AsstVars * ParseTree * ParseTree
      | NopAsstTree of AsstVars
      | ListAsstTree of AsstVars * (AsstVarsTree list)
        (* vars |- let [defs] in <asst> end *)
      | LetAsstTree of AsstVars * (ParseTree list) * AsstVarsTree
        (* vars |- case selExpr of [(pat1, asst1), ..., (patN, asstN)] endcase *)
      | CaseAsstTree of AsstVars * ParseTree * ((ParseTree * AsstVarsTree) list)
        (* vars |- if c1 then asst1 elsif c2 then asst2 .... else asstN *)
      | IfAsstTree of AsstVars * ((ParseTree * AsstVarsTree) list) * AsstVarsTree
      | ChooseAsstTree of AsstVars * (ParseTree list option) * ((ParseTree * AsstVarsTree) list)
      | ForeachAsstTree of AsstVars * (ParseTree list) * AsstVarsTree
      | LabeledAsstTree of AsstVars * ParseTree * AsstVarsTree

    (* Transition relation of a leaf module (not a composition of any other modules) *)
    datatype AtomicModel =
	AtomicModel of { name: ParseTree option,
			 uname: ParseTree,
			 assts: AsstVarsTree,
			 cone: Cone,
			 pvars: PrimitiveVars,
			 (* We might not need this list, as we have the master list *)
			 (* stateVars: ParseTree list *)
			 (* Abstraction modules nested into the current module.
			    These modules may have further abstractions inside them. *)
			 absModules: AtomicModel list
		       }

    (* Transition relation for the model representaiton *)
    datatype TransRel = 
	(* Parallel composition of two modules *)
	TransSync2 of TransRel * TransRel
      | TransAsync2 of TransRel * TransRel
	(* Parallel composition over the parameters' ranges *)
      | TransSync of { names: ParseTree list,
		       body: TransRel,
		       parent: ParseTree }
      | TransAsync of { names: ParseTree list,
		        body: TransRel,
			parent: ParseTree }
      | TransAtomic of AtomicModel

    (* Representation of a model in the sequent *)
    type Model = { trans: TransRel,
		   findObject: ParseTree -> ParseTree option,
		   (* List of StateVar objects *)
		   stateVars: ParseTree list,
		   (* The united set of primitive variables *)
		   pvars: PrimitiveVars,
		   abs: Abstraction }

    (* Check if the expression is a constant for the purposes of COI reduction *)
    val isConst: ParseTree -> bool

    val vtName: VarType -> ParseTree
    val vtFlag: VarType -> VarTypeFlag
    val vtWrap: VarTypeFlag -> ParseTree -> VarType
    val vtEq: VarType * VarType -> bool
    val vtEqName: VarType * VarType -> bool
    val cvName: ConeVar -> ParseTree option
    val cvEq: ConeVar * ConeVar -> bool
    val cvValue : ConeVar -> ParseTree
    val makeCone: unit -> Cone
    val addCone: Cone * VarType * ConeVar list -> Cone
    val addConeDestructive: Cone * VarType * ConeVar list -> Cone
    val addPrim: PrimitiveVars -> ParseTree * (ParseTree list) -> PrimitiveVars
    (* addToCone options pvars TypeSizeLimit cone var varList
       SIDE EFFECT: may add index vars to pvars *)
    val addToCone: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> Cone -> VarType * (ConeVar list) -> unit
    val addToCones: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> Cone -> AsstVars * (ConeVar list) -> unit
    (* getVarConeHash options pvars TypeSizeLimit depth cone var
       SIDE EFFECT: may add index vars to pvars *)
    val getVarConeHash: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> int -> Cone -> VarType -> (ConeVar,unit) Hash.Hash
    val getVarConeHashNoNext: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> int -> Cone -> VarType -> (ConeVar,unit) Hash.Hash
    val getVarConeN: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> int -> Cone -> VarType -> ConeVar list
    val getVarConeNnoNext: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> int -> Cone -> VarType -> ConeVar list
    val getVarCone: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> Cone -> VarType -> ConeVar list
    val getVarConeNoNext: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> Cone -> VarType -> ConeVar list
    (* inConeOf options pvars TypeSizeLimit cone var conevar
       SIDE EFFECT: may add index vars to pvars *)
    val inConeOf: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> Cone -> VarType -> ConeVar -> bool option
    (* Make a clean copy of the cone for later destructive modifications *)
    val copyCone: Cone -> Cone
    val makePrimitiveVars: unit -> PrimitiveVars
    val copyPrimitiveVars: PrimitiveVars -> PrimitiveVars

    (* Return the list of all primitive vars for a state var `s', or NONE
       if `s' is not in the database. *)
    val getPrimitiveVars: PrimitiveVars -> ParseTree -> ParseTree list option
    (* Get the first layer of `more primitive vars' of s *)
    val getPrimitiveVars1: PrimitiveVars -> ParseTree -> ParseTree list option
    val addPrimitiveVars: PrimitiveVars -> ParseTree -> ParseTree list -> PrimitiveVars
    val addPrimitiveVarsDestructive: PrimitiveVars -> ParseTree -> ParseTree list -> PrimitiveVars
    (* Find/insert an internal index variable for a state var of EnumType *)
    val getPrimitiveIndexVar: PrimitiveVars -> ParseTree -> ParseTree option
    val addPrimitiveIndexVar: PrimitiveVars -> ParseTree -> ParseTree -> PrimitiveVars
    val addPrimitiveIndexVarDestructive: PrimitiveVars -> ParseTree -> ParseTree -> PrimitiveVars

    (* Merge the two sets of primitive vars into a new one (no side effects) *)
    val mergePrimitiveVars: PrimitiveVars * PrimitiveVars -> PrimitiveVars

    (* updateCone options TypeSizeLimit pvars cone (var, expr)
       SIDE EFFECT: may add index vars to pvars *)
    val updateCone: options -> (ParseTree -> ParseTree option) -> int -> PrimitiveVars -> Cone -> VarType * ParseTree -> Cone
    val updateConeDestructive: options -> (ParseTree -> ParseTree option) -> int -> PrimitiveVars -> Cone
	  -> VarType * ParseTree -> Cone

    (* val getAsstSeq: AsstVarsTree -> AsstSeq *)
    (* val getPredTrees: AsstVarsTree -> AsstVarsTree list *)
    val getAsstVars: AsstVarsTree -> AsstVars
    (* Set new state variables to the top-level node *)
    val setAsstVars: AsstVars -> AsstVarsTree -> AsstVarsTree
    (* Balance the vars in the conditional branches.
       Assumes the sets of vars are "reasonable" at each node. *)
    val balanceAsstTree: AsstVarsTree -> AsstVarsTree
    (* Completely recomputes the set of variables and balances the tree *)
    val recomputeAsstVars: AsstVarsTree -> AsstVarsTree
    (* substAsstTree(v,expr,tree):
       substitute `expr' for var `v' in the `tree' *)
    val substAsstTree: ParseTree * ParseTree -> AsstVarsTree -> AsstVarsTree
    val union : ('a * 'a -> bool) -> 'a list list -> 'a list
    val unionC : ConeVar list list -> ConeVar list
    val unionV : VarType list list -> VarType list
    val unionAV : AsstVars list -> AsstVars
    val varsFrom: options -> (ParseTree -> ParseTree option)
	  -> PrimitiveVars -> int -> ParseTree -> ConeVar list

    val vt2str: VarType -> string
    val vtlist2str: string -> VarType list -> string

    val varCone2Str : VarType * (ConeVar list) -> Str
    val varCone2str : VarType * (ConeVar list) -> string
    val cv2str: ConeVar -> string
    val cvlist2str: string -> ConeVar list -> string
    val cone2Str : Cone  -> Str
    val cone2str : Cone  -> string

    val prim2Str : PrimitiveVars -> ParseTree -> Str
    val prim2str : PrimitiveVars -> ParseTree -> string
    val primVars2Str : PrimitiveVars -> Str
    val primVars2str : PrimitiveVars -> string
    val primVars2StrDebug : PrimitiveVars -> Str
    val primVars2strDebug : PrimitiveVars -> string

    val avt2Str : AsstVarsTree -> Str
    val avt2str : AsstVarsTree -> string

    val modelEq: Model * Model -> bool
    val model2str: Model -> string

    (* Check whether two state vars are similar upto non-value
       expressions as array indices. Sequent g1 ~= g2. *)
    val svSimilar: ParseTree * ParseTree -> bool
    (* Check whether g1 is a sub-var of g2.  Sequent g1 <=~ g2. *)
    val svPartOf: ParseTree * ParseTree -> bool

    val substituteType: (ParseTree * ParseTree) -> ParseTree -> ParseTree
    val instantiateType: (ParseTree list) * (ParseTree list) -> ParseTree -> ParseTree
  end

