functor TransSMVFun (structure Hash : HASH
		     structure Abstract : ABSTRACT): TRANS_SMV =

  struct

    structure Hash = Hash
    structure Abstract = Abstract

    open Str
    open ListOps
    open Hash
    open Abstract
    open Pos
    open ParserDefault
    open ParseTreeStruct
    open Interface
    open Options

    type VarSMVMapping = (ParseTree, Str) Hash
    type DefineMapping = (ParseTree, Str) Hash
    type TypeValueMapping = (ParseTree, Str list) Hash

    fun ptListEq ([], []) = true
      | ptListEq (x::l1, y::l2) =
	  if ptEq(x,y) then ptListEq(l1,l2)
	  else false
      | ptListEq _ = false

    (* Redefine name generator _not_ to include `!' symbols *)
    fun newName () = "X_"^(Int.toString(newNumber()))

    fun makeMapping () = makeHashDefault (ptEq, pt2string) 
    fun makeTVMapping () = makeHashDefault (ptEq, pt2string) 
    fun addMapDestructive(map, v, smv) = 
	(lazyVerbDebug (getOptions()) "addMap" (fn()=>"addMap("^(pt2string v)^")\n");
	 insertHashDestructive(map, v, smv))

    fun map2str map =
	  "["^(strlist2str ",\n  "
	       (List.map(fn(v,v')=>(pt2stringDebug v)^" -> "^(Str2string v'))
	                (hash2any(fn x=>x)(fn x=>x) map)))
	  ^"]"

    fun TVMap2str map =
	  "["^(strlist2str ",\n  "
	       (List.map(fn(v,v')=>(pt2stringDebug v)^" -> "^(Str2string(Strlist2Str "," v')))
		        (hash2any(fn x=>x)(fn x=>x) map)))
	  ^"]"

    fun getMapOpt(map, v) =
	let val _ = pushFunStackLazy("TransSMV.getMapOpt",
				     fn()=>(pt2string v)^", "^(map2str map))
	    fun loop(v as Builtin{name=v', ...}) =
		(case findHash(map, v) of
		     SOME s => SOME s
		   | NONE => loop v')
	      | loop v = findHash(map, v)
	    val res = loop v
	    val _ = popFunStackLazy("TransSMV.getMapOpt",
				    fn()=>option2string "NONE" (Option.map Str2string res))
	in res
	end

    fun getMap(map, v) =
	(case getMapOpt(map, v) of
	     SOME s => s
	   | NONE => raise SympBug
		 ("trans_smv:  variable not in hash: "^(pt2stringDebug v)))

    fun getTVMap(map, v) =
	let val _ = pushFunStackLazy("TransSMV.getTVMap",
				     fn()=>(pt2string v)^", "^(TVMap2str map))
	    val res = (case findHash (map, v) of
			   SOME s => s
			 | NONE => raise SympBug
			       ("trans_smv:  variable not in hash: "^(pt2stringDebug v)))
	    val _ = popFunStackLazy("TransSMV.getTVMap",
				    fn()=>"["^(Str2string(Strlist2Str "," res))^"]")
	in res: Str list
	end

    fun isComplex (Appl (_, Builtin{name=And _, ...}, _)) = true
      | isComplex (Appl (_, Builtin{name=Or _, ...}, _)) = true
      | isComplex (Appl (_, Builtin{name=Implies _, ...}, _)) = true
      | isComplex (Appl (_, Builtin{name=Eq _, ...}, _)) = true
      | isComplex (Appl (_, Builtin{name=NotEq _, ...}, _)) = true
      | isComplex (Appl (_, Builtin{name=Not _, ...}, _)) = true
      | isComplex x = false


    val bottom = Conc (Str (newName ()), Str "bot")
    val truth = Conc (Str (newName ()), Str "true")
    val falsity = Conc (Str (newName ()), Str "false")
    val eqTrue = Conc (Str " = ", truth)
    val eqFalse = Conc (Str " = ", falsity)
    val eqBottom = Conc (Str " = ", bottom)
    val thenTrue = Conc (Str " : ", Conc (truth, Str ";"))
    val thenFalse = Conc (Str " : ", Conc (falsity, Str ";"))
    val thenBottom = Conc (Str " : ", Conc (bottom, Str ";"))
    val otherwise = (Str "\n  1")
    val || = (Str " | ")
    val & = (Str " & ")
    val dummyBool = Conc (Str (newName ()), Str "ensureBoolean")

    (* Functions to translate 1- and 2-arity functions.
       There should be NO trailing `;' at the end. *)

    local fun op * (x,y) = Conc(x,y)
    in 
      fun codeFor1 (Not _, s) = (Str "!(") * s * (Str ")")
	| codeFor1 (Ag _, s) = (Str "AG(") * s * (Str ")")
	| codeFor1 (Af _, s) = (Str "AF(") * s * (Str ")")
	| codeFor1 (Eg _, s) = (Str "EG(") * s * (Str ")")
	| codeFor1 (Ef _, s) = (Str "EF(") * s * (Str ")")
	| codeFor1 (Ax _, s) = (Str "AX(") * s * (Str ")")
	| codeFor1 (Ex _, s) = (Str "EX(") * s * (Str ")")
	| codeFor1 (Object {def = d, ...}, s) = codeFor1 (d, s)
	| codeFor1 (ObjectInst {obj = d, ...}, s) = codeFor1 (d, s)
	| codeFor1 (Builtin {name=n, ...}, s) = codeFor1 (n, s)
	| codeFor1 (x, s) = raise SympBug 
	  ("trans_SMV:  Unexpected arity-1 operator in formula:\n  "
	   ^(pt2string x)^"("^(Str2string s)^")")
	  
      fun codeFor2 (And _, s1, s2) =
	   (Str "(") * s1 * (Str ")")* & * (Str "(") * s2 * (Str ")")
	| codeFor2 (Or _, s1, s2) =
	   (Str "(") * s1 * (Str ")") * || * (Str "(") * s2 * (Str ")")
	| codeFor2 (Implies _, s1, s2) =
	   (Str "(") * s1 * (Str ")") * (Str " -> ") * (Str "(") * s2 * (Str ")")
	| codeFor2 (Eq _, s1, s2) =
	   (Str "(") * s1 * (Str ")") * (Str " = ") * (Str "(") * s2 * (Str ")")
	| codeFor2 (NotEq _, s1, s2) =
	   (Str "! ((") * s1 * (Str ")") * (Str " = ") * (Str "(") * s2 * (Str "))")
	| codeFor2 (Au _, s1, s2) = (Str "A[") * s1 * (Str "U") * s2 * (Str "]")
	| codeFor2 (Eu _, s1, s2) = (Str "E[") * s1 * (Str "U") * s2 * (Str "]")
	| codeFor2 (Ar _, s1, s2) = codeFor1(Not dp,
					     codeFor2(Eu dp,
						      codeFor1(Not dp, s1),
						      codeFor1(Not dp, s2)))
	| codeFor2 (Er _, s1, s2) = codeFor1(Not dp,
					     codeFor2(Au dp,
						      codeFor1(Not dp, s1),
						      codeFor1(Not dp, s2)))
	| codeFor2 (Object {def = d, ...}, s1, s2) = codeFor2 (d, s1, s2)
	| codeFor2 (ObjectInst {obj = d, ...}, s1, s2) = codeFor2 (d, s1, s2)
	| codeFor2 (Builtin {name=n, ...}, s1, s2) = codeFor2 (n, s1, s2)
	| codeFor2 (x, s1, s2) = raise SympBug 
	  ("trans_SMV:  Unexpected arity-2 operator in formula:\n  "
	   ^(pt2string x)^"("^(Str2string s1)^", "^(Str2string s1)^")")
    end

    fun formulaStr options (varMap, formula) =
      let val debug = lazyVerbDebug options "formulaStr"
	  val _ = pushFunStackLazy("formulaStr", fn()=>(pt2string formula)
				   ^", varMap = "^(map2str varMap))
	  fun op * (x,y) = Conc(x,y)
	fun formStr a =
	    let val _ = pushFunStackLazy("formulaStr/formStr", fn()=>pt2string a)
		val res = 
		    (case getMapOpt(varMap, a) of
			 SOME x => x
		       | NONE => 
			     (case a of
				  (Appl (_, x, TupleExpr (_, [v1, v2]))) =>
				      codeFor2 (x, formStr v1, formStr v2)
				| (Appl (_, x, v)) => codeFor1 (x, formStr v)
				| (Next v) => (Str "next(")*(formStr v)*(Str ")")
				| (Builtin{name=True _, ...}) => truth
				| (Builtin{name=False _, ...}) => falsity
				| (Builtin{name=Undefined _, ...}) => bottom
				| True2 => Str "1"
				| False2 => Str "0"
				| x => raise SympBug
				      ("formulaStr: unexpected expression: "^(pt2stringDebug x))))
		val _ = popFunStackLazy("formulaStr/formStr", fn()=>Str2string res)
	    in
		res
	    end
	val _ = debug(fn()=>"varMap = ["
		      ^(strlist2str ",\n          "
			(List.map(fn(v,v')=>(pt2string v)^" -> "^(Str2string v'))
			 (hash2any(fn x=>x)(fn x=>x) varMap)))
		      ^"]\n")
	val str = formStr formula
	val _ = popFunStackLazy("formulaStr", fn()=>(Str2string str))
      in
	  (Str "", str)
      end
		    
    fun SMVcode options findObject limit specList varList initBool invarBool transBool =
      let
	val debug = lazyVerbDebug options "SMVcode"
        (* Eliminate repeated entries in a ParseTree list; namely varList *)
	fun eliminateRepeated lst =
	    let val hash = makeHashDefault(ptEq, pt2string)
		fun hash2list() = List.map #1 (hash2any(fn x=>x)(fn x=>x) hash)
		fun add x = (insertHashDestructive(hash, x, ()); ())
	    in 
		(List.app add lst; hash2list())
	    end
	val varList = eliminateRepeated varList
	val booleanSeen = ref false
	fun op * (x,y) = Conc(x,y)
	val formulaStr = formulaStr options
	val map = makeMapping ()
	val _ = (addMapDestructive (map, True(dp), truth);
		 addMapDestructive (map, False(dp), falsity))
	val TVmap = makeTVMapping ()
	fun innerTypeStr v = 
	  let 
	    val tvals = getTVMap(TVmap, v)
	  in
	    Strlist2Str ", " tvals
	  end
	fun typeStr v =  (Str " : {") *  (innerTypeStr v) * (Str "}")
	val _ = debug(fn()=>"\n-- SMVcode: generating types for vars:\n  ["
		      ^(ptlist2str "," varList)^"]\n")
	val str = Str "\nMODULE main\n\n"
	fun doVar v =
	    let val _ = addMapDestructive (map, v, (Str(newName ())))
		val t = getExprType findObject v
		val tvals = getTypeValues options limit t
		val _ = (case t of 
			     BoolType (_) => booleanSeen := true
			   | _ => ())
	    in
		case t of
		    AbstractType (alist) =>
			let val names = List.map 
				 (fn name => 
				  case (getMapOpt(map, name)) of
				      SOME n => n
				    | NONE => Str (newName ())) alist
			in
			     (addMapDestructive (TVmap, v, bottom :: names);
			      List.app (fn n =>
					case (getMapOpt (map, List.nth (alist, n))) of
					    NONE => (addMapDestructive 
						     (map, List.nth (alist, n),
						      List.nth (names, n));
						     ())
					  | SOME _ => ())
			      (List.tabulate (List.length alist, fn x => x)))
			end
		  | _ => 
			 (case tvals of
			      SOME vlist =>
				  let val names = List.map 
				      (fn name => 
				           case (getMapOpt (map, name)) of
					       SOME n => n
					     | NONE => Str (newName ())) vlist
				  in
				      (addMapDestructive (TVmap, v, bottom::names);
				       List.app (fn n =>
						 case (getMapOpt (map, List.nth (vlist, n))) of
						     NONE =>
							 (addMapDestructive 
							  (map, 
							   List.nth (vlist, n),
							   List.nth (names, n));
							  ())
						   | SOME _ => ())
				       (List.tabulate (List.length vlist, fn x => x)))
				  end
			    | NONE => raise ProverError
				  ("Infinite type in translation to SMV:\n  Type: "
				   ^(pt2stringDebug t)
				   ^"\n  Expr: "^(pt2string v)))
	    end
	val _ = List.app doVar varList
	val _ = debug(fn()=>"\n-- SMVcode: generating var mapping...\n")
	val comments = Strlist2Str "\n"
	  ((Str "-- SMV Code Automatically Generated by SyMP\n\n--Variable mappings:\n")
	   ::(List.map 
	      (fn v => (Str "-- ")*(pt2str v)*(Str " -> ")
	              *(getMap(map, v))*(typeStr v))
	      varList))
	val comments = comments
	    *(Str "\n\n--Mapping of all names (including variables)\n")
	    *(Strlist2Str "\n"
	      (List.map(fn(v,str)=>(Str "-- ") * v * (Str " -> ")* str)
	       ((Str "undefined", bottom)::(hash2any pt2str (fn x=>x) map))))

	(* val _ = debug(fn()=>"\n-- SMVcode: vars and their mappings:\n  ["
		      ^(strlist2str ",\n   "
			(List.map(fn v=>(pt2stringDebug v)^" -> "
				  ^(Str2string (getMap options (map, v)))
				  ^" : {"
				  ^(Str2string (Strlist2Str ", " (getTVMap options (TVmap, v))))
				  ^"}") varList))
		      ^"]\n") *)
	(* val constList = substract ptEq (List.map #1 (hash2any(fn x=>x)(fn x=>x) map), varList) *)
	val _ = debug(fn()=> Str2string comments)
	val _ = debug(fn()=>"\n-- SMVcode: generating VAR declarations...\n")
	val str = comments * str
	val str = str * (Strlist2Str "\n  "
			 (((Str ("\n\nVAR\n  ")) *
			   (if (!booleanSeen) then (Str "")
			    else
			      (dummyBool * (Str ": {") * bottom * (Str ", ") * truth 
			       * (Str", ") * falsity * (Str "};\n  "))))
			  ::(List.map (fn v => (getMap(map, v))
				              *(typeStr v)*(Str ";"))
			     varList)))
	val _ = debug(fn()=>"\n-- SMVcode: generating formula for INIT...\n")
	val (initDef, initForm) = formulaStr (map, initBool)
	val initStr = initDef * (Str "\nINIT\n  ") * initForm * (Str "\n")
	val _ = debug(fn()=>"\n-- SMVcode: generating formula for INVAR...\n")
	val (invarDef, invarForm) = formulaStr (map, invarBool)
	val invarStr = invarDef * (Str "\nINVAR\n  ") * invarForm * (Str "\n")
	val _ = debug(fn()=>"\n-- SMVcode: generating formula for TRANS...\n")
	val (transDef, transForm) = formulaStr (map, transBool)
	val transStr = transDef * (Str "\nTRANS\n  ") * transForm * (Str "\n")
	val specStr = List.foldr Conc (Str "") 
	  (List.map
	   (fn s => 
	    let
	      val (specDef, specForm) = formulaStr (map, s)
	    in
	      (Str "\nSPEC\n  ") * specForm * (Str "\n")
	    end) specList)
	val _ = debug(fn()=>"\n-- SMVcode: Translation finished.\n")
	val str =  str *  initStr * invarStr * transStr * specStr (* * errorStr *)
      in
	(str, map, TVmap)
      end

  end
