signature TRANS =
  sig

    (* "Proof terms" for keeping sets of assigned variables in
    different parts of assignments.  Corresponds to the sequent C;N;L;I |- A. *)

    structure Abstract: ABSTRACT
    structure Evaluate: EVALUATE
    sharing Evaluate.ParserDefault = Abstract.ParserDefault
    type Abstraction = Abstract.Abstraction
    type ParseTree = Evaluate.ParserDefault.ParseTreeStruct.ParseTree
    type Str = Str.Str
    type options = Evaluate.ParserDefault.Interface.Options.options

    exception TransError of string

    type Cone (* Cone of influence *)

    (* correspondence b/w SyMP state vars and primitive vars in the translation *)
    type PrimitiveVars

    datatype VarType =
	NormalVar of ParseTree
      | NextVar of ParseTree
      | InitVar of ParseTree

    type AsstVars = {norm: VarType list,
		     next: VarType list,
		     init: VarType list}

    datatype ConeVar =
	PVar of ParseTree
      | DelayedVar of ParseTree (* Delayed computation of cone, DelayedVar(expr) *)

    (* Representation of sequent N;L;I |- A1;...;An *)
    (* type AsstSeq = AsstVars * (ParseTree list) *)

    datatype AsstVarsTree =
        (* vars |- varName := expr *)
	NormalAsstTree of AsstVars * ParseTree * ParseTree
      | NextAsstTree of AsstVars * ParseTree * ParseTree
      | InitAsstTree of AsstVars * ParseTree * ParseTree
      | NopAsstTree of AsstVars
      | ListAsstTree of AsstVars * (AsstVarsTree list)
        (* vars |- let [defs] in <asst> end *)
      | LetAsstTree of AsstVars * (ParseTree list) * AsstVarsTree
        (* vars |- case selExpr of [(pat1, asst1), ..., (patN, asstN)] endcase *)
      | CaseAsstTree of AsstVars * ParseTree * ((ParseTree * AsstVarsTree) list)
        (* vars |- if c1 then asst1 elsif c2 then asst2 .... else asstN *)
      | IfAsstTree of AsstVars * ((ParseTree * AsstVarsTree) list) * AsstVarsTree
      | ChooseAsstTree of AsstVars * (ParseTree list option) * ((ParseTree * AsstVarsTree) list)
      | ForeachAsstTree of AsstVars * (ParseTree list) * AsstVarsTree
      | LabeledAsstTree of AsstVars * ParseTree * AsstVarsTree

    (* Transition relation of a leaf module (not a composition of any other modules) *)
    datatype AtomicModel =
	AtomicModel of { name: ParseTree option,
			 uname: ParseTree,
			 assts: AsstVarsTree,
			 cone: Cone,
			 pvars: PrimitiveVars,
			 (* We might not need this list, as we have the master list *)
			 (* stateVars: ParseTree list *)
			 (* Abstraction modules nested into the current module.
			    These modules may have further abstractions inside them. *)
			 absModules: AtomicModel list
		       }

    (* Transition relation for the model representaiton *)
    datatype TransRel = 
	(* Parallel composition of two modules *)
	TransSync2 of TransRel * TransRel
      | TransAsync2 of TransRel * TransRel
	(* Parallel composition over the parameters' ranges *)
      | TransSync of { names: ParseTree list,
		       body: TransRel,
		       parent: ParseTree }
      | TransAsync of { names: ParseTree list,
		        body: TransRel,
			parent: ParseTree }
      | TransAtomic of AtomicModel

    (* Representation of a model in the sequent *)
    type Model = { trans: TransRel,
		   findObject: ParseTree -> ParseTree option,
		   (* List of StateVar objects *)
		   stateVars: ParseTree list,
		   (* The united set of primitive variables *)
		   pvars: PrimitiveVars,
		   abs: Abstraction }

    type rebuildAsstOptions

    val vtName: VarType -> ParseTree

    (* val getAsstSeq: AsstVarsTree -> AsstSeq *)
    (* val getPredTrees: AsstVarsTree -> AsstVarsTree list *)
    val getAsstVars: AsstVarsTree -> AsstVars
    (* Set new state variables to the top-level node *)
    val setAsstVars: AsstVars -> AsstVarsTree -> AsstVarsTree

    val avt2Str : AsstVarsTree -> Str
    val avt2str : AsstVarsTree -> string
    val varCone2Str : VarType * (ConeVar list) -> Str
    val varCone2str : VarType * (ConeVar list) -> string
    val cone2Str : Cone  -> Str
    val cone2str : Cone  -> string
    val prim2Str : PrimitiveVars -> ParseTree -> Str
    val prim2str : PrimitiveVars -> ParseTree -> string
    val primVars2Str : PrimitiveVars -> Str
    val primVars2str : PrimitiveVars -> string
    (* Return the list of vars in the cone of a given var.
       getVarConeN collects only variables up to a given depth. 
       (~1 = `entire cone') *)
    val getVarConeN: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> int -> Cone -> VarType -> ConeVar list
    val getVarCone: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> Cone -> VarType -> ConeVar list
    val inConeOf: options -> (ParseTree -> ParseTree option) -> PrimitiveVars -> int
	  -> Cone -> VarType -> ConeVar -> bool option
    (* Make a clean copy of the cone for later destructive modifications *)
    val copyCone: Cone -> Cone

    (* Operations on PrimitiveVars *)
    val makePrimitiveVars: unit -> PrimitiveVars
    val mergePrimitiveVars: PrimitiveVars * PrimitiveVars -> PrimitiveVars

    (* substAVT options findObject pvars lim (tree, old, new) 
       Substitute `new' for `old' in the `tree' and return the new tree. *)
    val substAVT: options -> (ParseTree -> ParseTree option) -> PrimitiveVars
           -> int -> (AsstVarsTree * ParseTree * ParseTree) -> AsstVarsTree

    (* For a list of assignments generate an AsstVarsTree.  On the
       way, check for circular and multiple assignments, and if found
       any, raise TransError. *)

    (* We are not using Abstraction at this stage yet.  We generate
       the most general representation of the transition relation and
       reuse it later. *)

    val collectAsstVars: options -> (ParseTree -> ParseTree option) -> int
	  -> (ParseTree list * ParseTree list) 
          -> (AsstVarsTree * Cone * PrimitiveVars)

    (* Customize an AsstVarsTree to a particular property under a
       particular abstraction *)

    val customizeAsstVars: options -> (ParseTree -> ParseTree option) ->
      PrimitiveVars -> Cone -> int-> ParseTree list -> Abstraction -> 
      AsstVarsTree -> AsstVarsTree

    (* Traverse the AsstTree, extract the ground vars from the
       assignments and add them to the tree.  The resulting tree is
       then balanced.  On the way, construct the new cone and
       PrimitiveVars database.  Returns (asstTree, cone, pvars).

       This is a handy function to apply to an abstracted or
       COI-filtered assignment tree.

       rebuildAsstVars options findObject lim (stateVars, asstTree) *)

    val rebuildAsstVars: options -> (ParseTree -> ParseTree option)
          -> PrimitiveVars -> int -> (ParseTree list option * AsstVarsTree * rebuildAsstOptions)
          -> (AsstVarsTree * Cone * PrimitiveVars)

    val rebuildAsstOptionsDefault: rebuildAsstOptions
    val rebuildAsstOptionsCheckAssts: rebuildAsstOptions -> rebuildAsstOptions
    val rebuildAsstOptionsBalance: rebuildAsstOptions -> rebuildAsstOptions

    (* generateTrans options abs TypeSizeLimit asstTree
       Generate a formula representing the transition relation given
       by the AsstVarsTree. *)
    val generateTrans: options -> (ParseTree -> ParseTree option) -> Abstraction
	  -> PrimitiveVars -> int -> AsstVarsTree
	  -> ParseTree * ParseTree * ParseTree

    (* boolExpr options abs lim expr
       Translate a boolean expression into a boolean formula suitable for 
       the use in external model checkers *)
    val boolExpr: options -> (ParseTree -> ParseTree option) -> ParseTree list
	  -> Abstraction -> PrimitiveVars -> int -> ParseTree -> ParseTree

    (* Translate a theorem declaration into a primitive boolean formula *)
    val generateTheorem: options -> (ParseTree -> ParseTree option) -> ParseTree list 
	  -> Abstraction -> PrimitiveVars -> int -> ParseTree -> ParseTree

    (* Perform COI reduction on the model, along with other necessary
       required transformations before generating the transition
       relation *)
    val specializeModel: options -> Model -> ParseTree list -> Model

    (* Generate the 3 transition relation formulas (norm, next, init) for a Model *)
    val generateModelTrans: options -> Model -> ParseTree * ParseTree * ParseTree

    val getModuleAssts: ParseTree -> ParseTree list

    val makeModel: ParseTree * (ParseTree -> ParseTree option) -> Model
    val modelEq: Model * Model -> bool
    val model2str: Model -> string
    (* Later we either extend the functions above, or include more
       functions, to compute an abstracted transition relation,
       relative to a provided `Abstraction'. *)

  end
