(* SYMP parser in SML'97 *)

structure ParseTreeStruct = ParseTreeStruct
structure ParseError = ParseError
open ParseTreeStruct
open ParseError

fun tuple2list (TupleType(_,lst)) = lst
  | tuple2list (Group(_,x)) = [x]
  | tuple2list x = [x]

fun enum2list (EnumType(_,lst)) = lst
  | enum2list (Id x) = [Id x]
  | enum2list (Of x) = [Of x]
  | enum2list x = 
      let val str = ("\nError: invalid enumeration type entry at "
		     ^(pos2string(pos x))^": \n"
		     ^(pt2string x)^"\n")
      in
          ((* TextIO.output(TextIO.stdErr,str);
	   TextIO.flushOut(TextIO.stdErr); *)
	   wasError := true;
           raise ParseError str
	   (* [x] *))
      end

fun nondet2list (NondetExpr(_,lst)) = lst
  | nondet2list x = [x]

fun pattern2list (ChoicePattern(_,lst)) = lst
  | pattern2list x = [x]

fun ungroup(Group(_,x)) = x
  | ungroup x = x

fun tvtype(TypedVars(_,_,tp)) = tp
  | tvtype _ = NONE

fun tvlist(TypedVars(_,l,_)) = l
  | tvlist _ = []

fun sorttypes lst =
    let fun collect [] front acc = loop [] (front@acc)
          | collect ((Group(p,x))::tl) front acc =
               loop tl (x::(front@acc))
          | collect ((TypedVars(p,vars,tp))::tl) front acc =
               loop tl ((TypedVars(p,(List.rev front)@vars,tp))::acc)
          | collect ((Id x)::tl) front acc = collect tl ((Id x)::front) acc
          | collect (x::tl) front acc =
               loop tl (x::(front@acc))
        and loop [] acc = List.rev acc
          | loop x acc = collect x [] acc
    in loop lst [] end

%%

%name SYMP

%verbose

%pure

%header (functor SYMPLrValsFun(structure Token : TOKEN
			       structure ParseTreeStruct: PARSE_TREE
			       structure ParseError: PARSE_ERROR)
 : sig structure ParserData : PARSER_DATA
            where type result = ParseTreeStruct.ParseTree;
       structure Tokens : SYMP_TOKENS
   end)

%eop EOF
%pos int
%noshift EOF

%term MODULE of Pos.Pos
| BEGIN of Pos.Pos
| END of Pos.Pos
| LET of Pos.Pos
| IN of Pos.Pos
| VAR of Pos.Pos
| VAL of Pos.Pos
| FUN of Pos.Pos
| FN of Pos.Pos
| OP of Pos.Pos
| AS of Pos.Pos
| THEOREM of Pos.Pos
| REFINES of Pos.Pos
| SIMULATES of Pos.Pos
| ARRAY of Pos.Pos
| FAIRNESS of Pos.Pos
| CASE of Pos.Pos
| ENDCASE of Pos.Pos
| AND of Pos.Pos
| OR of Pos.Pos
| IMPLIES of Pos.Pos
| IFF of Pos.Pos
| NOT of Pos.Pos
| FORALL of Pos.Pos
| FOREACH of Pos.Pos
| ENDFOREACH of Pos.Pos
| EXISTS of Pos.Pos
| AG of Pos.Pos
| EG of Pos.Pos
| AF of Pos.Pos
| EF of Pos.Pos
| AX of Pos.Pos
| EX of Pos.Pos
| A of Pos.Pos
| E of Pos.Pos
| U of Pos.Pos
| R of Pos.Pos
| G of Pos.Pos
| F of Pos.Pos
| MU of Pos.Pos
| NU of Pos.Pos
| OF of Pos.Pos
| UNIT of Pos.Pos
| BOOL of Pos.Pos
| INT of Pos.Pos
| NAT of Pos.Pos
| NEXT of Pos.Pos
| INIT of Pos.Pos
| INCLUDE of Pos.Pos
| TYPE of Pos.Pos
| DATATYPE of Pos.Pos
| SYMMETRIC of Pos.Pos
| FINITE of Pos.Pos
| STATEVAR of Pos.Pos
| EXPORT of Pos.Pos
| ALL of Pos.Pos
| OPEN of Pos.Pos
| UNDEFINED of Pos.Pos
| ANYVALUE of Pos.Pos
| SELF of Pos.Pos
| IF of Pos.Pos
| THEN of Pos.Pos
| ELSIF of Pos.Pos
| ELSE of Pos.Pos
| ENDIF of Pos.Pos
| MIN of Pos.Pos
| MAX of Pos.Pos
| MOD of Pos.Pos
| DIV of Pos.Pos
| CHOOSE of Pos.Pos
| LABEL of Pos.Pos
| PICK of Pos.Pos
| ENDCHOOSE of Pos.Pos
| NOP of Pos.Pos
| FALSE of Pos.Pos
| TRUE of Pos.Pos
| WITH of Pos.Pos
| SYNC of Pos.Pos
| ASYNC of Pos.Pos
| LP of Pos.Pos
| RP of Pos.Pos
| LB of Pos.Pos
| RB of Pos.Pos
| LCB of Pos.Pos
| RCB of Pos.Pos
| UNDERSCORE of Pos.Pos
| EQDEF of Pos.Pos
| PLUS of Pos.Pos
| MINUS of Pos.Pos
| TILDE of Pos.Pos
| TIMES of Pos.Pos
| DIVIDE of Pos.Pos
| EQ of Pos.Pos
| NOTEQ of Pos.Pos
| LE of Pos.Pos
| GE of Pos.Pos
| LT of Pos.Pos
| GT of Pos.Pos
| THREEDOTS of Pos.Pos
| TWODOTS of Pos.Pos
| DOT of Pos.Pos
| ARROW of Pos.Pos
| DARROW of Pos.Pos
| BAR of Pos.Pos
| DBAR of Pos.Pos
| COMMA of Pos.Pos
| COLON of Pos.Pos
| SEMI of Pos.Pos
| MODELS of Pos.Pos
| ID of Pos.Pos * string
| TVAR of Pos.Pos * string
| NUMBER of Pos.Pos * int
| QUOTE of Pos.Pos * string
| EOF of Pos.Pos
| UMINUS of Pos.Pos
| TARROW of Pos.Pos
| TTIMES of Pos.Pos
| TBAR of Pos.Pos
| JUXT

 (* %prefer SEMI *)

%left SEMI
%nonassoc SYMMETRIC FINITE
%nonassoc REFINES SIMULATES MODELS
%left COMMA
%nonassoc LET IN VAR VAL FUN FN THEOREM FAIRNESS CASE ENDCASE INCLUDE TYPE
      DATATYPE STATEVAR EXPORT OPEN IF THEN ELSE ENDIF CHOOSE LABEL ENDCHOOSE
      FOREACH ENDFOREACH EOF
%left BAR DBAR
%nonassoc EQDEF
%nonassoc FORALL EXISTS SYNC ASYNC PICK
%left U R
%nonassoc AG EG AF EF AX EX A E F G
%nonassoc IFF
%right  IMPLIES ARROW DARROW
%left  OR
%left  AND
%nonassoc EQ NOTEQ 
%nonassoc NOT
%nonassoc LE GE LT GT
%left JUXT
%nonassoc COLON
%left ELSIF
%nonassoc WITH
%nonassoc ARRAY
%nonassoc  OF
%nonassoc TWODOTS
%left  MOD
%left  PLUS MINUS
%left  TIMES DIV DIVIDE
%nonassoc  TILDE		(* supplies precedence for unary minus *)
%nonassoc  NEXT INIT
%left TBAR  (* precedences for |, -> and * in types *)
%right TARROW
%left TTIMES
%left  DOT
%left  LP RP LB RB LCB RCB
%nonassoc AS UNIT BOOL INT NAT ALL UNDEFINED ANYVALUE SELF NOP FALSE TRUE
     UNDERSCORE THREEDOTS ID NUMBER QUOTE
%nonassoc OP

(* all nonterminals *)

%nonterm begin of ParseTree
| semi
| declarations of ParseTree list
| declaration of ParseTree
| localdecl of ParseTree
| localdecls of ParseTree list
| module of ParseTree
| statparam of ParseTree list option
| dynparam of ParseTree option
| statformals of ParseTree list
| statblks of ParseTree list
| statblock of ParseTree
| moduleexpr of ParseTree
| typedecl of ParseTree
| tvars of ParseTree list
| vardecl of ParseTree
| valdecl of ParseTree
| fundecl of ParseTree
| fundeclcases of ParseTree list
| fundeclcase of ParseTree
| patterncommalist of ParseTree list
| patternlist of ParseTree list
| recordpattern of ParseTree list
| pattern of ParseTree
| pattern_ of ParseTree
| opendecl of ParseTree
| theorem of ParseTree
| theoremexpr of ParseTree
| statevardecl of ParseTree
| export of ParseTree
| asstexprs of ParseTree list
| asstexpr of ParseTree
| asstexpr0 of ParseTree
| singleasst of ParseTree
| ifasst of ParseTree
| elsifassts of ParseTree list
| elsifasst of ParseTree
| letasst of ParseTree
| letlocals of ParseTree list
| letlocal of ParseTree
| caseasst of ParseTree
| caseclauseassts of ParseTree list
| caseclauseasst of ParseTree
| chooseasst of ParseTree
| foreachasst of ParseTree
| chooseclauseassts of ParseTree list
| chooseclauseasst of ParseTree
| statinstexprs of ParseTree list
| statinstexpr of ParseTree
| dyninst of ParseTree list
| xtypeexpr of ParseTree
| typeexpr of ParseTree
| ftype of ParseTree
| atype of ParseTree
| typelist of ParseTree list
| arraytype of ParseTree
| recordtype of ParseTree
| tupletype of ParseTree
| tupletypelist of ParseTree list
| formalvars of ParseTree list
| fvarblock of ParseTree
| typedvars of ParseTree
| tpvarslist of ParseTree list
| tpvarslistel of ParseTree
| typedvarslist of ParseTree list
| neexprlist of ParseTree list
| expr of ParseTree
| cexpr of ParseTree
| texpr of ParseTree
| nexpr of ParseTree
| statevarexpr of ParseTree
| fnexpr of ParseTree
| recordexpr of ParseTree
| recordexprassts of ParseTree list
| recordexprasst of ParseTree
| ifexpr of ParseTree
| elsifexprs of ParseTree list
| elsifexpr of ParseTree
| letexpr of ParseTree
| caseexpr of ParseTree
| caseclauseexprs of ParseTree list
| caseclauseexpr of ParseTree
| withexpr of ParseTree
| withassts of ParseTree list
| withasst of ParseTree

%start begin

%%

begin         : declarations (Program(pos(List.hd(declarations)),declarations))
semi          : SEMI () | ()
declarations  : declaration  semi ([declaration])
              | declaration semi declarations (declaration::declarations)
declaration   : module (module)
              | typedecl (typedecl)
	      | vardecl (vardecl)
	      | valdecl (valdecl)
              | fundecl (fundecl)
              | opendecl (opendecl)
              | theorem (theorem)
              | INCLUDE QUOTE (Include(QUOTE))
              | INCLUDE LP QUOTE RP (Include(QUOTE))
localdecl     : declaration (declaration)
	      | statevardecl (statevardecl)
              | export (export)
              | asstexpr (asstexpr)
localdecls    : localdecl semi ([localdecl])
              | localdecl semi localdecls (localdecl::localdecls)
module        : MODULE ID statparam dynparam EQ moduleexpr
                (Module(MODULE,
                        { name = Id(ID),
			  statparams = statparam,
			  dynparams = dynparam,
			  body = moduleexpr }))
statparam     : (NONE)
              | LB statformals RB (SOME(statformals))
dynparam      : (NONE)
              | pattern_ (SOME(pattern_))
statblock     : ID %prec SEMI (TypedVars(pos(Id(ID)),[Id ID],NONE))
              | ID COLON xtypeexpr (TypedVars(COLON,[Id ID],SOME xtypeexpr))
              | LP typedvars RP (Group(pos(typedvars),typedvars))
              | TYPE ID (PType(TYPE,Id(ID)))
              | TYPE SYMMETRIC ID (PType(TYPE,Symmetric(TYPE,Id(ID))))
              | TYPE FINITE ID (PType(TYPE,Finite(TYPE,Id(ID))))
statblks      : statblock %prec SEMI ([statblock])
              | statblks COMMA statblks (statblks1@statblks2)
statformals   : statblks %prec SEMI (sorttypes(statblks))
moduleexpr    : BEGIN localdecls END (BeginEnd(BEGIN,localdecls))
              | ID texpr (Appl(pos(Id(ID)),Id(ID),texpr))
              | ID %prec SEMI (Id(ID))
              | SELF %prec SEMI (Self(SELF))
              | ID LB statinstexprs RB texpr
                 (Appl(pos(Id(ID)),(ModuleInst(pos(Id(ID)),
                                      Id(ID),statinstexprs)),texpr))
              | ID LB statinstexprs RB  %prec SEMI
 	          (ModuleInst(pos(Id(ID)),Id(ID),statinstexprs))
              | LP moduleexpr RP  %prec SEMI (moduleexpr)
              | moduleexpr DOT nexpr (Dot(DOT,moduleexpr,nexpr))
              | moduleexpr BAR moduleexpr (Async2(BAR,moduleexpr1,moduleexpr2))
              | moduleexpr DBAR moduleexpr
                   (Sync2(DBAR,moduleexpr1,moduleexpr2))
              | ASYNC formalvars COLON moduleexpr
                   (Async(ASYNC,formalvars,moduleexpr))
              | SYNC formalvars COLON moduleexpr
                   (Sync(SYNC,formalvars,moduleexpr))
typedecl      : TYPE ID EQ xtypeexpr (TypeDecl(TYPE,Id(ID),[],xtypeexpr))
              | TYPE TVAR ID EQ xtypeexpr
                  (TypeDecl(TYPE,Id(ID),[Tvar(TVAR)],xtypeexpr))
              | TYPE LP tvars RP ID EQ xtypeexpr
                  (TypeDecl(TYPE,Id(ID),tvars,xtypeexpr))
              | DATATYPE ID EQ xtypeexpr
                  (DatatypeDecl(DATATYPE,Id(ID),[],xtypeexpr))
              | DATATYPE TVAR ID EQ xtypeexpr
                  (DatatypeDecl(DATATYPE,Id(ID),[Tvar(TVAR)],xtypeexpr))
              | DATATYPE LP tvars RP ID EQ xtypeexpr
                  (DatatypeDecl(DATATYPE,Id(ID),tvars,xtypeexpr))
tvars         : TVAR ([Tvar(TVAR)])
              | TVAR COMMA tvars (Tvar(TVAR)::tvars)
vardecl       : VAR typedvarslist (Var(VAR,typedvarslist))
valdecl       : VAL pattern EQ expr (Val(VAL,pattern,expr))
fundecl       : FUN fundeclcases (FunCases(FUN,fundeclcases))
fundeclcases  : fundeclcase ([fundeclcase])
              | fundeclcase BAR fundeclcases %prec SEMI
	         (fundeclcase::fundeclcases)
fundeclcase   : ID patternlist COLON xtypeexpr EQ expr
                  (FunCase(pos(Id(ID)),Id(ID),patternlist,
                           SOME(xtypeexpr),expr))
              | ID patternlist EQ expr
                  (FunCase(pos(Id(ID)),Id(ID),patternlist,NONE,expr))
opendecl      : OPEN moduleexpr (Open(OPEN,moduleexpr))
theorem       : THEOREM ID EQ theoremexpr (Theorem(THEOREM,Id(ID),theoremexpr))
theoremexpr   : moduleexpr MODELS expr (Models(MODELS,moduleexpr,expr))
              | moduleexpr REFINES moduleexpr
                  (Refines(REFINES,moduleexpr1,moduleexpr2))
              | moduleexpr SIMULATES moduleexpr
                  (Refines(SIMULATES,moduleexpr2,moduleexpr1))
statevardecl  : STATEVAR typedvarslist (StateVarDecl(STATEVAR,typedvarslist))
export        : EXPORT typedvarslist (Export(EXPORT,typedvarslist))
              | EXPORT ALL (Export(EXPORT,[]))
asstexprs     : asstexpr semi ([asstexpr])
              | asstexpr semi asstexprs (asstexpr::asstexprs)
asstexpr      : LABEL ID asstexpr0 (LabeledAsst(LABEL, Id(ID), asstexpr0))
              | asstexpr0 (asstexpr0)
asstexpr0     : NOP (Nop(NOP))
              | singleasst (singleasst)
              | ifasst (ifasst)
              | letasst (letasst)
              | caseasst (caseasst)
              | chooseasst (chooseasst)
              | foreachasst (foreachasst)
singleasst    : statevarexpr EQDEF expr (NormalAsst(EQDEF,statevarexpr,expr))
              | INIT LP statevarexpr RP EQDEF expr
                  (InitAsst(INIT,statevarexpr,expr))
              | NEXT LP statevarexpr RP EQDEF expr
                  (NextAsst(NEXT,statevarexpr,expr))
ifasst        : IF expr THEN asstexprs elsifassts ELSE asstexprs ENDIF
                 (IfAsst(IF,(CondAsst(IF,expr,asstexprs1))::elsifassts,
                         asstexprs2))
elsifassts    : ([])
              | elsifasst elsifassts (elsifasst::elsifassts)
elsifasst     : ELSIF expr THEN asstexprs (CondAsst(ELSIF,expr,asstexprs))
letasst       : LET letlocals IN asstexprs END
                  (LetAsst(LET,letlocals,asstexprs))
letlocals     : letlocal ([letlocal])
              | letlocal letlocals (letlocal::letlocals)
letlocal      : vardecl (vardecl)
              | valdecl (valdecl)
              | fundecl (fundecl)
              | typedecl (typedecl)
caseasst      : CASE expr OF caseclauseassts ENDCASE
                  (CaseAsst(CASE,expr,caseclauseassts))
caseclauseassts: caseclauseasst ([caseclauseasst])
              | caseclauseasst BAR caseclauseassts  %prec SEMI
                   (caseclauseasst::caseclauseassts)
caseclauseasst: pattern DARROW asstexprs %prec SEMI
                  (ChoiceAsst(DARROW,pattern,asstexprs))
chooseasst    : CHOOSE formalvars COLON chooseclauseassts ENDCHOOSE
                  (ChooseAsst(CHOOSE,SOME formalvars,chooseclauseassts))
              | CHOOSE chooseclauseassts ENDCHOOSE
                  (ChooseAsst(CHOOSE,NONE,chooseclauseassts))
chooseclauseassts: chooseclauseasst ([chooseclauseasst])
              | chooseclauseasst BAR chooseclauseassts %prec SEMI
                 (chooseclauseasst::chooseclauseassts)
chooseclauseasst: cexpr DARROW asstexprs %prec SEMI
                  (ChoiceAsst(DARROW,cexpr, asstexprs))
foreachasst    : FOREACH formalvars COLON asstexprs ENDFOREACH %prec CHOOSE
                  (ForeachAsst(FOREACH, formalvars, asstexprs))
statinstexprs : statinstexpr ([statinstexpr])
              | statinstexpr COMMA statinstexprs (statinstexpr::statinstexprs)
statinstexpr  : expr (expr)
              | TYPE xtypeexpr (Type(TYPE,xtypeexpr))
              | DATATYPE xtypeexpr (Datatype(DATATYPE,xtypeexpr))
              | BOOL (BoolType(BOOL))
              | INT (IntType(INT))
              | NAT (NatType(NAT))
xtypeexpr     : typeexpr %prec FINITE (typeexpr)
              | SYMMETRIC typeexpr (Symmetric(SYMMETRIC,typeexpr))
              | FINITE typeexpr (Finite(FINITE,typeexpr))
typeexpr      : atype (atype)
              | LP typelist RP ftype (TypeInst(pos(ftype),typelist,ftype))
              | typeexpr ftype (TypeInst(pos(ftype),[typeexpr],ftype))
              | typeexpr BAR typeexpr
                (EnumType(pos(typeexpr1),
                          (enum2list typeexpr1)@(enum2list typeexpr2)))
              | typeexpr ARROW typeexpr
                  (FunType(ARROW,typeexpr1,typeexpr2))
              | typeexpr TIMES typeexpr
		(TupleType(pos(typeexpr1),
			   (tuple2list typeexpr1)@(tuple2list typeexpr2)))
atype         : ID OF typeexpr %prec EQDEF (Of(pos(Id(ID)),Id(ID),typeexpr))
              | ID %prec WITH (Id(ID))
              | ID DOT ID (Dot(DOT,Id(ID1),Id(ID2)))
              | ID LB statinstexprs RB DOT ID %prec DOT
 	          (Dot(DOT,ModuleInst(pos(Id(ID1)),
                           Id(ID1),statinstexprs),Id(ID2)))
              | TVAR %prec WITH (Tvar(TVAR))
              | BOOL (BoolType(BOOL))
              | NAT (NatType(NAT))
              | INT (IntType(INT))
              | LB expr TWODOTS expr RB (RangeType(TWODOTS,expr1,expr2))
              | recordtype (recordtype)
              | LP xtypeexpr RP (Group(pos(xtypeexpr),ungroup(xtypeexpr)))
              | arraytype (arraytype)
typelist      : xtypeexpr COMMA xtypeexpr ([xtypeexpr1,xtypeexpr2])
              | xtypeexpr COMMA typelist (xtypeexpr::typelist)
ftype         : ID %prec WITH (Id(ID))
              | ID DOT ID (Dot(DOT,Id(ID1),Id(ID2)))
              | ID LB statinstexprs RB DOT ID %prec DOT
 	          (Dot(DOT,ModuleInst(pos(Id(ID1)),
                           Id(ID1),statinstexprs),Id(ID2)))
              | LP xtypeexpr RP (Group(pos(xtypeexpr),ungroup(xtypeexpr)))
arraytype     : ARRAY typeexpr OF typeexpr 
                 (ArrayType(ARRAY,typeexpr1,typeexpr2))
tpvarslistel  : ID %prec SEMI (Id ID)
              | ID COLON xtypeexpr (TypedVars(COLON,[Id ID],SOME xtypeexpr))
              | LP typedvars RP (Group(pos(typedvars),typedvars))
tpvarslist    : tpvarslistel %prec SEMI ([tpvarslistel])
              | tpvarslist COMMA tpvarslist (tpvarslist1@tpvarslist2)
typedvarslist : tpvarslist %prec SEMI (sorttypes(tpvarslist))
recordtype    : LCB typedvarslist RCB (RecordType(LCB,typedvarslist))
formalvars    : fvarblock %prec COMMA ([fvarblock])
              | fvarblock COMMA formalvars (fvarblock::formalvars)
fvarblock     : ID (Id(ID))
              | LP typedvars RP (typedvars)
typedvars     : ID COMMA typedvars
                 (TypedVars(pos(Id(ID)),(Id ID)::(tvlist typedvars),
                            tvtype typedvars))
              | ID COLON xtypeexpr (TypedVars(COLON,[Id ID],SOME xtypeexpr))
expr          : expr BAR expr
                (NondetExpr(BAR,(nondet2list expr1)@(nondet2list expr2)))
              | expr texpr %prec JUXT (Appl(pos(expr),expr,texpr))
              | expr COLON xtypeexpr (TypedExpr(COLON,expr,xtypeexpr))
              | expr DOT nexpr (Dot(DOT,expr,nexpr))
              | expr WITH LB withassts RB (WithExpr(WITH,expr,withassts))
              | TILDE expr (Appl(TILDE,UMinus(TILDE),expr))
              | MINUS expr %prec TILDE (Appl(MINUS,UMinus(MINUS),expr))
              | NOT expr (Appl(NOT,Not(NOT),expr))
              | expr EQ expr (Appl(EQ,Eq(EQ),TupleExpr(EQ,[expr1,expr2])))
              | expr NOTEQ expr (Appl(NOTEQ,NotEq(NOTEQ),
                                 TupleExpr(NOTEQ,[expr1,expr2])))
              | expr IFF expr (Appl(IFF,Iff(IFF),TupleExpr(IFF,[expr1,expr2])))
              | expr IMPLIES expr (Appl(IMPLIES,Implies(IMPLIES),
                                       TupleExpr(IMPLIES,[expr1,expr2])))
              | expr ARROW expr (Appl(ARROW,Arrow(ARROW),
                                      TupleExpr(ARROW,[expr1,expr2])))
              | expr DARROW expr (Appl(DARROW,Darrow(DARROW),
                                       TupleExpr(DARROW,[expr1,expr2])))
              | expr AND expr (Appl(AND,And(AND),TupleExpr(AND,[expr1,expr2])))
              | expr OR expr (Appl(OR,Or(OR),TupleExpr(OR,[expr1,expr2])))
              | expr LT expr (Appl(LT,Lt(LT),TupleExpr(LT,[expr1,expr2])))
              | expr GT expr (Appl(GT,Gt(GT),TupleExpr(GT,[expr1,expr2])))
              | expr LE expr (Appl(LE,Le(LE),TupleExpr(LE,[expr1,expr2])))
              | expr GE expr (Appl(GE,Ge(GE),TupleExpr(GE,[expr1,expr2])))
              | expr PLUS expr (Appl(PLUS,Plus(PLUS),
                                     TupleExpr(PLUS,[expr1,expr2])))
              | expr MINUS expr (Appl(MINUS,Minus(MINUS),
                                      TupleExpr(MINUS,[expr1,expr2])))
              | expr TIMES expr (Appl(TIMES,Times(TIMES),
                                      TupleExpr(TIMES,[expr1,expr2])))
              | expr DIV expr (Appl(DIV,Div(DIV),TupleExpr(DIV,[expr1,expr2])))
              | expr MOD expr (Appl(MOD,Mod(MOD),TupleExpr(MOD,[expr1,expr2])))
              | FORALL formalvars COLON expr %prec FORALL
                 (Forall(FORALL,formalvars,expr))
              | EXISTS formalvars COLON expr %prec EXISTS
                 (Exists(EXISTS,formalvars,expr))
              | PICK pattern_ COLON expr %prec FORALL
                 (ChooseExpr(PICK,pattern_,expr))
              | AG expr (Appl(AG,Ag(AG),expr))
              | AF expr (Appl(AF,Af(AF),expr))
              | EG expr (Appl(EG,Eg(EG),expr))
              | EF expr (Appl(EF,Ef(EF),expr))
              | AX expr (Appl(AX,Ax(AX),expr))
              | EX expr (Appl(EX,Ex(EX),expr))
              | A LB expr U expr RB
                  (Appl(A,Au(A),TupleExpr(A,[expr1,expr2])))
              | E LB expr U expr RB
                  (Appl(E,Eu(E),TupleExpr(E,[expr1,expr2])))
              | A LB expr R expr RB
                  (Appl(A,Ar(A),TupleExpr(A,[expr1,expr2])))
              | E LB expr R expr RB
                  (Appl(E,Er(E),TupleExpr(E,[expr1,expr2])))
              | expr U expr (Appl(U, Until(U), TupleExpr(U,[expr1,expr2])))
              | expr R expr (Appl(R, Releases(R), TupleExpr(R,[expr1,expr2])))
              | G expr (Appl(G, Globally(G), expr))
              | F expr (Appl(F, Eventually(F), expr))
              | NEXT expr (Appl(NEXT, NextTime(NEXT), expr))
              | MU ID COLON expr (Mu(MU, Id(ID), expr))
              | NU ID COLON expr (Nu(NU, Id(ID), expr))
              | FALSE (False(FALSE))
              | TRUE (True(TRUE))
              | UNDEFINED (Undefined(UNDEFINED))
              | ANYVALUE (Anyvalue(ANYVALUE))
              | NUMBER (Number(NUMBER))
              | recordexpr (recordexpr)
              | ifexpr (ifexpr)
              | caseexpr (caseexpr)
              | letexpr (letexpr)
              | fnexpr (fnexpr)
              | LP neexprlist RP
                (case neexprlist of
		     [] => raise SympParseInternalError "neexprlist is empty"
                   | [x] => x
                   | _::_ => TupleExpr(LP,neexprlist))
              | ID LB statinstexprs RB
                  (ModuleInst(pos(Id(ID)),Id(ID),statinstexprs))
              | ID (Id(ID))
              | OP AND (And(AND))
              | OP OR (Or(OR))
              | OP IMPLIES (Implies(OP))
              | OP ARROW (Arrow(ARROW))
              | OP DARROW (Darrow(DARROW))
              | OP IFF (Iff(IFF))
              | OP EQ (Eq(EQ))
              | OP NOTEQ (NotEq(NOTEQ))
              | OP NOT (Not(NOT))
              | OP AG (Ag(AG))
              | OP AF (Af(AF))
              | OP EG (Eg(EG))
              | OP EF (Ef(EF))
              | OP AX (Ax(AX))
              | OP EX (Ex(EX))
              | OP LT (Lt(LT))
              | OP GT (Gt(GT))
              | OP LE (Le(LE))
              | OP GE (Ge(GE))
              | OP PLUS (Plus(PLUS))
              | OP MINUS (Minus(MINUS))
              | OP TIMES (Times(TIMES))
              | OP DIV (Div(DIV))
              | OP MOD (Mod(MOD))
texpr         : texpr DOT nexpr (Dot(DOT,texpr,nexpr))
              | TILDE texpr (Appl(TILDE,UMinus(TILDE),texpr))
              | NOT texpr (Appl(NOT,Not(NOT),texpr))
              | FALSE (False(FALSE))
              | TRUE (True(TRUE))
              | UNDEFINED (Undefined(UNDEFINED))
              | ANYVALUE (Anyvalue(ANYVALUE))
              | NUMBER (Number(NUMBER))
              | recordexpr (recordexpr)
              | ifexpr (ifexpr)
              | caseexpr (caseexpr)
              | letexpr (letexpr)
              | fnexpr (fnexpr)
              | LP neexprlist RP
                (case neexprlist of
		     [] => raise SympParseInternalError "neexprlist is empty"
                   | [x] => x
                   | _::_ => TupleExpr(LP,neexprlist))
              | ID LB statinstexprs RB
                  (ModuleInst(pos(Id(ID)),Id(ID),statinstexprs))
              | ID (Id(ID))
              | OP AND (And(AND))
              | OP OR (Or(OR))
              | OP IMPLIES (Implies(OP))
              | OP ARROW (Arrow(ARROW))
              | OP DARROW (Darrow(DARROW))
              | OP IFF (Iff(IFF))
              | OP EQ (Eq(EQ))
              | OP NOTEQ (NotEq(NOTEQ))
              | OP NOT (Not(NOT))
              | OP AG (Ag(AG))
              | OP AF (Af(AF))
              | OP EG (Eg(EG))
              | OP EF (Ef(EF))
              | OP AX (Ax(AX))
              | OP EX (Ex(EX))
              | OP LT (Lt(LT))
              | OP GT (Gt(GT))
              | OP LE (Le(LE))
              | OP GE (Ge(GE))
              | OP PLUS (Plus(PLUS))
              | OP MINUS (Minus(MINUS))
              | OP TIMES (Times(TIMES))
              | OP DIV (Div(DIV))
              | OP MOD (Mod(MOD))
nexpr         : ID (Id(ID))
              | OP AND (And(AND))
              | OP OR (Or(OR))
              | OP IMPLIES (Implies(OP))
              | OP ARROW (Arrow(ARROW))
              | OP DARROW (Darrow(DARROW))
              | OP IFF (Iff(IFF))
              | OP EQ (Eq(EQ))
              | OP NOTEQ (NotEq(NOTEQ))
              | OP NOT (Not(NOT))
              | OP AG (Ag(AG))
              | OP AF (Af(AF))
              | OP EG (Eg(EG))
              | OP EF (Ef(EF))
              | OP AX (Ax(AX))
              | OP EX (Ex(EX))
              | OP LT (Lt(LT))
              | OP GT (Gt(GT))
              | OP LE (Le(LE))
              | OP GE (Ge(GE))
              | OP PLUS (Plus(PLUS))
              | OP MINUS (Minus(MINUS))
              | OP TIMES (Times(TIMES))
              | OP DIV (Div(DIV))
              | OP MOD (Mod(MOD))
cexpr          : cexpr BAR cexpr
                (NondetExpr(BAR,(nondet2list cexpr1)@(nondet2list cexpr2)))
              | cexpr texpr %prec JUXT (Appl(pos(cexpr),cexpr,texpr))
              | cexpr DOT nexpr (Dot(DOT,cexpr,nexpr))
              | cexpr WITH LB withassts RB (WithExpr(WITH,cexpr,withassts))
              | TILDE cexpr (Appl(TILDE,UMinus(TILDE),cexpr))
              | NOT cexpr (Appl(NOT,Not(NOT),cexpr))
              | cexpr EQ cexpr (Appl(EQ,Eq(EQ),TupleExpr(EQ,[cexpr1,cexpr2])))
              | cexpr NOTEQ cexpr (Appl(NOTEQ,NotEq(NOTEQ),
                                 TupleExpr(NOTEQ,[cexpr1,cexpr2])))
              | cexpr IFF cexpr (Appl(IFF,Iff(IFF),
                                      TupleExpr(IFF,[cexpr1,cexpr2])))
              | cexpr IMPLIES cexpr (Appl(IMPLIES,Implies(IMPLIES),
                                       TupleExpr(IMPLIES,[cexpr1,cexpr2])))
              | cexpr ARROW cexpr (Appl(ARROW,Arrow(ARROW),
                                      TupleExpr(ARROW,[cexpr1,cexpr2])))
              | cexpr AND cexpr (Appl(AND,And(AND),
                                      TupleExpr(AND,[cexpr1,cexpr2])))
              | cexpr OR cexpr (Appl(OR,Or(OR),TupleExpr(OR,[cexpr1,cexpr2])))
              | cexpr LT cexpr (Appl(LT,Lt(LT),TupleExpr(LT,[cexpr1,cexpr2])))
              | cexpr GT cexpr (Appl(GT,Gt(GT),TupleExpr(GT,[cexpr1,cexpr2])))
              | cexpr LE cexpr (Appl(LE,Le(LE),TupleExpr(LE,[cexpr1,cexpr2])))
              | cexpr GE cexpr (Appl(GE,Ge(GE),TupleExpr(GE,[cexpr1,cexpr2])))
              | cexpr PLUS cexpr (Appl(PLUS,Plus(PLUS),
                                     TupleExpr(PLUS,[cexpr1,cexpr2])))
              | cexpr MINUS cexpr (Appl(MINUS,Minus(MINUS),
                                      TupleExpr(MINUS,[cexpr1,cexpr2])))
              | cexpr TIMES cexpr (Appl(TIMES,Times(TIMES),
                                      TupleExpr(TIMES,[cexpr1,cexpr2])))
              | cexpr DIV cexpr (Appl(DIV,Div(DIV),
                                      TupleExpr(DIV,[cexpr1,cexpr2])))
              | cexpr MOD cexpr (Appl(MOD,Mod(MOD),
                                      TupleExpr(MOD,[cexpr1,cexpr2])))
              | FORALL formalvars COLON cexpr %prec FORALL
                 (Forall(FORALL,formalvars,cexpr))
              | EXISTS formalvars COLON cexpr %prec FORALL
                 (Exists(EXISTS,formalvars,cexpr))
              | PICK pattern_ COLON cexpr %prec FORALL
                 (ChooseExpr(PICK,pattern_,cexpr))
              | AG cexpr (Appl(AG,Ag(AG),cexpr))
              | AF cexpr (Appl(AF,Af(AF),cexpr))
              | EG cexpr (Appl(EG,Eg(EG),cexpr))
              | EF cexpr (Appl(EF,Ef(EF),cexpr))
              | AX cexpr (Appl(AX,Ax(AX),cexpr))
              | EX cexpr (Appl(EX,Ex(EX),cexpr))
              | A LB cexpr U cexpr RB
                  (Appl(A,Au(A),TupleExpr(A,[cexpr1,cexpr2])))
              | E LB cexpr U cexpr RB
                  (Appl(E,Eu(E),TupleExpr(E,[cexpr1,cexpr2])))
              | A LB cexpr R cexpr RB
                  (Appl(A,Ar(A),TupleExpr(A,[cexpr1,cexpr2])))
              | E LB cexpr R cexpr RB
                  (Appl(E,Er(E),TupleExpr(E,[cexpr1,cexpr2])))
              | cexpr U cexpr (Appl(U, Until(U), TupleExpr(U,[cexpr1,cexpr2])))
              | cexpr R cexpr (Appl(R, Releases(R), TupleExpr(R,[cexpr1,cexpr2])))
              | G cexpr (Appl(G, Globally(G), cexpr))
              | F cexpr (Appl(F, Eventually(F), cexpr))
              | NEXT cexpr (Appl(NEXT, NextTime(NEXT), cexpr))
              | MU ID COLON cexpr (Mu(MU, Id(ID), cexpr))
              | NU ID COLON cexpr (Nu(NU, Id(ID), cexpr))
              | FALSE (False(FALSE))
              | TRUE (True(TRUE))
              | UNDEFINED (Undefined(UNDEFINED))
              | ANYVALUE (Anyvalue(ANYVALUE))
              | NUMBER (Number(NUMBER))
              | recordexpr (recordexpr)
              | ifexpr (ifexpr)
              | caseexpr (caseexpr)
              | letexpr (letexpr)
              | fnexpr (fnexpr)
              | LP neexprlist RP
                (case neexprlist of
		     [] => raise SympParseInternalError "neexprlist is empty"
                   | [x] => x
                   | _::_ => TupleExpr(LP,neexprlist))
              | ID LB statinstexprs RB
                  (ModuleInst(pos(Id(ID)),Id(ID),statinstexprs))
              | ID (Id(ID))
              | OP AND (And(AND))
              | OP OR (Or(OR))
              | OP IMPLIES (Implies(OP))
              | OP ARROW (Arrow(ARROW))
              | OP DARROW (Darrow(DARROW))
              | OP IFF (Iff(IFF))
              | OP EQ (Eq(EQ))
              | OP NOTEQ (NotEq(NOTEQ))
              | OP NOT (Not(NOT))
              | OP AG (Ag(AG))
              | OP AF (Af(AF))
              | OP EG (Eg(EG))
              | OP EF (Ef(EF))
              | OP AX (Ax(AX))
              | OP EX (Ex(EX))
              | OP LT (Lt(LT))
              | OP GT (Gt(GT))
              | OP LE (Le(LE))
              | OP GE (Ge(GE))
              | OP PLUS (Plus(PLUS))
              | OP MINUS (Minus(MINUS))
              | OP TIMES (Times(TIMES))
              | OP DIV (Div(DIV))
              | OP MOD (Mod(MOD))
neexprlist    : expr ([expr])
              | expr COMMA neexprlist (expr::neexprlist)
statevarexpr  : nexpr %prec COMMA (nexpr)
              | statevarexpr texpr %prec JUXT
                 (Appl(pos(statevarexpr),statevarexpr,texpr))
              | statevarexpr DOT nexpr (Dot(DOT,statevarexpr,nexpr))
              | LP statevarexpr RP (statevarexpr)
fnexpr        : FN caseclauseexprs (Fn(FN,caseclauseexprs))
recordexpr    : LCB recordexprassts RCB (RecordExpr(LCB,recordexprassts))
recordexprassts: recordexprasst ([recordexprasst])
              | recordexprasst COMMA recordexprassts
                  (recordexprasst::recordexprassts)
recordexprasst: ID EQ expr (RecordAsst(EQ,Id(ID),expr))
ifexpr        : IF expr THEN expr elsifexprs ELSE expr ENDIF
                 (IfExpr(IF,(CondExpr(IF,expr1,expr2))::elsifexprs, expr3))
elsifexprs    : %prec COMMA([])
              | elsifexpr elsifexprs (elsifexpr::elsifexprs)
elsifexpr     : ELSIF expr THEN expr (CondExpr(ELSIF,expr1,expr2))
letexpr       : LET letlocals IN expr END (LetExpr(LET,letlocals,expr))
caseexpr      : CASE expr OF caseclauseexprs ENDCASE
                  (CaseExpr(CASE,expr,caseclauseexprs))
caseclauseexprs: caseclauseexpr  %prec SEMI ([caseclauseexpr])
              | caseclauseexpr BAR caseclauseexprs  %prec SEMI
                   (caseclauseexpr::caseclauseexprs)
caseclauseexpr: pattern DARROW expr %prec FORALL
                (ChoiceExpr(DARROW,pattern,expr))
withassts     : withasst ([withasst])
              | withasst COMMA withassts (withasst::withassts)
withasst      : ID EQ expr (WithAsst(EQ,Id(ID),expr))
pattern       : pattern BAR pattern
                (ChoicePattern(BAR,(pattern2list pattern1)@(pattern2list pattern2)))
              | ID pattern (ApplPattern(pos(Id(ID)),Id(ID),pattern))
              | ID AS pattern (AsPattern(AS,Id(ID),pattern))
              | ID (Id(ID))
              | NUMBER (Number(NUMBER))
              | TRUE (True(TRUE))
              | FALSE (False(FALSE))
              | pattern COLON xtypeexpr %prec WITH
                  (TypedPattern(COLON,pattern,xtypeexpr))
              | LP patterncommalist RP
                 (case patterncommalist of
                      [] => raise SympParseInternalError
                                          "patterncommalist is empty"
                   | [x] => x
                   | _::_ => TuplePattern(LP,patterncommalist))
              | LCB recordpattern RCB (RecordPattern(LCB,recordpattern))
              | UNDERSCORE (Underscore(UNDERSCORE))
patternlist   : pattern_ ([pattern_])
              | pattern_ patternlist (pattern_::patternlist)
patterncommalist : pattern %prec BAR ([pattern])
              | pattern COMMA patterncommalist (pattern::patterncommalist)
pattern_      : ID (Id(ID))
              | NUMBER (Number(NUMBER))
              | TRUE (True(TRUE))
              | FALSE (False(FALSE))
              | LP patterncommalist RP
                 (case patterncommalist of
                      [] => raise SympParseInternalError
                                          "patterncommalist is empty"
                   | [x] => x
                   | _::_ => TuplePattern(LP,patterncommalist))
              | LCB recordpattern RCB (RecordPattern(LCB,recordpattern))
              | UNDERSCORE (Underscore(UNDERSCORE))
recordpattern : ID EQ pattern COMMA THREEDOTS
                 ([RecordAsst(EQ,Id(ID),pattern),Ellipsis(THREEDOTS)])
              | ID EQ pattern COMMA recordpattern
                 (RecordAsst(EQ,Id(ID),pattern)::recordpattern)
              | ID EQ pattern ([RecordAsst(EQ,Id(ID),pattern)])
