structure Tokens = Tokens
structure ParseError = ParseError
open ParseError

type pos = int
type svalue = Tokens.svalue
type ('a,'b) token = ('a,'b) Tokens.token
type lexresult= (svalue,pos) token
type arg = string * pos * pos -> unit

val pos = ref 1
val yypos_bol = ref 0  (* remember the val. of yypos at the
			  beginning of the current line *)

val eof = fn _ => Tokens.EOF((!pos,0),0,0)

(* This is passed as a parameter to `lex' now *)
(* val error = fn (e,l,c) => (wasError := true;
			   printOut options "parseerror" 
		                ("at "^(Int.toString l)^"."
				 ^(Int.toString c)^": "^e^"\n"))
 *)

fun incLine (x: int ref) =  (x := !x + 1)
fun ripQuotes str = String.extract(str,1,SOME(String.size(str)-2))

fun tvar(s,pos,line) = Tokens.TVAR(((pos,line),String.extract(s,1,NONE)),pos,line)

fun id2token(s,pos,line) =
    let fun keyword "module" = SOME(Tokens.MODULE((pos,line),pos,line))
	  | keyword "begin" =  SOME(Tokens.BEGIN((pos,line),pos,line))
	  | keyword "end" =  SOME(Tokens.END((pos,line),pos,line))
	  | keyword "let" =  SOME(Tokens.LET((pos,line),pos,line))
	  | keyword "in" =  SOME(Tokens.IN((pos,line),pos,line))
	  | keyword "var" =  SOME(Tokens.VAR((pos,line),pos,line))
	  | keyword "val" =  SOME(Tokens.VAL((pos,line),pos,line))
	  | keyword "fun" =  SOME(Tokens.FUN((pos,line),pos,line))
	  | keyword "fn" =  SOME(Tokens.FN((pos,line),pos,line))
          | keyword "op" = SOME(Tokens.OP((pos,line),pos,line))
          | keyword "as" = SOME(Tokens.AS((pos,line),pos,line))
	  | keyword "theorem" =  SOME(Tokens.THEOREM((pos,line),pos,line))
	  | keyword "lemma" =  SOME(Tokens.THEOREM((pos,line),pos,line))
	  | keyword "proposition" =  SOME(Tokens.THEOREM((pos,line),pos,line))
	  | keyword "spec" =  SOME(Tokens.THEOREM((pos,line),pos,line))
	  | keyword "specification" =  SOME(Tokens.THEOREM((pos,line),pos,line))
	  | keyword "corollary" =  SOME(Tokens.THEOREM((pos,line),pos,line))
	  | keyword "conjecture" =  SOME(Tokens.THEOREM((pos,line),pos,line))
	  | keyword "implements" =  SOME(Tokens.REFINES((pos,line),pos,line))
	  | keyword "refines" =  SOME(Tokens.REFINES((pos,line),pos,line))
	  | keyword "simulates" =  SOME(Tokens.SIMULATES((pos,line),pos,line))
	  | keyword "array" =  SOME(Tokens.ARRAY((pos,line),pos,line))
	  | keyword "fairness" =  SOME(Tokens.FAIRNESS((pos,line),pos,line))
	  | keyword "case" =  SOME(Tokens.CASE((pos,line),pos,line))
	  | keyword "endcase" =  SOME(Tokens.ENDCASE((pos,line),pos,line))
	  | keyword "and" =  SOME(Tokens.AND((pos,line),pos,line))
	  | keyword "or" =  SOME(Tokens.OR((pos,line),pos,line))
	  | keyword "implies" =  SOME(Tokens.IMPLIES((pos,line),pos,line))
	  | keyword "iff" =  SOME(Tokens.IFF((pos,line),pos,line))
	  | keyword "not" =  SOME(Tokens.NOT((pos,line),pos,line))
	  | keyword "forall" =  SOME(Tokens.FORALL((pos,line),pos,line))
	  | keyword "foreach" =  SOME(Tokens.FOREACH((pos,line),pos,line))
	  | keyword "endforeach" =  SOME(Tokens.ENDFOREACH((pos,line),pos,line))
	  | keyword "exists" =  SOME(Tokens.EXISTS((pos,line),pos,line))
	  | keyword "ag" =  SOME(Tokens.AG((pos,line),pos,line))
	  | keyword "eg" =  SOME(Tokens.EG((pos,line),pos,line))
	  | keyword "af" =  SOME(Tokens.AF((pos,line),pos,line))
	  | keyword "ef" =  SOME(Tokens.EF((pos,line),pos,line))
	  | keyword "ax" =  SOME(Tokens.AX((pos,line),pos,line))
	  | keyword "ex" =  SOME(Tokens.EX((pos,line),pos,line))
	  | keyword "until" =  SOME(Tokens.U((pos,line),pos,line))
	  | keyword "releases" =  SOME(Tokens.R((pos,line),pos,line))
	  | keyword "always" =  SOME(Tokens.A((pos,line),pos,line))
	  | keyword "sometimes" =  SOME(Tokens.E((pos,line),pos,line))
          | keyword "globally" = SOME(Tokens.G((pos,line),pos,line))
          | keyword "eventually" = SOME(Tokens.F((pos,line),pos,line))
          | keyword "mu" = SOME(Tokens.MU((pos,line),pos,line))
          | keyword "nu" = SOME(Tokens.NU((pos,line),pos,line))
	  | keyword "of" =  SOME(Tokens.OF((pos,line),pos,line))
	  | keyword "mod" =  SOME(Tokens.MOD((pos,line),pos,line))
	  | keyword "bool" =  SOME(Tokens.BOOL((pos,line),pos,line))
	  | keyword "int" =  SOME(Tokens.INT((pos,line),pos,line))
	  | keyword "nat" =  SOME(Tokens.NAT((pos,line),pos,line))
	  | keyword "next" =  SOME(Tokens.NEXT((pos,line),pos,line))
	  | keyword "init" =  SOME(Tokens.INIT((pos,line),pos,line))
	  | keyword "include" =  SOME(Tokens.INCLUDE((pos,line),pos,line))
	  | keyword "type" =  SOME(Tokens.TYPE((pos,line),pos,line))
	  | keyword "datatype" =  SOME(Tokens.DATATYPE((pos,line),pos,line))
	  | keyword "symmetric" =  SOME(Tokens.SYMMETRIC((pos,line),pos,line))
	  | keyword "finite" =  SOME(Tokens.FINITE((pos,line),pos,line))
	  | keyword "statevar" =  SOME(Tokens.STATEVAR((pos,line),pos,line))
	  | keyword "export" =  SOME(Tokens.EXPORT((pos,line),pos,line))
	  | keyword "all" =  SOME(Tokens.ALL((pos,line),pos,line))
	  | keyword "open" =  SOME(Tokens.OPEN((pos,line),pos,line))
	  | keyword "undefined" =  SOME(Tokens.UNDEFINED((pos,line),pos,line))
	  | keyword "anyvalue" =  SOME(Tokens.ANYVALUE((pos,line),pos,line))
	  | keyword "self" =  SOME(Tokens.SELF((pos,line),pos,line))
	  | keyword "if" =  SOME(Tokens.IF((pos,line),pos,line))
	  | keyword "then" =  SOME(Tokens.THEN((pos,line),pos,line))
	  | keyword "elsif" =  SOME(Tokens.ELSIF((pos,line),pos,line))
	  | keyword "else" =  SOME(Tokens.ELSE((pos,line),pos,line))
	  | keyword "endif" =  SOME(Tokens.ENDIF((pos,line),pos,line))
	  | keyword "min" =  SOME(Tokens.MIN((pos,line),pos,line))
	  | keyword "max" =  SOME(Tokens.MAX((pos,line),pos,line))
	  | keyword "div" =  SOME(Tokens.DIV((pos,line),pos,line))
	  | keyword "choose" =  SOME(Tokens.CHOOSE((pos,line),pos,line))
	  | keyword "pick" =  SOME(Tokens.PICK((pos,line),pos,line))
          | keyword "label" =  SOME(Tokens.LABEL((pos,line),pos,line))
	  | keyword "endchoose" =  SOME(Tokens.ENDCHOOSE((pos,line),pos,line))
	  | keyword "nop" =  SOME(Tokens.NOP((pos,line),pos,line))
	  | keyword "false" =  SOME(Tokens.FALSE((pos,line),pos,line))
	  | keyword "true" =  SOME(Tokens.TRUE((pos,line),pos,line))
	  | keyword "with" =  SOME(Tokens.WITH((pos,line),pos,line))
	  | keyword "sync" =  SOME(Tokens.SYNC((pos,line),pos,line))
	  | keyword "async" =  SOME(Tokens.ASYNC((pos,line),pos,line))
	  | keyword x = NONE
    in
	case keyword(String.map Char.toLower s) of
	    SOME x => x
	  | NONE => Tokens.ID(((pos,line),s),pos,line)
    end
%%

%arg (error);

%header (functor SYMPLexFun(structure Tokens: SYMP_TOKENS
			    structure ParseError: PARSE_ERROR));
%s C CC CCC;
alpha=[A-Za-z_];
idchar=[A-Za-z_0-9?\'];
digit=[0-9];
ws = [\ \t\013];
comment = "--".*\n | "#".*\n;

%%

<INITIAL>\n           => (yypos_bol := yypos; incLine pos; continue());
<INITIAL>{ws}+        => (continue());
<INITIAL>{comment}    => (yypos_bol := yypos; incLine pos; continue());
<INITIAL>"(*"         => (YYBEGIN C; continue());
<INITIAL>"*)"         => (error("`*)' without `(*' (you may have exceeded max. depth of comment nesting)",!pos,yypos - !yypos_bol); continue());
<INITIAL>"()"         => (Tokens.UNIT((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"("          => (Tokens.LP((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>")"          => (Tokens.RP((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"["          => (Tokens.LB((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"]"          => (Tokens.RB((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"{"          => (Tokens.LCB((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"}"          => (Tokens.RCB((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>":="         => (Tokens.EQDEF((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"+"          => (Tokens.PLUS((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"-"          => (Tokens.MINUS((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"~"          => (Tokens.TILDE((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"*"          => (Tokens.TIMES((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"/"          => (Tokens.DIVIDE((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"="          => (Tokens.EQ((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"!="         => (Tokens.NOTEQ((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"<>"         => (Tokens.NOTEQ((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"<="         => (Tokens.LE((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>">="         => (Tokens.GE((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"<"          => (Tokens.LT((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>">"          => (Tokens.GT((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"..."        => (Tokens.THREEDOTS((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>".."         => (Tokens.TWODOTS((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"."          => (Tokens.DOT((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"->"         => (Tokens.ARROW((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"=>"         => (Tokens.DARROW((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"<->"        => (Tokens.IFF((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"|"          => (Tokens.BAR((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"||"         => (Tokens.DBAR((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"&"          => (Tokens.AND((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"!"          => (Tokens.NOT((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>","          => (Tokens.COMMA((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>":"          => (Tokens.COLON((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>";"          => (Tokens.SEMI((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"|="         => (Tokens.MODELS((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));
<INITIAL>"_"          => (Tokens.UNDERSCORE((!pos,yypos - !yypos_bol),!pos,yypos - !yypos_bol));

<INITIAL>{digit}+     => (Tokens.NUMBER (((!pos,yypos - !yypos_bol),
					  List.foldl (fn(a,r)=>ord(a)-ord(#"0")+10*r)
					  0 (explode yytext)),
					 !pos,yypos - !yypos_bol));
<INITIAL>{alpha}{idchar}* => (id2token(yytext,!pos,yypos - !yypos_bol));
<INITIAL>"'"{alpha}{idchar}* => (tvar(yytext,!pos,yypos - !yypos_bol));
<INITIAL>\"[^\"]*\"   => (Tokens.QUOTE(((!pos,yypos - !yypos_bol),
					ripQuotes yytext),
				       !pos,yypos - !yypos_bol));

<INITIAL>.            => (error ("Illigal character: "^yytext,!pos,yypos - !yypos_bol); continue());

<C>"*)"               => (YYBEGIN INITIAL; continue());
<C>"(*"               => (YYBEGIN CC; continue());
<C>.                  => (continue());

<CC>"*)"              => (YYBEGIN C; continue());
<CC>"(*"              => (YYBEGIN CCC; continue());
<CC>.                 => (continue());

<CCC>"*)"             => (YYBEGIN CC; continue());
<CCC>"(*"             => (error("Exceeded max.  depth of comment nexting",
				!pos,yypos - !yypos_bol); continue());
<CCC>.                => (continue());
