functor ParseTreeFun(structure Pos: POS): PARSE_TREE =
  struct
    open Pos
    open Str
    open SympBug

    exception SympParseInternalError of string

    datatype ParseTree = 
	Program of Pos * ParseTree list
      | Include of Pos * string
      (* Included(pos,file,parse_tree) Parsed included file *)
      | Included of Pos * string * ParseTree
      | Quote of Pos * string
	(* This is for creating temporary objects in the tree and
	   replacing them later with new values, in all places at once.
	   Once all objects are final, one could walk over the tree and
	   make them permanent. *)
      | SharedRef of ParseTree ref
      | Id of Pos * string
	(* Unique ID, internal names of objects, to distinguish
	   them from the user-given names. *)
      | Uid of string
      | Tvar of Pos * string
      (* An internal type variable introduced in typechecking.
       * It means the type is not fully deduced yet; it cannot be 
       * considered as a polymorphic type variable. *)
      | TvarN of int
      (* Type of the form "<type>,...,<type> name", like "int list" *)
      | TypeInst of Pos * (ParseTree list) * ParseTree
      | Ellipsis of Pos
(*        | Module of Pos * ParseTree * (ParseTree list option) *)
(*  	              * (ParseTree option) * ParseTree *)
      (* Built-in top level declarations (like +, *, etc.) *)
      | TopClosure of ParseTree list
      | Module of Pos.Pos *
	{ name: ParseTree,
	  statparams: ParseTree list option,
	  dynparams: ParseTree option,
	  body: ParseTree
	}
      | ModuleClosure of
	{ name: ParseTree option,
	  uname: ParseTree, (* unique module identifier *)
	  (* dynName: ParseTree option, *)
	  Sig: ModuleSig,
	  closure: ParseTree list,
	  (* def = NONE for begin-end, since only closure is important in
	     that case *)
	  def: ParseTree option,
	  parent: ParseTree
	}
      (* Parent closure of an instantiated module *)
      (* | ModuleParamClosure of
	{ closure: ParseTree list,
	  parent: ParseTree
	} *)
      (* This is passed as a "parent closure" to inner functions.
	 Originally it contains "FunClosure" that only defines
	 the name space, and later it's overwritten by the actual
	 function definition, thus making a circular structure for
	 recursive functions *)
(*        | FunClosureRef of ParseTree ref *)
      (* This is a preliminary version of the  "parent closure",
       * without the actual definition.  It is used only at 
       * the time of typechecking, and is later replaced by the full
       * version. Maybe, it's not even needed... *)
(*        | FunClosureSmall of *)
(*  	{ name: ParseTree option, *)
(*  	  formals: ParseTree, (* There can only be one argument *) *)
(*  	  parent: ParseTree (* The parent's closure *) *)
(*  	} *)
      (* The full version of the above.  If body = Fake, then it's the
       * same as the above. *)
      | FunClosure of
	{ name: ParseTree option,
	  formals: ParseTree, (* There can only be one argument *)
	  parent: ParseTree, (* The parent's closure *)
	  body: ParseTree
	}
      (* The "linearized" representation of recursive function `f':
       * RecurFun(fn f => <the curried def. of `f'>) *)
      | RecurFun of ParseTree
      (* Closure for `let' expression *)
      | LetClosure of
	{ locals: ParseTree list,
	  parent: ParseTree,
	  body: ParseTree }
      (* Same for assignments *)
      | LetAsstClosure of
	{ locals: ParseTree list,
	  parent: ParseTree,
	  body: ParseTree list }
      (* Serves as an additional "context" for recursive or
       * parameterized type declarations. *)
      | TypeClosure of { name: ParseTree,
			 uname: ParseTree,
			 params: ParseTree list,
			 def: ParseTree,
			 recursive: bool,
			 parent: ParseTree }
      (* Keeps bound variables in a choice of a `case' statement *)
      | ChoiceClosure of { pattern: ParseTree,
			   (* Internal name for selector expr *)
			   uname: ParseTree,
			   names: ParseTree list,
			   body: ParseTree,
			   parent: ParseTree }
      (* Same for assignments *)
      | ChooseAsstClosure of 
	{ names: ParseTree list option,
	  choices: ParseTree list,
	  parent: ParseTree}
      (* `choices' is a list of `ChoiceAsstClosure' elements *)
      | ChoiceAsstClosure of
	{ pattern: ParseTree,
	  (* Internal name for selector expr *)
	  uname: ParseTree,
	  names: ParseTree list,
	  body: ParseTree list,
	  parent: ParseTree }
      | ForeachAsstClosure of
	{ names: ParseTree list,
	  assts: ParseTree list,
	  parent: ParseTree }

      (* The closure for the CHOOSE operator.  It's the same as ChoiceClosure,
       * only serves a different purpose. *)
      | ChooseClosure of { pattern: ParseTree,
			   Type: ParseTree, (* Type of the result *)
			   names: ParseTree list,
			   body: ParseTree,
			   parent: ParseTree }
      (* Quantifiers *)
      | ForallClosure of { names: ParseTree list,
			   body: ParseTree,
			   parent: ParseTree }
      | ExistsClosure of { names: ParseTree list,
			   body: ParseTree,
			   parent: ParseTree }
      | MuClosure of { name: ParseTree,
		       body: ParseTree,
		       parent: ParseTree }
      | NuClosure of { name: ParseTree,
		       body: ParseTree,
		       parent: ParseTree }
      | QuantifiedVar of { name: ParseTree,
			   uname: ParseTree,
			   Type: ParseTree }
      (* Module parallel composition *)
      | SyncClosure of { names: ParseTree list,
			 body: ParseTree,
			 parent: ParseTree }
      | AsyncClosure of { names: ParseTree list,
			  body: ParseTree,
			  parent: ParseTree }

      (* Context for the sequent in the proof system *)
      | SequentClosure of { names: ParseTree list,
			    parent: ParseTree }

      (* This is a builtin top-level object *)
      | Builtin of {name: ParseTree, Type: ParseTree}
      (* A state variable object *)
      | StateVar of
	{ name: ParseTree,
	  uname: ParseTree,
	  Type: ParseTree,
	  (* State variables have to be distinguished later for
	     different module instances.  The `id' field will hold the
	     string of module instances (e.g. M1[i].M2[j].M3(y,x)) the
	     state var is coming from, where Mi are "uname"'s of
	     modules.  The context provides the direct link to the
	     transition system information. *)
	  id: ParseTree
	}
      | SkolemConst of {name: ParseTree,
			uname: ParseTree,
			Type: ParseTree }

      (* Uninterpreted constant similar to SkolemConst, only it's guaranteed
       to have different interpretation from any other AbstractConst *)
      | AbstractConst of {name: ParseTree,
			  uname: ParseTree,
			  Type: ParseTree }

      (* Object is a typechecked named object representation.
         It must carry enough info to disambiguate overloaded
	 identifiers. *)
      | Object of
	{ name: ParseTree,
	  uname: ParseTree,
	  Type: ParseTree,
	  def: ParseTree
	}
      (* Object instance that replaces a name in the program.  The
	 `subst' field is a function that properly instantiates the
         types within the object for this particular instance. *)
      | ObjectInst of {obj: ParseTree,
		       subst: ParseTree -> ParseTree}

      (**************************************************************)

      (* A variable bound by a pattern *)
      | PatternFormal of { name: ParseTree,
			   uname: ParseTree,
			   Type: ParseTree,
			   (* Function that wraps a proper extracter around 
			    * an actual parameter *)
			   extract: ParseTree -> ParseTree }
      (* A variable bound by a pattern in dynamic parameter of a module *)
      | DynPatternFormal of { name: ParseTree,
			      uname: ParseTree,
			      Type: ParseTree,
			     (* Function that wraps a proper extracter around 
			      * an actual parameter *)
			      extract: ParseTree -> ParseTree,
			      value: ParseTree option }
      (* Datatype constructor. 
       * The type is either the datatype itself (EnumType), or
       * paramType -> datatype (FunType). *)
      | TypeConstr of { name: ParseTree,
		        uname: ParseTree,
		        Type: ParseTree }
      (* Record field object.
         `uname' doesn't really make sense here. *)
      | RecordField of { name: ParseTree,
			 Type: ParseTree
		       (* Avoid circularity.  We probably do not need it. *)
		       (* recordType: ParseTree *) }
      (* Static formal constant and type parameters of a module.
         Type parameters are represented by PType. *)
      | StaticFormalConst of
	{ name: ParseTree,
	  uname: ParseTree,
	  Type: ParseTree,
	  value: ParseTree option }
      | StaticFormalType of
	{ name: ParseTree,
	  uname: ParseTree,
	  value: ParseTree option }
      | Theorem of Pos * ParseTree * ParseTree
      (* Parametric type in the static params of a module *)
      | PType of Pos * ParseTree
      | Symmetric of Pos * ParseTree
      | Finite of Pos * ParseTree
      | BeginEnd of Pos * ParseTree list
      | Sync2 of Pos * ParseTree * ParseTree
      | Async2 of Pos * ParseTree * ParseTree
      | Sync of Pos * ParseTree list * ParseTree
      | Async of Pos * ParseTree list * ParseTree
      (* TypeDecl(pos,name,paramlist,def) *)
      | TypeDecl of Pos * ParseTree * (ParseTree list) * ParseTree
      | DatatypeDecl of Pos * ParseTree * (ParseTree list) * ParseTree
      | Var of Pos * ParseTree list
      | StateVarDecl of Pos * ParseTree list
      | Val of Pos * ParseTree * ParseTree
      | FunCases of Pos * ParseTree list
      (* FunCase(pos,name,patterns,result_type,expr) *)
      | FunCase of Pos * ParseTree * (ParseTree list)
	* (ParseTree option) * ParseTree
      | Open of Pos * ParseTree
      | Export of Pos * ParseTree list
      | Models of Pos * ParseTree * ParseTree
      | Refines of Pos * ParseTree * ParseTree
      (* LabeledAsst(p, label, asst) *)
      | LabeledAsst of Pos * ParseTree * ParseTree
      | Nop of Pos
      | NormalAsst of Pos * ParseTree * ParseTree
      | InitAsst of Pos * ParseTree * ParseTree
      | NextAsst of Pos * ParseTree * ParseTree
      (* Wrapper to represent next(x) in boolean formulas for the
         transition relation (internal use only) *)
      | Next of ParseTree
      | IfAsst of Pos * (ParseTree list) * ParseTree list
      | CondAsst of Pos * ParseTree * (ParseTree list)
      | LetAsst of Pos * (ParseTree list) * (ParseTree list)
      | CaseAsst of Pos * ParseTree * (ParseTree list)
      | ChoiceAsst of Pos * ParseTree * (ParseTree list)
      | ChooseAsst of Pos * (ParseTree list option) * ParseTree list
      | ForeachAsst of Pos * ParseTree list * ParseTree list
      | IfExpr of Pos * (ParseTree list) * ParseTree
      | CondExpr of Pos * ParseTree * ParseTree
      | LetExpr of Pos * (ParseTree list) * ParseTree
      | CaseExpr of Pos * ParseTree * (ParseTree list)
      | ChoiceExpr of Pos * ParseTree * ParseTree
      | ChooseExpr of Pos * ParseTree * ParseTree
      | WithExpr of Pos * ParseTree * (ParseTree list)
      | WithAsst of Pos * ParseTree * ParseTree
      | RecordExpr of Pos * ParseTree list
      | TupleExpr of Pos * ParseTree list
      | NondetExpr of Pos * ParseTree list
      | RecordAsst of Pos * ParseTree * ParseTree
      (* ModuleInst(name,stat_param_list).
       Dynamic params are applied with Appl. *)
      | ModuleInst of Pos * ParseTree * (ParseTree list)
      | Dot of Pos * ParseTree * ParseTree
      | Type of Pos * ParseTree
      | Datatype of Pos * ParseTree
      | And of Pos
      | Or of Pos
      | Implies of Pos
      (* Normally, these two will also be "Implies", but the user may 
       * redefine them to something else *)
      | Arrow of Pos
      | Darrow of Pos
      | Iff of Pos
      | Eq of Pos
      | NotEq of Pos
      | Not of Pos
      | Forall of Pos * (ParseTree list) * ParseTree
      | Exists of Pos * (ParseTree list) * ParseTree
      (* CTL constructs *)
      | Ag of Pos
      | Af of Pos
      | Eg of Pos
      | Ef of Pos
      | Ax of Pos
      | Ex of Pos
      | Au of Pos
      | Eu of Pos
      | Ar of Pos
      | Er of Pos
      (* LTL constructs *)
      | Globally of Pos
      | Eventually of Pos
      | Until of Pos
      | Releases of Pos
      | NextTime of Pos
      (* Mu-Calculus constructs: ?u(pos, var, expr) *)
      | Mu of Pos * ParseTree * ParseTree
      | Nu of Pos * ParseTree * ParseTree
      (* Arithmetic operators *)
      | Lt of Pos
      | Gt of Pos
      | Le of Pos
      | Ge of Pos
      | Plus of Pos
      | Minus of Pos
      | UMinus of Pos
      | Times of Pos
      | Div of Pos
      | Mod of Pos
      | Number of Pos * int
      | BoolType of Pos
      (* Internal type for 2-valued boolean operations *)
      | BoolType2
      | NatType of Pos
      | IntType of Pos
      | RangeType of Pos * ParseTree * ParseTree
      | FunType of Pos * ParseTree * ParseTree
      | TupleType of Pos * ParseTree list
      | ArrayType of Pos * ParseTree * ParseTree
      | RecordType of Pos * ParseTree list
      | EnumType of Pos * ParseTree list
      | AbstractType of ParseTree list
      | Of of Pos * ParseTree * ParseTree
      | TypedVars of Pos * (ParseTree list) * (ParseTree option)
      | False of Pos
      | True of Pos
      (* Special 2-valued true and false, for the internal use of the
         transition relation generator *)
      | False2
      | True2
      | Min of Pos
      | Max of Pos
      | Undefined of Pos
      | Anyvalue of Pos
      (* Internal value representing "all the other values" in abstractions *)
      | OtherValue
      | Self of Pos
      | Appl of Pos * ParseTree * ParseTree
      | TypedExpr of Pos * ParseTree * ParseTree
      | Fn of Pos * ParseTree list
      | ChoicePattern of Pos * ParseTree list
      | ApplPattern of Pos * ParseTree * ParseTree
      | TypedPattern of Pos * ParseTree * ParseTree
      | TuplePattern of Pos * ParseTree list
      | RecordPattern of Pos * ParseTree list
	(* AsPattern(p,id,pattern) represents `id as pattern' *)
      | AsPattern of Pos * ParseTree * ParseTree
      | Underscore of Pos
      (* And pattern extractors, for internal use only, to provide a way
       * of assigning values to variables in the pattern *)
      (* ExtractAppl(DatatypeConstructer,Expr) *)
      | ExtractAppl of ParseTree * ParseTree
      (* Takes n-th element of a tuple *)
      | ExtractTuple of int * ParseTree
      (* ExtractRecord(FieldName, Expr) - no full record type is needed,
       * the expression must contain the field info *)
      | ExtractRecord of ParseTree * ParseTree
      (* ExtractIndex(Expr) - get the TypeConstr that makes the value
         of `Expr' that is of EnumType *)
      | ExtractIndex of ParseTree

      (* for emphasizing grouping, e.g. for tuple type *)
      | Group of Pos * ParseTree 
      | Fake (* for some errors and non-implemented commands *)
    and ModuleSig =
       (* Module's "type" for internal typechecking purposes.
       * In the future it is likely to be extended to ML-like signatures. *)
      ModuleSig of { statparams: ParseTree list option,
		     (* (name, type) of dyn. param *)
		     dynParam: ParseTree  option }


    fun isInfix(And _) = true
      | isInfix(Or _) = true
      | isInfix(Implies _) = true
      | isInfix(Arrow _) = true
      | isInfix(Darrow _) = true
      | isInfix(Iff _) = true
      | isInfix(Eq _) = true
      | isInfix(NotEq _) = true
      | isInfix(Plus _) = true
      | isInfix(Minus _) = true
      | isInfix(Times _) = true
      | isInfix(Div _) = true
      | isInfix(Mod _) = true
      | isInfix(Object{def=Builtin{name=n,...},...}) = isInfix n
      | isInfix(ObjectInst{obj=obj,...}) = isInfix obj
      | isInfix(Builtin{name=n,...}) = isInfix n
      | isInfix _ = false

    fun isUnary(Not _) = true
      | isUnary(UMinus _) = true
      | isUnary(Ag _) = true
      | isUnary(Af _) = true
      | isUnary(Eg _) = true
      | isUnary(Ef _) = true
      | isUnary(Ax _) = true
      | isUnary(Ex _) = true
      | isUnary(Object{def=Builtin{name=n,...},...}) = isUnary n
      | isUnary(ObjectInst{obj=obj,...}) = isUnary obj
      | isUnary _ = false

    (* This will eventually be a pretty printer.
       The `isDecl' is a boolean flag for printing the declaration of an
       object rather than just its name *)
    fun pt2strCommon isDecl tree =
	let val op * = fn (x,y) => Conc(x,y)
	    (* Different info to carry around for pretty printing *)
	    type Cxt = { (* Current substitution from `ObjectInst' *)
			 subst: ParseTree -> ParseTree,
			 (* whether it's a declaration or a use *)
			 isDecl: bool,
			 (* Whether it is a type or not (for type constructors) *)
			 isType: bool,
			 offset: int }
	    val cxt = {subst=(fn x=>x),
		       isDecl=isDecl,
		       isType=false,
		       offset=0}
	    fun upd_subst{subst=ff,
			  isDecl=d,
			  isType=t,
			  offset=n} newff =
		  {subst=newff,
		   isDecl=d,
		   isType=t,
		   offset=n}
	    fun upd_isDecl{subst=ff,
			   isDecl=d,
			   isType=t,
			   offset=n} newd =
		  {subst=ff,
		   isDecl=newd,
		   isType=t,
		   offset=n}
	    fun upd_isType{subst=ff,
			   isDecl=d,
			   isType=t,
			   offset=n} newt =
		  {subst=ff,
		   isDecl=d,
		   isType=newt,
		   offset=n}
	    fun upd_offset{subst=ff,
			   isDecl=d,
			   isType=t,
			   offset=n} offset =
		  {subst=ff,
		   isDecl=d,
		   isType=t,
		   offset=offset}
	    fun add_offset{subst=ff,
			   isDecl=d,
			   isType=t,
			   offset=n} offsetDiff =
		  {subst=ff,
		   isDecl=d,
		   isType=t,
		   offset=n+offsetDiff}
	    fun subst({subst=ff,...}:Cxt)  = ff
	    fun isDecl({isDecl=d,...}:Cxt) = d
	    fun isType({isType=d,...}:Cxt) = d
	    fun offset({offset=n,...}:Cxt) = n
	    fun offsetString({offset=n,...}:Cxt) = 
		 Vector.foldr(op ^) "" (Vector.tabulate(n,(fn _ => " ")))
	    fun offsetStr cxt = Str(offsetString cxt)
	    fun nls cxt = "\n"^(offsetString cxt)
	    fun nl cxt = (Str (nls cxt))
	    fun list2str cxt sep (x::y::tl) = 
		(loop cxt x)*(Str sep)*(list2str cxt sep (y::tl))
	      | list2str cxt _ ([x]) = loop cxt x
	      | list2str _ _ [] = Str ""
	    and option2str _ NONE = Str ""
	      | option2str cxt (SOME tree) = loop cxt tree
	    and optionLst2str _ _ NONE = Str ""
	      | optionLst2str cxt sep (SOME lst) = list2str cxt sep lst
	    and loop (cxt:Cxt) (Program(_,lst)) =
		  (list2str cxt (";"^(nls cxt)) lst)
	      | loop cxt (Include(_,str)) = (Str("include(\""^str^"\")"))
	      | loop cxt (Included(_,str,x)) = 
		(Str("-- include(\""^str^"\")"^(nls cxt)))
		*(loop cxt x)
		*(Str((nls cxt)^"-- end of file "
		      ^str^(nls cxt)))
	      | loop cxt (Quote(_,str)) = (Str ("\""^str^"\""))
	      | loop cxt (SharedRef x) = 
		  (Str "SharedRef(")*(loop cxt (!x))*(Str ")")
	      | loop cxt (Id(_,str)) = Str str
	      | loop cxt (Uid(str)) = Str("<"^str^">")
	      | loop cxt (Tvar(_,str)) = Str("'"^str)
	      | loop cxt (TvarN n) = Str("'X!"^(Int.toString n))
	      | loop cxt (TypeInst(_,parms,tp)) =
		  let val {subst=ff,...} = cxt
		  in (Str "(")*(list2str cxt "," parms)*(Str ") ")
		      *(loop cxt (ff tp))
		  end
	      | loop cxt (Ellipsis _) = Str "..."
	      | loop cxt (TopClosure lst) =
		  let val cxt = upd_isDecl(add_offset cxt 2) true
		  in (Str ((nls cxt)^"-- Top Closure"^(nls cxt)))
		      *(list2str cxt (";"^(nls cxt)) (List.rev lst))
		      *(Str ((nls cxt)^"-- End of Top Closure"^(nls cxt)))
		  end
	      | loop cxt (Module(_,{ name=name,
				   statparams=stat,
				   dynparams=dyn,
				   body=body})) = 
		(Str "module ")*(loop cxt name)
		  *(case stat of
			NONE => Str ""
		      | SOME lst => (Str "[")*(list2str cxt "," lst)*(Str "]"))
		  *(case dyn of
			NONE => Str ""
		      | SOME p => loop cxt p)
		  *(Str " = ")*(loop cxt body)
	      | loop cxt (ModuleClosure{name=name,
					uname=uname,
					Sig=ms,
				       def=def,
				       closure=cl,
				       parent=p}) =
		  if isDecl cxt then
		    (Str ((nls cxt)^"module "))
		    *(case name of
			  NONE => (loop cxt uname) (* (Str "<unnamed>") *)
			| SOME n => (loop cxt n))
		    *(Str ": ")*(ModuleSig2Str ms)
		    *(Str (" ="))
		    *(case def of
			  NONE =>
			      (nl cxt)
			      *(let val cxt = upd_isDecl(add_offset cxt 2) true
				in (Str ("begin"^(nls cxt)))
				    *(list2str cxt (";"^(nls cxt))
				      (List.rev cl))
				end)*(Str ((nls cxt)^"end"^(nls cxt)))
			| SOME d => (loop (upd_isDecl cxt false) d)*(nl cxt))
		  else (case name of
			  NONE => (Str "<unnamed module>")
			| SOME n => (loop cxt n))
	      (* | loop cxt (ModuleParamClosure{closure=cl, parent=p}) =
		  (Str "ModuleParamClosure")
		  *(nl (add_offset cxt 2))
		  *(Str "{closure = [")
		  *(let val cxt = (add_offset cxt 14)
		    in (list2str cxt (","^(nls cxt)) (List.rev cl))
		    end)
		  *(Str ("],"^(nls (add_offset cxt 3))^"parent = "))
		  (* *(loop cxt p) *) *(Str "<skipping...>")
		  *(Str ("}"^(nls cxt))) *)
	      | loop cxt (FunClosure{formals=parms,
				     body=body, ...}) =
		  let val cxt = (add_offset cxt 3)
		  in (Str "(fn ")*(loop cxt parms)
		      *(Str " => ")
		      *(loop (add_offset cxt 2) body)*(Str ")")
		  end
	      | loop cxt (RecurFun f) =
		  let val cxt = (add_offset cxt 9)
		  in (Str "RecurFun(")*(loop cxt f)*(Str ")")
		  end
	      | loop cxt (LetClosure{locals=locs,parent=p,body=b}) =
		  (Str ((nls cxt)^"let "))
		   *(let val cxt = upd_isDecl(add_offset cxt 4) true
		     in (list2str cxt (nls cxt) (List.rev locs))
		     end)
		  *(Str ((nls cxt)^"in "))
		   *(loop (upd_isDecl(add_offset cxt 3) false) b)
		  *(Str ((nls cxt)^"end"^(nls cxt)))
	      | loop cxt (LetAsstClosure{locals=locs,parent=p,body=b}) =
		  (Str ((nls cxt)^"let "))
		   *(let val cxt = upd_isDecl(add_offset cxt 4) true
		     in (list2str cxt (nls cxt) (List.rev locs))
		     end)
		  *(Str ((nls cxt)^"in "))
		   *(let val cxt = upd_isDecl(add_offset cxt 3) false
		     in list2str cxt (";"^(nls cxt)) b
		     end)
		  *(Str ((nls cxt)^"end"^(nls cxt)))
	      | loop cxt (TypeClosure{name=name,uname=uname,
				      params=parms,def=def,
				      recursive=r,parent=p}) =
		  (* Reset the substitution - the type closure has no
		     external type vars. *)
		  let val cxt = upd_isType(upd_subst cxt (fn x=>x)) true
		      val {isDecl=d,...} = cxt
		  in 
		      if d then
			  (Str (if r then "datatype " else "type "))
			  *(case parms of
				[] => (Str "")
			      | [a] =>(loop cxt a)
			      | _ => (Str "(")*(list2str cxt "," parms)
				    *(Str ")"))
			  *(Str " ")*(loop cxt name)*(Str " =")
			  *(let val cxt = add_offset (upd_isDecl cxt false) 3
			    in (Str "  ")*(loop cxt def)
			    end)
		      else (loop cxt name)
		  end
	      | loop cxt (ChoiceClosure{pattern=pat,names=names,
					body=b,parent=p,...}) =
		  (loop cxt pat)*(Str " => ")*(loop cxt b)
	      | loop cxt (ChooseAsstClosure{names=namesOpt,
					    choices=choices,
					    parent=p}) =
		  let val cxt2 = add_offset cxt 2
		      val cxt4 = add_offset cxt 4
		  in (Str "choose")
		      *(option2Str(Str "")
			(Option.map(fn l=>(list2str cxt ", " l)
				    *(Str ":")) namesOpt))
		      *(Str(nls cxt4))
		      *(list2str cxt ((nls cxt2)^"| ") choices)
		      *(Str ((nls cxt)^"endchoose"))
		  end
	      | loop cxt (ForeachAsstClosure{names=names,
					     assts=lst,
					     parent=p}) =
		  let val cxt2 = add_offset cxt 2
		  in (Str "forall ")
		      *(list2str cxt "," names)*(Str ":")
		      *(Str (nls cxt))
		      *(list2str cxt2 (nls cxt2) lst)
		      *(Str ((nls cxt)^"endforall"))
		  end
	      | loop cxt (ChoiceAsstClosure{pattern=pat,
					    names=names,
					    body=b,parent=p,...}) =
		  (loop cxt pat)
		  *(let val cxt = add_offset cxt 3
		    in (Str (" =>"^(nls cxt)))
			*(list2str cxt (nls cxt) b)
		    end)
	      | loop cxt (ChooseClosure{pattern=pat,names=names,
					body=b,parent=p,...}) =
		  (Str "pick ")
		  *(let val cxt = upd_isDecl cxt false
		    in (loop cxt pat)*(Str ": ")*(loop cxt b)
		    end)
	      | loop cxt (ForallClosure{names=names,body=body,parent=p}) =
		  let val cxt = upd_isDecl cxt false
		  in (Str "forall ")
		      *(list2str cxt "," names)*(Str ": ")
		      *(loop cxt body)
		  end
	      | loop cxt (ExistsClosure{names=names,body=body,parent=p}) =
		  let val cxt = upd_isDecl cxt false
		  in (Str "exists ")
		      *(list2str cxt "," names)*(Str ": ")
		      *(loop cxt body)
		  end
	      | loop cxt (MuClosure{name=name,body=body,parent=p}) =
		  let val cxt = upd_isDecl cxt false
		  in (Str "Mu ")
		      *(loop cxt name)*(Str ": ")
		      *(loop cxt body)
		  end
	      | loop cxt (NuClosure{name=name,body=body,parent=p}) =
		  let val cxt = upd_isDecl cxt false
		  in (Str "Nu ")
		      *(loop cxt name)*(Str ": ")
		      *(loop cxt body)
		  end
	      | loop cxt (QuantifiedVar{name=n,uname=_,Type=tp}) =
		  (Str "(")*(loop cxt n)*(Str ": ")
		  *(loop (upd_isType(upd_isDecl cxt false) true)
		         (subst cxt tp))*(Str ")")
	      | loop cxt (SyncClosure{names=names,body=body,parent=p}) =
		  let val cxt = upd_isDecl cxt false
		  in (Str "sync ")
		      *(list2str cxt "," names)*(Str ": ")
		      *(loop cxt body)
		  end
	      | loop cxt (AsyncClosure{names=names,body=body,parent=p}) =
		  let val cxt = upd_isDecl cxt false
		  in (Str "async ")
		      *(list2str cxt "," names)*(Str ": ")
		      *(loop cxt body)
		  end
	      | loop cxt (SequentClosure{names=names,parent=p}) =
		  let val cxt = upd_isDecl cxt false
		  in (Str "SequentClosure{names=[")
		      *(list2str cxt "," names)
		      *(Str "]\n               parent=<skipping>}")
		  end
	      | loop cxt (Builtin{name=n,Type=tp}) = 
		  if isDecl cxt then (Str "<Builtin(")*(loop cxt n)*(Str ")>")
		  else (loop cxt n)
	      | loop cxt (StateVar{name=name,uname=_,Type=tp,id=id}) =
		  if isDecl cxt then 
		      let val c = upd_isDecl cxt false
		      in  (Str "StateVar ")*(loop c name)
			  *(Str ": ")*(loop (upd_isType c true) (subst c tp))
			  *(Str " (* id=")*(loop c id)*(Str " *)")
		      end
		  else (loop cxt name)
	      | loop cxt (SkolemConst{name=name,...}) = (loop cxt name)
	      | loop cxt (AbstractConst{name=name,...}) = (loop cxt name)
	      | loop cxt (Object{name=name,uname=_,Type=tp,def=def}) =
		  if isDecl cxt then
		      (Str "val ")
		      *(case name of
			    Id _ => (loop cxt name)
			  | _ => (Str "op ")*(loop cxt name))
		      *(Str ": ")
		      *(let val cxt = upd_isDecl(add_offset cxt 5) false
			in (loop (upd_isType(add_offset cxt 3) true) tp)
			    *(Str " =")
			    *(nl cxt)*(loop cxt def)
			end)
		  else (loop cxt name)
	      (* Here we take the new substitution from ObjectInst *) 
	      | loop cxt (ObjectInst{obj=obj,subst=ff1}) =
		  (Str "@(")*
		  (loop (upd_isDecl 
			 (upd_subst cxt ((subst cxt) o ff1)) false) obj)
		  *(Str")")
	      | loop cxt (PatternFormal{name=name,Type=tp,...}) =
		  (Str "<[")*(loop cxt name)*(Str ": ")
		  *(loop (upd_isType cxt true) (subst cxt tp))
		  *(Str "]>")
	      | loop cxt (DynPatternFormal{name=name,Type=tp,value=v,...}) =
		  (Str "#<[")*(loop cxt name)*(Str ": ")
		  *(loop (upd_isType cxt true) (subst cxt tp))
		  *(case v of
			NONE => Str ""
		      | SOME x => (Str " = ")*(loop cxt x))
		  *(Str "]>")
	      | loop cxt (TypeConstr{name=name,uname=un,Type=tp}) =
		  if isDecl cxt then
		      let val cxt = add_offset cxt 3
		      in (Str "(* ")*(loop cxt name)
			  *(Str ": ")
			  *(loop (upd_isDecl (upd_isType cxt true) false)
			    (subst cxt tp))
			  *(Str " *)")
		      end
		  else (loop cxt name)
		      *(if isType cxt then 
			    (case tp of
				 FunType(_,t,_) => (Str " of ")*(loop cxt t)
			       | _ => Str "")
			else (Str ""))		      
	      | loop cxt (RecordField{name=name,Type=tp}) =
		  (loop cxt (TypedExpr(dp,name,(subst cxt tp))))
	      | loop cxt (StaticFormalConst{name=name,uname=_,Type=tp,value=v}) =
		  (if isDecl cxt then 
		      (loop (upd_isDecl cxt false)
		            (TypedExpr(dp,name,(subst cxt tp))))
		  else (loop cxt name))
		 *(case v of
		       NONE => (Str "")
		     | SOME x => (Str "< = ")*(loop cxt x)*(Str ">"))
	      | loop cxt (StaticFormalType{name=name,uname=_,value=v}) =
		   (Str "type ")*(loop cxt name)
		   *(case v of
			 NONE => Str ""
		       | SOME x => (Str "< = ")*(loop cxt x)*(Str ">"))
	      | loop cxt (Theorem(_,name,body)) = 
		   let val cxt2 = add_offset cxt 2
		   in (Str "theorem ")*(loop cxt2 name)
		       *(Str (" ="^(nls cxt2)))
		       *(loop (upd_isDecl cxt2 false) body)
		   end
	      | loop cxt (PType(_,tp)) = 
		  (if isDecl cxt then (Str "type ") else (Str ""))
		  *(loop cxt (subst cxt tp))
	      | loop cxt (Symmetric(_,tp)) = 
		  (Str "symmetric ")*(loop cxt (subst cxt tp))
	      | loop cxt (Finite(_,tp)) =
		  (Str "finite ")*(loop cxt (subst cxt tp))
	      | loop cxt (BeginEnd(_,exprs)) = 
		  (Str ((nls cxt)^"begin"))
		  *(let val cxt = add_offset cxt 2
		    in (Str ((nls cxt)^"  "))
			*(list2str cxt (";"^(nls cxt)) exprs)
		    end)
		  *(Str ((nls cxt)^"end -- module"^(nls cxt)))
	      | loop cxt (Sync2(_,e1,e2)) = 
		  (Str "(")*(loop cxt e1)*(Str ")")
		  *(Str " || ")*(Str "(")*(loop cxt e2)*(Str ")")
	      | loop cxt (Async2(_,e1,e2)) = 
		  (Str "(")*(loop cxt e1)*(Str ")")
		  *(Str " | ")*(Str "(")*(loop cxt e2)*(Str ")")
	      | loop cxt (Sync(_,vars,e)) =
		  (Str "sync ")*(list2str cxt ", " vars)*(Str ": ")
		  *(loop cxt e)
	      | loop cxt (Async(_,vars,e)) = 
		  (Str "async ")*(list2str cxt ", " vars)*(Str ": ")
		  *(loop cxt e)
	      | loop cxt (TypeDecl(_,name,parms,tp)) = 
		  (Str "type ")
		  *(case parms of
		       [] => (Str "")
		     | _ => (Str "(")*(list2str cxt "," parms)*(Str ") "))
		  *(loop cxt name)*(Str " = ")
		  *(loop (upd_isType cxt true) (subst cxt tp))
	      | loop cxt (DatatypeDecl(_,name,parms,tp)) = 
		  (Str "datatype ")
		  *(case parms of
		       [] => (Str "")
		     | _ => (Str "(")*(list2str cxt "," parms)*(Str ") "))
		  *(loop cxt name)
		  *(Str " = ")*(loop (upd_isType cxt true) (subst cxt tp))
	      | loop cxt (Var(_,vars)) = 
		  (Str "var ")*(list2str cxt ", " vars)
	      | loop cxt (StateVarDecl(_,vars)) = 
		  (Str "statevar ")*(list2str cxt ", " vars)
	      | loop cxt (Val(_,name,e)) = 
		  (Str "val ")*(loop cxt name)*(Str " = ")*(loop cxt e)
	      | loop cxt (FunCases(_,lst)) = 
		  let val cxt = add_offset cxt 2
		  in (Str "fun ")
		      *(list2str (add_offset cxt 2) ((nls cxt)^"| ") lst)
		  end
	      | loop cxt (FunCase(_,name,patterns,res_tp,e)) = 
		  (loop cxt name)*(Str " ")*(list2str cxt " " patterns)
		  *(case res_tp of
			NONE => Str ""
		      | SOME tp => (Str " : ")
			    *(loop (upd_isType cxt true) (subst cxt tp)))
		  *(Str " = ")
		  *(let val cxt = add_offset cxt 3
		    in (nl cxt)*(loop cxt e)
		    end)
	      | loop cxt (Open(_,e)) = (Str "open ")*(loop cxt e)
	      | loop cxt (Export(_,lst)) = 
		  (Str "export ")
		  *(case lst of
			[] => (Str "all")
		      | _  => list2str cxt ", " lst)
	      | loop cxt (Models(_,e1,e2)) = 
		  (loop cxt e1)*(Str ((nls cxt)^"|="^(nls cxt)))
		  *(loop cxt e2)
	      | loop cxt (Refines(_,e1,e2)) = 
		  (loop cxt e1)*(Str ((nls cxt)^"refines"^(nls cxt)))
		  *(loop cxt e2)
	      | loop cxt (LabeledAsst(_, l, a)) =
		  (Str "label ")*(loop cxt l)*(Str " ")*(loop cxt a)
	      | loop cxt (Nop _) = Str "nop"
	      | loop cxt (NormalAsst(_,name,e)) = 
		  (Str "(")*(loop cxt name)*(Str ") := ")
		  *(loop (upd_isDecl cxt false) e)
	      | loop cxt (InitAsst(_,name,e)) = 
		  (Str "init(")*(loop cxt name)*(Str ") := ")
		  *(loop (upd_isDecl cxt false) e)
	      | loop cxt (NextAsst(_,name,e)) = 
		  (Str "next(")*(loop cxt name)*(Str ") := ")
		  *(loop (upd_isDecl cxt false) e)
	      | loop cxt (Next e) = (Str "next(")*(loop cxt e)*(Str ")")
	      | loop cxt (IfAsst(_,cond_lst,else_lst)) = 
		  let val cxt2 = add_offset cxt 2
		  in (Str "if ")
		      *(list2str cxt2 ((nls cxt)^"elsif ") cond_lst)
		      *(Str ((nls cxt)^"else"^(nls cxt2)))
		      *(list2str cxt2 (nls cxt2) else_lst)
		      *(Str ((nls cxt)^"endif"))
		  end
	      | loop cxt (CondAsst(_,e,assts)) = 
		  (loop cxt e)*(Str (" then"^(nls cxt)))
		  *(list2str cxt (nls cxt) assts)
	      | loop cxt (LetAsst(_,locals,assts)) = 
		  (Str "let ")
		  *(let val cxt = upd_isDecl(add_offset cxt 4) true
		    in (list2str cxt (nls cxt) locals)
		    end)
		  *(Str ((nls cxt)^"in"^(nls cxt)^"  "))
		  *(let val cxt = add_offset cxt 3
		    in (list2str cxt (";"^(nls cxt)) assts)
		    end)
		  *(Str ((nls cxt)^"end "))
	      | loop cxt (CaseAsst(_,e,cases)) = 
		  let val cxt2 = add_offset cxt 2
		      val cxt4 = add_offset cxt 4
		  in (Str ((nls cxt)^"case "))*(loop cxt4 e)*(Str (" of"^(nls cxt4)))
		      *(list2str cxt4 ((nls cxt2)^"| ") cases)
		      *(Str ((nls cxt)^"endcase"))
		  end
	      | loop cxt (ChoiceAsst(_,e, assts)) =
		  let val cxt2 = add_offset cxt 2
		      val cxt6 = add_offset cxt 6
		  in (loop cxt2 e)
		      *(Str (" =>"^(nls cxt6)))
		      *(list2str cxt6 (";"^(nls cxt6)) assts)
		  end
	      | loop cxt (ChooseAsst(_,paramsOpt, lst)) = 
		  let val cxt2 = add_offset cxt 2
		      val cxt4 = add_offset cxt 4
		  in (Str "choose")
		      *(option2Str(Str "")
			(Option.map(fn l=>(list2str cxt ", " l)
				    *(Str ":")) paramsOpt))
		      *(Str(nls cxt4))
		      *(list2str cxt ((nls cxt2)^"| ") lst)
		      *(Str ((nls cxt)^"endchoose"))
		  end
	      | loop cxt (ForeachAsst(_, names, lst)) =
		  let val cxt2 = add_offset cxt 2
		  in (Str "forall ")
		      *(list2str cxt "," names)*(Str ":")
		      *(Str (nls cxt))
		      *(list2str cxt2 (nls cxt2) lst)
		      *(Str ((nls cxt)^"endforall"))
		  end
	      | loop cxt (IfExpr(_,conds,else_exp)) = 
		  let val cxt2 = add_offset cxt 2
		  in (Str "if ")*(list2str cxt2 ((nls cxt)^"elsif ") conds)
		      *(Str ((nls cxt)^"else "))*(loop cxt2 else_exp)
		      *(Str ((nls cxt)^"endif "))
		  end
	      | loop cxt (CondExpr(_,e1,e2)) = 
		  (loop (add_offset cxt 3) e1)
		  *(Str (" then"^(nls cxt)))*(loop cxt e2)
	      | loop cxt (LetExpr(_,locals,e)) =
		  (Str "let ")
		  *(let val cxt = upd_isDecl(add_offset cxt 4) true
		    in (list2str cxt (nls cxt) locals)
		    end)
		  *(Str ((nls cxt)^"in "))
		  *(loop (upd_isDecl(add_offset cxt 3) false) e)
		  *(Str ((nls cxt)^"end "))
	      | loop cxt (CaseExpr(_,e,cases)) = 
		  let val cxt2 = add_offset cxt 2
		      val cxt4 = add_offset cxt 4
		  in (Str ((nls cxt)^"case "))*(loop cxt4 e)
		      *(Str (" of"^(nls cxt4)))
		      *(list2str cxt4 ((nls cxt2)^"| ") cases)
		  end
		  *(Str ((nls cxt)^"endcase "))
	      | loop cxt (ChoiceExpr(_,p,e)) = 
		  let val cxt2 = add_offset cxt 2
		      val cxt6 = add_offset cxt 6
		  in (loop cxt2 p)*(Str (" =>"^(nls cxt6)))*(loop cxt6 e)
		  end
	      | loop cxt (ChooseExpr(_,p,e)) = 
		  (Str "pick ")*(loop cxt p)*(Str ": ")*(loop cxt e)
	      | loop cxt (WithExpr(_,e,lst)) = 
		  (loop cxt e)*(Str " with [ ")*(list2str cxt ", " lst)*(Str " ]")
	      | loop cxt (WithAsst(_,name,e)) = 
		  (loop cxt name)*(Str " = ")*(loop cxt e)
	      | loop cxt (RecordExpr(_,lst)) = 
		  (Str "{ ")*(list2str cxt ", " lst)*(Str " }")
	      | loop cxt (TupleExpr(_,lst)) = 
		  (Str "( ")*(list2str cxt ", " lst)*(Str " )")
	      | loop cxt (NondetExpr(_,lst)) = 
		  (Str "(")*(list2str cxt ") | (" lst)*(Str ")")
	      | loop cxt (RecordAsst(_,name,e)) = 
		  (loop cxt name)*(Str " = ")*(loop cxt e)
	      | loop cxt (ModuleInst(_,name,stats)) = 
		  (loop cxt name)*(Str "[")*(list2str cxt ", " stats)*(Str "]")
	      | loop cxt (Dot(_,e1,e2)) = 
		  (loop cxt e1)*(Str ".")*(loop cxt e2)
	      | loop cxt (Type(_,e)) = 
		  (Str "type ")*(loop cxt e)
	      | loop cxt (Datatype(_,e)) = 
		  (Str "datatype ")*(loop cxt e)
	      | loop cxt (And _) = (Str "and")
	      | loop cxt (Or _) = (Str "or")
	      | loop cxt (Implies _) = (Str "implies")
	      | loop cxt (Arrow _) = (Str "->")
	      | loop cxt (Darrow _) = (Str "=>")
	      | loop cxt (Iff _) = (Str "<=>")
	      | loop cxt (Eq _) = (Str "=")
	      | loop cxt (NotEq _) = (Str "!=")
	      | loop cxt (Not _) = (Str "!")
	      | loop cxt (Forall(_,vars,e)) = 
		  (Str "forall ")*(list2str cxt ", " vars)*(Str ": ")*(loop cxt e)
	      | loop cxt (Exists(_,vars,e)) = 
		  (Str "exists ")*(list2str cxt ", " vars)*(Str ": ")*(loop cxt e)
	      | loop cxt (Ag _) = (Str "AG")
	      | loop cxt (Af _) = (Str "AF")
	      | loop cxt (Eg _) = (Str "EG")
	      | loop cxt (Ef _) = (Str "EF")
	      | loop cxt (Ax _) = (Str "AX")
	      | loop cxt (Ex _) = (Str "EX")
	      | loop cxt (Au _) = (Str "AU")
	      | loop cxt (Eu _) = (Str "EU")
	      | loop cxt (Ar _) = (Str "AR")
	      | loop cxt (Er _) = (Str "ER")
	      | loop cxt (Globally _) = (Str "Globally")
	      | loop cxt (Eventually _) = (Str "Eventually")
	      | loop cxt (Until _) = (Str "Until")
	      | loop cxt (Releases _) = (Str "Releases")
	      | loop cxt (NextTime _) = Str "Next"
	      | loop cxt (Mu(_, var, e)) = 
		  (Str "Mu ")*(loop cxt var)*(Str ": (")*(loop cxt e)*(Str ")")
	      | loop cxt (Nu(_, var, e)) = 
		  (Str "Nu ")*(loop cxt var)*(Str ": (")*(loop cxt e)*(Str ")")
	      | loop cxt (Lt _) = (Str "<")
	      | loop cxt (Gt _) = (Str ">")
	      | loop cxt (Le _) = (Str "<=")
	      | loop cxt (Ge _) = (Str ">=")
	      | loop cxt (Plus _) = (Str "+")
	      | loop cxt (Minus _) = (Str "-")
	      | loop cxt (UMinus _) = (Str "~")
	      | loop cxt (Times _) = (Str "*")
	      | loop cxt (Div _) = (Str "/")
	      | loop cxt (Mod _) = (Str "mod")
	      | loop cxt (Number(_,n)) = (Str (Int.toString n))
	      | loop cxt (BoolType _) = (Str "bool")
	      | loop cxt (BoolType2) = (Str "bool2")
	      | loop cxt (NatType _) = (Str "nat")
	      | loop cxt (IntType _) = (Str "int")
	      | loop cxt (RangeType(_,x,y)) =
		  (Str "[")*(loop cxt x)*(Str "..")*(loop cxt y)*(Str "]")
	      | loop cxt (FunType(_,tp1,tp2)) =
		  (Str "(")*(loop cxt (subst cxt tp1))*(Str ")")
		  *(Str " -> ")*(Str "(")*(loop cxt (subst cxt tp2))*(Str ")")
	      | loop cxt (TupleType(_,lst)) = 
		  (Str "(")*(list2str cxt ") * (" lst)*(Str ")")
	      | loop cxt (ArrayType(_,tp1,tp2)) = 
		  (Str "array ")*(loop cxt (subst cxt tp1))*(Str " of ")
		  *(loop cxt (subst cxt tp2))
	      | loop cxt (RecordType(_,lst)) = 
		  (Str "{ ")*(list2str cxt ", " lst)*(Str " }")
	      | loop cxt (EnumType(_,lst)) = 
		  (Str "(")*(list2str cxt " | " lst)*(Str ")")
	      | loop cxt (AbstractType(lst)) = 
		  (Str "AbstractType[")*(list2str cxt ", " lst)*(Str "]")
	      | loop cxt (Of(_,name,tp)) = 
		  (loop cxt name)*(Str " of ")*(loop cxt (subst cxt tp))
	      | loop cxt (TypedVars(_,vars,tp)) = 
		  (Str "(")*(list2str cxt ", " vars)
		  *(case tp of
			NONE => (Str "")
		      | SOME tp => (Str ": ")
			    *(loop (upd_isType cxt true) (subst cxt tp)))
		  *(Str ")")
	      | loop cxt (False _) = (Str "false")
	      | loop cxt (True _) = (Str "true")
	      | loop cxt (False2) = (Str "false2")
	      | loop cxt (True2) = (Str "true2")
	      | loop cxt (Min _) = (Str "min")
	      | loop cxt (Max _) = (Str "max")
	      | loop cxt (Undefined _) = (Str "undefined")
	      | loop cxt (Anyvalue _) = (Str "anyvalue")
	      | loop cxt OtherValue = (Str "OtherValue")
	      | loop cxt (Self _) = (Str "self")
	      | loop cxt (Appl(_,Au _,TupleExpr(_,[e1,e2]))) =
		  (Str "Always[")*(loop cxt e1)*(Str " Until ")*(loop cxt e2)*(Str "]")
	      | loop cxt (Appl(_,Eu _,TupleExpr(_,[e1,e2]))) =
		  (Str "Sometimes[")*(loop cxt e1)*(Str " Until ")
		  *(loop cxt e2)*(Str "]")
	      | loop cxt (Appl(_,Ar _,TupleExpr(_,[e1,e2]))) =
		  (Str "Always[")*(loop cxt e1)*(Str " Releases ")
		  *(loop cxt e2)*(Str "]")
	      | loop cxt (Appl(_,Er _,TupleExpr(_,[e1,e2]))) =
		  (Str "Sometimes[")*(loop cxt e1)*(Str " Releases ")
		  *(loop cxt e2)*(Str "]")
	      | loop cxt (Appl(_, Until _, TupleExpr(_,[e1,e2]))) =
		  (Str "(")*(loop cxt e1)*(Str " Until ")*(loop cxt e2)*(Str ")")
	      | loop cxt (Appl(_, Releases _, TupleExpr(_,[e1,e2]))) =
		  (Str "(")*(loop cxt e1)*(Str " Releases ")*(loop cxt e2)*(Str ")")
	      | loop cxt (Appl(_,e1,e2)) = 
		  let val cxt = upd_isDecl cxt false
		  in if isInfix(e1) then
		       (case e2 of
			    TupleExpr(_,[x,y]) =>
			      (Str "(")*(loop cxt x)*(Str " ")
			      *(loop cxt e1)*(Str " ")*(loop cxt y)*(Str ")")
			  | x => (Str "(op ")*(loop cxt e1)*(Str " ")
			      *(loop cxt e2)*(Str ")"))
		     else if isUnary(e1) then
		      (loop cxt e1)*(Str "(")*(loop cxt e2)*(Str ")")
		     else
		      (Str "(")*(loop cxt e1)*(Str " ")*(loop cxt e2)*(Str ")")
		  end
	      | loop cxt (TypedExpr(_,e,tp)) = 
		  (Str "((")*(loop cxt e)*(Str "): ")
		  *(loop (upd_isType cxt true) (subst cxt tp))*(Str ")")
	      | loop cxt (Fn(_,lst)) =
		  let val cxt2 = add_offset cxt 2
		      val cxt4 = add_offset cxt 4
		  in (Str "fn ")*(list2str cxt4 ((nls cxt2)^"| ") lst)
		  end
	      | loop cxt (ChoicePattern(_,lst)) = (list2str cxt " | " lst)
	      | loop cxt (ApplPattern(_,p1,p2)) =
		  (Str "(")*(loop cxt p1)*(Str " ")*(loop cxt p2)*(Str ")")
	      | loop cxt (AsPattern(_,p1,p2)) =
		  (Str "(")*(loop cxt p1)*(Str " as ")*(loop cxt p2)*(Str ")")
	      | loop cxt (TypedPattern(_,p,tp)) = 
		  (Str "(")*(loop cxt p)*(Str ": ")*(loop cxt (subst cxt tp))*(Str ")")
	      | loop cxt (TuplePattern(_,lst)) = 
		  (Str "(")*(list2str cxt ", " lst)*(Str ")")
	      | loop cxt (RecordPattern(_,lst)) = 
		  (Str "{")*(list2str cxt ", " lst)*(Str "}")
	      | loop cxt (Underscore _) = (Str "_")
	      | loop cxt (ExtractAppl(c,e)) =
		  (Str "ExtractAppl(")*(list2str cxt ", " [c,e])
		  *(Str ")")
	      | loop cxt (ExtractTuple(n,e)) =
		  (Str("ExtractTuple("^(Int.toString n)^","))
		  *(loop cxt e)*(Str ")")
	      | loop cxt (ExtractRecord(name,e)) =
		  (Str "ExtractRecord(")*(loop cxt name)*(Str ",")
		  *(loop cxt e)*(Str ")")
	      | loop cxt (ExtractIndex e) =
		  (Str "ExtractIndex(")*(loop cxt e)*(Str ")")
	      | loop cxt (Group(_,x)) = (Str "(")*(loop cxt x)*(Str ")")
	      | loop cxt  Fake = (Str "<Fake>")
(*  	      | loop cxt  x = raise SympBug("Can't print a ParseTree element.") *)
	in loop cxt tree
	end
    and ModuleSig2Str(ModuleSig{statparams=stat, dynParam=dyn}) =
	let val op * = fn (x,y) => Conc(x,y)
	in
	    (Str "ModuleSig{statparams=")
	    *(case stat of
		  NONE => Str "[]"
		| SOME lst => (Str "[")*(Strlist2Str "," (List.map(pt2strCommon true) lst)
					 *(Str "]")))
	    *(Str ",\n  dynParam=")
	    *(case dyn of
		  NONE => Str "<none>"
		| SOME x => pt2strCommon true x)
	    *(Str "}")
	end


    fun pt2str tree = pt2strCommon false tree

    fun pt2string tree = Str2string(pt2str tree)
    fun ModuleSig2string s = Str2string(ModuleSig2Str s)

    fun pt2shortStr length tree =
	let val str = pt2string tree
	in if (String.size str) > length then
	     (String.substring(str,0,length-3))^"..."
	   else str
	end

    (* Print a ParseTree with lots of debug info.  `print' is a
       function to print a single string and/or return another string
       representation.  `concat' concatenates two such string
       representations into one.  The return result is concatenation
       of all individual `print' results. *)
    local
    fun ptPrintDebugCommon' (print, concat) =
	let val op * = concat
	    fun list2str ff sep (x::y::tl) = 
		(loop ff x)*(print sep)*(list2str ff sep (y::tl))
	      | list2str ff _ ([x]) = loop ff x
	      | list2str _ _ [] = print ""
	    and option2str _ NONE = print ""
	      | option2str ff (SOME tree) = loop ff tree
	    and optionLst2str _ _ NONE = print ""
	      | optionLst2str ff sep (SOME lst) = list2str ff sep lst
	    and loop ff (Program(_,lst)) = (list2str ff ";\n" lst)
	      | loop ff (Include(_,str)) = (print("include(\""^str^"\")"))
	      | loop ff (Included(_,str,x)) = 
		(print("-- include(\""^str^"\")\n"))*(loop ff x)
		*(print("\n-- end of file "^str^"\n"))
	      | loop ff (Quote(_,str)) = (print ("\""^str^"\""))
	      | loop ff (SharedRef x) = (print "SharedRef(")*(loop ff(!x))*(print ")")
	      | loop ff (Id(_,str)) = print str
	      | loop ff (Uid(str)) = print("<"^str^">")
	      | loop ff (Tvar(_,str)) = print("'"^str)
	      | loop ff (TvarN n) = print("'X!"^(Int.toString n))
	      | loop ff (TypeInst(_,parms,tp)) =
		  (print "(")*(list2str ff "," parms)*(print ") ")
		  *(loop ff (ff tp))
	      | loop ff (Ellipsis _) = print "..."
	      | loop ff (TopClosure lst) =
		(print "\n -- Top Closure\n  ")
		*(list2str ff ";\n  " (List.rev lst))
		*(print "\n -- End of Top Closure\n")
	      | loop ff (Module(_,{ name=name,
				   statparams=stat,
				   dynparams=dyn,
				   body=body})) = 
		(print "module ")*(loop ff name)
		  *(case stat of
			NONE => print ""
		      | SOME lst => (print "[")*(list2str ff "," lst)*(print "]"))
		  *(case dyn of
			NONE => print ""
		      | SOME p => loop ff p)
		  *(print " = ")*(loop ff body)
	      | loop ff (ModuleClosure{name=name,
				       uname=uname,
				       (* dynName=dn, *)
				       Sig=ms,
				       def=def,
				       closure=cl,
				       parent=p}) =
		  (print "ModuleClosure{name=")
		  *(case name of
			NONE => (print "<unnamed>")
		      | SOME n => (loop ff n))
		  *(print ",\n              uname=")
		  *(loop ff uname)
		  *(print ",\n              Sig=")*(ptPrintModuleSig ff ms)
		  *(print ",\n              closure=[")
		  *(list2str ff ",\n                 " (List.rev cl))
		  *(print "],\n              def=")
		  *(case def of
			NONE => (print "NONE")
		      | SOME d => (loop ff d))
		  *(print ",\n              parent=")
		  *(print(pt2string p)) (* *(loop ff p) *) (*  *(print "<skipping...>") *)
		  *(print "}\n")
	      (* | loop ff (ModuleParamClosure{closure=cl, parent=p}) =
		  (print "ModuleParamClosure{closure = [")
		  *(list2str ff ", " (List.rev cl))
		  *(print "],\n                 parent = ")
		  (* *(loop ff p) *) *(print "<skipping...>")
		  *(print "}\n") *)
	      | loop ff (FunClosure{ name=name,
				 formals=parms,
				 body=body, ...}) =
		  (print "FunClosure{ name=")
		  *(case name of
			NONE => (print "<unnamed>")
		      | SOME(n) => (loop ff n))
		  *(print ", formals=")*(loop ff parms)
		  *(print ", body=")*(loop ff body)
		  *(print "}")
	      | loop ff (RecurFun f) = (print "RecurFun(")*(loop ff f)*(print ")")
	      | loop ff (LetClosure{locals=locs,parent=p,body=b}) =
		  (print "LetClosure{locals=[")*(list2str ff ", " (List.rev locs))
		  *(print "],\n           parent=")(* *(loop ff p) *)*(print "<skipping...>")
		  *(print ",\n           body=")*(loop ff b)
		  *(print "}\n")
	      | loop ff (LetAsstClosure{locals=locs,parent=p,body=b}) =
		  (print "LetAsstClosure{locals=[")*(list2str ff ", " (List.rev locs))
		  *(print "],\n           parent=")(* *(loop ff p) *)*(print "<skipping...>")
		  *(print ",\n           body=[")*(list2str ff ";\n    " b)
		  *(print "] }\n")
	      | loop ff (TypeClosure{name=name,uname=uname,params=parms,
				     def=def,recursive=r,parent=p}) =
		  (* Reset the substitution - the type closure has no
		     external type vars. *)
		  let val ff = (fn x=>x)
		  in (print "TypeClosure{name=")*(loop ff name)
		      *(print ",\n            uname=")*(loop ff uname)
		      *(print ",\n            params=[")*(list2str ff ", " parms)
		      *(print "],\n")
		      *(print "            def=")*(loop ff def)*(print ",\n")
		      *(if r then (print "            recursive=true,\n")
			else (print "            recursive=false,\n"))
		      *(print "            parent=")(* *(loop ff p) *)
		      *(print "<skipping...>")
		      *(print "}\n")
		  end
	      | loop ff (ChoiceClosure{pattern=pat,
				       uname=uname,
				       names=names,body=b,parent=p}) =
		  (print "ChoiceClosure{pattern=")*(loop ff pat)*(print ",\n")
		  *(print "              uname=")*(loop ff uname)*(print ",\n")
		  *(print "              names=[")
		  *(list2str ff "," names)*(print "],\n")
		  *(print "              body=")*(loop ff b)*(print ",\n")
		  *(print "              parent=")(* *(loop ff p) *)*(print "<skipping...>")
		  *(print "}\n")
	      | loop ff (ChoiceAsstClosure{pattern=pat,
					   uname=uname,
					   names=names,
					   body=b,parent=p}) =
		  (print "ChoiceAsstClosure{pattern=")*(loop ff pat)*(print ",\n")
		  *(print "                  uname=")*(loop ff uname)*(print ",\n")
		  *(print "                  names=[")
		  *(list2str ff "," names)*(print "],\n")
		  *(print "              body=[\n   ")
		  *(list2str ff "\n   " b)*(print "],\n")
		  *(print "              parent=")(* *(loop ff p) *)*(print "<skipping...>")
		  *(print "}\n")
	      | loop ff (ChooseAsstClosure{names=paramsOpt,
					   choices=lst, ...}) = 
		  (print "choose")
		  *(case paramsOpt of
		        NONE => print ""
		      | SOME l => (list2str ff ", " l)*(print ":"))
		  *(print "\n    ")
		  *(list2str ff "\n  | " lst)
		  *(print "\nendchoose ")
	      | loop ff (ForeachAsstClosure{names=names,
					    assts=lst,...}) =
		  (print "foreach ")*(list2str ff ", " names)*(print ":\n  ")
		  *(list2str ff "\n  " lst)
		  *(print "\nendforeach")
	      | loop ff (ChooseClosure{pattern=pat,Type=tp,names=names,body=b,parent=p}) =
		  (print "ChooseClosure{pattern=")*(loop ff pat)*(print ",\n")
		  *(print "              Type=")*(loop ff (ff tp))*(print ",\n")
		  *(print "              names=[")
		  *(list2str ff "," (List.rev names))*(print "],\n")
		  *(print "              body=")*(loop ff b)*(print ",\n")
		  *(print "              parent=")(* *(loop ff p) *)*(print "<skipping...>")
		  *(print "}\n")
	      | loop ff (ForallClosure{names=names,body=body,parent=p}) =
		  (print "ForallClosure{names=[")*(list2str ff "," names)
		  *(print "],\n              body=")*(loop ff body)
		  *(print ",\n              parent=")(* *(loop ff p) *)
		  *(print "<skipping...>")*(print "}\n")
	      | loop ff (ExistsClosure{names=names,body=body,parent=p}) =
		  (print "ExistsClosure{names=[")*(list2str ff "," names)
		  *(print "],\n              body=")*(loop ff body)
		  *(print ",\n              parent=")(* *(loop ff p) *)
		  *(print "<skipping...>")*(print "}\n")
	      | loop ff (MuClosure{name=name,body=body,parent=p}) =
		  (print "MuClosure{name=")*(loop ff name)
		  *(print ",\n         body=")*(loop ff body)
		  *(print ",\n         parent=")(* *(loop ff p) *)
		  *(print "<skipping...>")*(print "}\n")
	      | loop ff (NuClosure{name=name,body=body,parent=p}) =
		  (print "NuClosure{name=")*(loop ff name)
		  *(print ",\n         body=")*(loop ff body)
		  *(print ",\n         parent=")(* *(loop ff p) *)
		  *(print "<skipping...>")*(print "}\n")
	      | loop ff (QuantifiedVar{name=n,uname=uname,Type=tp}) =
		  (print "QuantifiedVar{name=")*(loop ff n)
		  *(print ",\n              uname=")*(loop ff uname)
		  *(print ",\n              Type=")*(loop ff (ff tp))
		  *(print " }")
	      | loop ff (SyncClosure{names=names,body=body,parent=p}) =
		  (print "SyncClosure{names=[")*(list2str ff "," names)
		  *(print "],\n            body=")*(loop ff body)
		  *(print ",\n            parent=")(* *(loop ff p) *)
		  *(print "<skipping...>")*(print "}\n")
	      | loop ff (AsyncClosure{names=names,body=body,parent=p}) =
		  (print "AsyncClosure{names=[")*(list2str ff "," names)
		  *(print "],\n             body=")*(loop ff body)
		  *(print ",\n             parent=")(* *(loop ff p) *)
		  *(print "<skipping...>")*(print "}\n")
	      | loop ff (SequentClosure{names=names,parent=p}) =
		  (print "SequentClosure{names=[")*(list2str ff "," names)
		  *(print "],\n               parent=")(* *(loop ff p) *)
		  *(print "<skipping...>")*(print "}\n")
	      | loop ff  (Builtin{name=n,Type=tp}) =
		  (print "<Builtin(")*(loop ff n)*(print ": ")*(loop ff (ff tp))*(print ")>")
	      | loop ff (StateVar{name=name,uname=uname,Type=tp,id=id}) =
		  (print "StateVar{name=")*(loop ff name)
		  *(print ",\n         uname=")*(loop ff uname)
		  *(print ",\n         Type=")*(loop ff (ff tp))
		  *(print ",\n         id=")*(loop ff id)*(print "}")
	      | loop ff (SkolemConst{name=name,uname=uname,Type=tp}) =
		  (print "SkolemConst{name=")*(loop ff name)
		  *(print ",\n            uname=")*(loop ff uname)
		  *(print ",\n            Type=")*(loop ff (ff tp))*(print "}")
	      | loop ff (AbstractConst{name=name,uname=uname,Type=tp}) =
		  (print "AbstractConst{name=")*(loop ff name)
		  *(print ",\n            uname=")*(loop ff uname)
		  *(print ",\n            Type=")*(loop ff (ff tp))*(print "}")
	      | loop ff (Object{name=name,uname=uname,Type=tp,def=def}) =
		  (print "{{")*(loop ff name)*(print " ")*(loop ff uname)
		  *(print " : ")*(loop ff (ff tp))
		  *(print " = ")*(loop ff def)*(print "}}")
	      (* Here we take the new substitution from ObjectInst *) 
	      | loop ff (ObjectInst{obj=obj,subst=ff1}) =
(*  		  (print "@")*(loop ff obj) *)
		  (print "@(")*(loop (ff o ff1) obj)*(print ")")
	      | loop ff (PatternFormal{name=name,uname=uname,Type=tp,...}) =
		  (print "<[")*(loop ff name)*(print " ")*(loop ff uname)
		  *(print " : ")*(loop ff (ff tp))
		  *(print "]>")
	      | loop ff (DynPatternFormal{name=name,uname=uname,Type=tp,value=v,...}) =
		  (print "#<[")*(loop ff name)*(print " ")*(loop ff uname)
		  *(print " : ")*(loop ff (ff tp))
		  *(case v of 
			NONE => print ""
		      | SOME x => (print " = ")*(loop ff x))
		  *(print "]>")
	      | loop ff (TypeConstr{name=name,uname=uname,Type=tp}) =
		  (print "[<")*(loop ff name)*(print " ")*(loop ff uname)
		  *(print " : ")*(loop ff (ff tp))*(print ">]")
	      | loop ff (RecordField{name=name,Type=tp}) =
		  (print "<{")*(loop ff name)*(print " : ")*(loop ff (ff tp))
		  *(print "}>")
	      | loop ff (StaticFormalConst{name=name,uname=uname,Type=tp,value=v}) =
		  (print "[[")*(loop ff name)*(print " ")*(loop ff uname)
		  *(print " : ")*(loop ff (ff tp))
		  *(case v of 
			NONE => print ""
		      | SOME x => (print " = ")*(loop ff x))
		  *(print "]]")
	      | loop ff (StaticFormalType{name=name,uname=uname,value=v}) =
		  (print "[[")*(loop ff name)*(print " ")*(loop ff uname)
		  *(print " : type")
		  *(case v of 
			NONE => print ""
		      | SOME x => (print " = ")*(loop ff x))
		  *(print "]]")
	      | loop ff (Theorem(_,name,body)) = 
		  (print "theorem ")*(loop ff name)*(print " = ")*(loop ff body)
	      | loop ff (PType(_,tp)) = (print "type ")*(loop ff (ff tp))
	      | loop ff (Symmetric(_,tp)) = 
		  (print "symmetric ")*(loop ff (ff tp))
	      | loop ff (Finite(_,tp)) = (print "finite ")*(loop ff (ff tp))
	      | loop ff (BeginEnd(_,exprs)) = 
		  (print "\nbegin\n  ")
		  *(list2str ff ";\n  " exprs)
		  *(print "\nend -- module\n")
	      | loop ff (Sync2(_,e1,e2)) = 
		  (print "(")*(loop ff e1)*(print ")")
		  *(print " || ")*(print "(")*(loop ff e2)*(print ")")
	      | loop ff (Async2(_,e1,e2)) = 
		  (print "(")*(loop ff e1)*(print ")")
		  *(print " | ")*(print "(")*(loop ff e2)*(print ")")
	      | loop ff (Sync(_,vars,e)) =
		  (print "sync ")*(list2str ff ", " vars)*(print ": ")*(loop ff e)
	      | loop ff (Async(_,vars,e)) = 
		  (print "async ")*(list2str ff ", " vars)*(print ": ")*(loop ff e)
	      | loop ff (TypeDecl(_,name,parms,tp)) = 
		  (print "type ")
		  *(case parms of
		       [] => (print "")
		     | _ => (print "(")*(list2str ff "," parms)*(print ") "))
		  *(loop ff name)*(print " = ")*(loop ff (ff tp))
	      | loop ff (DatatypeDecl(_,name,parms,tp)) = 
		  (print "datatype ")
		  *(case parms of
		       [] => (print "")
		     | _ => (print "(")*(list2str ff "," parms)*(print ") "))
		  *(loop ff name)
		  *(print " = ")*(loop ff (ff tp))
	      | loop ff (Var(_,vars)) = 
		  (print "var ")*(list2str ff ", " vars)
	      | loop ff (StateVarDecl(_,vars)) = 
		  (print "statevar ")*(list2str ff ", " vars)
	      | loop ff (Val(_,name,e)) = 
		  (print "val ")*(loop ff name)*(print " = ")*(loop ff e)
	      | loop ff (FunCases(_,lst)) = 
		  (print "fun ")*(list2str ff "\n  | " lst)
	      | loop ff (FunCase(_,name,patterns,res_tp,e)) = 
		  (loop ff name)*(print " ")*(list2str ff " " patterns)
		  *(case res_tp of
			NONE => print ""
		      | SOME tp => (print " : ")*(loop ff (ff tp)))
		  *(print " = ")*(loop ff e)
	      | loop ff (Open(_,e)) = (print "open ")*(loop ff e)
	      | loop ff (Export(_,lst)) = 
		  (print "export ")
		  *(case lst of
			[] => (print "all")
		      | _  => list2str ff ", " lst)
	      | loop ff (Models(_,e1,e2)) = 
		  (print "(")*(loop ff e1)*(print ") |= (")*(loop ff e2)*(print ")")
	      | loop ff (Refines(_,e1,e2)) = 
		  (print "(")*(loop ff e1)*(print ") refines (")*(loop ff e2)*(print ")")
	      | loop ff (LabeledAsst(_, l, a)) =
		  (print "label ")*(loop ff l)*(print " ")*(loop ff a)
	      | loop ff (Nop _) = print "nop"
	      | loop ff (NormalAsst(_,name,e)) = 
		  (print "(")*(loop ff name)*(print ") := ")*(loop ff e)
	      | loop ff (InitAsst(_,name,e)) = 
		  (print "init(")*(loop ff name)*(print ") := ")*(loop ff e)
	      | loop ff (NextAsst(_,name,e)) = 
		  (print "next(")*(loop ff name)*(print ") := ")*(loop ff e)
	      | loop ff (Next e) = (print "next(")*(loop ff e)*(print ")")
	      | loop ff (IfAsst(_,cond_lst,else_lst)) = 
		  (print "if ")
		  *(list2str ff "\nelsif\n  " cond_lst)
		  *(print "\nelse\n  ")*(list2str ff "\n  " else_lst)
		  *(print "\nendif ")
	      | loop ff (CondAsst(_,e,assts)) = 
		  (loop ff e)*(print " then\n  ")*(list2str ff "\n  " assts)
	      | loop ff (LetAsst(_,locals,assts)) = 
		  (print "let ")*(list2str ff "\n    " locals)
		  *(print "\nin\n  ")*(list2str ff ";\n  " assts)
		  *(print "\nend ")
	      | loop ff (CaseAsst(_,e,cases)) = 
		  (print "case ")*(loop ff e)*(print " of\n    ")
		  *(list2str ff "\n  | " cases)
		  *(print "\nendcase ")
	      | loop ff (ChoiceAsst(_,e,assts)) =
		  (loop ff e)*(print " =>\n      ")
		  *(list2str ff ";\n      " assts)
	      | loop ff (ChooseAsst(_,paramsOpt, lst)) = 
		  (print "choose")
		  *(case paramsOpt of
		        NONE => print ""
		      | SOME l => (list2str ff ", " l)*(print ":"))
		  *(print "\n    ")
		  *(list2str ff "\n  | " lst)
		  *(print "\nendchoose ")
	      | loop ff (ForeachAsst(_, names, lst)) =
		  (print "foreach ")*(list2str ff ", " names)*(print ":\n  ")
		  *(list2str ff "\n  " lst)
		  *(print "\nendforeach")
	      | loop ff (IfExpr(_,conds,else_exp)) = 
		  (print "if ")*(list2str ff "\nelsif " conds)
		  *(print "\nelse ")*(loop ff else_exp)
		  *(print "\nendif ")
	      | loop ff (CondExpr(_,e1,e2)) = 
		  (loop ff e1)*(print " then ")*(loop ff e2)
	      | loop ff (LetExpr(_,locals,e)) = 
		  (print "let ")
		  *(list2str ff "\n    " locals)
		  *(print "\nin ")*(loop ff e)*(print "\nend ")
	      | loop ff (CaseExpr(_,e,cases)) = 
		  (print "case ")*(loop ff e)*(print " of\n    ")
		  *(list2str ff "\n  | " cases)
		  *(print "\nendcase ")
	      | loop ff (ChoiceExpr(_,p,e)) = 
		  (loop ff p)*(print " => ")*(loop ff e)
	      | loop ff (ChooseExpr(_,p,e)) = 
		  (print "pick ")*(loop ff p)*(print ": ")*(loop ff e)
	      | loop ff (WithExpr(_,e,lst)) = 
		  (loop ff e)*(print " with [ ")*(list2str ff ", " lst)*(print " ]")
	      | loop ff (WithAsst(_,name,e)) = 
		  (loop ff name)*(print " = ")*(loop ff e)
	      | loop ff (RecordExpr(_,lst)) = 
		  (print "{ ")*(list2str ff ", " lst)*(print " }")
	      | loop ff (TupleExpr(_,lst)) = 
		  (print "( ")*(list2str ff ", " lst)*(print " )")
	      | loop ff (NondetExpr(_,lst)) = 
		  (print "(")*(list2str ff ") | (" lst)*(print ")")
	      | loop ff (RecordAsst(_,name,e)) = 
		  (loop ff name)*(print " = ")*(loop ff e)
	      | loop ff (ModuleInst(_,name,stats)) = 
		  (loop ff name)*(print "[")*(list2str ff ", " stats)*(print "]")
	      | loop ff (Dot(_,e1,e2)) = 
		  (loop ff e1)*(print ".")*(loop ff e2)
	      | loop ff (Type(_,e)) = 
		  (print "type ")*(loop ff e)
	      | loop ff (Datatype(_,e)) = 
		  (print "\ndatatype ")*(loop ff e)
	      | loop ff (And _) = (print "and")
	      | loop ff (Or _) = (print "or")
	      | loop ff (Implies _) = (print "implies")
	      | loop ff (Arrow _) = (print "->")
	      | loop ff (Darrow _) = (print "=>")
	      | loop ff (Iff _) = (print "<=>")
	      | loop ff (Eq _) = (print "=")
	      | loop ff (NotEq _) = (print "!=")
	      | loop ff (Not _) = (print "!")
	      | loop ff (Forall(_,vars,e)) = 
		  (print "forall ")*(list2str ff ", " vars)*(print ": ")*(loop ff e)
	      | loop ff (Exists(_,vars,e)) = 
		  (print "exists ")*(list2str ff ", " vars)*(print ": ")*(loop ff e)
	      | loop ff (Ag _) = (print "AG")
	      | loop ff (Af _) = (print "AF")
	      | loop ff (Eg _) = (print "EG")
	      | loop ff (Ef _) = (print "EF")
	      | loop ff (Ax _) = (print "AX")
	      | loop ff (Ex _) = (print "EX")
	      | loop ff (Au _) = (print "AU")
	      | loop ff (Eu _) = (print "EU")
	      | loop ff (Ar _) = (print "AR")
	      | loop ff (Er _) = (print "ER")
	      | loop ff (Globally _) = (print "Globally")
	      | loop ff (Eventually _) = (print "Eventually")
	      | loop ff (Until _) = (print "Until")
	      | loop ff (Releases _) = (print "Releases")
	      | loop ff (NextTime _) = print "Next"
	      | loop ff (Mu(_, var, e)) = 
		  (print "Mu")*(loop ff var)*(print ": (")*(loop ff e)*(print ")")
	      | loop ff (Nu(_, var, e)) = 
		  (print "Nu")*(loop ff var)*(print ": (")*(loop ff e)*(print ")")
	      | loop ff (Lt _) = (print "<")
	      | loop ff (Gt _) = (print ">")
	      | loop ff (Le _) = (print "<=")
	      | loop ff (Ge _) = (print ">=")
	      | loop ff (Plus _) = (print "+")
	      | loop ff (Minus _) = (print "-")
	      | loop ff (UMinus _) = (print "~")
	      | loop ff (Times _) = (print "*")
	      | loop ff (Div _) = (print "/")
	      | loop ff (Mod _) = (print "mod")
	      | loop ff (Number(_,n)) = (print (Int.toString n))
	      | loop ff (BoolType _) = (print "bool")
	      | loop ff (BoolType2) = (print "bool2")
	      | loop ff (NatType _) = (print "nat")
	      | loop ff (IntType _) = (print "int")
	      | loop ff (RangeType(_,x,y)) =
		  (print "[")*(loop ff x)*(print "..")*(loop ff y)*(print "]")
	      | loop ff (FunType(_,tp1,tp2)) =
		  (print "(")*(loop ff (ff tp1))*(print ")")
		  *(print " -> ")*(print "(")*(loop ff (ff tp2))*(print ")")
	      | loop ff (TupleType(_,lst)) = 
		  (print "(")*(list2str ff ") * (" lst)*(print ")")
	      | loop ff (ArrayType(_,tp1,tp2)) = 
		  (print "array ")*(loop ff (ff tp1))*(print " of ")
		  *(loop ff (ff tp2))
	      | loop ff (RecordType(_,lst)) = 
		  (print "{ ")*(list2str ff ", " lst)*(print " }")
	      | loop ff (EnumType(_,lst)) = 
		  (print "(")*(list2str ff " | " lst)*(print ")")
	      | loop ff (AbstractType(lst)) = 
		  (print "AbstractType[")*(list2str ff ", " lst)*(print "]")
	      | loop ff (Of(_,name,tp)) = 
		  (loop ff name)*(print " of ")*(loop ff (ff tp))
	      | loop ff (TypedVars(_,vars,tp)) = 
		  (print "(")*(list2str ff ", " vars)
		  *(case tp of
			NONE => (print "")
		      | SOME tp => (print ": ")*(loop ff (ff tp)))
		  *(print ")")
	      | loop ff (False _) = (print "false")
	      | loop ff (True _) = (print "true")
	      | loop ff (False2) = (print "false2")
	      | loop ff (True2) = (print "true2")
	      | loop ff (Min _) = (print "min")
	      | loop ff (Max _) = (print "max")
	      | loop ff (Undefined _) = (print "undefined")
	      | loop ff (Anyvalue _) = (print "anyvalue")
	      | loop ff OtherValue = (print "OtherValue")
	      | loop ff (Self _) = (print "self")
	      | loop ff (Appl(_,Au _,TupleExpr(_,[e1,e2]))) =
		  (print "Always[")*(loop ff e1)*(print " Until ")*(loop ff e2)*(print "]")
	      | loop ff (Appl(_,Eu _,TupleExpr(_,[e1,e2]))) =
		  (print "Sometimes[")*(loop ff e1)*(print " Until ")
		  *(loop ff e2)*(print "]")
	      | loop ff (Appl(_,Ar _,TupleExpr(_,[e1,e2]))) =
		  (print "Always[")*(loop ff e1)*(print " Releases ")
		  *(loop ff e2)*(print "]")
	      | loop ff (Appl(_,Er _,TupleExpr(_,[e1,e2]))) =
		  (print "Sometimes[")*(loop ff e1)*(print " Releases ")
		  *(loop ff e2)*(print "]")
	      | loop ff (Appl(_,Until _,TupleExpr(_,[e1,e2]))) =
		  (print "(")*(loop ff e1)*(print " Until ")
		  *(loop ff e2)*(print ")")
	      | loop ff (Appl(_,Releases _,TupleExpr(_,[e1,e2]))) =
		  (print "(")*(loop ff e1)*(print " Releases ")
		  *(loop ff e2)*(print ")")
	      | loop ff (Appl(_,e1,e2)) = 
		  if isInfix(e1) then
		      (case e2 of
			   TupleExpr(_,[x,y]) =>
			       (print "(")*(loop ff x)*(print " ")
			       *(loop ff e1)*(print " ")*(loop ff y)*(print ")")
			 | x => (print "(op ")*(loop ff e1)*(print " ")
			       *(loop ff e2)*(print ")"))
		  else if isUnary(e1) then
		      (loop ff e1)*(print "(")*(loop ff e2)*(print ")")
		  else
		      (print "(")*(loop ff e1)*(print " ")*(loop ff e2)*(print ")")
	      | loop ff (TypedExpr(_,e,tp)) = 
		  (print "((")*(loop ff e)*(print "): ")
		  *(loop ff (ff tp))*(print ")")
	      | loop ff (Fn(_,lst)) =
		  (print "fn ")*(list2str ff "\n  | " lst)
	      | loop ff (ChoicePattern(_,lst)) = (list2str ff " | " lst)
	      | loop ff (ApplPattern(_,p1,p2)) =
		  (print "(")*(loop ff p1)*(print " ")*(loop ff p2)*(print ")")
	      | loop ff (AsPattern(_,p1,p2)) =
		  (print "(")*(loop ff p1)*(print " as ")*(loop ff p2)*(print ")")
	      | loop ff (TypedPattern(_,p,tp)) = 
		  (print "(")*(loop ff p)*(print ": ")*(loop ff (ff tp))*(print ")")
	      | loop ff (TuplePattern(_,lst)) = 
		  (print "(")*(list2str ff ", " lst)*(print ")")
	      | loop ff (RecordPattern(_,lst)) = 
		  (print "{")*(list2str ff ", " lst)*(print "}")
	      | loop ff (Underscore _) = (print "_")
	      | loop ff (ExtractAppl(c,e)) =
		  (print "ExtractAppl(")*(list2str ff ", " [c,e])
		  *(print ")")
	      | loop ff (ExtractTuple(n,e)) =
		  (print("ExtractTuple("^(Int.toString n)^","))
		  *(loop ff e)*(print ")")
	      | loop ff (ExtractRecord(name,e)) =
		  (print "ExtractRecord(")*(loop ff name)*(print ",")
		  *(loop ff e)*(print ")")
	      | loop ff (ExtractIndex e) =
		  (print "ExtractIndex(")*(loop ff e)*(print ")")
	      | loop ff (Group(_,x)) = (print "(")*(loop ff x)*(print ")")
	      | loop ff  Fake = (print "<Fake>")
            and ptPrintModuleSig ff (ModuleSig{statparams=stat, dynParam=dyn}) =
		(print "ModuleSig{statparams=")
		*(case stat of
		      NONE => print "[]"
		    | SOME lst => (print "[")*(list2str ff "," lst)*(print "]"))
		*(print ",\n   dynParam=")
		*(case dyn of
		      NONE => print "<none>"
		    | SOME x => loop ff x)
		*(print "}")
	in (loop (fn x=>x), ptPrintModuleSig (fn x=>x)) end
    in
	fun ptPrintDebugCommon (print, concat) x = (#1 (ptPrintDebugCommon' (print, concat))) x
	fun ModuleSigPrintDebug (print, concat) x = (#2 (ptPrintDebugCommon' (print, concat))) x
    end


    val pt2strDebug = ptPrintDebugCommon(fn x=>Str x, fn (x,y) => Conc(x,y))
    val ModuleSig2StrDebug = ModuleSigPrintDebug(fn x=>Str x, fn (x,y) => Conc(x,y))

    fun pt2stringDebug tree = Str2string(pt2strDebug tree)
    fun ModuleSig2stringDebug s = Str2string(ModuleSig2StrDebug s)

    fun ptPrintDebug printFun tree = ptPrintDebugCommon(printFun, fn((),())=>()) tree

    (* Create a string from the list of objects separated with sep *)
    fun ptlist2str sep [] = ""
      | ptlist2str sep [hd] = pt2string hd
      | ptlist2str sep (hd::tl) = (pt2string hd)^sep^(ptlist2str sep tl)

    fun ptlist2strDebug sep [] = ""
      | ptlist2strDebug sep [hd] = pt2stringDebug hd
      | ptlist2strDebug sep (hd::tl) =
	  (pt2stringDebug hd)^sep^(ptlist2strDebug sep tl)

    fun pos(Program(p,_)) = p
      | pos(Include(p,_)) = p
      | pos(Included(p,_,_)) = p
      | pos(Quote(p,_)) = p
      | pos(Id(p,_)) = p
      | pos(Tvar(p,_)) = p
      | pos(Ellipsis p) = p
      | pos(Module(p,_)) = p
      | pos(Theorem(p,_,_)) = p
      | pos(PType(p,_)) = p
      | pos(Symmetric(p,_)) = p
      | pos(Finite(p,_)) = p
      | pos(BeginEnd(p,_)) = p
      | pos(Sync2(p,_,_)) = p
      | pos(Async2(p,_,_)) = p
      | pos(Sync(p,_,_)) = p
      | pos(Async(p,_,_)) = p
      | pos(TypeDecl(p,_,_,_)) = p
      | pos(DatatypeDecl(p,_,_,_)) = p
      | pos(TypeInst(p,_,_)) = p
      | pos(Var(p,_)) = p
      | pos(StateVarDecl(p,_)) = p
      | pos(Val(p,_,_)) = p
      | pos(FunCases(p,_)) = p
      | pos(FunCase(p,_,_,_,_)) = p
      | pos(Open(p,_)) = p
      | pos(Export(p,_)) = p
      | pos(Models(p,_,_)) = p
      | pos(Refines(p,_,_)) = p
      | pos(LabeledAsst(p,_,_)) = p
      | pos(Nop p) = p
      | pos(NormalAsst(p,_,_)) = p
      | pos(InitAsst(p,_,_)) = p
      | pos(NextAsst(p,_,_)) = p
      | pos(IfAsst(p,_,_)) = p
      | pos(CondAsst(p,_,_)) = p
      | pos(LetAsst(p,_,_)) = p
      | pos(CaseAsst(p,_,_)) = p
      | pos(ChoiceAsst(p,_,_)) = p
      | pos(ChooseAsst(p,_,_)) = p
      | pos(ForeachAsst(p,_,_)) = p
      | pos(IfExpr(p,_,_)) = p
      | pos(CondExpr(p,_,_)) = p
      | pos(LetExpr(p,_,_)) = p
      | pos(CaseExpr(p,_,_)) = p
      | pos(ChoiceExpr(p,_,_)) = p
      | pos(ChooseExpr(p,_,_)) = p
      | pos(WithExpr(p,_,_)) = p
      | pos(WithAsst(p,_,_)) = p
      | pos(RecordExpr(p,_)) = p
      | pos(TupleExpr(p,_)) = p
      | pos(NondetExpr(p,_)) = p
      | pos(RecordAsst(p,_,_)) = p
      | pos(ModuleInst(p,_,_)) = p
      | pos(Dot(p,_,_)) = p
      | pos(Type(p,_)) = p
      | pos(Datatype(p,_)) = p
      | pos(And(p)) = p
      | pos(Or(p)) = p
      | pos(Implies(p)) = p
      | pos(Iff(p)) = p
      | pos(Eq(p)) = p
      | pos(NotEq(p)) = p
      | pos(Not(p)) = p
      | pos(Forall(p,_,_)) = p
      | pos(Exists(p,_,_)) = p
      | pos(Ag(p)) = p
      | pos(Af(p)) = p
      | pos(Eg(p)) = p
      | pos(Ef(p)) = p
      | pos(Ax(p)) = p
      | pos(Ex(p)) = p
      | pos(Au(p)) = p
      | pos(Eu(p)) = p
      | pos(Ar(p)) = p
      | pos(Er(p)) = p
      | pos(Globally p) = p
      | pos(Eventually p) = p
      | pos(Until p) = p
      | pos(Releases p) = p
      | pos(NextTime p) = p
      | pos(Mu(p,_,_)) = p
      | pos(Nu(p,_,_)) = p
      | pos(Lt(p)) = p
      | pos(Gt(p)) = p
      | pos(Le(p)) = p
      | pos(Ge(p)) = p
      | pos(Plus(p)) = p
      | pos(Minus(p)) = p
      | pos(UMinus(p)) = p
      | pos(Times(p)) = p
      | pos(Div(p)) = p
      | pos(Mod(p)) = p
      | pos(Number(p,_)) = p
      | pos(BoolType p) = p
      | pos(NatType p) = p
      | pos(IntType p) = p
      | pos(RangeType(p,_,_)) = p
      | pos(FunType(p,_,_)) = p
      | pos(TupleType(p,_)) = p
      | pos(ArrayType(p,_,_)) = p
      | pos(RecordType(p,_)) = p
      | pos(EnumType(p,_)) = p
      | pos(Of(p,_,_)) = p
      | pos(TypedVars(p,_,_)) = p
      | pos(False p) = p
      | pos(True p) = p
      | pos(Min p) = p
      | pos(Max p) = p
      | pos(Undefined p) = p
      | pos(Anyvalue p) = p
      | pos(Self p) = p
      | pos(Appl(p,_,_)) = p
      | pos(TypedExpr(p,_,_)) = p
      | pos(Fn(p,_)) = p
      | pos(ChoicePattern(p,_)) = p
      | pos(ApplPattern(p,_,_)) = p
      | pos(AsPattern(p,_,_)) = p
      | pos(TypedPattern(p,_,_)) = p
      | pos(TuplePattern(p,_)) = p
      | pos(RecordPattern(p,_)) = p
      | pos(Underscore p) = p
      | pos(Group(p,_)) = p
      | pos x = dp (* Return an "unknown" position *)

(*  raise SympParseInternalError("pos: Found illegal value: "^(pt2string x)) *)

    fun pos2string (l,c) = (Int.toString l)^"."^(Int.toString c)

    (* Equality of two parse trees upto the position field.
     * If noParent=true, then don't recurse into parent closures either,
     * consider them irrelevant for equality. *)
    fun ptEqCommon noParent (t1,t2) =
	let fun looplist(hd1::tl1,hd2::tl2) = 
	           (loop(hd1,hd2)) andalso (looplist(tl1,tl2))
	      | looplist([],[]) = true
	      | looplist _ = false
	    and loopOpt(NONE,NONE) = true
	      | loopOpt(SOME x, SOME y) = loop(x,y)
	      | loopOpt _ = false
	    and looplistOpt(NONE,NONE) = true
	      | looplistOpt(SOME x, SOME y) = looplist(x,y)
	      | looplistOpt _ = false
	    and loop(Program(_,lst1),Program(_,lst2)) = looplist(lst1,lst2)
	      | loop(Include(_,s1),Include(_,s2)) = (s1 = s2)
	      | loop(Included(_,s1,t1),Included(_,s2,t2)) =
		   (s1 = s2) andalso (loop(t1,t2))
	      | loop(Quote(_,s1),Quote(_,s2)) = (s1 = s2)
	      | loop(SharedRef x,SharedRef y) = (x=y)
	      | loop(Id(_,s1),Id(_,s2)) = (s1 = s2)
	      | loop(Uid s1,Uid s2) = (s1 = s2)
	      | loop(Tvar(_,s1),Tvar(_,s2)) = (s1 = s2)
	      | loop(TvarN n1,TvarN n2) = (n1 = n2)
	      | loop(TypeInst(_,lst1,x1), TypeInst(_,lst2,x2)) =
		   looplist(lst1,lst2) andalso loop(x1,x2)
	      | loop(Ellipsis _,Ellipsis _) = true
	      | loop(TopClosure lst1, TopClosure lst2) = looplist(lst1,lst2)
	      | loop(Module(_,{name=name1,statparams=s1,dynparams=d1,
			       body=b1}),
		     Module(_,{name=name2,statparams=s2,dynparams=d2,
			       body=b2})) =
		    (loop(name1,name2))
		    andalso (looplistOpt(s1,s2))
		    andalso (loopOpt(d1,d2))
		    andalso (loop(b1,b2))
	      | loop(ModuleClosure{name=n1,uname=un1,Sig=s1,def=d1,
				   closure=c1,parent=p1},
		     ModuleClosure{name=n2,uname=un2,Sig=s2,def=d2,
				   closure=c2,parent=p2}) =
		    (loopOpt(n1,n2))
		    andalso (loop(un1,un2))
		    andalso (loopSig(s1,s2))
		    andalso (loopOpt(d1,d2))
		    andalso (looplist(c1,c2))
		    andalso (noParent orelse loop(p1,p2))
	      | loop(FunClosure{name=n1,formals=f1,parent=p1,body=b1},
		     FunClosure{name=n2,formals=f2,parent=p2,body=b2}) =
		    (loopOpt(n1,n2))
		    andalso (loop(f1,f2))
		    andalso (loop(b1,b2))
		    andalso (noParent orelse (loop(p1,p2)))
	      | loop(RecurFun x1, RecurFun x2) = loop(x1,x2)
	      | loop(LetClosure{locals=lst1,parent=p1,body=b1},
		     LetClosure{locals=lst2,parent=p2,body=b2}) =
		    looplist(lst1,lst2) andalso loop(b1,b2) 
		    andalso (noParent orelse loop(p1,p2))
	      | loop(LetAsstClosure{locals=lst1,parent=p1,body=b1},
		     LetAsstClosure{locals=lst2,parent=p2,body=b2}) =
		    looplist(lst1,lst2) andalso looplist(b1,b2)
		    andalso (noParent orelse loop(p1,p2))
	      (* This one might be depricated soon *)
	      | loop(TypeClosure{name=n1,uname=un1,params=lst1,def=d1,
				 recursive=r1,parent=p1},
		     TypeClosure{name=n2,uname=un2,params=lst2,def=d2,
				 recursive=r2,parent=p2}) =
		    loop(n1,n2) andalso loop(un1,un2) andalso r1 = r2
		    andalso looplist(lst1,lst2) andalso loop(d1,d2)
		    andalso (noParent orelse loop(p1,p2))
	      | loop(ChoiceClosure{pattern=pat1,uname=u1,
				   names=n1,body=b1,parent=p1},
		     ChoiceClosure{pattern=pat2,uname=u2,
				   names=n2,body=b2,parent=p2}) =
		    loop(pat1,pat2) andalso looplist(n1,n2)
		    andalso loop(u1,u2) andalso loop(b1,b2)
		    andalso (noParent orelse loop(p1,p2))
	      | loop(ChoiceAsstClosure{pattern=pat1,uname=u1,names=n1,
				       body=b1,parent=p1},
		     ChoiceAsstClosure{pattern=pat2,uname=u2,names=n2,
				       body=b2,parent=p2}) =
		    loop(pat1,pat2) andalso looplist(n1,n2)
		    andalso loop(u1,u2) andalso looplist(b1,b2)
		    andalso (noParent orelse loop(p1,p2))
	      | loop(ChooseAsstClosure{names=n1,choices=a1,parent=p1},
		     ChooseAsstClosure{names=n2,choices=a2,parent=p2}) =
		     looplistOpt(n1,n2) andalso looplist(a1,a2)
		     andalso (noParent orelse loop(p1,p2))
	      | loop(ForeachAsstClosure{names=n1, assts=a1, parent=p1},
		     ForeachAsstClosure{names=n2, assts=a2, parent=p2}) =
		     looplist(n1,n2) andalso looplist(a1,a2)
		     andalso (noParent orelse loop(p1,p2))
	      | loop(ChooseClosure{pattern=pat1,Type=tp1,names=n1,body=b1,parent=p1},
		     ChooseClosure{pattern=pat2,Type=tp2,names=n2,body=b2,parent=p2}) =
		    loop(pat1,pat2) andalso looplist(n1,n2)
		    andalso loop(tp1,tp2) andalso loop(b1,b2)
		    andalso (noParent orelse loop(p1,p2))
	      | loop(ForallClosure{names=n1,body=b1,parent=p1},
		     ForallClosure{names=n2,body=b2,parent=p2}) =
		    looplist(n1,n2) andalso loop(b1,b2)
		    andalso (noParent orelse loop(p1,p2))
	      | loop(ExistsClosure{names=n1,body=b1,parent=p1},
		     ExistsClosure{names=n2,body=b2,parent=p2}) =
		    looplist(n1,n2) andalso loop(b1,b2)
		    andalso (noParent orelse loop(p1,p2))
	      | loop(MuClosure{name=n1, body=b1, parent=p1},
		     MuClosure{name=n2, body=b2, parent=p2}) =
		    loop(n1,n2) andalso loop(b1,b2) andalso (noParent orelse loop(p1,p2))
	      | loop(NuClosure{name=n1, body=b1, parent=p1},
		     NuClosure{name=n2, body=b2, parent=p2}) =
		    loop(n1,n2) andalso loop(b1,b2) andalso (noParent orelse loop(p1,p2))
	      | loop(QuantifiedVar{name=n1,uname=un1,Type=tp1},
		     QuantifiedVar{name=n2,uname=un2,Type=tp2}) =
		    loop(n1,n2) andalso loop(un1,un2) andalso loop(tp1,tp2)
	      | loop(SyncClosure{names=n1,body=b1,parent=p1},
		     SyncClosure{names=n2,body=b2,parent=p2}) =
		    looplist(n1,n2) andalso loop(b1,b2)
		    andalso (noParent orelse loop(p1,p2))
	      | loop(AsyncClosure{names=n1,body=b1,parent=p1},
		     AsyncClosure{names=n2,body=b2,parent=p2}) =
		    looplist(n1,n2) andalso loop(b1,b2)
		    andalso (noParent orelse loop(p1,p2))
	      | loop(SequentClosure{names=n1,parent=p1},
		     SequentClosure{names=n2,parent=p2}) =
		    looplist(n1,n2) andalso (noParent orelse loop(p1,p2))
	      | loop(Builtin{name=n1,Type=tp1},Builtin{name=n2,Type=tp2}) =
		    loop(n1,n2) andalso loop(tp1,tp2)
	      | loop(StateVar{name=n1,uname=un1,Type=tp1,id=id1},
		     StateVar{name=n2,uname=un2,Type=tp2,id=id2}) =
		    (loop(n1,n2)) andalso loop(un1,un2) andalso (loop(tp1,tp2))
		    andalso (loop(id1,id2))
	      | loop(SkolemConst{name=n1,uname=un1,Type=tp1},
		     SkolemConst{name=n2,uname=un2,Type=tp2}) =
		    (loop(n1,n2)) andalso loop(un1,un2) andalso (loop(tp1,tp2))
	      | loop(AbstractConst{name=n1,uname=un1,Type=tp1},
		     AbstractConst{name=n2,uname=un2,Type=tp2}) =
		    (loop(n1,n2)) andalso loop(un1,un2) andalso (loop(tp1,tp2))
	      | loop(Object{name=n1,uname=un1,Type=t1,def=d1},
		     Object{name=n2,uname=un2,Type=t2,def=d2}) =
		    (loop(n1,n2)) andalso(loop(un1,un2)) andalso (loop(t1,t2))
		    andalso (loop(d1,d2))
	      | loop(ObjectInst{obj=obj1,...},
		     ObjectInst{obj=obj2,...}) = loop(obj1,obj2)
	      | loop(PatternFormal{name=n1,uname=un1,Type=t1,...},
		     PatternFormal{name=n2,uname=un2,Type=t2,...}) =
		    (loop(n1,n2)) andalso (loop(un1,un2)) andalso (loop(t1,t2))
	      | loop(DynPatternFormal{name=n1,uname=un1,Type=t1,value=v1,...},
		     DynPatternFormal{name=n2,uname=un2,Type=t2,value=v2,...}) =
		    (loop(n1,n2)) andalso (loop(un1,un2)) andalso (loop(t1,t2))
		    andalso (loopOpt(v1,v2))
	      | loop(TypeConstr{name=n1,uname=un1,Type=t1},
		     TypeConstr{name=n2,uname=un2,Type=t2}) =
		    (loop(n1,n2)) andalso (loop(un1,un2)) andalso (loop(t1,t2))
	      | loop(RecordField{name=n1,Type=f1},
		     RecordField{name=n2,Type=f2}) =
		    (loop(n1,n2))  andalso (loop(f1,f2))
	      | loop(StaticFormalConst{name=n1,uname=un1,Type=d1,value=v1},
		     StaticFormalConst{name=n2,uname=un2,Type=d2,value=v2}) =
		    (loop(n1,n2)) andalso (loop(un1,un2)) andalso (loop(d1,d2))
		    andalso loopOpt(v1,v2)
	      | loop(StaticFormalType{name=n1,uname=un1,value=v1},
		     StaticFormalType{name=n2,uname=un2,value=v2}) =
		    (loop(n1,n2)) andalso (loop(un1,un2))
		    andalso loopOpt(v1,v2)
	      | loop(Theorem(_,x1,y1),Theorem(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(PType(_,x1),PType(_,x2)) = loop(x1,x2)
	      | loop(Symmetric(_,x1),Symmetric(_,x2)) = loop(x1,x2)
	      | loop(Finite(_,x1),Finite(_,x2)) = loop(x1,x2)
	      | loop(BeginEnd(_,lst1),BeginEnd(_,lst2)) = looplist(lst1,lst2)
	      | loop(Sync2(_,x1,y1),Sync2(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(Async2(_,x1,y1),Async2(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(Sync(_,x1,y1),Sync(_,x2,y2)) =
		    (looplist(x1,x2)) andalso (loop(y1,y2))
	      | loop(Async(_,x1,y1),Async(_,x2,y2)) =
		    (looplist(x1,x2)) andalso (loop(y1,y2))
	      | loop(TypeDecl(_,x1,lst1,y1),TypeDecl(_,x2,lst2,y2)) =
		    (loop(x1,x2)) andalso (looplist(lst1,lst2))
		    andalso (loop(y1,y2))
	      | loop(DatatypeDecl(_,x1,lst1,y1),DatatypeDecl(_,x2,lst2,y2)) =
		    (loop(x1,x2)) andalso (looplist(lst1,lst2))
		    andalso (loop(y1,y2))
	      | loop(Var(_,x1),Var(_,x2)) = looplist(x1,x2)
	      | loop(StateVarDecl(_,x1),StateVarDecl(_,x2)) = looplist(x1,x2)
	      | loop(Val(_,x1,y1),Val(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(FunCases(_,x1),FunCases(_,x2)) = looplist(x1,x2)
	      | loop(FunCase(_,x1,lst1,o1,y1),FunCase(_,x2,lst2,o2,y2)) =
		    (loop(x1,x2)) andalso (looplist(lst1,lst2))
		    andalso (loopOpt(o1,o2))
		    andalso (loop(y1,y2))
	      | loop(Open(_,x1),Open(_,x2)) = loop(x1,x2)
	      | loop(Export(_,lst1),Export(_,lst2)) = looplist(lst1,lst2)
	      | loop(Models(_,x1,y1),Models(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(Refines(_,x1,y1),Refines(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(LabeledAsst(_, l1, a1), LabeledAsst(_, l2, a2)) =
		    (loop(l1,l2)) andalso loop(a1,a2)
	      | loop(Nop _,Nop _) = true
	      | loop(NormalAsst(_,x1,y1),NormalAsst(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(InitAsst(_,x1,y1),InitAsst(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(NextAsst(_,x1,y1),NextAsst(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(Next x1, Next x2) = loop(x1,x2)
	      | loop(IfAsst(_,x1,y1),IfAsst(_,x2,y2)) =
		    (looplist(x1,x2)) andalso (looplist(y1,y2))
	      | loop(CondAsst(_,x1,y1),CondAsst(_,x2,y2)) =
		    (loop(x1,x2)) andalso (looplist(y1,y2))
	      | loop(LetAsst(_,x1,y1),LetAsst(_,x2,y2)) =
		    (looplist(x1,x2)) andalso (looplist(y1,y2))
	      | loop(CaseAsst(_,x1,y1),CaseAsst(_,x2,y2)) =
		    (loop(x1,x2)) andalso (looplist(y1,y2))
	      | loop(ChoiceAsst(_,x1,y1),ChoiceAsst(_,x2,y2)) =
		    (loop(x1,x2)) andalso (looplist(y1,y2))
	      | loop(ChooseAsst(_,n1,lst1),ChooseAsst(_,n2,lst2)) = 
		    looplistOpt(n1,n2) andalso looplist(lst1,lst2)
	      | loop(ForeachAsst(_, n1, l1), ForeachAsst(_, n2, l2)) =
		    looplist(n1,n2) andalso looplist(l1,l2)
	      | loop(IfExpr(_,lst1,x1),IfExpr(_,lst2,x2)) =
		    (looplist(lst1,lst2)) andalso (loop(x1,x2))
	      | loop(CondExpr(_,x1,y1),CondExpr(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(LetExpr(_,lst1,x1),LetExpr(_,lst2,x2)) =
		    (looplist(lst1,lst2)) andalso (loop(x1,x2))
	      | loop(CaseExpr(_,x1,lst1),CaseExpr(_,x2,lst2)) =
		    (looplist(lst1,lst2)) andalso (loop(x1,x2))
	      | loop(ChoiceExpr(_,x1,y1),ChoiceExpr(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(ChooseExpr(_,x1,y1),ChooseExpr(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(WithExpr(_,x1,lst1),WithExpr(_,x2,lst2)) =
		    (loop(x1,x2)) andalso (looplist(lst1,lst2))
	      | loop(WithAsst(_,x1,y1),WithAsst(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(RecordExpr(_,lst1),RecordExpr(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(TupleExpr(_,lst1),TupleExpr(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(NondetExpr(_,lst1),NondetExpr(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(RecordAsst(_,x1,y1),RecordAsst(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(ModuleInst(_,x1,lst1),ModuleInst(_,x2,lst2)) =
		    (loop(x1,x2)) andalso (looplist(lst1,lst2))
	      | loop(Dot(_,x1,y1),Dot(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(Type(_,x1),Type(_,x2)) = (loop(x1,x2))
	      | loop(Datatype(_,x1),Datatype(_,x2)) = (loop(x1,x2))
	      | loop(And _,And _) = true
	      | loop(Or _,Or _) = true
	      | loop(Implies _,Implies _) = true
	      | loop(Arrow _, Arrow _) = true
	      | loop(Darrow _, Darrow _) = true
	      | loop(Iff _,Iff _) = true
	      | loop(Eq _,Eq _) = true
	      | loop(NotEq _,NotEq _) = true
	      | loop(Not _,Not _) = true
	      | loop(Forall(_,x1,y1),Forall(_,x2,y2)) =
		    (looplist(x1,x2)) andalso (loop(y1,y2))
	      | loop(Exists(_,x1,y1),Exists(_,x2,y2)) =
		    (looplist(x1,x2)) andalso (loop(y1,y2))
	      | loop(Ag _,Ag _) = true
	      | loop(Af _,Af _) = true
	      | loop(Eg _,Eg _) = true
	      | loop(Ef _,Ef _) = true
	      | loop(Ax _,Ax _) = true
	      | loop(Ex _,Ex _) = true
	      | loop(Au _,Au _) = true
	      | loop(Eu _,Eu _) = true
	      | loop(Ar _,Ar _) = true
	      | loop(Er _,Er _) = true
              | loop(Globally _, Globally _) = true
	      | loop(Eventually _, Eventually _) = true
	      | loop(Until _, Until _) = true
	      | loop(Releases _, Releases _) = true
	      | loop(NextTime _, NextTime _) = true
	      | loop(Mu(_, v1, e1), Mu(_, v2, e2)) = loop(v1,v2) andalso loop(e1,e2)
	      | loop(Nu(_, v1, e1), Nu(_, v2, e2)) = loop(v1,v2) andalso loop(e1,e2)
	      | loop(Lt _,Lt _) = true
	      | loop(Gt _,Gt _) = true
	      | loop(Le _,Le _) = true
	      | loop(Ge _,Ge _) = true
	      | loop(Plus _,Plus _) = true
	      | loop(Minus _,Minus _) = true
	      | loop(UMinus _,UMinus _) = true
	      | loop(Times _,Times _) = true
	      | loop(Div _,Div _) = true
	      | loop(Mod _,Mod _) = true
	      | loop(Number(_,n1),Number(_,n2)) = (n1 = n2)
	      | loop(BoolType _,BoolType _) = true
	      | loop(BoolType2,BoolType2) = true
	      | loop(NatType _,NatType _) = true
	      | loop(IntType _,IntType _) = true
	      | loop(RangeType(_,x1,y1),RangeType(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(FunType(_,x1,y1),FunType(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(TupleType(_,lst1),TupleType(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(ArrayType(_,x1,y1),ArrayType(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(RecordType(_,lst1),RecordType(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(EnumType(_,lst1),EnumType(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(AbstractType(lst1),AbstractType(lst2)) =
		    (looplist(lst1,lst2))
	      | loop(Of(_,x1,y1),Of(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(TypedVars(_,lst1,o1),TypedVars(_,lst2,o2)) =
		    (looplist(lst1,lst2)) andalso
		    (loopOpt(o1,o2))
	      | loop(False _,False _) = true
	      | loop(True _,True _) = true
	      | loop(False2,False2) = true
	      | loop(True2,True2) = true
	      | loop(Min _,Min _) = true
	      | loop(Max _,Max _) = true
	      | loop(Undefined _,Undefined _) = true
	      | loop(Anyvalue _,Anyvalue _) = true
	      | loop(OtherValue,OtherValue) = true
	      | loop(Self _,Self _) = true
	      | loop(Appl(_,x1,y1),Appl(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(TypedExpr(_,x1,y1),TypedExpr(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(Fn(_,lst1),Fn(_,lst2)) = (looplist(lst1,lst2))
	      | loop(ChoicePattern(_,lst1),ChoicePattern(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(ApplPattern(_,x1,y1),ApplPattern(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(AsPattern(_,x1,y1),AsPattern(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(TypedPattern(_,x1,y1),TypedPattern(_,x2,y2)) =
		    (loop(x1,x2)) andalso (loop(y1,y2))
	      | loop(TuplePattern(_,lst1),TuplePattern(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(RecordPattern(_,lst1),RecordPattern(_,lst2)) =
		    (looplist(lst1,lst2))
	      | loop(Underscore _,Underscore _) = true
	      | loop(ExtractAppl(y1,z1),ExtractAppl(y2,z2)) =
		    loop(y1,y2) andalso loop(z1,z2)
	      | loop(ExtractTuple(n1,x1),ExtractTuple(n2,x2)) =
		    n1 = n2 andalso loop(x1,x2)
	      | loop(ExtractRecord(x1,y1),ExtractRecord(x2,y2)) =
		    loop(x1,x2) andalso loop(y1,y2)
	      | loop(ExtractIndex x1, ExtractIndex x2) = loop(x1,x2)
	      | loop(Group(_,x1),Group(_,x2)) = loop(x1,x2)
	      | loop(Fake,Fake) = true
	      | loop _ = false
	    and loopSig(ModuleSig{statparams=s1,dynParam=d1},
			ModuleSig{statparams=s2,dynParam=d2}) =
		    (case (s1,s2) of
			 (NONE,NONE) => true
		       | (SOME(x1),SOME(x2)) => looplist(x1,x2)
		       | _ => false)
		   andalso 
		    (case (d1,d2) of
			 (NONE, NONE) => true
		       | (SOME x1, SOME x2) => loop(x1,x2)
		       | _ => false)

	in loop(t1,t2)
	end

    (* Make the default equality NOT to compare parent closures.  It
       doesn't seem to be necessary, and is too expensive. *)
    val ptEq = ptEqCommon true

    (* Don't compare parent closures, consider objects equivalent if their
     * local definitions are the same. *)
    val ptEqLoose = ptEqCommon true

(* Takes a function `ff' that transforms the parse tree, and applies
   it to every subexpression of `expr' of ParseTree type.  This
   function may call `ptTransform' on its argument, and thus,
   implement the recursive descend for the entire tree without having
   to mention all of the cases.

   Parent closures are not changed.  If you need to recurse into
   parent closures, consider those cases in `ff'. 

   Also, since this is a very critical function, we explicitly
   enumerate all cases in the ParseTree, to let the ML typechecker
   catch anything we've missed. *)

    fun ptTransform ff expr =
	let val fflist = List.map ff
	    val ffOpt = Option.map ff
	    fun loop (Program(p,lst)) = Program(p,fflist lst)
	      | loop (x as Include _) = x
	      | loop (Included(p,s,x)) = Included(p,s,ff x)
	      | loop (x as Quote _) = x
	      (* Open the reference *)
	      | loop (SharedRef x) = ff(!x)
	      | loop (x as Id _) = x
	      | loop (x as Uid _) = x
	      | loop (x as Tvar _) = x
	      | loop (x as TvarN _) = x
	      | loop (TypeInst(p,lst,x)) = TypeInst(p,fflist lst, ff x)
	      | loop (x as Ellipsis _) = x
	      | loop (TopClosure lst) = TopClosure(fflist lst)
	      | loop (Module(p,{name=name,
				statparams=stats,
				dynparams=dyn,
				body=body})) =
		  Module(p,{name=name,
			    statparams=Option.map fflist stats,
			    dynparams=Option.map ff dyn,
			    body=ff body})
	      | loop (ModuleClosure{name=name,
				    uname=uname,
				    Sig=msig,
				    closure=cl,
				    def=def,
				    parent=p}) =
		  ModuleClosure{name=name,
				uname=uname,
				Sig=loopSig msig,
				closure=fflist cl,
				def=Option.map ff def,
				parent=p}
	      | loop (FunClosure{name=name,
				 formals=arg,
				 parent=parent,
				 body=body}) =
		  FunClosure{name=name,
			     formals=ff arg,
			     parent=parent,
			     body=ff body}
	      | loop (RecurFun x) = RecurFun(ff x)
	      | loop (LetClosure{locals=locals,
				 parent=parent,
				 body=body}) =
		  LetClosure{locals=fflist locals,
			     parent=parent,
			     body=ff body}
	      | loop (LetAsstClosure{locals=locals,
				     parent=parent,
				     body=body}) =
		  LetAsstClosure{locals=fflist locals,
				 parent=parent,
				 body=fflist body}
	      | loop (TypeClosure{name=name,
				  uname=uname,
				  params=params,
				  def=def,
				  recursive=r,
				  parent=parent}) =
		  TypeClosure{name=name,
			      uname=uname,
			      params=params,
			      def=ff def,
			      recursive=r,
			      parent=parent}
	      | loop (ChoiceClosure{pattern=pat,
				    uname=uname,
				    names=names,
				    body=body,
				    parent=parent}) =
		  ChoiceClosure{pattern=ff pat,
				uname=uname,
				names=fflist names,
				body=ff body,
				parent=parent}
	      | loop (ChoiceAsstClosure{pattern=pat,
					uname=uname,
					names=names,
					body=body,
					parent=parent}) =
		  ChoiceAsstClosure{pattern=ff pat,
				    uname=uname,
				    names=fflist names,
				    body=fflist body,
				    parent=parent}
	      | loop (ChooseAsstClosure{names=namesOpt,
					choices=assts,
					parent=parent}) =
		      ChooseAsstClosure{names=Option.map fflist namesOpt,
					choices=fflist assts,
					parent=parent}
	      | loop (ForeachAsstClosure{names=names,
					 assts=assts,
					 parent=parent}) =
		      ForeachAsstClosure{names=fflist names,
					 assts=fflist assts,
					 parent=parent}
	      | loop (ChooseClosure{pattern=pat,
				    Type=tp,
				    names=names,
				    body=body,
				    parent=parent}) =
		  ChooseClosure{pattern=ff pat,
				Type=ff tp,
				names=fflist names,
				body=ff body,
				parent=parent}
	      | loop (ForallClosure{names=names,
				    body=body,
				    parent=parent}) =
		  ForallClosure{names=fflist names,
				body=ff body,
				parent=parent}
	      | loop (ExistsClosure{names=names,
				    body=body,
				    parent=parent}) =
		  ExistsClosure{names=fflist names,
				body=ff body,
				parent=parent}
	      | loop (MuClosure{name=name,
				body=body,
				parent=parent}) =
		  MuClosure{name=ff name,
			    body=ff body,
			    parent=parent}
	      | loop (NuClosure{name=name,
				body=body,
				parent=parent}) =
		  NuClosure{name=ff name,
			    body=ff body,
			    parent=parent}
	      | loop (QuantifiedVar{name=n,uname=un,Type=tp}) =
		  QuantifiedVar{name=n, uname=un, Type=ff tp}
	      | loop (SyncClosure{names=names,
				  body=body,
				  parent=parent}) =
		  SyncClosure{names=fflist names,
			      body=ff body,
			      parent=parent}
	      | loop (AsyncClosure{names=names,
				   body=body,
				   parent=parent}) =
		  AsyncClosure{names=fflist names,
			       body=ff body,
			       parent=parent}
	      | loop (SequentClosure{names=names,
				     parent=parent}) =
		  SequentClosure{names=fflist names,
				 parent=parent}
	      | loop (Builtin{name=name, Type=tp}) =
		  Builtin{name=name, Type=ff tp}
	      | loop (StateVar{name=n, uname=un, Type=tp, id=id}) =
		  StateVar{name=n, uname=un, Type=ff tp, id=ff id}
	      | loop (SkolemConst{name=n, uname=un, Type=tp}) =
		  SkolemConst{name=n, uname=un, Type=ff tp}
	      | loop (AbstractConst{name=n, uname=un, Type=tp}) =
		  AbstractConst{name=n, uname=un, Type=ff tp}
	      | loop (Object{name=name,
			     uname=uname,
			     Type=tp,
			     def=def}) =
		  Object{name=name,
			 uname=uname,
			 Type=ff tp,
			 def=ff def}
	      | loop (ObjectInst{obj=obj,subst=f}) =
		  ObjectInst{obj=ff obj, subst=f}
	      | loop (PatternFormal{name=name,
				    uname=uname,
				    Type=tp,
				    extract=extract}) =
		  PatternFormal{name=name,
				uname=uname,
				Type=ff tp,
				extract=extract}
	      | loop (DynPatternFormal{name=name,
				       uname=uname,
				       Type=tp,
				       value=v,
				       extract=extract}) =
		  DynPatternFormal{name=name,
				   uname=uname,
				   Type=ff tp,
				   value=Option.map ff v,
				   extract=extract}
	      | loop (TypeConstr{name=name,
				 uname=uname,
				 Type=tp}) =
		  TypeConstr{name=name, 
			     uname=uname,
			     Type=ff tp}
	      | loop (RecordField{name=name,
				  Type=tp}) =
		  RecordField{name=name,
			      Type=ff tp}
	      | loop (StaticFormalConst{name=name,
					uname=uname,
					Type=def,
					value=v}) =
		  StaticFormalConst{name=name,
				    uname=uname,
				    Type=ff def,
				    value=Option.map ff v}
	      | loop (StaticFormalType{name=name,
				       uname=uname,
				       value=v}) =
		  StaticFormalType{name=name,
				   uname=uname,
				   value=Option.map ff v}
	      | loop (Theorem(p,name,def)) = Theorem(p,name,ff def)
	      | loop (PType(p,tp)) = PType(p,ff tp)
	      | loop (Symmetric(p,tp)) = Symmetric(p,ff tp)
	      | loop (Finite(p,tp)) = Finite(p,ff tp)
	      | loop (BeginEnd(p,lst)) = BeginEnd(p,fflist lst)
	      | loop (Sync2(p,m1,m2)) = Sync2(p,ff m1,ff m2)
	      | loop (Async2(p,m1,m2)) = Async2(p,ff m1,ff m2)
	      | loop (Sync(p,lst,m)) = Sync(p,fflist lst,ff m)
	      | loop (Async(p,lst,m)) = Async(p,fflist lst,ff m)
	      | loop (TypeDecl(p,name,parms,def)) =
		  TypeDecl(p,name,parms,ff def)
	      | loop (DatatypeDecl(p,name,parms,def)) =
		  DatatypeDecl(p,name,parms,ff def)
	      | loop (Var(p,lst)) = Var(p,fflist lst)
	      | loop (StateVarDecl(p,lst)) = StateVarDecl(p,fflist lst)
	      | loop (Val(p,name,def)) = Val(p,name,ff def)
	      | loop (FunCases(p,lst)) = FunCases(p,fflist lst)
	      | loop (FunCase(p,name,pats,tp,def)) =
		  FunCase(p,name,fflist pats,
			  (case tp of
			       NONE => NONE
			     | SOME(t) => SOME(ff t)), ff def)
	      | loop (Open(p,m)) = Open(p,ff m)
	      | loop (Export(p,lst)) = Export(p,fflist lst)
	      | loop (Models(p,x,y)) = Models(p,ff x,ff y)
	      | loop (Refines(p,x,y)) = Refines(p,ff x,ff y)
	      | loop (LabeledAsst(p,l,x)) = LabeledAsst(p, l, ff x)
	      | loop (x as Nop _) = x
	      | loop (NormalAsst(p,name,e)) = NormalAsst(p,ff name,ff e)
	      | loop (InitAsst(p,name,e)) = InitAsst(p,ff name,ff e)
	      | loop (NextAsst(p,name,e)) = NextAsst(p,ff name,ff e)
	      | loop (Next e) = Next(ff e)
	      | loop (IfAsst(p,lst1,lst2)) = IfAsst(p,fflist lst1,fflist lst2)
	      | loop (CondAsst(p,x,lst)) = CondAsst(p,ff x,fflist lst)
	      | loop (LetAsst(p,defs,lst)) = LetAsst(p,fflist defs,fflist lst)
	      | loop (CaseAsst(p,e,lst)) = CaseAsst(p,ff e,fflist lst)
	      | loop (ChoiceAsst(p,x,lst)) = ChoiceAsst(p,ff x,fflist lst)
	      | loop (ChooseAsst(p,n,lst)) = ChooseAsst(p, Option.map fflist n, fflist lst)
	      | loop (ForeachAsst(p, n, lst)) = ForeachAsst(p, fflist n, fflist lst)
	      | loop (IfExpr(p,lst,x)) = IfExpr(p,fflist lst,ff x)
	      | loop (CondExpr(p,x,y)) = CondExpr(p,ff x,ff y)
	      | loop (LetExpr(p,defs,e)) = LetExpr(p,fflist defs,ff e)
	      | loop (CaseExpr(p,e,lst)) = CaseExpr(p,ff e,fflist lst)
	      | loop (ChoiceExpr(p,x,y)) = ChoiceExpr(p,ff x,ff y)
	      | loop (ChooseExpr(p,x,y)) = ChooseExpr(p,ff x,ff y)
	      | loop (WithExpr(p,x,lst)) = WithExpr(p,ff x,fflist lst)
	      | loop (WithAsst(p,name,y)) = WithAsst(p,name,ff y)
	      | loop (RecordExpr(p,lst)) = RecordExpr(p,fflist lst)
	      | loop (TupleExpr(p,lst)) = TupleExpr(p,fflist lst)
	      | loop (NondetExpr(p,lst)) = NondetExpr(p,fflist lst)
	      | loop (RecordAsst(p,x,y)) = RecordAsst(p,ff x,ff y)
	      | loop (ModuleInst(p,m,lst)) = ModuleInst(p,ff m,fflist lst)
	      | loop (Dot(p,x,y)) = Dot(p,ff x,ff y)
	      | loop (Type(p,tp)) = Type(p,ff tp)
	      | loop (Datatype(p,tp)) = Datatype(p,tp)
	      | loop (x as And _) = x
	      | loop (x as Or _) = x
	      | loop (x as Implies _) = x
	      | loop (x as Arrow _) = x
	      | loop (x as Darrow _) = x
	      | loop (x as Iff _) = x
	      | loop (x as Eq _) = x
	      | loop (x as NotEq _) = x
	      | loop (x as Not _) = x
	      | loop (Forall(p,lst,x)) = Forall(p,fflist lst,ff x)
	      | loop (Exists(p,lst,x)) = Exists(p,fflist lst,ff x)
	      | loop (x as Ag _) = x
	      | loop (x as Af _) = x
	      | loop (x as Eg _) = x
	      | loop (x as Ef _) = x
	      | loop (x as Ax _) = x
	      | loop (x as Ex _) = x
	      | loop (x as Au _) = x
	      | loop (x as Eu _) = x
	      | loop (x as Ar _) = x
	      | loop (x as Er _) = x
	      | loop (x as Globally _) = x
	      | loop (x as Eventually _) = x
	      | loop (x as Until _) = x
	      | loop (x as Releases _) = x
	      | loop (x as NextTime _) = x
	      | loop (Mu(p,v,e)) = Mu(p, ff v, ff e)
	      | loop (Nu(p,v,e)) = Nu(p, ff v, ff e)
	      | loop (x as Lt _) = x
	      | loop (x as Gt _) = x
	      | loop (x as Le _) = x
	      | loop (x as Ge _) = x
	      | loop (x as Plus _) = x
	      | loop (x as Minus _) = x
	      | loop (x as UMinus _) = x
	      | loop (x as Times _) = x
	      | loop (x as Div _) = x
	      | loop (x as Mod _) = x
	      | loop (x as Number _) = x
	      | loop (x as BoolType _) = x
	      | loop (x as BoolType2) = x
	      | loop (x as NatType _) = x
	      | loop (x as IntType _) = x
	      | loop (RangeType(p,x,y)) = RangeType(p,ff x,ff y)
	      | loop (FunType(p,x,y)) = FunType(p,ff x,ff y)
	      | loop (TupleType(p,x)) = TupleType(p,fflist x)
	      | loop (ArrayType(p,x,y)) = ArrayType(p,ff x,ff y)
	      | loop (RecordType(p,x)) = RecordType(p,fflist x)
	      | loop (EnumType(p,lst)) = EnumType(p,fflist lst)
	      | loop (AbstractType(lst)) = AbstractType(fflist lst)
	      | loop (Of(p,name,tp)) = Of(p,name,ff tp)
	      | loop (TypedVars(p,lst,tp)) = TypedVars(p,fflist lst, Option.map ff tp)
	      | loop (x as False _) = x
	      | loop (x as True _) = x
	      | loop (x as False2) = x
	      | loop (x as True2) = x
	      | loop (x as Min _) = x
	      | loop (x as Max _) = x
	      | loop (x as Undefined _) = x
	      | loop (x as Anyvalue _) = x
	      | loop (x as OtherValue) = x
	      | loop (x as Self _) = x
	      | loop (Appl(p,x,y)) = Appl(p,ff x,ff y)
	      | loop (TypedExpr(p,x,y)) = TypedExpr(p,ff x,ff y)
	      | loop (Fn(p,lst)) = Fn(p,fflist lst)
	      | loop (ChoicePattern(p,lst)) = ChoicePattern(p,fflist lst)
	      | loop (ApplPattern(p,x,y)) = ApplPattern(p,ff x,ff y)
	      | loop (TypedPattern(p,x,y)) = TypedPattern(p,ff x,ff y)
	      | loop (TuplePattern(p,lst)) = TuplePattern(p,fflist lst)
	      | loop (RecordPattern(p,lst)) = RecordPattern(p,fflist lst)
	      | loop (AsPattern(p,x,y)) = AsPattern(p,ff x,ff y)
	      | loop (x as Underscore _) = x
	      | loop (ExtractAppl(y,z)) = ExtractAppl(ff y,ff z)
	      | loop (ExtractTuple(n,x)) = ExtractTuple(n,ff x)
	      | loop (ExtractRecord(x,y)) = ExtractRecord(ff x,ff y)
	      | loop (ExtractIndex x) = ExtractIndex(ff x)
	      | loop (Group(p,x)) = Group(p,ff x)
	      | loop (x as Fake) = x
	    and loopSig (ModuleSig{statparams=stats, dynParam=dyn}) =
		  ModuleSig{statparams=Option.map fflist stats,
			    dynParam=Option.map ff dyn}
	in loop expr
	end

    (* Same as ptTransformParent, only ff takes context and ParseTree
     and updates the parent closures correctly, so that they point to
     the closures of the current tree. 

     If the initial context is "Fake", then the outermost context is
     kept unchanged.  That is, if you want the expression to remain in
     the context it comes from, simply give "Fake" as the current
     context. *)

    fun ptTransformParent ff context expr =
	let fun fflist context lst = List.map (ff context) lst
	    fun pickC(parent,context) =
		  (case context of
		       Fake => parent
		     | _ => context)
	    fun descend c acc [] = acc
	      | descend c acc (d::lst) = descend c ((ff (c acc) d)::acc) lst
	    fun recur c x = ptTransform (ff c) x
	    (* Attention: Module doesn't form a context, only
	       ModuleClosure does, so this case should be taken care
	       externally if needed. *)
	    fun loop context (Module(p,{name=name,
					statparams=stats,
					dynparams=dyn,
					body=body})) =
		let val sp = Option.map(fflist context) stats
		    val dp = Option.map(ff context) dyn
		in Module(p,{name=name,
			     statparams=sp,
			     dynparams=dp,
			     body=ff context body})
		end
	      | loop context (ModuleClosure{name=name,
					    uname=uname,
					    Sig=msig,
					    closure=cl,
					    def=def,
					    parent=p}) =
		let val newdef = Option.map (ff context) def
		    val newSig=loopSig context msig
		    fun c lst = 
			 ModuleClosure{name=name,
				       uname=uname,
				       Sig=newSig,
				       closure=lst,
				       def=newdef,
				       parent=pickC(p,context)}
		    val cl = descend c [] (List.rev cl)
		in (c cl)
		end
	      | loop context (FunClosure{name=name,
				 formals=arg,
				 parent=p,
				 body=body}) =
		let val cxt = pickC(p,context)
		    val arg = ff context arg
		    val b = ff (FunClosure{name=name,
					   formals=arg,
					   parent=cxt,
					   body=Fake}) body
		in FunClosure{name=name,
			      formals=arg,
			      parent=cxt,
			      body=b}
		end
	      | loop context (LetClosure{locals=locals,
				 parent=p,
				 body=body}) =
		let fun c b lst = 
		      LetClosure{locals=lst,
				 parent=pickC(p,context),
				 body=b}
		    val locs = descend (c Fake) [] (List.rev locals)
		in c (ff (c Fake locs) body) locs
		end
	      | loop context (LetAsstClosure{locals=locals,
					     parent=p,
					     body=body}) =
		let fun c b lst = 
		      LetAsstClosure{locals=lst,
				     parent=pickC(p,context),
				     body=b}
		    val locs = descend (c []) [] (List.rev locals)
		    val cxt = c [] locs
		in c (fflist cxt body) locs
		end
	      | loop context (TypeClosure{name=name,
				  uname=uname,
				  params=params,
				  def=def,
				  recursive=r,
				  parent=p}) =
		  TypeClosure{name=name,
			      uname=uname,
			      params=params,
			      def=ff (pickC(p,context)) def,
			      recursive=r,
			      parent=(pickC(p,context))}
	      | loop context (ChoiceClosure{pattern=pat,
					    uname=uname,
					    names=names,
					    body=body,
					    parent=pc}) =
		  let val cxt=pickC(pc,context)
		      val newnames = fflist cxt names
		      fun c b pat =
		        ChoiceClosure{pattern=pat,
				      uname=uname,
				      names=newnames,
				      body=b,
				      parent=cxt}
		      val pat = ff cxt pat
		  in c (ff (c Fake pat) body) pat
		  end
	      | loop context (ChoiceAsstClosure{pattern=pat,
						uname=uname,
						names=names,
						body=body,
						parent=pc}) =
		  let val cxt=pickC(pc,context)
		      val newnames = fflist cxt names
		      fun c b pat =
		        ChoiceAsstClosure{pattern=pat,
					  uname=uname,
					  names=newnames,
					  body=b,
					  parent=cxt}
		      val pat = ff cxt pat
		  in c (fflist (c [] pat) body) pat
		  end
	      | loop context (ChooseAsstClosure{names=namesOpt,
						choices=assts,
						parent=pc}) =
		  let val cxt=pickC(pc,context)
		      val newnames = Option.map(fflist cxt) namesOpt
		      fun c assts = ChooseAsstClosure{names=newnames,
						      choices=assts,
						      parent=cxt}
		  in c (fflist (c []) assts)
		  end
	      | loop context (ForeachAsstClosure{names=names,
						 assts=assts,
						 parent=pc}) =
		  let val cxt=pickC(pc,context)
		      val newnames = fflist cxt names
		      fun c assts = ForeachAsstClosure{names=newnames,
						       assts=assts,
						       parent=cxt}
		  in c (fflist (c []) assts)
		  end
	      | loop context (ChooseClosure{pattern=pat,
					    Type=tp,
					    names=names,
					    body=body,
					    parent=pc}) =
		  let val cxt=pickC(pc,context)
		      val newnames = fflist cxt names
		      val pat = ff cxt pat
		      val tp = ff cxt tp
		      fun c b =
		        ChooseClosure{pattern=pat,
				      Type=tp,
				      names=newnames,
				      body=b,
				      parent=cxt}
		  in c (ff (c Fake) body)
		  end
	      | loop context (ForallClosure{names=names,
				    body=body,
				    parent=pc}) =
		  let val cxt=pickC(pc,context)
		      fun c b n =
		        ForallClosure{names=n,
				      body=b,
				      parent=cxt}
		      val names=fflist cxt names
		  in c (ff (c Fake names) body) names
		  end
	      | loop context (ExistsClosure{names=names,
				    body=body,
				    parent=pc}) =
		  let val cxt=pickC(pc,context)
		      fun c b n =
		        ExistsClosure{names=n,
				      body=b,
				      parent=cxt}
		      val names=fflist cxt names
		  in c (ff (c Fake names) body) names
		  end
	      | loop context (MuClosure{name=name,
				    body=body,
				    parent=pc}) =
		  let val cxt=pickC(pc,context)
		      fun c b n =
			  MuClosure{name=n,
				    body=b,
				    parent=cxt}
		      val name=ff cxt name
		  in c (ff (c Fake name) body) name
		  end
	      | loop context (NuClosure{name=name,
				    body=body,
				    parent=pc}) =
		  let val cxt=pickC(pc,context)
		      fun c b n =
		        NuClosure{name=n,
				  body=b,
				  parent=cxt}
		      val name=ff cxt name
		  in c (ff (c Fake name) body) name
		  end
	      | loop context (StateVar{name=n, uname=un, Type=tp, id=id}) =
		  StateVar{name=n,
			   uname=un,
			   Type=ff context tp,
			   id= ff context id}
	      | loop context (SkolemConst{name=n, uname=un, Type=tp}) =
		  SkolemConst{name=n,
			      uname=un,
			      Type=ff context tp}
	      | loop context (AbstractConst{name=n, uname=un, Type=tp}) =
		  AbstractConst{name=n,
				uname=un,
				Type=ff context tp}
	      | loop context (SyncClosure{names=names,
					  body=body,
					  parent=pc}) =
		  let val cxt=pickC(pc,context)
		      fun c b n =
			  SyncClosure{names=n,
				      body=b,
				      parent=cxt}
		      val names=fflist cxt names
		  in c (ff (c Fake names) body) names
		  end
	      | loop context (AsyncClosure{names=names,
					   body=body,
					   parent=pc}) =
		  let val cxt=pickC(pc,context)
		      fun c b n =
			  AsyncClosure{names=n,
				       body=b,
				       parent=cxt}
		      val names=fflist cxt names
		  in c (ff (c Fake names) body) names
		  end
	      | loop context (SequentClosure{names=names,
					     parent=pc}) =
		  let val cxt=pickC(pc,context)
		      fun c n =
			  SequentClosure{names=n,
					 parent=cxt}
		      val names=fflist cxt names
		  in c names
		  end
	      | loop context (TopClosure lst) =
		  let fun c lst = TopClosure lst
		  in TopClosure(descend c [] (List.rev lst))
		  end
	      | loop context x = recur context x
            and loopSig context (ModuleSig{statparams=stat, dynParam=dyn}) =
		let fun c lst = LetClosure{locals=lst, body=Fake, parent=context}
		in ModuleSig{statparams=Option.map(descend c []) stat,
			     dynParam=Option.map(loop context) dyn}
		end
	in loop context expr
	end

    (* transforms in a bottom-up manner *)


    fun ptTransformUp ff expr =
      ff (ptTransform (ptTransformUp ff) expr)

    (* Define what is considered a name, in one row with Id *)
    fun isName (Id _) = true
      | isName (And _) = true
      | isName (Or _) = true
      | isName (Implies _) = true
      | isName (Darrow _) = true
      | isName (Arrow _) = true
      | isName (Iff _) = true
      | isName (Eq _) = true
      | isName (NotEq _) = true
      | isName (Not _) = true
      | isName (Ag _) = true
      | isName (Eg _) = true
      | isName (Af _) = true
      | isName (Ef _) = true
      | isName (Ax _) = true
      | isName (Ex _) = true
      | isName (Au _) = true
      | isName (Eu _) = true
      | isName (Ar _) = true
      | isName (Er _) = true
      | isName (Globally _) = true
      | isName (Eventually _) = true
      | isName (Until _) = true
      | isName (Releases _) = true
      | isName (NextTime _) = true
      | isName (Lt _) = true
      | isName (Gt _) = true
      | isName (Le _) = true
      | isName (Ge _) = true
      | isName (Plus _) = true
      | isName (Minus _) = true
      | isName (UMinus _) = true
      | isName (Times _) = true
      | isName (Div _) = true
      | isName (Mod _) = true
      | isName (Anyvalue _) = true
      | isName (Undefined _) = true
      | isName (True _) = true
      | isName (False _) = true
      | isName (Self _) = true
      | isName _ = false

    fun isObject (ModuleClosure _) = true
      | isObject (FunClosure _) = true
      | isObject (LetClosure _) = true
      | isObject (ChooseClosure _) = true
      | isObject (ForallClosure _) = true
      | isObject (ExistsClosure _) = true
      | isObject (MuClosure _) = true
      | isObject (NuClosure _) = true
      | isObject (QuantifiedVar _) = true
      | isObject (Builtin _) = true
      | isObject (StateVar _) = true
      | isObject (SkolemConst _) = true
      | isObject (AbstractConst _) = true
      | isObject (Object _) = true
      | isObject (ObjectInst _) = true
      | isObject (PatternFormal _) = true
      | isObject (DynPatternFormal _) = true
      | isObject (StaticFormalConst _) = true
      | isObject _ = false

    (* Extracts the parent closure from the given one, if one exists.
     * Returns SOME(parent_closure) on success, or NONE on failure. *)
    fun parentClosure (ModuleClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (FunClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (LetClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (LetAsstClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (TypeClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (ChoiceClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (ChoiceAsstClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (ChooseAsstClosure{parent=cl,...}) = SOME cl
      | parentClosure (ForeachAsstClosure{parent=cl,...}) = SOME cl
      | parentClosure (ChooseClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (ForallClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (ExistsClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (MuClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (NuClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (SyncClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (AsyncClosure{parent=cl,...}) = SOME(cl)
      | parentClosure (SequentClosure{parent=cl,...}) = SOME cl
      | parentClosure _ = NONE

    (* Extract the type field from any object. Return NONE or SOME tp. *)
    fun GetObjectType (Object{Type=tp,...}) = SOME tp
      | GetObjectType (ObjectInst{obj=obj,subst=ff}) = Option.map ff (GetObjectType obj)
      | GetObjectType (QuantifiedVar{Type=tp,...}) = SOME tp
      | GetObjectType (StateVar{Type=tp,...}) = SOME tp
      | GetObjectType (SkolemConst{Type=tp,...}) = SOME tp
      | GetObjectType (AbstractConst{Type=tp,...}) = SOME tp
      | GetObjectType (PatternFormal{Type=tp,...}) = SOME tp
      | GetObjectType (DynPatternFormal{Type=tp,...}) = SOME tp
      | GetObjectType (TypeConstr{Type=tp,...}) = SOME tp
      | GetObjectType (RecordField{Type=tp,...}) = SOME tp
      | GetObjectType (StaticFormalConst{Type=tp,...}) = SOME tp
      | GetObjectType (Type(_,tp)) = SOME tp
      | GetObjectType (Datatype(_,tp)) = SOME tp
      | GetObjectType _ = NONE

    (* Extract the name field from any object *)
    fun GetObjectName (Object{name=n,...}) = SOME n
      | GetObjectName (ObjectInst{obj=obj,...}) = GetObjectName obj
      | GetObjectName (Builtin{name=n, ...}) = SOME n
      | GetObjectName (QuantifiedVar{name=n,...}) = SOME n
      | GetObjectName (StateVar{name=n,...}) = SOME n
      | GetObjectName (SkolemConst{name=n,...}) = SOME n
      | GetObjectName (AbstractConst{name=n,...}) = SOME n
      | GetObjectName (PatternFormal{name=n,...}) = SOME n
      | GetObjectName (DynPatternFormal{name=n,...}) = SOME n
      | GetObjectName (TypeConstr{name=n,...}) = SOME n
      | GetObjectName (RecordField{name=n,...}) = SOME n
      | GetObjectName (StaticFormalConst{name=n,...}) = SOME n
      | GetObjectName (StaticFormalType{name=n,...}) = SOME n
      | GetObjectName (Module(_,{name=n,...})) = SOME n
      | GetObjectName (ModuleClosure{name=no,...}) = no
      | GetObjectName (TypeClosure{name=n,...}) = SOME n
      | GetObjectName (PType(_,n)) = SOME n
      | GetObjectName (FunClosure{name=no,...}) = no
      | GetObjectName (Theorem(_,n,_)) = SOME n
      | GetObjectName x = if isName x then SOME x else NONE

    (* Extract the uname field from any object *)
    fun GetObjectUName (Object{uname=n,...}) = SOME n
      | GetObjectUName (ObjectInst{obj=obj,...}) = GetObjectUName obj
      | GetObjectUName (QuantifiedVar{uname=n,...}) = SOME n
      | GetObjectUName (StateVar{uname=n,...}) = SOME n
      | GetObjectUName (SkolemConst{uname=n,...}) = SOME n
      | GetObjectUName (AbstractConst{uname=n,...}) = SOME n
      | GetObjectUName (PatternFormal{uname=n,...}) = SOME n
      | GetObjectUName (DynPatternFormal{uname=n,...}) = SOME n
      | GetObjectUName (TypeConstr{uname=n,...}) = SOME n
      | GetObjectUName (StaticFormalConst{uname=n,...}) = SOME n
      | GetObjectUName (StaticFormalType{uname=n,...}) = SOME n
      | GetObjectUName (ModuleClosure{uname=n,...}) = SOME n
      | GetObjectUName (TypeClosure{uname=n,...}) = SOME n
      | GetObjectUName (n as Uid _) = SOME n
      | GetObjectUName x = NONE
      
    (* Determine if a ParseTree represents a constant value or a non-value. *)

    fun isValue parseTree = 
      case parseTree of
	FunClosure {body = x,...} => isValue x
      | RecurFun (F) => isValue F
      | Builtin{name=Anyvalue _,...} => false
      | Builtin _ => true
      | Object {def=x,...} => isValue (x)
      | Appl (_, TypeConstr _,y) => isValue y
      | TypeConstr _ => true
      | Number(_) => true
      | True(x) => true
      | False(x) => true
      | StaticFormalConst(x) => false
      | PatternFormal(x) => false
      | DynPatternFormal(x) => false
      | RecordExpr (rpos, rlist) => List.all 
	    (fn RecordAsst(_,_,PT) => 
	     isValue PT
	  | _ => raise SympParseInternalError 
	     ("Badly formed record: " 
	      ^ (pt2stringDebug parseTree))) rlist
      | TupleExpr (rpos, tlist) => List.all isValue tlist
      | ObjectInst {obj = x,...} => isValue x
      | _ => false

    (* Determine if a ParseTree represents an undefined value.  Note
       that tuples and records are considered undefined if they
       contain an undefined field. *)

    fun isUndefined (Undefined _) = true
      | isUndefined (Builtin{name=Undefined _,...}) = true
      | isUndefined (RecordExpr(_,lst)) = List.exists isUndefined lst
      | isUndefined (RecordAsst(_,_,x)) = isUndefined x
      | isUndefined (TupleExpr(_,lst)) = List.exists isUndefined lst
      | isUndefined (ExtractRecord(_,x)) = isUndefined x
      | isUndefined (ExtractTuple(_,x)) = isUndefined x
      | isUndefined (ExtractAppl(_,x)) = isUndefined x
      | isUndefined (ExtractIndex x) = isUndefined x
      | isUndefined (Object{def=x,...}) = isUndefined x
      | isUndefined (ObjectInst{obj=x,...}) = isUndefined x
      | isUndefined _ = false

    (* Check if the object is a temporal operator *)
    fun isTemporal (Ag _) = true
      | isTemporal (Af _) = true
      | isTemporal (Eg _) = true
      | isTemporal (Ef _) = true
      | isTemporal (Ax _) = true
      | isTemporal (Ex _) = true
      | isTemporal (Au _) = true
      | isTemporal (Eu _) = true
      | isTemporal (Ar _) = true
      | isTemporal (Er _) = true
      | isTemporal (Globally _) = true
      | isTemporal (Eventually _) = true
      | isTemporal (Releases _) = true
      | isTemporal (NextTime _) = true
      | isTemporal (Builtin{name=name, ...}) = isTemporal name
      | isTemporal (Object{def=def, ...}) = isTemporal def
      | isTemporal _ = false

    (* Strip object down to its component values, without names. *)

    fun stripObjects (Object{def=x,...}) = stripObjects (x)
      | stripObjects (TupleExpr(tpos,tlist)) = TupleExpr (tpos,
							List.map 
							stripObjects tlist)
      | stripObjects (re as RecordExpr (rpos, rlist)) = 
      RecordExpr (rpos, List.map (fn RecordAsst(a, b, PT) => 
				  RecordAsst (a, b, stripObjects PT)
    | _ => raise SympParseInternalError 
				  ("Badly formed record: " 
				   ^ (pt2stringDebug re))) rlist)
      | stripObjects (Appl (p, x, y)) = Appl (p, stripObjects x, 
					      stripObjects y)
      | stripObjects (StaticFormalConst{value=SOME v, ...}) = stripObjects v
      | stripObjects (DynPatternFormal{value=SOME v, ...}) = stripObjects v
      | stripObjects x = x

    (* Strip all ObjectInst's from the expression, instantiating the types properly. *)
    fun stripObjectInst expr =
      let fun looplist ff cxt lst = List.map (loop ff cxt) lst
	  and recur ff context x = ptTransformParent (loop ff) context  x
	  and loop ff cxt (ObjectInst{obj=x,subst=subst}) = loop (ff o subst) cxt x
	    | loop ff cxt (TypeInst(p,parms,tp)) = TypeInst(p,List.map ff parms, ff tp)
	    | loop ff cxt (ChooseClosure{pattern=pat,
					 Type=tp,
					 names=names,
					 body=b,
					 parent=p}) =
	        let val newNames = looplist ff cxt names
		    val newPattern = loop ff cxt pat
		    fun cc body =
			ChooseClosure{pattern=newPattern,
				      Type=ff tp,
				      names=newNames,
				      body=body,
				      parent=cxt}
		    val body = loop ff (cc Fake) b
		in cc body
		end
	    | loop ff cxt (QuantifiedVar{name=n,uname=uname,Type=tp}) =
		QuantifiedVar{name=n,uname=uname,Type=ff tp}
	    | loop ff cxt (StateVar{name=name,uname=uname,Type=tp,id=id}) =
		StateVar{name=name,uname=uname,Type=ff tp,id=loop ff cxt id}
	    | loop ff cxt (SkolemConst{name=name,uname=uname,Type=tp}) =
		SkolemConst{name=name,uname=uname,Type=ff tp}
	    | loop ff cxt (AbstractConst{name=name,uname=uname,Type=tp}) =
		AbstractConst{name=name,uname=uname,Type=ff tp}
	    | loop ff cxt (Object{name=name,uname=uname,Type=tp,def=def}) =
		Object{name=name,uname=uname,Type=ff tp,def=loop ff cxt def}
	    | loop ff cxt (PatternFormal{name=name,uname=uname,Type=tp,extract=extract}) =
		PatternFormal{name=name,uname=uname,Type=ff tp,extract=extract}
	    | loop ff cxt (DynPatternFormal{name=name,uname=uname,Type=tp,
					    extract=extract,value=v}) =
		DynPatternFormal{name=name,uname=uname,Type=ff tp,
				 extract=extract,value=Option.map(loop ff cxt) v}
	    | loop ff cxt (TypeConstr{name=name,uname=uname,Type=tp}) =
		TypeConstr{name=name,uname=uname,Type=ff tp}
	    | loop ff cxt (RecordField{name=name,Type=tp}) =
		RecordField{name=name,Type=ff tp}
	    | loop ff cxt (StaticFormalConst{name=name,uname=uname,Type=tp,value=v}) =
		StaticFormalConst{name=name,uname=uname,Type=ff tp,
				  value=Option.map(loop ff cxt) v}
	    | loop ff cxt (PType(p,tp)) = PType(p,ff tp)
	    | loop ff cxt (Symmetric(p,tp)) = Symmetric(p,ff tp)
	    | loop ff cxt (Finite(p,tp)) = Finite(p,ff tp)
	    | loop ff cxt (TypedExpr(p,e,tp)) = TypedExpr(p,loop ff cxt e,ff tp)
	    | loop ff cxt (TypedPattern(p,e,tp)) = TypedPattern(p,loop ff cxt e,ff tp)
	    | loop ff cxt x = recur ff cxt x
      in loop (fn x=>x) Fake expr
      end

    fun eqUNames (e1, e2) =
      case (GetObjectUName e1) of
	SOME u1 =>
	  (case (GetObjectUName e2) of
	     SOME u2 =>
	       ptEq (u1, u2)
	   | NONE => false)
      | NONE => false

    (* Additional functions to extract a type of any typechecked expression. *)
    fun getExprType findObject expr =
	let fun loop ff (ObjectInst{obj=obj,subst=subst}) =
	         loop (ff o subst) obj
	      | loop ff (Object{Type=tp,...}) = ff tp
	      | loop ff (Next e) = loop ff e
	      | loop ff (FunClosure{formals=arg,body=body,...}) =
		 FunType(dp,loop ff arg, loop ff body)
	      | loop ff (RecurFun F) =
		 (case loop ff F of
		      FunType(_,_,tp) => tp
		    | tp => raise SympBug("getExprType: RecurFun is not FunType: "
					  ^(pt2stringDebug tp)))
	      | loop ff (LetClosure{body=body,...}) = loop ff body
	      | loop ff (ChoiceClosure{body=body,...}) = loop ff body
	      | loop ff (ChooseClosure{Type=tp,...}) = ff tp
	      | loop ff (ForallClosure _) = BoolType dp
	      | loop ff (ExistsClosure _) = BoolType dp
	      | loop ff (MuClosure _) = BoolType dp
	      | loop ff (NuClosure _) = BoolType dp
	      | loop ff (QuantifiedVar{Type=tp,...}) = ff tp
	      | loop ff (Builtin{Type=tp,...}) = ff tp
	      | loop ff (Number _) = IntType dp
	      | loop ff (True _) = BoolType dp
	      | loop ff (False _) = BoolType dp
	      | loop ff True2 = BoolType2
	      | loop ff False2 = BoolType2
	      | loop ff (StateVar{Type=tp,...}) = ff tp
	      | loop ff (SkolemConst{Type=tp,...}) = ff tp
	      | loop ff (AbstractConst{Type=tp,...}) = ff tp
	      | loop ff (PatternFormal{Type=tp,...}) = ff tp
	      | loop ff (DynPatternFormal{Type=tp,...}) = ff tp
	      | loop ff (TypeConstr{Type=tp,...}) = ff tp
	      | loop ff (RecordField{Type=tp,...}) = ff tp
	      | loop ff (StaticFormalConst{Type=tp,...}) = ff tp
	      | loop ff (e as (IfExpr(_,[],_))) = raise SympBug
		      ("getExprType: IfExpr has empty `if' list: "
		       ^(pt2stringDebug e))
	      | loop ff (IfExpr(_,hd::_,_)) = loop ff hd
	      | loop ff (CondExpr(_,_,e)) = loop ff e
	      | loop ff (LetExpr(_,_,e)) = loop ff e
	      | loop ff (e as CaseExpr(_,_,[])) = raise SympBug
		      ("getExprType: CaseExpr has empty list of cases: "
		       ^(pt2stringDebug e))
	      | loop ff (CaseExpr(_,_,e::_)) = loop ff e
	      | loop ff (WithExpr(_,e,_)) = loop ff e
	      | loop ff (RecordExpr(_,lst)) = RecordType(dp,List.map(loop ff) lst)
	      | loop ff (RecordAsst(_,field,e)) = RecordField{name=field,Type=loop ff e}
	      | loop ff (TupleExpr(_,lst)) = TupleType(dp,List.map(loop ff) lst)
	      | loop ff (NondetExpr(_,e::_)) = loop ff e
	      | loop ff (NondetExpr(p,[])) = raise SympBug
		      ("getExprType: empty NondetExpr at "^(pos2string p))
	      | loop ff (ExtractAppl(c,_)) =
		  (case loop ff c of
		       FunType(_,tp,_) => ff tp
		     | tp => raise SympBug
			   ("getExprType: type constructor in ExtractAppl is not FunType: "
			    ^(pt2stringDebug tp)))
	      | loop ff (ExtractTuple(n,e)) =
		  (case loop ff e of
		       TupleType(_,lst) =>
			   if (List.length lst) > n then List.nth(lst,n)
			   else raise SympBug
			       ("getExprType: expr in ExtractTuple is a shorter tuple than n")
		     | x => raise SympBug
			       ("getExprType: expr in ExtractTuple is not TupleType"))
	      | loop ff (ExtractRecord(f,e)) =
		  let fun findField ((RecordField{name=name,Type=tp})::lst) = 
		           if ptEq(f,name) then ff tp else (findField lst)
			| findField (x::_) = raise SympBug
			   ("getExprType: not RecordField: "^(pt2stringDebug x))
			| findField [] = raise SympBug
			   ("getExprType: no such field in a record: "
			    ^(pt2stringDebug f))
		  in (case loop ff e of
			  RecordType(_,lst) => findField lst
			| x => raise SympBug
			   ("getExprType: expr in ExtractRecord is not RecordType: "
			    ^(pt2stringDebug x)))
		  end
	      (* The type of index is an AbstractType that lists all TypeConstr's. *)
	      | loop ff (ExtractIndex e) =
		  let fun getClist (EnumType(_,lst)) = lst
			| getClist (TypeClosure{def=EnumType(_,lst),...}) = lst
			| getClist (u as Uid _) = 
		           (case findObject u of
				SOME tp => getClist tp
			      | NONE => raise SympBug
				  ("getExprType: no object with such Uid: "
				   ^(pt2string u)))
			| getClist t = raise SympBug
			   ("getExprType: not an EnumType expression in ExtractIndex:\n  "
			    ^(pt2string(TypedExpr(dp,e,t))))
		      val lst = getClist(loop ff e)
		  in AbstractType lst
		  end
	      | loop ff (Appl(_,e,_)) =
		  (case loop ff e of
		       FunType(_,_,tp) => ff tp
		     | tp => raise SympBug("getExprType: function in Appl is not FunType: "
					   ^(pt2stringDebug tp)))
	      | loop ff x = raise SympBug("getExprType: unexpected expression: "
					  ^(pt2stringDebug x))
	in loop (fn x=>x) expr
	end

    (* Checks whether the expression might have types in it.  If it
       does, we have to keep Object and ObjectInst wrappers. *)
    fun hasTypes e =
	(case e of
	     FunClosure _ => true
	   | RecurFun _ => true
(* We cannot wrap these back into Object/ObjectInst, since we lose the
   ability to evaluate them uniformly.  But if we don't, we lose the
   type information.  Maybe it's a tolerable loss, we'll see. *)
(*  	   | Builtin(Undefined _) => true *)
(*  	   | Builtin(Eq _) => true *)
	   | Object _ => true
	   | ObjectInst _ => true
	   | RecordExpr(_,rlist) => List.exists hasTypes rlist
	   | RecordAsst(_,_,x) => hasTypes x
	   | TupleExpr(_,lst) => List.exists hasTypes lst
	   | x => not(isValue x))

    (* Create an `undefined' value of type `tp' *)
    fun bot tp = Builtin{name=Undefined dp, Type=tp}

    (* Try to extract a function value from f.  Also, if the function
       is wrapped into ObjectInst, return a function that will wrap
       any parse tree into the same ObjectInst.  Later, when the
       extracted function is applied, the result will be wrapped this
       way to preserve the type information. *)

    fun extractFun ff f =
	let fun loop ff hh (f as FunClosure _) = SOME(f,hh)
	      | loop ff hh (RecurFun(F)) = SOME(RecurFun(F),hh)
	      | loop ff hh (oi as ObjectInst {obj = x,subst=subst}) =
		  loop (ff o subst) (fn e=>hh(ObjectInst{obj=e,subst=subst})) x
	      | loop ff hh (Object {def = x,...}) = loop ff hh x
	      (* Here we assume the typechecker guarantees they _are_ functions *)
	      | loop ff hh (Builtin{name=f,Type=tp}) = SOME(Builtin{name=f,Type=ff tp},hh)
	      | loop ff hh (TypeConstr{name=name,uname=uname,Type=tp}) =
		  SOME(TypeConstr{name=name,uname=uname,Type=ff tp},hh)
	      | loop _ _ _ = NONE
	in loop ff (fn x=>x) f
	end

    fun extractTuple n (t as TupleExpr(_,l)) = 
	  if List.length l > n then SOME(List.nth(l,n))
	  else raise SympBug("Evaluator/extractTuple: tuple is shorter than "
			     ^(Int.toString n)^": "
			     ^(pt2string t))
      | extractTuple n (oi as ObjectInst{obj = x,subst=subst}) = 
	  let val res = extractTuple n x
	  in 
	     Option.map(fn e=>if hasTypes e then ObjectInst{obj=e,subst=subst}
			      else e) res
	  end
      | extractTuple n (Object{def=x,...}) = extractTuple n x
      | extractTuple _ x = if isUndefined x then SOME(Undefined dp)
			   else NONE

    fun extractRecord field (r as RecordExpr(_,rlist)) = 
	  let val fieldMatch = List.find (fn RecordAsst (_, fname, e) =>
					       ptEq (fname, field)
		 | x => raise SympBug
					       ("Badly formed record field: " ^
						pt2stringDebug x)) rlist
		 in
		   (case fieldMatch of
		      SOME (RecordAsst(_, _, e)) => SOME e
		    | SOME x => raise SympBug ("This can't happen!")
		    | NONE => raise SympBug ("Field extraction failed: " ^
					       pt2stringDebug(ExtractRecord (field, r))))
		 end
      | extractRecord f (w as WithExpr (_, _, wlist)) =
	  let val fieldMatch = List.find (fn WithAsst (_, fname, e) =>
					  ptEq (fname, f)
	| x => raise SympBug
					  ("Badly formed with field: " ^
					   pt2stringDebug x)) wlist
	  in
	    (case fieldMatch of
	       SOME (WithAsst (_, _, e)) => SOME e
	       | SOME x => raise SympBug ("This can't happen!")
	       | NONE => if isUndefined w then SOME(Undefined dp)
			 else NONE)
	  end
      | extractRecord f (oi as ObjectInst{obj = x,subst=subst}) = 
	  let val res = extractRecord f x
	  in 
	     Option.map(fn e=>if hasTypes e then ObjectInst{obj=e,subst=subst}
			      else e) res
	  end
      | extractRecord f (Object{def=x,...}) = extractRecord f x
      | extractRecord _ x = if isUndefined x then SOME(Undefined dp)
			    else NONE

    fun extractAppl c1 (Appl (_, c2, x)) =
      if eqUNames (c1, c2) then
	SOME (x)
      else
	raise SympBug "Incorrect application extraction"
      | extractAppl _ x = if isUndefined x then SOME(Undefined dp)
			  else NONE

    (* newName(): Generate new unique type variable names (string).
     * Not for the user, the names are "illegal" for the parser. *)
    val (newNumber,
	 nextNumber,
	 resetNumber) =
	let val num = ref 0
	in 
	    (fn () => (!num before (num := (!num) + 1)),
	     fn () => (!num),
	     fn () => (num:=0))
	end

    fun newName () = "X!"^(Int.toString(newNumber()))

  end
