(* Parse tree type and related signature *)

(* This must be on top level *)
(*  type Pos = int * int *)
    
signature PARSE_TREE =
  sig
    exception SympParseInternalError of string
    exception SympBug of string

    type Str = Str.Str

    datatype ParseTree = 
	Program of Pos.Pos * ParseTree list
      | Include of Pos.Pos * string
      (* Included(pos,file,parse_tree) Parsed included file *)
      | Included of Pos.Pos * string * ParseTree
      | Quote of Pos.Pos * string
	(* This is for creating temporary objects in the tree and
	   replacing them later with new values, in all places at once.
	   Once all objects are final, one could walk over the tree and
	   make them permanent. *)
      | SharedRef of ParseTree ref
      | Id of Pos.Pos * string
	(* Unique ID, internal names of objects, to distinguish
	   them from the user-given names. *)
      | Uid of string
      | Tvar of Pos.Pos * string
      (* An internal type variable introduced in typechecking.
       * It means the type is not fully deduced yet; it cannot be 
       * considered as a polymorphic type variable. *)
      | TvarN of int
      (* Type of the form "<type>,...,<type> name", like "int list" *)
      | TypeInst of Pos.Pos * (ParseTree list) * ParseTree
      | Ellipsis of Pos.Pos
(*        | Module of Pos.Pos * ParseTree * (ParseTree list option) *)
(*  	              * (ParseTree option) * ParseTree *)
      (* Built-in top level declarations (like +, *, etc.) *)
      | TopClosure of ParseTree list
      | Module of Pos.Pos *
	{ name: ParseTree,
	  statparams: ParseTree list option,
	  dynparams: ParseTree option,
	  body: ParseTree
	}
      | ModuleClosure of
	{ name: ParseTree option,
	  uname: ParseTree, (* unique module identifier *)
	  (* dynName: ParseTree option, *)
	  Sig: ModuleSig,
	  closure: ParseTree list,
	  (* def = NONE for begin-end, since only closure is important in
	     that case *)
	  def: ParseTree option,
	  parent: ParseTree
	}
      (* Parent closure of an instantiated module *)
      (* | ModuleParamClosure of
	{ closure: ParseTree list,
	  parent: ParseTree
	} *)
      (* This is passed as a "parent closure" to inner functions.
	 Originally it contains "FunClosure" that only defines
	 the name space, and later it's overwritten by the actual
	 function definition, thus making a circular structure for
	 recursive functions *)
(*        | FunClosureRef of ParseTree ref *)
      (* This is a preliminary version of the  "parent closure",
       * without the actual definition.  It is used only at 
       * the time of typechecking, and is later replaced by the full
       * version. Maybe, it's not even needed... *)
(*        | FunClosureSmall of *)
(*  	{ name: ParseTree option, *)
(*  	  formals: ParseTree, (* There can only be one argument *) *)
(*  	  parent: ParseTree (* The parent's closure *) *)
(*  	} *)
      (* The full version of the above.  If body = Fake, then it's the
       * same as the above. *)
      | FunClosure of
	{ name: ParseTree option,
	  formals: ParseTree, (* There can only be one argument *)
	  parent: ParseTree, (* The parent's closure *)
	  body: ParseTree
	}
      (* The "linearized" representation of recursive function `f':
       * RecurFun(fn f => <the curried def. of `f'>) *)
      | RecurFun of ParseTree
      (* Closure for `let' expression *)
      | LetClosure of
	{ locals: ParseTree list,
	  parent: ParseTree,
	  body: ParseTree }
      (* Same for assignments *)
      | LetAsstClosure of
	{ locals: ParseTree list,
	  parent: ParseTree,
	  body: ParseTree list }
      (* Serves as an additional "context" for recursive or
       * parameterized type declarations. *)
      | TypeClosure of { name: ParseTree,
			 uname: ParseTree,
			 params: ParseTree list,
			 def: ParseTree,
			 recursive: bool,
			 parent: ParseTree }
      (* Keeps bound variables in a choice of a `case' statement *)
      | ChoiceClosure of { pattern: ParseTree,
			   (* Internal name for selector expr *)
			   uname: ParseTree,
			   names: ParseTree list,
			   body: ParseTree,
			   parent: ParseTree }
      (* Same for assignments *)
      | ChooseAsstClosure of 
	{ names: ParseTree list option,
	  choices: ParseTree list,
	  parent: ParseTree}
      (* `choices' is a list of `ChoiceAsstClosure' elements *)
      | ChoiceAsstClosure of
	{ pattern: ParseTree,
	  (* Internal name for selector expr *)
	  uname: ParseTree,
	  names: ParseTree list,
	  body: ParseTree list,
	  parent: ParseTree }
      | ForeachAsstClosure of
	{ names: ParseTree list,
	  assts: ParseTree list,
	  parent: ParseTree }

      (* The closure for the CHOOSE operator.  It's the same as ChoiceClosure,
       * only serves a different purpose. *)
      | ChooseClosure of { pattern: ParseTree,
			   Type: ParseTree, (* Type of the result *)
			   names: ParseTree list,
			   body: ParseTree,
			   parent: ParseTree }
      (* Quantifiers *)
      | ForallClosure of { names: ParseTree list,
			   body: ParseTree,
			   parent: ParseTree }
      | ExistsClosure of { names: ParseTree list,
			   body: ParseTree,
			   parent: ParseTree }
      | MuClosure of { name: ParseTree,
		       body: ParseTree,
		       parent: ParseTree }
      | NuClosure of { name: ParseTree,
		       body: ParseTree,
		       parent: ParseTree }
      | QuantifiedVar of { name: ParseTree,
			   uname: ParseTree,
			   Type: ParseTree }
      (* Module parallel composition *)
      | SyncClosure of { names: ParseTree list,
			 body: ParseTree,
			 parent: ParseTree }
      | AsyncClosure of { names: ParseTree list,
			  body: ParseTree,
			  parent: ParseTree }

      (* Context for the sequent in the proof system *)
      | SequentClosure of { names: ParseTree list,
			    parent: ParseTree }

      (* This is a builtin top-level object *)
      | Builtin of {name: ParseTree, Type: ParseTree}
      (* A state variable object *)
      | StateVar of
	{ name: ParseTree,
	  uname: ParseTree,
	  Type: ParseTree,
	  (* State variables have to be distinguished later for
	     different module instances.  The `id' field will hold the
	     string of module instances (e.g. M1[i].M2[j].M3(y,x)) the
	     state var is coming from, where Mi are "uname"'s of
	     modules.  The context provides the direct link to the
	     transition system information. *)
	  id: ParseTree
	}
      | SkolemConst of {name: ParseTree,
			uname: ParseTree,
			Type: ParseTree }

      (* Uninterpreted constant similar to SkolemConst, only it's guaranteed
       to have different interpretation from any other AbstractConst *)
      | AbstractConst of {name: ParseTree,
			  uname: ParseTree,
			  Type: ParseTree }

      (* Object is a typechecked named object representation.
         It must carry enough info to disambiguate overloaded
	 identifiers. *)
      | Object of
	{ name: ParseTree,
	  uname: ParseTree,
	  Type: ParseTree,
	  def: ParseTree
	}
      (* Object instance that replaces a name in the program.  The
	 `subst' field is a function that properly instantiates the
         types within the object for this particular instance. *)
      | ObjectInst of {obj: ParseTree,
		       subst: ParseTree -> ParseTree}

      (**************************************************************)

      (* A variable bound by a pattern *)
      | PatternFormal of { name: ParseTree,
			   uname: ParseTree,
			   Type: ParseTree,
			   (* Function that wraps a proper extracter around 
			    * an actual parameter *)
			   extract: ParseTree -> ParseTree }
      (* A variable bound by a pattern in dynamic parameter of a module *)
      | DynPatternFormal of { name: ParseTree,
			      uname: ParseTree,
			      Type: ParseTree,
			     (* Function that wraps a proper extracter around 
			      * an actual parameter *)
			      extract: ParseTree -> ParseTree,
			      value: ParseTree option }
      (* Datatype constructor. 
       * The type is either the datatype itself (EnumType), or
       * paramType -> datatype (FunType).  In the definition of recursive
       * datatypes, the datatype is referred to by its name which later is
       * instantiated as needed. *)
      (* When a datatype is introduced, all of its constructors are
       * introduced at the same level as functions/constants *)
      | TypeConstr of { name: ParseTree,
		        uname: ParseTree,
		        Type: ParseTree }
      (* Record field object.
         `uname' doesn't really make sense here. *)
      | RecordField of { name: ParseTree,
			 Type: ParseTree
		       (* Avoid circularity.  We probably do not need it. *)
		       (* recordType: ParseTree *) }
      (* Static formal constant and type parameters of a module.
         Type parameters are represented by PType. *)
      | StaticFormalConst of
	{ name: ParseTree,
	  uname: ParseTree,
	  Type: ParseTree,
	  value: ParseTree option }
      | StaticFormalType of
	{ name: ParseTree,
	  uname: ParseTree,
	  value: ParseTree option }
      | Theorem of Pos.Pos * ParseTree * ParseTree
      (* Parametric type in the static params of a module *)
      | PType of Pos.Pos * ParseTree
      | Symmetric of Pos.Pos * ParseTree
      | Finite of Pos.Pos * ParseTree
      | BeginEnd of Pos.Pos * ParseTree list
      | Sync2 of Pos.Pos * ParseTree * ParseTree
      | Async2 of Pos.Pos * ParseTree * ParseTree
      | Sync of Pos.Pos * ParseTree list * ParseTree
      | Async of Pos.Pos * ParseTree list * ParseTree
      (* TypeDecl(pos,name,paramlist,def) *)
      | TypeDecl of Pos.Pos * ParseTree * (ParseTree list) * ParseTree
      | DatatypeDecl of Pos.Pos * ParseTree * (ParseTree list) * ParseTree
      | Var of Pos.Pos * ParseTree list
      | StateVarDecl of Pos.Pos * ParseTree list
      | Val of Pos.Pos * ParseTree * ParseTree
      | FunCases of Pos.Pos * ParseTree list
      (* FunCase(pos,name,patterns,result_type,expr) *)
      | FunCase of Pos.Pos * ParseTree * (ParseTree list)
	* (ParseTree option) * ParseTree
      | Open of Pos.Pos * ParseTree
      | Export of Pos.Pos * ParseTree list
      | Models of Pos.Pos * ParseTree * ParseTree
      | Refines of Pos.Pos * ParseTree * ParseTree
      (* LabeledAsst(p, label, asst) *)
      | LabeledAsst of Pos.Pos * ParseTree * ParseTree
      | Nop of Pos.Pos
      | NormalAsst of Pos.Pos * ParseTree * ParseTree
      | InitAsst of Pos.Pos * ParseTree * ParseTree
      | NextAsst of Pos.Pos * ParseTree * ParseTree
      (* Wrapper to represent next(x) in boolean formulas for the
         transition relation (internal use only) *)
      | Next of ParseTree
      | IfAsst of Pos.Pos * (ParseTree list) * ParseTree list
      | CondAsst of Pos.Pos * ParseTree * (ParseTree list)
      | LetAsst of Pos.Pos * (ParseTree list) * (ParseTree list)
      | CaseAsst of Pos.Pos * ParseTree * (ParseTree list)
      | ChoiceAsst of Pos.Pos * ParseTree * (ParseTree list)
      | ChooseAsst of Pos.Pos * (ParseTree list option) * ParseTree list
      | ForeachAsst of Pos.Pos * ParseTree list * ParseTree list
      | IfExpr of Pos.Pos * (ParseTree list) * ParseTree
      | CondExpr of Pos.Pos * ParseTree * ParseTree
      | LetExpr of Pos.Pos * (ParseTree list) * ParseTree
      | CaseExpr of Pos.Pos * ParseTree * (ParseTree list)
      | ChoiceExpr of Pos.Pos * ParseTree * ParseTree
      | ChooseExpr of Pos.Pos * ParseTree * ParseTree
      | WithExpr of Pos.Pos * ParseTree * (ParseTree list)
      | WithAsst of Pos.Pos * ParseTree * ParseTree
      | RecordExpr of Pos.Pos * ParseTree list
      | TupleExpr of Pos.Pos * ParseTree list
      | NondetExpr of Pos.Pos * ParseTree list
      | RecordAsst of Pos.Pos * ParseTree * ParseTree
      (* ModuleInst(module,stat_param_list).
       Dynamic params are applied with Appl. *)
      | ModuleInst of Pos.Pos * ParseTree * (ParseTree list)
      | Dot of Pos.Pos * ParseTree * ParseTree
      | Type of Pos.Pos * ParseTree
      | Datatype of Pos.Pos * ParseTree
      | And of Pos.Pos
      | Or of Pos.Pos
      | Implies of Pos.Pos
      (* Normally, these two will also be "Implies", but the user may 
       * redefine them to something else *)
      | Arrow of Pos.Pos
      | Darrow of Pos.Pos
      | Iff of Pos.Pos
      | Eq of Pos.Pos
      | NotEq of Pos.Pos
      | Not of Pos.Pos
      | Forall of Pos.Pos * (ParseTree list) * ParseTree
      | Exists of Pos.Pos * (ParseTree list) * ParseTree
      (* CTL constructs *)
      | Ag of Pos.Pos
      | Af of Pos.Pos
      | Eg of Pos.Pos
      | Ef of Pos.Pos
      | Ax of Pos.Pos
      | Ex of Pos.Pos
      | Au of Pos.Pos
      | Eu of Pos.Pos
      | Ar of Pos.Pos
      | Er of Pos.Pos
      (* LTL constructs *)
      | Globally of Pos.Pos
      | Eventually of Pos.Pos
      | Until of Pos.Pos
      | Releases of Pos.Pos
      | NextTime of Pos.Pos
      (* Mu-Calculus constructs: ?u(pos, var, expr) *)
      | Mu of Pos.Pos * ParseTree * ParseTree
      | Nu of Pos.Pos * ParseTree * ParseTree
      (* Arithmetic operators *)
      | Lt of Pos.Pos
      | Gt of Pos.Pos
      | Le of Pos.Pos
      | Ge of Pos.Pos
      | Plus of Pos.Pos
      | Minus of Pos.Pos
      | UMinus of Pos.Pos
      | Times of Pos.Pos
      | Div of Pos.Pos
      | Mod of Pos.Pos
      | Number of Pos.Pos * int
      | BoolType of Pos.Pos
      (* Internal type for 2-valued boolean operations *)
      | BoolType2
      | NatType of Pos.Pos
      | IntType of Pos.Pos
      | RangeType of Pos.Pos * ParseTree * ParseTree
      | FunType of Pos.Pos * ParseTree * ParseTree
      | TupleType of Pos.Pos * ParseTree list
      | ArrayType of Pos.Pos * ParseTree * ParseTree
      | RecordType of Pos.Pos * ParseTree list
      | EnumType of Pos.Pos * ParseTree list
      | AbstractType of ParseTree list
      | Of of Pos.Pos * ParseTree * ParseTree
      | TypedVars of Pos.Pos * (ParseTree list) * (ParseTree option)
      | False of Pos.Pos
      | True of Pos.Pos
      (* Special 2-valued true and false, for the internal use of the
         transition relation generator *)
      | False2
      | True2
      | Min of Pos.Pos
      | Max of Pos.Pos
      | Undefined of Pos.Pos
      | Anyvalue of Pos.Pos
      (* Internal value representing "all the other values" in abstractions *)
      | OtherValue
      | Self of Pos.Pos
      | Appl of Pos.Pos * ParseTree * ParseTree
      | TypedExpr of Pos.Pos * ParseTree * ParseTree
      | Fn of Pos.Pos * ParseTree list
      | ChoicePattern of Pos.Pos * ParseTree list
      | ApplPattern of Pos.Pos * ParseTree * ParseTree
      | TypedPattern of Pos.Pos * ParseTree * ParseTree
      | TuplePattern of Pos.Pos * ParseTree list
      | RecordPattern of Pos.Pos * ParseTree list
	(* AsPattern(p,id,pattern) represents `id as pattern' *)
      | AsPattern of Pos.Pos * ParseTree * ParseTree
      | Underscore of Pos.Pos
      (* And pattern extractors, for internal use only, to provide a way
       * of assigning values to variables in the pattern *)
      (* ExtractAppl(DatatypeConstructor,Expr) *)
      | ExtractAppl of ParseTree * ParseTree
      (* Takes n-th element of a tuple *)
      | ExtractTuple of int * ParseTree
      (* ExtractRecord(FieldName, Expr) - no full record type is needed,
       * the expression must contain the field info *)
      | ExtractRecord of ParseTree * ParseTree
      (* ExtractIndex(Expr) - get the TypeConstr that makes the value
         of `Expr' that is of EnumType *)
      | ExtractIndex of ParseTree

      (* for emphasizing grouping, e.g. for tuple type *)
      | Group of Pos.Pos * ParseTree 
      | Fake (* for some errors and non-implemented commands *)
    and ModuleSig =
       (* Module's "type" for internal typechecking purposes.
       * In the future it is likely to be extended to ML-like signatures. *)
      ModuleSig of { statparams: ParseTree list option,
		     (* (name, type) of dyn. param *)
		     dynParam: ParseTree option }
	
    val pos: ParseTree -> Pos.Pos
    val pos2string: Pos.Pos -> string
    (* More or less pretty printing functions *)
    val Str2string: Str -> string
    val strlist2str: string -> string list -> string
    val Strlist2Str: string -> Str list -> Str
    val pt2str: ParseTree -> Str
    val pt2string: ParseTree -> string
    val ModuleSig2string: ModuleSig -> string
    val ptlist2str: string -> (ParseTree list) -> string
    val pt2shortStr: int -> ParseTree -> string	
    (* Really nasty chatty printing *)
    val pt2strDebug: ParseTree -> Str
    val pt2stringDebug: ParseTree -> string
    val ModuleSig2stringDebug: ModuleSig -> string
    (* Print the parse tree with a given print function incrementally *)
    val ptPrintDebug: (string -> unit) -> ParseTree -> unit
    val ptlist2strDebug: string -> (ParseTree list) -> string
    (* Equality of parse trees upto position fields *)
    val ptEq: ParseTree * ParseTree -> bool
    (* Same as ptEq, but upto parent closures.  More efficient, when
     * parent closures are irrelevant. *)
    val ptEqLoose: ParseTree * ParseTree -> bool
    (* Transformation builder *)
    val ptTransform: (ParseTree -> ParseTree) -> ParseTree -> ParseTree
    val ptTransformParent: (ParseTree -> ParseTree -> ParseTree)
	                -> ParseTree -> ParseTree -> ParseTree
    val ptTransformUp: (ParseTree -> ParseTree) -> ParseTree -> ParseTree
(*      val ptTransformLazy: (ParseTree -> ('a -> ParseTree)) -> ParseTree *)
(*  	                   -> ('a -> ParseTree) *)

    val isName: ParseTree -> bool
    val isObject: ParseTree -> bool
    val parentClosure: ParseTree -> ParseTree option
    val GetObjectType: ParseTree -> ParseTree option
    val GetObjectName: ParseTree -> ParseTree option
    val GetObjectUName: ParseTree -> ParseTree option

    val isValue: ParseTree -> bool 
    val isUndefined: ParseTree -> bool 
    val isTemporal: ParseTree -> bool
    val stripObjects : ParseTree -> ParseTree
    val stripObjectInst: ParseTree -> ParseTree
    val eqUNames: ParseTree * ParseTree -> bool
    (* Extract a type from any typechecked expression *)
    val getExprType: (ParseTree -> ParseTree option) -> ParseTree -> ParseTree

    val hasTypes : ParseTree -> bool
    val extractRecord : ParseTree -> ParseTree -> ParseTree option
    val extractTuple : int -> ParseTree -> ParseTree option
    val extractFun : (ParseTree -> ParseTree)
      -> ParseTree -> (ParseTree * (ParseTree -> ParseTree)) option
    val extractAppl : ParseTree -> ParseTree -> ParseTree option

    val newNumber: unit -> int
    val nextNumber: unit -> int
    val resetNumber: unit -> unit
    val newName: unit -> string

  end
