(* The default implementation of SEQUENT interface.  It is not
   restricted to the SEQUENT signature here, this is done later in the
   ProofSystem module.  We need to see the internals of this module in
   other places like ProofRules module. *)

functor SequentDefaultFun(structure Hash: HASH
			  structure Interface: INTERFACE) =
  struct

    structure SequentBase = 
      struct
	structure InputLanguage = 
	    InputLanguageDefaultFun(structure Hash = Hash
				    structure Interface = Interface)

	structure Abstract = AbstractFun(structure Hash = Hash
					 structure ParserDefault = InputLanguage.ParserDefault)

	structure Trans = TransFun(structure Hash = Hash
				   structure Abstract = Abstract
				   structure Evaluate = InputLanguage.Typecheck.Evaluate)

	open InputLanguage

	(* Dummy type until we figure it out *)
	type SequentHints = unit

	(* Sequent: (M,I); Sigma; Gamma |- Delta *)
	type Sequent = { model: Trans.Model,
			(* may or may not have the initial state predicate *)
			init: ParseTree option,
			invar: ParseTree list,
			(* Context specific to the sequent.  In particular, 
			   all Skolem constants go here *)
			context: ProgramContext,
			assumptions: ParseTree list,
			conclusions: ParseTree list }
	    
	datatype SequentPart = InvarPart | AssumptionPart | ConclusionPart
	  | AnyPart | SomeParts of SequentPart list

	fun formulaInd2str(part, nOpt) =
	    let fun part2str AnyPart = "*"
		  | part2str InvarPart = "i"
		  | part2str AssumptionPart = "-"
		  | part2str ConclusionPart = "+"
		  | part2str (SomeParts lst) = strlist2str "" (List.map part2str lst)
	    in
		(part2str part)
		^(case nOpt of
		      NONE => ""
		    | SOME n => Int.toString n)
	    end

	local fun op *(x,y) = Conc(x,y)
	in 
	    fun sequent2str ({init=init,
			      invar=invar,
			      assumptions=prems,
			      conclusions=concs,...}: Sequent) =
		let fun list2str pref (n,[]) = (Str "")
		      | list2str pref (n,hd::tl) = ((Str ("["^pref^(Int.toString n)^"]  "))
						    *(pt2str hd)*(Str "\n")
						    *(list2str pref (n+1, tl)))
		in ((list2str "i" (1,invar))
		    *(list2str "-" (1,prems))
		    *(Str ("|--------"^(case init of
					    NONE => "------"
					  | SOME _ => " INIT ")
			   ^"------------\n"))
		    *(list2str "" (1,concs)))
		end
	end

	fun sequent2string seq = Str2string(sequent2str seq)

	fun hints2string hints = ""
      end

    (* Open them here, before SML figures out their signatures and constrains them *)
    open SequentBase
    open InputLanguage

    (* Some common types and functions *)
    structure SequentCommon = SequentCommonFun(structure Interface = Interface
					       structure SequentBase = SequentBase)

    open SequentCommon

    open SympBug
    open Str
    open ListOps
    open Pos
    open Hash
    open ParserDefault
    open Trans
    open Abstract
    open ParseTreeStruct

    type TermInd = int (* Term index *)

    (* Sequent template types *)

    datatype Template =
      (* Formula variable *)
	FormulaTemp of string
      (* Template var matching a list of formulas which possibly match a common template *)
      | ListFormulaTemp of string * (Template option)
      (* Built-in function (explicitly specified) *)
      | BuiltinTemp of ParseTree
      (* Arbitrary value of any type, unspecified functions, etc. *)
      | ValueTemp of string
      (* Program context variable *)
      | ProgramContextTemp of string
      (* Same as ValueTemp, only the expression is thrown away *)
      | WildCardTemp
      (* Same as ListValueTemp, only the expressions are thrown away *)
      | ListWildCardTemp
      (* List of object values *)
      | ListValueTemp of string * (Template option)
      | TupleTemp of Template list
      | ApplTemp of Template * Template
	(* QuantifierTemp(vars,body,context) *)
      | ForallTemp of Template list * Template * Template
      | ExistsTemp of Template list * Template * Template
      (* Context variable C[T] that contains an object matching T inside *)
      | ContextTemp of string * Template
      (* T1 : T2 - an object matching T1 has a type matching T2 *)
      | TypedTemp of Template * Template
      (* T :: pi - a template constrained to certain objects *)
      | ConstrainedTemp of Template * TempConstrain
      (* A new Skolem constant; doesn't match anything, only used for instantiation *)
      | SkolemTemp
      (* Matches or instantiates this very constant *)
      | ConstantTemp of TemplateValue
      (* Assigns a value to a string that matches a template *)
      | NamedTemp of string * Template

    (* Extra constraints to the templates (possibly taken from the rule's parameters) *)
    and TempConstrain =
	(* No restriction *)
	AnyConstr
      (* Model value *)
      | ModelConstr
      (* Program Context value *)
      | ProgramContextConstr
      (* Initial predicate value *)
      | InitConstr
	(* The template must be a formula matching a specification if given *)
      | FormulaConstr of (SequentPart * (int option)) option
	(* Template is a bound variable possibly matching a template *)
      | BoundConstr of Template option
	(* Any object, possibly equal to the given one *)
      | ObjectConstr of ParseTree option
      (* Value is a parse tree constant (for special cases like parent closures, etc.) *)
      | ParseTreeConstr
      (* The value must be a context *) 
      | ContextConstr
      (* Disjunction of several constraints *)
      | SomeConstr of TempConstrain list
	(* List of elements constrained to TempConstrain *)
      | ListConstr of TempConstrain

    (* Intermediate representation of expressions (ParseTree) using some sort of indexing *)

    (* Defines the term and all of its parents (terms that contain it
       as an immediate subterm).  The parents are used to determine
       quickly all the term's occurrences. *)
    and TermID = TermID of { ind: TermInd, parents: TermID list ref }

    and Term = (* The actual representation of terms *)
	ConstTerm of ParseTree
      | TupleTerm of TermID list
      | ApplTerm of TermID * TermID
      | TypedTerm of TermID * TermID
      | ForallTerm of { names: TermID list,
		        body: TermID,
			parent: ParseTree }
      | ExistsTerm of { names: TermID list,
		        body: TermID,
			parent: ParseTree }

    (* Values associated with template variables *)
    and TemplateValue =
	ModelValue of Model
      (* Set of initial states of the model, if any *)
      | InitValue of ParseTree option
      (* Context, usually of the current sequent *)
      | ProgramContextValue of ProgramContext
      (* Object can be a formula, or a constant, but not a bound var. *)
      | ObjectValue of TermID
      (* Special constants like parent closures, etc. *)
      | ParseTreeValue of ParseTree
      (* Bound variable *)
      | BoundValue of TermID
      | NamedValue of string * TemplateValue
      (* Value of list variables (ListVarTemp and BoundVarList) *)
      | ListValue of TemplateValue list
      | SubgoalValue of Sequent
      | ContextValue of TemplateValue -> TemplateValue
      | ConstrainedValue of TemplateValue * TempConstrain

    type SequentTemplate = { model: Template,
			     init: Template,
			     context: Template,
			     (* Allow several templates in the formula lists.
			        The order shouldn't be important. *)
			     invar: Template list,
			     assumptions: Template list,
			     conclusions: Template list }
			     
    (* Representation of a set of particular occurrences of a term.
       The `contexts' are paths from the term through its ancestors
       all the way to the top. *)

    type TermOccurs = { ind: TermInd, contexts: TermInd list list }

    (* Hashes that connect indices with terms in both directions *)
    type TermHash = { ind2termHash: (TermInd, Term * TermID) Hash,
		      term2idHash: (Term, TermID) Hash,
		      (* Next free term index for this hash *)
		      nextInd: TermInd ref,
		      (* Unique ID for each hash *)
		      hashID: int }

    (* Term representation of a sequent *)
    type SequentTerm = { model: Trans.Model,
			 init: ParseTree option,
			 context: ProgramObject,
			 invar: TermID list,
			 assumptions: TermID list,
			 conclusions: TermID list,
			 hash: TermHash }

    (* Associate template variable names with values *)
    type Substitution = (string, TemplateValue) Hash * TermHash

    (* Default value of the inf. rule argument *)
    datatype DefaultArgValue =
	(* Match the template with the sequent, if it makes sense.
	   E.g. formula number doesn't need a template, and then it's NONE. *)
	DefaultMatch of Template option
	(* Use this static value *)
      | DefaultValue of InferenceRuleArgumentValue
	(* No default value, the user is required to supply the value *)
      | DefaultRequired

    type InferenceRuleArgumentSpec =
	{ name: string,
	  Type: InferenceRuleArgumentType,
	  (* When default is NONE, the argument is required, otherwise it's optional *)
	  default: DefaultArgValue }

    (* Internal representation of compiled rule arguments *)
    datatype RuleArgValue =
	ArgString of string
      | ArgNumber of int
      | ArgSubgoal of string
      | ArgFormulaNum of SequentPart * (int option)
      | ArgObject of ProgramObject
      | ArgList of RuleArgValue list

    type RuleArg = {name: string, value: RuleArgValue}

    fun IRArgSpecName({name=name,...}: InferenceRuleArgumentSpec) = name
    fun IRArgSpecType({Type=tp,...}: InferenceRuleArgumentSpec) = tp

    (* Printing of the inference rule argument, for help and debugging *)
    fun RuleArgValue2string (ArgString s) = s
      | RuleArgValue2string (ArgNumber n) = Int.toString n
      | RuleArgValue2string (ArgSubgoal s) = s
      | RuleArgValue2string (ArgFormulaNum pair) = formulaInd2str pair
      | RuleArgValue2string (ArgObject (pt, _)) = pt2string pt
      | RuleArgValue2string (ArgList lst) = 
	  "("^(strlist2str " " (List.map RuleArgValue2string lst))^")"

    fun RuleArg2string{name=name, value=v} =
	  name^" : "^(RuleArgValue2string v)


    (* The `conclusion' is a sequent template that may contain free variables, some of them
       from `params', and they are instantiated to match the actual sequent to which this
       rule is applied.  `match' complements `conclusion' or replaces it.  Similarly,
       `premisses' are new sequent templates that are instantiated with values that
       variables in `conclusion' receive.  `apply' function, if present, complements or
       replaces `premisses'. *)

    type InferenceRule = { name: string,
			   help: string, (* Help text *)
			   params: InferenceRuleArgumentSpec list,
			   conclusion: SequentTemplate option,
			   premisses: SequentTemplate list option,
			   (* Given a concrete sequent, try to match it with the conclusion.
			      May apply some other rule-specific tests.
			      Without this function, match the conclusion in the standard way. *)
			   match: (Sequent * ProgramContext * (RuleArg list)
				   * (Substitution option) -> Substitution option) option,
			   (* Apply the rule to a concrete sequent with the list of arguments and
			      the substitution computed by the match function.  Return NONE if
			      failed, or a list of new sequents.

			      The first argument is a function to print info as the rule
			      is being applied (progress report, comments, etc.) *)
			   apply: ((string -> unit)
				   -> Sequent * ProgramContext * (RuleArg list) * Substitution
				   ->  InferenceRuleResult) option }

    type ProofSystemCommandResult = unit

    type ProofSystemCommand = { name: string,
			        help: string,
				params: InferenceRuleArgumentSpec list,
				apply: (string -> unit)
                                         -> Sequent * (InferenceRuleArgument list)
                                         -> ProofSystemCommandResult option }

    fun listEq eq (lst1,lst2) =
	(case zipOpt(lst1, lst2) of
	     NONE => false
	   | SOME l => List.all eq l)

    (* Check if l1 is a subset of l2 w.r.t. the equality `eq'.
       Actually, `eq' could be an inequality, yielding the obvious
       extended notion of subset. *)
    fun subset eq (l1,l2) = List.all(fn x=>List.exists (fn y=>eq(x,y)) l2) l1
    fun eqset eq (l1,l2) = subset eq (l1,l2) andalso subset eq (l2,l1)

    (* Set-like operations on lists *)
    fun intersectLists eq (l1,l2) = List.filter(fn x=>List.exists(fn y=>eq(x,y)) l2) l1
    fun uniteLists eq (l1,l2) = (List.filter(fn x=>not(List.exists(fn y=>eq(x,y)) l2)) l1)@l2

    (* Extract sequent part from a UIArg specification *)
    fun UIArg2FormulaIndex(UInumber n) =
	  if n >= 0 then (ConclusionPart, SOME n)
	  else (AssumptionPart, SOME(~n))
      | UIArg2FormulaIndex(UIstring str) =
	  (case String.explode str of
	       [#"*"] => (AnyPart, NONE)
	     | [#"-"] => (AssumptionPart, NONE)
	     | [#"+"] => (ConclusionPart, NONE)
	     | [#"i"] => (InvarPart, NONE)
	     | #"i"::digits => 
		   ((case Int.fromString(String.implode digits) of
			 NONE => raise ProverError("Bad formula number: "^str)
		       | SOME n => (InvarPart, SOME n))
			 handle _ => raise ProverError
			     ("Too large formula number: "^str))
	     | #"*"::digits => 
		   ((case Int.fromString(String.implode digits) of
			 NONE => raise ProverError("Bad formula number: "^str)
		       | SOME n => (AnyPart, SOME n))
			 handle _ => raise ProverError
			     ("Too large formula number: "^str))
	     | #"-"::digits => 
		   ((case Int.fromString(String.implode digits) of
			 NONE => raise ProverError("Bad formula number: "^str)
		       | SOME n => (AssumptionPart, SOME n))
			 handle _ => raise ProverError
			     ("Too large formula number: "^str))
	     | digits as (_::_) => 
		   ((case Int.fromString(String.implode digits) of
			 NONE => raise ProverError("Bad formula number: "^str)
		       | SOME n => (ConclusionPart, SOME n))
			 handle _ => raise ProverError
			     ("Too large formula number: "^str))
	     | _ =>  raise ProverError
			 ("Bad formula number: \""^str^"\""))
      | UIArg2FormulaIndex arg = raise ProverError
	       ("Bad formula number: "^(UIArg2string arg))

    fun FormulaIndex2UIArg part = UIstring(formulaInd2str part)
	    

    (* Functions for manipulating terms *)
    fun termHashID({hashID=id,...}: TermHash) = id

    fun termEq (ConstTerm e1, ConstTerm e2) = ptEq(e1,e2)
      | termEq (TupleTerm l1, TupleTerm l2) =
	  (case zipOpt(l1, l2) of
	       SOME pairs => List.all (op =) pairs
	     | NONE => false)
      | termEq (ApplTerm(x1,y1),ApplTerm(x2,y2)) =
	  x1 = x2 andalso y1 = y2
      | termEq (TypedTerm(x1,y1),TypedTerm(x2,y2)) =
	  x1 = x2 andalso y1 = y2
      (* The order of bound vars doesn't matter, nor does parent closure *)
      | termEq (ForallTerm{names=n1, body=b1, parent=p1},
		ForallTerm{names=n2, body=b2, parent=p2}) =
	  eqset (op =) (n1,n2) andalso b1 = b2
      | termEq (ExistsTerm{names=n1, body=b1, parent=p1},
		ExistsTerm{names=n2, body=b2, parent=p2}) =
	  eqset (op =) (n1,n2) andalso b1 = b2
      | termEq _ = false

    fun termIDeq(TermID{ind=i1, ...}, TermID{ind=i2,...}) = i1 = i2

    fun termID2ind(TermID{ind=ind,...}) = ind
    fun termInd2str ind = Int.toString ind
    fun termID2str(TermID{ind=ind, parents=ref p}) = Int.toString ind
(*    fun termID2str(TermID{ind=ind, parents=ref p}) =
	  "["^(Int.toString ind)^"<-"
	  ^(strlist2str "," (List.map (Int.toString o termID2ind) p))^"]" *)
    fun term2strHash (ConstTerm p) = pt2stringDebug p
      | term2strHash (TupleTerm lst) = "Tuple("^(strlist2str ", " (List.map termID2str lst))^")"
      | term2strHash (ApplTerm(t1,t2)) = "Appl("^(termID2str t1)^", "^(termID2str t2)^")"
      | term2strHash (TypedTerm(t1,t2)) = "Typed("^(termID2str t1)^", "^(termID2str t2)^")"
      | term2strHash (ForallTerm{names=n, body=b, ...}) =
	  "Forall{"^(strlist2str ", " (List.map termID2str n))^"}("^(termID2str b)^")"
      | term2strHash (ExistsTerm{names=n, body=b, ...}) =
	  "Exists{"^(strlist2str ", " (List.map termID2str n))^"}("^(termID2str b)^")"

    (* String representation of templates and constraints (for debugging) *)
    fun temp2str (FormulaTemp n) = "<formula "^n^">"
      | temp2str (ListFormulaTemp(n, tOpt)) =
	  "<formulas "^n
	  ^(case tOpt of
		SOME t => " ["^(temp2str t)^"]"
	      | NONE => "")
	  ^">"
      | temp2str (BuiltinTemp pt) = pt2string pt
      | temp2str (ValueTemp n) = "<value "^n^">"
      | temp2str (ProgramContextTemp n) = "<program context "^n^">"
      | temp2str WildCardTemp = "<_>"
      | temp2str ListWildCardTemp = "<__>"
      | temp2str (ListValueTemp(n, tOpt)) =
	  "<values "^n
	  ^(case tOpt of
		SOME t => " ["^(temp2str t)^"]"
	      | NONE => "")
	  ^">"
      | temp2str (TupleTemp lst) = "("^(strlist2str ", " (List.map temp2str lst))^")"
      | temp2str (ApplTemp(t1,t2)) = "("^(temp2str t1)^" "^(temp2str t2)^")"
      | temp2str (ForallTemp(vars, body, _)) =
	  "(forall "^(strlist2str ", " (List.map temp2str vars))^": "^(temp2str body)^")"
      | temp2str (ExistsTemp(vars, body, _)) =
	  "(exists "^(strlist2str ", " (List.map temp2str vars))^": "^(temp2str body)^")"
      | temp2str (ContextTemp(n, t)) = "<context "^n^"["^(temp2str t)^"]>"
      | temp2str (TypedTemp(t1,t2)) = "("^(temp2str t1)^": "^(temp2str t2)^")"
      | temp2str (ConstrainedTemp(t,c)) = "("^(temp2str t)^"::"^(constr2str c)^")"
      | temp2str SkolemTemp = "<Skolem>"
      | temp2str (ConstantTemp v) = "<constant "^(value2str v)^">"
      | temp2str (NamedTemp(name, t)) = "<named "^name^" = "^(temp2str t)^">"

    and constr2str AnyConstr = "<*>"
      | constr2str ModelConstr = "<model>"
      | constr2str InitConstr = "<init>"
      | constr2str ProgramContextConstr = "<program context>"
      | constr2str (FormulaConstr indOpt) = 
	"<formula"
	^(case indOpt of
	      SOME ind => "["^(formulaInd2str ind)^"]"
	    | NONE => "")^">"
      | constr2str (BoundConstr tOpt) =
	  "<bound"
	  ^(case tOpt of
		SOME t => "["^(temp2str t)^"]"
	      | NONE => "")^">"
      | constr2str (ObjectConstr ptOpt) =
	  "<object"
	  ^(case ptOpt of
		SOME pt => "["^(pt2string pt)^"]"
	      | NONE => "")^">"
      | constr2str ParseTreeConstr = "<ParseTree>"
      | constr2str ContextConstr = "<context>"
      | constr2str (SomeConstr lst) = 
	  "SomeConstr("^(strlist2str " | " (List.map constr2str lst))^")"
      | constr2str (ListConstr c) = "<"^(constr2str c)^" list>"

    (* Give a unique string representation of a term (for hashing and debugging) *)
    (* String representation of values (for debugging) *)
    and value2str (ModelValue m) = "MODEL = "^(model2str m)
      | value2str (InitValue ptOpt) = "INIT"^(case ptOpt of
						  SOME pt => " = "^(pt2string pt)
						| NONE => "")
      | value2str (ProgramContextValue(c,_)) = "ProgramContextValue("^(pt2string c)^")"
      | value2str (ObjectValue id) = "ObjectValue("^(termID2str id)^")"
      | value2str (ParseTreeValue pt) = (pt2string pt)
      | value2str (BoundValue id) = "BoundValue("^(termID2str id)^")"
      | value2str (NamedValue(n,v)) = "("^n^"="^(value2str v)^")"
      | value2str (ListValue lst) = "["^(strlist2str ", " (List.map value2str lst))^"]"
      | value2str (SubgoalValue s) = "SubgoalValue"
      | value2str (ContextValue ff) = 
	  "ContextValue(fn X => "^(value2str(ff(ParseTreeValue(Id(dp,"X")))))^")"
      | value2str (ConstrainedValue(v,c)) = "("^(value2str v)^"::"^(constr2str c)^")"

    fun termHash2str(tHash: TermHash) =
	let val { ind2termHash=iHash, hashID=id, ... } = tHash
	    fun pair2str(_, (term, id)) = "<id="^(termID2str id)^", "^(term2strHash term)^">"
	in
	    "TermHash[id="^(Int.toString id)^", "
	    ^(strlist2str ", " (List.map pair2str (hash2any(fn x=>x)(fn x=>x) iHash)))
	    ^"]"
	end

    (* Substitution -> string, mostly for debugging *)
    fun subst2str((hash, termHash): Substitution) =
	let fun val2str(n,v) = "("^n^"="^(value2str v)^")"
	    val vals = hash2any(fn x=>x)(fn x=>x) hash
	in
	    "Substitution{hash=["^(strlist2str "," (List.map val2str vals))
	    ^"],\n            "^(termHash2str termHash)^"}\n"
	end

    fun seqTemp2str(seqTemp: SequentTemplate) =
	let val { model = model,
		  init = init,
		  context = context,
		  invar = invar,
		  assumptions = assump,
		  conclusions = conc } = seqTemp
	in
	    "SequentTemplate{model = "^(temp2str model)
	    ^",\n                init = "^(temp2str init)
	    ^",\n                context = "^(temp2str context)
	    ^",\n                invar = ["^(strlist2str ", " (List.map temp2str invar))
	    ^"],\n                assumptions = ["^(strlist2str ", " (List.map temp2str assump))
	    ^"],\n                conclusions = ["^(strlist2str ", " (List.map temp2str conc))
	    ^"] }"
	end
					      
    fun string2hints _ = ()

    local val idRef = ref 0
    in
	fun makeTermHash() =
	    ({ ind2termHash=makeHashDefault(op =, Int.toString),
	       term2idHash=makeHashDefault(termEq, term2strHash),
	       nextInd=ref 0,
	       hashID = !idRef }: TermHash) before (idRef := (!idRef) + 1)

	fun copyTermHash({ ind2termHash=iHash,
			   term2idHash=tHash,
			   nextInd=ref i,...}: TermHash) =
	    ({ ind2termHash=copyHash iHash,
	       term2idHash=copyHash tHash,
	       nextInd=ref i,
	       hashID = !idRef}: TermHash) before (idRef := (!idRef) + 1)
    end

    (* Extract the list of all immediate subterm IDs of a term *)
    fun subtermIDs (ConstTerm _) = []
      | subtermIDs (TupleTerm lst) = lst
      | subtermIDs (ApplTerm(t1,t2)) = [t1,t2]
      | subtermIDs (TypedTerm(t1,t2)) = [t1,t2]
      | subtermIDs (ForallTerm{names=n, body=b,...}) = b::n
      | subtermIDs (ExistsTerm{names=n, body=b,...}) = b::n

    (* Convert terms back to ParseTree, together with some other useful conversions *)
    local 
	fun ind2pair hash ind =
	    (case findHash(hash, ind) of
		 SOME x => x
	       | NONE => raise SympBug
		     ("ind2pair: term index not in hash: "^(termInd2str ind)))
	fun ind2term hash ind = let val (t,_) = ind2pair hash ind in t end
	fun ind2id hash ind = let val (_,id) = ind2pair hash ind in id end
	fun id2term hash (id as TermID{ind=ind,...}) = ind2term hash ind
	fun id2pt hash id = loop hash (id2term hash id)
	and loop hash (ConstTerm p) = p
	  | loop hash (TupleTerm lst) = TupleExpr(dp, List.map (id2pt hash) lst)
	  | loop hash (ApplTerm(t1,t2)) = Appl(dp, id2pt hash t1, id2pt hash t2)
	  | loop hash (TypedTerm(t1,t2)) = TypedExpr(dp, id2pt hash t1, id2pt hash t2)
	  | loop hash (ForallTerm{names=n, body=b, parent=p}) =
	      ForallClosure{names=List.map (id2pt hash) n,
			    body=id2pt hash b, parent=p}
	  | loop hash (ExistsTerm{names=n, body=b, parent=p}) =
	      ExistsClosure{names=List.map (id2pt hash) n,
			    body=id2pt hash b, parent=p}
    in
	fun term2pt ({ind2termHash=hash,...}: TermHash) term = loop hash term
	fun termID2pt ({ind2termHash=hash,...}: TermHash) term = id2pt hash term
	fun termInd2termID ({ind2termHash=hash,...}: TermHash) ind = ind2id hash ind
	fun termInd2pt ({ind2termHash=hash,...}: TermHash) ind = id2pt hash (ind2id hash ind)
	fun termInd2term ({ind2termHash=hash,...}: TermHash) ind = ind2term hash ind
	fun termID2term ({ind2termHash=hash,...}: TermHash) id = id2term hash id
    end


    (* Check if the term is already in the hash and return its ID; otherwise
       generate a new ID and insert it into the hash *)
    fun term2termID (termHash: TermHash) t =
	let val debug = lazyVerbDebug (getOptions()) "term2termID"
	    val { ind2termHash=iHash,
		  term2idHash=tHash,
		  nextInd=nextInd, ... } = termHash
	    val _ = pushFunStackLazy("term2termID", 
				     fn()=>((term2strHash t)
					    ^",\n  "^(termHash2str termHash)))
	    val res = 
		(case findHash(tHash, t) of
		     SOME id => (debug(fn()=>"term2termID: term is in hash\n");
				 id)
		   | NONE =>
			 let val ind = !nextInd
			     val id = TermID{ind=ind, parents=ref[]}
			     val _ = debug(fn()=>"term2termID: term is not in hash\n")
			 in id before (insertHashDestructive(tHash,t,id);
				       insertHashDestructive(iHash,ind,(t,id));
				       nextInd := ind+1)
			 end)
	    val _ = popFunStackLazy("term2termID", fn()=>("termHash = "^(termID2str res)))
	in
	    res
	end

    (* Convert ParseTree to the Term representation, DESTRUCTIVELY
       modifying the given TermHash and returning the termID.  If the
       term is not yet in the hash, its list of parents is empty. *)
    fun pt2termID (hash: TermHash) pt =
	let val { ind2termHash=iHash,
		  term2idHash=tHash,
		  nextInd=nextInd, ... } = hash
	    (* DESTRUCTIVELY insert id into the parent list, if not already there *)
	    fun insertParent (id as TermID{ind=ind,...}) (TermID{parents = parents,...}) =
		    (case List.find(fn TermID{ind=i,...} => i=ind)(!parents) of
			 SOME _ => ()
		       | NONE => parents := id::(!parents))
		(* For each subterm, DESTRUCTIVELY insert the parent's
		   id into its list, if not already there *)
	    fun updateParents id term = List.app (insertParent id) (subtermIDs term)
	    (* Descend through the parse tree and match existing terms bottom up *)
	    fun update term =
		let val id = term2termID hash term
		in id before (updateParents id term)
		end
	    and loop (TupleExpr(_, lst)) = update(TupleTerm(List.map loop lst))
	      | loop (Appl(_, x, y)) = update(ApplTerm(loop x, loop y))
	      | loop (TypedExpr(_, x, y)) = update(TypedTerm(loop x, loop y))
	      | loop (ForallClosure{names=n, body=b, parent=p}) =
		  update(ForallTerm{names = List.map loop n,
				    body = loop b,
				    parent=p})
	      | loop (ExistsClosure{names=n, body=b, parent=p}) =
		  update(ExistsTerm{names = List.map loop n,
				    body = loop b,
				    parent=p})
	      | loop (Object{def=def,...}) = loop def
	      | loop e = update(ConstTerm e)
	in
	    loop pt
	end

    (* Substitution of term IDs: replace all occurances of oldID by
       newID in the term represented by the `id', and return the new
       term ID.  No existing IDs or terms are replaced. *) 
    fun substTermID termHash (oldID,newID) id =
	let val debug = lazyVerbDebug (getOptions()) "substTermID"
	    val _ = pushFunStackLazy("substTermID",
				     fn()=>((termID2str oldID)^" => "
					    ^(termID2str newID)^", "^(termID2str id)
					    ^" = "^(pt2string(termID2pt termHash id))))
	    (* val pt = termID2pt termHash id
	    val oldPt = termID2pt termHash oldID
	    val newPt = termID2pt termHash newID *)
	    fun recur (t as ConstTerm p) = t
	         (* ConstTerm(substitute (getOptions()) (newPt, oldPt, pt)) *)
	      | recur (TupleTerm lst) = TupleTerm(List.map replace lst)
	      | recur (ApplTerm(id1,id2)) = ApplTerm(replace id1, replace id2)
	      | recur (TypedTerm(id1,id2)) = TypedTerm(replace id1, replace id2)
	      | recur (ForallTerm{names=names, body=body, parent=p}) =
	         ForallTerm{names=List.map replace names,
			    body=replace body,
			    parent=p}
	      | recur (ExistsTerm{names=names, body=body, parent=p}) =
	         ExistsTerm{names=List.map replace names,
			    body=replace body,
			    parent=p}
	    and replace id =
		let val _ = pushFunStackLazy("substTermID/replace",
					     fn()=>((termID2str id)^" = "
						    ^(pt2string(termID2pt termHash id))))
		    val res = if id = oldID then newID
			      else term2termID termHash (recur(termID2term termHash id))
		    val _ = popFunStackLazy("substTermID/replace",
					    fn()=>((termID2str res)^" = "
						   ^(pt2string(termID2pt termHash res))))
		in
		    res
		end
	    val res = replace id
	    val _ = popFunStackLazy("substTermID",
				    fn()=>((termID2str res)^" = "
					   ^(pt2string(termID2pt termHash res))
					   ^",\n  "^(termHash2str termHash)))
	in
	    res
	end

    (* Convert the TermID tree into the set of term occurrences `TermOccurs' *)
    fun termID2TermOccurs (termId as TermID{ind=ind,...}) =
	let fun loop stack (TermID{ind=ind, parents=ref []}) = [List.rev(ind::stack)]
	      | loop stack (TermID{ind=ind, parents=ref lst}) =
	          List.foldr(op @) [] (List.map(loop(ind::stack)) lst)
	in
	    {ind=ind, contexts = loop [] termId}: TermOccurs
	end

    fun term2TermOccurs hash term = termID2TermOccurs(term2termID hash term)	

    (* Convert Sequent into SequentTerm representation *)
    fun sequent2term(seq: Sequent, hashOpt) =
	let val { model=model,
		  init=init,
		  context=context,
		  invar=invar,
		  assumptions=assump,
		  conclusions=conc } = seq
	    val hash = (case hashOpt of
			    SOME hash => hash
			  | NONE => makeTermHash())
	    val invarTerm = List.map (pt2termID hash) invar
	    val assumpTerm = List.map (pt2termID hash) assump
	    val concTerm = List.map (pt2termID hash) conc
	in
	    { model = model,
	      init = init,
	      context=context,
	      invar = invarTerm,
	      assumptions = assumpTerm,
	      conclusions = concTerm,
	      hash = hash }: SequentTerm
	end

    fun makeSequent context (Theorem(_,name,Models(_,model,thm)), findObject) =
	{ model = makeModel(model,findObject), (* handle TransError str => raise SympError str, *)
	  init = NONE,
	  context = context,
	  invar = [],
	  assumptions = [],
	  conclusions = [thm] }: Sequent
      | makeSequent _ (x,_) = raise SympBug
	("makeSequent (default): not a Theorem:\n  "
	 ^(pt2string x))

(*      fun getHints({hints=hints,...}: Sequent) = hints *)
    fun hintsEq(h1,h2) = (h1 = h2)

    fun sequentEq({model=m1, init=init1, context=cxt1, invar=invar1,
		   assumptions=p1, conclusions=c1}: Sequent,
		  {model=m2, init=init2, context=cxt2, invar=invar2,
		   assumptions=p2, conclusions=c2}: Sequent) =
	modelEq(m1,m2) 
	andalso (case (init1,init2) of
		     (NONE, NONE) => true
		   | (SOME x1, SOME x2) => ptEq(x1,x2)
		   | _ => false)
	andalso ptEq(#1 cxt1, #1 cxt2)
	andalso eqset ptEq (invar1,invar2)
	andalso eqset ptEq (p1,p2)
	andalso eqset ptEq (c1,c2)

    (* FIXME: make this more sane (however, it's already sound). *)
    val sequentSubsumes = sequentEq	

    (* Each rule has a unique string identifier *)
    fun getRuleName ({name=name,...}:InferenceRule) = name
    fun getCommandName ({name=name,...}:ProofSystemCommand) = name

    (* Each rule has a set of arguments *)
    fun getRuleArgs ({params=params,...}:InferenceRule) = params
    fun getCommandArgs ({params=params,...}:ProofSystemCommand) = params

    (* Surround the string `s' (usually the name and type of the
       argument) with default value annotations *)
    fun defaultArgValue2str s d =
	let fun d2pair(DefaultMatch _) = ("&optional ", "")
	      | d2pair(DefaultValue v) = ("&optional ", " [= "^(InfRuleArgValue2string v)^" ]")
	      | d2pair DefaultRequired = ("","")
	    val (pre, post) = d2pair d
	in pre^s^post
	end

    fun InfRuleSpec2str({name=name,
			 Type=t,
			 default=d}: InferenceRuleArgumentSpec) =
	let val NAME = String.implode(List.map Char.toUpper (String.explode name))
	in
	    defaultArgValue2str (NAME^": "^(InfRuleArgType2string t)) d
	end

    fun formatComm command =
	let val {interface=UI,...} = getOptions()
	in formatProverCommand UI command
	end

    fun getRuleHelp ({help=help,
		      name=name,
		      params=params, ...}:InferenceRule) =
	(formatComm(name, List.map InfRuleSpec2str params)^"\n\n"^help)

    fun getCommandHelp ({help=help,
			 name=name,
			 params=params, ...}:ProofSystemCommand) =
	(formatComm(name, List.map InfRuleSpec2str params)^"\n\n"^help)

    (* Unification and substitution code *)

    local
	fun loopC (AnyConstr, AnyConstr) = true
	  | loopC (ProgramContextConstr, ProgramContextConstr) = true
	  | loopC (FormulaConstr x1, FormulaConstr x2) = x1 = x2
	  | loopC (BoundConstr x1, BoundConstr x2) =
	      (case (x1,x2) of
		   (NONE, NONE) => true
		 | (SOME t1, SOME t2) => loop(t1,t2)
		 | _ => false)
	  | loopC (ObjectConstr x1, ObjectConstr x2) =
	      (case (x1,x2) of
		   (NONE, NONE) => true
		 | (SOME p1, SOME p2) => ptEq(p1,p2)
		 | _ => false)
	  | loopC (ListConstr l1, ListConstr l2) = loopC (l1,l2)
	  | loopC (SomeConstr l1, SomeConstr l2) = eqset loopC (l1,l2)
	  | loopC (SomeConstr [x], y) = loopC(x,y)
	  | loopC (x, SomeConstr [y]) = loopC(x,y)
	  | loopC _ = false
	and loop (FormulaTemp s1, FormulaTemp s2) = s1 = s2
	  | loop (ListFormulaTemp(s1, t1), ListFormulaTemp(s2, t2)) =
	      s1 = s2 andalso
	      (case (t1, t2) of
		   (NONE,NONE) => true
		 | (SOME x1, SOME x2) => loop(x1,x2)
		 | _ => false)
	  | loop (BuiltinTemp p1, BuiltinTemp p2) = ptEq(p1,p2)
	  | loop (ValueTemp s1, ValueTemp s2) = s1 = s2
	  | loop (ProgramContextTemp n1, ProgramContextTemp n2) = n1 = n2
	  | loop (WildCardTemp, WildCardTemp) = true
	  | loop (ListWildCardTemp, ListWildCardTemp) = true
	  | loop (ListValueTemp(s1, t1), ListValueTemp(s2,t2)) =
	      s1 = s2 andalso
	      (case (t1, t2) of
		   (NONE,NONE) => true
		 | (SOME x1, SOME x2) => loop(x1,x2)
		 | _ => false)
	  | loop (TupleTemp lst1, TupleTemp lst2) = listEq loop (lst1, lst2)
	  | loop (ApplTemp(x1,y1), ApplTemp(x2,y2)) =
	      loop(x1,x2) andalso loop(y1,y2)
	  | loop (ForallTemp(n1, b1, c1), ForallTemp(n2, b2, c2)) =
	      listEq loop (n1,n2) andalso loop(b1,b2) andalso loop(c1,c2)
	  | loop (ExistsTemp(n1, b1, c1), ExistsTemp(n2, b2, c2)) =
	      listEq loop (n1,n2) andalso loop(b1,b2) andalso loop(c1,c2)
	  | loop (ContextTemp(s1, t1), ContextTemp(s2, t2)) =
	      s1 = s2 andalso loop(t1,t2)
	  | loop (TypedTemp(x1, y1), TypedTemp(x2, y2)) =
	      loop(x1,x2) andalso loop(y1,y2)
	  | loop (ConstrainedTemp(t1, c1), ConstrainedTemp(t2, c2)) =
	      loop(t1,t2) andalso loopC(c1,c2)
	  | loop (SkolemTemp, SkolemTemp) = true
	  | loop (ConstantTemp v1, ConstantTemp v2) = loopV(v1,v2)
	  | loop (NamedTemp(n1, t1), NamedTemp(n2, t2)) =
	      n1 = n2 andalso loop(t1, t2)
	  | loop _ = false

	and loopV (ModelValue m1, ModelValue m2) = modelEq(m1,m2)
	  | loopV (InitValue i1, InitValue i2) =
	    (case (i1, i2) of
		 (NONE, NONE) => true
	       | (SOME p1, SOME p2) => ptEq(p1,p2)
	       | _ => false)
	  | loopV (ProgramContextValue(c1,_), ProgramContextValue(c2,_)) = ptEq(c1,c2)
	  | loopV (ObjectValue id1, ObjectValue id2) = termIDeq(id1, id2)
	  | loopV (ParseTreeValue p1, ParseTreeValue p2) = ptEq(p1, p2)
	  | loopV (BoundValue id1, BoundValue id2) = termIDeq(id1, id2)
	  | loopV (NamedValue(n1, v1), NamedValue(n2, v2)) =
	      n1 = n2 andalso loopV(v1, v2)
	  | loopV (ListValue l1, ListValue l2) = listEq loopV (l1,l2)
	  (* Finish this.  Although, this case is hardly needed. *)
	  | loopV (SubgoalValue _, _) = false
	  (* We assume that the context function doesn't modify the argument,
	   only substitutes it.  So, we plug in something simple as a test. *)
	  | loopV (ContextValue f1, ContextValue f2) =
	      loopV(f1(ParseTreeValue Fake), f2(ParseTreeValue Fake))
	  | loopV (ConstrainedValue(v1, c1), ConstrainedValue(v2, c2)) =
	      loopV(v1, v2) andalso loopC(c1, c2)
	  | loopV _ = false
    in
	fun TemplateEq(t1, t2) = loop(t1, t2)
	fun TempConstrEq(c1, c2) = loopC(c1, c2)
	fun valueEq(v1, v2) = loopV(v1, v2)
    end

    (* Partial order on Template Constraints; implements the "\pi_1 <= \pi_2" sequent. *)
    fun TempConstrLE(c1, c2) =
	let fun spLE (_, AnyPart) = true
	      | spLE (SomeParts l1, SomeParts l2) = subset (op =) (l1,l2)
	      | spLE (p1, SomeParts l2) = subset (op =) ([p1], l2)
	      | spLE (p1, p2) = p1 = p2
	    fun alphaLE((sp1, n1), (sp2,n2)) =
	         spLE(sp1,sp2) andalso
		 (case (n1,n2) of
		      (_, NONE) => true
		    | (SOME x1, SOME x2) => x1 = x2
		    | _ => false)
	    fun loop (_, AnyConstr) = true (* Top of the lattice *)
	      (* Nothing matches the empty disjunctive constraint, 
	         so it's the bottom of the lattice *)
	      | loop (SomeConstr [], x) = true
	      | loop (SomeConstr l1, SomeConstr l2) = subset loop (l1,l2)
	      | loop (SomeConstr [x], y) = loop (x,y)
	      | loop (x, SomeConstr l) = subset loop ([x], l)
	      | loop (FormulaConstr(SOME _), FormulaConstr NONE) = true
	      | loop (FormulaConstr(SOME a1), FormulaConstr(SOME a2)) = alphaLE(a1,a2)
	      | loop (BoundConstr(SOME _), BoundConstr NONE) = true
	      | loop (ObjectConstr(SOME _), ObjectConstr NONE) = true
	      | loop (BoundConstr _, ObjectConstr NONE) = true
	      | loop (FormulaConstr _, ObjectConstr NONE) = true
	      | loop (ListConstr c1, ListConstr c2) = loop(c1,c2)
	      | loop (c1,c2) = TempConstrEq(c1,c2)
	in
	    loop(c1,c2)
	end

    (* Pick the minimal constraint in the preorder (the strongest), if they are comparable. *)
    fun TempConstrMin(c1, c2) =
	if TempConstrLE(c1, c2) then SOME c1
	else if TempConstrLE(c2, c1) then SOME c2
	else NONE

    (* Flatten all the ListValue's into one single list *)
    fun flattenValue((ListValue lst)::tl) = (flattenValue lst)@(flattenValue tl)
      | flattenValue (x::tl) = x::(flattenValue tl)
      | flattenValue [] = []

    fun makeSubstitution(hash) = (makeHashDefault(op =, fn x=>x), hash): Substitution

    (* NOTE: termHash is NOT copied (it doesn't make sense to do so);
       it is expected that the set of terms is fixed at the time of
       generating a substitution. *)
    fun copySubstitution((hash, termHash): Substitution) =
	  (copyHash hash, termHash): Substitution

    (* Raw update to the substitution; nothing is checked, be careful with it. *)
    fun updateSubstitutionRaw(name, v)(sub as (hash, termHash): Substitution) =
	let val _ = pushFunStackLazy("updateSubstitutionRaw",
				     fn()=>(subst2str sub))
	    val res = (insertHashDestructive(hash, name, v), termHash)
	    val _ = popFunStackLazy("updateSubstitutionRaw",
				    fn()=>(subst2str res))
	in res
	end

    (* Adds a new name substitution to the current `sub', updates the
       given substitution DESTRUCTIVELY, and returns it.  If the name
       already defined, it must be exactly the same as `v'; if not,
       return NONE. *)

    fun updateSubstitution(name, v)(sub as (hash, _): Substitution) =
	(case findHash(hash, name) of
	     SOME v' => if valueEq(v,v') then SOME sub else NONE
	   | NONE => (insertHashDestructive(hash, name, v);
		      SOME sub))

    (* Update a list variable: if it exists, it's value must be
       ListValue, and the list of new values is added to the list. *)

    fun updateListSubstitution(name, vList)(sub as (hash, _): Substitution) =
	let val debug = lazyVerbDebug(getOptions()) "updateSubstitution"
	    val _ = pushFunStackLazy("updateListSubstitution",
				     fn()=>(name^", "
					    ^(value2str(ListValue vList))))
	    val res = 
		(case findHash(hash, name) of
		     SOME(v' as ListValue vList') =>
			 (debug(fn()=>"updateListSubstitution: merging with old value "
				^(value2str v')^"\n");
			  insertHashDestructive(hash, name,
						ListValue(uniteLists valueEq (vList,vList')));
			  SOME sub)
		   | SOME v' => (debug(fn()=>"updateListSubstitution: non-list value: "
				      ^(value2str v')^"\n");
				 NONE)
		   | NONE => (insertHashDestructive(hash, name, ListValue vList);
			      SOME sub))
	    val _ = popFunStackLazy("updateListSubstitution",
				    fn()=>(case res of
					       SOME _ => "success"
					     | NONE => "failure"))
	in res
	end

    (* Applies a substitution to a template, obtaining a TemplateValue.  All vars must be
       mapped to values. *)

    fun subst (sub as (sigma, termHash): Substitution) c temp =
	let val debug = lazyVerbDebug (getOptions()) "SequentDefault.subst" 
	    val _ = pushFunStackLazy("SequentDefault.subst",
				     fn()=>((constr2str c)^", "
					    ^(temp2str temp)^",\n  "^(subst2str sub)))
	    fun substName n =
	         (case findHash(sigma, n) of
		      SOME x => x
		    | NONE => raise SympBug
			  ("sequent-default/subst: no such name in the substitution: "^n))
	    fun stripListConstr(ListConstr c) = c
	      | stripListConstr c = c
	    fun stripFormulaConstr(FormulaConstr _) = AnyConstr
	      | stripFormulaConstr(ListConstr c) = ListConstr(stripFormulaConstr c)
	      | stripFormulaConstr(SomeConstr lst) = SomeConstr(List.map stripFormulaConstr lst)
	      | stripFormulaConstr c = c
	    (* Check if the value satisfies the constraint *)
	    fun constrValue c v =
		let val _ = pushFunStackLazy("SequentDefault.subst/constrValue",
					     fn()=>((constr2str c)^", "
						    ^(value2str v)))
		    val res = constrValue' c v
		    val _ = popFunStackLazy("SequentDefault.subst/constrValue",
					    fn()=>(case res of
						       SOME v => value2str v
						     | NONE => "NONE"))
		in res
		end
	    and constrValue' c (res as ConstrainedValue(v, c')) =
		  if TempConstrLE(c',c) then
		      Option.map(fn v=>ConstrainedValue(v, c')) (constrValue c' v)
		  else NONE
	      | constrValue' c (res as NamedValue(n, v)) =
		  Option.map(fn v=>NamedValue(n, v))(constrValue c v)
	      | constrValue' AnyConstr v = SOME v
	      | constrValue' ModelConstr v =
		  (case v of
		       ModelValue _ => SOME v
		     | _ => NONE)
	      | constrValue' InitConstr v =
		  (case v of
		       InitValue _ => SOME v
		     | _ => NONE)
	      | constrValue' ProgramContextConstr v =
		  (case v of
		       ProgramContextValue _ => SOME v
		     | _ => NONE)
	      (* FormulaConstr without a specifier matches any ObjectValue. *)
	      | constrValue' (FormulaConstr NONE) v =
		  (case v of
		       ObjectValue _ => SOME v
		     | _ => NONE)
	      | constrValue' (c as FormulaConstr(SOME _)) v =
		  (case v of
		       ConstrainedValue(v', c') =>
			   if TempConstrLE(c', c) then constrValue (FormulaConstr NONE) v'
			   else NONE
		     | ObjectValue _ => SOME v
		     | _ => NONE)
	      (* The value must be an object expression.  
	         We assume that ContexValue is used only for expressions. *)
	      | constrValue' (ObjectConstr spec) v =
		  let fun checkSpec id = 
		           (case spec of
				NONE => SOME v
			      | SOME pt => if ptEq(pt, termID2pt termHash id) then SOME v
					   else NONE)
		  in 
		      (case v of
			   ObjectValue id => checkSpec id
			 | BoundValue id => checkSpec id
			 | _ => NONE)
		  end
	      | constrValue' (BoundConstr spec) v =
		  (case v of
		       BoundValue id => 
			   (case spec of
				NONE => SOME v
			      (* Eventually we need to match the
			         template with the term `id'somehow *)
			      | SOME temp => NONE)
		     | _ => NONE)
	      | constrValue' ParseTreeConstr v =
		  (case v of
		       ParseTreeValue _ => SOME v
		     | _ => NONE)
	      | constrValue' ContextConstr v =
		  (case v of
		       ContextValue _ => SOME v
		     | _ => NONE)
	      | constrValue' (SomeConstr lst) v =
		  let fun ff [] = NONE
			| ff (c::lst) =
		             (case constrValue c v of
				  SOME x => SOME x
				| NONE => ff lst)
		  in
		      ff lst
		  end
	      | constrValue' (ListConstr c) v =
		  (case v of
		       ListValue lst =>
			   SOME(ListValue(List.mapPartial (constrValue c) (flattenValue lst)))
		     | _ => NONE)
	    (* Extract the term ID from the value, if possible *)
	    fun value2id (ObjectValue id) = id
	      | value2id (BoundValue id) = id
	      | value2id (NamedValue(_,v)) = value2id v
	      | value2id (ConstrainedValue(v,_)) = value2id v
	      | value2id _ = raise SympBug
		  ("sequent-default/subst/value2id: the value doesn't have a term ID")

	    (* The main loop *)
	    fun loop(temp, c) = 
		let val _ = pushFunStackLazy("SequentDefault.subst/loop",
					     fn()=>((temp2str temp)^", "^(constr2str c)))
		      
		    val res = loop'(temp, c)
		    val _ = popFunStackLazy("SequentDefault.subst/loop",
					    fn()=>(case res of
						       SOME v => value2str v
						     | NONE => "NONE"))
		in res
		end
	    and loop'(FormulaTemp n, c) = 
		  constrValue (stripFormulaConstr(stripListConstr c)) (substName n)
	      | loop'(ValueTemp n, c) = constrValue (stripListConstr c) (substName n)
	      | loop'(ProgramContextTemp n, c) = constrValue c (substName n)
	      | loop'(ListFormulaTemp(n,_), c) = constrValue c (substName n)
	      | loop'(BuiltinTemp tree, c) = SOME(ObjectValue(pt2termID termHash tree))
	      | loop'(WildCardTemp,_) = raise SympBug
		  ("sequent-default/subst: WildCardTemp can't be instantiated")
	      | loop'(ListWildCardTemp,_) = raise SympBug
		  ("sequent-default/subst: ListWildCardTemp can't be instantiated")
	      | loop'(ListValueTemp(n,_), c) = constrValue c (substName n)
	      | loop'(TupleTemp lst, c) =
		  let val elems = looplist (lst, ListConstr c)
		      val term = 
			   if List.all isSome (List.map (constrValue (ObjectConstr NONE)) elems)
			   then TupleTerm(List.map value2id elems)
			   else raise SympBug
			       ("Tuple template has non-object elements")
		  in SOME(ObjectValue(term2termID termHash term))
		  end
	      | loop'(ApplTemp(t1,t2), c) = 
		  (case (Option.mapPartial(constrValue (ObjectConstr NONE)) (loop(t1, c)),
			 Option.mapPartial(constrValue (ObjectConstr NONE)) (loop(t2, c))) of
		       (SOME v1, SOME v2) => 
			   let val term = ApplTerm(value2id v1, value2id v2)
			   in SOME(ObjectValue(term2termID termHash term))
			   end
		     | (_, _) => NONE)
	      | loop'(ForallTemp(lst, t, ct), c) =
		  let val vars = looplist(lst, ListConstr(BoundConstr NONE))
		      val body = loop(t,c)
		      val cxt = loop(ct, ParseTreeConstr)
		  in
		    (case (body, cxt) of
			 (SOME b, SOME(ParseTreeValue p)) =>
			   let val varIDs = List.map value2id vars
			       val term = ForallTerm{ names=varIDs,
						      body=value2id b,
						      parent=p }
			   in SOME(ObjectValue(term2termID termHash term))
			   end
		       | _ => raise SympBug
			   ("sequent-default/subst: ForallTemp parts have wrong values:\n  "
			    ^"body = "^(option2string "NONE" (Option.map value2str body))
			    ^"\n  cxt = "^(option2string "NONE" (Option.map value2str cxt))))
		  end
	      | loop'(ExistsTemp(lst, t, ct), c) =
		  let val vars = looplist(lst, ListConstr(BoundConstr NONE))
		      val body = loop(t,c)
		      val cxt = loop(ct, ParseTreeConstr)
		  in
		    (case (body, cxt) of
			 (SOME b, SOME(ParseTreeValue p)) =>
			   let val varIDs = List.map value2id vars
			       val term = ExistsTerm{ names=varIDs,
						      body=value2id b,
						      parent=p }
			   in SOME(ObjectValue(term2termID termHash term))
			   end
		       | _ => raise SympBug
			   ("sequent-default/subst: ExistsTemp parts have wrong values"))
		  end
	      | loop'(ContextTemp(n, t), c) =
		  let val cxt = constrValue ContextConstr (substName n)
		      val v = loop(t, AnyConstr)
		  in
		     case (cxt, v) of
			 (SOME(ContextValue f), SOME v') =>
			     constrValue c (f v')
		       | _ => NONE
		  end
	      | loop'(TypedTemp(t1, t2), c) =
		  (case (loop(t1, c), loop(t2, ObjectConstr NONE)) of
		       (SOME v1, SOME v2) =>
			   let val term = TypedTerm(value2id v1, value2id v2)
			   in SOME(ObjectValue(term2termID termHash term))
			   end
		     | _ => NONE)
	      | loop'(ConstrainedTemp(t, c'), c) = 
		  Option.mapPartial(fn c=>loop(t, c))(TempConstrMin(c', c))
	      (* Generate new Skolem constant - need to figure out where to get the type info *)
	      | loop'(SkolemTemp, c) = NONE
	      | loop'(ConstantTemp v, c) = SOME v
	      | loop'(NamedTemp(n, _), c) = constrValue (stripListConstr c) (substName n)
		  (* let val sk = xxx
		  in SOME(ObjectValue(term2termID termHash (ConstTerm sk)))
		  end *)
	    and looplist(lst, c) =
		  let val _ = pushFunStackLazy
		          ("SequentDefault.subst/looplist",
			   fn()=>("["^(strlist2str "," (List.map temp2str lst))
				    ^"], "^(constr2str c)))
		      fun ff c (t as ListFormulaTemp _) = loop(t, c)
		        | ff c (t as ListValueTemp _) = loop(t, c)
		        | ff (ListConstr c) t = loop(t, c)
		        | ff c t = raise SympBug
		           ("sequent-default/subst/looplist: constraint is not ListConstr")
		      val res = flattenValue(List.mapPartial (ff c) lst)
		      val _ = popFunStackLazy
		          ("SequentDefault.subst/looplist",
			   fn()=>("["^(strlist2str "," (List.map value2str res))))
		  in
		      res
		  end
	    val res = loop (temp, c)
	    val _ = popFunStackLazy("SequentDefault.subst",
				     fn()=>(case res of
						NONE => "NONE"
					      | SOME s => value2str s))
	in
	    res
	end

    (* "Typecheck" the rule's arguments (`InferenceRuleArgument') and
       return SOME error message if they don't match.  Otherwise
       return NONE. *)

    fun checkRuleArgs specs args =
	let (* val specs = getRuleArgs rule *)
            fun findSpec name = List.find(fn {name=x,...} => x=name) specs
	    fun findArg name = List.find(fn {name=x,...} => x=name) args
	    (* Check that every argument has the right type *)
	    fun checkArg {name=n,value=v} =
		 (case (findSpec n, getInfRuleArgType v) of
		      (NONE, _) => SOME("Unrecognized argument: "^n)
		    | (SOME{Type=tp,...}, tp') =>
			  if tp = tp' orelse tp' = ProverArgUnknownType then NONE
			  else SOME("Argument has the wrong type: "
				    ^n^": "^(InfRuleArgType2string tp')^", expected: "
				    ^(InfRuleArgType2string tp)))
	    (* Check that all required args are present *)
	    fun checkDefault {name=name, default=d,Type=tp} =
		 (case d of
		      (* No default, the arg is required *)
		      DefaultRequired =>
			  (case findArg name of
			       NONE => SOME("Missing required argument "
					    ^name^": "^(InfRuleArgType2string tp))
			     | SOME _ => NONE)
		    | _ => NONE)
	    val errors = List.mapPartial (fn x=>x) 
		          ((List.map checkArg args)@(List.map checkDefault specs))
	in
	    (case errors of
		 [] => NONE
	       | lst => SOME(strlist2str "\n" lst))
	end

    (* Goes through the arguments and instantiates the defaults for
       missing ones.  The only default args it does NOT instantiate
       are template matches.  Those must be done in the sequent match.
       Returns the list of new arguments sans the missing ones with
       template defauts. *)

    fun completeRuleArgs specs args =
	let val funName = "completeRuleArgs"
	    fun args2strFn args () = "["^(strlist2str ", " (List.map InfRuleArg2string args))^"]"
	    val _ = pushFunStackLazy(funName, args2strFn args)
	    fun findArg name = List.find(fn {name=x,...} => x=name) args

	    (* Check that all required args are present, and add the
	       missing ones with SOME(defaultVal). *)

	    fun checkDefault {name=name, default=d,Type=tp} =
		 (case findArg name of
		      NONE =>
			  (case d of
			       (* the arg is required *)
			       DefaultRequired => raise ProverError
				   ("Missing required argument "
				    ^name^": "^(InfRuleArgType2string tp))
			     | DefaultValue v => SOME{name=name, value=v}
			     (* Templates are not instantiated here *)
			     | DefaultMatch _ => NONE)
		    | SOME _ => NONE)
	    val newArgs = List.mapPartial checkDefault specs
	    val res = newArgs@args
	    val _ = popFunStackLazy(funName, args2strFn res)
	in
	    res
	end

    (* Transforms a TemplateValue into a ParseTree if possible, or returns NONE. *)
    fun TemplateValue2ParseTree tHash v = 
	let fun loop(InitValue t) = t
	      | loop(ObjectValue t) = SOME(termID2pt tHash t)
	      | loop(ParseTreeValue p) = SOME p
	      | loop(BoundValue t) = SOME(termID2pt tHash t)
	      | loop(NamedValue(_,v)) = loop v
	      | loop(ConstrainedValue(v,_)) = loop v
	      | loop _ = NONE
	in loop v
	end

    (* Transforms a TemplateValue into a ParseTree list.  Consider a
       single ParseTree element as a one-element list. *)

    fun TemplateValue2ParseTreeList tHash v =
	let val trees = List.map (TemplateValue2ParseTree tHash) (flattenValue [v])
	in 
	    if List.all isSome trees then
		SOME(List.mapPartial (fn x=>x) trees)
	    else NONE
	end

    fun TemplateValue2Model(ModelValue m) = SOME m
      | TemplateValue2Model _ = NONE

    fun TemplateValue2ProgramContext(ProgramContextValue pc) = SOME pc
      | TemplateValue2ProgramContext _ = NONE

    (* Takes a sequent template and a substitution and returns a sequent.
       All the variables in the template must have values in the substitution.
       Upon failure return NONE. *)

    fun substSequent (sigma:Substitution) (seqTemp: SequentTemplate) =
	let val _ = pushFunStackLazy("substSequent",
				     fn()=>((seqTemp2str seqTemp)^",\n"
					    ^(subst2str sigma)))
	    val { model=modelTemp,
		  init=initTemp,
		  context=contextTemp,
		  invar=invarTemp,
		  assumptions=assumptionsTemp,
		  conclusions=conclusionsTemp } = seqTemp
	    val (_, termHash) = sigma
	    val model = Option.mapPartial TemplateValue2Model 
		           (subst sigma ModelConstr modelTemp)
	    val init = Option.mapPartial (TemplateValue2ParseTree termHash)
		           (subst sigma InitConstr initTemp)
	    val context = Option.mapPartial TemplateValue2ProgramContext
		           (subst sigma ProgramContextConstr contextTemp)
	    val invarC = ListConstr(FormulaConstr(SOME(InvarPart, NONE)))
	    val assumpC = ListConstr(FormulaConstr(SOME(AssumptionPart, NONE)))
	    val concC = ListConstr(FormulaConstr(SOME(ConclusionPart, NONE)))
	    val invar = TemplateValue2ParseTreeList termHash
		  (ListValue(flattenValue(List.mapPartial (subst sigma invarC) invarTemp)))
	    val assump = TemplateValue2ParseTreeList termHash
		(ListValue(flattenValue(List.mapPartial (subst sigma assumpC) assumptionsTemp)))
	    val conc = TemplateValue2ParseTreeList termHash
		(ListValue(flattenValue(List.mapPartial (subst sigma concC) conclusionsTemp)))
	    val res = (case (model, init, context, invar, assump, conc) of
			   (SOME m, init, SOME context, SOME invar, SOME assump, SOME conc) =>
			       SOME({ model=m, init=init, context=context,
				      invar=invar,
				      assumptions=assump,
				      conclusions=conc }: Sequent)
			 | _ => NONE)
	    val _ = popFunStackLazy("substSequent",
				    fn()=>(case res of 
					       NONE => "NONE"
					     | SOME seq => (sequent2string seq)))
	in res
	end

(*     fun occursInTree tree x =
	let exception Stop
	    fun loop tree = if ptEq(tree,x) then raise Stop
			    else ptTransform loop tree
	in (loop tree; false) handle Stop => true
	end
*)
    (* Returns a substitution that makes template `temp' match the tree, or NONE. *)


    (* Takes a parse tree and checks whether it is a bound variable or not *)
    fun isBoundVar (QuantifiedVar _) = true
      | isBoundVar (PatternFormal _) = true
      | isBoundVar _ = false

    (* Match a term (given as TermID and its formula's position in the
       sequent) with a scalar template under a constraint.  Return
       either the new substitution (without altering the original), or
       NONE.  Implements T::pi|-e~>sigma.

       This implements a direct matching - no search for a "good"
       choice is performed for the templates that occur several times. *)

    fun matchTerm (temp, constr) (term as (termID, formulaInd)) sub =
	let val sub as (hash, termHash) = copySubstitution sub (* Make a clean copy first *)
	    val debug = lazyVerbDebug(getOptions()) "matchTerm"
	    val _ = pushFunStackLazy("matchTerm",
				     fn()=>((temp2str temp)^", "
					    ^(constr2str constr)^", "
					    ^(term2strHash(termID2term termHash termID))
					    ^" <id="^(termID2str termID)^">, "
					    ^(formulaInd2str formulaInd)))
	    fun loop (t, c) id =
		let val _ = pushFunStackLazy("matchTerm/loop",
					     fn ()=>((temp2str t)^" :: "
						     ^(constr2str c)^", ("
						     ^(term2strHash(termID2term termHash id))
						     ^") <id="^(termID2str termID)))
		    val res = loop' (t,c) id
		    val _ = popFunStackLazy("matchTerm/loop",
					    fn ()=>(case res of
							SOME _ => "matched"
						      | NONE => "unmatched"))
		in res
		end
	    and loop' (t, c as SomeConstr lst) id =
		let fun ff [] = NONE
		      | ff (c::lst) =
		         (* Call `matchTerm' to preserve the `sub', not `loop' *)
		         (case matchTerm (t, c) (id, formulaInd) sub of
			      SOME s => SOME s
			    | NONE => ff lst)
		in ff lst
		end
	      | loop' (t, c as FormulaConstr _) id =
		  if TempConstrLE(FormulaConstr(SOME formulaInd), c) then
		      loop (t, AnyConstr) id
		  else NONE
	      (* FIXME: check that the expression is boolean *)
	      | loop' (FormulaTemp n, AnyConstr) id =
		  updateSubstitution (n, ConstrainedValue(ObjectValue id,
							  FormulaConstr(SOME formulaInd))) sub
	      | loop' (BuiltinTemp tree, AnyConstr) id =
		  if ptEq(tree, termID2pt termHash id) then SOME sub
		  else NONE
	      | loop' (temp as ValueTemp n, BoundConstr t) id =
		  if isBoundVar(termID2pt termHash id) then
		    (case t of
			 SOME t =>
			   Option.mapPartial(updateSubstitution (n, BoundValue id))
			             (matchTerm (t, AnyConstr) (id, formulaInd) sub)
			 | NONE => updateSubstitution (n, BoundValue id) sub)
		  else NONE
	      | loop' (ValueTemp n, AnyConstr) id =
		  updateSubstitution (n, ObjectValue id) sub
	      | loop' (WildCardTemp, _) id = SOME sub
	      | loop' (temp, ObjectConstr pt) id =
		  let fun matchPT(SOME p) = ptEq(p, termID2pt termHash id)
			| matchPT _ = true
		  in 
		      if matchPT pt then loop(temp, AnyConstr) id
		      else NONE
		  end
	      | loop' (TupleTemp temps, constr) id =
		  let val term = termID2term termHash id
		  in
		      case term of
			  TupleTerm ids =>
			      matchTerms (temps, ListConstr constr) 
			                 (List.map(fn id=>(id, formulaInd)) ids) sub
			| _ => NONE
		  end
	      | loop' (ApplTemp(t1, t2), constr) id =
		  let val term = termID2term termHash id
		  in
		      case term of
			  ApplTerm(id1,id2) => 
			      Option.mapPartial(matchTerm (t2, constr) (id2, formulaInd))
			                       (matchTerm (t1, constr) (id1, formulaInd) sub)
			| _ => NONE
		  end
	      | loop' (TypedTemp(t1, t2), constr) id =
		  let val term = termID2term termHash id
		  in
		      case term of
			  TypedTerm(id1,id2) => 
			      Option.mapPartial(matchTerm (t2, constr) (id2, formulaInd))
			                       (matchTerm (t1, constr) (id1, formulaInd) sub)
			| _ => NONE
		  end
	      | loop' (ConstrainedTemp(temp, c1), c2) id =
		  Option.mapPartial(fn c=> loop(temp, c) id)(TempConstrMin(c1, c2))
	      | loop' (ForallTemp(vars, body, cxt), constr) id =
		  let val term = termID2term termHash id
		  in
		      case term of
			  ForallTerm{ names=nameIDs,
				      body=bodyID,
				      parent=p } =>
			    let val sub1 = matchTerms (vars, ListConstr(BoundConstr NONE))
				             (List.map(fn id=>(id, formulaInd)) nameIDs) sub
				val sub2 = Option.mapPartial
				      (matchTerm (body, FormulaConstr NONE)
				                 (bodyID, formulaInd)) sub1
			    in Option.mapPartial
				(fn s=> (case cxt of
					     ValueTemp n =>
						 updateSubstitution (n, ParseTreeValue p) s
					   | WildCardTemp => SOME s
					   | _ => NONE)) sub2
			    end
		        | _ => NONE
		  end
	      | loop' (ExistsTemp(vars, body, cxt), constr) id =
		  let val term = termID2term termHash id
		  in
		      case term of
			  ExistsTerm{ names=nameIDs,
				      body=bodyID,
				      parent=p } =>
			    let val sub1 = matchTerms (vars, ListConstr(BoundConstr NONE))
				             (List.map(fn id=>(id, formulaInd)) nameIDs) sub
				val sub2 = Option.mapPartial
				      (matchTerm (body, FormulaConstr NONE)
				                 (bodyID, formulaInd)) sub1
			    in Option.mapPartial
				(fn s=> (case cxt of
					     ValueTemp n =>
						 updateSubstitution (n, ParseTreeValue p) s
					   | WildCardTemp => SOME s
					   | _ => NONE)) sub2
			    end
		        | _ => NONE
		  end
	      | loop' (ContextTemp(n, temp), constr) id =
		  (* Perform BFS on subterms and try to match them
		     with `temp'.  Return the ID together with the
		     resulting substitution for the first that
		     matches. *)
		  let fun matchSubterms [] [] = NONE
			| matchSubterms [] next = matchSubterms (List.rev next) []
			| matchSubterms (id::ids) next =
		            (case matchTerm (temp, ObjectConstr NONE) (id, formulaInd) sub of
				 SOME sub => SOME(sub, id)
			       | NONE => matchSubterms ids 
				           ((List.rev (subtermIDs(termID2term termHash id)))@next))
		      (* Given the term ID and the ID of a subterm,
		         build a substitution function that would
		         construct a new term with the subterm
		         replaced by any new ID. *)
		      fun getIdSubst(id, sid) newID =
			  let fun loop id = getIdSubst(id, sid) newID
			  in 
			      if termIDeq(id, sid) then newID
			      else (case termID2term termHash id of
					t as ConstTerm _ => id
				      | TupleTerm lst =>
					    term2termID termHash (TupleTerm(List.map loop lst))
				      | ApplTerm(id1,id2) =>
					    term2termID termHash (ApplTerm(loop id1, loop id2))
				      | TypedTerm(id1,id2) =>
					    term2termID termHash (TypedTerm(loop id1, loop id2))
				(* Don't recur into bound vars, only into body *)
				      | ForallTerm{names=names,
						   body=b,
						   parent=p} =>
					    term2termID termHash (ForallTerm{names=names,
									     body=loop b,
									     parent=p})
				      | ExistsTerm{names=names,
						   body=b,
						   parent=p} =>
					    term2termID termHash (ExistsTerm{names=names,
									     body=loop b,
									     parent=p}))
			  end
		      fun getTempSubst(id, sid) (ObjectValue newID) =
			    ObjectValue(getIdSubst(id, sid) newID)
			| getTempSubst _ v = v
		      fun updateSub(sub, sid) = 
			    (updateSubstitution (n, ContextValue(getTempSubst(id, sid))) sub;
			     sub)
		  in
		      Option.map updateSub (matchSubterms [id] [])
		  end
	      | loop'(ConstantTemp _, constr) id = raise SympBug
		  ("SequentDefault.matchTerm: ConstantTemp matching is not implemented yet.")
	      | loop'(NamedTemp(name, temp), constr) id =
		  Option.mapPartial(updateSubstitution(name, ObjectValue id))
		                   (loop(temp, constr) id)
	      | loop' _ _ = NONE
	    val res = loop (temp, constr) termID
	    val _ = popFunStackLazy("matchTerm",
				    fn()=>(case res of
					       SOME s => "matched, "^(subst2str s)
					     | NONE => "unmatched"))
	in
	    res
	end

    (* `terms' is a list of pairs (termID, formulaInd) *)
    and matchTerms (temps, constr) terms  sub =
	let val (_, termHash) = sub
	    val debug = lazyVerbDebug(getOptions()) "matchTerms"
	    fun termlist2str terms =
		"["^(strlist2str "," 
		     (List.map (fn (id,fInd)=>"("
				(*  ^(term2strHash(termID2term termHash id)) *)
				^" <id="^(termID2str id)^">, "
				^(formulaInd2str fInd)^")") terms))
		^"]"
	    val _ = pushFunStackLazy("matchTerms",
				     fn()=>("["^(strlist2str "," (List.map temp2str temps))^"], "
					    ^(constr2str constr)^", "^(termlist2str terms)))
	    (* try to match terms and return those that match together with the final subst.
	       Arguments:
	       matchFilter sub formulaIndConstraint (temp, constr) terms matched unmatched *)
	    fun matchFilter sub (temp, c) ids matched unmatched =
		let fun first(x,_) = x
		    fun pairs2str pairs = 
		         "["^(strlist2str ", " (List.map(termID2str o first) pairs))^"]"
		    fun argsFn() = ((temp2str temp)^", "^(constr2str c)^", "
				    ^(pairs2str ids)^", "^(pairs2str matched))
		    val _ = pushFunStackLazy("matchTerms/matchFilter", argsFn)
		    fun loop [] = (List.rev matched, List.rev unmatched, sub)
		      | loop ((term as (id, fInd))::ids) =
			(case matchTerm (temp, c) term sub of
			     NONE => matchFilter sub (temp, c) ids matched (term::unmatched)
			   | SOME s => matchFilter s (temp, c) ids (term::matched) unmatched)
		    val res as (matched, _, _) = loop ids
		    val _ = popFunStackLazy("matchTerms/matchFilter",fn()=>pairs2str matched)
		in res
		end 
		  
	    (* Match the first term in the list with a scalar template *)
	    fun matchFirst sub (temp, c) ids unmatched =
		let fun first(x,_) = x
		    fun pairs2str pairs = 
		         "["^(strlist2str ", " (List.map(termID2str o first) pairs))^"]"
		    fun argsFn() = ((temp2str temp)^", "^(constr2str c)^", "^(pairs2str ids))
		    val _ = pushFunStackLazy("matchTerms/matchFirst", argsFn)
		    fun loop [] = NONE
		      | loop ((term as (id, _))::ids) = 
			(case matchTerm (temp, c) term sub of
			     NONE => matchFirst sub (temp, c) ids (term::unmatched)
			   | SOME s => SOME(id, (List.rev unmatched)@ids, s))
		    val res = loop ids
		    fun resFn(SOME(id, _, _)) () = termID2str id
		      | resFn NONE () = "NONE"
		    val _ = popFunStackLazy("matchTerms/matchFirst", resFn res)
		in
		    res 
		end
	    fun stripListConstr (ListConstr c) = SOME c
	      | stripListConstr AnyConstr = SOME AnyConstr
	      | stripListConstr _ = NONE
	    (* Match terms with `temps' and return the new sub and
	       remaining unmatched terms, or NONE *)
	    fun loop (temps, c) terms sub =
		let val sub = copySubstitution sub (* Always make a clean copy *)
		    val _ = pushFunStackLazy
			("matchTerms/loop",
			 fn()=>("["^(strlist2str "," (List.map temp2str temps))
				^"] :: "^(constr2str c)^", "
				^(termlist2str terms)))
		    val res = loop' (temps, c) terms sub
		    val _ = popFunStackLazy
			("matchTerms/loop",
			 fn()=>(case res of
				    SOME(_, terms) => (termlist2str terms)
				  | NONE => "unmatched"))
		in res
		end
	    and loop' ([], _) terms sub = SOME(sub, terms)
	      (* FIXME: Should probably constrain the formula list to FormulaConstr... *)
	      | loop' ((ListFormulaTemp(n, tempOpt))::temps, constr) ids sub =
		  let val cOpt = stripListConstr constr
		      val (matched, unmatched, sub1) =
		            case (cOpt, tempOpt) of
				(SOME c, SOME t) =>  matchFilter sub (t, c) ids [] []
			      | (SOME c, NONE) => matchFilter sub (WildCardTemp, c) ids [] []
			      | _ => ([], ids, sub)
		      fun ff(id, fInd) = ConstrainedValue(ObjectValue id, FormulaConstr(SOME fInd))
		      val sub2 = updateListSubstitution(n, List.map ff matched) sub1
		  in 
		      Option.mapPartial(loop (temps, constr) unmatched) sub2
		  end
	      | loop' ((ListValueTemp(n, tempOpt))::temps, constr) ids sub =
		  let val cOpt = stripListConstr constr
		      val (matched, unmatched, sub1) =
		            case (cOpt, tempOpt) of
				(SOME c, SOME t) =>  matchFilter sub (t, c) ids [] []
			      | (SOME c, NONE) => matchFilter sub (WildCardTemp, c) ids [] []
			      | _ => ([], ids, sub)
		      fun makeValue(id, _) =
			  (case cOpt of
			       SOME(BoundConstr _) => BoundValue id
			     | _ => ObjectValue id)
		      val sub2 = updateListSubstitution(n, List.map makeValue matched) sub1
		  in 
		      Option.mapPartial(loop (temps, constr) unmatched) sub2
		  end
	      | loop' (ListWildCardTemp::temps, constr) ids sub =
		  let val cOpt = stripListConstr constr
		      val (matched, unmatched, sub1) =
		            case cOpt of
				SOME c => matchFilter sub (WildCardTemp, c) ids [] []
			      | _ => ([], ids, sub)
		  in 
		      loop (temps, constr) unmatched sub1
		  end
	      | loop' ((ConstrainedTemp(t, c))::temps, constr) ids sub =
		  (case TempConstrMin(c, constr) of
		       NONE => Option.mapPartial(fn (sub, ids) =>loop (temps, constr) ids sub)
			                        (loop ([t], AnyConstr) [] sub)
		     | SOME c => Option.mapPartial(fn (sub, ids)=>loop (temps, constr) ids sub)
						  (loop ([t], c) ids sub))
	      | loop' (temp::temps, constr) ids sub =
		  let val c = (case stripListConstr constr of
				   SOME c => c
				 | NONE => constr)
		  in Option.mapPartial(fn (_, ids, sub)=> loop (temps, constr) ids sub)
		                      (matchFirst sub (temp, c) ids [])
		  end
	    (* All terms in the list must match for the final result *)
	    val res = (case loop (temps, constr) terms sub of
			   SOME(sub, []) => SOME sub
			 | _ => NONE)
	    val _ = popFunStackLazy("matchTerms",
				    fn()=>(case res of
					       SOME _ => "matched"
					     | NONE => "unmatched"))
	in
	    res
	end
		       

    fun matchModel (temp, model) sub =
	(case temp of
	     ValueTemp n => updateSubstitution (n, ModelValue model) sub
	   | _ => NONE)

    fun matchInit (temp, init) sub =
	(case temp of
	     ValueTemp n => updateSubstitution (n, InitValue init) sub
	   | _ => NONE)

    fun matchProgramContext (ProgramContextTemp n, pc) sub =
	  updateSubstitution (n, ProgramContextValue pc) sub
      | matchProgramContext _ _ = NONE

    (* Matches a sequent against a sequent template. Uses `sub' as
       the initial substitution.  Returns a new substitution or NONE *)

    fun matchSeq(sub: Substitution)(seqTemp: SequentTemplate,
				    hints: SequentHints option,
				    seqTerm: SequentTerm) =
	let val debug = lazyVerbDebug(getOptions()) "matchSeq"
	    val { model=modelTemp,
		  init=initTemp,
		  context=contextTemp,
		  invar=invarTemp,
		  assumptions=assumpTemp,
		  conclusions=concTemp } = seqTemp
	    val {model=model,
		 init=init,
		 context=pc,
		 invar=invar,
		 assumptions=assump,
		 conclusions=conc,
		 hash = termHash } = seqTerm
	    val _ = pushFunStackLazy("matchSeq", fn()=>(seqTemp2str seqTemp))
	    val _ = debug(fn()=>termHash2str termHash)
	    (* Pair formulas with their indices (sequent part + formula #) *)
	    fun annotate part n [] = []
	      | annotate part n (id::ids) = (id, (part, SOME n))::(annotate part (n+1) ids)
	    val sub = matchModel (modelTemp, model) sub
	    val _ = debug(fn()=>"matchSeq: model "
			  ^(case sub of
				SOME _ => "matched"
			      | NONE => "unmatched")^"\n")
	    val sub = Option.mapPartial(matchInit(initTemp, init)) sub
	    val _ = debug(fn()=>"matchSeq: init "
			  ^(case sub of
				SOME _ => "matched"
			      | NONE => "unmatched")^"\n")
	    val sub = Option.mapPartial(matchProgramContext(contextTemp, pc)) sub
	    val _ = debug(fn()=>"matchSeq: context "
			  ^(case sub of
				SOME _ => "matched"
			      | NONE => "unmatched")^"\n")
	    val sub = Option.mapPartial
		  (matchTerms(invarTemp, AnyConstr) (annotate InvarPart 1 invar)) sub
	    val _ = debug(fn()=>"matchSeq: invariants "
			  ^(case sub of
				SOME _ => "matched"
			      | NONE => "unmatched")^"\n")
	    val sub = Option.mapPartial
		  (matchTerms(assumpTemp, AnyConstr) (annotate AssumptionPart 1 assump)) sub
	    val _ = debug(fn()=>"matchSeq: assumptions "
			  ^(case sub of
				SOME _ => "matched"
			      | NONE => "unmatched")^"\n")
	    val sub = Option.mapPartial
		  (matchTerms(concTemp, AnyConstr) (annotate ConclusionPart 1 conc)) sub
	    val _ = popFunStackLazy("matchSeq",
				    fn()=>(case sub of
					       SOME s => subst2str s
					     | NONE => "unmatched"))
	in
	    sub
	end

    (* Converts arguments from the general to our own representation - RuleArg,
       also add ArgObject values to the substitution. *)
    fun compileArgs(sub, context, args) =
	let val funName = "compileArgs"
	    fun argsFn(args, arg2str) () = "["^(strlist2str ", " (List.map arg2str args))^"]"
	    val _ = pushFunStackLazy(funName, argsFn(args, InfRuleArg2string))
	    val (_, termHash) = sub
	    fun tc expr =
		let val (progObj,_) = typeCheckExpr (getOptions()) context (expr, Tvar(dp, "a"))
		in progObj
		end
	    fun eval(expr, findObject) =
		  (evaluateExpr(getOptions()) findObject expr, findObject)
	    fun loop (ProverArgString s) = ArgString s
	      | loop (ProverArgNumber n) = ArgNumber n
	      | loop (ProverArgSubgoal s) = ArgSubgoal s
	      | loop (ProverArgFormulaNum ind) = ArgFormulaNum ind
	      | loop (ProverArgObject str) =
		let val obj = ParseInput (getOptions())
		                (TextIO.openString(expressionKeyword^" "^str))
		    val progObj = eval(tc obj) 
		in ArgObject progObj
		end
	      | loop (ProverArgFormula str) =
		let val obj = ParseInput (getOptions())
		                (TextIO.openString(expressionKeyword^" "^str))
		    val (progObj,_) = typeCheckExpr (getOptions()) context (obj, BoolType dp)
		in ArgObject progObj
		end
	      | loop (ProverArgList lst) = ArgList(List.map loop lst)
	    fun convertArg{name=name, value=v} =
		let fun ff (v as ArgObject obj) =
		          ((case updateSubstitution
				   (name, ObjectValue(pt2termID termHash(#1 obj))) sub of
				SOME _ => ()
			      | NONE => raise SympBug
				    ("compileArgs: name is already in substitution with diff. value: "
				     ^name)); v)
		      | ff v = v
		in {name=name, value=ff(loop v)}: RuleArg
		end
	    val res = List.map convertArg args
	    val _ = popFunStackLazy(funName, argsFn(res, RuleArg2string))
	in
	    res
	end

    (* Sort through the rule's arguments and the conclusion template,
       and annotate the vars with the constraints from the arguments *)
    fun applyArgs(temp, args) =
	let fun findArgName (args: RuleArg list) name =
	        (case List.find(fn {name=n,...}=>n=name) args of
		     SOME {value=v,...} => SOME v
		   | NONE => NONE)
	    fun arg2constr(ArgFormulaNum fInd) = SOME(FormulaConstr(SOME fInd))
	      (* FIXME: this only allows top-level expressions.
		 We also need types - add another arg type? *)
	      | arg2constr(ArgObject(pt,_)) = SOME(ObjectConstr(SOME pt))
	      (* Anything else is not allowed. *)
	      (* FIXME: list args should probably be handled, but we do not have
	         disjunctive constraints yet... *)
	      | arg2constr (ArgList lst) =
		let val cs = List.map arg2constr lst
		in
		    if List.all isSome cs then SOME(SomeConstr(List.mapPartial(fn x=>x) cs))
		    else NONE
		end
	      | arg2constr _ = NONE
	    (* Join the two functions *)
	    fun applyConstr (name, t, isList) =
		let val constr = Option.mapPartial arg2constr (findArgName args name)
		in 
		    case constr of
			NONE => t
		      | SOME c => if isList then ConstrainedTemp(t, ListConstr c)
				  else ConstrainedTemp(t, c)
		end		
	    (* Recursively descend into `temp' and constrain all vars according to the args *)
	    fun loop (t as FormulaTemp n) = applyConstr(n, t, false)
	      | loop (t as ListFormulaTemp(n,_)) = applyConstr(n, t, true)
	      | loop (t as ValueTemp n) = applyConstr(n, t, false)
	      | loop (t as ListValueTemp(n,_)) = applyConstr(n, t, true)
	      | loop (TupleTemp lst) = TupleTemp(List.map loop lst)
	      | loop (ApplTemp(t1,t2)) = ApplTemp(loop t1, loop t2)
	      | loop (TypedTemp(t1,t2)) = TypedTemp(loop t1, loop t2)
	      (* Do not change the context template for the quantifiers *)
	      | loop (ForallTemp(lst, t1, t2)) = ForallTemp(List.map loop lst, loop t1, t2)
	      | loop (ExistsTemp(lst, t1, t2)) = ExistsTemp(List.map loop lst, loop t1, t2)
	      | loop (ContextTemp(n, t)) = applyConstr(n, ContextTemp(n, loop t), false)
	      | loop (ConstrainedTemp(t, c)) = ConstrainedTemp(loop t, c)
	      (* Nothing else contains interesting vars *)
	      | loop t = t
	in
	    loop temp
	end

    (* Same as above, only for the sequent template. *)
    fun applySeqArgs(seqTemp: SequentTemplate, args) =
	let val {model=model,
		 init=init,
		 context=context,
		 invar=invar,
		 assumptions=assump,
		 conclusions=conc} = seqTemp
	    val newInvar=List.map(fn t=>applyArgs(t, args)) invar
	    val newAssump=List.map(fn t=>applyArgs(t, args)) assump
	    val newConc=List.map(fn t=>applyArgs(t, args)) conc
	in
	    {model=model,
	     init=init,
	     context=context,
	     invar=newInvar,
	     assumptions=newAssump,
	     conclusions=newConc}: SequentTemplate
	end

    (* See if the sequent matches the conclusion of the rule, and if it does,
       compute the substitution for it.  Otherwise return NONE.  *)
    fun matchRuleSubst (sub: Substitution) (seq, context, seqTerm) (rule: InferenceRule, args) = 
	let val {name=name,match=match,conclusion=conc,...} = rule
	    (* FIXME: Add args of `ProverArgObjectType' to the substitution *)
	in 
	    case (match,conc) of
		(SOME ff, _) => ff(seq, context, args, SOME sub)
	      | (NONE, SOME seqTemp) =>
		    matchSeq sub (applySeqArgs(seqTemp, args), NONE, seqTerm)
	      | _ => raise SympBug
		    ("matchRule: the rule "^name
		     ^" has neither match function nor conclusion")
	end

    fun matchRule (seq, context) (rule, args) =
	let val seqTerm as {hash = hash, ...} = sequent2term(seq,NONE)
	    val sub = makeSubstitution(hash)
	    val newArgs = compileArgs(sub, context, completeRuleArgs(getRuleArgs rule) args)
	in 
	    matchRuleSubst sub (seq, context, seqTerm) (rule, newArgs)
	end

    fun applyCommand printFun (seq, _) (command: ProofSystemCommand, args) =
	let val {name=name, apply=apply, params=specs, ...} = command
	    val _ = (case checkRuleArgs specs args of
			 SOME str => raise ProverError str
		       | NONE => ())
	in
	    apply printFun (seq, args)
	end

    (* Try to apply the rule and get a list of new sequents as premisses.
       NONE means the rule doesn't apply. *)
    fun applyRule printFun (seq, context) (rule: InferenceRule, args, substOpt) =
	let val funName = "SequentDefault.applyRule"
	    fun argsFn() = ("seq = "^(sequent2string seq)
			    ^", rule = "^(getRuleName rule)
			    ^", args = ["^(strlist2str ", " (List.map InfRuleArg2string args))
			    ^"]")
	    val _ = pushFunStackLazy(funName, argsFn)
	    val { name=name, premisses=prems, apply=apply,...} = rule
	    (* Ignore the given context, and use the sequent context instead *)
	    val ({ context=context,...}: Sequent) = seq
	    val debug = verbDebug(getOptions()) "applyRule"
	    (* Check the arguments *)
	    val _ = (case checkRuleArgs (getRuleArgs rule) args of
			 SOME str => raise ProverError str
		       | NONE => ())
	    (* Figure out the substitution: if none is given, create an empty one *)
	    val sub as (_, hash) =
		 (case substOpt of
		      SOME s => s
		    | NONE => makeSubstitution(makeTermHash()))
	    val seqTerm = sequent2term(seq, SOME hash)
	    val newArgs = compileArgs(sub, context, completeRuleArgs(getRuleArgs rule) args)
	    val res =
	    case (apply, prems) of
		(SOME ff, _) => ff printFun (seq, context, newArgs, sub)
	      | (NONE, SOME lst) => 
		    (* We have to match the rule in the standard way *)
		    let val newSub = matchRuleSubst sub (seq, context, seqTerm) (rule, newArgs)
			val _ = if isSome newSub then
			           debug("The rule "
					 ^(getRuleName rule)
					 ^" matched, substitution generated\n")
				else debug("The rule "
					   ^(getRuleName rule)
					   ^" didn't match, no substitution generated\n")
			val seqsOpt = Option.map(fn sub=>List.map (substSequent sub) lst) newSub
			fun optSeq2str NONE = "NONE"
			  | optSeq2str (SOME seq) = sequent2string seq
		    in 
			case seqsOpt of
			    SOME seqs =>
				if List.all isSome seqs then
				    (InferenceRuleSuccess(List.map (fn x=>(valOf x, NONE)) seqs))
				    before
				    (debug("Rule "
					   ^(getRuleName rule)
					   ^": instantiation of premisses successful\n"))
				else (debug("Rule "
					    ^(getRuleName rule)
					    ^": instantiation of premisses failed.  Sequents =\n ["
					    ^(strlist2str "," (List.map optSeq2str seqs))
					    ^"]\n");
				      InferenceRuleFailure)
			  | NONE => InferenceRuleFailure
		    end
	      | _ => raise SympBug
		    ("applyRule: rule "^name^" has neither apply function nor premisses")
	    val _ = popFunStackLazy(funName, fn()=> InfRuleResult2string res)
	in
	    res
	end
	    
    (* Get the list of rules that match a given sequent *)
    fun matchAllRules seq = raise SympBug("SequentDefault/matchAllRules: not implemented")

    (* Getting extra information about a sequent.  The type of request
       is specified by a string, the output is also a string which the
       user will see. *)
     fun getSequentInfo (seq: Sequent) req =
	let val { model=model, ...} = seq
	    fun loop "model" = model2str model
	      | loop x = raise SequentError
		 ("Unknown info request for the sequent: \""^x^"\"")
	in loop req
	end

     (* Reimplementation of SequentCommon functions for our internal representation *)
    fun ArgStringValue(ArgString str) = SOME str
      | ArgStringValue _ = NONE
    fun ArgNumberValue(ArgNumber n) = SOME n
      | ArgNumberValue _ = NONE
    fun ArgSubgoalValue(ArgSubgoal str) = SOME str
      | ArgSubgoalValue _ = NONE
    fun ArgFormulaNumValue(ArgFormulaNum x) = SOME x
      | ArgFormulaNumValue _ = NONE
    fun ArgObjectValue(ArgObject obj) = SOME obj
      | ArgObjectValue _ = NONE
    fun ArgListValue(ArgList lst) = SOME lst
      | ArgListValue _ = NONE

    fun findArg args name =
	  Option.map(fn {value=v,...}: RuleArg => v)
	            (List.find(fn {name=n,...} => n=name) args)

    fun findArgAnyValue ff args name = Option.mapPartial ff (findArg args name)
	    
    val findArgValue = findArgAnyValue(fn x=>SOME x)
    val findArgStringValue = findArgAnyValue ArgStringValue
    val findArgNumberValue = findArgAnyValue ArgNumberValue
    val findArgSubgoalValue = findArgAnyValue ArgSubgoalValue
    val findArgFormulaNumValue = findArgAnyValue ArgFormulaNumValue
    val findArgObjectValue = findArgAnyValue ArgObjectValue
    val findArgListValue = findArgAnyValue ArgListValue

  end
