(* The default implementation of the PROOF_SYSTEM abstract interface *)

functor ProofSystemDefaultFun(structure Hash: HASH
			      structure Interface: INTERFACE): PROOF_SYSTEM =
  struct
    structure ProofRules = 
	ProofRulesDefaultFun(structure Hash = Hash
			     structure Interface = Interface)

    open Interface
    open ProofRules
    open Sequent
    open SequentCommon
    open SequentBase
    open Hash
    open SympBug
    open Str

    local
	val rules = makeHashDefault(op =, fn x=>x): (string, InferenceRule) Hash
	val commands = makeHashDefault(op =, fn x=>x): (string, ProofSystemCommand) Hash
	val tactics = makeHashDefault(op =, fn x=>x): (string, ProofSystemTactic) Hash
    in
	fun findRuleByName str = findHash(rules, str)
	fun findCommandByName str = findHash(commands, str)
	fun findTacticByName str = findHash(tactics, str)
	fun addNewRule r = (insertHashDestructive(rules, getRuleName r, r); ())
	fun addNewCommand c = (insertHashDestructive(commands, getCommandName c, c); ())
	fun addNewTactic t = (insertHashDestructive(tactics, getTacticName t, t); ())

	(* Initialize the set of inference rules from the ProofRules module *)
	fun initProofSystem () = 
	      (List.app addNewRule (proofRules());
	       List.app addNewCommand (proverCommands()))
	fun getAllRules() = List.map #2 (hash2any(fn x => x)(fn x => x) rules)
	fun getAllCommands() = List.map #2 (hash2any(fn x => x)(fn x => x) commands)
	fun getAllTactics() = List.map #2 (hash2any(fn x => x)(fn x => x) tactics)

	fun getInitRule() = 
	    (case findRuleByName "init" of
		 SOME r => r
	       | NONE => raise SympBug
		     ("ProofSystemDefault/getInitRule: no \"init\" rule"))

	(* Apply the rule, and check if the resulting sequents are trivially
	   true by applying the init rule.  All sequents that are proven this way
	   are discarded from the result. *)

	fun applyRule printFun (seq, context) (rule, args, substOpt) =
	    let val debug = verbDebug(getOptions()) "applyRule"
		val res = Sequent.applyRule printFun (seq, context) (rule, args, substOpt)
		val _ = debug("\napplyRule: the `raw' result of applying the rule "
			      ^(getRuleName rule)^":\n  "
			      ^(InfRuleResult2string res)^"\n")
		val _ = debug("\napplyRule: checking for trivially true sequents\n")
		(* Annihilate the sequent if it's trivially true *)
		fun checkSeq (pair as (seq, _)) =
		    (case Sequent.applyRule printFun (seq, context) (getInitRule(), [], NONE) of
			 InferenceRuleSuccess [] => 
			     (debug("\napplyRule: this sequent is trivially true:\n  "
				    ^(sequent2string seq));
			      NONE)
		       | _ => SOME pair)
		(* Clean up all trivial sequents *)
		fun checkSeqs(InferenceRuleSuccess seqs) =
		      InferenceRuleSuccess(List.mapPartial checkSeq seqs)
		  | checkSeqs x = x
	    in
		checkSeqs res
	    end
    end

  end
