functor ProofRulesDefaultFun(structure Hash: HASH
			     structure Interface: INTERFACE): PROOF_RULES_DEFAULT =
  struct
    structure Sequent = 
	SequentDefaultFun(structure Hash = Hash
			  structure Interface = Interface)

    open Sequent
    open InputLanguage
    open Typecheck

    structure Trans_SMV = TransSMVFun (structure Hash = Hash
				       structure Abstract = Abstract)

    open Hash
    open Interface
    open Options

    open SympBug
    open Str
    open Pos
    open Hash
    open ParserDefault
    open ParseTreeStruct
    open Abstract
    open Trans
    open Trans_SMV

    (* Eventually it will be a datatype multiplexing states of each tactic implemented *)
    type ProofSystemTacticState = unit

    type ProofSystemTactic =
	{ name: string,
	  help: string,
	  params: InferenceRuleArgumentSpec list,
	  (* Implements one step of a tactic.  If state is present,
	     it's an intermediate step, otherwise it's the first step.
	     The first `inline' argument controls inlining of nested
	     tactics.

	     apply inline printFun (args, context, stateOpt) (seq, hintsOpt) *)
	  apply: (bool -> (string -> unit)
		  -> ((InferenceRuleArgument list) * ProgramContext
		      * (ProofSystemTacticState option))
		  -> (Sequent * (SequentHints option))

		  (* Result: optional rule name with args which was
		     applied, the result of this rule's application,
		     and an optional new state. *)

		  -> (string * (InferenceRuleArgument list)
		      * ((Sequent * (SequentHints option)) list)
		      * (ProofSystemTacticState option)) option) }

    fun getTacticName({name=name,...}: ProofSystemTactic) = name

    fun getTacticHelp ({help=help,
			name=name,
			params=params, ...}:ProofSystemTactic) =
	(name^"("^(strlist2str ", " (List.map InfRuleSpec2str params))^")\n\n"^help)

    fun getTacticArgs({params=specs,...}:ProofSystemTactic) = specs
	    
    fun applyTactic inline printFun (tactic, args, context, stateOpt) (seq, hintsOpt) =
	let val {name=name, apply=apply, params=specs, ...}:ProofSystemTactic = tactic
	    val _ = (case checkRuleArgs specs args of
			 SOME str => raise ProverError str
		       | NONE => ())
	in
	    apply inline printFun (args, context, stateOpt) (seq, hintsOpt)
	end

    (* Local exception that aborts application of a rule.  It is supposed to be
       caught in rule-specific `applyRule' functions, and the rule then fails after
       printing the message from the exception. *)
    exception AbortRule of string

    val trueVal = Builtin{name=True dp, Type=BoolType dp}
    val falseVal = Builtin{name=False dp, Type=BoolType dp}
    val notVal = Builtin{name=Not dp, Type=FunType(dp,BoolType dp, BoolType dp)}
    val andVal = Builtin{name=And dp, 
			 Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
				      BoolType dp)}
    val orVal = Builtin{name=Or dp, 
			Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
				     BoolType dp)}
    val equalVal = Builtin{name=Eq dp, 
			   Type=FunType(dp,TupleType(dp,[Tvar(dp,"a"),Tvar(dp,"a")]),
					BoolType dp)}
    val impliesVal = Builtin{name=Implies dp, 
			     Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
					  BoolType dp)}
    fun negFormulaTemp f = ApplTemp(BuiltinTemp notVal, f)
    fun negFormula f = Appl(dp, notVal, f)
    fun impliesFormulaTemp(f,g) = ApplTemp(BuiltinTemp impliesVal, TupleTemp[f,g])
    fun impliesFormula(f,g) = Appl(dp, impliesVal, TupleExpr(dp, [f,g]))
    fun eqExprTemp(f,g) = ApplTemp(BuiltinTemp equalVal, TupleTemp[f,g])
    fun foldOpTemp (Op,default) lst =
	let fun foldFun(f1,f2) = ApplTemp(Op,TupleTemp[f1,f2])
	    fun doFold [] = default
	      | doFold (f::lst) = List.foldr foldFun f lst
	in doFold(List.rev lst)
	end
    val foldAndTemp = foldOpTemp(BuiltinTemp andVal, BuiltinTemp trueVal)
    val foldOrTemp = foldOpTemp(BuiltinTemp orVal, BuiltinTemp falseVal)

    (* Check if the term under `id' is the constant `const' *)
    fun compareTermIDconst termHash const id =
	let val term = termID2term termHash id
	in
	    case term of
		ConstTerm c => ptEq(c, const)
	      | _ => false
	end

    (* Given a TemplateValue `v' and a matching and substitution
       templates matchTemp and substTemp, return a TemplateValue
       updated accordingly, or NONE if something fails on the way. *)

    fun matchSubstValue sub (matchTemp, substTemp) v =
	let (* Wrap the constraints back on the value *)
	    fun wrapConstr(c::constr) v = wrapConstr constr (ConstrainedValue(v, c))
	      | wrapConstr [] v = v
	    (* Strip all the constraints from ObjectValue and return
	       (constraints, termID, formulaIndOpt) *)
	    fun stripConstr(ConstrainedValue(v, c), constr, ind) =
		let fun pickMin(NONE, NONE) = NONE
		      | pickMin(NONE, c) = c
		      | pickMin(c, NONE) = c 
		      | pickMin(c1, c2) =
		         if TempConstrLE(FormulaConstr c1, FormulaConstr c2) then c2 else c1
		    val newInd = (case c of
				      FormulaConstr p => pickMin(p, ind)
				    | _ => ind)
		in stripConstr(v, c::constr, newInd)
		end
	      | stripConstr(ObjectValue id, constr, ind) =
		  (case ind of
		       SOME p => (constr, id, p)
		     | NONE => (constr, id, (AnyPart, NONE)))
	      | stripConstr(v, c, _) = raise SympBug
		  ("ProofRulesDefault.matchSubstValue/stripConstr:\n"
		   ^"  not an ObjectValue: "^(value2str (wrapConstr c v)))
	    val (constr, id, fInd) = stripConstr(v, [], NONE)
	    val subOpt = matchTerm(matchTemp, AnyConstr) (id, fInd) sub
	    val newVal = Option.mapPartial(fn sub=>subst sub AnyConstr substTemp) subOpt
	in Option.map(wrapConstr constr) newVal
	end

    (* Take the list of formulas with constraints and compute the list
       of pairs (formula, part) *)

    fun getPairs termHash (ListValue lst) =
	let fun c2part _ (p as SOME _) = p
	      | c2part(FormulaConstr(SOME(InvarPart, _))) NONE = SOME InvarPart
	      | c2part(FormulaConstr(SOME(AssumptionPart, _))) NONE = SOME AssumptionPart
	      | c2part(FormulaConstr(SOME(ConclusionPart, _))) NONE = SOME ConclusionPart
	      | c2part(SomeConstr[c]) NONE = c2part c NONE
	      | c2part _ _ = NONE
	    fun getPair part (ConstrainedValue(v, c)) = getPair (c2part c part) v
	      | getPair (SOME part) (ObjectValue id) = (termID2pt termHash id, part)
	      | getPair NONE (ObjectValue id) = raise SympBug
		("ProofRules/getPairs: cannot determine sequent part\n"
		 ^"for the formula: "^(pt2string(termID2pt termHash id)))
	      | getPair _ x = raise SympBug
		("ProofRules/getPairs: not a formula in the argument:\n  "
		 ^(value2str x)
		 ^"\n  Term hash = "^(termHash2str termHash))
	in
	    List.map (getPair NONE) lst
	end
      | getPairs termHash v = raise SympBug
	("ProofRules.replaceApply/getPairs: not a ListValue argument: "
	 ^(value2str v)
	 ^"\n  Term hash = "^(termHash2str termHash))

    (* Take a pair (formula, part) and convert it back to its
       TemplateValue representation constrained with its index. *)

    fun pair2value termHash (formula, part) =
	  ConstrainedValue(ObjectValue(pt2termID termHash formula),
			   FormulaConstr(SOME(part, NONE)))

    (* Similarly for a list of pairs *)
    fun pairs2value termHash pairs = ListValue(List.map(pair2value termHash) pairs)

    (* Debugging print functions *)

    fun pair2str(formula, part) = "("^(pt2string formula)^": "^(formulaInd2str(part, NONE))^")"
    fun pairs2str pairs = "["^(strlist2str ", " (List.map pair2str pairs))^"]"


    (*************************************************************)
    (**************** The Cone of Influence rule *****************)
    (*************************************************************)

    (* Construct the formula suitable for model checking and generate
       a new model reduced by the cone of influence reduction relative
       to it.  Return the pair (formula, model). *)

    fun doCOI options (seq: Sequent, lim) =
	let val {model=model,
		 init=init,
		 context=context,
		 invar=invar,
		 assumptions=assump,
		 conclusions=conc} = seq
	    val {trans=trans,
		 findObject=findObject,
		 pvars=pvars,
		 abs=abs,...} = model
	    val verb = verb options
	    val verbStr = verbStr options
	    val lazyVerbDebug = lazyVerbDebug options
	    val verbDebug = verbDebug options
	    val lazyVerbDebugStr = lazyVerbDebugStr options
	    val trueVal = Builtin{name=True dp, Type=BoolType dp}
	    val falseVal = Builtin{name=False dp, Type=BoolType dp}
	    val andVal = Builtin{name=And dp, 
				 Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
					      BoolType dp)}
	    val orVal = Builtin{name=Or dp, 
				Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
					     BoolType dp)}
	    val impliesVal = Builtin{name=Implies dp, 
				     Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
						  BoolType dp)}
	    fun impliesFormula(f,g) = Appl(dp, impliesVal, TupleExpr(dp,[f,g]))
	    fun foldOp (Op,default) lst =
		let fun foldFun(f1,f2) = Appl(dp,Op,TupleExpr(dp,[f1,f2]))
		    fun doFold [] = default
		      | doFold (f::lst) = List.foldr foldFun f lst
		in doFold(List.rev lst)
		end
	    val foldAnd = foldOp(andVal, trueVal)
	    val foldOr = foldOp(orVal, falseVal)

	    val theorem = impliesFormula(foldAnd assump, foldOr conc)
	    val _ = verb("Specializing model to the property...")
	    val newModel = specializeModel options model [theorem]
	    val _ = verb("Specializing model to the property...done")
	in
	    (theorem, newModel)
	end

    val COIspecs =
	let val options = getOptions()
	    val { limit=lim, ...} = options
	in
	    [{name="limit",
	      Type=ProverArgNumberType,
	      default=DefaultValue(ProverArgNumber lim)}]
	end


    fun matchCOI(seq: Sequent, context, args: RuleArg list, subOpt) =
	    
	let (* An empty dummy substitution - we don't really need one *)
	    val sigma=(case subOpt of
			   SOME sub => sub
			 | NONE => makeSubstitution(makeTermHash()))
	(* This rule should be generally applicable, at least so far,
	   so just return SOME *)
	in SOME sigma
	end


    fun applyCOI printFun (seq: Sequent, context, args: RuleArg list, sigma: Substitution) =
	let val {init=init,
		 context=context,
		 invar=invar,
		 assumptions=assump,
		 conclusions=conc, ...} = seq
	    val lim =
		(case findArgNumberValue args "limit" of
		     NONE => raise SympBug
			 ("default ProofRules/applyCOI: the limit argument "
			  ^"is not a number or not defined")
		   | SOME n => n)
	    val (_, newModel) = doCOI (getOptions()) (seq, lim)
	    val newSeq = {model=newModel,
			  init=init,
			  context=context,
			  invar=invar,
			  assumptions=assump,
			  conclusions=conc}
	in
	    InferenceRuleSuccess[(newSeq, NONE)]
	end



    val COIRule =
	{ name="cone",
	  help=("Perform the Cone of Influence reduction on the model relative to the"
		^"current formulas in the sequent."),
	  params=COIspecs,
	  conclusion=NONE,
	  premisses=NONE,
	  match=SOME matchCOI,
	  apply=SOME applyCOI }: InferenceRule

    (*************************************************************)
    (******************* The model check rule ********************)
    (*************************************************************)

    fun matchMC(seq: Sequent, context, args: RuleArg list, subOpt) =
	    
	let (* An empty dummy substitution - we don't really need one *)
	    val sigma=(case subOpt of
			   SOME sub => sub
			 | NONE => makeSubstitution(makeTermHash()))
	(* This rule should be generally applicable, at least so far,
	   so just return SOME *)
	in SOME sigma
	end

    fun translateTo "smv" options (findObject, lim, specs, vars, init, invar, next) =
	let val _ = verb options ("Translating to SMV...")
	    val (smvStr, varHash, typeHash) = 
		  SMVcode options findObject lim specs vars init invar next
	    val _ = verb options ("Translating to SMV...done")
	in (smvStr, varHash, typeHash)
	end
      | translateTo x _ _ = raise ProverError
	  ("Unsupported back-end model checker: "^x)

    val MCspecs =
	let val options = getOptions()
	    val { limit=lim, ...} = options
	in
	    [{name="mc",
	      Type=ProverArgStringType,
	      default=DefaultValue(ProverArgString "smv")},
	     {name="limit",
	      Type=ProverArgNumberType,
	      default=DefaultValue(ProverArgNumber lim)},
	     {name="args",
	      Type=ProverArgListType ProverArgStringType,
	      default=DefaultValue(ProverArgList[])}]
	end

    fun applyMC printFun (seq: Sequent, context, args: RuleArg list, sigma: Substitution) =
	let fun op * (s1,s2) = Conc(s1,s2)
	    val {model=model,
		 init=init,
		 context=context,
		 invar=invar,
		 assumptions=assump,
		 conclusions=conc, ...} = seq
	    val {trans=trans,
		 findObject=findObject,
		 pvars=pvars,
		 abs=abs,...} = model
	    (* The back-end used to be determined in the `options', 
	       but now it's in the rule's argument *)
	    val backendMC =
		(case findArgStringValue args "mc" of
		     NONE => raise SympBug
			 ("default ProofRules/applyMC: the mc argument "
			  ^"is not a string or not defined")
		   | SOME s => s)
	    val lim =
		(case findArgNumberValue args "limit" of
		     NONE => raise SympBug
			 ("default ProofRules/applyMC: the limit argument "
			  ^"is not a number or not defined")
		   | SOME n => n)
	    val MCargs =
		(case findArgListValue args "args" of
		     NONE => raise SympBug
			 ("default ProofRules/applyMC: the `args' argument "
			  ^"is not a list or not defined")
		   | SOME lst => List.mapPartial ArgStringValue lst)
	    val options = getOptions()
	    val verb = verb options
	    val verbStr = verbStr options
	    val lazyVerbDebug = lazyVerbDebug options
	    val verbDebug = verbDebug options
	    val lazyVerbDebugStr = lazyVerbDebugStr options
(*	    val trueVal = Builtin{name=True dp, Type=BoolType dp}
	    val falseVal = Builtin{name=False dp, Type=BoolType dp}
	    val andVal = Builtin{name=And dp, 
				 Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
					      BoolType dp)}
	    val orVal = Builtin{name=Or dp, 
				Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
					     BoolType dp)}
	    val impliesVal = Builtin{name=Implies dp, 
				     Type=FunType(dp,TupleType(dp,[BoolType dp, BoolType dp]),
						  BoolType dp)}
	    fun impliesFormula(f,g) = Appl(dp, impliesVal, TupleExpr(dp,[f,g]))
	    fun foldOp (Op,default) lst =
		let fun foldFun(f1,f2) = Appl(dp,Op,TupleExpr(dp,[f1,f2]))
		    fun doFold [] = default
		      | doFold (f::lst) = List.foldr foldFun f lst
		in doFold(List.rev lst)
		end
	    val foldAnd = foldOp(andVal, trueVal)
	    val foldOr = foldOp(orVal, falseVal)

	    val theorem = impliesFormula(foldAnd assump, foldOr conc)
	    val _ = verb("Specializing model to the property...")
	    val newModel = specializeModel options model [theorem]
	    val _ = verb("Specializing model to the property...done")
*)
	    val (theorem, newModel) = doCOI options (seq, lim)
	    val { stateVars=vars, ... } = newModel
	    (* Check that the model is finite *)
	    val infVars = ref[]
	    val largeVars = ref[]
	    val error = ref (NONE: string option)
	    fun addError str =
		error := SOME(case !error of
				  NONE =>  str
				| SOME s => (s^"\n\n"^str))
	    fun checkVarType v =
		let val tp = getExprType findObject v
		    val size = getTypeSize options tp
		in
		    (case size of
		        InfiniteSize => infVars := (TypedExpr(dp,v,tp))::(!infVars)
		      | FiniteSize n =>
			    if n > lim then largeVars := ((TypedExpr(dp,v,tp)), n)::(!largeVars)
			    else ())
		end
	    val _ = List.app checkVarType vars
	    val _ = (case !infVars of
			 [] => ()
		       | lst => addError
			     ("The model has variables of infinite types:\n  "
			      ^(ptlist2str "\n  " lst)))
	    val _ = (case !largeVars of
			 [] => ()
		       | lst =>
			     let fun pr(pt, n) = (pt2string pt)^", size = "^(Int.toString n)
				 val strList = List.map pr lst
			     in addError("The model has variables of too large types:\n  "
					 ^(strlist2str "\n  " strList))
			     end)
	    val _ = (case !error of
			 NONE => ()
		       | SOME str => raise TransError str)
	    (* At this point, the model should be finite and small enough.
	       Continue with the translation. *)
	    val _ = verb("Translating theorems...")
	    val spec = boolExpr options findObject vars abs pvars lim theorem
	    val _ = verb("Translating theorems...done")
	    val _ = verb("Generating transition relation...")
	    val (invar, next, init) = generateModelTrans options newModel
	    val _ = verb("Generating transition relation...done")
		
	    val _ = lazyVerbDebugStr "common" 
		     (fn()=>(Str "\nINIT\n  ")*(pt2str init)*(Str "\n"))
	    val _ = lazyVerbDebugStr "common" 
		    (fn()=>(Str "\nINVAR\n  ")*(pt2str invar)*(Str "\n"))
	    val _ = lazyVerbDebugStr "common" 
		    (fn()=>(Str "\nTRANS\n  ")*(pt2str next)*(Str "\n"))
	    val _ = lazyVerbDebugStr "common" (fn()=>(Str "\nSPEC ")*(pt2str spec)*(Str "\n"))
	    (* Although called `smvStr', it is actually a program in
	       the language of whatever back-end model checker we are
	       using. *)
	    val (smvStr, varHash, typeHash) = 
		    translateTo backendMC options 
		      (findObject, lim, [spec], vars, init, invar, next)
	    val _ = verbDebugStr options "common" ((Str "\n\nSMV program:\n\n")
						   *smvStr
						   *(Str "-- end of SMV program\n\n"))
	    fun dump outs (Str s) = TextIO.output(outs,s)
	      | dump outs (Conc(x,y)) = ((dump outs x);(dump outs y))
	    fun wrapper "smv" = "smvrun"
	      | wrapper x = raise ProverError("Unsupported back-end model checker: "^x)
		(* Assume that the wrapper first reads the entire program from stdin,
		   and only then starts up the actual model checker and prints
		   something on stdout. *)
	    fun runExternalMC(smvStr,smvArgs) =
		let val proc = (Unix.execute("/bin/sh",
					     ["-c",
					      (wrapper backendMC)^" - -- "
					      ^(strlist2str " " smvArgs)])
				(* Any exception will be thrown in the model checker thread,
				   so we just exit with error *)
		               handle _ => (print("Can't execute "^backendMC);
					    raise SympExitError))
		    val (ins, outs) = (Unix.streamsOf proc
		               handle _ => raise ProverError
				   ("Can't execute/connect to "^backendMC))
		    val msg = ("Running "^backendMC^"...")
		    fun readOutput() = 
			let val str = TextIO.inputLine ins
			in
			    if str = "" then ()
			    else (printFun str; readOutput())
			end
		    fun isSuccess Posix.Process.W_EXITED = true
		      | isSuccess _ = false
		    (* Returns true if property is true *)
		    fun closeMC() = isSuccess(Unix.reap proc)
		in (verb(msg);
		    dump outs (smvStr); TextIO.closeOut outs;
		    readOutput();
		    (closeMC()) before (verb(msg^"done")))
		end
	in
	    if runExternalMC(smvStr,MCargs) then
		InferenceRuleSuccess([]: (Sequent * (SequentHints option)) list)
	    else InferenceRuleDisproven
	end handle TransError str => (printFun("\nCan't generate finite model:\n  "
					       ^str^"\n");
				      InferenceRuleFailure)

    val modelCheckRule =
	{ name="modelcheck",
	  help=("Runs an external model checker on the current sequent.\n"
		^"MC specifies the shell command to call the model checker,\n"
		^"and ARGS is the list of command line arguments to the program."),
	  params=MCspecs,
	  conclusion=NONE,
	  premisses=NONE,
	  match=SOME matchMC,
	  apply=SOME applyMC }: InferenceRule

    (* Delete some formulas from the sequent (weakening) *)
    val deleteRule =
	{ name = "delete",
	  help = "Deletes specified formulas from the sequent",
	  params = [{name="formulas",
		     Type=ProverArgListType ProverArgFormulaNumType,
		     default=DefaultRequired}],
	  conclusion = SOME{ model=ValueTemp "model",
			     init=ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar=[ListFormulaTemp("formulas",NONE),
				    ListFormulaTemp("invar",NONE)],
			     assumptions=[ListFormulaTemp("formulas",NONE),
					  ListFormulaTemp("assump",NONE)],
			     conclusions=[ListFormulaTemp("formulas",NONE),
					  ListFormulaTemp("conc",NONE)] },
	  premisses = SOME[{ model=ValueTemp "model",
			     init=ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar=[ListFormulaTemp("invar",NONE)],
			     assumptions=[ListFormulaTemp("assump",NONE)],
			     conclusions=[ListFormulaTemp("conc",NONE)] }],
	  match = NONE,
	  apply = NONE }: InferenceRule

    (* The "A is A" axioms: when one of the formulas in the conclusion
       also appears either in the assumptions or the invariants.
       These axioms are tried on every new sequent automatically. *)

    fun initMatch(seq, context, args: RuleArg list, subOpt: Substitution option) =
	let val sub as (_, termHash) =
	          (case subOpt of
		       SOME sub => sub
		     | NONE => makeSubstitution(makeTermHash()))
	    val seqTerm = sequent2term(seq, SOME termHash)
	    val { model=model,
		  init=init,
		  context=context,
		  invar=invar,
		  assumptions=assump,
		  conclusions=conc, ... } = seqTerm
	    (* find a common ID in the two lists *)
	    fun findCommon ([], _) = NONE
	      | findCommon (id1::l1, l2) =
		  (case List.find(fn id2 => id1 = id2) l2 of
		       SOME id => SOME id
		     | NONE => findCommon(l1,l2))
	in
	    if List.exists (compareTermIDconst termHash trueVal) conc
		orelse List.exists (compareTermIDconst termHash falseVal) invar
		orelse List.exists (compareTermIDconst termHash falseVal) assump
	    then SOME sub
	    else
		(case findCommon(invar, conc) of
		     SOME _ => SOME sub
		   | NONE => (case findCommon(assump, conc) of
				  SOME _ => SOME sub
				| NONE => NONE))
	end

    fun initApply printFun (seq, context, args, sub) =
	(case initMatch(seq, context, args, SOME sub) of
	     SOME _ => InferenceRuleSuccess[]
	   | NONE => InferenceRuleFailure)

    val initRule =
	{ name = "init",
	  help = ("Rule for the axioms `A => A', `FALSE => A', and `A => TRUE'.\n"
		  ^"Completes the proof if the invariants or assumptions have common\n"
		  ^"formulas with the conclusion part, or if the sequent is\n"
		  ^"trivially true.\n"),
	  params = [],
	  conclusion = NONE,
	  premisses = NONE,
	  match = SOME initMatch,
	  apply = SOME initApply }: InferenceRule

    (* Make a copy of a formula in the sequent *)
    val copyRule =
	{ name = "copy",
	  help = "Makes a copy of the specified formula",
	  params = [{name = "formula",
		     Type = ProverArgFormulaNumType,
		     default = DefaultRequired}],
	  conclusion = SOME{ model = ValueTemp "model",
			     init = ValueTemp "init",
			     invar = [ListFormulaTemp("formula",NONE),
				      ListFormulaTemp("invar",NONE)],
			     context=ProgramContextTemp "context",
			     assumptions = [ListFormulaTemp("formula",NONE),
					    ListFormulaTemp("assump",NONE)],
			     conclusions = [ListFormulaTemp("formula",NONE),
					    ListFormulaTemp("conc",NONE)] },
	  premisses = SOME[{ model = ValueTemp "model",
			     init = ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar = [ListFormulaTemp("formula",NONE),
				      ListFormulaTemp("formula",NONE),
				      ListFormulaTemp("invar",NONE)],
			     assumptions = [ListFormulaTemp("formula",NONE),
					    ListFormulaTemp("formula",NONE),
					    ListFormulaTemp("assump",NONE)],
			     conclusions = [ListFormulaTemp("formula",NONE),
					    ListFormulaTemp("formula",NONE),
					    ListFormulaTemp("conc",NONE)] }],
	  match = NONE,
	  apply = NONE }: InferenceRule

    (* Copy formulas from the invariant set to the assumptions *)
    val useInvarRule =
	{ name = "useinvar",
	  help = "Copies a specified formula from the invariant to the assumption part",
	  params = [{name = "formula",
		     Type = ProverArgFormulaNumType,
		     default = DefaultRequired}],
	  conclusion = SOME{ model = ValueTemp "model",
			     init = ValueTemp "init",
			     context=ProgramContextTemp "context",
			     (* The `formulas' var should match all the formulas by default *)
			     invar = [FormulaTemp "formula", ListFormulaTemp("invar",NONE)],
			     assumptions = [ListFormulaTemp("assump",NONE)],
			     conclusions = [ListFormulaTemp("conc",NONE)] },
	  premisses = SOME[{ model = ValueTemp "model",
			     init = ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar = [FormulaTemp "formula",
				      ListFormulaTemp("invar",NONE)],
			     assumptions = [ListFormulaTemp("assump",NONE),
					    FormulaTemp "formula"],
			     conclusions = [ListFormulaTemp("conc",NONE)] }],
	  match = NONE,
	  apply = NONE }: InferenceRule

    (* The `normal' cut rule, adds a formula to the assumptions and then to the conclusions.
       The cut for the invariant is more complex and requires modification of the
       formula. *)
    val caseRule =
	{ name = "case",
	  help = "Split cases on whether a given formula is true or false",
	  params = [{name = "formula",
		     (* The user supplies his own new formula *)
		     Type = ProverArgFormulaType,
		     default = DefaultRequired}],
	  conclusion = SOME{ model = ValueTemp "model",
			     init = ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar = [ListFormulaTemp("invar",NONE)],
			     assumptions = [ListFormulaTemp("assump",NONE)],
			     conclusions = [ListFormulaTemp("conc",NONE)] },
	  premisses = SOME[{ model = ValueTemp "model",
			     init = ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar = [ListFormulaTemp("invar",NONE)],
			     (* Add the new formula to the assumptions *)
			     assumptions = [FormulaTemp "formula",
					    ListFormulaTemp("assump",NONE)],
			     conclusions = [ListFormulaTemp("conc",NONE)] },
			   { model = ValueTemp "model",
			     init = ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar = [ListFormulaTemp("invar",NONE)],
			     assumptions = [ListFormulaTemp("assump",NONE)],
			     (* Add the new formula to the conclusions *)
			     conclusions = [FormulaTemp "formula",
					    ListFormulaTemp("conc",NONE)] }],
	  match = NONE,
	  apply = NONE }: InferenceRule

    (* $\forall Case_R$ rule *)
    val forallCaseRule =
	{ name = "forallcase",
	  help = "Splitting cases on all possible values of the given term",
	  params = [{name="var",
		     Type=ProverArgObjectType,
		     default=DefaultRequired},
		    {name="term",
		     Type=ProverArgObjectType,
		     default=DefaultRequired},
		    {name="formula",
		     Type=ProverArgFormulaNumType,
		     default=DefaultMatch NONE}],
	  conclusion = SOME{ model = ValueTemp "model",
			     init = ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar = [ListFormulaTemp("invar",NONE)],
			     (* Add the new formula to the assumptions *)
			     assumptions = [ListFormulaTemp("assump",NONE)],
			     conclusions = [FormulaTemp "formula",
					    ListFormulaTemp("conc",NONE)] },
	  premisses = SOME[{ model = ValueTemp "model",
			     init = ValueTemp "init",
			     context=ProgramContextTemp "context",
			     invar = [ListFormulaTemp("invar",NONE)],
			     (* Add the new formula to the assumptions *)
			     assumptions = [ListFormulaTemp("assump",NONE)],
			     conclusions = [ListFormulaTemp("conc",NONE),
					    ForallTemp([FormulaTemp "var"],
						       impliesFormulaTemp
						         (eqExprTemp(FormulaTemp "var",
								     FormulaTemp "term"),
							  FormulaTemp "formula"),
							FormulaTemp "context")] }],
	  match = NONE,
	  apply = NONE }: InferenceRule

    (* Flatten rule: eliminate all conjunctions in the invariants and
       assumptions and disjunctions in the conclusions.  For now, it
       doesn't take any arguments. *)

    fun flattenMatch(seq, context, args: RuleArg list, subOpt) =
	let val sub as (_, termHash) =
	          (case subOpt of
		       SOME sub => sub
		     | NONE => makeSubstitution(makeTermHash()))
	    val seqTerm = sequent2term(seq, SOME termHash)
	    val { model=model,
		  init=init,
		  context=context,
		  invar=invar,
		  assumptions=assump,
		  conclusions=conc, ... } = seqTerm
	    (* Flag whether the sequent has changed *)
	    val changed = ref false
	    (* Extract arguments of "and", "or", "not", and "implies" *)
	    fun splitArgs id =
		(case termID2term termHash id of
		     TupleTerm lst => SOME lst
		   | _ => SOME[id])

	    fun splitOp Op id =
		let val term = termID2term termHash id
		in
		    case term of
			ApplTerm(opId, argId) =>
			    (if compareTermIDconst termHash Op opId then splitArgs argId
			     else NONE)
		      | _ => NONE
		end

	    (* In the invariant, only split conjunctions.  Negations
	       are kept in to preserve completeness. *)
	    fun splitInvar id = Option.map(fn l => (l, [], [])) (splitOp andVal id)

	    (* Assumptions are split on conjunctions and negations *)
	    fun splitAssump id =
		  (case splitOp andVal id of
		       (* Keep conjuncts in the assumptions *)
		       SOME l => SOME([], l, [])
		     | NONE => 
			   (case splitOp notVal id of
				(* Move un-negated formulas to conclusions *)
				SOME l => SOME([], [], l)
			      | NONE => NONE))

	    fun splitConc id =
		  (case splitOp orVal id of
		       SOME l => SOME([], [], l)
		     | NONE => 
			   (case splitOp impliesVal id of
				SOME[id1,id2] => SOME([], [id1], [id2])
			      | NONE =>
				    (case splitOp notVal id of
					 SOME l => SOME([], l, [])
				       | NONE => NONE)
			      | SOME l => raise SympBug
				  ("ProofRulesDefault/flattenMatch/splitConc: "
				   ^"splitImplies returned bad list:\n  ["
				   ^(strlist2str ", " (List.map termID2str l))
				   ^"]")))

	    (* 6 arguments: 3 lists of formulas to split, 3 lists of
	       processed formulas.  War plan: look at the first
	       formula in each "to do" list, and if it splits, put the
	       parts in the corresponding "to do" lists again; if it
	       doesn't split, move it to the corresponding processed
	       list *)

	    fun loop([], [], [], invarDone, assumpDone, concDone) =
		  (List.rev invarDone, List.rev assumpDone, List.rev concDone)
	      | loop(x::invar, assump, conc, invarDone, assumpDone, concDone) =
		  (case splitInvar x of
		       NONE => loop(invar, assump, conc, x::invarDone, assumpDone, concDone)
		     | SOME(l1,l2,l3) => (loop(l1@invar, l2@assump, l3@conc,
					       invarDone, assumpDone, concDone)
					  before changed := true))
	      | loop (invar, x::assump, conc, invarDone, assumpDone, concDone) =
		  (case splitAssump x of
		       NONE => loop(invar, assump, conc, invarDone, x::assumpDone, concDone)
		     | SOME(l1,l2,l3) => (loop(l1@invar, l2@assump, l3@conc,
					       invarDone, assumpDone, concDone)
					  before changed := true))
	      | loop (invar, assump, x::conc, invarDone, assumpDone, concDone) =
		  (case splitConc x of
		       NONE => loop(invar, assump, conc, invarDone, assumpDone, x::concDone)
		     | SOME(l1,l2,l3) => (loop(l1@invar, l2@assump, l3@conc,
					       invarDone, assumpDone, concDone)
					  before changed := true))
	    val (invar, assump, conc) = loop(invar, assump, conc, [], [], [])
	    val invarValue = ListValue(List.map (fn id => ObjectValue id) invar)
	    val assumpValue = ListValue(List.map (fn id => ObjectValue id) assump)
	    val concValue = ListValue(List.map (fn id => ObjectValue id) conc)
	    fun updateSub() = 
		let val sOpt = updateSubstitution("model", ModelValue model) sub
		    val sOpt = Option.mapPartial
			         (updateSubstitution("init", InitValue init)) sOpt
		    val sOpt = Option.mapPartial
			         (updateSubstitution("context", ProgramContextValue context)) sOpt
		    val sOpt = Option.mapPartial
			         (updateSubstitution("invar", invarValue)) sOpt
		    val sOpt = Option.mapPartial
			         (updateSubstitution("assump", assumpValue)) sOpt
		    val sOpt = Option.mapPartial
			         (updateSubstitution("conc", concValue)) sOpt
		in
		    sOpt
		end
	in
	    if !changed then updateSub() else NONE
	end

    fun flattenApply printFun (seq, context, args, sigma) =
	let val seqTemp = { model = ValueTemp "model",
			    init = ValueTemp "init",
			    context=ProgramContextTemp "context",
			    invar = [ListFormulaTemp("invar",NONE)],
			    assumptions = [ListFormulaTemp("assump",NONE)],
			    conclusions = [ListFormulaTemp("conc",NONE)] }
	    val newSub = flattenMatch(seq, context, args, SOME sigma)
	    val newSeq = Option.mapPartial(fn sub=>substSequent sub seqTemp) newSub
	in
	    case newSeq of
		SOME seq => InferenceRuleSuccess[(seq,NONE)]
	      | NONE => InferenceRuleFailure
	end

    val flattenRule =
	{ name = "flatten",
	  help = ("Propositionally flattens the sequent.  That is, applies disjunction on"
		  ^"the right, conjunction on the left, and rules for negation as much as"
		  ^"possible."),
	  params = [],
	  conclusion = NONE,
	  premisses = NONE,
	  match = SOME flattenMatch,
	  apply = SOME flattenApply }: InferenceRule

    (* Universal Skolemization *)
    val skolemSpecs = 
	[{ name="formula",
	   Type=ProverArgFormulaNumType,
	   default=DefaultValue(ProverArgFormulaNum(AnyPart, NONE)) } ]

    fun skolemMatch(seq, context, args: RuleArg list, subOpt: Substitution option) =
	let val _ = pushFunStackLazy("skolemMatch",
				     fn()=>((sequent2string seq)^", "
					    ^(case subOpt of
						  SOME sub => subst2str sub
						| NONE => "Substitution = NONE")))
	    val sub as (_, termHash) =
	          (case subOpt of
		       SOME sub => sub
		     | NONE => makeSubstitution(makeTermHash()))
	    val seqTerm = sequent2term(seq, SOME termHash)
	    val { model=model,
		  init=init,
		  context=context,
		  invar=invar,
		  assumptions=assump,
		  conclusions=conc, ... } = seqTerm

	    (* Three versions of the template for matching only conclusions, only
	       assumptions, and only invariants *)

	    val seqTempConc =
		{ model = ValueTemp "model",
		  init = ValueTemp "init",
		  context = ProgramContextTemp "context",
		  invar = [ListFormulaTemp("invar", NONE)],
		  assumptions = [ListFormulaTemp("assump", NONE)],
		  conclusions = [ListFormulaTemp
				 ("formula",
				  SOME(ForallTemp([ListWildCardTemp],WildCardTemp,WildCardTemp))),
				 ListFormulaTemp("conc", NONE)] }
	    val seqTempAssump =
		{ model = ValueTemp "model",
		  init = ValueTemp "init",
		  context = ProgramContextTemp "context",
		  invar = [ListFormulaTemp("invar", NONE)],
		  assumptions = [ListFormulaTemp
				 ("formula",
				  SOME(ExistsTemp([ListWildCardTemp],WildCardTemp,WildCardTemp))),
				 ListFormulaTemp("assump", NONE)],
		  conclusions = [ListFormulaTemp("conc", NONE)] }
	    val seqTempInvar =
		{ model = ValueTemp "model",
		  init = ValueTemp "init",
		  context = ProgramContextTemp "context",
		  invar = [ListFormulaTemp
			   ("formula",
			    SOME(ExistsTemp([ListWildCardTemp],WildCardTemp,WildCardTemp))),
			   ListFormulaTemp("invar", NONE)],
		  assumptions = [ListFormulaTemp("assump", NONE)],
		  conclusions = [ListFormulaTemp("conc", NONE)] }

	    (* Matching one of the templates.  If something matched
	       previously, keep that match.  Otherwise try to match. *)
	    fun tryMatch (seqTemp, SOME newSub) = SOME newSub
	      | tryMatch (seqTemp, NONE) =
		let val seqTemp = applySeqArgs(seqTemp, args)
		    val newSub = matchSeq sub (seqTemp, NONE, seqTerm)
		    (* Take the first formula from the "formula"
		       list and return (f, rest) pair, or NONE *)
		    fun getFirst sub =
			(case subst sub AnyConstr (ListFormulaTemp("formula", NONE)) of
				 SOME(ListValue(hd::tl)) => SOME(sub, hd, tl)
			       | SOME(ListValue[]) => NONE
			       | SOME x => raise SympBug
				     ("skolemMatch/tryMatch: 'formula' is not a ListValue:\n  "
				      ^(value2str x))
			       | NONE => raise SympBug
				     ("skolemMatch/tryMatch: 'formula' is undefined "
				      ^"in the substitution"))
		    (* Move the formula `v' to the "formula1" var from the `formula' list *)
		    fun ff(sub, v,lst) =
			(updateSubstitutionRaw("formula1", v)
			      (updateSubstitutionRaw("formula", ListValue lst) sub))
		in
		    Option.map ff (Option.mapPartial getFirst newSub)
		end
	    (* Try matching in the listed order *)
	    val res = List.foldl tryMatch NONE [seqTempConc,seqTempAssump,seqTempInvar]
	    val _ = popFunStackLazy("skolemMatch",
				    fn()=>(case res of
					       SOME sub => subst2str sub
					     | NONE => "Substitution = NONE"))
	in res
	end

    (* Given the name (string), find an unused name based on it for a Skolem constant *)
    fun newConst (name,tp,cxt) =
	let fun loop n =
	    let val newName = Id(dp, name^(Int.toString n))
	    in
		if isSome(findNameInContext cxt newName) then loop(n+1)
		else newName
	    end
	    val name' = 
		if isSome(findNameInContext cxt (Id(dp,name))) then loop 1
		else  Id(dp,name)
	in
	    SkolemConst{name=name',
			uname=Uid(newName()),
			Type=tp}
	end

    fun updateCxt (SequentClosure{names=names, parent=p}) obj =
	   SequentClosure{names=obj::names, parent=p}
      | updateCxt cxt obj =
	   SequentClosure{names=[obj], parent=cxt}

    (* Generate Skolem constants for all names, and return the 
       list of pairs (var, const) - both termIDs, and the new context *)
    fun newSkolemConsts cxt vars =
	let fun loop [] cxt accum = (accum, cxt)
	      | loop (var::vars) cxt accum =
	        let (* val varPt = termID2pt termHash var *)
		    val name =
			(case GetObjectName var of
			     SOME str => str
			   | NONE => raise SympBug
				 ("newSkolemConsts: bound var has no name:\n  "
				  ^(pt2string var)))
		    val tp =
			(case GetObjectType var of
			     SOME tp => tp
			   | NONE => raise SympBug
				 ("newSkolemConsts: bound var has no type:\n  "
				  ^(pt2string var)))
		    val nameStr = (case name of
				       Id(_, str) => str
				     | _ => "a")
		    val const = newConst(nameStr,tp,cxt)
		    (* val const = pt2termID termHash constPt *)
		in 
		    loop vars (updateCxt cxt const) ((var, const)::accum)
		end
	in
	    loop vars cxt []
	end

    fun skolemApply printFun (seq: Sequent, context, args: RuleArg list, sigma: Substitution) =
	let val debug = lazyVerbDebug (getOptions()) "skolemApply"
	    val {init=init,
		 context=context,
		 invar=invar,
		 assumptions=assump,
		 conclusions=conc, ...} = seq
	    val _ = pushFunStackLazy("skolemApply",
				     fn()=>((sequent2string seq)^", "
					    ^(subst2str sigma)))
	    val (cxt, findObject) = context
	    val subOpt = skolemMatch(seq, context, args, NONE)
	    val _ = debug(fn()=>("\nskolemApply: subOpt = "
				 ^(case subOpt of
				       NONE => "NONE"
				     | SOME sub => subst2str sub)))
	    (* Strip all the constraints and return the pair (constraints, value) *)
	    fun stripConstr(ConstrainedValue(v, c)) constr =
		  stripConstr v (c::constr)
	      | stripConstr v constr = (constr, v)
	    (* Wrap the constraints back on the value *)
	    fun wrapConstr v (c::constr) = wrapConstr (ConstrainedValue(v, c)) constr
	      | wrapConstr v [] = v

	    fun getFormulaInd(ConstrainedValue(v, FormulaConstr(SOME ind))) = ind
	      | getFormulaInd(ConstrainedValue(v, _)) = getFormulaInd v
	      | getFormulaInd _ = raise SympBug
		 ("skolemApply/getFormulaInd: no formula index found") 
	    (* Instantiate all the vars with the Skolem constants *)
	    fun instList termHash id [] = id
	      | instList termHash id (pair::lst) =
		  instList termHash (substTermID termHash pair id) lst
            (* Do the actual Skolemization and construct the final sequent *)
	    fun skolem sub =
		let val _ = pushFunStackLazy("skolemApply/skolem",
					     fn()=>(subst2str sub))
		    val (_, termHash) = sub
		    val v = (case subst sub AnyConstr (FormulaTemp "formula1") of
				    SOME v => v
				  | NONE => raise SympBug
					("skolemApply/skolem: `formula1' is undefined"))
		    val formulaInd = getFormulaInd v

		    (* Strip the value from constraints.  Initialize the accumulator with
		       some bogus value *)

		    val (constr, v) = stripConstr v []
		    val id = (case v of
				  ObjectValue id => id
				| _ => raise SympBug
				      ("skolemApply/skolem: formula is not an ObjectValue:\n  "
				       ^(value2str v)))
		    (* Extract the quantifier parts *)
		    val {names=names, body=body, parent=parent} =
			(case termID2term termHash id of
			     ForallTerm parms => parms
			   | ExistsTerm parms => parms
			   | t => raise SympBug
				 ("skolemApply/skolem: formula is not a quantifier:\n  "
				  ^(term2strHash t)))
		    val (pairsPt, cxt) = newSkolemConsts cxt (List.map (termID2pt termHash) names)
		    val pairs = 
			  List.map(fn(v,c)=>(pt2termID termHash v, pt2termID termHash c)) pairsPt
		    val formulaID = instList termHash body pairs
		    val pcxt = ProgramContextValue(cxt, findObject)
		    (* insert the new ID in place of `formula1' and generate the final sequent *)
		    val newSub = 
			(case (updateListSubstitution
			       ("formula", [wrapConstr(ObjectValue formulaID) constr])
			       (updateSubstitutionRaw("context",pcxt) sub)) of
			     SOME sub => sub
			   | NONE => raise SympBug
				 ("skolemApply/skolem: updating `formula' in substitution failed"))
		    val seqTemp = 
			{ model = ValueTemp "model",
			  init = ValueTemp "init",
			  context = ProgramContextTemp "context",
			  invar = [ListFormulaTemp("formula", NONE),
				   ListFormulaTemp("invar", NONE)],
			  assumptions = [ListFormulaTemp("formula", NONE),
					 ListFormulaTemp("assump", NONE)],
			  conclusions = [ListFormulaTemp("formula", NONE),
					 ListFormulaTemp("conc", NONE)] }
		    val res = substSequent newSub seqTemp
		    val _ = popFunStackLazy("skolemApply/skolem",
					    fn()=>(case res of
						       SOME seq => sequent2string seq
						     | NONE => "Sequent = NONE"))
		in
		    res
		end
	    val resOpt = Option.mapPartial skolem subOpt
	    val _ = popFunStackLazy("skolemApply",
				    fn()=>(case resOpt of
					       NONE => "NONE"
					     | SOME seq => sequent2string seq))
	in
	    case resOpt of
		NONE => InferenceRuleFailure
	      | SOME seq => InferenceRuleSuccess[(seq,NONE)]
	end

    val skolemRule =
	{ name = "skolem",
	  help=("Skolemize a universal quantifier in the conclusion or an existential"
		^"quantifier in the assumption or the invariant."),
	  params=skolemSpecs,
	  conclusion=NONE,
	  premisses=NONE,
	  match=SOME skolemMatch,
	  apply=SOME skolemApply }: InferenceRule

    (* Existential instantiation *)
    val instSpecs = 
	[{ name="terms",
	   Type=ProverArgListType ProverArgObjectType,
	   default=DefaultRequired },
	 { name="formula",
	   Type=ProverArgFormulaNumType,
	   default=DefaultValue(ProverArgFormulaNum(AnyPart, NONE)) } ]

    fun instMatch(seq, context, args: RuleArg list, subOpt: Substitution option) =
	let val sub as (_, termHash) =
	          (case subOpt of
		       SOME sub => sub
		     | NONE => makeSubstitution(makeTermHash()))
	    val seqTerm = sequent2term(seq, SOME termHash)
	    val { model=model,
		  init=init,
		  context=PrContext,
		  invar=invar,
		  assumptions=assump,
		  conclusions=conc, ... } = seqTerm
	    val (context, findObject) = PrContext
	    (* Extract the type from the term supplied by the user *)
	    fun extractObject arg = 
		(case ArgObjectValue arg of
		     SOME obj => obj
		   | NONE => raise SympBug
			 ("instMatch/extractObject: argument not an `ArgObject'"))
	    fun extractObjType(term, findObject) = getExprType findObject term
	    val termArgs = (case findArg args "terms" of
				SOME arg => 
				    (case ArgListValue arg of
					 SOME lst => lst
				       | NONE => raise SympBug
					     ("instMatch: `terms' argument is not a list"))
			      | NONE => raise SympBug
					 ("instMatch: The argument `terms' is not found"))
	    val terms = List.map extractObject termArgs
	    val termsPt = List.map ProgramObject2pt terms
	    (* val termTypes = List.map extractObjType terms *)
	    (* The list of wild card templates of the corresponding length *)
	    val termWCTemps = List.map (fn _ => WildCardTemp) terms

	    (* Three versions of the template for matching only conclusions, only
	       assumptions, and only invariants *)

	    val seqTempConc =
		{ model = ValueTemp "model",
		  init = ValueTemp "init",
		  context = ProgramContextTemp "context",
		  invar = [ListFormulaTemp("invar", NONE)],
		  assumptions = [ListFormulaTemp("assump", NONE)],
		  conclusions = [ListFormulaTemp
				 ("formula",
				  SOME(ExistsTemp(termWCTemps,WildCardTemp,WildCardTemp))),
				 ListFormulaTemp("conc", NONE)] }
	    val seqTempAssump =
		{ model = ValueTemp "model",
		  init = ValueTemp "init",
		  context = ProgramContextTemp "context",
		  invar = [ListFormulaTemp("invar", NONE)],
		  assumptions = [ListFormulaTemp
				 ("formula",
				  SOME(ForallTemp(termWCTemps,WildCardTemp,WildCardTemp))),
				 ListFormulaTemp("assump", NONE)],
		  conclusions = [ListFormulaTemp("conc", NONE)] }
	    val seqTempInvar =
		{ model = ValueTemp "model",
		  init = ValueTemp "init",
		  context = ProgramContextTemp "context",
		  invar = [ListFormulaTemp
			   ("formula",
			    SOME(ForallTemp(termWCTemps,WildCardTemp,WildCardTemp))),
			   ListFormulaTemp("invar", NONE)],
		  assumptions = [ListFormulaTemp("assump", NONE)],
		  conclusions = [ListFormulaTemp("conc", NONE)] }

	    (* Extract the list of vars from the quantifier *)
	    fun getParts (ForallTerm{names=names, body=body, ...}) = (names, body)
	      | getParts (ExistsTerm{names=names, body=body, ...}) = (names, body)
	      | getParts x = raise SympBug
		  ("instMatch/getParts: not a quantifier term: "
		   ^(term2strHash x))

	    (* Try to restrict the expression to the type and typecheck it.  If it
	       works, return the new typechecked expression associated with the
	       corresponding variable, otherwise return NONE. *)

	    fun tryTypeCheck(expr, tp) =
		let val (newE, _, _) = typeCheckExpr(getOptions()) context (expr, tp)
		in
		    SOME newE
		end handle TypeError _ => NONE

	    (* Check that the types of the two terms unify, and if
	       they do, return true ; otherwise return false.  All
	       terms are given as TermID's. *)

	    fun getTermIDType termHash id = getExprType findObject (termID2pt termHash id)
	    fun checkType termHash (e, tp) = isSome(tryTypeCheck(e, tp))

	    (* Substitute vars with terms from the list of pairs (var,
	       term) in expression given by `id'. *)

	    fun substList termHash [] id = SOME id
	      | substList termHash ((old, new)::lst) id =
		let val tmp = substTermID termHash (old, new) id
		in
		    substList termHash lst tmp
		end

	    (* Strip all the constraints and return the pair (constraints, value) *)
	    fun stripConstr(ConstrainedValue(v, c)) constr =
		  stripConstr v (c::constr)
	      | stripConstr v constr = (constr, v)
	    (* Wrap the constraints back on the value *)
	    fun wrapConstr v (c::constr) = wrapConstr (ConstrainedValue(v, c)) constr
	      | wrapConstr v [] = v

	    (* Try to match terms from the list with the correct
	       variable types; pick the first one that matches and
	       return its list of vars and the body, and the other
	       formulas in the unmatched list. *)

	    fun tryMatchList [] unmatched = NONE
	      | tryMatchList (hd::lst) unmatched =
		let val (constr, v) = stripConstr hd []
		    val id = (case v of
				  ObjectValue id => id
				| _ => raise SympBug
				  ("instMatch/tryMatchList: formula is not an object value:\n  "
				   ^(value2str v)))
		    val (varIDs, body) = getParts(termID2term termHash id)
		    val varTypes = List.map (getTermIDType termHash) varIDs
		    val termTypePairs = 
			(case zipOpt(termsPt, varTypes) of
			     SOME lst => lst
			   | NONE => raise SympBug
			  ("instMatch/tryMatchList: number of terms and vars differ:\n  Terms: ["
			   ^(ptlist2str ", " termsPt)^"]\n Vars: ["
			   ^(ptlist2str ", " (List.map (termID2pt termHash) varIDs))^"]"))
		    fun wrapUp(varIDs, body, unmatched) =
			let val termValues = List.map (pt2termID termHash) termsPt
			    val pairs = (case zipOpt(varIDs, termValues) of
					     SOME lst => lst
					   | NONE => raise SympBug
			  ("instMatch/tryMatch: number of terms and vars differ:\n  Terms: ["
			   ^(ptlist2str ", " termsPt)^"]\n Vars: ["
			   ^(ptlist2str ", " (List.map (termID2pt termHash) varIDs))^"]"))
			    val newExprOpt = substList termHash pairs body
			    fun eval id =
				let val e = termID2pt termHash id
				    val newExpr = evaluateExpr(getOptions()) findObject e
				in
				    pt2termID termHash newExpr
				end
			in Option.map(fn e=> (wrapConstr(ObjectValue(eval e)) constr, unmatched))
			             newExprOpt
			end
		in
		    if List.all (checkType termHash) termTypePairs then
			wrapUp(varIDs, body, (List.rev unmatched)@lst)
		    else tryMatchList lst (hd::unmatched)
		end

(* 		    fun substFun (body, vars) (ListValue lst) =
			let fun conv(ObjectValue id) = id
			      | conv x = raise AbortRule("Not a term: "^(value2str x))
			in
			    ObjectValue(case substList termHash (vars, List.map conv lst) body of
					    SOME id => id
					  | NONE => body)
			end
		      | substFun _ v = raise AbortRule("The argument is not a list of terms: "
						       ^(value2str v))
*)

	    (* Matching one of the templates.  If something matched
	       previously, keep that match.  Otherwise try to match. *)
	    fun tryMatch (seqTemp, SOME newSub) = SOME newSub
	      | tryMatch (seqTemp, NONE) =
		let val seqTemp = applySeqArgs(seqTemp, args)
		    val newSub = matchSeq sub (seqTemp, NONE, seqTerm)
		    fun ff sub =
			(* Take the first formula from the "formula"
			   list that matches the types, and put it in the "formula1" var *)
			let val pairOpt = 
			    (case subst sub AnyConstr (ListFormulaTemp("formula", NONE)) of
				 SOME(ListValue lst) => tryMatchList lst []
			       | SOME v => raise SympBug
				     ("instMatch/tryMatch: 'formula' is not a list: "
				      ^(value2str v))
			       | NONE => raise SympBug
				     ("instMatch/tryMatch: 'formula' is undefined "
				      ^"in the substitution"))
			    fun updateSub (newExpr, lst) =
				 updateSubstitutionRaw("formula", ListValue(newExpr::lst)) sub
			in
			    Option.map updateSub pairOpt
			end
		in
		    Option.mapPartial ff newSub
		end
	in  (* Try matching in the listed order *)
	    List.foldl tryMatch NONE [seqTempConc,seqTempAssump,seqTempInvar]
	end

    fun instApply printFun (seq: Sequent, context, args: RuleArg list, sigma: Substitution) =
	let val {init=init,
		 context=context,
		 invar=invar,
		 assumptions=assump,
		 conclusions=conc, ...} = seq
	    val (cxt, findObject) = context
	    val subOpt = instMatch(seq, context, args, NONE)
	    (* Strip all the constraints and return the pair (constraints, value) *)
	    fun stripConstr(ConstrainedValue(v, c)) constr =
		  stripConstr v (ConstrainedValue(constr, c))
	      | stripConstr v constr = (constr, v)
	    (* Wrap the constraints back on the value *)
	    fun wrapConstr v (ConstrainedValue(v', c)) = wrapConstr (ConstrainedValue(v, c)) v'
	      | wrapConstr v _ = v
	    fun getFormulaInd(ConstrainedValue(v, FormulaConstr(SOME ind))) = ind
	      | getFormulaInd(ConstrainedValue(v, _)) = getFormulaInd v
	      | getFormulaInd _ = raise SympBug
		 ("instApply/getFormulaInd: no formula index found") 
            (* Do the actual instantiation and construct the final sequent *)
	    fun inst sub =
		let val seqTemp = 
			{ model = ValueTemp "model",
			  init = ValueTemp "init",
			  context = ProgramContextTemp "context",
			  invar = [ListFormulaTemp("formula", NONE),
				   ListFormulaTemp("invar", NONE)],
			  assumptions = [ListFormulaTemp("formula", NONE),
					 ListFormulaTemp("assump", NONE)],
			  conclusions = [ListFormulaTemp("formula", NONE),
					 ListFormulaTemp("conc", NONE)] }
		in
		    substSequent sub seqTemp
		end
	    val resOpt = Option.mapPartial inst subOpt
	in
	    case resOpt of
		NONE => InferenceRuleFailure
	      | SOME seq => InferenceRuleSuccess[(seq,NONE)]
	end

    val instRule =
	{ name = "inst",
	  help=("Instantiate an existential quantifier in the conclusion or a universal"
		^"quantifier in the assumption or the invariant."),
	  params=instSpecs,
	  conclusion=NONE,
	  premisses=NONE,
	  match=SOME instMatch,
	  apply=SOME instApply }: InferenceRule

    (* Replace Rule *)

    val replaceHelp = 
	"Rewriting rule: if an assumption (init or invar) has a form A = B,\n"
	^"substitute A with B wherever possible, except in the formula itself.\n"
	^"If it's an \"init\" formula, replace only in propositional\n"
	^"(non-temporal) formulas and the initial assignments of the model.  If\n"
	^"it's an \"invar\" formula, replace everywhere.\n\n"

	^"If the specified formula F is not an equality, then treat it as F =\n"
	^"true.  If the formula is in the conclusion, treat it as an \"init\"\n"
	^"formula of the form \"F = false\".\n\n"

	^"Arguments: \n\n"

	^"FORMULA is the list of formulas taken as rewrite rules; by default,\n"
	^"all the formulas from the assumption and invariant parts are collected.\n"
	^"The rewriting is done **in parallel**, that is, the rewrite rules\n"
	^"(selected formulas) do not change while the rule is being applied.\n\n"

	^"TARGET is a string list describing what to rewrite.  Default value\n"
	^"is [\"all\"].  Other possible values: \"model\" and formula number\n"
	^"specifications.  All mentioned objects from the list are rewritten, if\n"
	^"possible.\n\n"

	^"REVERSE is a boolean string; if the value is \"true\", \"t\", or \"yes\",\n"
	^"then a formula A=B replaces B with A (that is, in the reverse\n"
	^"direction)."

    val replaceSpecs =
	[{ name="formulas",
	   Type=ProverArgListType ProverArgFormulaNumType,
	   default=DefaultValue
	   (ProverArgList[ProverArgFormulaNum(SomeParts[InvarPart,AssumptionPart], NONE)]) },
	 { name="target",
	   Type=ProverArgListType ProverArgStringType,
	   default=DefaultValue(ProverArgList[ProverArgString "all"]) },
	 { name="reverse",
	   Type=ProverArgStringType,
	   default=DefaultValue(ProverArgString "false")} ]

    fun replaceMatch(seq, context, args: RuleArg list, subOpt) =
	let val sub as (_, hash) =
	           (case subOpt of
			SOME sub => sub
		      | NONE => makeSubstitution(makeTermHash()))
	    val seqTerm as {hash = hash, ...} = sequent2term(seq, SOME hash)
	    (* val a2bool = FunType(dp,TupleType(dp,[Tvar(dp,"a"),Tvar(dp,"a")]),BoolType dp)
	    val defaultTemplate = ApplTemp(BuiltinTemp(Builtin{name=Eq dp, Type=a2bool}),
					   WildCardTemp)
	    val temp = (case findArgListValue args "formula" of
			    NONE => SOME(defaultTemplate)
			  | SOME _ => NONE) *)
	    val seqTemp = { model = ValueTemp "model",
			    init = ValueTemp "init",
			    context = ProgramContextTemp "context",
			    invar = [ListFormulaTemp("formulas", NONE),
				     ListFormulaTemp("invar", NONE)],
			    assumptions = [ListFormulaTemp("formulas", NONE),
					   ListFormulaTemp("assump", NONE)],
			    conclusions = [ListFormulaTemp("formulas", NONE),
					   ListFormulaTemp("conc", NONE)] }
	in
	    matchSeq sub (applySeqArgs(seqTemp, args), NONE, seqTerm)
	end

    fun replaceApply printFun (seq: Sequent, context, args: RuleArg list, sub) =
	let val funName = "replaceApply"
	    fun argsFn() = ("seq = "^(sequent2string seq)
			    ^", args = ["^(strlist2str ", " (List.map RuleArg2string args))^"]")
	    val _ = pushFunStackLazy(funName, argsFn)
	    (* Datatype for sequent parts proper *)
	    (* datatype LocalPart = Invar | Assump | Conc *)
	    (* fun part2local ConclutionPart = Conc
	      | part2local AssumptionPart = Assump
	      | part2local InvarPart = Invar *)

	    val subOpt = replaceMatch(seq, context, args, SOME sub)
	    fun isNotProp (Builtin{name=name,...}) = isNotProp name
	      | isNotProp (Object{def=def,...}) = isNotProp def
	      | isNotProp (MuClosure _) = true
	      | isNotProp (NuClosure _) = true
	      | isNotProp x = isTemporal x

	    (* For a quick hack: transform everything to the
	       ParseTree, replace as we usually do, and convert back
	       to the termHash representation. *)

	    (* If `prop' is true, do not substitute inside temporal operators *)
	    fun replaceFormula prop (term, old, new) =
		let fun recur t = ptTransform subst t
		    and subst t =
			if ptEq(old, t) then new
			else (case t of
				  FunClosure{formals=f,...} => 
				      if eqUNames(f, old) then t else recur t
				| Appl(_, Op, _) => 
				      if prop andalso isNotProp Op then t else recur t
				| _ => recur t)
		in subst term
		end
	    fun replaceAVT initOnly (tree, old, new) =
		let fun subst e = replaceFormula false (e, old, new)
		    fun substVar(NormalVar v) = NormalVar(subst v)
		      | substVar(NextVar v) = NextVar(subst v)
		      | substVar(InitVar v) = InitVar(subst v)
		    fun substVars{init=initV, next=nextV, norm=normV} =
			let val initV = List.map substVar initV
			    val nextV = if initOnly then nextV
					else List.map substVar nextV
			    val normV = if initOnly then normV
					else List.map substVar normV
			in {init=initV, next=nextV, norm=normV}
			end			  
		    fun loop(t as NormalAsstTree(vars, v, e)) =
		          if initOnly then t
			  else NormalAsstTree(substVars vars, subst v, subst e)
		      | loop(t as NextAsstTree(vars, v, e)) =
		          if initOnly then t
			  else NextAsstTree(substVars vars, subst v, subst e)
		      | loop(t as InitAsstTree(vars, v, e)) = 
			  InitAsstTree(substVars vars, subst v, subst e)
		      | loop(NopAsstTree vars) = NopAsstTree(substVars vars)
		      | loop(ListAsstTree(vars, lst)) =
			  ListAsstTree(substVars vars, List.map loop lst)
		      | loop(LetAsstTree(vars, decls, t)) =
			  LetAsstTree(substVars vars, decls, loop t)
		      | loop(CaseAsstTree(vars, sel, pairs)) =
			  let fun ff(p, t) = (subst p, loop t)
			  in CaseAsstTree(substVars vars, subst sel, List.map ff pairs)
			  end
		      | loop(IfAsstTree(vars, pairs, t)) =
			  let fun ff(c, t) = (subst c, loop t)
			  in IfAsstTree(substVars vars, List.map ff pairs, loop t)
			  end
		      | loop(ChooseAsstTree(vars, paramsOpt, pairs)) =
			  let fun ff(c, t) = (subst c, loop t)
			  in ChooseAsstTree(substVars vars, paramsOpt, List.map ff pairs)
			  end
		      | loop(ForeachAsstTree(vars, params, t)) =
			  ForeachAsstTree(substVars vars, params, loop t)
		      | loop(LabeledAsstTree(vars, label, t)) =
			  LabeledAsstTree(substVars vars, label, loop t)
		in loop tree
		end
	    fun replaceModel initOnly (model: Model, old, new) =
		let val {trans=trans,
			 findObject=findObject,
			 pvars=pvars,
                         abs=abs,...} = model
		    val globalPvars = ref(makePrimitiveVars())
		    val stateVars = ref []
		    fun replaceAtomicModel(AtomicModel{name=name,
						       uname=uname,
						       assts=assts,
						       cone=cone,
						       pvars=pvars,
						       absModules=absModules}) =
			let val options = getOptions()
			    val newAssts = replaceAVT initOnly (assts, old, new)
			    (* Rebuilding the tree may introduce vars that
			       were previously removed with the cone, so do
			       not use the actual tree, only cone and
			       pvars. *)
			    val opt = if initOnly then
				        rebuildAsstOptionsDefault
				      else rebuildAsstOptionsBalance rebuildAsstOptionsDefault
			    val (_, newCone, newPvars) =
			       rebuildAsstVars options findObject pvars (get_limit options)
			           (NONE, newAssts, opt)
			in
			    (* TODO: handle `absModules'; now they are left untouched *)
			    AtomicModel{name=name,
					uname=uname,
					assts=newAssts,
					cone=newCone,
					pvars=newPvars,
					absModules=absModules}
			end
		    fun replaceTrans (TransSync2(t1,t2)) =
			  TransSync2(replaceTrans t1, replaceTrans t2)
		      | replaceTrans (TransAsync2(t1,t2)) =
			  TransAsync2(replaceTrans t1, replaceTrans t2)
		      | replaceTrans (TransSync{names=names,
						body=t,
						parent=p}) =
			  TransSync{names=names,
				    body=replaceTrans t,
				    parent=p}
		      | replaceTrans (TransAsync{names=names,
						 body=t,
						 parent=p}) =
			  TransAsync{names=names,
				     body=replaceTrans t,
				     parent=p}
		      | replaceTrans (TransAtomic m) =
			  let val newM as AtomicModel{pvars=newPvars,
						      assts=assts, ...} = replaceAtomicModel m
			      val {init=initvars, norm=normvars, next=nextvars} = getAsstVars assts
			      val vars = List.map vtName (normvars@initvars)
			  in (globalPvars := mergePrimitiveVars(newPvars, !globalPvars);
			      stateVars := vars@(!stateVars);
			      TransAtomic newM)
			  end
		in
		    {trans=replaceTrans trans,
		     findObject=findObject,
		     stateVars = !stateVars,
		     pvars = (!globalPvars),
		     abs=abs}: Model
		end

	    val b0tp = BoolType dp
	    val truth = Builtin{name=True dp, Type=b0tp}
	    val falsity = Builtin{name=False dp, Type=b0tp}
	    fun addToSeq((f, part),(invar, assump, conc)) =
		(case part of
		     ConclusionPart => (invar, assump, f::conc)
		   | AssumptionPart => (invar, f::assump, conc)
		   | InvarPart => (f::invar, assump, conc)
		   | _ => raise SympBug
		      ("ProofSystem/replaceApply/addToSeq: bad sequent part: "
		       ^(formulaInd2str(part, NONE))))
	    fun addPairsToSeq parts pairs = List.foldr addToSeq parts pairs
	    (* Take a formula and the rest of the sequent (by parts),
	       and rewrite relative to the formula, taking into
	       account the sequent part the formula is from and the
	       direction of rewriting. *)
	    fun replaceOne(model, invar, assump, conc)(f, part, reverse) =
		let fun splitF(Appl(_, Builtin{name=Eq _, ...}, TupleExpr(_, [a, b]))) =
		          if reverse then (b, a) else (a, b)
		      | splitF f = (f, truth)
		    (* Return (A, B, initOnlyFlag), where A is to be replaced by B *)
		    fun getRewriteParts(f, ConclusionPart) = (f, falsity, true)
		      | getRewriteParts(f, InvarPart) =
			let val (a, b) = splitF(f)
			in (a, b, false)
			end
		      | getRewriteParts(f, AssumptionPart) =
			let val (a, b) = splitF(f)
			in (a, b, true)
			end
		      | getRewriteParts(_, part) = raise SympBug
			  ("ProofSystem/replaceApply/replaceOne: bad sequent part: "
		       ^(formulaInd2str(part, NONE)))
		    val (a, b, initOnly) = getRewriteParts(f, part)
		    fun rewriteList lst = List.map(fn x=>replaceFormula initOnly (x, a, b)) lst
		    val (newInvar, newAssump, newConc) =
			 (if initOnly then invar else rewriteList invar,
			  rewriteList assump,
			  rewriteList conc)
		    val newModel = replaceModel initOnly (model, a, b)
		in
		    (newModel, newInvar, newAssump, newConc)
		end

	    (* Strip all the constraints from ObjectValue and return the corresponding ParseTree *)
	    fun getFormula termHash v =
		let fun loop(ConstrainedValue(v, _)) = loop v
		      | loop(ObjectValue id) = termID2pt termHash id
		      | loop x = raise SympBug
			  ("ProofRules.replaceApply/getFormula: not a formula:\n  "
			   ^(value2str x)
			   ^"\n  Term hash = "^(termHash2str termHash))
		in loop v
		end
	    fun ptlist2value termHash lst =
		let fun doOne pt = ObjectValue(pt2termID termHash pt)
		in ListValue(List.map doOne lst)
		end
	    (* Direction of rewrite *)
	    val reverse = (case findArgStringValue args "reverse" of
			       SOME "true" => true
			     | SOME "yes" => true
			     | SOME "t" => true
			     | _ => false)
	    val target = (case  findArgListValue args "target" of
			       SOME lst => lst
			     | NONE => raise SympBug
				 ("ProofRules.replaceApply: no \"target\" parameter"))
	    (* Orchestrate all the work to be done for the rule *)
	    fun doFormulas(sub, SOME(formulas, model, invar, assump, conc)) =
		let val (_, termHash) = sub
		    fun lv2list(ListValue lst) = List.map (getFormula termHash) lst
		      | lv2list x = raise SympBug
			 ("ProofRules.replaceApply: list of formulas is not a ListValue: "
			  ^(value2str x)
			  ^"\n  Term hash = "^(termHash2str termHash))
		    fun mv2model(ModelValue m) = m
		      | mv2model x = raise SympBug
			 ("ProofRules.replaceApply: model is not a ModelValue: "
			  ^(value2str x)
			  ^"\n  Term hash = "^(termHash2str termHash))
		    fun ptl2val lst = ptlist2value termHash lst
		    val (model, invar, assump, conc) =
			(mv2model model, lv2list invar, lv2list assump, lv2list conc)
			 
		    val pairs = getPairs termHash formulas
		    fun loop((f, part)::pairs, model, invar, assump, conc) =
			let val (model, invar, assump, conc) =
			          replaceOne(model, invar, assump, conc)(f, part, reverse)
			in
			    loop(pairs, model, invar, assump, conc)
			end
		      | loop([], model, invar, assump, conc) = (model, invar, assump, conc)
		    val (model, invar, assump, conc) = loop(pairs, model, invar, assump, conc)
		    (* val (invar, assump, conc) = addPairsToSeq(invar, assump, conc) pairs *)
		    val sub = updateSubstitutionRaw("model", ModelValue model) sub
		    val sub = updateSubstitutionRaw("invar", ptl2val invar) sub
		    val sub = updateSubstitutionRaw("assump", ptl2val assump) sub
		    val sub = updateSubstitutionRaw("conc", ptl2val conc) sub
		    val seqTemp = { model = ValueTemp "model",
				   init = ValueTemp "init",
				   context = ProgramContextTemp "context",
				   invar = [ListFormulaTemp("formulas", NONE),
					    ListFormulaTemp("invar", NONE)],
				   assumptions = [ListFormulaTemp("formulas", NONE),
						  ListFormulaTemp("assump", NONE)],
				   conclusions = [ListFormulaTemp("formulas", NONE),
						  ListFormulaTemp("conc", NONE)] }
		    val newSeqOpt = substSequent sub seqTemp
		in 
		    case newSeqOpt of
			SOME seq => InferenceRuleSuccess[(seq, NONE)]
		      | NONE => InferenceRuleFailure
		end
	      | doFormulas(_, NONE) = InferenceRuleFailure
	    fun doSub(SOME sub) =
		    (* Formulas to consider as rewrite rules *)
		let val formulasOpt = subst sub AnyConstr (ListFormulaTemp("formulas", NONE))
		    fun getModel f = Option.map(fn m=>(f,m))
			  (subst sub AnyConstr (ValueTemp "model"))
		    fun getInvar (f,m) = Option.map(fn i=>(f,m,i))
			  (subst sub AnyConstr (ListFormulaTemp("invar", NONE)))
		    fun getAssump (f,m,i) = Option.map(fn a=>(f,m,i,a))
			  (subst sub AnyConstr (ListFormulaTemp("assump", NONE)))
		    fun getConc(f,m,i,a) =  Option.map(fn c=>(f,m,i,a,c))
			  (subst sub AnyConstr (ListFormulaTemp("conc", NONE)))
		    val modelOpt = Option.mapPartial getModel formulasOpt
		    val invarOpt = Option.mapPartial getInvar modelOpt
		    val assumpOpt = Option.mapPartial getAssump invarOpt
		    val partsOpt = Option.mapPartial getConc assumpOpt
		in
		    doFormulas(sub, partsOpt)
		end
	      | doSub NONE = InferenceRuleFailure
	    val res = doSub subOpt
	    val _ = popFunStackLazy(funName, fn()=>InfRuleResult2string res)
	in
	    res
	end

    val replaceRule = { name="replace",
		        help=replaceHelp,
			params = replaceSpecs,
			conclusion = NONE,
			premisses=NONE,
			match=SOME replaceMatch,
			apply=SOME replaceApply }: InferenceRule

    (* Match only AG formulas in the conclusion.  If the user specifies
       anything else, it's an error. *)
    (* fun inductAGmatch(seq: Sequent, context, args RuleArg list, subOpt)a = NONE *)

    local
	val b1tp = FunType(dp,BoolType dp,BoolType dp)
	val ag = Builtin{name=Ag dp, Type=b1tp}
	val ax = Builtin{name=Ax dp, Type=b1tp}
	val AGtemp = ApplTemp(BuiltinTemp ag, FormulaTemp "formula")
	val AXtemp = ApplTemp(BuiltinTemp ax, FormulaTemp "invariant")
	val inductAGspecs =
	    [{name = "invariant",
	      Type=ProverArgObjectType,
	      default=DefaultRequired},
	     {name = "formula",
	      Type=ProverArgFormulaNumType,
	      default=DefaultValue(ProverArgFormulaNum(AnyPart, NONE))}]

    in
	val inductAGseqTemp =
	    { model = ValueTemp "model",
	      init = ValueTemp "init",
	      context=ProgramContextTemp "context",
	      invar = [ListFormulaTemp("invar",NONE)],
	      assumptions = [ListFormulaTemp("assump",NONE)],
	      conclusions = [AGtemp, ListFormulaTemp("conc",NONE)] }: SequentTemplate

	val inductAGrule =
	    { name = "induct_ag",
	     help = ("Prove an AG(f) formula by induction on time\n"
		     ^"using a given inductive INVARIANT."),
	     params = inductAGspecs,
	     conclusion = SOME inductAGseqTemp,
	     premisses = SOME[(* Invariant implies the property *)
			      { model = ValueTemp "model",
			        init = ConstantTemp(InitValue NONE)(* ValueTemp "init"*),
				context = ProgramContextTemp "context",
				invar = [FormulaTemp "invariant",
					 ListFormulaTemp("invar", NONE)],
				assumptions = [ListFormulaTemp("assump", NONE)],
				conclusions = [FormulaTemp "formula",
					       ListFormulaTemp("conc", NONE)] },
			      (* Invariant holds at the initial state *)
			      { model = ValueTemp "model",
			        init = ValueTemp "init",
				context = ProgramContextTemp "context",
				invar = [ListFormulaTemp("invar", NONE)],
				assumptions = [ListFormulaTemp("assump", NONE)],
				conclusions = [FormulaTemp "invariant",
					       ListFormulaTemp("conc", NONE)] },
			      (* Invariant imlies itself at the next step *)
			      { model = ValueTemp "model",
			        init = ConstantTemp(InitValue NONE),
				context = ProgramContextTemp "context",
				invar = [ListFormulaTemp("invar", NONE)],
				assumptions = [FormulaTemp "invariant",
					       ListFormulaTemp("assump", NONE)],
				conclusions = [AXtemp, ListFormulaTemp("conc", NONE)] }],
	  match = NONE,
	  apply = NONE }: InferenceRule
    end

    (* Same as above, only returs two lists: (matchedAndReplaced, unmatched) *)
    fun matchSubstValues sub (matchTemp, substTemp) vlist =
	let val match = matchSubstValue sub (matchTemp, substTemp)
	    fun loop([], matched, unmatched) = (List.rev matched, List.rev unmatched)
	      | loop(v::lst, matched, unmatched) =
	          (case match v of
		       NONE => loop(lst, matched, v::unmatched)
		     | SOME v' => loop(lst, v'::matched, v::unmatched))
	in loop(vlist, [], [])
	end

    (* Lifting forall from under the AX:
       AX(forall ...) -> forall ...: AX ... *)

    local 
	val b1tp = FunType(dp,BoolType dp,BoolType dp)
	val ax = Builtin{name=Ax dp, Type=b1tp}
	val AXtemp = ApplTemp(BuiltinTemp ax, 
			      ForallTemp([ListValueTemp("vars", NONE)],
					 FormulaTemp "body", ValueTemp "cxt"))
	val resTemp = ForallTemp([ListValueTemp("vars", NONE)],
				 ApplTemp(BuiltinTemp ax, FormulaTemp "body"),
				 ValueTemp "cxt")
	val AXtempWildcard =
	       ApplTemp(BuiltinTemp ax, 
			ForallTemp([ListWildCardTemp], WildCardTemp, WildCardTemp))
	val AXtempList = ListFormulaTemp("formulas", SOME AXtempWildcard)
	val specs =
	    [{name = "formulas",
	      Type=ProverArgListType ProverArgFormulaNumType,
	      default=DefaultValue(ProverArgList[ProverArgFormulaNum(AnyPart, NONE)])}]
	val concTemp = { model = ValueTemp "model",
			 init = ValueTemp "init",
			 context = ProgramContextTemp "context",
			 invar = [AXtempList, ListFormulaTemp("invar",NONE)],
			 assumptions = [AXtempList, ListFormulaTemp("assump",NONE)],
			 conclusions = [AXtempList, ListFormulaTemp("conc",NONE)] }
	val resSeqTemp =  { model = ValueTemp "model",
			    init = ValueTemp "init",
			    context = ProgramContextTemp "context",
			    invar = [ListFormulaTemp("formulas", NONE),
				     ListFormulaTemp("invar",NONE)],
			    assumptions = [ListFormulaTemp("formulas", NONE),
					   ListFormulaTemp("assump",NONE)],
			    conclusions = [ListFormulaTemp("formulas", NONE),
					   ListFormulaTemp("conc",NONE)] }

	fun apply printFun (seq: Sequent, context, args: RuleArg list, sub) =
	    let val (_, hash) = sub
		val seqTerm = sequent2term(seq, SOME hash)
		val subOpt = matchSeq sub (applySeqArgs(concTemp, args), NONE, seqTerm)
		fun newSeq sub formulas =
		    let val (matched, unmatched) = matchSubstValues sub (AXtemp, resTemp) formulas
			val newSub = updateSubstitutionRaw("formulas",
							   ListValue(matched@unmatched)) sub
		    in substSequent newSub resSeqTemp
		    end
		fun getFormulas sub =
		    let val ListValueOpt = 
			  Option.mapPartial(fn s=>subst s AnyConstr
					    (ListFormulaTemp("formulas", NONE))) subOpt
			fun getList(ListValue lst) = SOME lst
			  | getList(ConstrainedValue(v, _)) = getList v
			  | getList _ = NONE
			val formulasOpt = Option.mapPartial getList ListValueOpt
		    in
			Option.mapPartial(newSeq sub) formulasOpt
		    end
		val newSeqOpt = Option.mapPartial getFormulas subOpt
	    in
		case newSeqOpt of
		    NONE => InferenceRuleFailure
		  | SOME seq => InferenceRuleSuccess[(seq, NONE)]
	    end
    in
	val liftForallAXrule =
	    { name = "lift_forall_ax",
	      help = ("Lift the universal quantifier from inside AX to the top level."),
	      params = specs,
	      conclusion = SOME concTemp,
	      premisses = NONE,
	      match = NONE,
	      apply = SOME apply }: InferenceRule
	val concTemp = { model = ValueTemp "model",
			 init = ValueTemp "init",
			 context = ProgramContextTemp "context",
			 invar = [ListFormulaTemp("invar",NONE)],
			 assumptions = [ListFormulaTemp("assump",NONE)],
			 conclusions = [ListFormulaTemp("conc",NONE)] }
    end

    local
	val splitLabelSpecs =
	    [{name="label",
	      Type=ProverArgStringType,
	      default=DefaultRequired}]

	exception Quit of string

	fun apply printFun (seq: Sequent, context, args: RuleArg list, sub) =
	    let val {model=model,
		     init=init,
		     context=context,
		     invar=invar,
		     assumptions=assump,
		     conclusions=conc, ...} = seq
		val {trans=trans,
		     findObject=findObject,
		     pvars=pvars,
		     stateVars=stateVars,
                     abs=abs,...} = model
		val (_, termHash) = sub
		val options = getOptions()
		val lim = get_limit options
		val (cxt, findObject) = context
		fun extractAtomic(TransAtomic am) = am
		  | extractAtomic _ = raise Quit 
		      "Sorry, this rule does not support parallel composition yet."
		val AtomicModel{assts=assts,
				name=name,
				uname=uname,
				absModules=absModules, ...} = extractAtomic trans
		val labelStr = (case findArgStringValue args "label" of
				 NONE => raise SympBug
				     ("split_label rule: argument LABEL is not found")
			       | SOME str => str)
		val label = Id(dp, labelStr)

		(* Search for the labeled assignment on the top level
		   and return a pair (asstOpt, ff), where `ff' takes a
		   new assingment subtree and returns a new asst. tree
		   with the labeled assignment replaced by the given
		   one. *)

		fun findAsst(asst as LabeledAsstTree(vars, l, a)) =
		    if ptEq(label, l) then SOME(a, fn a'=> LabeledAsstTree(vars, l, a'))
		    else NONE
		  | findAsst(asst as ListAsstTree(vars, assts)) =
		    let fun loop [] acc = NONE
			  | loop (a::lst) acc = 
			      (case findAsst a of
				   NONE => loop lst (a::acc)
				 | SOME(a', ff) =>
				     SOME(a', fn a => ListAsstTree
					       (vars, (List.rev acc)@[ff a]@lst)))
		    in loop assts []
		    end
		  | findAsst _ = NONE
		(* Return the list of pairs: (condition, new_asst_subtree) *)
		fun splitAsst(ChooseAsstTree(vars, paramsOpt, pairs), ff) =
		    let val (varPairs, newCxt) =
			    (case paramsOpt of
				 SOME params => newSkolemConsts cxt params
			       | NONE => ([], cxt))
			fun subst(old, new) expr =
			    let fun loop e = if ptEq(e, old) then new
					     else ptTransform loop e
			    in loop expr
			    end
			fun substParams c =
			    let fun fold(pair, expr) = subst pair expr
			    in List.foldr fold c varPairs
			    end
			fun substParamsAsst a =
			    let fun fold((old, new), asst) =
				    substAVT options findObject pvars 
				      lim (asst, old, new)
			    in List.foldr fold a varPairs
			    end
			fun doPair(c, a) = (substParams c, ff(substParamsAsst a))
		    in (newCxt, List.map doPair pairs)
		    end
		  | splitAsst _ = raise Quit ("Cannot split the assignment labeled "^labelStr)

		(* Take the list of pairs (condition, asst) and compute the list of
		   sequents with the condition in the assumptions and the model built out
		   of `asst' *)
		fun pairs2seqs(cxt, pairs) =
		    let fun pair2seq(c, asst) =
			    let val opt = rebuildAsstOptionsBalance rebuildAsstOptionsDefault
				val (asst, cone, pvars) = 
				  rebuildAsstVars options findObject pvars lim (NONE, asst, opt)
				val trans = TransAtomic(AtomicModel{name=NONE,
								    uname=uname,
								    absModules=absModules,
								    cone=cone,
								    pvars=pvars,
								    assts=asst})
				val model = {trans=trans,
					     findObject=findObject,
					     stateVars=stateVars,
					     pvars=pvars,
					     abs=abs}: Model
				val seq = {model=model,
					   init=init,
					   context=(cxt, findObject),
					   invar=invar,
					   assumptions=c::assump,
					   conclusions=conc}: Sequent
			    in (seq, NONE: SequentHints option)
			    end
		    in List.map pair2seq pairs
		    end
		(* Check that the rule is sound to apply to the formulas *)
		fun checkSeq(invar, assump, conc) =
		    (printFun("\nATTENTION: make sure it is sound to apply this rule;\n"
			      ^"the automatic check is not implemented yet.\n");
		     true)
		    
		(* Now execute all the stuff *)
		val _ = if checkSeq(invar, assump, conc) then ()
			else raise Quit("Sequent has temporal operators other than AX, "
				     ^"or has nested AX.")
		val pairOpt = findAsst assts
		val newSeqsOpt = Option.map(pairs2seqs o splitAsst) pairOpt
	    in
		case newSeqsOpt of
		    SOME lst => InferenceRuleSuccess lst
		  | NONE => InferenceRuleFailure
	    end handle Quit str => (printFun(str^"\n"); InferenceRuleFailure)

    in
	val splitLabelRule =
	    { name = "split_label",
	      help = ("Split cases on a nondeterministic choice in the assignment\n"
		      ^"labeled with the given LABEL."),
	      params = splitLabelSpecs,
	      conclusion = SOME concTemp,
	      premisses = NONE,
	      match = NONE,
	      apply = SOME apply }: InferenceRule
    end

    (* Split rule: combination of and_R, or_L, etc. that produce
       several new subgoals to proof. *)

    local
	val specs = [{name="formulas",
		      Type=ProverArgListType ProverArgFormulaNumType,
		      default=DefaultValue(ProverArgList[ProverArgFormulaNum(AnyPart, NONE)])}]
	val concTemp = { model = ValueTemp "model",
			 init = ValueTemp "init",
			 context = ProgramContextTemp "context",
			 invar = [ListFormulaTemp("formulas",NONE),
				  ListFormulaTemp("invar",NONE)],
			 assumptions = [ListFormulaTemp("formulas",NONE),
					ListFormulaTemp("assump",NONE)],
			 conclusions = [ListFormulaTemp("formulas",NONE),
					ListFormulaTemp("conc",NONE)] }
	fun apply printFun (seq: Sequent, context, args: RuleArg list, sub) =
	    let val funName = "ProofRules.split.apply"
		val debug = lazyVerbDebug(getOptions()) funName
		val (_, termHash) = sub
		val seqTerm = sequent2term(seq, SOME termHash)
		val subOpt = matchSeq sub (applySeqArgs(concTemp, args), NONE, seqTerm)
		fun getFormulas sub =
		    (case subst sub AnyConstr (ListFormulaTemp("formulas",NONE)) of
			 SOME v => v
		       | NONE => raise SympBug
			     ("ProofRules/split rule/apply: "
			      ^"`formulas' is not in the substitution:\n"
			      ^(subst2str sub)))
		val (_, findObject) = context 
		val evalExpr = evaluateExpr(getOptions()) findObject

		(* Take a pair (formula, ind) and split it into a list of lists of
		   similar pairs, each list representing the new set of formulas
		   replacing the original one in the sequent for one of the cases.
		   The top level operator must be conjunction or IFF for `ConcPart',
                   and disjunction/implication for the other parts.  In any other case
		   return NONE. *)

		fun splitPair(f, part) =
		    let val funName = funName^"/splitPair"
			val _ = pushFunStackLazy(funName, fn()=>pair2str (f, part))
			fun loop(f as Appl(_, Op, args), part as ConclusionPart) =
			      (case GetObjectName Op of
				   SOME(And _) => SOME[[(evalExpr(ExtractTuple(0, args)), part)],
						       [(evalExpr(ExtractTuple(1, args)), part)]]
				 | SOME(Iff _) =>
				       let val arg1 = evalExpr(ExtractTuple(0, args))
					   val arg2 = evalExpr(ExtractTuple(1, args))
				       in SOME[[(arg1, AssumptionPart), (arg2, part)],
					       [(arg2, AssumptionPart), (arg1, part)]]
				       end
				 | _ => NONE)
			  | loop (f as Appl(_, Op, args), part as AssumptionPart) =
			      (case GetObjectName Op of
				   SOME(Or _) => SOME[[(evalExpr(ExtractTuple(0, args)), part)],
						      [(evalExpr(ExtractTuple(1, args)), part)]]
				 | SOME(Implies _) => 
				       SOME[[(evalExpr(ExtractTuple(0, args)), ConclusionPart)],
					    [(evalExpr(ExtractTuple(1, args)), part)]]
				 | _ => NONE)
			  | loop (f as Appl(_, Op, args), part as InvarPart) =
			      (case GetObjectName Op of
				   SOME(Or _) => SOME[[(evalExpr(ExtractTuple(0, args)), part)],
						      [(evalExpr(ExtractTuple(1, args)), part)]]
				 (* Can't move negation from invariant to conclusion that easily,
				    may lose too much information *)
				 | SOME(Implies _) => 
				       SOME[[(evalExpr(negFormula(ExtractTuple(0, args))), part)],
					    [(evalExpr(ExtractTuple(1, args)), part)]]
				 | _ => NONE)
			  | loop _ = NONE
			fun res2str lst = "["^(strlist2str ", " (List.map pairs2str lst))^"]"
			val res = loop(f, part) handle EvalError str => raise SympBug
			    ("ProofRules/split rule/apply/splitPair: "^str)
			val _ = popFunStackLazy(funName, fn()=>optionPrint res2str "NONE" res)
		    in
			res
		    end

		(* Return the list of lists of cases for all formula pairs, or NONE if
		   none of the pairs can be split. *)

		fun splitPairs(pairs, sub) =
		    let val splits = List.map splitPair pairs
			fun loop([], acc, true) = SOME(List.rev acc)
			  | loop([], _, false) = NONE
			  | loop (pair::lst, acc, hadSplits) =
			     (case splitPair pair of
				  SOME res => loop(lst, res::acc, true)
				| NONE => loop(lst, [[pair]]::acc, hadSplits))
			fun splits2cases lst =
			    let val combinations = comblist lst
				val foldAppend = List.foldr(op @) []
				val caseLists = List.map foldAppend combinations
			    in List.map(pairs2value termHash) caseLists
			    end
			val splitsOpt = loop(pairs, [], false)
		    in Option.map (fn lst => (splits2cases lst, sub)) splitsOpt
		    end

		(* Given the list of formula pairs (as one case in the split),
		   generate a new sequent for this case *)

		fun genSequent(formulas, sub) =
		    let val seqTemp =
			  { model = ValueTemp "model",
			    init = ValueTemp "init",
			    context = ProgramContextTemp "context",
			    invar = [ListFormulaTemp("formulas",NONE),
				     ListFormulaTemp("invar",NONE)],
			    assumptions = [ListFormulaTemp("formulas",NONE),
					   ListFormulaTemp("assump",NONE)],
			    conclusions = [ListFormulaTemp("formulas",NONE),
					   ListFormulaTemp("conc",NONE)] }
			val sub = copySubstitution sub
			val sub = updateSubstitutionRaw("formulas", formulas) sub
		    in (case substSequent sub seqTemp of 
			    SOME seq => seq
			  | NONE => raise SympBug
			 ("ProofRules/split rule/apply/genSequent:\n"
			  ^" cannot instantiate new sequent for the case:\n  "
			  ^(value2str formulas)))
		    end

		fun genSequents(cases, sub) =
		      List.map(fn c=> (genSequent(c, sub), NONE: SequentHints option)) cases

		fun cases2str cases = "["^(strlist2str ",\n " (List.map value2str cases))^"]"

		(* Now execute the stuff *)
		(* ParseTree values of the formulas paired with their Formula Index *)
		val formulaPairsOpt = 
		      Option.map (fn sub => (getPairs termHash(getFormulas sub), sub)) subOpt
		val _ = debug(fn()=>"\n"^funName^": formulaPairs = "
			      ^(optionPrint (pairs2str o #1) "NONE" formulaPairsOpt)^"\n")
		val casesOpt = Option.mapPartial splitPairs formulaPairsOpt
		val _ = debug(fn()=>"\n"^funName^": cases = "
			      ^(optionPrint (cases2str o #1) "NONE" casesOpt)^"\n"
			      ^(termHash2str termHash)^"\n")
		val newSeqsOpt = Option.map genSequents casesOpt
	    in
		case newSeqsOpt of
		    NONE => InferenceRuleFailure
		  | SOME seqs => InferenceRuleSuccess seqs
	    end
    in
	val splitRule =
	    { name = "split",
	      help = ("Split cases on conjunction in conclusions, disjuntion and implication\n"
		      ^"in assumptions, etc."),
	      params = specs,
	      conclusion = SOME concTemp,
	      premisses = NONE,
	      match = NONE,
	      apply = SOME apply }: InferenceRule
    end

    local
	val specs = []
	val concTemp = { model = ValueTemp "model",
			 init = ValueTemp "init",
			 context = ProgramContextTemp "context",
			 invar = [ListFormulaTemp("invar",NONE)],
			 assumptions = [ListFormulaTemp("assump",NONE)],
			 conclusions = [ListFormulaTemp("conc",NONE)] }
	fun apply printFun (seq: Sequent, context, args: RuleArg list, sub) =
	    let val funName = "ProofRules.abstractSplit.apply"
		val debug = lazyVerbDebug(getOptions()) funName
		val (_, termHash) = sub
		(* val seqTerm = sequent2term(seq, SOME termHash)
		val subOpt = matchSeq sub (applySeqArgs(concTemp, args), NONE, seqTerm) *)
	    in
		InferenceRuleFailure
	    end
    in
	val abstractSplitRule =
	    { name = "abstract_split",
	      help = ("Split cases on equalities between Skolem constants present in the\n"
		      ^"current sequent and convert these Skolem constants into abstract\n"
		      ^"constants.\n\n"
		      ^"The rule is invertible when Skolem constants are only compared for\n"
		      ^"equality or used as indices in array state variables.  Otherwise,\n"
		      ^"invertibility is not guaranteed, meaning that you may get a spurious\n"
		      ^"counterexample by model checking the resulting sequents."),
	      params = specs,
	      conclusion = SOME concTemp,
	      premisses = NONE,
	      match = NONE,
	      apply = SOME apply }: InferenceRule
    end

    fun proofRules() = [modelCheckRule,
			COIRule,
			deleteRule,
			initRule,
			copyRule,
			useInvarRule,
			caseRule,
			forallCaseRule,
			flattenRule,
			skolemRule,
			instRule,
			replaceRule,
			inductAGrule,
			liftForallAXrule,
			splitRule,
			splitLabelRule,
			abstractSplitRule]: InferenceRule list

    (*********************************************************************)
    (*****************   Proof System Commands   *************************)
    (*********************************************************************)

    fun applyStatModel printFun (seq: Sequent, args) =
	let val {model=model,...} = seq
	in
	    (printFun("Model in this sequent:\n\n"
		     ^(model2str model)^"\n");
	     NONE: ProofSystemCommandResult option)
	end

    val statModelCommand =
	{ name="statmodel",
	  help=("Give some statistics and state information about the model in the"
		^" current sequent."),
	  params = [],
	  apply = applyStatModel }: ProofSystemCommand

    fun proverCommands() =
	[statModelCommand]: ProofSystemCommand list

  end
