functor ParserDefaultFun(structure Interface: INTERFACE) =
  struct
    open SympBug
    open Str

    structure Interface = Interface
    structure ParseTreeStruct = ParseTreeFun(structure Pos = Pos)

    open Interface
    open Options
    open ParseTreeStruct;

    structure ParseError: PARSE_ERROR =
      struct
	exception ParseError of string 
	val wasError = ref false
      end
    open ParseError

    structure SYMPLrVals =
	SYMPLrValsFun(structure Token = LrParser.Token
		      structure ParseTreeStruct = ParseTreeStruct
		      structure ParseError = ParseError)

    structure SYMPLex =
	SYMPLexFun(structure Tokens = SYMPLrVals.Tokens
		   structure ParseError = ParseError)

    structure SYMPParser =
	SYMPJoin(structure LrParser = LrParser
		 structure ParseTreeStruct = ParseTreeStruct
		 structure ParserData = SYMPLrVals.ParserData
		 structure Lex = SYMPLex)

    (* Similar parser for user input *)

    structure SYMPInputLrVals =
	SYMPInputLrValsFun(structure Token = LrParser.Token
			   structure ParseTreeStruct = ParseTreeStruct
			   structure ParseError = ParseError)

    structure SYMPInputLex =
	SYMPLexFun(structure Tokens = SYMPInputLrVals.Tokens
		   structure ParseError = ParseError)

    structure SYMPInputParser =
	SYMPJoin(structure LrParser = LrParser
		 structure ParseTreeStruct = ParseTreeStruct
		 structure ParserData = SYMPInputLrVals.ParserData
		 structure Lex = SYMPInputLex)

    type result = SYMPLrVals.ParserData.result
    open SYMPLrVals.ParserData

    fun Parse options ins = 
	let val print_error = fn (s,l,c) =>
	    (wasError := true;
	     printOutCommand options
	     ("error",[UIstring "parse",
		       UIstring ((Int.toString l)^"."^(Int.toString c)^", "^s^"\n")]))
	(* Do not use error correction *)
	    val invoke = fn lexstream =>
		(SYMPParser.parse(15,lexstream,print_error,())
		 handle SYMPParser.ParseError => raise Interface.ParseError("Fatal parse error")
		      | ParseError.ParseError str => raise Interface.ParseError str)
	    val lexer = SYMPParser.makeLexer 
		           (fn (i: int) => TextIO.inputLine ins) print_error
	    val (result,_) = (SYMPLex.UserDeclarations.pos := 1;
			      SYMPLex.UserDeclarations.yypos_bol := 0;
			      wasError := false;
			      invoke lexer)
			      
	in result
	end

    (* Parse user input *)
    fun ParseInput options ins =
	let val print_error = fn (s,l,c) =>
	    (wasError := true;
	     printOutCommand options
	     ("error",[UIstring "parse",
		       UIstring ((Int.toString l)^"."^(Int.toString c)^", "^s^"\n")]))
	(* Do not use error correction *)
	    val invoke = fn lexstream => 
		(SYMPInputParser.parse(0,lexstream,print_error,())
		 handle SYMPParser.ParseError => raise Interface.ParseError("Fatal parse error")
		      | ParseError.ParseError str => raise Interface.ParseError str)
	    val lexer = SYMPInputParser.makeLexer 
		           (fn (i: int) => TextIO.inputLine ins) print_error
	    val (result,_) = (SYMPInputLex.UserDeclarations.pos := 1;
			      SYMPInputLex.UserDeclarations.yypos_bol := 0;
			      wasError := false;
			      invoke lexer)
			      
	in result
	end

    (* Extracts the directory name from the file name.  Basically,
     * it's all the string upto and including the last `/' or `\'
     * symbol if there is one, and empty otherwise *)

    fun dirname file =
	let fun loop n str = (case String.sub(str,n) of
				  #"/" => String.substring(str,0,n+1)
				| #"\\" =>String.substring(str,0,n+1)
				| _ => if n = 0 then "" else loop (n-1) str)
	in loop ((String.size file)-1) file
	end

    (* Parses and opens includes recursively *)
    fun ParseFile (options: options) file =
	let val _ = verbDebug options "ParseFile" ("\nSyMP Default: Parsing file "^file^"...\n")
	    val ins = (if file = "-" then TextIO.stdIn
		       else TextIO.openIn file)
	             handle _ => raise SympError("Cannot open input file: "^file)
	    val {verbose = verbose, ... } = options
	    val verb = verb options
	    val program = 
		(verb("Parsing "^file^"...");
		 let val program = Parse options (ins) 
		 in program before
		  (if file = "-" then () else TextIO.closeIn ins;
		   if !wasError then
		       raise SympError("\nThere were parse errors in "
				       ^file^". Exiting.")
		   else (verb("Parsing "^file^"...done")))
		 end)
	    fun expand (Program(pos,lst)) = Program(pos,map expand lst)
	      | expand (Include(pos,file1)) =
		 Included(pos,file1,
			  ParseFile options ((dirname file)^file1))
	      | expand x = x
	in expand program
	end

  end
