functor EvaluateFun(structure ParserDefault: PARSER_DEFAULT): EVALUATE =

  struct

    structure ParserDefault = ParserDefault
    structure ParseTreeStruct = ParserDefault.ParseTreeStruct
    structure PatternMatch = PatternMatchFun (structure ParserDefault = ParserDefault)

    open Pos
    open ParserDefault
    open ParseTreeStruct
    open Interface
    open Options
    open PatternMatch

    exception EvalError of string

    (* Create an `undefined' value of type `tp' *)
    fun bot tp = Builtin{name=Undefined dp, Type=tp}

    fun isVal options x =
      let val _ = lazyVerbDebug options "isValue" (fn()=>("\nisValue("^(pt2stringDebug x)))
	  val res = isValue x
	  val _ = lazyVerbDebug options "isValue" 
	            (fn()=>(") = "^(if res then "true\n" else "false\n")))
      in res 
      end

    (* Checks whether the expression might have types in it.  If it
       does, we have to keep Object and ObjectInst wrappers. *)
    fun hasTypes e =
	(case e of
	     FunClosure _ => true
	   | RecurFun _ => true
(* We cannot wrap these back into Object/ObjectInst, since we lose the
   ability to evaluate them uniformly.  But if we don't, we lose the
   type information.  Maybe it's a tolerable loss, we'll see. *)
(*  	   | Builtin(Undefined _) => true *)
(*  	   | Builtin(Eq _) => true *)
	   | Object _ => true
	   | ObjectInst _ => true
	   | RecordExpr(_,rlist) => List.exists hasTypes rlist
	   | RecordAsst(_,_,x) => hasTypes x
	   | TupleExpr(_,lst) => List.exists hasTypes lst
	   | x => not(isValue x))

    (* Check if the expressions are always the same or not.  If they
       are always the same, return true, if always different - false,
       if can't tell - NONE. *)
    fun theSame(e1, e2) =
	let datatype Comparable = No | OneWay | BothWays
	    fun comparable (StateVar _, StateVar _) = OneWay
	      | comparable (SkolemConst _, SkolemConst _) = OneWay
	      | comparable (AbstractConst _, AbstractConst _) = BothWays
	      | comparable (PatternFormal _, PatternFormal _) = OneWay
	      | comparable (e1, e2) = 
		  if (isValue e1) andalso (isValue e2) then BothWays
		  else No
	in
	    (case comparable(e1, e2) of
		 No => NONE
	       | OneWay => if (eqUNames (e1, e2) orelse ptEq (e1, e2)) then SOME true else NONE
	       | BothWays => SOME(eqUNames (e1, e2) orelse ptEq (e1, e2)))
	end

    local fun subs options (expr, arg) body =
      let 
	val result = 
	  (lazyVerbDebug options "Evaluate.subs" (fn () => 
					 ("\nSubs called on:[\n" ^ 
					  (pt2stringDebug body)));
	   (case body of
	      FunClosure{formals=f,...} => 
		if eqUNames(arg,f) then body
		else ptTransform ((subs options) (expr, arg)) body
	    | ChoiceClosure{uname=u,...} => 
		if eqUNames(arg,u) then body
		else ptTransform ((subs options) (expr, arg)) body
	    | _ => 
		if eqUNames (body, arg) then
		  expr
		else
		  ptTransform ((subs options) (expr, arg)) body))
      in
	(lazyVerbDebug options "Evaluate.subs" (fn () =>
				       ("\nSubs returning:\n" ^ 
					  (pt2stringDebug result)^"]"));
	 result)
      end
    in
      fun substitute options (expr, arg, body) =
	(lazyVerbDebug options "Evaluate.subs" (fn () => 
					 ("\nSubstitute replacing:\n"
					  ^(pt2stringDebug arg)
					  ^"\nwith:\n"
					  ^(pt2stringDebug expr)));
	 subs options (expr, arg) body)
(*  	 ptTransform ((subs options) (expr, arg)) body) *)
    end

    fun handleConds findObject rc ff (pos, [], else_exp) = rc ff else_exp
      | handleConds findObject rc ff (pos, rest as (CondExpr(cpos, cond, e)::t), else_exp) =
         (case rc ff cond of
	      Builtin{name=True dp,...} => rc ff e
	    | Builtin{name=False dp,...} => handleConds findObject rc ff (pos, t, else_exp)
	    | x => if isUndefined x then bot(ff(getExprType findObject e))
		   else IfExpr (pos, rest, else_exp))
      | handleConds _ _ _ (_, c::t, _)  = raise 
	(SympBug
	 ("Badly constructed conditional: " ^ (pt2stringDebug c)))

    (* Assumption: e (the case selector) is fully evaluated *)
    fun handleCases rc (options, pos, e, [], tp) = bot tp
      | handleCases rc (options, pos, e, 
			(c as ChoiceClosure{pattern=pat,
					    uname=selName,
					    body=exp,...})::t,
			tp) =
      (case (matches options pat e) of
	 SOME true => rc(substitute options (e,selName,exp))
       | SOME false => handleCases rc (options, pos, e, t, tp)
       | NONE => CaseExpr (pos, e, (c::t)))
      | handleCases _ (_, _, _, c::t, _) = raise SympBug
	("Badly formed choice in case" ^ pt2stringDebug c)

    fun handleWith rc (pos, rpos, rlist, wlist) =
      let 
	val erlist = List.map rc rlist
	val ewlist = List.map rc wlist
      in
	RecordExpr (rpos, (List.map 
			   (fn WithAsst(p,n1,n2) => RecordAsst (p,n1,n2)
			 | w => raise SympBug
			    ("Badly formed with: " 
			     ^ (pt2stringDebug w)))
			   ewlist) @ (List.filter 
				     (fn RecordAsst(_,name1,_) =>
				      not (List.exists 
					   (fn WithAsst(_,name2,_) =>
					    ptEq (name1, name2)
					 | w => raise SympBug 
					    ("Badly formed with: " 
					     ^ (pt2stringDebug w)))
					   ewlist)
				   | r => raise SympBug 
				      ("Badly formed record: " 
				       ^ (pt2stringDebug r))
				      ) erlist))
      end

    fun evaluateExpr options findObject parseTree1 =
      let val funName = "evaluateExpr"
	val isValue = isVal options
	val _ = pushFunStackLazy(funName, fn () => pt2string parseTree1)
	fun recurCall ff parseTree1 = 
	 let val funName = "evaluateExpr/recurCall"
	     val _ = pushFunStackLazy(funName, fn()=>pt2string parseTree1)
	     fun trueobj BoolType2 = True2 
	       | trueobj _ = Builtin{name=True dp, Type=BoolType dp}
	     fun falseobj BoolType2 = False2
	       | falseobj _ = Builtin{name=False dp, Type=BoolType dp}
	     (* Tries to apply function f to argument e.
	     Invariant: ee1 and ee2 are fully evaluated *)
	     fun applyFun tp (f, e) =
		 let val funName = "Evaluate.applyFun"
		     val _ = pushFunStackLazy(funName,
					      fn()=>((pt2string f)^", "^(pt2string e)
						     ^", tp = "^(pt2string tp)))
		     val res = applyFun' tp (f, e)
		     val _ = popFunStackLazy(funName, fn()=> pt2string res)
		 in res
		 end
	     and applyFun' tp (Builtin{name=Plus _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  Number (dp, n1 + n2)
	       | applyFun' tp (Builtin{name=Times _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  Number (dp, n1 * n2)
	       | applyFun' tp (Builtin{name=Minus _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  Number (dp, n1 - n2)
	       | applyFun' tp (Builtin{name=Div _, ...}, TupleExpr(_, [_, Number(_, 0)])) = bot(IntType dp)
	       | applyFun' tp (Builtin{name=Div _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  Number (dp, n1 div n2)
	       | applyFun' tp (Builtin{name=Mod _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  Number (dp, n1 mod n2)
	       | applyFun' tp (Builtin{name=Gt _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  if n1 > n2 then trueobj tp else falseobj tp
	       | applyFun' tp (Builtin{name=Lt _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  if n1 < n2 then trueobj tp else falseobj tp
	       | applyFun' tp (Builtin{name=Ge _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  if n1 >= n2 then trueobj tp else falseobj tp
	       | applyFun' tp (Builtin{name=Le _, ...}, TupleExpr(_, [Number(_, n1), Number(_, n2)])) =
		  if n1 <= n2 then trueobj tp else falseobj tp
	       | applyFun' tp (Builtin{name=And _, ...}, 
			      TupleExpr (_,[Builtin{name=True _,...}, e])) = e
	       | applyFun' tp (Builtin{name=And _, ...}, 
			      TupleExpr (_,[e, Builtin{name=True _,...}])) = e
	       | applyFun' tp (Builtin{name=And _, ...},
			      TupleExpr (_, [Builtin{name=False(_),...}, _])) = falseobj tp 
	       | applyFun' tp (Builtin{name=And _, ...},
			      TupleExpr (_, [_, Builtin{name=False(_),...}])) = falseobj tp 
	       | applyFun' tp (Builtin{name=Or _, ...},
			      TupleExpr (_, [Builtin{name=False(_),...}, e])) = e
	       | applyFun' tp (Builtin{name=Or _, ...},
			      TupleExpr (_, [e, Builtin{name=False(_),...}])) = e
	       | applyFun' tp (Builtin{name=Or _, ...},
			      TupleExpr (_, [Builtin{name=True(_),...}, _])) = trueobj tp 
	       | applyFun' tp (Builtin{name=Or _, ...},
			      TupleExpr (_, [_, Builtin{name=True(_),...}])) = trueobj tp
	       | applyFun' tp (Builtin{name=Implies _, ...},
			      TupleExpr (_, [Builtin{name=True(_),...},e])) = e
	       | applyFun' tp (Builtin{name=Implies _, ...},
			      TupleExpr (_, [Builtin{name=False(_),...}, _])) = trueobj tp 
	       | applyFun' tp (Builtin{name=Implies _, ...},
			      TupleExpr (_, [_, Builtin{name=True(_),...}])) = trueobj tp
	       | applyFun' tp (Builtin{name=Implies _, ...},
			      TupleExpr (_, [e, Builtin{name=False(_),...}])) =
		      Appl(dp, Builtin{name=Not dp, Type=FunType(dp,tp,tp)}, e)
	       (* And now the same boolean operators for BoolType2 *)
	       | applyFun' tp (Builtin{name=And _, ...}, 
			      TupleExpr (_,[True2, e])) = e
	       | applyFun' tp (Builtin{name=And _, ...}, 
			      TupleExpr (_,[e, True2])) = e
	       | applyFun' tp (Builtin{name=And _, ...},
			      TupleExpr (_, [False2, _])) = falseobj tp 
	       | applyFun' tp (Builtin{name=And _, ...},
			      TupleExpr (_, [_, False2])) = falseobj tp 
	       | applyFun' tp (Builtin{name=Or _, ...},
			      TupleExpr (_, [False2, e])) = e
	       | applyFun' tp (Builtin{name=Or _, ...},
			      TupleExpr (_, [e, False2])) = e
	       | applyFun' tp (Builtin{name=Or _, ...},
			      TupleExpr (_, [True2, _])) = trueobj tp 
	       | applyFun' tp (Builtin{name=Or _, ...},
			      TupleExpr (_, [_, True2])) = trueobj tp
	       | applyFun' tp (Builtin{name=Implies _, ...},
			      TupleExpr (_, [True2,e])) = e
	       | applyFun' tp (Builtin{name=Implies _, ...},
			      TupleExpr (_, [False2, _])) = trueobj tp 
	       | applyFun' tp (Builtin{name=Implies _, ...},
			      TupleExpr (_, [_, True2])) = trueobj tp 
	       | applyFun' tp (Builtin{name=Implies _, ...},
			      TupleExpr (_, [e, False2])) =
		      Appl(dp, Builtin{name=Not dp, Type=FunType(dp,tp,tp)}, e)
	       (* in Eq and NotEq the Undefined is treated as a normal value *)
	       | applyFun' tp (ee1 as (Builtin{name=Eq _, ...}), ee2) =
		  (case ee2 of
		       TupleExpr (_, [eq1, eq2]) => 
			   (case theSame(eq1, eq2) of
				SOME true => trueobj tp
			      | SOME false => falseobj tp
			      | NONE => Appl (dp, ee1, ee2))
		     | _ => Appl (dp, ee1, ee2))
	       | applyFun' tp (ee1 as (Builtin{name=NotEq _, ...}), ee2) =
		  (case ee2 of
		       TupleExpr (_, [eq1, eq2]) => 
			   (case theSame(eq1, eq2) of
				SOME true => falseobj tp
			      | SOME false => trueobj tp
			      | NONE => Appl (dp, ee1, ee2))
		     | _ => Appl (dp, ee1, ee2))
	       (* Arrow and Darrow are translated to "Builtin(Implies)" now *)
	       (* Unary operators *)
	       | applyFun' tp (Builtin{name=UMinus(_),...}, Number(p,n)) = Number(p,~n)
	       | applyFun' tp (Builtin{name=Not(_),...}, Builtin{name=False p, ...}) = trueobj tp
	       | applyFun' tp (Builtin{name=Not(_),...}, Builtin{name=True p,...}) = falseobj tp
	       | applyFun' tp (Builtin{name=Not(_),...}, Appl(_, Builtin{name=Not _,...}, e)) = e
	       (* Same for BoolType2 *)
	       | applyFun' tp (Builtin{name=Not(_),...}, False2) = trueobj tp
	       | applyFun' tp (Builtin{name=Not(_),...}, True2) = falseobj tp
	       (* Some boolean optimizations *) 
	       | applyFun' tp (Op as Builtin{name=And _,...},
			      arg as TupleExpr(_, [e1, Appl(_, Builtin{name=Not _,...}, e2)])) =
		  if ptEq(e1, e2) then falseobj tp else Appl(dp, Op, arg)
	       | applyFun' tp (Op as Builtin{name=And _,...},
			      arg as TupleExpr(_, [Appl(_, Builtin{name=Not _,...}, e1), e2])) =
		  if ptEq(e1, e2) then falseobj tp else Appl(dp, Op, arg)
	       | applyFun' tp (Op as Builtin{name=And _,...},
			      arg as TupleExpr(_, [e1, e2])) =
		  if ptEq(e1, e2) then e1 else Appl(dp, Op, arg)
	       | applyFun' tp (Op as Builtin{name=Or _,...},
			      arg as TupleExpr(_, [e1, Appl(_, Builtin{name=Not _,...}, e2)])) =
		  if ptEq(e1, e2) then trueobj tp else Appl(dp, Op, arg)
	       | applyFun' tp (Op as Builtin{name=Or _,...},
			      arg as TupleExpr(_, [Appl(_, Builtin{name=Not _,...}, e1), e2])) =
		  if ptEq(e1, e2) then trueobj tp else Appl(dp, Op, arg)
	       | applyFun' tp (Op as Builtin{name=Or _,...},
			      arg as TupleExpr(_, [e1, e2])) =
		  if ptEq(e1, e2) then e1 else Appl(dp, Op, arg)
	       | applyFun' tp (Op as Builtin{name=Implies _,...},
			      arg as TupleExpr(_, [e1, e2])) =
		  if ptEq(e1, e2) then trueobj tp else Appl(dp, Op, arg)
	       (* Errors, undefined values, etc. *)
	       | applyFun' tp (ee1 as (Builtin _), ee2) =
		  if isUndefined ee2 then 
		      (case getExprType findObject ee1 of
			   FunType(_,_,tp) => bot tp
			 | x => raise SympBug
			       ("evaluateExpr/applyFun/Builtin: not FunType: "^(pt2string x)))
		  else Appl(dp,ee1,ee2)
	       | applyFun' tp (ee1,ee2) =
		  if isUndefined ee1 then 
		      (case getExprType findObject ee1 of
			   FunType(_,_,tp) => bot tp
			 | x => raise SympBug
			       ("evaluateExpr/applyFun: not FunType: "^(pt2string x)))
		  else
		      let val pfun = extractFun ff ee1
		      in
			  (case pfun of
			       SOME (FunClosure {formals = arg, body = body,...}, wrap) =>
				   let val sub = wrap(substitute options (ee2, arg, body))
				   in recurCall ff sub
				   end
			     | SOME (RecurFun (F), wrap) => 
				   let val f = recurCall ff (wrap(applyFun tp(F, RecurFun(F))))
				   in  recurCall ff (applyFun tp(f, ee2))
				   end
			     | SOME(tc as (TypeConstr _), wrap) => wrap(Appl (dp, tc, ee2))
			     | _ => Appl (dp, ee1, ee2))
		      end
 

	  val result =
	    case parseTree1 of
		IfExpr (pos, conds, else_exp) => (handleConds findObject recurCall ff)
		    (pos, conds, else_exp)
	      | ce as CaseExpr (pos, e, clist) => 
		    let val sel =  recurCall ff e
			val tp = getExprType findObject ce
		    in if isUndefined sel then bot tp
		       else (handleCases(recurCall ff)
			      (options, pos, sel, clist, ff(tp)))
		    end
	      | WithExpr (pos, record, wlist) => 
		    let
			val erecord = recurCall ff record
		    in
			(case erecord of 
			     (RecordExpr (rpos, rlist)) =>
				 (handleWith(recurCall ff)) (pos, rpos, rlist, wlist)
			   | _ => WithExpr (pos, erecord, wlist))
		    end
	      | oInst as ObjectInst {obj=object,subst=subst} =>
		    let 
			val eobject = recurCall (ff o subst) object
		    in
			if (hasTypes eobject) then 
			    ObjectInst {obj=eobject,subst=subst}
			else
			    eobject
		    end
	      | object as Object {def=def,name=name,uname=uname,Type=tp} =>
		    let
			val edef = recurCall ff def
		    in
			if (hasTypes edef) then 
			    (* Keep only one the innermost object in a sequence of 
			     nested objects. *)
			    (case edef of
				 Object _ => edef
			       | _ => Object {def=edef,name=name,uname=uname,Type=ff tp})
			else
			    edef
		    end
	      | app as Appl (p, e1, e2) =>
		    let
			val ee1 = recurCall ff e1
			val ee2 = recurCall ff e2
			val tp = getExprType findObject (Appl(p, ee1, ee2))
		    in applyFun tp (ee1,ee2)
		    end
	      | a as ExtractAppl (c1, e) =>
		    let val ee = recurCall ff e
			fun getParmType(FunType(_,tp,_)) = tp
			  | getParmType (u as Uid _) = 
			      (case findObject u of
				   SOME tp => getParmType tp
				 | NONE => raise SympBug
				     ("evaluateExpr/ExtractAppl: no object with such Uid: "
				      ^(pt2string u)))
			  | getParmType x = raise SympBug
			      ("evaluateExpr/ExtractAppl: c1 is not FunType: "
			       ^(pt2string x))
			val tp = getParmType(getExprType findObject c1)
		    in
			case ee of
			    Appl (_, c2, e) =>
				if eqUNames (c1, c2) then e
				else
				    raise EvalError ("Application extraction failed: " ^
						     pt2string a)
			  | TypeConstr _ => raise EvalError
				("Application extraction failed: " ^ pt2string a)
			  | x => if isUndefined x then bot tp
				 else ExtractAppl (c1, ee)
		    end
	      | a as ExtractIndex e =>
		    let
			val ee = recurCall ff e
			val tp = getExprType findObject a
		    in
			case ee of
			    Appl (_, c as TypeConstr _, e) => c
			  | c as TypeConstr _ => c
			  | x => if isUndefined x then bot tp
				 else ExtractIndex ee
		    end
	      | ExtractTuple (n, tuple) => 
		    let
			val etuple = recurCall ff tuple
		    in
	      (* After evaluation some tuple elements might not be values,
	         but we might still extract the element we need *)
			(case extractTuple n etuple of
			     SOME x =>
			      (case x of
				   Undefined _ => bot(getExprType findObject 
						      (ExtractTuple(n, etuple)))
				 | _ => x)
			   | NONE => ExtractTuple (n, etuple))
		    end
	      | r as ExtractRecord (field, record) =>
		    let 
			val erecord = recurCall ff record
		    in
	      (* After evaluation some field expressions might not be values,
	         but we might still extract the field we need *)
			(case extractRecord field erecord of
			     SOME x =>
			      (case x of
				   Undefined _ => bot(getExprType findObject 
						      (ExtractRecord(field, erecord)))
				 | _ => x)
			   | NONE => 
			       (case erecord of
				    (* No updated field matched, thus
				       top level `with' is irrelevant.
				       IMPORTANT: OTHER MODULES RELY ON THIS FEATURE! *)
				    WithExpr(_,e,_) => ExtractRecord(field,e)
				  | _ => ExtractRecord (field, erecord)))
		    end
	      | TupleExpr (_, tlist) =>
		    TupleExpr (dp, List.map(recurCall ff) tlist)
	      | RecordExpr (_, rlist) =>
		    RecordExpr (dp, List.map(recurCall ff) rlist)
	      | RecordAsst (_, fname, fval) =>
		    RecordAsst (dp, fname, recurCall ff fval)
	      | WithAsst (_, fname, fval) =>
		    WithAsst (dp, fname, recurCall ff fval)
	      | NondetExpr (_, ndlist) =>
		    NondetExpr (dp, List.map(recurCall ff) ndlist)
	      | LetClosure {locals = llist, parent = par, body = body} =>
		    let val ebody = recurCall ff body
		    (* Do not evaluate definition list, it's been evaluated at typechecking *)
		    in if isValue ebody then ebody
		       else LetClosure {locals = llist, 
					parent = par, body = ebody}
		    end
	      | Builtin{name=n, Type=tp} => Builtin{name=n, Type=ff tp}
	      | u as Uid _ => (case findObject u of
				   SOME e => recurCall ff e
				 | NONE => raise SympBug
				     ("evaluateExpr: no object with such Uid: "
				      ^(pt2string u)))
	      | Next e => Next(recurCall ff e)
	      | _ => parseTree1
	 in
	    (popFunStackLazy(funName,fn () => pt2stringDebug result);
	     result)
	 end
	val res = recurCall (fn x=>x) parseTree1
	val _ = popFunStackLazy(funName,fn () => pt2stringDebug res)
      in res
      end

    (* fun evaluateExprTree options parseTree1 =
      ptTransform (evaluateExpr options) parseTree1 *)

  end
