functor AbstractFun(structure Hash: HASH
		 structure ParserDefault: PARSER_DEFAULT): ABSTRACT =
  struct
    structure Hash = Hash
    structure ParserDefault = ParserDefault

    open Hash
    open Pos
    open Str
    open ParserDefault
    open Interface
    open ParseTreeStruct
    open Options

    exception AbstractError of string

    datatype TypeSize = FiniteSize of int | InfiniteSize

    (* Representation of an abstraction: 
       TypeAbstraction is an abstract interpretation of a single type.
       `Type' is the type's (finite) abstraction,
       `absInt' holds abstract interpretations of some functions like
       equality, inequalities, +, *, etc. in associative list format.
       An abstract interpretation is just a symp function, and may
       possibly return a nondeterministic expression. *)

    type TypeAbstraction = {Type: ParseTree, (* Abstraction of the type: AbstractType list *)
			    (* Abstract interpretation of some related functions *)
			    absInt: (ParseTree * ParseTree) list,
			    (* Mapping from actual values to the abstract values *)
			    mapping: ParseTree -> ParseTree }

    (* Global abstraction is a hash associating types with their
       abstract interpretations. *)

    type Abstraction = (ParseTree,TypeAbstraction) Hash

    fun makeTypeAbstraction(tp,mapping,absIntList) =
	 ({Type=tp, mapping=mapping, absInt=absIntList}: TypeAbstraction)

    fun addAbstractInterpretations ({Type=tp,mapping=m,absInt=a}: TypeAbstraction) newa =
	  ({Type=tp, mapping=m, absInt=newa@a}: TypeAbstraction)

    (* Creates a new empty abstraction that doesn't abstract anything yet.
       Do indexing by the type only. *)
    fun makeAbstraction() =
	  makeHashDefault(ptEq, pt2string): Abstraction

    (* (addAbstraction opt A (tp,abs_tp)) adds abstraction information
       about type tp whose abstraction will be abs_tp, and return the
       updated A. *)

    fun addAbstraction options abs (tp, abstp) = insertHash(abs,tp,abstp)

    fun getTypeAbstraction options abs tp = findHash(abs,tp)

    (* Find an abstract interpretation of a function in the abstract type *)
    fun getAbsInt options ({absInt=ai,...}: TypeAbstraction) f =
	  Option.map(fn (x,y)=>y)(List.find(fn (x,y)=>ptEq(x,f)) ai)

    fun getAbsType({Type=tp,...}: TypeAbstraction) = tp

    fun getAbsMap({mapping=m,...}: TypeAbstraction) = m

    (* Determine if the type is finite.  The type may be an
       abstraction of some concrete type. *)
    fun isFiniteType (TypeClosure{def=tp,...}) = isFiniteType tp
      (* A raw Id can only occur in recursive datatypes as a self-reference.
         If we find one, then the type is infinite.  (Is it too much of a hack?) *)
      | isFiniteType (Id _) = false
      | isFiniteType (StaticFormalType _) = false
      | isFiniteType (PType _) = false
      | isFiniteType (Type(p,t)) = isFiniteType t
      | isFiniteType (Datatype(p,t)) = isFiniteType t
      | isFiniteType (BoolType _) = true
      | isFiniteType (NatType _) = false
      | isFiniteType (IntType _) = false
      | isFiniteType (RangeType _) = true
      | isFiniteType (FunType(_,t1,t2)) =
	  (isFiniteType t1) andalso (isFiniteType t2)
      | isFiniteType (ArrayType(_,t1,t2)) =
	  (isFiniteType t1) andalso (isFiniteType t2)
      | isFiniteType (RecordType(_,lst)) = List.all isFiniteType lst
      | isFiniteType (RecordField{Type=tp,...}) = isFiniteType tp
      | isFiniteType (TupleType(_,lst)) = List.all isFiniteType lst
      | isFiniteType (EnumType(_,lst)) = List.all isFiniteType lst
      | isFiniteType (TypeConstr{Type=t,...}) = isFiniteType t
      | isFiniteType _ = false

    (* Determine the type size.  It is possible that the type is
       larger than SML's int, in which case getTypeSize may return
       InfiniteSize, even if isFiniteType says it's finite.  In the
       future we may even control the threshold of when a finite type
       is considered "infinite" through command line options. *)
    fun getTypeSize options tp = 
	let fun log2 x = trunc((Math.ln(Real.fromInt x)) / (Math.ln 2.0))
	    val maxlog2Int = Option.map(fn x=>log2 x) Int.maxInt
	    fun ltMaxlog n =
		(case maxlog2Int of
		     NONE => true
		   | SOME m => n < m)
	    fun exp(x,0) = 1
	      | exp(x,y) = x*(exp(x,y-1))
	    fun sumloglist [] = SOME 0
	      | sumloglist (InfiniteSize::_) = NONE
	      | sumloglist ((FiniteSize n)::tl) = 
		  (case sumloglist tl of
		       NONE => NONE
		     | SOME r => SOME(r+(log2 n)))
	    fun multlist [] = FiniteSize 1
	      | multlist (InfiniteSize::_) = InfiniteSize
	      | multlist ((FiniteSize n)::tl) = 
		  (case multlist tl of
		       InfiniteSize => InfiniteSize
		     | FiniteSize r => FiniteSize(n*r))
	    fun sumlist [] = FiniteSize 1
	      | sumlist (InfiniteSize::_) = InfiniteSize
	      | sumlist ((FiniteSize n)::tl) = 
		  (case sumlist tl of
		       InfiniteSize => InfiniteSize
		     | FiniteSize r => FiniteSize(n+r))
	    fun loop (Id _) = InfiniteSize (* means we are in recursive datatype *)
	      | loop (StaticFormalType _) = InfiniteSize
	      | loop (PType _) = InfiniteSize
	      | loop (Type(_,t)) = loop t
	      | loop (Datatype(_,t)) = loop t
	      | loop (BoolType _) = FiniteSize 2
	      | loop (NatType _) = InfiniteSize
	      | loop (IntType _) = InfiniteSize
	      | loop (r as (RangeType(p,Number(_,n1),Number(_,n2)))) =
		 (* Let's just hope the user doesn't use minInt..maxInt range... *)
	         if n1 <= n2 then
		   (FiniteSize(n2-n1+1)
		      handle _ => raise AbstractError
			 ((pos2string p)^": Range is too wide: "^(pt2string r)))
		 else FiniteSize 0
	      | loop (FunType(_,t1,t2)) = 
		 let val s1 = loop t1
		     val s2 = loop t2
		 in (case (s1,s2) of
			 (InfiniteSize,_) => InfiniteSize
		       | (_,InfiniteSize) => InfiniteSize
		       | (FiniteSize n1, FiniteSize n2) => 
			     if ltMaxlog(n2 * (log2 n1)) then FiniteSize(exp(n1,n2))
			     else InfiniteSize)
		 end
	      | loop (ArrayType(_,t1,t2)) = 
		 let val s1 = loop t1
		     val s2 = loop t2
		 in (case (s1,s2) of
			 (InfiniteSize,_) => InfiniteSize
		       | (_,InfiniteSize) => InfiniteSize
		       | (FiniteSize n1, FiniteSize n2) => 
			     if ltMaxlog(n2 * (log2 n1)) then FiniteSize(exp(n1,n2))
			     else InfiniteSize)
		 end
	      | loop (RecordType(_,lst)) =
		 let val slist=List.map loop lst
		 in (case sumloglist slist of
			 NONE => InfiniteSize
		       | SOME n =>
			  if ltMaxlog n then multlist slist
			  else InfiniteSize)
		 end
	      | loop (RecordField{Type=tp,...}) = loop tp
	      | loop (TupleType(_,lst)) =
		 let val slist=List.map loop lst
		 in (case sumloglist slist of
			 NONE => InfiniteSize
		       | SOME n =>
			  if ltMaxlog n then multlist slist
			  else InfiniteSize)
		 end
	      | loop (EnumType(_,lst)) = sumlist(List.map loop lst)
	      | loop (TypeConstr{Type=tp,...}) =
		 (case tp of
		      FunType(_,t,_) => loop t
		    | _ => FiniteSize 1)
	      | loop (AbstractType lst) = FiniteSize(List.length lst)
	      | loop _ = InfiniteSize
	in loop tp
	end

    val trueVal = Builtin{name=True dp, Type=BoolType dp}
    val falseVal = Builtin{name=False dp, Type=BoolType dp}

    (* (getTypeValues opt sizeLimit tp) Return the list of all possible values
       of the type tp, if possible.  Return NONE if the type is
       infinite or uninterpreted, or is larger than the `limit'. *)
    fun getTypeValues options limit tp = 
	let val funName = "getTypeValues"
	    val _ = pushFunStackLazy(funName, fn()=>pt2string tp)
	    fun rangeList(n1,n2) =
	         if n2 < n1 then []
		 else (Number(dp,n1))::(rangeList(n1+1,n2))

	    (* Take N value lists (in a list) and generate a list of
	       all combinations of those values as N-lists.  For
	       instance, on input [[0,1],[true,false],[a,b:T]] it will
	       produce a list [[0,true,a],[0,true,b],...,[1,false,b]]
	       (8 elements) *)

	    fun comblist [] = [[]]
	      | comblist (hd::tl) =
		let val lst = comblist tl
		    fun ff x = List.map(fn l=>x::l) lst
		in List.foldr(op @) [] (List.map ff hd)
		end
	    (* If one of the elements is NONE, return NONE, otherwise
	       extract all lists from SOME, apply ff to them and wrap
	       it into SOME. *)
	    fun mapWorst ff [] = SOME []
	      | mapWorst ff (x::tl) =
		 (case ff x of
		      NONE => NONE
		    | SOME r => 
			  Option.map(fn l=>r::l) (mapWorst ff tl))
	    fun l2str lst = "["^(ptlist2str ", " lst)^"]"
	    fun resStrFn res () = option2string "NONE" (Option.map l2str res)
	    fun loop tp =
		let val funName = "getTypeValues/loop"
		    val _ = pushFunStackLazy(funName, fn()=>pt2string tp)
		    val res = loop' tp
		    val _ = popFunStackLazy(funName, resStrFn res)
		in res
		end
	    and loop' (Id _) = NONE (* means we are in recursive datatype *)
	      | loop' (StaticFormalType{value=SOME t,...}) = loop t
	      | loop' (StaticFormalType _) = NONE
	      | loop' (PType _) = NONE
	      | loop' (Type(_,t)) = loop t
	      | loop' (Datatype(_,t)) = loop t
	      | loop' (BoolType _) = SOME[trueVal, falseVal]
	      | loop' (NatType _) = NONE
	      | loop' (IntType _) = NONE
		(* Don't check for limits here, it'll be checked globally *)
	      | loop' (r as (RangeType(p,Number(_,n1),Number(_,n2)))) = SOME(rangeList(n1,n2))
	      (* Here we should have generated a list of
	         FunClosure's, but it's hardly useful, so we don't. *)
	      | loop' (FunType(_,t1,t2)) = NONE
	      | loop' (ArrayType(_,t1,t2)) = NONE
	      | loop' (RecordType(_,lst)) =
		 let val slist = mapWorst loop lst (* values of each field *)
		     val clist = Option.map(fn l=>comblist l) slist (* Cartesian product *)
		 in (Option.map(fn cl=>List.map(fn l=>RecordExpr(dp,l)) cl) clist)
		     : ParseTree list option
		 end
	      | loop' (RecordField{Type=tp,name=name}) =
		 Option.map(List.map(fn x=>RecordAsst(dp,name,x))) (loop tp)
	      | loop' (TupleType(_,lst)) =
		 let val slist = mapWorst loop lst (* values of each field *)
		     val clist = Option.map(fn l=>comblist l) slist (* Cartesian product *)
		 in Option.map(List.map(fn l=>TupleExpr(dp,l))) clist
		 end
	      | loop' (EnumType(_,lst)) =
		 Option.map(List.foldr(op @) []) (mapWorst loop lst)
	      | loop' (tc as (TypeConstr{Type=tp,...})) =
		 (case tp of
		      FunType(_,t,_) => Option.map(List.map(fn x=>Appl(dp,tc,x))) (loop t)
		    | _ => SOME[tc])
	      | loop' (TypeClosure{recursive=false, def=t,...}) = loop t
	      | loop' (AbstractType lst) = SOME lst
	      | loop' x = NONE
	    val res = (case getTypeSize options tp of
			   InfiniteSize => NONE
			 | FiniteSize n =>
			       if n > limit then NONE
			       else loop tp)
	    val _ = popFunStackLazy(funName, resStrFn res)
	in res
	end

    (* (abstract opt A tp) returns an abstraction of type tp according to A *)
    fun abstract options (abs: Abstraction) tp =
	let val recur = abstract options abs
	    val recurlist = List.map recur
	in
	  (case findHash(abs,tp) of
	       SOME ta => SOME ta
	     | NONE => NONE
	       (* (case tp of
		    TypeInst(p,lst,t) => TypeInst(p, recurlist lst, recur t)
		  | TypeClosure{name=name,
				uname=uname,
				params=params,
				def=def,
				recursive=r,
				parent=parent} =>
			TypeClosure{name=name,
				    uname=uname,
				    params=params,
				    def=recur def,
				    recursive=r,
				    parent=parent}
		   (* These two types are uninterpreted *)
		   | StaticFormalType _ => tp
		   | PType _ => tp
		   | Type(p,t) => Type(p, recur t)
		   | Datatype(p,t) => Datatype(p, recur t)
		   | BoolType _ => tp
		   | NatType _ => tp
		   | IntType _ => tp
		   | RangeType _ => tp
		   | FunType(p,t1,t2) => FunType(p,recur t1,recur t2)
		   | ArrayType(p,t1,t2) => ArrayType(p,recur t1,recur t2)
		   | TupleType(p,lst) => TupleType(p,recurlist lst)
		   | RecordType(p,lst) => RecordType(p,recurlist lst)
		   | RecordField{name=name,Type=t} => RecordField{name=name,
								   Type=recur t}
		   | EnumType(p,lst) => EnumType(p,recurlist lst)
		   | TypeConstr{name=name,uname=uname,Type=t} => 
			(case t of
			     FunType(p,t1,t2) =>
			       TypeConstr{name=name,uname=uname,
					  Type=FunType(p,recur t1,t2)}
			   | _ => tp)
		   | _ => raise SympBug("abstract: invalid type: "
					^(pt2string tp))) *) )
	end

    fun isAbstract options (abs: Abstraction) tp =
	let val recur = isAbstract options abs
	    val recurlist = List.exists recur
	in
	  (case findHash(abs,tp) of
	       SOME _ => true
	     | NONE =>
	       (case tp of
		    TypeInst(p,lst,t) => (recurlist lst) orelse recur t
		  | TypeClosure{name=name,
				uname=uname,
				params=params,
				def=def,
				recursive=r,
				parent=parent} => recur def
		   (* These two types are uninterpreted *)
		   | StaticFormalType _ => false
		   | PType _ => false
		   | Type(p,t) => recur t
		   | Datatype(p,t) => recur t
		   | BoolType _ => false
		   | NatType _ => false
		   | IntType _ => false
		   | RangeType _ => false
		   | FunType(p,t1,t2) => (recur t1) orelse (recur t2)
		   | ArrayType(p,t1,t2) => (recur t1) orelse (recur t2)
		   | TupleType(p,lst) => recurlist lst
		   | RecordType(p,lst) => recurlist lst
		   | RecordField{name=name,Type=t} => recur t
		   | EnumType(p,lst) => recurlist lst
		   | TypeConstr{name=name,uname=uname,Type=t} => 
			(case t of
			     FunType(_,t1,_) => recur t1
			   | _ => true)
		   | _ => raise SympBug("isAbstract: invalid type: "
					^(pt2string tp))))
	end

  end
