signature ABSTRACT =
  sig
    structure ParserDefault: PARSER_DEFAULT

    type TypeAbstraction
    type Abstraction
    datatype TypeSize = FiniteSize of int | InfiniteSize


    type ParseTree = ParserDefault.ParseTreeStruct.ParseTree
    type options = ParserDefault.Interface.Options.options

    (* (abstract opt A tp) returns the abstracted representation of
       type tp according to A, or NONE. *)

    val abstract: options -> Abstraction -> ParseTree -> TypeAbstraction option

    (* Check whether the type is (partially) abstracted;
       that is, whether it has an abstracted type in its definition. *)
    val isAbstract: options -> Abstraction -> ParseTree -> bool

    (* Returns the abstract interpretation of a type, if there is one.
       It is only defined over types that are explicitly abstracted,
       and will give NONE for any other type, even if it uses an
       abstracted type in its definition. 

       Basically, we want to avoid constructing complicated abstract
       functions by induction over the type structure. *)
    val getTypeAbstraction: options -> Abstraction -> ParseTree
	  -> TypeAbstraction option

    val getAbsType: TypeAbstraction -> ParseTree
    val getAbsMap: TypeAbstraction -> (ParseTree -> ParseTree)

    (* Find an abstract interpretation of a function in the abstract type *)
    val getAbsInt: options -> TypeAbstraction -> ParseTree -> ParseTree option

    (* Creates a new empty abstraction that doesn't abstract anything yet *)
    val makeAbstraction: unit -> Abstraction

    (* makeTypeAbstraction(tp, absMapping, absInterpretList) Creates a
       new type abstraction, with absMapping a function mapping the
       values of the original type to the abstract type `tp', and
       (optional) list of abstract interpretations of some related
       functions. *)
    val makeTypeAbstraction:
	 ParseTree * (ParseTree -> ParseTree) * ((ParseTree * ParseTree) list)
	  -> TypeAbstraction

    val addAbstractInterpretations: TypeAbstraction -> ((ParseTree * ParseTree) list)
	  -> TypeAbstraction

    (* (addAbstraction opt A (tp,abs_tp)) adds abstraction information
       about type tp whose abstraction will be abs_tp, and return the
       updated A. *)

    val addAbstraction: options -> Abstraction -> (ParseTree * TypeAbstraction)
	  -> Abstraction

    (* Determine if the type is finite.  The type may be an
       abstraction of some concrete type. *)
    val isFiniteType: ParseTree -> bool

    (* Determine the type size.  It is possible that the type is
       larger than SML's int, in which case getTypeSize may return
       InfiniteSize, even if isFiniteType says it's finite.  In the
       future we may even control the threshold of when a finite type
       is considered "infinite" through command line options. *)
    val getTypeSize: options -> ParseTree -> TypeSize

    (* (getTypeValues opt sizeLimit tp) Return the list of all possible values
       of the type tp, if possible.  Return NONE if the type is
       infinite or uninterpreted, or is larger than the sizeLimit. *)
    val getTypeValues: options -> int -> ParseTree -> ParseTree list option

  end
