(* Typechecking without overloading of identifiers *)

functor AthenaTypecheckFun(structure Hash: HASH
			   structure ParseTreeStruct: ATHENA_PARSE_TREE
			   structure Interface: INTERFACE): 
  ATHENA_TYPECHECK =
  struct
    structure Hash = Hash
    structure ParseTreeStruct = ParseTreeStruct
    structure Interface = Interface

    open Str
    open Pos
    open Interface
    open ParseTreeStruct
    open Hash
    open Options

    (* Some vars for statistics *)

    val tHashMaxLength = ref 0

    (* Just a dummy for now. *)
    fun typeCheckExpr options expr _ =
      expr

    (* Converts keys (and keys only) to strings for some errors. *)
    fun key2string (Priv(_,Id(_,id))) =
      "PVK " ^ id
      | key2string (Pub(_,Id(_,id))) =
      "PK " ^ id
      | key2string _ =
      "NOT A KEY"

    (* Tells if the key given is available in knownKeys. *)
    fun knowKey (Priv(_,Id(_,id)), knownKeys) =
      (List.exists (fn (Priv(_,Id(_,x))) => x = id |
		    _ => false) knownKeys)
      | knowKey (Pub(_,Id(_,id)), knownKeys) =
      (List.exists (fn (Pub(_,Id(_,x))) => (x = id) |
		    _ => false) knownKeys)
      | knowKey (x, _) = 
      raise TypeError ((pos2string (pos x)) ^
		       ": attempts to extract key from non-principal (" ^
		       (pt2string x) ^ ").")

    (* Returns inverse of a given key for decryption via pattern-matching. *)
    fun keyInv (Priv(p,id)) =
      (Pub(p,id))
      | keyInv (Pub(p,id)) =
      (Priv(p,id))
      | keyInv x = x

    (* Makes a hash of the parameter types for a role definition, and
       establishes the initial knownKey list based on types. *)
    fun paramTypes (NONE, hash, knownKeys) =
      (hash, knownKeys)
      | paramTypes (SOME [], hash, knownKeys) =
      (hash, knownKeys)
      | paramTypes (SOME (Parameter(_, ids, tp) :: rest), hash, knownKeys) =
      (List.app (fn Id (_,id) => (insertHashDestructive (hash, id, tp); ()) |
		 _ => raise SympBug "Non-ID in parameter list!") ids;
       case tp of
	 SelfType(_) => 
	   (paramTypes (SOME rest, hash,
			((List.map (fn x => (Pub (Pos.dp, x))) ids) @
			 (List.map (fn x => (Priv (Pos.dp, x))) ids) @
			 knownKeys)))
       | PrincipalType(_) =>
	   (paramTypes (SOME rest, hash,
			((List.map (fn x => (Pub (Pos.dp, x))) ids) @
			 knownKeys)))
       | _ =>
	   paramTypes (SOME rest, hash, knownKeys))
      | paramTypes (SOME (h::_), hash, knownKeys) =
      raise SympBug ((pos2string (pos h)) ^ 
		     ": non-parameter in role parameters!")

    (* Checks a Send action for proper use of keys, types. *)
    fun checkSend (pHash, Crypt (p, m, k), initialized, knownKeys) =
      if (not (knowKey (k, knownKeys))) then
	raise TypeError ((pos2string p) ^
			 ": attempts to encrypt with unavailable key (" ^
			 (key2string k) ^ ").")
      else
	checkSend (pHash, m, initialized, knownKeys)
      | checkSend (pHash, i as Id (p, id), initialized, knownKeys) =
	(case findHash (pHash, id) of
	   NONE => raise TypeError ((pos2string p) ^
				    ": undeclared identifier (" ^
				    id ^ ").")
	 | SOME (FreshNonceType(_)) =>
	     if (not (List.exists (fn (Id(_, x)) => x = id |
				   _ => false) initialized)) then
	       ((Id (p, id)) :: initialized)
	     else
	       initialized
         | SOME _ => initialized)
      | checkSend (pHash, MessageTuple (_, mlist), initialized, knownKeys) =
	List.foldr (fn (m, init) => checkSend (pHash, m, init, knownKeys))
	initialized mlist
      | checkSend (pHash, KeyPair (p, (k1, k2)), initialized, knownKeys) =
	raise SympBug "KeyPair is not supported yet."
      | checkSend (pHash, key as Pub (p, k), initialized, knownKeys) =
	if (not (knowKey (key, knownKeys))) then
	  raise TypeError ((pos2string p) ^
			   ": attempts to send unavailable key (" ^
			   key2string (key) ^ ").")
	else
	  initialized
      | checkSend (pHash, key as Priv (p, k), initialized, knownKeys) =
	if (not (knowKey (key, knownKeys))) then
	  raise TypeError ((pos2string p) ^
			   ": attempts to send unavailable key (" ^
			   key2string (key) ^ ").")
	else
	  initialized
      | checkSend (pHash, m, initialized, knownKeys) =
	raise SympBug ((pos2string (pos m)) ^ 
		       ": SHOULD NOT REACH THIS CASE.  checkSend in Athena.")

    (* Checks a Send action for proper use of keys, types, freshNonces. *)
    fun checkReceive (pHash, Crypt (p, m, k), initialized, knownKeys) =
      if (not (knowKey (keyInv k, knownKeys))) then
	raise TypeError ((pos2string p) ^
			 ": attempts to decrypt with unavailable key (" ^
			 (key2string (keyInv k)) ^ ").")
      else
	checkReceive (pHash, m, initialized, knownKeys)
      | checkReceive (pHash, i as Id (p, id), initialized, knownKeys) =
	(case findHash (pHash, id) of
	   NONE => raise TypeError ((pos2string p) ^
				    ": undeclared identifier (" ^
				    id ^ ").")
	 | SOME (FreshNonceType(_)) =>
	     if (not (List.exists (fn Id(_, x) => x = id |
				   _ => false) initialized)) then
	       raise TypeError ((pos2string p) ^ 
				": tries to receive uncreated freshNonce (" ^
				id ^ ").")
	     else
		knownKeys		
         | SOME _ => knownKeys)
      | checkReceive (pHash, MessageTuple (_, mlist), initialized, knownKeys) =
	List.foldr (fn (m, known) => checkReceive (pHash, m, initialized, 
						   known))
	   knownKeys mlist
      | checkReceive (pHash, KeyPair (p, (k1, k2)), initialized, knownKeys) =
	raise SympBug "KeyPair is not supported yet."
      | checkReceive (pHash, pvk as Priv (p, i as Id(_,id)), initialized, 
		      knownKeys) =
	(case findHash (pHash, id) of
	   NONE => raise TypeError ((pos2string (pos i)) ^
				    ": undeclared identifier ("
				    ^ id ^ ").")
	 | SOME (PrincipalType(_)) =>
	     if (not (knowKey (pvk, knownKeys))) then
	       (pvk :: knownKeys)
	     else
	       knownKeys
	 | SOME (SelfType (_)) =>
	     knownKeys
         | SOME _ => 
	     raise TypeError ((pos2string p) ^
			      ": attempts to extract key from non-principal ("^
			      id ^ ")."))
      | checkReceive (pHash, m, initialized, knownKeys) =
	raise SympBug 
	  ((pos2string (pos m)) ^ 
	   ": SHOULD NOT REACH THIS CASE.  checkReceive in Athena.")

    (* Checks the body of a role, tracking labels as well. *)
    fun checkBody (pHash, [], _, _, labels) = labels
      | checkBody (pHash, Send (p, m) :: rest, initialized, knownKeys, 
		   labels) =
      checkBody (pHash, rest, checkSend (pHash, m, initialized, 
					 knownKeys),
		 knownKeys, labels)
      | checkBody (pHash, Receive (p, m) :: rest, initialized, knownKeys,
		   labels) =
      checkBody (pHash, rest, initialized,
		 checkReceive (pHash, m, initialized, knownKeys), labels)
      | checkBody (pHash, Labeled (lid as Id(p,id), act) :: rest, initialized, 
		   knownKeys, labels) =
      if (not (List.exists (fn Id(_, x) => x = id |
			    _ => false) labels)) then
	checkBody (pHash, act :: rest, initialized, knownKeys, lid :: labels)
      else
	raise TypeError ((pos2string p) ^ ": duplicated label in role body.")
      | checkBody (pHash, h :: _, initialized, knownKeys, labels) =
      raise SympBug ((pos2string (pos h)) ^ 
		     ": non send/receive in protocol body.")

    (* Return the role "closure" for the given expression, which in
       the case of a role is essentially unchanged.  First return
       value is the Role, second is the parameters (for easy access
       in typechecking) and the third is a list of labels in
       reverse order. *)
    fun typeCheckRoleExpr options 
      (role as (Role (p, {name=Id(_,id), params=pa, body=RoleBody(_,b)})))  =
      let
	val (paramHash, knownKeys : ParseTree list) = 
	  paramTypes (pa, 
		      (makeHashDefault(op=, 
				       (fn x => x))),
		      [])
      in
	(role, pa, checkBody (paramHash, b, [], knownKeys, []))
      end
      | typeCheckRoleExpr options r = 
      raise SympBug ((pos2string (pos r)) ^ 
		     ": proper role expression expected.")

    (* Raises a TypeError if two types are not compatible (A <: B). *)
    fun typeLTEqual ((FreshNonceType _, _), NonceType _) = true
      | typeLTEqual ((SelfType _, _), PrincipalType _) = true
      | typeLTEqual ((PrincipalType _, _), SelfType _) = true
      | typeLTEqual ((NonceType _, _), NonceType _) = true
      | typeLTEqual (_, MessageType _) = true
      | typeLTEqual ((FreshNonceType _, _), FreshNonceType _) = true
      | typeLTEqual ((SelfType _, _), SelfType _) = true
      | typeLTEqual ((PrincipalType _, _), PrincipalType _) = true
      | typeLTEqual ((tp1, Id (p, id)), tp2) =
      raise TypeError ((pos2string p) ^ ": " ^ id ^
		       " has type " ^ 
		       (pt2string tp1) ^ ", but should have type " ^
		       (pt2string tp2) ^ " (from " ^
		       (pos2string (pos tp2)) ^ ").")
      | typeLTEqual ((_, expr), _) =
      raise SympBug ((pos2string (pos expr)) ^
		     ": non-ID in instantiated parameters!")

    (* Substitutes types for full parameters in an instance of a 
       role/theorem, to ease typechecking. *)
    fun substitute (scopeParams, []) = []
      | substitute (scopeParams, (Id(p,id)::t)) =
      (case (List.find (fn (Parameter (_, list, _)) =>
			List.exists (fn Id (_,x) => x = id |
				     _ => false) list |
			_ => false) scopeParams) of
	 NONE => raise TypeError ((pos2string p) ^ 
				  ": undeclared identifier (" ^
				  id ^ ").")
       | SOME (Parameter (_, _, tp)) => (tp, Id(p,id))
	   :: (substitute (scopeParams, t))
       | SOME _ => raise SympBug 
	   "This cannot be.  If it is, see athena-typcheck.sml:substitute.")
      | substitute (scopeParams, (expr::t)) =
      raise SympBug ((pos2string (pos expr) ^ 
		      ": non-ID in instantiated parameters!"))

    (* Checks if all actual parameters given to a role/theorem have
       types <: the formal parameter types. *)
    fun checkLTEqual (p, [], []) = ()
      | checkLTEqual (p, (Parameter (_, [], tp1)) :: t1, given) =
      checkLTEqual (p, t1, given)
      | checkLTEqual (p, (Parameter (_, h::ids, tp1)) :: t1, tp2 :: t2) =
      (typeLTEqual (tp2, tp1);
       checkLTEqual (p, (Parameter (p, ids, tp1)) :: t1, t2))
      | checkLTEqual (p, [], h::t) = 
      raise TypeError ((pos2string p) ^ 
		       ": extra parameters given to role.")
      | checkLTEqual (p, h::t, []) = 
      raise TypeError ((pos2string p) ^ 
		       ": too few parameters given to role.")
      | checkLTEqual (p, expr :: _, _) =
      raise SympBug ((pos2string (pos expr)) ^ 
		     ": non parameter in parameter list!")

    (* Checks if the parameters can be unified (and that types are <:). *)
    fun canUnify (p, scopeParams, SOME givenParams, SOME realParams) =
      let
	val scopePs = (case scopeParams of
			 SOME l => l
		       | NONE => [])
	val subs = substitute (scopePs, givenParams)
      in
	(checkLTEqual (p, realParams, subs))
      end
      | canUnify (p, _, NONE, NONE) = ()
      | canUnify (p, _, SOME x, NONE) =
      raise TypeError ((pos2string p) ^ 
		       ": role given parameters, expects none.")
      | canUnify (p, _, NONE, SOME x) =
      raise TypeError ((pos2string p) ^ 
		       ": role not given parameters, expects them.")

    (* Checks that ranges are in correct order and exist in the
       given role. *)
    fun checkLabelOrder (rid, p, label1, false, label2, true, []) =
      raise TypeError ((pos2string p) ^ ": " ^ label1 ^ 
		       "not defined in role " ^ rid ^ 
		       " in ranged strand expression.")
      | checkLabelOrder (rid, p, label1, true, label2, false, []) =
      raise TypeError ((pos2string p) ^ ": " ^ label2 ^ 
		       "not defined in role " ^ rid ^ 
		       " in ranged strand expression.")
      | checkLabelOrder (rid, p, label1, false, label2, false, []) =
      raise TypeError ((pos2string p) ^ ": " ^ label1 ^ 
		       "not defined in role " ^ rid ^ 
		       " in ranged strand expression.")
      | checkLabelOrder (rid, p, label1, false, label2, false, Id(_,h)::t) =
      if (h = label2) then
	checkLabelOrder (rid, p, label1, false, label2, true, t)
      else
	if (h = label1) then
	  checkLabelOrder (rid, p, label1, true, label2, false, t)
	else
	  checkLabelOrder (rid, p, label1, false, label2, false, t)
      | checkLabelOrder (rid, p, label1, true, label2, false, Id(_,h)::t) =
      if (h = label2) then
	raise TypeError ((pos2string p) ^ ": " ^ label2 ^ " precedes " ^
			 label1 ^ 
			 " in role, but not in ranged strand expression.")
      else
	checkLabelOrder (rid, p, label1, true, label2, false, t)
      | checkLabelOrder (rid, p, label1, false, label2, true, Id(_,h)::t) =
      if (h = label1) then
	()
      else
	checkLabelOrder (rid, p, label1, false, label2, true, t)
      | checkLabelOrder (rid, p, label1, _, label2, _, _) =
      raise SympBug 
	((pos2string p) ^ 
	 ": label listing contains non-label, or true, true given!")

    (* Typechecks the strands in an atomic formula, expanding roles. *)
    fun checkStrands ([], defined, params) = []
      | checkStrands (RoleLabel (p1, Id(_,rid), sparameters, Id(p2,label)) 
		      :: t,
		      defined, params) =
      (case (List.find (fn (Role (_, {name=Id(_,x),...}), _, _) => x = rid |
			_ => false) defined) of
	 NONE => raise TypeError ((pos2string p1) ^
				  ": undefined role (" ^
				  rid ^ ").")
       | SOME (role, tparams, labels) => 
	   (canUnify (p1, params, sparameters, tparams);
	    if (List.exists (fn (Id(_,x)) => x = label |
			     _ => false)
		labels) then
	      ((RoleLabel (p1, role, sparameters, Id (p2,label))) ::
	       (checkStrands (t, defined, params)))
	    else
	      raise TypeError ((pos2string p2) ^
			       ": label " ^ label ^ " not defined in role " ^
			       rid ^ ".")))
      | checkStrands (RoleLabel (p1, Id(_,rid), sparameters, 
				 Range(p2,Id(p3,label1),Id(p4,label2))) :: t,
		      defined, params) =
      (case (List.find (fn (Role (_, {name=Id(_,x),...}), _, _) => x = rid |
			_ => false) 
	     defined) of
	 NONE => raise TypeError ((pos2string p1) ^
				  ": undefined role (" ^
				  rid ^ ").")
       | SOME (role, tparams, labels) => 
	   (canUnify (p1, params, sparameters, tparams);
	    checkLabelOrder (rid, p1, label1, false, label2, false, labels);
	    (RoleLabel (p1, role, sparameters,
			Range(p2,Id(p3,label1),Id(p4,label2)))) ::
	    checkStrands (t, defined, params)))
      | checkStrands (RoleStrand (p, Id(_,rid), sparameters) :: t,
		      defined, params) =
      (case (List.find (fn (Role (_,{name=Id(_,x),...}), _, _) => x = rid |
			_ => false) defined) of
	 NONE => raise TypeError ((pos2string p) ^
				  ": undefined role (" ^
				  rid ^ ").")
       | SOME (role, tparams, _) => 
	   (canUnify (p, params, sparameters, tparams);
	    (RoleStrand (p, role, sparameters)) ::
	    checkStrands (t, defined, params)))
      | checkStrands (expr::t, _, _) =
	 raise SympBug ((pos2string (pos expr)) ^ 
			": not a strand expression!")

    (* Recurses into a formula structure, expanding theorems and roles
       and performing typecheck on parameters. *)
    fun checkFormulaDown (Atomic (p, slist), defined, params) =
      Atomic (p, checkStrands (slist, defined, params))
      | checkFormulaDown (Not (p, f), defined, params) =
      Not (p, checkFormulaDown (f, defined, params))
      | checkFormulaDown (And (p, f1, f2), defined, params) =
      And (p, checkFormulaDown (f1, defined, params),
	   checkFormulaDown (f2, defined, params))
      | checkFormulaDown (Or (p, f1, f2), defined, params) =
      Or (p, checkFormulaDown (f1, defined, params),
	  checkFormulaDown (f2, defined, params))
      | checkFormulaDown (Implies (p, f1, f2), defined, params) =
      Implies (p, checkFormulaDown (f1, defined, params),
	       checkFormulaDown (f2, defined, params))
      | checkFormulaDown (FormulaInstance (p, Id(_,id), sparameters), 
			  defined, params) =
      (case (List.find (fn (Predicate (_, {name=Id(_,x),...}),
			    _, _) => x = id |
			(Theorem (_, {name=Id(_,x),...}), _, _) =>
			  x = id |
			_ => false) defined) of
	 NONE => (case (List.find (fn (Role (_, {name=Id(_,x),...}), _, _) =>
				   x = id |
				   _ => false) defined) of
		    NONE => raise TypeError 
		      ((pos2string p) ^
		       ": undefined role, predicate or theorem (" ^
		       id ^ ").")
		    | SOME _ =>
		      Atomic (p, checkStrands 
			      ([RoleStrand (p, Id(p,id), sparameters)],
			       defined, params)))
       | SOME (predtheorem, tparams, _) => 
	   (canUnify (p, params, sparameters, tparams);
	    predtheorem))
      | checkFormulaDown (expr, _, _) =
      raise SympBug ((pos2string (pos expr)) ^ 
		     ": not a formula!")

    (* Performs typechecking/expansion on a theorem. *)
    fun checkTheorem options (Id(p,n), pa, def) defined =
      let
	val formula = checkFormulaDown (def, defined, pa);
      in
	(Theorem (p, {name=Id(p,n), params=pa, def=formula}), pa, [])
      end
      | checkTheorem options (expr, _, _) _ =
      raise SympBug ((pos2string (pos expr)) ^ 
		     ": name of theorem is not an ID!")
	
    (* Performs typechecking/expansion on a predicate.  Note that
       the two names are merely syntactic sugar. *)
    fun checkPredicate options (Id(p,n), pa, def) defined =
      let
	val formula = checkFormulaDown (def, defined, pa);
      in
	(Predicate (p, {name=Id(p,n), params=pa, def=formula}), pa, [])
      end
      | checkPredicate options (expr, _, _) _ =
      raise SympBug ((pos2string (pos expr)) ^ 
		     ": name of predicate is not an ID!")

    (* Typecheck a theorem expression.  A theorem is either defined
       or referred to by its name.  Expands any roles/theorems
       referred to in the formula.  Return is the same as for 
       typeCheckRoleExpr, except that third item is always empty. *)
    fun typeCheckTheoremInternal options (Theorem (_, {name=n, params=pa,
					       def=d})) defined =
      (checkTheorem options (n, pa, d) defined)
    | typeCheckTheoremInternal options (Predicate (_, {name=n, params=pa,
					       def=d})) defined =
      (checkPredicate options (n, pa, d) defined)
    | typeCheckTheoremInternal options expr _ =
      raise SympBug 
	((pos2string (pos expr)) ^ 
	 ": typeCheckTheoremInternal called on non-Predicate/Theorem!")
	
    (* Returns the name and position in a definition, for checking
       against redefinition. *)
    fun namePos (Role (p,{name=Id(_,n),...})) = (n, p)
      | namePos (Predicate (p,{name=Id(_,n),...})) = (n, p)
      | namePos (Theorem (p,{name=Id(_,n),...})) = (n, p)
      | namePos _ = raise SympBug ("Definition of non role/predicate/theorem!")

    (* Checks to make sure a name isn't being redefined.  We use a 
       single namespace for roles, predicates, and theorems to avoid
       confusion. *)
    fun checkRedefined (thing, []) = ()
      | checkRedefined (thing, ((h,_,_)::t)) =
      let
	val ((n1, p1), (n2, p2)) = (namePos thing, namePos h)
      in
	if (n1 = n2) then
	  raise TypeError ((pos2string p1) ^ ": redefinition of " ^ n1 ^
			   ", originally defined at " ^ (pos2string p2) ^ ".")
	else
	  checkRedefined (thing, t)
      end
      
    (* Recurses into definitions, accumulating them for use in later
       definitions, performing typechecking and expanding definitions when
       they are used. *)
    fun typeCheckAll options (Protocol (_, n, [])) _ = []
      | typeCheckAll options (Protocol (_, n, (role as Role (p, _)) :: t)) 
      defs =
      let
	val newDefined as (role, _, _) = typeCheckRoleExpr options role
      in
	(checkRedefined (role, defs);
	 role :: (typeCheckAll options (Protocol (p,n,t)) 
		  (newDefined :: defs)))
      end
      | typeCheckAll options 
      (Protocol (_, n, ((pred as Predicate (p,_)) :: t))) defs =
      let
	val newDefined as (pred, _, _) = 
	  typeCheckTheoremInternal options pred defs
      in
	(checkRedefined (pred, defs);
	 pred :: (typeCheckAll options (Protocol (p,n,t)) 
		  (newDefined :: defs)))
      end
      | typeCheckAll options 
      (Protocol (_, n, ((theorem as Theorem (p,_)) :: t))) defs=  
      let
	val newDefined as (theorem, _, _) = 
	  typeCheckTheoremInternal options theorem defs
      in
	(checkRedefined (theorem, defs);
	 theorem :: (typeCheckAll options (Protocol (p,n,t)) 
		     (newDefined :: defs)))
      end
      | typeCheckAll options (Protocol (p, n, h::t)) defs = 
      raise SympBug ((pos2string p) ^ 
		     ": role, predicate, or theorem expected (" ^
		     (pt2string h) ^ ").")
      | typeCheckAll options expr defs =
      raise SympBug ((pos2string (pos expr)) ^ 
		     ": protocol expected (in typeCheckAll)!")

    (* Return the Protocol, with all roles and theorems checked and
       (for theorems) expanded fully.  Raises TypeError if there is
       any problem. *)
    fun typeCheckProtocol options (expr as Protocol (p, name, _))=
      let
	val checked = (typeCheckAll options expr [])
      in
	Protocol (p, name, checked)
      end
      | typeCheckProtocol options expr =
      raise SympBug ((pos2string (pos expr)) ^ 
		     ": protocol expected (in typeCheckProtocol)!")

    (* Return the Program, with all roles and theorems checked and
       (for theorems) expanded fully.  Raises TypeError if there is
       any problem. *)
    fun typeCheckProgram options (Program (p, decls)) =
      let
	val debug = verbDebug options "Athena:typeCheckProgram"
	  "Starting typechecking.\n"
	val newDecls = List.map (typeCheckProtocol options) decls
	val debug = verbDebug options "Athena:typeCheckProgram"
	  "Finished typechecking.\n"
      in
	(verbDebug options "Athena:typeCheckProgram"
	 ((pt2string (Program (p, decls))));
	 Program (p, newDecls))
      end
      | typeCheckProgram options expr =
      raise SympBug ((pos2string (pos expr)) ^ 
		     ": non-program given to typeCheckProgram!")

    (* First parsetree should be a context (a Protocol), second should
       be an ID.  Returns the associated Theorem, if one exists.  
       Otherwise raises TypeError. *)
    fun typeCheckTheorem options (Protocol (_, Id(_,pname), decls)) 
      (Id(_,name)) =
      (case (List.find (fn Theorem (_,{name=Id(_,x),...}) => x = name |
			_ => false) decls) of
	 SOME theorem => theorem
       | NONE => raise TypeError ("The theorem " ^
				  name ^ " is not defined in this context ("
				  ^ pname ^ ")."))
      | typeCheckTheorem _ _ _ =
      raise SympBug ("Improper context or theorem in Athena:typeCheckTheorem.")

    (* First parsetree should be a context (a Protocol), second should
       be an ID.  Returns the associated Theorem, if one exists.  
       Otherwise raises TypeError. *)
    fun typeCheckContext options (Program (_, decls)) (Id(_,name)) =
      (case (List.find (fn Protocol (_, Id(_,x) , _) => x = name |
			_ => false) decls) of
	 SOME protocol => protocol
       | NONE => raise TypeError ("The protocol " ^
				  name ^ " is not defined in the program."))
      | typeCheckContext _ _ _ =
      raise SympBug ("Improper context or theorem in Athena:typeCheckContext.")
end
