(* SYMP parser in SML'97 *)

structure AthenaParseTreeStruct = ParseTreeStruct
structure ParseError = ParseError
open ParseTreeStruct
open ParseError

%%

%name AthenaInput

%verbose

%pure

%header (functor AthenaInputLrValsFun(structure Token : TOKEN
			       structure ParseTreeStruct: ATHENA_PARSE_TREE
			       structure ParseError: PARSE_ERROR)
 : sig structure ParserData : PARSER_DATA
            where type result = ParseTreeStruct.ParseTree;
       structure Tokens : Athena_TOKENS
   end)

%eop EOF
%pos int
%noshift EOF

%term PROTOCOL of Pos.Pos
| ROLE of Pos.Pos
| BEGIN of Pos.Pos
| END of Pos.Pos
| RP of Pos.Pos
| LP of Pos.Pos
| LB of Pos.Pos
| RB of Pos.Pos
| LCB of Pos.Pos
| RCB of Pos.Pos
| SEND of Pos.Pos
| RECEIVE of Pos.Pos
| NONCE of Pos.Pos
| FRESHNONCE of Pos.Pos
| MESSAGE of Pos.Pos
| SYMKEY of Pos.Pos
| PRIVKEY of Pos.Pos
| PUBKEY of Pos.Pos
| DOT of Pos.Pos
| EQ of Pos.Pos
| NOTEQ of Pos.Pos
| IMPLIES of Pos.Pos
| ARROW of Pos.Pos
| DARROW of Pos.Pos
| BAR of Pos.Pos
| DBAR of Pos.Pos
| COMMA of Pos.Pos
| COLON of Pos.Pos
| SEMI of Pos.Pos
| AND of Pos.Pos
| OR of Pos.Pos
| NOT of Pos.Pos
| IFF of Pos.Pos
| EOF of Pos.Pos
| EQDEF of Pos.Pos
| PUB of Pos.Pos
| PRIV of Pos.Pos
| KEYPAIR of Pos.Pos
| PREDICATE of Pos.Pos
| PRINCIPAL of Pos.Pos
| THEOREM of Pos.Pos
| QUOTE of Pos.Pos * string
| ID of Pos.Pos * string
| MINUS of Pos.Pos
| SELF of Pos.Pos

 (* %prefer SEMI *)

%left SEMI
%left COMMA
%left OR
%left AND
%right  IMPLIES ARROW DARROW
%nonassoc COLON
%nonassoc NOT

%nonterm protocol of ParseTree
| semi
| comma
| role of ParseTree
| declarations of ParseTree list
| declaration of ParseTree
| parameters of ParseTree list option
| sparameters of ParseTree list option
| roleexpr of ParseTree
| ids of ParseTree list
| params of ParseTree list
| param of ParseTree
| sparams of ParseTree list
| sparam of ParseTree
| mexpr of ParseTree
| mtuple of ParseTree list
| pactions of ParseTree list
| paction of ParseTree
| typeexpr of ParseTree
| predicates of ParseTree
| predicate of ParseTree
| theorems of ParseTree
| theorem of ParseTree
| formula of ParseTree
| atomicformula of ParseTree list
| label of ParseTree
| labeling of ParseTree
| strand of ParseTree
| strands of ParseTree list

%start protocol

%%

protocol	: declaration (declaration)
semi		: SEMI () | ()
comma		: COMMA () | ()
mexpr		: ID (Id(ID))
		| LP mtuple RP ((case mtuple of
                                    [x] => x
                                  | _ => MessageTuple(LP,mtuple)))
		| LCB mexpr RCB mexpr (Crypt (LCB, mexpr1, mexpr2))
		| KEYPAIR LP mexpr COMMA mexpr RP (KeyPair (KEYPAIR, (mexpr1, mexpr2)))
		| PUB mexpr (Pub (PUB, mexpr))
		| PRIV mexpr (Priv (PRIV, mexpr))
mtuple		: mexpr ([mexpr])
		| mexpr COMMA mtuple (mexpr::mtuple)
declarations	: declaration semi ([declaration])
                | declaration semi declarations (declaration::declarations)
declaration	: THEOREM ID (Id(ID))
                | PROTOCOL ID (Id(ID))
paction		: SEND mexpr (Send(SEND,mexpr))
		| RECEIVE mexpr (Receive(RECEIVE,mexpr))
		| ID COLON paction (Labeled(Id(ID), paction))
pactions	: paction semi ([paction])
                | paction semi pactions (paction::pactions)
roleexpr	: BEGIN pactions END (RoleBody(BEGIN,pactions))
role		: ROLE ID parameters EQ roleexpr
                  (Role(ROLE,
			{ name = Id(ID),
			  params = parameters,
			  body = roleexpr }))
predicate	: PREDICATE ID parameters EQ formula
                  (Predicate(PREDICATE,
			{ name = Id(ID),
			  params = parameters,
			  def = formula }))
theorem		: THEOREM ID parameters EQ formula
                  (Theorem(THEOREM,
			{ name = Id(ID),
			  params = parameters,
			  def = formula }))
formula		: atomicformula (Atomic(pos(List.hd(atomicformula)),atomicformula))
		| formula OR formula (Or(pos(formula1), formula1, formula2))
		| formula AND formula (And(pos(formula1), formula1, formula2))
		| formula IMPLIES formula (Implies(pos(formula1), formula1, formula2))
		| formula ARROW formula (Implies(pos(formula1), formula1, formula2))
		| formula DARROW formula (Implies(pos(formula1), formula1, formula2))
		| NOT formula (Not(NOT, formula))
                | LP formula RP (formula)
		| ID sparameters (FormulaInstance(pos(Id(ID)), Id(ID), sparameters))
atomicformula	: LCB strands RCB (strands)
                | strand ([strand])
strands		: strand ([strand])
		| strand COMMA strands (strand::strands)
strand		: ID sparameters (RoleStrand (pos(Id(ID)), Id(ID), sparameters))
		| ID sparameters DOT labeling (RoleLabel (pos(Id(ID)), Id(ID), sparameters, labeling))
labeling	: ID (Id (ID))
		| ID MINUS label (Range (pos(Id(ID)), Id(ID), label))
label		: ID (Id (ID))
typeexpr	: MESSAGE (MessageType(MESSAGE))
		| NONCE (NonceType(NONCE))
		| FRESHNONCE (FreshNonceType(FRESHNONCE))
		| PRINCIPAL (PrincipalType(PRINCIPAL))
		| SELF (SelfType(SELF))
sparameters	: (NONE)
		| LB sparams RB (SOME(sparams))
sparam		: ID (Id (ID))
sparams		: sparam ([sparam])
		| sparam COMMA sparams (sparam::sparams)
parameters	: (NONE)
		| LB params RB (SOME(params))
ids		: ID comma ([Id(ID)])
		| ID comma ids ((Id(ID))::ids)
param		: ids COLON typeexpr (Parameter(COLON, ids, typeexpr))
params		: param semi ([param])
		| param semi params (param::params)
