#include <stdlib.h>
#include <stdio.h>

#include "config.h"
#include "var.h"
#include "util.h"

/*------------------------------------------------------------------------*/

extern int verbose;

/*------------------------------------------------------------------------*/

static Variable * variables = 0, first_variable = 0, last_variable = 0;
static unsigned max_variables = 0, num_variables = 0, size_variables = 0;
static unsigned visited_variables = 0, lookups_variables = 0;

/*------------------------------------------------------------------------*/

static const char * gen_syn_prefix = "S";
static unsigned gen_syn_idx = 0;

/*------------------------------------------------------------------------*/

static unsigned hash_string(const char * name)
{
  const char * p;
  unsigned res, tmp;

  for(res = 0, p = name; *p; p++)
    {
      res <<= 4;
      res += (unsigned)*p;
      tmp = res & 0xf0000000;
      if(tmp) 
        {
	  res ^= tmp >> 28;
	  res ^= tmp;
	}
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static Variable * _find_Variable(const char * name)
{
  unsigned h;
  Variable * p;

  h = hash_string(name) % size_variables;

  lookups_variables++;

  for(p = &variables[h]; 
      *p && strcmp((*p) -> name, name) != 0; p = &(*p) -> next)
    visited_variables++;

  return p;
}

/*------------------------------------------------------------------------*/

Variable find_Variable(const char * name)
{
  return *_find_Variable(name);
}

/*------------------------------------------------------------------------*/

unsigned num_Variable()
{
  return num_variables;
}

/*------------------------------------------------------------------------*/

static void resize_Variable()
{
  unsigned old_size, i, h;
  Variable * old_variables, v, tmp;

  old_size = size_variables;
  old_variables = variables;

  size_variables = next_size(2 * old_size);
  variables = (Variable*) calloc(size_variables, sizeof(Variable));

  for(i = 0; i < old_size; i++)
    for(v = old_variables[i]; v; v = tmp)
      {
        h = hash_string(v -> name) % size_variables;
	tmp = v -> next;
	v -> next = variables[h];
	variables[h] = v;
      }
  
  free(old_variables);
}

/*------------------------------------------------------------------------*/

static Variable _new_Variable(const char * name, Variable * p)
{
  Variable res;

  assert(!*p);

  res = (Variable) malloc(sizeof(struct Variable_));
  res -> name = strcpy((char*) malloc(strlen(name) + 1), name);
  res -> size = 1;
  res -> next = 0;
  res -> next_chronological = 0;
  res -> cache.idx = 0;
  res -> cache.varsets = 0;
  res -> assignment = 0;
  res -> is_in_COI = 1;		/* as default print all variables */

  *p = res;			/* put it into hash table */

  /* and put it into chronlogical list of predicates 
   */
  if(last_variable)
    {
      last_variable -> next_chronological = res;
      last_variable = res;
    }
  else first_variable = last_variable = res;

  if(++num_variables >= 2 * size_variables) resize_Variable();
  if(num_variables >= max_variables) max_variables = num_variables;

  return res;
}

/*------------------------------------------------------------------------*/

Variable new_Variable(const char * name)
{
  Variable * p, res;

  p = _find_Variable(name);
  res = *p ? *p : _new_Variable(name, p);

  return res;
}

/*------------------------------------------------------------------------*/

void init_Variable()
{
  size_variables = next_size(100);
  variables = (Variable*) calloc(size_variables, sizeof(Variable));
  max_variables = num_variables = 0;
  visited_variables = lookups_variables = 0;
  first_variable = last_variable = 0;
}

/*------------------------------------------------------------------------*/

void exit_Variable()
{
  Variable v, tmp;
  double avg;

  for(v = first_variable; v; v = tmp)
    {
      tmp = v -> next_chronological;
      assert(!v -> assignment);
      free(v -> name);
      free(v);
    }

  free(variables);

  if(verbose)
    {
      if(lookups_variables)
	{
	  avg = ((double)visited_variables) / ((double)lookups_variables);
	  print_verbose(
	    "variables: num %u, max %u, size %u, avg %.2f\n",
	    num_variables, max_variables, size_variables, avg);
	}
      else
	{
	  print_verbose(
	    "variables: num %u, max %u, size %u\n",
	    num_variables, max_variables, size_variables);
	}
    }

  variables = 0;
  num_variables = size_variables = 0;
}

/*------------------------------------------------------------------------*/

void forall_Variable(void (*f)(Variable))
{
  Variable v, tmp;

  for(v = first_variable; v; v = tmp)
    {
      tmp = v -> next_chronological;
      f(v);
    }
}

/*------------------------------------------------------------------------*/

Variable generate_Symbol()
{
  char buffer[200];
  Variable * p, res;

  do {
    sprintf(buffer, "%s%u", gen_syn_prefix, gen_syn_idx);
    p = _find_Variable(buffer);
    gen_syn_idx++;
  } while(*p);

  res = _new_Variable(buffer, p);

  return res;
}

/*------------------------------------------------------------------------*/

void reset_idx_cache_Variable()
{
  Variable v;

  for(v = first_variable; v; v = v -> next_chronological) v -> cache.idx = 0;
}
/*------------------------------------------------------------------------*/

void reset_varsets_cache_Variable()
{
  Variable v;
  unsigned i;

  for(v = first_variable; v; v = v -> next_chronological)
    {
      if(v -> cache.varsets)
	{
	  for(i = 0; i < v -> size; i++)
	    {
	      if(v -> cache.varsets[i])
		free_VarSet(v -> cache.varsets[i]);
	    }
	  
	  free(v -> cache.varsets);
	  v -> cache.varsets = 0;
	}
    }
}
