#include <stdio.h>

#include "assign.h"
#include "config.h"
#include "list.h"
#include "pp.h"
#include "ctl.h"
#include "util.h"

/*------------------------------------------------------------------------*/

extern CTL init, trans, spec, invar;
extern List fairness;
extern unsigned max_on_line;
extern int flatten_flag;

/*------------------------------------------------------------------------*/

char * NOTEMPOPS = "no temporal operators in propositional fmt\n";

/*------------------------------------------------------------------------*/

static void _print_pp_CTL_base(CTL f, PP_Format fmt)
{
  if(f -> data.base.variable -> size > 1 &&
     f -> data.base.variable -> size != f -> data.base.idx)
    {
      switch(fmt)
	{
	  case SMV_PP_Format:
	    if(flatten_flag)
	      {
		fputs(f -> data.base.variable -> name, stdout);
		putc('[', stdout);
		putu(f -> data.base.idx, stdout);
		putc(']', stdout);
	      }
	    else
	      {
		printf("((%s / %u) mod 2 = 1)",
		  f -> data.base.variable -> name, pow2(f -> data.base.idx));
	      }
	    break;

	  case BMC_PP_Format:
	    printf("%s", f -> data.base.variable -> name);
	    printf("[%u]", f -> data.base.idx);
	    break;
	  
	  case PROVE_PP_Format:
	    printf("%s", f -> data.base.variable -> name);
	    printf("_%u", f -> data.base.idx);
	    break;
	  
	  default:
	    fatal(POSITION, "unsupported fmt\n");
	    break;
	}
    }
  else printf("%s", f -> data.base.variable -> name);
}

/*------------------------------------------------------------------------*/

static void _print_pp_cases_aux(
  char * str, CTL f, PP_Format fmt, unsigned indent)
{
  CTL g;

  while(f -> tag == CASE_Tag || f -> tag == SWITCH_Tag)
    {
      g = f -> data.arg[0];
      assert(g -> tag == COLON_Tag);

      fputs(str, stdout);
      tab(indent);
      _print_pp_CTL(str, g -> data.arg[0], fmt, 0, indent + 2, 1);
      fputs(" : ", stdout);
      _print_pp_CTL(str, g -> data.arg[1], fmt, 0, indent + 2, 1);
      fputs(";\n", stdout);

      f = f -> data.arg[1];
    }

  assert(f -> tag == COLON_Tag);

  fputs(str, stdout);
  tab(indent);
  _print_pp_CTL(str, f -> data.arg[0], fmt, 0, indent + 2, 1);
  fputs(" : ", stdout);
  _print_pp_CTL(str, f -> data.arg[1], fmt, 0, indent + 2, 1);
  fputs(";\n", stdout);
}

/*------------------------------------------------------------------------*/

static void _print_pp_cases(char * str, CTL f, PP_Format fmt, unsigned indent)
{
  putc('\n', stdout);
  fputs(str, stdout);
  tab(indent);
  fputs("case\n", stdout);
  _print_pp_cases_aux(str, f, fmt, indent + 2);
  fputs(str, stdout);
  tab(indent);
  fputs("esac", stdout);
}

/*------------------------------------------------------------------------*/

static void _print_pp_switches_aux(
  char * str, CTL f, PP_Format fmt, unsigned indent)
{
  CTL g, h;

  while(f -> tag == SWITCH_Tag)
    {
      g = f -> data.arg[0];
      assert(g -> tag == COLON_Tag);
      h = g -> data.arg[0];
      assert(h -> tag == EQUAL_Tag);
      assert(h -> data.arg[1] -> tag == NUMBER_Tag);

      fputs(str, stdout);
      tab(indent);
      printf("%u : ", h -> data.arg[1] -> data.number);
      _print_pp_CTL(str, g -> data.arg[1], fmt, 0, indent + 2, 1);
      fputs(";\n", stdout);

      f = f -> data.arg[1];
    }

  assert(f -> tag == COLON_Tag);
  h = f -> data.arg[0];
  assert(h -> tag == EQUAL_Tag);
  assert(h -> data.arg[1] -> tag == NUMBER_Tag);

  fputs(str, stdout);
  tab(indent);
  printf("%u : ", h -> data.arg[1] -> data.number);
  _print_pp_CTL(str, f -> data.arg[1], fmt, 0, indent + 2, 1);
  fputs(";\n", stdout);
}

/*------------------------------------------------------------------------*/

static void _print_pp_switches(
  char * str, CTL f, PP_Format fmt, unsigned indent)
{
  putc('\n', stdout);
  fputs(str, stdout);
  tab(indent);
  if(fmt == SMV_PP_Format)
    {
      fputs("case\n", stdout);
      _print_pp_cases_aux(str, f, fmt, indent + 2);
      fputs(str, stdout);
      tab(indent);
      fputs("esac", stdout);
    }
  else
    {
      fputs("switch(", stdout);
      _print_pp_CTL_base(f -> data.arg[0] -> data.arg[0] -> data.arg[0], fmt);
      fputs("){\n", stdout);
      _print_pp_switches_aux(str, f, fmt, indent + 2);
      fputs(str, stdout);
      tab(indent);
      fputs("}", stdout);
    }
}

/*------------------------------------------------------------------------*/

void _print_pp_CTL(
  char * str,
  CTL f,
  PP_Format fmt,
  unsigned out_prior,
  unsigned indent,
  int one_line)
{
  unsigned in_prior, in_ind;
  int print_parentheses;

  if(!one_line && f -> size <= max_on_line)
    {
      printf("%s", str);
      tab(indent);
      _print_pp_CTL(str, f, fmt, out_prior, 0, 1);
      printf("\n");
      return;
    }

  in_prior = priority_CTL(f);

  print_parentheses = in_prior < out_prior || (in_prior == out_prior &&
     (f -> tag == IMPLIES_Tag ||
      f -> tag == EQUAL_Tag ||
      f -> tag == EQUIV_Tag));

  if(print_parentheses)
    {
      if(one_line) printf("(");
      else
        {
	  printf("%s", str);
          tab(indent);
          printf("(\n");
        }
      in_ind = indent + 2;
    }
  else in_ind = indent;

  switch(f -> tag)
    {
      case NUMBER_Tag:

	if(fmt == PROVE_PP_Format) fatal(POSITION, "unsupported fmt\n");
	if(!one_line)
	  {
	    printf("%s", str);
	    tab(in_ind);
	  }
	printf("%u", f -> data.number);
	if(!one_line) printf("\n");
	break;
      
      case CONSTANT_Tag:

	if(!one_line)
	  {
	    printf("%s", str);
	    tab(in_ind);
	  }
        switch(fmt)
	  {
	    case PROVE_PP_Format:
	      fputs(f -> data.number ? "(a # ~a)" : "(a & ~a)", stdout);
	      break;

	    default:
	      putc(f -> data.number ? '1' : '0', stdout);
	      break;
	  }
	if(!one_line) printf("\n");
	break;
      

      case CURRENT_Tag:

	if(!one_line)
	  {
	    printf("%s", str);
	    tab(in_ind);
	  }
	_print_pp_CTL_base(f, fmt);
	if(!one_line) printf("\n");
        break;
      
      case NEXT_Tag:

	if(!one_line)
	  {
	    printf("%s", str);
	    tab(in_ind);
	  }
	switch(fmt)
	  {
	    case PROVE_PP_Format:
	      fatal(POSITION,
		"PROVE fmt does not support next state variables\n");
	      break;

	    case BMC_PP_Format:
	    case SMV_PP_Format:
	      printf("next(");
	      _print_pp_CTL_base(f, fmt);
	      printf(")");
	      if(!one_line) printf("\n");
	      break;

	    default:
	      fatal(POSITION, "unsupported fmt\n");
	      break;
	  }
        break;

      case INC_Tag:
	switch(fmt)
	  {
	    case BMC_PP_Format:
	      if(!one_line)
		{
		  printf("%s", str);
		  tab(in_ind);
		}
	      printf("inc(");
	      _print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, 1);
	      printf(")");
	      if(!one_line) printf("\n");
	      break;
	    
	    case SMV_PP_Format:
	      if(!one_line)
		{
		  printf("%s", str);
		  tab(in_ind);
		}
	      printf("((");
	      _print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, 1);
	      printf(" + 1) mod %u)", pow2(width_CTL(f)));
	      if(!one_line) printf("\n");
	      break;
	    
	    default:
	      fatal(POSITION, "unsupported fmt\n");
	      break;
	  }
	break;
      
      case ADD_Tag:
      case SUB_Tag:

	switch(fmt)
	  {
	    case BMC_PP_Format:

	      if(!one_line)
		{
		  fputs(str, stdout);
		  tab(in_ind);
		}
	      if(f -> tag == ADD_Tag) fputs("add(", stdout);
	      else fputs("sub(", stdout);
	      _print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, 1);
	      fputs(", ", stdout);
	      _print_pp_CTL(str, f -> data.arg[1], fmt, in_prior, in_ind, 1);
	      putc(')', stdout);
	      if(!one_line) printf("\n");
	      break;
	    
	    case SMV_PP_Format:

	      if(!one_line)
		{
		  printf("%s", str);
		  tab(in_ind);
		}
	      printf("((");
	      _print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, 1);
	      if(f -> tag == ADD_Tag) fputs(" + ", stdout);
	      else fputs(" - ", stdout);
	      _print_pp_CTL(str, f -> data.arg[1], fmt, in_prior, in_ind, 1);
	      printf(") mod %u)", pow2(width_CTL(f)));
	      if(!one_line) printf("\n");
	      break;
	    
	    default:
	      fatal(POSITION, "unsupported fmt\n");
	      break;
	  }
	break;
      

      case DEC_Tag:

	switch(fmt)
	  {
	    case BMC_PP_Format:

	      if(!one_line)
		{
		  printf("%s", str);
		  tab(in_ind);
		}
	      printf("dec(");
	      _print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, 1);
	      printf(")");
	      if(!one_line) printf("\n");
	      break;
	    
	    case SMV_PP_Format:
	       
	      if(!one_line)
		{
		  printf("%s", str);
		  tab(in_ind);
		}
	      printf("((");
	      _print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, 1);
	      printf(" - 1) mod %u)", pow2(width_CTL(f)));
	      if(!one_line) printf("\n");
	      break;
	    
	    default:
	      fatal(POSITION, "unsupported fmt\n");
	      break;
	  }
	break;

      case EQUAL_Tag:
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, 1);
	printf(" = ");
	_print_pp_CTL(str, f -> data.arg[1], fmt, in_prior, in_ind, 1);
        if(!one_line) printf("\n");
        break;

      case EQUIV_Tag:
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
	if(one_line) printf(" <-> ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("<->\n");
	  }
	_print_pp_CTL(str, f -> data.arg[1], fmt, in_prior, in_ind, one_line);
        break;

      case IMPLIES_Tag:
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
	if(one_line) printf(" -> ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("->\n");
	  }
	_print_pp_CTL(str, f -> data.arg[1], fmt, in_prior, in_ind, one_line);
        break;

      case OR_Tag:
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
	if(!one_line)
	  {
	    printf("%s", str);
	    tab(in_ind);
	  }
	switch(fmt)
	  {
	    case SMV_PP_Format:
	    case BMC_PP_Format:
	      if(one_line) printf(" | "); else printf("|\n");
	      break;

	    case PROVE_PP_Format:
	      if(one_line) printf(" # "); else printf("#\n");
	      break;

	    default:
	      fatal(POSITION, "unsupported fmt\n");
	      break;
	  }
	_print_pp_CTL(str, f -> data.arg[1], fmt, in_prior, in_ind, one_line);
        break;

      case AND_Tag:
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
	if(one_line) printf(" & ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("&\n");
	  }
	_print_pp_CTL(str, f -> data.arg[1], fmt, in_prior, in_ind, one_line);
	break;
      
      case EX_Tag:
	if(is_propositional_PP_Format(fmt)) fatal(POSITION, NOTEMPOPS);
	if(one_line) printf("EX ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("EX\n");
	  }
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
        break;

      case EG_Tag:
	if(is_propositional_PP_Format(fmt)) fatal(POSITION, NOTEMPOPS);
	if(one_line) printf("EG ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("EG\n");
	  }
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
        break;

      case EF_Tag:
	if(is_propositional_PP_Format(fmt)) fatal(POSITION, NOTEMPOPS);
	if(one_line) printf("EF ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("EF\n");
	  }
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
        break;

      case AX_Tag:
	if(is_propositional_PP_Format(fmt)) fatal(POSITION, NOTEMPOPS);
	if(one_line) printf("AX ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("AX\n");
	  }
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
        break;

      case AG_Tag:
	if(is_propositional_PP_Format(fmt)) fatal(POSITION, NOTEMPOPS);
	if(one_line) printf("AG ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("AG\n");
	  }
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
        break;

      case AF_Tag:
	if(is_propositional_PP_Format(fmt)) fatal(POSITION, NOTEMPOPS);
	if(one_line) printf("AF ");
	else
	  {
	    printf("%s", str);
	    tab(in_ind);
	    printf("AF\n");
	  }
	_print_pp_CTL(str, f -> data.arg[0], fmt, in_prior, in_ind, one_line);
        break;

      case NOT_Tag:

	if(fmt == BMC_PP_Format && f -> data.arg[0] -> tag == EQUAL_Tag)
	  {
	    _print_pp_CTL(
	      str, f -> data.arg[0] -> data.arg[0], fmt, in_prior, in_ind, 1);

	    printf(" != ");

	    _print_pp_CTL(
	      str, f -> data.arg[0] -> data.arg[1], fmt, in_prior, in_ind, 1);

	    if(!one_line) printf("\n");
	  }
	else
	  {
	    if(!one_line)
	      {
		printf("%s", str);
		tab(in_ind);
	      }
	    switch(fmt)
	      {
		case SMV_PP_Format:
		case BMC_PP_Format:
		  if(one_line) printf("!"); else printf("!\n");
		  break;

		case PROVE_PP_Format:
		  if(one_line) printf("~"); else printf("~\n");
		  break;

		default:
		  fatal(POSITION, "unsupported fmt\n");
		  break;
	      }
	    _print_pp_CTL(str,
	      f -> data.arg[0], fmt, in_prior, in_ind, one_line);
	  }
	break;

      case CASE_Tag:
        if(fmt != BMC_PP_Format && fmt != SMV_PP_Format)
	  fatal(POSITION, "`case' statements not supported for this format\n");
        _print_pp_cases(str, f, fmt, indent);
	break;

      case SWITCH_Tag:
        if(fmt != BMC_PP_Format && fmt != SMV_PP_Format)
	  fatal(POSITION,
	    "`switch' statements not supported for this format\n");
        _print_pp_switches(str, f, fmt, indent);
	break;

      default:
	fatal(POSITION, "non valid tag\n");
    }

  if(print_parentheses)
    {
      if(one_line) printf(")");
      else
	{
	  printf("%s", str);
	  tab(indent);
	  printf(")\n");
	}
    }
}

/*------------------------------------------------------------------------*/

void print_CTL(CTL f)
{
  _print_pp_CTL("", f, BMC_PP_Format, 0, 2, 0);
}

/*------------------------------------------------------------------------*/

void printnl_CTL(CTL f)
{
  print_CTL(f);
  printf("\n");
}

/*------------------------------------------------------------------------*/

static unsigned count_pp_CTL_arrays_Variable_counter = 0;

/*------------------------------------------------------------------------*/

static void count_pp_CTL_arrays_Variable(Variable p)
{
  if(p -> size > 1 && p -> is_in_COI)
    count_pp_CTL_arrays_Variable_counter++;
}

/*------------------------------------------------------------------------*/

static unsigned count_pp_CTL_arrays()
{
  unsigned  res;

  forall_Variable(count_pp_CTL_arrays_Variable);
  res = count_pp_CTL_arrays_Variable_counter;
  count_pp_CTL_arrays_Variable_counter = 0;

  return res;
}

/*------------------------------------------------------------------------*/

static void print_pp_CTL_arraydecl(Variable p)
{
  if(p -> size > 1 && p -> is_in_COI)
    printf("  %s[%u]\n", p -> name, p -> size);
}

/*------------------------------------------------------------------------*/

static void print_pp_CTL_vars()
{
  forall_Variable(print_pp_CTL_arraydecl);
}

/*------------------------------------------------------------------------*/

static PP_Format fmt_for_functor;

/*------------------------------------------------------------------------*/

static void print_pp_define_assignment(Assignment a)
{
  int print_nl;
  unsigned i;
  Variable v;

  v = a -> variable;
  print_nl = 0;

  if(a -> define)
    {
      print_nl = 1;
      printf("  %s := ", v -> name);
      _print_pp_CTL("", a -> define, fmt_for_functor, 0, 4, 1);
      fputs(";\n", stdout);
    }
  
  if(a -> defines)
    {
      for(i = 0; i < v -> size; i++)
	{
	  if(a -> defines[i])
	    {
	      print_nl = 1;
	      printf("  %s[%u] := ", v -> name, i);
	      _print_pp_CTL("", a -> defines[i], fmt_for_functor, 0, 4, 1);
	      fputs(";\n", stdout);
	    }
	}
    }

  if(print_nl) putc('\n', stdout);
}

/*------------------------------------------------------------------------*/

static void print_pp_next_init_assignment(Assignment a)
{
  int print_nl;
  unsigned i;
  Variable v;

  v = a -> variable;
  print_nl = 0;

  if(a -> init)
    {
      print_nl = 1;
      printf("  init(%s) := ", v -> name);
      _print_pp_CTL("", a -> init, fmt_for_functor, 0, 4, 1);
      fputs(";\n", stdout);
    }
  
  if(a -> inits)
    {
      for(i = 0; i < v -> size; i++)
	{
	  if(a -> inits[i])
	    {
	      print_nl = 1;
	      printf("  init(%s[%u]) := ", v -> name, i);
	      _print_pp_CTL("", a -> inits[i], fmt_for_functor, 0, 4, 1);
	      fputs(";\n", stdout);
	    }
	}
    }

  if(print_nl) putc('\n', stdout);

  print_nl = 0;
  
  if(a -> next)
    {
      print_nl = 1;
      printf("  next(%s) := ", v -> name);
      _print_pp_CTL("", a -> next, fmt_for_functor, 0, 4, 1);
      fputs(";\n", stdout);
    }
 
  if(a -> nexts)
    {
      for(i = 0; i < v -> size; i++)
	{
	  if(a -> nexts[i])
	    {
	      print_nl = 1;
	      printf("  next(%s[%u]) := ", v -> name, i);
	      _print_pp_CTL("", a -> nexts[i], fmt_for_functor, 0, 4, 1);
	      fputs(";\n", stdout);
	    }
	}
    }

  if(print_nl) putc('\n', stdout);
}

/*------------------------------------------------------------------------*/

void print_pp_next_init_assignments(PP_Format fmt)
{
  if(fmt != SMV_PP_Format && fmt != BMC_PP_Format)
    fatal(POSITION, "non valid format\n");

  fmt_for_functor = fmt;
  forall_Assignment(print_pp_next_init_assignment);
}

/*------------------------------------------------------------------------*/

void print_pp_define_assignments(PP_Format fmt)
{
  if(fmt != SMV_PP_Format && fmt != BMC_PP_Format)
    fatal(POSITION, "non valid format\n");

  fmt_for_functor = fmt;
  forall_Assignment(print_pp_define_assignment);
}

/*------------------------------------------------------------------------*/

static void print_pp_fairness(void * f)
{
  CTL g;

  g = (CTL) f;
  printf("FAIRNESS\n\n");
  printnl_CTL(g);
}

/*------------------------------------------------------------------------*/

void print_pp()
{
  start_verbose("pretty printing");

  if(count_pp_CTL_arrays() > 0)
    {
      printf("VAR\n\n");
      print_pp_CTL_vars();
      printf("\n");
    }
  
  if(num_init_assignments + num_next_assignments)
    {
      printf("ASSIGN\n\n");
      print_pp_next_init_assignments(BMC_PP_Format);
    }

  if(num_define_assignments)
    {
      printf("DEFINE\n\n");
      print_pp_define_assignments(BMC_PP_Format);
    }

  if(!is_true_CTL(init))
    {
      printf("INIT\n\n");
      printnl_CTL(init);
    }

  if(!is_true_CTL(trans))
    {
      printf("TRANS\n\n");
      printnl_CTL(trans);
    }
  
  if(!is_true_CTL(invar))
    {
      printf("INVAR\n\n");
      printnl_CTL(invar);
    }
  
  if(fairness)
    {
      forall_List(fairness, print_pp_fairness);
    }

  if(!is_true_CTL(spec))
    {
      printf("SPEC\n\n");
      printnl_CTL(spec);
    }

  end_verbose("pretty printing");
}
