%{
#include <stdio.h>
#include <stdarg.h>

#include "assign.h"
#include "config.h"
#include "ctl.h"
#include "list.h"
#include "util.h"
#include "var.h"

/*------------------------------------------------------------------------*/

extern CTL init, trans, invar, spec;
extern List fairness;
extern char * prefix;
extern int only_propositional;

/*------------------------------------------------------------------------*/

int yyerror(char*, ...);
void yymessage(char*);
int yylex();

/*------------------------------------------------------------------------*/

static void check_var(Variable p)
{
  char * s, * t;

  for(s = p -> name, t = prefix; *s == *t; s++, t++)
    ;
  
  if(!*s) return;	/* name is shorter or same length than the prefix */
  if(*t) return;	/* prefix is no prefix of the name */

  if(*s < '0' || *s > '9') return;	/* prefix is followed by at least
  				         * one digit */
  
  s++;
  while('0' <= *s && *s <= '9') s++;	/* skip other numbers */

  if(*s == '_' && s[1])
    yyerror("`%s' collides with the prefix `%s'", p -> name, prefix);
}

/*------------------------------------------------------------------------*/
/* returns number of cases 
 */

static unsigned check_cases(CTL f)
{
  unsigned i;
  CTL g;

  for(i=0; f -> tag == CASE_Tag; i++)
    {
      g = f -> data.arg[0];
      assert(g -> tag == COLON_Tag);
      f = f -> data.arg[1];
    }
  
  assert(f -> tag == COLON_Tag);

  if(!is_true_CTL(f -> data.arg[0]))
    yyerror("a `case' statement must end with a `1' guard");
  
  return i + 1;
}

/*------------------------------------------------------------------------*/

static void check_switches(Variable v, CTL f)
{
  CTL g, h;
  unsigned i, n, m;
  int found;

  for(g = f; g -> tag == SWITCH_Tag; g = g -> data.arg[1])
    {
      assert(g -> data.arg[0] -> tag == COLON_Tag);
      assert(g -> data.arg[0] -> data.arg[0] -> tag == NUMBER_Tag);

      n = g -> data.arg[0] -> data.arg[0] -> data.number;

      for(h = g -> data.arg[1]; h -> tag == SWITCH_Tag; h = h -> data.arg[1])
        {
	  assert(h -> data.arg[0] -> tag == COLON_Tag);
	  assert(h -> data.arg[0] -> data.arg[0] -> tag == NUMBER_Tag);

	  m = h -> data.arg[0] -> data.arg[0] -> data.number;

	  if(n == m) yyerror(
	    "number `%u' guard in switch(%s){..} occured twice",
	    n, v -> name);
	}

      assert(h -> tag == COLON_Tag);
      assert(h -> data.arg[0] -> tag == NUMBER_Tag);

      m = h -> data.arg[0] -> data.number;

      if(n == m) yyerror(
	"number `%u' in guard for switch(%s){..} occured twice",
	n, v -> name);
    }
  
  if(v -> size > 31) yyerror(
    "switch statement for variable of width > 31 not supported");
  
  m = pow2(v -> size);
  for(i = 0; i < m; i++)
    {
      found = 0;
      for(g = f; !found && g -> tag == SWITCH_Tag; g = g -> data.arg[1])
        {
	  assert(g -> data.arg[0] -> tag == COLON_Tag);
	  assert(g -> data.arg[0] -> data.arg[0] -> tag == NUMBER_Tag);

	  n = g -> data.arg[0] -> data.arg[0] -> data.number;
	  if(i == n) found = 1;
	}
      
      if(!found)
        {
	  assert(g -> tag == COLON_Tag);
	  assert(g -> data.arg[0] -> tag == NUMBER_Tag);

	  n = g -> data.arg[0] -> data.number;
	  if(i != n) yyerror(
	    "no guard with number `%u' in switch(%s){..} found",
	    i, v -> name);
	}
    }
}

/*------------------------------------------------------------------------*/

static CTL gen_switches(Variable v, CTL f)
{
  CTL tmp, res, l, r;
  
  if(f -> tag == SWITCH_Tag)
    {
      res = gen_switches(v, f -> data.arg[1]);

      assert(f -> data.arg[0] -> tag == COLON_Tag);
      assert(f -> data.arg[0] -> data.arg[0] -> tag == NUMBER_Tag);

      l = binary_CTL(
            EQUAL_Tag,
	    current_CTL(v, v -> size),
	    copy_CTL(f -> data.arg[0] -> data.arg[0]));
      
      r = copy_CTL(f -> data.arg[0] -> data.arg[1]);

      tmp = binary_CTL(COLON_Tag, l, r);
      res = binary_CTL(SWITCH_Tag, tmp, res);
    }
  else
    {
      assert(f -> tag == COLON_Tag);
      assert(f -> data.arg[0] -> tag == NUMBER_Tag);

      l = binary_CTL(
            EQUAL_Tag,
	    current_CTL(v, v -> size),
            copy_CTL(f -> data.arg[0]));
      
      r = copy_CTL(f -> data.arg[1]);
      res = binary_CTL(COLON_Tag, l, r);
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static void check_assignment(Variable v, unsigned idx, CTL f)
{
  check_var(v); 

  if(v -> size == idx)
    {
      if(!check_width_CTL(v -> size, f))
	yyerror("right hand side of assignment for `%s' has incorrect width",
	  v -> name);
    }
  else
    {
      if(!check_width_CTL(1, f))
        yyerror(
	  "right hand side of assignment for `%s[%u]' has incorrect width",
	  v -> name, idx);
    }
}

/*------------------------------------------------------------------------*/

static CTL gen_equal(CTL f, CTL g)
{
  CTL res;
  unsigned f_width, g_width;

  f_width = contains_var_with_width(f);
  g_width = contains_var_with_width(g);
  
  if(!f_width && g_width)
    {
      if(!check_width_CTL(g_width, f))
        {
	  yyerror("left hand side does not have correct width");
	}

      if(!check_width_CTL(g_width, g))
        {
	  yyerror("right hand side contains variables with different width");
	}
    }
  else
  if(f_width && !g_width)
    {
      if(!check_width_CTL(f_width, g))
        {
	  yyerror("right hand side does not have correct width");
	}

      if(!check_width_CTL(f_width, f))
        {
	  yyerror("left hand side contains variables with different width");
	}
    }
  else
  if(f_width && g_width)
    {
      if(f_width != g_width)
	{
	  yyerror("types of arguments of `=' do not match");
	}
    }

  res = binary_CTL(EQUAL_Tag, f, g);

  return res;
}

/*------------------------------------------------------------------------*/

%}

%union {
  Variable variable;
  char * str;
  CTL formula;
  unsigned number;
}

%token AND_Token OR_Token NOT_Token EQUIV_Token IMPLIES_Token EQUAL_Token
%token NOTEQUAL_Token

%token AG_Token AX_Token AF_Token

%token INIT_Token TRANS_Token SPEC_Token VAR_Token DEFINE_Token ASSIGN_Token
%token INVAR_Token FAIRNESS_Token MODULE_Token ARRAY_Token OF_Token
%token DOTDOT_Token BOOLEAN_Token

%token NEXT_Token INC_Token DEC_Token CASE_Token ESAC_Token BECOMES_Token
%token ADD_Token SUB_Token SWITCH_Token init_Token PLUS_Token

%token STRING_Token
%token NUMBER_Token

%type <formula> formula equiv ors ands unary basic cases term case
%type <formula> switches switch
%type <formula> assignment_var
%type <number> NUMBER_Token
%type <number> assignment_head
%type <str> STRING_Token
%type <variable> variable_name

%left PLUS_Token

%%

start
:
modules
|
formula
{
  spec = $1;
  if(!spec -> is_propositional)
    yyerror("expected a single propositional formula");
  only_propositional = 1;
}
;

modules
:
module
;

module
:
module_head sections
;

module_head
:
/* epsilon */
|
MODULE_Token STRING_Token
{
  if(strcmp($2, "main") != 0) 
    yyerror("no other modules than `main' supported yet\n");
  
  free($2);
}
;

sections
:
/* epsilon */
|
section sections
;

section
:
VAR_Token vardecls
|
INIT_Token formula
{
  if(!$2 -> is_current) 
    yyerror("no `next' operator allowed in INIT predicate");

  init = binary_CTL(AND_Tag, init, $2);
}
|
TRANS_Token formula
{
  trans = binary_CTL(AND_Tag, trans, $2);
}
|
SPEC_Token formula
{
  spec = binary_CTL(AND_Tag, spec, $2);
}
|
ASSIGN_Token assignments
|
DEFINE_Token defines
|
INVAR_Token formula
{
  if(!$2 -> is_current) yyerror("can not use `next' in INVAR section");
  if(!$2 -> is_propositional) yyerror("INVAR section has to be propositional");

  invar = binary_CTL(AND_Tag, invar, $2);
}
|
FAIRNESS_Token formula
{
  if(!$2 -> is_current) yyerror("can not use `next' in FAIRNESS section");
  if(!$2 -> is_propositional) yyerror(
    "FAIRNESS constraint must be propositional");
  
  fairness = cons($2, fairness);
}
;

vardecls
:
vardecl
|
vardecl vardecls
;

vardecl
:
variable_name '[' NUMBER_Token ']'
{
  check_var($1); 
  if($3 == 0)
    yyerror("array `%s' must have length greater than 0", $1 -> name);

  $1 -> size = $3;
}
|
variable_name
':'
  ARRAY_Token 
    NUMBER_Token DOTDOT_Token NUMBER_Token
  OF_Token
    BOOLEAN_Token
';'
{
  check_var($1); 
  if($4 != 0) yyerror("array has to start at `0'\n");
  if($6 == 0) yyerror("array must end with number > `0'\n");
  $1 -> size = $6 - $4 + 1;
}
|
variable_name
':'
BOOLEAN_Token
';'
{
  check_var($1); 
  if($1 -> size != 1) yyerror("variable `%s' declared twice\n", $1 -> name);
  $1 -> size = 1;
}
;

assignments
:
assignment
|
assignment assignments
;

assignment
:
assignment_head  '(' assignment_var ')' BECOMES_Token formula ';'
{
  Assignment_Type type;
  Assignment a;
  Variable v;
  unsigned idx;

  v = $3 -> data.base.variable;
  idx = $3 -> data.base.idx;
  free_CTL($3);

  if(!$6 -> is_current) 
    yyerror("can not use `next' in right hand side of `:='");
  
  check_assignment(v, idx, $6);

  a = find_Assignment(v);
  if(a)
    {
      if(a -> define || a -> defines) yyerror(
	"`%s' already has an `define' assignment",
	v -> name);

      if(idx == v -> size)
        {
	  if($1 == (unsigned) init_Token)
	    {
	      if(a -> init || a -> inits)
	        yyerror("`init(%s)' assigned twice", v -> name);
	    }

	  if($1 == (unsigned) NEXT_Token)
	    {
	      if(a -> next || a -> nexts)
	        yyerror("`next(%s)' assigned twice", v -> name);
	    }
	}
      else
        {
	  if($1 == (unsigned) init_Token)
	    {
	      if(a -> init) yyerror("`init(%s)' already assigned", v -> name);
	      if(a -> inits && (a -> inits)[idx])
	        yyerror("`init(%s[%u])' already assigned", v -> name, idx);
	    }

	  if($1 == (unsigned) NEXT_Token)
	    {
	      if(a -> next) yyerror("`next(%s)' already assigned", v -> name);
	      if(a -> nexts && (a -> nexts)[idx])
	        yyerror("`next(%s[%u])' already assigned", v -> name, idx);
	    }
	}
    }

  type = ($1 == (unsigned) init_Token) ? INIT_Assignment : NEXT_Assignment;
  insert_Assignment(type, v, idx, $6);
}
;

assignment_var
:
variable_name
{
  if($1 -> size > 1) $$ = current_CTL($1, $1 -> size);
  else $$ = current_CTL($1, 0);
}
|
variable_name '[' NUMBER_Token ']'
{
  if($1 -> size <= $3)
    {
      yyerror("index `%u' is out of range for `%s'", $3, $1 -> name);
    }
  else $$ = current_CTL($1, $3);
}
;

assignment_head
:
init_Token
{
  $$ = (unsigned) init_Token;
}
|
NEXT_Token
{
  $$ = (unsigned) NEXT_Token;
}
;

defines
:
define
|
define defines
;

define
:
variable_name BECOMES_Token formula ';'
{
  unsigned idx;
  Assignment a;
  
  a = find_Assignment($1);
  if(a)
    {
      if(a -> init || a -> inits || a -> next || a -> nexts) yyerror(
        "`%s' already has an `init' or `next' assignment",
	$1 -> name);
    }

  if($1 -> size > 1) idx = $1 -> size;
  else idx = 0;

  check_assignment($1, idx, $3);
  insert_Assignment(DEFINE_Assignment, $1, idx, $3);
}
|
variable_name '[' NUMBER_Token ']' BECOMES_Token formula ';'
{
  unsigned idx;
  Assignment a;
  Variable v;
  CTL formula;

  v = $1;
  idx = $3;
  formula = $6;

  if(v -> size <= idx) 
    yyerror("index `%u' for `%s' is out of range", idx, v -> name);
  
  a = find_Assignment(v);

  if(a)
    {
      if(a -> init || a -> inits || a -> next || a -> nexts) yyerror(
        "`%s' already has an `init' or `next' assignment",
	v -> name);

      if(a -> define) yyerror("`%s' already defined", v -> name);
      if(a -> defines && (a -> defines)[idx])
	yyerror("`%s[%u]' already defined", v -> name, idx);
    }

  check_assignment(v, idx, formula);
  insert_Assignment(DEFINE_Assignment, v, idx, formula);
}
;

formula
:
equiv
;

equiv
:
ors EQUIV_Token ors
{
  $$ = binary_CTL(EQUIV_Tag, $1, $3);
}
|
ors IMPLIES_Token ors
{
  $$ = binary_CTL(IMPLIES_Tag, $1, $3);
}
|
ors
;

ors
:
ands OR_Token ors
{
  $$ = binary_CTL(OR_Tag, $1, $3);
}
|
ands
;

ands
:
unary AND_Token ands
{
  $$ = binary_CTL(AND_Tag, $1, $3);
}
|
unary
;

unary
:
NOT_Token unary
{
  $$ = unary_CTL(NOT_Tag, $2);
}
|
AX_Token unary
{
  $$ = unary_CTL(AX_Tag, $2);
}
|
AF_Token unary
{
  $$ = unary_CTL(AF_Tag, $2);
}
|
AG_Token unary
{
  $$ = unary_CTL(AG_Tag, $2);
}
|
basic
;

basic
:
term
|
term EQUAL_Token term
{
  $$ = gen_equal($1, $3);
}
|
term NOTEQUAL_Token term
{
  $$ = gen_equal($1, $3);
  $$ = unary_CTL(NOT_Tag, $$);
}
|
'(' formula ')'
{
  $$ = $2;
}
;

term
:
variable_name
{
  check_var($1); 

  if($1 -> size > 1) $$ = current_CTL($1, $1 -> size);
  else $$ = current_CTL($1, 0);
}
|
NEXT_Token '(' variable_name ')'
{
  check_var($3); 

  if($3 -> size > 1) $$ = next_CTL($3, $3 -> size);
  else $$ = next_CTL($3, 0);
}
|
variable_name '[' NUMBER_Token ']'
{
  check_var($1); 

  if($3 >= $1 -> size)
    yyerror("index `%u' is out of range for `%s'", $3, $1 -> name);
  
  $$ = current_CTL($1, $3);
}
|
NEXT_Token '(' variable_name '[' NUMBER_Token ']' ')'
{
  check_var($3); 

  if($5 >= $3 -> size)
    yyerror("index `%u' is out of range for `%s'", $5, $3 -> name);
  
  $$ = next_CTL($3, $5);
}
|
INC_Token '(' term ')'
{
  $$ = unary_CTL(INC_Tag, $3);
}
|
DEC_Token '(' term ')'
{
  $$ = unary_CTL(DEC_Tag, $3);
}
|
term PLUS_Token term
{
  $$ = binary_CTL(ADD_Tag, $1, $3);
}
|
ADD_Token '(' term ',' term ')'
{
  $$ = binary_CTL(ADD_Tag, $3, $5);
}
|
SUB_Token '(' term ',' term ')'
{
  $$ = binary_CTL(SUB_Tag, $3, $5);
}
|
NUMBER_Token
{
  $$ = number_CTL($1);
}
|
CASE_Token cases ESAC_Token
{
  if(check_cases($2) == 1)		/* only one case */
    {
      assert($2 -> tag == COLON_Tag);
      assert(is_true_CTL($2 -> data.arg[0]));
      $$ = copy_CTL($2 -> data.arg[1]);
      free_CTL($2);
    }
  else $$ = $2;
}
|
SWITCH_Token '(' variable_name ')' '{' switches '}'
{
  check_switches($3, $6);
  $$ = gen_switches($3, $6);
  free_CTL($6);
}
;

variable_name
:
STRING_Token
{
  $$ = new_Variable($1);
  free($1);
}
;

cases
:
case ';'
{
  $$ = $1;
}
|
case ';' cases
{
  $$ = binary_CTL(CASE_Tag, $1, $3);
}
;

case
:
formula ':' term
{
  $$ = binary_CTL(COLON_Tag, $1, $3);
}
;

switches
:
switch ';'
{
  $$ = $1;
}
|
switch ';' switches 
{
  $$ = binary_CTL(SWITCH_Tag, $1, $3);
}
;

switch
:
NUMBER_Token ':' term
{
  $$ = binary_CTL(COLON_Tag, number_CTL($1), $3);
}
;

%%

extern int yylineno;
extern char yytext[];

int yyerror(char * fmt, ...)
{
  va_list ap;

  fprintf(stderr, "*** bmc: line %d: ", yylineno); 
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
  fprintf(stderr, " before `%s'\n", yytext);
  fflush(stderr);
  exit(1);
}
