#include <stdio.h>
#include <stdlib.h>

#include "graph.h"
#include "util.h"

/*------------------------------------------------------------------------*/

Graph new_Graph()
{
  Graph res;

  res = (Graph) malloc(sizeof(struct Graph_));
  res -> size_nodes = next_size(100);

  res -> count_nodes = 0;
  res -> max_nodes = 0;
  res -> lookup_nodes = 0;
  res -> visited_nodes = 0;
  res -> resize_nodes = 0;

  res -> count_edges = 0;
  res -> size_edges = 0;
  res -> max_edges = 0;
  res -> lookup_edges = 0;
  res -> visited_edges = 0;
  res -> resize_edges = 0;

  res -> nodes = (Node*) calloc(res -> size_nodes, sizeof(Node));

  return res;
}

/*------------------------------------------------------------------------*/

static void free_Node(Node node)
{
  Edge p, tmp;
  unsigned i;

  for(i = 0; i < node -> size; i++)
    {
      for(p = node -> edges[i]; p; p = tmp)
        {
	  tmp = p -> next;
	  free(p);
	}
    }
  
  free(node -> edges);
  free(node);
}

/*------------------------------------------------------------------------*/

void free_Graph(Graph graph)
{
  Node p, tmp;
  unsigned i;
  double avg;

  for(i = 0; i < graph -> size_nodes; i++)
    {
      for(p = graph -> nodes[i]; p; p = tmp)
        {
	  tmp = p -> next;
	  free_Node(p);
	}
    }
  
  if(graph -> lookup_nodes)
    {
      avg = ((double)graph->visited_nodes) / ((double)graph->lookup_nodes);
      print_verbose(
	"Nodes: size %u, max %u, lookups %u, avg %.2f, resized %u\n",
        graph -> size_nodes, graph -> max_nodes, graph -> lookup_nodes,
	avg, graph -> resize_nodes);
    }
  
  if(graph -> lookup_edges)
    {
      avg = ((double)graph->visited_edges) / ((double)graph->lookup_edges);
      print_verbose(
	"Edges: size %u, max %u, lookups %u, avg %.2f, resized %u\n",
        graph -> size_edges, graph -> max_edges, graph -> lookup_edges,
	avg, graph -> resize_edges);
    }
  
  free(graph -> nodes);
  free(graph);
}

/*------------------------------------------------------------------------*/

static Node new_Node(Graph graph, Variable v, unsigned idx)
{
  Node res;

  res = (Node) malloc(sizeof(struct Node_));
  res -> variable = v;
  res -> idx = idx;

  res -> is_marked = 0;
  res -> is_on_stack = 0;
  res -> next = 0;

  res -> size = next_size(0);
  res -> count = 0;
  res -> edges = (Edge*) calloc(res -> size, sizeof(Edge));

  graph -> size_edges += res-> size;
  graph -> count_nodes++;
  if(graph -> count_nodes > graph -> max_nodes) 
    graph -> max_nodes = graph -> count_nodes;
  
  return res;
}

/*------------------------------------------------------------------------*/

static unsigned hash_Variable(Variable v, unsigned idx)
{
  unsigned res;

  res = ((unsigned)v) * 49999;
  res = (res + ((unsigned)idx)) * 9973;

  return res;
}

/*------------------------------------------------------------------------*/

static unsigned hash_Node(Node n)
{
  return hash_Variable(n -> variable, n -> idx);
}

/*------------------------------------------------------------------------*/

static int is_equal_Node(Node n, Variable v, unsigned idx)
{
  return n -> variable == v && n -> idx == idx;
}

/*------------------------------------------------------------------------*/

static void resize_Nodes(Graph graph)
{
  Node * old_nodes, p, tmp;
  unsigned old_size, i, h;

  graph -> resize_nodes++;

  old_nodes = graph -> nodes;
  old_size = graph -> size_nodes;

  graph -> size_nodes = next_size(graph -> count_nodes);
  graph -> nodes = (Node*) calloc(graph -> size_nodes, sizeof(Node));

  for(i = 0; i < old_size; i++)
    {
      for(p = old_nodes[i]; p; p = tmp)
        {
	  tmp = p -> next;
	  h = hash_Node(p) % graph -> size_nodes;
	  p -> next = graph -> nodes[h];
	  graph -> nodes[h] = p;
	}
    } 

  free(old_nodes);
}

/*------------------------------------------------------------------------*/

static Node find_Node(Graph graph, Variable v, unsigned idx)
{
  unsigned h;
  Node * p;

  if(graph -> count_nodes > 4 * graph -> size_nodes) resize_Nodes(graph);

  h = hash_Variable(v, idx) % graph -> size_nodes;
  graph -> lookup_nodes++;
  for(p = &graph -> nodes[h]; *p; p = &(*p) ->next)
    {
      graph -> visited_nodes++;
      if(is_equal_Node(*p, v, idx)) break;
    }
  
  if(!*p) *p = new_Node(graph, v, idx);

  return *p;
}

/*------------------------------------------------------------------------*/

static Edge new_Edge(Graph graph, Node src, Node dst)
{
  Edge res;

  res = (Edge) malloc(sizeof(struct Edge_));
  res -> dst = dst;
  res -> next = 0;
  src -> count++;
  graph -> count_edges++;
  if(graph -> max_edges < graph -> count_edges)
    graph -> max_edges = graph -> count_edges;
  
  return res;
}

/*------------------------------------------------------------------------*/

static void resize_Edges(Graph graph, Node src)
{
  Edge * old_edges, p, tmp;
  unsigned old_size, h, i;

  graph -> resize_edges++;

  old_edges = src -> edges;
  old_size = src -> size;

  src -> size = next_size(src -> count);
  src -> edges = (Edge*) calloc(src -> size, sizeof(Edge));

  for(i = 0; i < old_size; i++)
    {
      for(p = old_edges[i]; p; p = tmp)
        {
	  tmp = p -> next;
	  h = hash_Node(p -> dst) % src -> size;
	  p -> next = src -> edges[h];
	  src -> edges[h] = p;
	}
    }

  graph -> size_edges += src -> size;
  graph -> size_edges -= old_size;

  free(old_edges);
}

/*------------------------------------------------------------------------*/

static void insert_Edge(Graph graph, Node src, Node dst)
{
  Edge * p;
  unsigned h;

  if(src -> count > 4 * src -> size) resize_Edges(graph, src);
  h = hash_Node(dst) % src -> size;
  graph -> lookup_edges++;
  for(p = &src -> edges[h]; *p; p = &(*p) -> next)
    {
      graph -> visited_edges++;
      if(is_equal_Node((*p) -> dst, dst -> variable, dst -> idx)) break;
    }
  
  if(!*p) *p = new_Edge(graph, src, dst);
}

/*------------------------------------------------------------------------*/

void add_Edge(Graph graph, Variable u, unsigned i, Variable v, unsigned j)
{
  Node src, dst;

  assert(i < u -> size);
  assert(j < v -> size);

  src = find_Node(graph, u, i);
  dst = find_Node(graph, v, j);
  insert_Edge(graph, src, dst);
}

/*------------------------------------------------------------------------*/

static void reset_Graph(Graph graph)
{
  unsigned i;
  Node p;

  for(i = 0; i < graph -> size_nodes; i++)
    for(p = graph -> nodes[i]; p; p = p -> next)
      p -> is_marked = p -> is_on_stack = 0;
}

/*------------------------------------------------------------------------*/

static Node _find_Cycle(Node src)
{
  Node res;
  unsigned i;
  Edge e;

  if(src -> is_on_stack) return src;		/* found backedge */
  if(src -> is_marked) return 0;		/* found crossedge */

  src -> is_marked = 1;
  src -> is_on_stack = 1;			/* mark as `on stack' */

  for(i = 0, res = 0; !res && i < src -> size; i++)
    for(e = src -> edges[i]; !res && e; e = e -> next)
      res = _find_Cycle(e -> dst);
  
  src -> is_on_stack = 0;
  
  return res;
}

/*------------------------------------------------------------------------*/

Node find_Cycle(Graph graph)
{
  Node res, p;
  unsigned i;

  for(i = 0, res = 0; !res && i < graph -> size_nodes; i++)
    for(p = graph -> nodes[i]; !res && p; p = p -> next)
      res = _find_Cycle(p);
  
  reset_Graph(graph);

  return res;
}

/*------------------------------------------------------------------------*/

void init_DFS(Graph graph)
{
  /* nothing to done here */
}

/*------------------------------------------------------------------------*/

void _dfs(Node src)
{
  unsigned i;
  Edge e;

  if(src -> is_marked) return;

  src -> is_marked = 1;

  for(i = 0; i < src -> size; i++)
    for(e = src -> edges[i]; e; e = e -> next)
      _dfs(e -> dst);
}

/*------------------------------------------------------------------------*/

void dfs(Graph g, Variable v, unsigned idx)
{
  Node src;

  src = find_Node(g, v, idx);
  _dfs(src);
}

/*------------------------------------------------------------------------*/

void forall_reachable_Graph(Graph graph, void(*f)(Variable,unsigned))
{
  Node p;
  unsigned i;
  for(i = 0; i < graph -> size_nodes; i++)
    for(p = graph -> nodes[i]; p; p = p -> next)
      if(p -> is_marked) f(p -> variable, p -> idx);
}

/*------------------------------------------------------------------------*/

void exit_DFS(Graph graph)
{
  reset_Graph(graph);
}
