#include <stdio.h>
#include <stdlib.h>
#include <time.h>

static void usage()
{
  printf( "*** usage: genqueue [-implies][-invar] <length> [<width>]\n");
  printf(" ***        default width is `1'\n");
}

static int width, length;

static int log2(int e)
{
  if(e < 0x2)        return 1;
  if(e < 0x4)        return 2;
  if(e < 0x8)        return 3;
  if(e < 0x10)       return 4;
  if(e < 0x20)       return 5;
  if(e < 0x40)       return 6;
  if(e < 0x80)       return 7;
  if(e < 0x100)      return 8;
  if(e < 0x200)      return 9;
  if(e < 0x400)      return 10;
  if(e < 0x800)      return 11;
  if(e < 0x1000)     return 12;
  if(e < 0x2000)     return 13;
  if(e < 0x4000)     return 14;
  if(e < 0x8000)     return 15;
  if(e < 0x10000)    return 16;
  if(e < 0x20000)    return 17;
  if(e < 0x40000)    return 18;
  if(e < 0x80000)    return 19;
  if(e < 0x100000)   return 20;
  if(e < 0x200000)   return 21;
  if(e < 0x400000)   return 22;
  if(e < 0x800000)   return 23;
  if(e < 0x1000000)  return 24;
  if(e < 0x2000000)  return 25;
  if(e < 0x4000000)  return 26;
  if(e < 0x8000000)  return 27;
  if(e < 0x10000000) return 28;
  if(e < 0x20000000) return 29;
  if(e < 0x40000000) return 30;
  if(e < 0x80000000) return 31;
  return 32;
}

static int pow2(int e)
{
  int res, i;

  for(res = 1, i = 0; i < e; i++)
    res *= 2;
  
  return res;
}

int main(int argc, char ** argv)
{
  int i, j, log, bound, invar, implies;

  length = -1;
  width = -1;
  invar = 0;
  implies = 0;

  for(i = 1; i < argc; i++)
    {
      if('0' <= argv[i][0] && argv[i][0] <= '9')
        {
	  if(length == -1)
	    {
	      length = atoi(argv[i]);
	    }
	  else
	  if(width == -1)
	    {
	      width = atoi(argv[i]);
	    }
	  else
	    {
	      fprintf(stderr, "*** three numbers specified\n");
	      usage();
	      exit(1);
	    }
	}
      else
      if(strcmp("-invar", argv[i]) == 0)
        {
	  invar = 1;
	}
      else
      if(strcmp("-implies", argv[i]) == 0)
        {
	  implies = 1;
	}
      else
        {
	  fprintf(stderr, "*** unknown command line option `%s'\n", argv[i]);
	  usage();
	  exit(1);
	}
    }
  
  if(implies && invar)
    {
      fprintf(stderr, "*** can not specify both `-invar' and `-implies'\n");
      usage();
      exit(1);
    }
  
  if(length == -1)
    {
      fprintf(stderr, "*** no length specified\n");
      usage();
      exit(1);
    }
  
  if(width == -1) width = 1;

  log = log2(length - 1);
  bound = pow2(log);

  if(!length)
    {
      fprintf(stderr, "*** length must be greater zero\n");
      usage();
      exit(1);
    }

  if(!width)
    {
      fprintf(stderr, "*** width must be greater zero\n");
      usage();
      exit(1);
    }

  printf("VAR\n\n");

  if(length > 2)
    {
      printf("  atop[%d]\n", log);
      printf("  chead[%d]\n", log);
      printf("  ctail[%d]\n", log);
    }

  printf("  input[%d]\n", width);
  printf("  aout[%d]\n", width);
  printf("  cout[%d]\n", width);

  for(i = 0; i < length; i++) printf("  a%d[%d]\n", i, width);
  if(invar) for(i = 0; i < length; i++) printf("  b%d[%d]\n", i, width);
  for(i = 0; i < length; i++) printf("  c%d[%d]\n", i, width);

  printf("\n");

  printf("DEFINE\n\n");

  if(length == 1)
    {
      printf("  afull :=\n");
      printf("    case\n");
      printf("      aempty : 0;\n");
      printf("      1 : 1;\n");
      printf("    esac;\n\n");

      printf("  cfull :=\n");
      printf("    case\n");
      printf("      cempty : 0;\n");
      printf("      1 : 1;\n");
      printf("    esac;\n\n");
    }
  else
    {
      printf("  afull :=\n");
      printf("    case\n");
      printf("      atop = %d : 1;\n", length - 1);
      printf("      1 : 0;\n");
      printf("    esac;\n\n");

      printf("  cfull :=\n");
      printf("    case\n");
      printf("      ctail = %d :\n", length - 1);
      printf("         case\n");
      printf("           chead = 0 : 1;\n");
      printf("           1 : 0;\n");
      printf("         esac;\n");
      printf("      inc(ctail) = chead : 1;\n");
      printf("      1 : 0;\n");
      printf("    esac;\n\n");
    }

  printf("  read :=\n");
  printf("    case\n");
  printf("      afull | cfull : 0;\n");
  printf("      aempty | cempty : 1;\n");
  printf("      1 : oracle;\n");
  printf("    esac;\n\n");

  printf("  write :=\n");
  printf("    case\n");
  printf("      read : 0;\n");
  printf("      1 : 1;\n");
  printf("    esac;\n\n");

  printf("  aout :=\n");
  printf("    case\n");
  printf("      write : a0;\n");
  printf("      1 : 0;\n");
  printf("    esac;\n\n");
            
  printf("  cout :=\n");
  printf("    case\n");
  printf("      write :\n");
  printf("        switch(chead){\n");
  for(i = 0; i < length; i++)
    printf("          %d : c%d;\n", i, i);
  for(i = length; i < bound; i++)
    printf("          %d : 0;\n", i);
  printf("        };\n");
  printf("      1 : 0;\n");
  printf("    esac;\n\n");

  printf("ASSIGN\n\n");

  /*
   *  +---+---+---+---+   atop moves to the right when a new element is 
   *  | X | X | X |   |   entered while reading. During the writing the
   *  +---+---+---+---+   first element is extracted and the whole contents
   *         <- ^ ->      of a is moved to the left.
   *            atop 
   */

  if(!implies) printf("  init(aempty) := 1;\n");

  printf("  next(aempty) :=\n");
  printf("    case\n");
  printf("      read : 0;\n");
  printf("      write & atop = 0 : 1;\n");
  printf("      1 : aempty;\n");
  printf("    esac;\n\n");

  if(!implies) printf("  init(atop) := 0;\n");

  printf("  next(atop) :=\n");
  printf("    case\n");
  printf("      write & atop = 0: 0;\n");
  printf("      !write & atop = %d: %d;\n", length - 1, length -1);
  printf("      write : ");
  if(length > 1) printf("dec(atop);\n");
  else printf("atop;\n");
  printf("      aempty : 0;\n");
  printf("      1 : inc(atop);\n");
  printf("    esac;\n\n");

  for(i = 0; i < length; i++)
    {
      if(!implies) printf("  init(a%i) := 0;\n", i);

      printf("  next(a%i) :=\n", i);
      printf("    case\n");

      if(i == length - 1) printf("      write : 0;\n");
      else printf("      write : a%d;\n", i + 1);

      if(i == 0) printf("      aempty : input;\n");
      else if(i == 1) printf("      !aempty & atop = 0 : input;\n");
      else  printf("      atop = %d : input;\n", i - 1);

      printf("      1 : a%i;\n", i);
      printf("    esac;\n\n");
    }
  
  /*
   *                 ctail moving to the right
   *                 v ->
   *       +---+---+---+---+  New elements are put at the tail when reading.
   *    c  | X | X | X |   |  When writing from the queue then elements are
   *       +---+---+---+---+  taken from the head.
   *
   *         ^  ->
   *         chead moving to the right as well
   *
   *    Note:  When chead = ctail then cempty is needed to distinguish 
   *           the case where there are elements in c from the case
   *           where c is empty.
   */

  if(!implies) printf("  init(cempty) := 1;\n");

  printf("  next(cempty) :=\n");
  printf("    case\n");
  printf("      read : 0;\n");
  printf("      chead = ctail : 1;\n");
  printf("      1 : 0;\n");
  printf("    esac;\n\n");

  if(!implies) printf("  init(chead) := 0;\n");

  printf("  next(chead) :=\n");
  printf("    case\n");
  printf("      chead != ctail & write :\n");
  printf("        case\n");
  printf("          chead = %d : 0;\n", length - 1);
  printf("          1 : inc(chead);\n");
  printf("        esac;\n");
  printf("      1 : chead;\n");
  printf("    esac;\n\n");

  if(!implies) printf("  init(ctail) := 0;\n");

  printf("  next(ctail) :=\n");
  printf("    case\n");
  printf("      !cempty & read :\n");
  printf("         case\n");
  printf("           ctail = %d : 0;\n", length - 1);
  printf("           1 : inc(ctail);\n");
  printf("         esac;\n");
  printf("      1 : ctail;\n");
  printf("    esac;\n\n");

  for(i = 0; i < length; i++)
    {
      if(!implies) printf("  init(c%i) := 0;\n", i);

      printf("  next(c%i) :=\n", i);
      printf("    case\n");
      printf(
"      !write & (cempty & ctail = %d | !cempty & ctail = %d) : input;\n",
       i, (i + length - 1) % length);
      printf("      write & chead = %d : 0;\n", i);
      printf("      1 : c%d;\n", i);
      printf("    esac;\n\n");
    }

  /*
  printf("SPEC\n\n  AG afull = cfull\n");
  printf("SPEC\n\n  AG aempty = cempty\n");
  */

  if(invar || implies)
    {
      printf("DEFINE\n\n");

      printf("  invar :=\n");

      printf("    case\n");
      for(i = 0; i < length; i++)
        printf("      a%d = b%d\n      &\n", i, i);
      
      printf("      afull = cfull\n");
      printf("      &\n");
      printf("      aempty = cempty\n");
      printf("      &\n");
      printf("      (aempty -> atop = 0)\n");
      printf("      &\n");
      printf("      (cempty -> chead = ctail)\n");

      if(length < bound)
        {
	  printf("      &\n");
	  printf("      (");
	  for(j = length; j < bound - 1; j++) printf("atop != %d & ", j);
	  printf("atop != %d)\n", bound - 1);

	  printf("      &\n");
	  printf("      (");
	  for(j = length; j < bound - 1; j++) printf("ctail != %d & ", j);
	  printf("ctail != %d)\n", bound - 1);

	  printf("      &\n");
	  printf("      (");
	  for(j = length; j < bound - 1; j++) printf("chead != %d & ", j);
	  printf("chead != %d)\n", bound - 1);

	  for(i = 0; i < length; i++)
	    for(j = 0; j < length; j++)
	      {
		printf("      &\n");
		printf("      (chead = %d & ctail = %d -> atop = %d)\n",
		  i, j, (j + (length - i)) % length);
	      }
	}
      else 
        {
	  printf("      &\n");
	  printf("      ctail = add(chead, atop)\n");
	}

      printf("      &\n");
      printf("      (aempty -> ");
      for(j = 0; j < length - 1; j++) printf("a%d=0 & ", j);
      printf("a%d=0)\n", length - 1);
      for(i = 0; i < length - 1; i++)
        {
	  printf("      &\n");
	  printf("      (!aempty & atop = %d -> ", i);
	  for(j = i + 1; j < length - 1; j++) printf("a%d=0 & ", j);
	  printf("a%d=0)\n", length - 1);
	}

      printf("      : 1;\n");
      printf("    1 : 0;\n");
      printf("  esac;\n");

      printf("\n");

      for(i = 0; i < length; i++)
        {
	  printf("  b%d :=\n", i);
	  printf("    switch(chead){\n");
	  for(j = 0; j < length; j++)
	    printf("      %d : c%d;\n", j, (j + i) % length);
	  for(j = length; j < bound; j++)
	    printf("      %d : 0;\n", j);
	  printf("    };\n\n");
	}
    }

  if(invar)
    {
      printf("SPEC\n\n  AG invar\n");
    }
  else 
  if(implies)
    {
      /*
      printf("SPEC\n\n  invar -> aout = cout\n");
      */
      printf("INIT invar\n\n");
      printf("SPEC\n\n  aout = cout\n");
    }
  else printf("SPEC\n\n  AG aout = cout\n");

  exit(0);
  return 0;
}
