extern "C" {
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
};

/*------------------------------------------------------------------------*/

static bool BMC_output = false;
static bool fairness = false;
static bool bug = false;

/*------------------------------------------------------------------------*/

class String
{
  static char concat_buffer[1000];

  char * _str;

public:
  
  String(const char * a) : _str(strdup(a)) { }

  String(const char * a, int i)
  {
    sprintf(concat_buffer, "%s%d", a, i);
    _str = strdup(concat_buffer);
  }

  ~String() { free(_str); }

  String * extend(const char * a) const
  {
    String * res;

    sprintf(concat_buffer, "%s_%s", str(), a);
    res = new String(concat_buffer);

    return res;
  }

  String * extend(const char * a, int i) const
  {
    String * res;

    sprintf(concat_buffer, "%s_%s%d", str(), a, i);
    res = new String(concat_buffer);

    return res;
  }

  String * prefix(const char * a) const
  {
    String * res;

    sprintf(concat_buffer, "%s%s", a, str());
    res = new String(concat_buffer);

    return res;
  }

  const char * str() const { return _str; }
};

char String::concat_buffer[1000];

/*------------------------------------------------------------------------*/

class Gate
{
  String * _name;
  Gate * next;

  static Gate * first, * last;

  static void _print(void (Gate::*f)())
  {
    Gate * g;

    for(g = first; g; g = g -> next) (g ->* f)();
  }

  virtual void print_ASSIGN() { }
  virtual void print_TRANS() { }
  virtual void print_DECL() { printf("  %s : boolean;\n", name()); }
  virtual void print_FAIRNESS() { printf("FAIRNESS o_%s\n", name()); }

public:
  
  static void exit()
  {
    Gate * p, * tmp;

    for(p = first; p; p = tmp)
      {     
	tmp = p -> next;
	delete p;
      }
  }

  Gate(String * s)
  :
    _name(s),
    next(0)
  {
    if(last)
      {
	last -> next = this;
	last = this;
      }
    else first = last = this;
  }

  virtual ~Gate() { delete _name; }

  virtual Gate * child(int) const { return 0; }

  const char * name() const { return _name -> str(); }
  const String * string() const { return _name; }

  static void print()
  { 
    if(!BMC_output)
      {
	printf("MODULE main\n\nVAR\n\n");
	_print(&print_DECL);
	printf("\n");
      }

    printf("ASSIGN\n\n");
    _print(&print_ASSIGN);

    if(fairness) _print(&print_FAIRNESS);

    printf("\nTRANS\n\n");
    _print(&print_TRANS);
    printf("  1\n");
  }
};

/*------------------------------------------------------------------------*/

class NotSignal
:
  public Gate		/* thats no contradiction */
{
  Gate * in;

public:
  
  NotSignal(Gate * i) : Gate(i -> string() -> prefix("!")) { }
  
  Gate * child(int i)
  {
    Gate * res;

    if(i == 0) res = in;
    else res = 0;

    return res;
  }

  void print_DECL() { }
  void print_FAIRNESS() { }
};

/*------------------------------------------------------------------------*/

class BinGate
:
  public Gate
{
protected:

  Gate * left, * right;

  void print_BinGate(int init, char * op)
  {
    printf("  init(%s) := %d;\n", name(), init);
    printf("  next(%s) := o_%s & (%s %s %s) | !o_%s & %s;\n",
      name(), name(), left -> name(), op, right -> name(), name(), name());
  }

public:
  
  BinGate(String * s, Gate * l, Gate * r) : Gate(s), left(l), right(r) { }
  BinGate(String * s) : Gate(s), left(0), right(0) { }

  Gate * child(int i) const
  {
    if(i == 0) return left;
    else if(i == 1) return right;
    else return 0;
  }

  void connect(Gate * l, Gate * r) { left = l; right = r; }
  void connect_left(Gate * l) { left = l; }
  void connect_right(Gate * r) { right = r; }

  void print_DECL()
  {
    printf("  o_%s : boolean;\n", name());
    printf("  %s : boolean;\n", name());
  }
};

/*------------------------------------------------------------------------*/

class AndGate
:
  public BinGate
{
  int init;

public:
  
  AndGate(String * s, Gate * l, Gate * r, int _init = 0) 
  :
    BinGate(s, l, r), init(_init)
  {}

  AndGate(String * s, int _init = 0) : BinGate(s), init(_init) { }

  void print_ASSIGN() { print_BinGate(init, "&"); }
};

/*------------------------------------------------------------------------*/

class OrGate
:
  public BinGate
{
public:
  
  OrGate(String * s, Gate * l, Gate * r) : BinGate(s, l, r) { }
  OrGate(String * s) : BinGate(s) { }

  void print_ASSIGN() { print_BinGate(0, "|"); }
};

/*------------------------------------------------------------------------*/

class CElement
:
  public BinGate
{
public:
  
  CElement(String * s, Gate * l, Gate * r) : BinGate(s, l, r) { }
  CElement(String * s) : BinGate(s) { }

  void print_ASSIGN()
  {
    printf("  init(%s) := 0;\n", name());
    printf("  next(%s) :=\n", name());
    printf("    case\n");
    printf("      o_%s & %s = %s : %s;\n", 
      name(), left -> name(), right -> name(), left -> name());
    printf("      1 : %s;\n", name());
    printf("    esac;\n");
  }
};

/*------------------------------------------------------------------------*/

class MutexHalf
:
  public BinGate
{
public:
  
  MutexHalf(String * s, Gate * l, Gate * r) : BinGate(s,l,r) { }
  MutexHalf(String * s) : BinGate(s) { }

  void print_ASSIGN()
  {
    printf("  init(%s) := 0;\n", name());
    printf("  next(%s) := o_%s & %s | !o_%s & %s;\n",
      name(), name(), left -> name(), name(), name());
  }

  void print_TRANS()
  {
    if(fairness || !bug)
      printf("  !(next(%s) & next(%s)) &\n", name(), right -> name());
  }

  void print_FAIRNESS() 
  { 
    if(!bug) printf("FAIRNESS o_%s\n", name());
  }
};

/*------------------------------------------------------------------------*/

class User
:
  public Gate
{
  Gate * _ack;
  int initial_req;

public:
  
  User(String * s, int _intial_req = 0)
  :
    Gate(s), _ack(0), initial_req(_intial_req)
  {}

  void connect_ack(Gate * r) { _ack = r; }

  void print_ASSIGN()
  {
    printf("  init(%s) := %d;\n", name(), initial_req);
    printf("  next(%s) := o_%s & !%s | !o_%s & %s;\n",
      name(), name(), ack() -> name(), name(), name());
  }

  Gate * req() { return this; }
  Gate * ack() const { return _ack; }

  void print_DECL()
  {
    printf("  o_%s : boolean;\n", name());
    printf("  %s : boolean;\n", name());
  }
};

/*------------------------------------------------------------------------*/
/*
 *  q : and-gate(f.out,n.out);
 *  f : c-element(d.out,i.out);
 *  d : and-gate(b.out,!u.ack);
 *  b : mutex-half(left.req,a.out);
 *  i : and-gate(h.out,!j.out);
 *  h : c-element(g.out,j.out);
 *  n : and-gate-init(!e.out,!m.out,!token);
 *  u : user;
 *  a : mutex-half(u.req,b.out);
 *  c : and-gate(a.out,!left.ack);
 *  g : or-gate(c.out,d.out);
 *  e : c-element(c.out,i.out);
 *  k : and-gate(g.out,!h.out);
 *  l : and-gate(k.out,m.out);
 *  p : and-gate(k.out,n.out);
 *  m : and-gate-init(!f.out,!n.out,token);
 *  r : and-gate(e.out,m.out);
 *  j : or-gate(l.out,ack);
 */

class Cell
:
  public Gate
{
  Cell * left, * right;

  BinGate * q;
  BinGate * f;
  BinGate * d;
  BinGate * b;
  BinGate * i;
  BinGate * h;
  BinGate * n;
  User * u;
  BinGate * a;
  BinGate * c;
  BinGate * g;
  BinGate * e;
  BinGate * k;
  BinGate * l;
  BinGate * p;
  BinGate * m;
  BinGate * r;
  BinGate * j;

public:
  
  Cell(String * str, int token)
  :
    Gate(str),
    left(0),
    right(0)
  {
    q = new AndGate(str -> extend("q"));
    f = new CElement(str -> extend("f"));
    d = new AndGate(str -> extend("d"));
    b = new MutexHalf(str -> extend("b"));
    i = new AndGate(str -> extend("i"));
    h = new CElement(str -> extend("h"));
    n = new AndGate(str -> extend("n"), !token);
    u = new User(str -> extend("u"), token);
    a = new MutexHalf(str -> extend("a"));
    c = new AndGate(str -> extend("c"));
    g = new OrGate(str -> extend("g"));
    e = new CElement(str -> extend("e"));
    k = new AndGate(str -> extend("k"));
    l = new AndGate(str -> extend("l"));
    p = new AndGate(str -> extend("p"));
    m = new AndGate(str -> extend("m"), token);
    r = new AndGate(str -> extend("r"));
    j = new OrGate(str -> extend("j"));

    q -> connect(f, n);
    f -> connect(d, i);
    d -> connect(b, new NotSignal(r));
    b -> connect_right(a);
    i -> connect(h, new NotSignal(j));
    h -> connect(g, j);
    n -> connect(new NotSignal(e), new NotSignal(m));
    u -> connect_ack(r);
    a -> connect(u -> req(), b);
    c -> connect(a, new NotSignal(q));
    g -> connect(c, d);
    e -> connect(c, i);
    k -> connect(g, new NotSignal(h));
    l -> connect(k, m);
    p -> connect(k, n);
    m -> connect(new NotSignal(f), new NotSignal(n));
    r -> connect(e, m);
    j -> connect_left(l);
  }
  
  void connect(Cell * l, Cell * r)
  {
    left = l;
    right = r;

    b -> connect_left(left -> req());
    j -> connect_right(ack());
  }

  Gate * req() const { return p; }
  Gate * ack() const { return right -> q; }
  User * user() const { return u; }
  
  Gate * child(int num) const
  {
    Gate * res;

    switch(num)
      {
        case 0: res = q;
        case 1: res = f;
        case 2: res = d;
        case 3: res = b;
        case 4: res = i;
        case 5: res = h;
        case 6: res = n;
        case 7: res = u;
        case 8: res = a;
        case 9: res = c;
        case 10: res = g;
        case 11: res = e;
        case 12: res = k;
        case 13: res = l;
        case 14: res = p;
        case 15: res = m;
        case 16: res = r;
        case 17: res = j;
	default: res = 0;
      }
    
    return res;
  }

  void print_DECL() { }
  void print_FAIRNESS() { }
};

/*-----------------------------------------------------------------------*/

Gate * Gate::first = 0, * Gate::last = 0;

/*------------------------------------------------------------------------*/

static void usage()
{
  fprintf(stderr,
    "*** usage: gendme1 [-fair][-bmc][-bug] <number>\n");
}

int main(int argc, char ** argv)
{
  int i, j, n;
  Cell ** cells;
  bool found_number;

  for(found_number = false, i = 1; i < argc; i++)
    {
      if(strcmp(argv[i], "-bmc") == 0)
        {
	  BMC_output = 1;
	}
      else
      if('0' <= argv[i][0] && argv[i][0] <= '9')
        {
	  n = atoi(argv[i]);
	  found_number = true;
	}
      else
      if(strcmp(argv[i], "-bug") == 0)
        {
	  bug = true;
	}
      else
      if(strcmp(argv[i], "-fair") == 0)
        {
	  fairness = true;
	}
      else
	{
	  fprintf(stderr, "*** unknown command line option `%s'\n", argv[i]);
	  usage();
	  exit(1);
	}
    }
  
  if(!found_number)
    {
      fprintf(stderr, "*** no number specified\n");
      usage();
      exit(1);
    }
  
  if(n<2)
    {
      fprintf(stderr, "*** gendme1: number must be greater 1\n");
      exit(1);
    }
  
  cells = new Cell * [ n ];
  for(i=n-1; i >= 0; i--) cells[i] = new Cell(new String("cell", i), i==n-1);
  for(i=0; i<n; i++) cells[i] -> connect(cells[(i+1)%n], cells[(i+n-1)%n]);

  Gate::print();

  if(fairness)
    {
      printf("\nSPEC AF %s\n", cells[n-1] -> user() -> ack() -> name());
    }
  else
    {
      printf("\nSPEC  AG\n\n");
      printf("  (\n");
      for(i = 0; i < n - 1; i++)
	for(j = i + 1; j < n; j ++)
	  printf("    !(%s & %s) &\n",
	    cells[i] -> user() -> ack() -> name(), 
	    cells[j] -> user() -> ack() -> name());

      printf("    1\n");
      printf("  )\n");
    }
  
  Gate::exit();
  delete [] cells;
  exit(0);
  return 0;
}
