#include <stdlib.h>

#include "assign.h"
#include "config.h"
#include "ctl.h"
#include "dep.h"
#include "graph.h"
#include "util.h"
#include "var.h"

/*------------------------------------------------------------------------*/

extern CTL init, trans, spec, invar;
extern char * comment_string;

#ifdef VERYDEBUG
extern int debug;
#endif

extern int use_varset;

/*------------------------------------------------------------------------*/

static VarSet _dependencies_CTL(CTL f)
{
  VarSet res, l, r, tmp;
  unsigned i, idx;
  Variable v;

  if(f -> is_marked)
    {
      res = copy_VarSet(f -> cache.varset);

      assert(res ||
	f -> tag == COLON_Tag ||
	f -> tag == CONSTANT_Tag ||
	f -> tag == NUMBER_Tag);
    }
  else
    {
      switch(f -> tag)
	{
	  case ADD_Tag:
	  case AND_Tag:
	  case CASE_Tag:
	  case COLON_Tag:
	  case EQUAL_Tag:
	  case EQUIV_Tag:
	  case IMPLIES_Tag:
	  case OR_Tag:
	  case SUB_Tag:
	  case SWITCH_Tag:

	    l = _dependencies_CTL(f -> data.arg[0]);
	    r = _dependencies_CTL(f -> data.arg[1]);
	    res = join_VarSet(l, r);
	    free_VarSet(l);
	    free_VarSet(r);
	    break;

	  case NOT_Tag:
	  case INC_Tag:
	  case DEC_Tag:
	  case EX_Tag:
	  case EF_Tag:
	  case EG_Tag:
	  case AX_Tag:
	  case AF_Tag:
	  case AG_Tag:

	    res = _dependencies_CTL(f -> data.arg[0]);
	    break;
	  
	  case CURRENT_Tag:
	  case NEXT_Tag:

	    v = f -> data.base.variable;
	    idx = f -> data.base.idx;

	    if(v -> size == idx)
	      {
		res = mt_VarSet();

		for(i = 0; i < v -> size; i++)
		  {
		    if(v -> cache.varsets && v -> cache.varsets[i])
		      {
			tmp = join_VarSet(res, v -> cache.varsets[i]);
			free_VarSet(res);
			res = tmp;
		      }

		    tmp = add_VarSet(res, v, i);
		    free_VarSet(res);
		    res = tmp;
		  }
	      }
	    else
	      {
		if(v -> cache.varsets && v -> cache.varsets[idx])
		  {
		    tmp = copy_VarSet(v -> cache.varsets[idx]);
		  }
		else tmp = mt_VarSet();

		res = add_VarSet(tmp, v, idx);
		free_VarSet(tmp);
	      }
	    break;

	  case NUMBER_Tag:
	  case CONSTANT_Tag:

	    res = mt_VarSet();
	    break;

	  default:

	    fatal(POSITION, "non valid tag\n");
	    res = (VarSet) 0;
	    break;
	}

      /* Now all marked nodes of `f' have a reference to a VarSet
       * in its cache.varset.
       */
      f -> is_marked = 1;
      f -> cache.varset = copy_VarSet(res);
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static void reset_marked_varsets_cache_CTL(CTL f)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(g -> is_marked)
	{
	  g -> is_marked = 0;
	  if(g -> cache.varset)
	    {
              free_VarSet(g -> cache.varset);
	      g -> cache.varset = 0;
	    }
	  push_CTL(stack, g);
	}
    }

  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

static VarSet dependencies_CTL(CTL f)
{
  VarSet res;

  res = _dependencies_CTL(f);
  reset_marked_varsets_cache_CTL(f);

  return res;
}

/*------------------------------------------------------------------------*/

static void _print_dependencies_Variable(Variable v)
{
  unsigned i;
  int found;

  if(v -> cache.varsets)
    {
      for(found = 0, i = 0; !found && i < v -> size; i++)
        found = (v -> cache.varsets[i] != 0);

      if(found)
        {
	  for(i = 0; i < v -> size; i++)
	    {
	      if(v -> cache.varsets[i])
		{
		  printf("%sDEP %s", comment_string, v -> name);
		  if(v -> size > 1) printf("[%u]", i);
		  fputs(" <- ", stdout);
		  print_VarSet(v -> cache.varsets[i]);
		  putc('\n', stdout);
		}
	    }
	}
    }
}

/*------------------------------------------------------------------------*/

void print_dependencies()
{
  printf("%sDEP begin\n", comment_string);
  forall_Variable(_print_dependencies_Variable);
  printf("%sDEP end\n", comment_string);
}

/*------------------------------------------------------------------------*/

static void insert_dependencies(Variable v, unsigned idx, VarSet s)
{
  VarSet tmp;

  assert(idx < v -> size);		/* no arrays allowed here */

  if(!v -> cache.varsets)
    {
      v -> cache.varsets = (VarSet*) calloc(v -> size, sizeof(VarSet));
      tmp = mt_VarSet();
    }
  else
  if(v -> cache.varsets[idx])
    {
      tmp = v -> cache.varsets[idx];
    }
  else tmp = mt_VarSet();		/* do not assume that an empty
  				         * VarSet is a null pointer !
					 */

  v -> cache.varsets[idx] = join_VarSet(tmp, s);
  free_VarSet(tmp);
}

/*------------------------------------------------------------------------*/

static int insert_mutual_dependencies(VarSet s)
{
  int res;
  VarSet t;
  Variable v;
  unsigned idx;

  res = 0;
  for(t = s; t; t = t -> child)
    {
      idx = t -> idx;
      v = t -> variable;

      assert(idx < v -> size);		/* no arrays allowed here */

      if(!v -> cache.varsets || !subset_VarSet(s, v -> cache.varsets[idx]))
	{
	  insert_dependencies(v, idx, s);
	  res = 1;
	}
    }
  
  return res;
}

/*------------------------------------------------------------------------*/
/* Don't use this function for anything else, since it uses 
 * `_dependencies_CTL' and does not reset the values calculated by
 * that function.
 */

static int gen_formula_dependencies(CTL f)
{
  int res, arg0_changed, arg1_changed;
  VarSet s;

  if(is_true_CTL(f)) res = 0;
  else
    {
      if(f -> tag == AND_Tag) 
	{
	  arg0_changed = gen_formula_dependencies(f -> data.arg[0]);
	  arg1_changed = gen_formula_dependencies(f -> data.arg[1]);

	  res = arg0_changed || arg1_changed;
	}
      else
	{
	  s = _dependencies_CTL(f);
	  res = insert_mutual_dependencies(s);
	  free_VarSet(s);
	}
    }

  return res;
}

/*------------------------------------------------------------------------*/

static int changed_for_functor_flag = 0;

/*------------------------------------------------------------------------*/
/* Don't use this function for anything else, since it uses 
 * `_dependencies_CTL' and does not reset the values calculated by
 * that function.
 */

static void _gen_assignment_dependencies(
  Variable v, CTL formula, CTL * formulas)
{
  VarSet s;
  unsigned i;
  int changed;

  changed = 0;

  if(formula)				/* like  next(a) = inc(a) */
    {
      assert(!formulas);

      s = _dependencies_CTL(formula);

      for(i = 0; i < v -> size; i++)
	{
	  if(!v -> cache.varsets ||
	     !subset_VarSet(s, v -> cache.varsets[i]))
	    {
	      insert_dependencies(v, i, s);
	      changed = 1;
	    }
	}

      free_VarSet(s);
    }
  
  if(formulas)				/* like next(a[0]) = a[1] */
    {
      for(i = 0; i < v -> size; i++)
	{
	  if(formulas[i])
	    {
	      s = _dependencies_CTL(formulas[i]);

	      if(!v -> cache.varsets || 
		 !subset_VarSet(s, v -> cache.varsets[i]))
		{
	          insert_dependencies(v, i, s);
		  changed = 1;
		}

	      free_VarSet(s);
	    }
	}
    }

  if(changed) changed_for_functor_flag = 1;
}

/*------------------------------------------------------------------------*/
/* Don't use this function for anything else, since it uses 
 * `_dependencies_CTL' and does not reset the values calculated by
 * that function.
 */

static void gen_assignment_dependencies(Assignment a)
{
  Variable v;

  v = a -> variable;

  _gen_assignment_dependencies(v, a -> init, a -> inits);
  _gen_assignment_dependencies(v, a -> next, a -> nexts);
  _gen_assignment_dependencies(v, a -> define, a -> defines);
}

/*------------------------------------------------------------------------*/
/* Don't use this function for anything else, since it uses 
 * `_dependencies_CTL' and does not reset the values calculated by
 * that function.
 */

static int gen_assignments_dependencies()
{
  int res;

  changed_for_functor_flag = 0;
  forall_Assignment(gen_assignment_dependencies);
  res = changed_for_functor_flag;

  return res;
}

/*------------------------------------------------------------------------*/

static void _reset_assignment_dependencies_CTL(
  Variable v, CTL formula, CTL * formulas)
{
  unsigned i;

  if(formula) reset_marked_varsets_cache_CTL(formula);

  if(formulas)
    {
      for(i = 0; i < v -> size; i++)
	if(formulas[i]) reset_marked_varsets_cache_CTL(formulas[i]);
    }
}

/*------------------------------------------------------------------------*/

static void reset_assignment_dependencies_CTL(Assignment a)
{
  Variable v;

  v = a -> variable;
  _reset_assignment_dependencies_CTL(v, a -> init, a -> inits);
  _reset_assignment_dependencies_CTL(v, a -> next, a -> nexts);
  _reset_assignment_dependencies_CTL(v, a -> define, a -> defines);
}

/*------------------------------------------------------------------------*/
/* Reset all the dependencies in `cache.varset' of CTL nodes.
 */

static void reset_varsets_cache_CTL()
{
  reset_marked_varsets_cache_CTL(init);
  reset_marked_varsets_cache_CTL(trans);
  reset_marked_varsets_cache_CTL(invar);
  forall_Assignment(reset_assignment_dependencies_CTL);
}

/*------------------------------------------------------------------------*/

static int _gen_dependencies()
{
  int init_changed, trans_changed, invar_changed, assign_changed;

  init_changed = gen_formula_dependencies(init);
  trans_changed = gen_formula_dependencies(trans);
  invar_changed = gen_formula_dependencies(invar);
  assign_changed = gen_assignments_dependencies();
  reset_varsets_cache_CTL();

  return init_changed || trans_changed || assign_changed;
}

/*------------------------------------------------------------------------*/

void gen_dependencies()
{
  unsigned iteration;

  start_verbose("generating dependencies");

  start_progress();
  for(iteration = 1; _gen_dependencies(); iteration++)
    progress();
  end_progress();

  end_verbose("generated dependencies (%u iterations)", iteration);

# ifdef VERYDEBUG
    if(debug)
      {
	print_dependencies();
      }
# endif
}

/*------------------------------------------------------------------------*/

static VarSet varset_for_functor = 0;

/*------------------------------------------------------------------------*/

static void _coi_remove_assignment(
  Variable v, CTL * formula_ptr, CTL * formulas, unsigned * counter)
{
  CTL formula;
  VarSet s;
  unsigned i;
  int remove;
  
  formula = *formula_ptr;
  s = varset_for_functor;

  if(formula)
    {
      remove = !contains_VarSet(s, v, v -> size);

      for(i = 0; remove && i < v -> size; i++)
	remove = !contains_VarSet(s, v, i);

      if(remove)
	{
	  free_CTL(formula);
	  *formula_ptr = 0;
	  (*counter)--;
	}
    }

  if(formulas)
    {
      if(!contains_VarSet(s, v, v -> size))
        {
	  for(i = 0; i < v -> size; i++)
	    {
	      if(formulas[i] && !contains_VarSet(s, v, i))
		{
		  free_CTL(formulas[i]);
		  formulas[i] = 0;
		  (*counter)--;
		}
	    }
	}
    }
}

/*------------------------------------------------------------------------*/

static void coi_remove_assignment(Assignment a)
{
  Variable v;

  v = a -> variable;

  _coi_remove_assignment(v, &a -> init, a -> inits, &num_init_assignments);
  _coi_remove_assignment(v, &a -> next, a -> nexts, &num_next_assignments);
  _coi_remove_assignment(
    v, &a -> define, a -> defines, &num_define_assignments);

  remove_if_mt_Assignment(a);
}

/*------------------------------------------------------------------------*/

static void coi_remove_assignments(VarSet s)
{
  unsigned old, old_init, old_next, old_define, delta;

  old_init = num_init_assignments;
  old_next = num_next_assignments;
  old_define = num_define_assignments;

  old = old_init + old_next + old_define;
  if(old)
    {
      start_verbose("trying to remove assignments");

      varset_for_functor = s;
      forall_Assignment(coi_remove_assignment);
      varset_for_functor = 0;

      delta = old_init - num_init_assignments;
      if(delta) print_verbose("removed %u `init' assignments\n", delta);

      delta = old_next - num_next_assignments;
      if(delta) print_verbose("removed %u `next' assignments\n", delta);

      delta = old_define - num_define_assignments;
      if(delta) print_verbose("removed %u `define' assignments\n", delta);

      delta = old - num_init_assignments
	          - num_next_assignments 
	          - num_define_assignments;

      if(delta) end_verbose("removed %u assignments", delta);
      else end_verbose("could not remove any assignment");
    }
}

/*------------------------------------------------------------------------*/

static void mark_unused_variable(Variable v)
{
  int keep_it;
  unsigned i;

  keep_it = contains_VarSet(varset_for_functor, v, v -> size);

  for(i = 0; !keep_it && i < v -> size; i++)
    keep_it = contains_VarSet(varset_for_functor, v, i);
  
  if(!keep_it) v -> is_in_COI = 0;
}

/*------------------------------------------------------------------------*/

static void mark_unused_variables(VarSet s)
{
  varset_for_functor = s;
  forall_Variable(mark_unused_variable);
  varset_for_functor = 0;
}

/*------------------------------------------------------------------------*/

static void gen_define_dependencies_Assignment(Assignment a)
{
  _gen_assignment_dependencies(a -> variable, a -> define, a -> defines);
}

/*------------------------------------------------------------------------*/

static int _gen_define_dependencies()
{
  int res;

  changed_for_functor_flag = 0;
  forall_Assignment(gen_define_dependencies_Assignment);
  res = changed_for_functor_flag;
  forall_Assignment(reset_assignment_dependencies_CTL);

  return res;
}

/*------------------------------------------------------------------------*/

static void gen_define_dependencies()
{
  unsigned iteration;

  start_verbose("generating `define' dependencies");

  start_progress();
  for(iteration = 1; _gen_define_dependencies(); iteration++)
    progress();
  end_progress();

  end_verbose("generated `define' dependencies (%u iterations)", iteration);

# ifdef VERYDEBUG
    if(debug)
      {
	print_dependencies();
      }
# endif
}

/*------------------------------------------------------------------------*/

static void detected_circular_dependency(Variable v, unsigned idx)
{
  if(v -> size == idx || v -> size == 1)
    {
      fprintf(stderr,
	"*** detected circular dependency of `%s' on itself\n",
	v -> name);
    }
  else
    {
      fprintf(stderr,
	"*** detected circular dependency of `%s[%u]' on itself\n",
	v -> name, idx);
    }

  exit(1);
}

/*------------------------------------------------------------------------*/

static void check_circular_dependency(Assignment a)
{
  Variable v;
  VarSet s;
  unsigned i, j;
  int found_dependency;

  v = a -> variable;

  if(a -> define)
    {
      if(v -> cache.varsets)
	{
	  for(i = 0; i < v -> size; i++)
	    {
	      s = v -> cache.varsets[i];
	      if(s)
		{
		  found_dependency = contains_VarSet(s, v, v -> size);
		  for(j = 0; !found_dependency && j < v -> size; j++)
		    found_dependency = contains_VarSet(s, v, j);
		 
		  if(found_dependency)
		    detected_circular_dependency(v, v -> size);
		}
	    }
	}
    }
  
  if(a -> defines)
    {
      for(i = 0; i < v -> size; i++)
        {
	  if(v -> cache.varsets[i])
	    {
	      if(contains_VarSet(v -> cache.varsets[i], v, v -> size) ||
	         contains_VarSet(v -> cache.varsets[i], v, i))
	        {
		  detected_circular_dependency(v, i);
		}
	    }
	}
    }
}

/*------------------------------------------------------------------------*/

static void gen_dependencies_Graph_CTL(
  Graph graph, Variable src, unsigned src_idx, CTL f)
{
  CTL g;
  Stack stack;
  unsigned i, j, dst_idx;
  Variable dst;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);

      assert(g -> tag != NEXT_Tag);

      if(!g -> is_marked)
        {
	  g -> is_marked = 1;
	  if(g -> tag == CURRENT_Tag)
	    {
	      dst = g -> data.base.variable;
	      dst_idx = g -> data.base.idx;

	      if(src_idx == src -> size)
	        {
		  if(dst_idx == dst -> size)
		    {
		      for(i = 0; i < src -> size; i++)
		        for(j = 0; j < dst -> size; j++)
			  add_Edge(graph, src, i, dst, j);
		    }
		  else
		    {
		      for(i = 0; i < src -> size; i++)
		        add_Edge(graph, src, i, dst, dst_idx);
		    }
		}
	      else
	        {
		  if(dst_idx == dst -> size)
		    {
		      for(j = 0; j < dst -> size; j++)
		        add_Edge(graph, src, src_idx, dst, j);
		    }
		  else
		    {
		      add_Edge(graph, src, src_idx, dst, dst_idx);
		    }
		}
	    }
	  else push_CTL(stack, g);
	}
    }

  unmark_CTL(f);
  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

static Graph graph_for_functor = 0;

/*------------------------------------------------------------------------*/

static void _gen_dependencies_Graph_Assignment(
  Variable v, CTL formula, CTL * formulas)
{
  unsigned i;

  if(formula) 
    gen_dependencies_Graph_CTL(graph_for_functor, v, v -> size, formula);
  
  if(formulas)
    {
      for(i = 0; i < v -> size; i++)
        if(formulas[i])
	  gen_dependencies_Graph_CTL(graph_for_functor, v, i, formulas[i]);
    }
}

/*------------------------------------------------------------------------*/

static void gen_define_dependencies_Graph_Assignment(Assignment a)
{
  _gen_dependencies_Graph_Assignment(a -> variable, a -> define, a -> defines);
}

/*------------------------------------------------------------------------*/


static Graph gen_define_dependencies_Graph()
{
  Graph res;

  graph_for_functor = new_Graph();
  forall_Assignment(gen_define_dependencies_Graph_Assignment);
  res = graph_for_functor;
  graph_for_functor = 0;

  return res;
}

/*------------------------------------------------------------------------*/

void check_defines()
{
  Graph graph;
  Node node;

  if(!num_define_assignments) return;
  start_verbose("checking circular dependencies");

  if(use_varset)
    {
      gen_define_dependencies();
      forall_Assignment(check_circular_dependency);
      reset_varsets_cache_Variable();
    }
  else
    {
      graph = gen_define_dependencies_Graph();
      node = find_Cycle(graph);
      if(node) detected_circular_dependency(node -> variable, node -> idx);
      free_Graph(graph);
    }

  end_verbose("circular dependency checking");
}

/*------------------------------------------------------------------------*/
/* This code turned out to be very slow, but I keep it for
 * comparison with the `Graph' version.  Note that VarSets are still
 * used with the `Graph' code.
 */

static void coi_with_varset()
{
  VarSet s;

  gen_dependencies();
  s = dependencies_CTL(spec);

# ifdef VERYDEBUG
    if(debug)
      {
	printf("%sSPEC dependencies: ", comment_string);
	print_VarSet(s);
	putc('\n', stdout);
      }
# endif

  coi_remove_assignments(s);
  mark_unused_variables(s);
  free_VarSet(s);
  reset_varsets_cache_Variable();
}

/*------------------------------------------------------------------------*/

static void _insert_mutual_dependencies_Graph(
  Variable u, unsigned uidx, VarSet s)
{
  VarSet t;
  Variable v;
  unsigned vidx;

  for(t = s; t; t = t -> child)
    {
      vidx = t -> idx;
      v = t -> variable;

      assert(vidx < v -> size);
      add_Edge(graph_for_functor, u, uidx, v, vidx);
    }
}

/*------------------------------------------------------------------------*/

static void insert_mutual_dependencies_Graph(VarSet s)
{
  VarSet t;
  Variable v;
  unsigned idx;

  for(t = s; t; t = t -> child)
    {
      idx = t -> idx;
      v = t -> variable;

      assert(idx < v -> size);
      
      _insert_mutual_dependencies_Graph(v, idx, s);
    }
}

/*------------------------------------------------------------------------*/

static void coi_with_graph_CTL(CTL f)
{
  VarSet s;

  if(f -> is_marked)
    {
      /* do nothing */
    }
  else
  if(f -> tag == AND_Tag)
    {
      f -> is_marked = 1;
      coi_with_graph_CTL(f -> data.arg[0]);
      coi_with_graph_CTL(f -> data.arg[1]);
    }
  else
    {
      s = _dependencies_CTL(f);
      insert_mutual_dependencies_Graph(s);
      free_VarSet(s);
    }
}

/*------------------------------------------------------------------------*/

static void dfs_coi_Graph(CTL f)
{
  CTL g;
  Stack stack;
  unsigned i, idx;
  Variable v;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;
	  if(g -> tag == CURRENT_Tag || g -> tag == NEXT_Tag)
	    {
	      v = g -> data.base.variable;
	      idx = g -> data.base.idx;
	      if(v -> size == idx)
	        {
		  for(i = 0; i < v -> size; i++)
		    dfs(graph_for_functor, v, i);
		}
	      else dfs(graph_for_functor, v, idx);
	    }
	  else push_CTL(stack, g);
	}
    }

  unmark_CTL(f);
  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

static void add_to_varset_for_functor(Variable v, unsigned idx)
{
  VarSet tmp;

  tmp = varset_for_functor;
  varset_for_functor = add_VarSet(tmp, v, idx);
  free_VarSet(tmp);
}

/*------------------------------------------------------------------------*/

static void gen_dependencies_Graph_Assignment(Assignment a)
{
  _gen_dependencies_Graph_Assignment(a -> variable, a -> init, a -> inits);
  _gen_dependencies_Graph_Assignment(a -> variable, a -> next, a -> nexts);
  _gen_dependencies_Graph_Assignment(a -> variable, a -> define, a -> defines);
}

/*------------------------------------------------------------------------*/

static VarSet coi_with_graph_dependencies()
{
  VarSet res;

  init_DFS(graph_for_functor);
  dfs_coi_Graph(spec);

  /* Have to include these, since I am to lazy to code the correct version.
   * This means that whenever you write INIT, TRANS or INVAR section then
   * COI will include all the variables mentionend in these section into
   * the cone of influence of the specification.
   */
  dfs_coi_Graph(init);
  dfs_coi_Graph(trans);
  dfs_coi_Graph(invar);

  varset_for_functor = mt_VarSet();
  forall_reachable_Graph(graph_for_functor, add_to_varset_for_functor);
  exit_DFS(graph_for_functor);
  free_Graph(graph_for_functor);
  graph_for_functor = 0;
  res = varset_for_functor;
  varset_for_functor = 0;

  return res;
}

/*------------------------------------------------------------------------*/

static void coi_with_graph()
{
  VarSet s;

  graph_for_functor = new_Graph();

  forall_Assignment(gen_dependencies_Graph_Assignment);
  s = coi_with_graph_dependencies();
  print_verbose("SPEC depends on %u (flattened) variables\n", size_VarSet(s));
  coi_remove_assignments(s);
  mark_unused_variables(s);

# ifdef VERYDEBUG
    if(debug)
      {
	printf("%sSPEC dependencies: ", comment_string);
	print_VarSet(s);
	putc('\n', stdout);
      }
# endif

  free_VarSet(s);
}

/*------------------------------------------------------------------------*/

void coi()
{
  if(!num_init_assignments &&
     !num_next_assignments &&
     !num_define_assignments) return;

  start_verbose("cone of influence reduction");
  if(use_varset) coi_with_varset();
  else coi_with_graph();
  end_verbose("cone of influence reduction");
}
