#include <stdlib.h>

#include "assign.h"
#include "ctl.h"
#include "list.h"
#include "prop.h"
#include "util.h"

/*------------------------------------------------------------------------*/

extern CTL init, trans, invar, spec;
extern List fairness;

/*------------------------------------------------------------------------*/

unsigned num_init_assignments = 0;
unsigned num_next_assignments = 0;
unsigned num_define_assignments = 0;
int bitwise_assignment = 0;

/*------------------------------------------------------------------------*/

static Assignment first_assignment = 0, last_assignment = 0;

/*------------------------------------------------------------------------*/

static void init_formulas(
  Variable v, unsigned idx, 
  CTL * bit, CTL ** bits, int * counter, CTL formula)
{
  if(idx == v -> size || v -> size == 1)
    {
      assert(v -> size > 1 || idx == 0);

      if(*bits || *bit) fatal(POSITION, "`%s' assigned twice\n", v -> name);
      *bit = formula;
    }
  else
    {
      if(*bit) fatal(POSITION,
	         "`%s' and `%s[%u]' assigned\n", v -> name, v -> name, idx);

      if(*bits)
	{
	  if((*bits)[idx]) 
	    fatal(POSITION, "`%[%u]' assigned twice\n", v -> name, idx);
	}
      else *bits = (CTL*) calloc(v -> size, sizeof(CTL));

      bitwise_assignment = 1;
      (*bits)[idx] = formula;
    }

  (*counter)++;
}

/*------------------------------------------------------------------------*/

static Assignment new_Assignment(
  Assignment_Type type, Variable v, unsigned idx, CTL formula)
{
  Assignment res;

  res = (Assignment) malloc(sizeof(struct Assignment_));
  res -> variable = v;
  res -> idx = idx;
  res -> next_chronological = 0;
  res -> prev_chronological = 0;

  res -> init = 0;
  res -> next = 0;
  res -> define = 0;
  res -> inits = 0;
  res -> nexts = 0;
  res -> defines = 0;

  if(last_assignment)
    {
      res -> prev_chronological = last_assignment;
      last_assignment -> next_chronological = res;
      last_assignment = res;
    }
  else 
    {
      first_assignment = last_assignment = res;
    }

  switch(type)
    {
      case INIT_Assignment:
	init_formulas(
	  v, idx, &res -> init, &res -> inits, &num_init_assignments, formula);
        break;

      case NEXT_Assignment:
	init_formulas(
	  v, idx, &res -> next, &res -> nexts, &num_next_assignments, formula);
        break;

      case DEFINE_Assignment:
	init_formulas(
	  v, idx, &res -> define, &res -> defines,
	  &num_define_assignments, formula);
        break;

      default:
        fatal(POSITION, "non valid type\n");
	break;
    }

  return res;
}

/*------------------------------------------------------------------------*/

void free_Assignment(Assignment a)
{
  unsigned i;

  assert(a -> variable);
  assert(a -> variable -> assignment == a);

  /* First dequeue of doubly linked list
   */

  if(a -> prev_chronological)
    {
      a -> prev_chronological -> next_chronological = a -> next_chronological;
    }
  else first_assignment = a -> next_chronological;

  if(a -> next_chronological)
    {
      a -> next_chronological -> prev_chronological = a -> prev_chronological;
    }
  else last_assignment = a -> prev_chronological;

  a -> variable -> assignment = 0;

  if(a -> init)
    {
      num_init_assignments--;
      free_CTL(a -> init);
    }
  else
  if(a -> inits)
    {
      for(i = 0; i < a -> variable -> size; i++)
	if(a -> inits[i])
	  {
	    free_CTL(a -> inits[i]);
	    num_init_assignments--;
	  }
      
      free(a -> inits);
    }

  if(a -> next)
    {
      num_next_assignments--;
      free_CTL(a -> next);
    }
  else
  if(a -> nexts)
    {
      for(i = 0; i < a -> variable -> size; i++)
	if(a -> nexts[i])
	  {
	    free_CTL(a -> nexts[i]);
	    num_next_assignments--;
	  }
      
      free(a -> nexts);
    }


  if(a -> define)
    {
      num_define_assignments--;
      free_CTL(a -> define);
    }
  else
  if(a -> defines)
    {
      for(i = 0; i < a -> variable -> size; i++)
	if(a -> defines[i])
	  {
	    free_CTL(a -> defines[i]);
	    num_define_assignments--;
	  }
      
      free(a -> defines);
    }

  free(a);
}

/*------------------------------------------------------------------------*/

void init_Assignment()
{
  num_init_assignments = num_next_assignments = num_define_assignments = 0;
  first_assignment = last_assignment = 0;
}

/*------------------------------------------------------------------------*/

void exit_Assignment()
{
  print_verbose("assignments: %u init, %u next, %u define\n",
    num_init_assignments, num_next_assignments, num_define_assignments);
  forall_Assignment(free_Assignment);

  assert(!num_init_assignments);
  assert(!num_next_assignments);
  assert(!num_define_assignments);

  first_assignment = last_assignment = 0;
}

/*------------------------------------------------------------------------*/

Assignment find_Assignment(Variable v)
{
  return v -> assignment;
}

/*------------------------------------------------------------------------*/

int contains_Assignment(Assignment_Type type, Variable v, unsigned idx)
{
  int res;
  Assignment a;
  CTL formula, * formulas;

  a = v -> assignment;

  if(a)
    {
      if((type == DEFINE_Assignment && 
	    (a -> init || a -> inits || a -> next || a -> nexts))
	  ||
	  (type != DEFINE_Assignment && (a -> define || a -> defines)))
	{
	  res = 1;
	}
      else
	{
	  switch(type)
	    {
	      case INIT_Assignment:

		formula = a -> init;
		formulas = a -> inits;
		break;

	      case NEXT_Assignment:

		formula = a -> next;
		formulas = a -> nexts;
		break;

	      case DEFINE_Assignment:

		formula = a -> define;
		formulas = a -> defines;
		break;
	      
	      default:
		fatal(POSITION, "non valid type\n");
		formula = 0;
		formulas = 0;
		break;
	    }

	  if(v -> size == idx) res = formula || formulas;
	  else res = formula || (formulas && formulas[idx]);
	}
    }
  else res = 0;

  return res;
}

/*------------------------------------------------------------------------*/

void insert_Assignment(
  Assignment_Type type, Variable v, unsigned idx, CTL formula)
{
  Assignment a;

  a = v -> assignment;

  if(a)
    {
      switch(type)
        {
	  case INIT_Assignment:
	    init_formulas(
	      v, idx, &a -> init, &a -> inits, &num_init_assignments, formula);
	    break;

	  case NEXT_Assignment:
	    init_formulas(
	      v, idx, &a -> next, &a -> nexts, &num_next_assignments, formula);
	    break;

	  case DEFINE_Assignment:
	    init_formulas(v, idx, &a -> define, &a -> defines, 
	      &num_define_assignments, formula);
	    break;

	  default:
	    fatal(POSITION, "non valid type\n");
	    break;
	}
    }
  else v -> assignment = new_Assignment(type, v, idx, formula);
}

/*------------------------------------------------------------------------*/
/* This forall construct has two nice properties:
 *
 *   (1) `f' might even manipulate f -> next_chronological, e.g. freeing
 *       the memory of a.
 *
 *   (2) `f' might add new assignments that later not traversed!
 */

void forall_Assignment(void(*f)(Assignment))
{
  Assignment a, tmp, end;
  int finished;

  if(first_assignment)
    {
      a = first_assignment;
      end = last_assignment;
      
      do {
	finished = (a == end);
	tmp = a -> next_chronological;
	f(a);
	a = tmp;
      } while(!finished);
    }
}

/*------------------------------------------------------------------------*/

static CTL _gen_assignments(CTL_Tag tag, Assignment_Type type, CTL f)
{
  CTL res, f0, f1;

  if(f -> is_marked) res = true_CTL();
  else
    {
      f -> is_marked = 1;
      switch(f -> tag)
	{
	  case AND_Tag:

	    f0 = _gen_assignments(tag, type, f -> data.arg[0]);
	    f1 = _gen_assignments(tag, type, f -> data.arg[1]);
	    res = binary_CTL(AND_Tag, f0, f1);
	    break;
	  
	  case NOT_Tag:

	    f0 = f -> data.arg[0];
	    if(f0 -> tag == tag &&
	        !contains_Assignment(
		  type, f0 -> data.base.variable, f0 -> data.base.idx))
	      {
		insert_Assignment(
		  type,
		  f0 -> data.base.variable,
		  f0 -> data.base.idx,
		  number_CTL(0));

		res = true_CTL();
	      }
	    else res = copy_CTL(f);
	    break;
	  
	  case EQUIV_Tag:

	    f0 = f -> data.arg[0];
	    f1 = f -> data.arg[1];
	    if(f0 -> tag == tag &&
	        (type != NEXT_Assignment || f1 -> is_current) &&
	        !contains_Assignment(
		  type, f0 -> data.base.variable, f0 -> data.base.idx))
	      {
		insert_Assignment(
		  type,
		  f0 -> data.base.variable,
		  f0 -> data.base.idx,
		  copy_CTL(f1));

		res = true_CTL();
	      }
	    else
	    if(f1 -> tag == tag &&
	        (type != NEXT_Assignment || f0 -> is_current) &&
	        !contains_Assignment(
		  type, f1 -> data.base.variable, f1 -> data.base.idx))
	      {
		insert_Assignment(
		  type,
		  f1 -> data.base.variable,
		  f1 -> data.base.idx,
		  copy_CTL(f0));

		res = true_CTL();
	      }
	    else res = copy_CTL(f);
	    break;
	  
	  case EQUAL_Tag:

	    f0 = f -> data.arg[0];
	    f1 = f -> data.arg[1];
	    if(f0 -> tag == tag &&
	        (type != NEXT_Assignment || f1 -> is_current) &&
	        !contains_Assignment(
		  type, f0 -> data.base.variable, f0 -> data.base.idx))
	      {
		insert_Assignment(
		  type,
		  f0 -> data.base.variable,
		  f0 -> data.base.idx,
		  copy_CTL(f1));

		res = true_CTL();
	      }
	    else
	    if(f1 -> tag == tag &&
	        (type != NEXT_Assignment || f0 -> is_current) &&
	        !contains_Assignment(
		  type, f1 -> data.base.variable, f1 -> data.base.idx))
	      {
		insert_Assignment(
		  type,
		  f1 -> data.base.variable,
		  f1 -> data.base.idx,
		  copy_CTL(f0));

		res = true_CTL();
	      }
	    else res = copy_CTL(f);
	    break;
	  
	  case NEXT_Tag:
	  case CURRENT_Tag:

	    if(f -> tag == tag &&
	        !contains_Assignment(
		  type, f -> data.base.variable, f -> data.base.idx))
	      {
		insert_Assignment(
		  type,
		  f -> data.base.variable,
		  f -> data.base.idx,
		  number_CTL(1));

		res = true_CTL();
	      }
	    else res = copy_CTL(f);
	    break;
          
	  default:
	    res = copy_CTL(f);
	    break;
	}
    }

  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_init_assignments(CTL f)
{
  CTL res;
  
  res =_gen_assignments(CURRENT_Tag, INIT_Assignment, f);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_next_assignments(CTL f)
{
  CTL res;
  
  res = _gen_assignments(NEXT_Tag, NEXT_Assignment, f);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_invar_assignments(CTL f)
{
  CTL res;

  /* This is more complicated since we have to ensure noncyclicity
   * of the added `define' assignments.
   */
  res = copy_CTL(f);

  return res;
}

/*------------------------------------------------------------------------*/

void gen_assignments()
{
  CTL tmp;
  unsigned old;
  double old_size, old_nodes, new_size, new_nodes;

  if(is_true_CTL(init) && is_true_CTL(trans) && is_true_CTL(invar)) return;

  start_verbose("generating assignments");

  if(!is_true_CTL(init))
    {
      old = num_init_assignments;
      old_size = init -> size;
      old_nodes = count_CTL(init);

      tmp = gen_init_assignments(init);
      unmark_CTL(init);
      free_CTL(init);
      init = tmp;

      print_verbose(
	"generated %u `init' assignments\n", num_init_assignments - old);

      new_size = init -> size;
      new_nodes = count_CTL(init);

      print_verbose("INIT: size %.0f (%.0f nodes = %.0f%%) x %.2f (%.2f)\n",
	new_size, new_nodes, 100.0 * new_nodes / new_size,
	new_size / old_size, new_nodes / old_nodes);
    }

  if(!is_true_CTL(trans))
    {
      old = num_next_assignments;
      old_size = trans -> size;
      old_nodes = count_CTL(trans);

      tmp = gen_next_assignments(trans);
      unmark_CTL(trans);
      free_CTL(trans);
      trans = tmp;
      print_verbose(
	"generated %u `next' assignments\n", num_next_assignments - old);

      new_size = trans -> size;
      new_nodes = count_CTL(trans);

      print_verbose("TRANS: size %.0f (%.0f nodes = %.0f%%) x %.2f (%.2f)\n",
	new_size, new_nodes, 100.0 * new_nodes / new_size,
	new_size / old_size, new_nodes / old_nodes);
    }

  if(!is_true_CTL(invar))
    {
      old = num_define_assignments;
      old_size = invar -> size;
      old_nodes = count_CTL(invar);

      tmp = gen_invar_assignments(invar);
      unmark_CTL(invar);
      free_CTL(invar);
      invar = tmp;
      print_verbose(
	"generated %u `define' assignments\n", num_define_assignments - old);

      new_size = invar -> size;
      new_nodes = count_CTL(invar);

      print_verbose("INVAR: size %.0f (%.0f nodes = %.0f%%) x %.2f (%.2f)\n",
	new_size, new_nodes, 100.0 * new_nodes / new_size,
	new_size / old_size, new_nodes / old_nodes);
    }

  end_verbose("extracted assignments from TRANS, INIT and INVAR");
}

/*------------------------------------------------------------------------*/

void remove_if_mt_Assignment(Assignment a)
{
  Variable v;
  unsigned i;
  int remove;

  v = a -> variable;

  if(!a -> init && !a -> next && !a -> define)
    {
      remove = 1;

      if(remove && a -> inits)
	{
          for(i = 0; remove && i < v -> size; i++)
	    remove = !a -> inits[i];
	}

      if(remove && a -> nexts)
	{
          for(i = 0; remove && i < v -> size; i++)
	    remove = !a -> nexts[i];
	}

      if(remove && a -> defines)
	{
          for(i = 0; remove && i < v -> size; i++)
	    remove = !a -> defines[i];
	}

      if(remove) free_Assignment(a);
    }
}

/*------------------------------------------------------------------------*/

unsigned num_not_flattened_assignments(Assignment_Type type)
{
  Assignment a, tmp;
  unsigned res, i;
  CTL formula, * formulas;

  res = 0;
  for(a = first_assignment; a; a = tmp)
    {
      tmp = a -> next_chronological;

      switch(type)
        {
	  case INIT_Assignment:
	    formula = a -> init;
	    formulas = a -> inits;
	    break;

	  case NEXT_Assignment:
	    formula = a -> next;
	    formulas = a -> nexts;
	    break;

	  case DEFINE_Assignment:
	    formula = a -> define;
	    formulas = a -> defines;
	    break;

	  default:
	    formula = 0;
	    formulas = 0;
	    fatal(POSITION, "non valid type\n");
	    break;
	}
      
      if(formula && (a -> variable -> size > 1 || !formula -> is_flattened))
	{
	  res++;
	}

      if(formulas)
        {
	  for(i = 0; i < a -> variable -> size; i++)
	    if(formulas[i] && !formulas[i] -> is_flattened) res++;
	}
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static void _move_defines_to_INVAR(Assignment a)
{
  unsigned i;
  Variable v;
  CTL tmp;

  v = a -> variable;

  if(a -> define)
    {
      if(v -> size > 1)
        {
	  tmp = current_CTL(v, v -> size);
	  tmp = binary_CTL(EQUAL_Tag, tmp, a -> define);
	}
      else
        {
	  tmp = current_CTL(v, 0);
	  tmp = binary_CTL(EQUIV_Tag, tmp, extract_bit(a -> define, 0, 0));
	  free_CTL(a -> define);
	}

      invar = binary_CTL(AND_Tag, invar, tmp);
      a -> define = 0;
      num_define_assignments--;
    }
  else
  if(a -> defines)
    {
      for(i = 0; i < v -> size; i++)
        {
	  if(a -> defines[i])
	    {
	      tmp = current_CTL(v, i);
	      tmp = binary_CTL(
	        EQUIV_Tag, tmp, extract_bit(a -> defines[i], 0, 0));
	      free_CTL(a -> defines[i]);
	      a -> defines[i] = 0;
	      invar = binary_CTL(AND_Tag, invar, tmp);
	      num_define_assignments--;
	    }
	}
      
      free(a -> defines);
      a -> defines = 0;
    }

  remove_if_mt_Assignment(a);
}

/*------------------------------------------------------------------------*/

void move_defines_to_INVAR()
{
  unsigned old;

  start_verbose("moving `define' assignments to INVAR section");
  start_changing();
  old = num_define_assignments;
  forall_Assignment(_move_defines_to_INVAR);
  end_changing();
  end_verbose("moved %u assignments to INVAR section",
    old - num_define_assignments);
}

/*------------------------------------------------------------------------*/

static double init_nodes_before, init_nodes_after;
static double init_size_before, init_size_after;
static double trans_nodes_before, trans_nodes_after;
static double trans_size_before, trans_size_after;
static double invar_nodes_before, invar_nodes_after;
static double invar_size_before, invar_size_after;
static double spec_size_before, spec_size_after;
static double spec_nodes_before, spec_nodes_after;
static double fairness_nodes_before, fairness_nodes_after;
static double fairness_size_before, fairness_size_after;

/*------------------------------------------------------------------------*/

static unsigned nodes_sum_for_functor, size_sum_for_functor;

/*------------------------------------------------------------------------*/

static void count_nodes_and_size_CTL(void * f)
{
  CTL g;

  g = (CTL) f;

  nodes_sum_for_functor += count_CTL(g);
  size_sum_for_functor += g -> size;
}

/*------------------------------------------------------------------------*/

void start_changing()
{
  init_nodes_before = count_CTL(init);
  init_size_before = init -> size;
  trans_nodes_before = count_CTL(trans);
  trans_size_before = trans -> size;
  invar_nodes_before = count_CTL(invar);
  invar_size_before = invar -> size;
  spec_nodes_before = count_CTL(spec);
  spec_size_before = spec -> size;

  nodes_sum_for_functor = 0;
  size_sum_for_functor = 0;
  forall_List(fairness, count_nodes_and_size_CTL);
  fairness_nodes_before = nodes_sum_for_functor;
  fairness_size_before = size_sum_for_functor;
}

/*------------------------------------------------------------------------*/

void _end_changing(
  const char * section,
  double size_before, double size_after,
  double nodes_before, double nodes_after)
{
  if(size_after == size_before && nodes_after == nodes_before)
    {
      /*
      print_verbose("%s: no change\n", section);
      */
    }
  else
    {
      print_verbose("%s: size %.0f (%0.f nodes = %.0f%%) x %.2f (%.2f)\n",
	section, size_after, nodes_after, 100.0 * nodes_after / size_after,
	size_after / size_before, nodes_after / nodes_before);
    }
}

/*------------------------------------------------------------------------*/

void end_changing()
{
  init_nodes_after = count_CTL(init);
  init_size_after = init -> size;
  trans_nodes_after = count_CTL(trans);
  trans_size_after = trans -> size;
  invar_nodes_after = count_CTL(invar);
  invar_size_after = invar -> size;
  spec_nodes_after = count_CTL(spec);
  spec_size_after = spec -> size;

  nodes_sum_for_functor = 0;
  size_sum_for_functor = 0;
  forall_List(fairness, count_nodes_and_size_CTL);
  fairness_nodes_before = nodes_sum_for_functor;
  fairness_size_before = size_sum_for_functor;

  if(spec_size_after == spec_size_before &&
     spec_nodes_after == spec_nodes_before &&
     invar_size_after == invar_size_before &&
     invar_nodes_after == invar_nodes_before &&
     trans_size_after == trans_size_before &&
     trans_nodes_after == trans_nodes_before &&
     init_size_after == init_size_before &&
     init_nodes_after == init_nodes_before &&
     fairness_size_after == fairness_size_before &&
     fairness_nodes_after == fairness_nodes_before)
    {
      print_verbose("no change\n");
    }
  else
    {
      _end_changing("INIT", init_size_before, init_size_after,
	init_nodes_before, init_nodes_after);

      _end_changing("TRANS", trans_size_before, trans_size_after,
	trans_nodes_before, trans_nodes_after);

      _end_changing("INVAR", invar_size_before, invar_size_after,
	invar_nodes_before, invar_nodes_after);

      _end_changing("FAIRNESS", fairness_size_before, fairness_size_after,
	fairness_nodes_before, fairness_nodes_after);

      _end_changing("SPEC", spec_size_before, spec_size_after,
	spec_nodes_before, spec_nodes_after);
    }
}
