#include <stdio.h>
#include <stdlib.h>

#include "varset.h"
#include "util.h"

/*------------------------------------------------------------------------*/

extern int verbose;

/*------------------------------------------------------------------------*/

static VarSet * varsets = 0;
static unsigned num_varsets = 0, size_varsets = 0, max_varsets = 0;
static unsigned visited_varsets = 0, lookups_varsets = 0;

/*------------------------------------------------------------------------*/

void init_VarSet()
{
  size_varsets = next_size(100);
  varsets = (VarSet*) calloc(size_varsets, sizeof(VarSet));
  num_varsets = max_varsets = visited_varsets = lookups_varsets = 0;
}

/*------------------------------------------------------------------------*/

void exit_VarSet()
{
  unsigned i;
  VarSet p, tmp;
  double avg;

  if(verbose)
    {
      if(lookups_varsets)
        {
	  avg = ((double)visited_varsets) / ((double)lookups_varsets);
	  print_verbose("varsets: num %u, max %u, size %u, avg %.2f\n",
	    num_varsets, max_varsets, size_varsets, avg);
	}
      else
        {
	  print_verbose("varsets: num %u, max %u, size %u\n",
	    num_varsets, max_varsets, size_varsets);
	}
    }

  if(varsets)
    {
      for(i = 0; i < size_varsets; i++)
        {
	  for(p = varsets[i]; p; p = tmp)
	    {
	      tmp = p -> next;
	      free(p);
	    }
	}
    }
  
  free(varsets);
  varsets = 0;
  num_varsets = size_varsets = 0;
}


/*------------------------------------------------------------------------*/

static unsigned hash_VarSet(VarSet s, Variable v, unsigned idx)
{
  unsigned res;

  res = ((unsigned) s) * 199999;
  res = (res + ((unsigned)v)) * 49999;
  res = (res + ((unsigned)idx)) * 9973;

  return res;
}

/*------------------------------------------------------------------------*/

VarSet mt_VarSet()
{
  return (VarSet) 0;
}

/*------------------------------------------------------------------------*/

void free_VarSet(VarSet s)
{
  VarSet t;
  unsigned h;
  VarSet * p;

  while(s && --s -> ref == 0)
    {
      t = s -> child;

      h = hash_VarSet(s -> child, s -> variable, s -> idx);
      h %= size_varsets;
      for(p = &varsets[h]; *p && *p != s; p = &(*p) -> next)
	;

      if(!*p) fatal(POSITION, "could not find VarSet in unique table\n");

      *p = s -> next;

      free(s);
      num_varsets--;
      s = t;
    }
  
  assert(!s || s -> ref > 0);
}

/*------------------------------------------------------------------------*/

VarSet copy_VarSet(VarSet s)
{
  VarSet res;

  if(s)
    {
      assert(s -> ref);
      s -> ref++;
      res = s;
    }
  else res = (VarSet) 0;

  return res;
}

/*------------------------------------------------------------------------*/

static void resize_VarSet()
{
  VarSet * old_varsets, p, tmp;
  unsigned old_size, i, h;

  old_size = size_varsets;
  old_varsets = varsets;
  size_varsets = next_size(num_varsets);
  varsets = (VarSet*) calloc(size_varsets, sizeof(VarSet));

  for(i = 0; i < old_size; i++)
    {
      for(p = old_varsets[i]; p; p = tmp)
        {
	  tmp = p -> next;
	  h = hash_VarSet(p -> child, p -> variable, p -> idx);
	  h %= size_varsets;
	  p -> next = varsets[h];
	  varsets[h] = p;
	}
    }
  
  free(old_varsets);
}

/*------------------------------------------------------------------------*/

static VarSet new_VarSet(VarSet s, Variable v, unsigned idx)
{
  VarSet * p, res;
  unsigned h;

  if(v -> size == idx)
    fatal(POSITION, "array dependencies not implemented\n");

  if(num_varsets >= 3 * size_varsets) resize_VarSet();

  lookups_varsets++;
  h = hash_VarSet(s, v, idx);
  h %= size_varsets;

  for(p = &varsets[h]; *p; p = &(*p) -> next)
    {
      visited_varsets++;
      if((*p) -> child == s &&
         (*p) -> variable == v &&
	 (*p) -> idx == idx) break;
    }
  
  if(*p) res = copy_VarSet(*p);
  else
    {
      res = (VarSet) malloc(sizeof(struct VarSet_));
      *p = res;

      res -> ref = 1;
      res -> child = copy_VarSet(s);
      res -> variable = v;
      res -> idx = idx;
      res -> next = 0;

      num_varsets++;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static int cmp_variables(Variable u, unsigned i, Variable v, unsigned j)
{
  int res;

  if(u == v)
    {
      if(i == j) return 0;
      if(i < j) return -1;
      return 1;
    }

  res = strcmp(u -> name, v -> name);

  return res;
}

/*------------------------------------------------------------------------*/

static int cmp_top_VarSet(VarSet s, VarSet t)
{
  int res;

  if(s == t) return 0;
  if(!s) return -1;
  if(!t) return 1;

  res = cmp_variables(s -> variable, s -> idx, t -> variable, t -> idx);

  return res;
}

/*------------------------------------------------------------------------*/

VarSet add_VarSet(VarSet s, Variable v, unsigned idx)
{
  int cmp;
  VarSet res, tmp;

  if(s)
    {
      cmp = cmp_variables(s -> variable, s -> idx, v, idx);
      if(cmp == 0) 
	{
	  /* Skip variables that are already in the set.
	   */
	  res = copy_VarSet(s);
	}
      else
      if(cmp < 0)
	{
	  /* Toplevel variable of `s' is smaller than (v,idx).
	   * Thus we insert it into the child of `s' and combine
	   * the result with the toplevel variable of `s'.
	   */
	  tmp = add_VarSet(s -> child, v, idx);
	  res = new_VarSet(tmp, s -> variable, s -> idx);
	  free_VarSet(tmp);
	}
      else 
	{
	  /* Toplevel variable of `s' is greater than (v, idx)
	   * and we can simply put it on top of `s'.
	   */
	  res = new_VarSet(s, v, idx);
	}
    }
  else res = new_VarSet(0, v, idx);

  return res;
}

/*------------------------------------------------------------------------*/

VarSet join_VarSet(VarSet s, VarSet t)
{
  VarSet res, tmp;
  Variable v;
  unsigned idx;
  int cmp;

  if(!s || s == t) return copy_VarSet(t);
  if(!t) return copy_VarSet(s);

  cmp = cmp_top_VarSet(s, t);

  if(cmp == 0)
    {
      v = s -> variable;
      idx = s -> idx;
      tmp = join_VarSet(s -> child, t -> child);
    }
  else
  if(cmp < 0)
    {
      v = s -> variable;
      idx = s -> idx;
      tmp = join_VarSet(s -> child, t);
    }
  else
    {
      v = t -> variable;
      idx = t -> idx;
      tmp = join_VarSet(s, t -> child);
    }
  
  res = add_VarSet(tmp, v, idx);
  free_VarSet(tmp);

  return res;
}

/*------------------------------------------------------------------------*/

int contains_VarSet(VarSet s, Variable v, unsigned idx)
{
  VarSet t;

  for(t  = s;  t; t = t -> child)
    if(t -> variable == v && t -> idx == idx) return 1;

  return 0;
}

/*------------------------------------------------------------------------*/

void print_VarSet(VarSet s)
{
  while(s)
    {
      fputs(s -> variable -> name, stdout);

      if(s -> variable -> size > 1 &&
	 s -> variable -> size != s -> idx)
        {
	  putc('[', stdout);
	  putu(s -> idx, stdout);
	  putc(']', stdout);
	}

      if(s -> child) putc(' ', stdout);
      s = s -> child;
    }
}

/*------------------------------------------------------------------------*/

void printnl_VarSet(VarSet s)
{
  print_VarSet(s);
  putc('\n', stdout);
}

/*------------------------------------------------------------------------*/

int intersects_VarSet(VarSet a, VarSet b)
{
  int res, cmp;

  res = 0;
  while(a && b && !res)
    {
      cmp = cmp_top_VarSet(a, b);
      if(cmp == 0) res = 1;
      else if(cmp < 0) a = a -> child;
      else b = b -> child;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

int subset_VarSet(VarSet a, VarSet b)
{
  int res, cmp;

  res = 1;
  while(a && res)
    {
      if(b)
	{
	  cmp = cmp_top_VarSet(a, b);
	  if(cmp == 0)
	    {
	      a = a -> child;
	      b = b -> child;
	    }
	  else if(cmp < 0) res = 0;
	  else b = b -> child;
	}
      else res = 0;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static unsigned size_arrays_VarSet(VarSet a)
{
  unsigned res;
  VarSet b;
  Variable v;

  for(res = 0, b = a; b; b = b -> child)
    {
      v = b -> variable;
      if(v -> size == b -> idx) res += v -> size;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static unsigned size_elements_VarSet(VarSet a)
{
  unsigned res;
  VarSet b;
  Variable v;

  for(res = 0, b = a; b; b = b -> child)
    {
      v = b -> variable;
      if(b -> idx < v -> size && !contains_VarSet(a, v, v -> size)) res++;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

unsigned size_VarSet(VarSet a)
{
  unsigned res;
  
  res = size_arrays_VarSet(a) + size_elements_VarSet(a);

  return res;
}
