#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>                     

#include "config.h"
#include "util.h"

/*------------------------------------------------------------------------*/

extern int verbose;

/*------------------------------------------------------------------------*/

void fatal(char * fun, char * file, int lineno, char * fmt, ...)
{
  va_list ap;

  fprintf(stderr, "***");
  if(fun) fprintf(stderr, " in %s", fun);
  if(file) fprintf(stderr, " at %s", file);
  if(lineno) fprintf(stderr, ":%d", lineno);
  fprintf(stderr, " ");

  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);

  fflush(stderr);

  exit(1);
}

/*------------------------------------------------------------------------*/

static unsigned pow2_table[]
=
{
  0x1, 0x2, 0x4, 0x8,
  0x10, 0x20, 0x40, 0x80,
  0x100, 0x200, 0x400, 0x800,
  0x1000, 0x2000, 0x4000, 0x8000,
  0x10000, 0x20000, 0x40000, 0x80000,
  0x100000, 0x200000, 0x400000, 0x800000,
  0x1000000, 0x2000000, 0x4000000, 0x8000000,
  0x10000000, 0x20000000, 0x40000000, 0x80000000,
  0x0
};

/*------------------------------------------------------------------------*/

unsigned pow2(unsigned i)
{
  unsigned res;

  assert(sizeof(pow2_table) / sizeof(unsigned) == 33);
  if(i > 32) fatal(POSITION, "number out of range\n");
  res = pow2_table[i];

  return res;
}

/*------------------------------------------------------------------------*/

unsigned log2(unsigned e)
{
  assert(sizeof(unsigned) * 8 == 32);

  if(e < 0x2)        return 1;
  if(e < 0x4)        return 2;
  if(e < 0x8)        return 3;
  if(e < 0x10)       return 4;
  if(e < 0x20)       return 5;
  if(e < 0x40)       return 6;
  if(e < 0x80)       return 7;
  if(e < 0x100)      return 8;
  if(e < 0x200)      return 9;
  if(e < 0x400)      return 10;
  if(e < 0x800)      return 11;
  if(e < 0x1000)     return 12;
  if(e < 0x2000)     return 13;
  if(e < 0x4000)     return 14;
  if(e < 0x8000)     return 15;
  if(e < 0x10000)    return 16;
  if(e < 0x20000)    return 17;
  if(e < 0x40000)    return 18;
  if(e < 0x80000)    return 19;
  if(e < 0x100000)   return 20;
  if(e < 0x200000)   return 21;
  if(e < 0x400000)   return 22;
  if(e < 0x800000)   return 23;
  if(e < 0x1000000)  return 24;
  if(e < 0x2000000)  return 25;
  if(e < 0x4000000)  return 26;
  if(e < 0x8000000)  return 27;
  if(e < 0x10000000) return 28;
  if(e < 0x20000000) return 29;
  if(e < 0x40000000) return 30;
  if(e < 0x80000000) return 31;
  return 32;
}

/*------------------------------------------------------------------------*/

static void _tab(FILE * file, unsigned n)
{
  unsigned i;

  for(i=0; i<n; i++) putc(' ', file);
}

/*------------------------------------------------------------------------*/

void tab(unsigned n)
{
  _tab(stdout, n);
}

/*------------------------------------------------------------------------*/

double get_time()
{
  double res;
  struct rusage usage;

  (void) getrusage(RUSAGE_SELF, &usage);

  /* INCLUDE system time */

  res = usage.ru_utime.tv_usec + usage.ru_stime.tv_usec;
  res *= 1e-6;
  res += usage.ru_utime.tv_sec + usage.ru_stime.tv_sec; 
	
  return res;
}

/*------------------------------------------------------------------------*/

#define max_verbose_started 5
static unsigned verbose_started = 0;
static double verbose_start_time[max_verbose_started];

/*------------------------------------------------------------------------*/

static void _print_verbose()
{
  fprintf(stderr, "<%u> ", verbose_started);
  _tab(stderr, 2 * verbose_started);
}

/*------------------------------------------------------------------------*/

void start_verbose(char * fmt, ...)
{
  double start_time;
  va_list ap;

  if(verbose)
    {
      if(verbose_started >= max_verbose_started)
	fatal(POSITION,
	  "maximal number of nested verbosity outputs exceeded\n");

      start_time = get_time();
      _print_verbose();
      fprintf(stderr, "START ");
      verbose_start_time[verbose_started++] = start_time;
      va_start(ap, fmt);
      vfprintf(stderr, fmt, ap);
      va_end(ap);
      fprintf(stderr, " at %.2f sec\n", start_time);
      fflush(stderr);
    }
}

/*------------------------------------------------------------------------*/

void end_verbose(char * fmt, ...)
{
  va_list ap;
  double diff;

  if(verbose)
    {
      if(verbose_started==0) fatal(POSITION, "verbose level reached zero\n");
      verbose_started--;
      _print_verbose();
      fprintf(stderr, "END ");
      va_start(ap, fmt);
      vfprintf(stderr, fmt, ap);
      va_end(ap);

      diff = get_time() - verbose_start_time[verbose_started];
      if(diff < 0) diff = 0.0;
      fprintf(stderr, " in %.2f sec\n", diff);
      fflush(stderr);
    }
}

/*------------------------------------------------------------------------*/

void print_verbose(char * fmt, ...)
{
  va_list ap;

  if(verbose)
    {
      _print_verbose();
      va_start(ap, fmt);
      vfprintf(stderr, fmt, ap);
      va_end(ap);
    }
}

/*------------------------------------------------------------------------*/

void putu(unsigned u, FILE * file)
{
  char buffer[30], * p;
  unsigned tmp;

  if(u)
    {
      buffer[30] = '\0';
      for(p = buffer + 29; u; p--, u = tmp)
	{
	  tmp = u / 10;
	  *p = u - 10 * tmp + '0';
	}

      fputs(p + 1, file);
    }
  else putc('0', file);
}

/*------------------------------------------------------------------------*/

static unsigned primes[] =
{
  3, 7, 17, 37, 79, 163, 331, 673, 1361, 2729, 5471, 
  10949, 21911, 43853, 87719, 175447, 350899, 701819, 1403641, 2807303, 
  5614657, 11229331, 22458671, 44917381, 89834777, 179669557, 359339171, 
  718678369, 1437356741
};

/*------------------------------------------------------------------------*/

unsigned next_size(unsigned n)
{
  unsigned i, max = sizeof(primes)/sizeof(unsigned);

  if(primes[0] < n)
    {
      for(i = 1; n > primes[i] && i < max; i++)
        ;

      if(i >= max) fatal(POSITION, "maximal number of variables exceeded\n");

      return n - primes[i - 1] < primes[i] - n ? primes[i - 1] : primes[i];
    }
  else return primes[0];
}

/*------------------------------------------------------------------------*/

static unsigned num_progress = 0;

/*------------------------------------------------------------------------*/

void start_progress()
{
  if(verbose)
    {
      num_progress = 0;
    }
}

/*------------------------------------------------------------------------*/

void progress()
{
  if(verbose)
    {
      if(num_progress)
	{
	  if(num_progress == 1)
	    {
	      _print_verbose();
	      fputs("..", stderr);
	    }
	  else putc('.', stderr);

	  if(num_progress > 74 - 3 - 2 * verbose_started)
	    {
	      num_progress = 0;
	      putc('\n', stderr);
	    }
	  else num_progress++;

	  fflush(stderr);
	}
      else num_progress = 1;
    }
}

/*------------------------------------------------------------------------*/

void end_progress()
{
  if(verbose)
    {
      if(num_progress > 1)
	{
	  fputs(".\n", stderr);
	  fflush(stderr);
	}
      
      num_progress = 0;
    }
}
