#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

/*------------------------------------------------------------------------*/

typedef struct Anchor_ Anchor;
typedef struct Assignment_ * Assignment;
typedef struct Link_ Link;
typedef struct TimedVariable_ * TimedVariable;
typedef struct Variable_ * Variable;

/*------------------------------------------------------------------------*/

struct Link_ { Link * prev, * next; };
struct Anchor_ { Link * first, * last; };

/*------------------------------------------------------------------------*/

struct Variable_
{
  Link link;
  char * name;
  Anchor timed;
};

/*------------------------------------------------------------------------*/

struct TimedVariable_
{
  Link link;
  Variable variable;
  unsigned time;
  unsigned long value;
  Anchor assignment;
};

/*------------------------------------------------------------------------*/

struct Assignment_
{
  Link link;
  TimedVariable variable;
  unsigned pos, idx, value;
};

/*------------------------------------------------------------------------*/

static Anchor variables = { 0, 0 };
static unsigned max_time;

/*------------------------------------------------------------------------*/

static Assignment * indices = 0;
static unsigned indices_size;

/*------------------------------------------------------------------------*/

static void enqueue(Anchor * anchor, Link * link)
{
  assert(!anchor -> first == !anchor -> last);
 
  if(anchor -> last)
    {
      assert(!anchor -> last -> next);
 
      link -> prev = anchor -> last;
      link -> prev -> next = link;
      link -> next = 0;
      anchor -> last = link;
    }
  else
    {
      anchor -> first = anchor -> last = link;
      link -> prev = link -> next = 0;
    }
 
  assert(!anchor -> first == !anchor -> last);
}

/*------------------------------------------------------------------------*/

static void insert_variable_aux(
  char * name, unsigned time, unsigned pos, unsigned idx)
{
  TimedVariable q;
  Assignment a;
  Variable p;
  unsigned i;

  for(p = (Variable) variables.first;
      p && strcmp(p -> name, name) != 0;
      p = (Variable) p -> link.next)
    ;
  
  if(!p)
    {
      p = (Variable) malloc(sizeof(struct Variable_));
      enqueue(&variables, &p -> link);
      p -> name = strdup(name);
      p -> timed.first = p -> timed.last = 0;
    }
  
  for(q = (TimedVariable) p -> timed.first;
      q && q -> time != time;
      q = (TimedVariable) q -> link.next)
    ;
  
  if(!q)
    {
      q = (TimedVariable) malloc(sizeof(struct TimedVariable_)); 
      enqueue(&p -> timed, &q -> link);
      q -> variable = p;
      q -> time = time;
      q -> assignment.first = q -> assignment.last = 0;

      if(q -> time + 1 > max_time) max_time = q -> time + 1;
    }
  
  assert(q -> variable == p);
  
  for(a = (Assignment) q -> assignment.first;
      a && a -> pos != pos;
      a = (Assignment) a -> link.next)
    ;

  if(!a)
    {
      a = (Assignment) malloc(sizeof(struct Assignment_));
      enqueue(&q -> assignment, &a -> link);
      a -> variable = q;

      if(pos >= 8 * sizeof(unsigned long))
        {
	  fprintf(stderr,
	    "*** position %u to %s is too large\n",
	    pos,
	    name);
	  exit(1);
	}

      a -> pos = pos;
      a -> idx = idx;
      a -> value = 0;

      if(!indices)
        {
	  indices = (Assignment*)
	    malloc(sizeof(Assignment) * (indices_size = 2 * idx));
	  
	  for(i = 0; i < indices_size; i++) indices [ i ] = 0;
	}

      while(indices_size <= idx)
	{
	  indices = (Assignment*)
	    realloc(indices, sizeof(Assignment) * (indices_size * 2));
	  
	  for(i = 0; i < indices_size; i++) indices [ i + indices_size ] = 0;

	  indices_size *= 2;
	}
      
      assert(!indices [ idx ]);

      indices [ idx ] = a;
    }

  assert(a -> variable == q);
}

/*------------------------------------------------------------------------*/

static char * buffer = 0;
static int buffer_size = 0;

/*------------------------------------------------------------------------*/

static int read_line(FILE * file)
{
  int ch, res, pos;

  pos = 0;

  while((ch = getc(file)) != EOF && ch != '\n')
    {
      if(!buffer) buffer = (char*) malloc(buffer_size = 10000);
      else if(pos + 1 >= buffer_size)
        buffer = (char*) realloc(buffer, buffer_size *= 2);
      
      buffer[pos++] = ch;
    }
  
  assert(pos < buffer_size);

  buffer[pos] = 0;

  res = (ch != EOF || pos > 0);

  return res;
}

/*------------------------------------------------------------------------*/

static int is_decl(
  char ** name, unsigned * idx, unsigned * time, unsigned * pos)
{
  char * p, * time_prefix, * pos_start;
  int res;

  if(buffer[0] == 'c' && buffer[1] == ' ')
    {
      for(p = buffer + 2; *p && isdigit((int) *p); p++)
        ;
      
      if(*p == ' ' && p[1] == '=' && p[2] == ' ' && p[3] == 'V')
        {
	  *idx = (unsigned) atoi(buffer + 2);
	  *p = 0;
	  time_prefix = p + 4;

	  for(p = time_prefix; *p && isdigit((int) *p); p++)
	    ;
	  
	  if(*p == '_')
	    {
	      *p = 0;
	      *time = atoi(time_prefix);
	      *name = p + 1;

	      for(p = *name; *p && *p != '['; p++)
	        ;
	      
	      if(*p == '[')
	        {
		  *p = 0;	/* otherwise `name' is not correct */

		  for(pos_start = ++p; *p && *p != ']'; p++)
		    ;
		  
		  if(*p == ']')
		    {
		      *p = 0;
		      *pos = (unsigned) atoi(pos_start);
		      res = 1;
		    }
		  else res = 0;
		}
	      else 
	        {
		  *pos = 0;
		  res = 1;
		}
	    }
	  else res = 0;
	}
      else res = 0;
    }
  else res = 0;

  return res;
}

/*------------------------------------------------------------------------*/

static void insert_variable()
{
  unsigned idx, pos, time;
  char * name;

  if(is_decl(&name, &idx, &time, &pos))
     insert_variable_aux(name, time, pos, idx);
}

/*------------------------------------------------------------------------*/

static void read_cnf(FILE * file)
{
  while(read_line(file)) insert_variable();
}

/*------------------------------------------------------------------------*/

static void calculate_value()
{
  unsigned long value;
  TimedVariable t;
  Assignment a;
  Variable v;

  for(v = (Variable) variables.first; v; v = (Variable) v -> link.next)
    for(t = (TimedVariable) v -> timed.first;
        t;
        t = (TimedVariable) t -> link.next)
       {
	 for(a = (Assignment) t -> assignment.first, value = 0;
	     a;
	     a = (Assignment) a -> link.next)
	    {
	      if(a -> value) value |= (((unsigned long)1) << a -> pos);
	      else value &= ~(((unsigned long)1) << a -> pos);
	    }
	  
	  t -> value = value;
       }
}

/*------------------------------------------------------------------------*/

static void insert_assignment()
{
  char * p, * start;
  int literal, done;
  unsigned atom;

  p = buffer;
  done = 0;
  
  while(!done)
    {
      while(*p && (*p != '-') && !isdigit((int)*p))
        p++;
      
      if(*p)
        {
	  start = p;

	  if(*p == '-')
	    {
	      p++;
	      if(!*p) done = 1;
	    }
	  
	  if(*p && isdigit((int) *p))
	    {
	      while(*p && isdigit((int) *p))
		p++;
	      
	      if(!*p || isspace((int) *p))
		{
		  if(!*p) done = 1;

		  literal = atoi(start);

		  if(literal < 0) atom = (unsigned) - literal;
		  else atom = (unsigned) literal;

		  if(atom < indices_size && indices[atom])
		    {
		      if(literal < 0) indices[atom] -> value = 0;
		      else indices[atom] -> value = 1;
		    }
		}
	    }
	  
	  p++;
	}
      else done = 1;
    }
}

/*------------------------------------------------------------------------*/

static void read_assignment(FILE * file)
{
  while(read_line(file)) insert_assignment();
}

/*------------------------------------------------------------------------*/

static void print()
{
  TimedVariable p, t;
  Variable v;
  unsigned i;

  for(i = 0; i < max_time; i++)
    {
      printf("\n--- STATE %u ---\n\n", i);

      for(v = (Variable) variables.first; v; v = (Variable) v -> link.next)
        {
	  if(i)
	    {
	      for(p = (TimedVariable) v -> timed.first;
		  p && p -> time != (i - 1);
		  p = (TimedVariable) p -> link.next)
		;
	    }
	  else p = 0;

	  for(t = (TimedVariable) v -> timed.first;
	      t && t -> time != i;
	      t = (TimedVariable) t -> link.next)
	     ;

	  assert(!t || !p || t -> variable == p -> variable);
	  assert(!t || !p || t -> time  == (p -> time + 1));

	   if(t && (!p || p -> value != t -> value))
	     {
	       printf("%s = %lu\n", v -> name, t -> value);
	     }
	}
    }
}

/*------------------------------------------------------------------------*/

int main(int argc, char ** argv) 
{
  FILE * cnf_file, * assignment_file;

  if(argc != 3)
    {
      fprintf(stderr, "*** usage: scamid <cnf_file> <assignment>\n");
      exit(1);
    }
  
  if(strcmp(argv[1], "-") == 0 && strcmp(argv[2], "-") == 0)
    {
      fprintf(stderr, "*** scamid: can not have `-' for both args\n");
      exit(1);
    }
  
  if(strcmp(argv[1], "-") == 0) cnf_file  = stdin;
  else
    {
      cnf_file = fopen(argv[1], "r");
      if(!cnf_file)
	{
	  fprintf(stderr,
	    "*** scamid: could not open cnf file `%s'\n", argv[1]);
	  exit(1);
	}
    }

  if(strcmp(argv[2], "-") == 0) assignment_file = stdin;
  else
    {
      assignment_file = fopen(argv[2], "r");
      if(!assignment_file)
	{
	  fprintf(stderr,
	    "*** scamid: could not open assignment file `%s'\n", argv[2]);
	  exit(1);
	}
    }

  read_cnf(cnf_file);
  read_assignment(assignment_file);
  calculate_value();
  print();

  exit(0);
  return 0;
}
