#include <stdlib.h>

#include "assign.h"
#include "config.h"
#include "ctl.h"
#include "idx.h"
#include "list.h"
#include "prop.h"
#include "stack.h"
#include "util.h"

/*------------------------------------------------------------------------*/

extern CTL init, trans, spec, invar;
extern List fairness;
extern unsigned bound;
extern int diameter_flag, only_propositional, check_invariant, check_base;
extern int assume_property_on_the_way, exact_flag, alternative_translation;

/*------------------------------------------------------------------------*/

static void _gen_stabilizer(CTL f, CTL * res_ptr)
{
  CTL tmp, g;
  Stack stack;
  Variable v;
  unsigned i;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;
	  switch(g -> tag)
	    {
	      case CONSTANT_Tag:
	      case NUMBER_Tag:
	        break;

	      case CURRENT_Tag:
	      case NEXT_Tag:
		v = g -> data.base.variable;

		if(!v -> cache.idx)
		  {
		    for(i = 0; i < v -> size; i++)
		      {
			v -> cache.idx = 1;
			tmp = binary_CTL(
			  EQUIV_Tag, current_CTL(v, i), next_CTL(v, i));
			if(*res_ptr)
			  {
			    *res_ptr = binary_CTL(AND_Tag, *res_ptr, tmp);
			  }
			else *res_ptr = tmp;
		      }
		  }
		break;
	      
	      case AND_Tag:
	      case OR_Tag:
	      case EQUIV_Tag:
	      case IMPLIES_Tag:
		push(stack, g -> data.arg[0]);
		push(stack, g -> data.arg[1]);
		break;
	      
	      case NOT_Tag:
	      case EX_Tag:
	      case EF_Tag:
	      case EG_Tag:
	      case AX_Tag:
	      case AF_Tag:
	      case AG_Tag:
		push(stack, g -> data.arg[0]);
		break;

	      default:
		fatal(POSITION, "non valid tag\n");
		break;
	    }
	}
    }
  
  free_Stack(stack);
  unmark_CTL(f);
}

/*------------------------------------------------------------------------*/

static CTL gen_stabilizer()
{
  CTL res;

  assert(init || trans || spec || invar);

  res = 0;

  _gen_stabilizer(init, &res);
  _gen_stabilizer(trans, &res);
  _gen_stabilizer(spec, &res);
  _gen_stabilizer(invar, &res);

  assert(res);
  
  reset_idx_cache_Variable();

  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_diameter()
{
  unsigned i, j;
  CTL res, c, stab, tmp;

  assert(trans);
  assert(bound > 0);

  if(fairness) fatal(POSITION,
    "can not combine `-diameter' with FAIRNESS\n");

  for(i = 1, res = sub_prop_CTL(trans, 0, 1); i < bound; i++)
    res = binary_CTL(AND_Tag, res, sub_prop_CTL(trans, i, i+1));

  stab = gen_stabilizer();
  c = 0;

  for(i = 0; i < bound; i++)
    {
      for(j = i + 1; j <= bound; j++)
	{
	  tmp = sub_prop_CTL(stab, i, j);
	  if(c) c = binary_CTL(OR_Tag, c, tmp);
	  else c = tmp;
	}
    }
  
  res = binary_CTL(AND_Tag, res, unary_CTL(NOT_Tag, c));
  free_CTL(stab);
  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_invariant_checking_step()
{
  CTL current, next, tmp, res;

  assert(spec -> tag == AG_Tag);
  assert(spec -> data.arg[0] -> is_propositional);

  current = sub_prop_CTL(spec -> data.arg[0], 0, 1);
  next = unary_CTL(NOT_Tag, sub_prop_CTL(spec -> data.arg[0], 1, 2));

  tmp = binary_CTL(AND_Tag, current, sub_prop_CTL(trans, 0, 1));
  res = binary_CTL(AND_Tag, tmp, next);
  
  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_invariant_checking_base()
{
  CTL res;

  assert(spec -> tag == AG_Tag);
  assert(spec -> data.arg[0] -> is_propositional);

  res = binary_CTL(AND_Tag, sub_prop_CTL(init, 0, 1),
          unary_CTL(NOT_Tag, sub_prop_CTL(spec -> data.arg[0], 0, 1)));

  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_prop_spec()
{
  CTL res;
  unsigned num_nodes;

  assert(spec -> is_propositional);

  if(fairness) fatal(POSITION,
    "combination of FAIRNESS with propositional SPEC not implemented\n");

  if(bound) fatal(POSITION, 
    "it does not make sense to have k > 0 for propositional SPEC\n");

  start_verbose("(unrolling) propositional SPEC");

  res = unary_CTL(NOT_Tag, sub_prop_CTL(spec, 0, 1));
  res = binary_CTL(AND_Tag, sub_prop_CTL(init,0,1), res);
  res = binary_CTL(AND_Tag, res, sub_prop_CTL(invar,0,1));

  num_nodes = count_CTL(res);
  end_verbose(
    "unrolled SPEC: size %u (%u nodes = %.0f%%)",
     res -> size, num_nodes,
     100.0 * ((double)num_nodes) / ((double)res -> size));

  return res;
}

/*------------------------------------------------------------------------*/
/*  The basic idea to handle fairness is the following:
 *
 *                 /\
 *  s_k = s_k  &  /  \ fairness_j(s_k)
 *                 j
 *  |
 *                     /\
 *  s_k = s_{k-1}  &  /  \ (fairness_j(s_k) | fairness_j(s_{k-1}))
 *                     j
 *  |
 *
 *  ...
 *
 *  |                       l
 *                     /\  \  /
 *  s_k = s_{k-l}  &  /  \  \/    fairness_j(s_{k - i})
 *                     j   i = 0
 *
 *
 *  now calculate this over l = 0, ..., k  (with k = bound) and
 *  update list `ored_fairness', where in the l'th iteration
 *
 *    ored_fairness = 
 *
 *        l                         l                            
 *       \  /                      \  /                          
 *     {  \/ fairness_0(s_{k - i}), \/ fairness_1(s_{k - i}), ... }
 *       i = 0                    i = 0                         
 */

static List init_ored_fairness()
{
  List reversed, l, res;

  /* result should be in the same order as `fairness'
   */
  reversed = reverse_List(fairness);

  for(l = reversed, res = 0; l; l = cdr(l))
    {
      res = cons(sub_prop_CTL((CTL) car(l), bound - 1, bound), res);
    }

  free_List(reversed);

  return res;
}

/*------------------------------------------------------------------------*/

static void update_ored_fairness(List ored_fairness, unsigned i)
{
  List l0, l1;

  for(l0 = fairness, l1 = ored_fairness; l0; l0 = cdr(l0), l1 = cdr(l1))
    {
      l1 -> head = binary_CTL(OR_Tag,
	sub_prop_CTL((CTL) car(l0), i, i + 1), (CTL) l1 -> head);
    }
  
  assert(!l1);
}

/*------------------------------------------------------------------------*/

static void free_ored_fairness(List l)
{
  forall_List(l, (void(*)(void*)) free_CTL);
  free_List(l);
}

/*------------------------------------------------------------------------*/

CTL gen_fair_loop(CTL f)
{
  CTL tmp, fair_loop, stab, res;
  unsigned i, num_nodes;
  List l, ored_fairness;
  int done;

  start_verbose("adding fairness loop");

  /* If there are fairness constraints then we also need to have
   * a loop that contains all the fairness constraints.
   */
  fair_loop = false_CTL();

  ored_fairness = init_ored_fairness();
  stab = gen_stabilizer();
  i = bound - 1;
  done = 0;

  do
    {
      tmp = sub_prop_CTL(stab, bound, i);

      for(l = ored_fairness; l; l = cdr(l))
	tmp = binary_CTL(AND_Tag, tmp, copy_CTL((CTL) car(l)));
      
      fair_loop = binary_CTL(OR_Tag, fair_loop, tmp);

      if(i) update_ored_fairness(ored_fairness, i - 1);
      else done = 1;

      i--;
    }
  while(!done);
  
  free_ored_fairness(ored_fairness);
  free_CTL(stab);

  res = binary_CTL(AND_Tag, f, fair_loop);

  num_nodes = count_CTL(res);
  end_verbose(
    "added fairness: size %u (%u nodes = %.0f%%)",
    res -> size,
    num_nodes,
    100.0 * ((double)num_nodes) / ((double)res -> size));
  
  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_AG_spec()
{
  CTL res, disj, tmp;
  unsigned i, num_nodes;

  assert(spec -> tag == AG_Tag);

  if(!spec -> data.arg[0] -> is_propositional)
    fatal(POSITION, "only non nested safety properties implemented yet\n");

  start_verbose("unrolling AG");

  /*
   *
   *  not |=_k AG prop
   *
   *  <=>
   *
   *  |=_k EF !prop
   *
   *  <=>
   *
   *  init(0)
   *  &
   *  trans(0,1) & ... & trans(k-1, k)
   *  &
   *  (!prop(0) | ... !prop(k))
   *
   *  is satisfiable
   *  
   *  -----------------------
   *  If the exact_flag is set (`-exact' on the command line) then
   *  the last big disjunction is replaced by just `!prop(k)'.
   *  This is simply a toplevel case split by the user.
   */

  for(i = 1, res = sub_prop_CTL(init, 0, 1); i <= bound; i++)
    {
      res = binary_CTL(AND_Tag, res, sub_prop_CTL(trans, i - 1, i));

      if(assume_property_on_the_way)
        {
	  res = binary_CTL(
	    AND_Tag, res, sub_prop_CTL(spec -> data.arg[0], i - 1, i));
	}
    }

  if(exact_flag)
    {
      tmp = sub_prop_CTL(spec -> data.arg[0], bound, bound + 1);
      res = binary_CTL(AND_Tag, res, unary_CTL(NOT_Tag, tmp));
    }
  else
    {
      disj = false_CTL();

      for(i = 0; i <= bound; i++)
        {
	  tmp = sub_prop_CTL(spec -> data.arg[0], i, i + 1);
	  disj = binary_CTL(OR_Tag, disj, unary_CTL(NOT_Tag, tmp));
	}

      res = binary_CTL(AND_Tag, res, disj);
    }

  num_nodes = count_CTL(res);
  end_verbose(
    "unrolled AG for k = %u: size %u (%u nodes = %.0f%%)",
     bound, res -> size, num_nodes,
     100.0 * ((double)num_nodes) / ((double)res -> size));

  if(fairness) res = gen_fair_loop(res);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_loop(CTL f)
{
  CTL stab, tmp, res;
  unsigned i, num_nodes;

  assert(bound);

  stab = gen_stabilizer();
  tmp = sub_prop_CTL(stab, 0, bound);

  for(i = 1; i < bound; i++)
    tmp = binary_CTL(OR_Tag, tmp, sub_prop_CTL(stab, i, bound));

  res = binary_CTL(AND_Tag, f, tmp);
  free_CTL(stab);

  num_nodes = count_CTL(res);
  end_verbose(
    "added loop: size %u (%u nodes = %.0f%%)",
    res -> size,
    num_nodes,
    100.0 * ((double)num_nodes) / ((double)res -> size));
  
  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_AF_spec()
{
  CTL res, prop;
  unsigned i, num_nodes;

  assert(spec -> tag == AF_Tag);

  if(!spec -> data.arg[0] -> is_propositional)
    fatal(POSITION, "only non nested liveness properties implemented yet\n");
  
  if(bound == 0)
    fatal(POSITION, "can not unroll safety property for <bound> = 0\n");

  start_verbose("unrolling AF");

  /*
   *  not |=_k  AF prop 
   *
   * <=>
   *
   * |=_k EG !prop
   *
   * <=>
   *
   * init(0)
   * & 
   * trans(0,1) & ... & trans(k-1, k)
   * &
   * !prop(0) & !prop(1) & ... & !prop(k)
   * &
   * (stab(0,k) | stab(1,k) | ... | stab(k-1,k))
   *
   * is satisfiable
   *---------------------------------
   *  If we have to consider fairness a fair loop is added instead of
   *  the simple loops (the last big disjunction)
   */

  res = sub_prop_CTL(init, 0, 1);
  prop = unary_CTL(NOT_Tag, sub_prop_CTL(spec -> data.arg[0], 0, 1));
  res = binary_CTL(AND_Tag, res, prop);

  for(i = 1; i <= bound; i++)
    {
      res = binary_CTL(AND_Tag, res, sub_prop_CTL(trans, i - 1, i));
      prop = unary_CTL(NOT_Tag, sub_prop_CTL(spec -> data.arg[0], i - 1, i));
      res = binary_CTL(AND_Tag, res, prop);
    }

  num_nodes = count_CTL(res);
  end_verbose(
    "unrolled AF for k = %u: size %u (%u nodes = %.0f%%)",
     bound, res -> size, num_nodes,
     100.0 * ((double)num_nodes) / ((double)res -> size));

  if(fairness) res = gen_fair_loop(res);
  else res = gen_loop(res);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL _translate_ECTL(Idx w, CTL f, unsigned i)
{
  CTL res, tmp, t;
  Idx u, v;
  unsigned j;

  u = new_Idx(w, i);

  if(f -> is_propositional)
    {
      res = sub_prop_CTL(f, u -> num, u -> num + 1);
    }
  else
    {
      switch(f -> tag)
        {
	  case EX_Tag:
	    v = new_Idx(w, i + 1);
	    tmp = _translate_ECTL(v, f -> data.arg[0], 0);
	    t = sub_prop_CTL(trans, u -> num, v -> num);
	    res = binary_CTL(AND_Tag, t, tmp);
	    break;
	  
	  case EF_Tag:
	    res = _translate_ECTL(u, f -> data.arg[0], 0);
	    if(i < bound)
	      {
		v = new_Idx(w, i + 1);
		t = sub_prop_CTL(trans, u -> num, v -> num);
		tmp = _translate_ECTL(w, f, i+1);
		tmp = binary_CTL(AND_Tag, t, tmp);
		res = binary_CTL(OR_Tag, res, tmp);
	      }
	    break;

	  case EG_Tag:
	    res = _translate_ECTL(u, f -> data.arg[0], 0);
	    if(i < bound)
	      {
		v = new_Idx(w, i + 1);
		t = sub_prop_CTL(trans, u -> num, v -> num);
		tmp = _translate_ECTL(w, f, i+1);
		tmp = binary_CTL(AND_Tag, t, tmp);
		res = binary_CTL(AND_Tag, res, tmp);
	      }
	    else
	      {
	        for(tmp = 0, j = 0; j <= bound; j++)
		  {
		    v = new_Idx(w, j);
		    t = sub_prop_CTL(trans, u -> num, v -> num);
		    if(tmp) tmp = binary_CTL(OR_Tag, tmp, t);
		    else tmp = t;
		  }

		res = binary_CTL(AND_Tag, res, tmp);
	      }
	    break;
	    
	  default:
	    res = 0;
	    fatal(POSITION, "non valid tag\n");
	    break;
	}
    }

  return res;
}

/*------------------------------------------------------------------------*/

static CTL translate_ECTL()
{
  CTL res, tmp;
  Idx u;

  init_Idx(bound);
  tmp = nnf_CTL(unary_CTL(NOT_Tag, copy_CTL(spec)), 1);
  res = _translate_ECTL(0, tmp, 0);
  u = new_Idx(0, 0);
  res = binary_CTL(AND_Tag, sub_prop_CTL(init, u -> num, u -> num + 1), res);
  free_CTL(tmp);
  exit_Idx(bound);

  return res;
}

/*------------------------------------------------------------------------*/

CTL gen_prop()
{
  flatten_all();
  substitute_defines();
  add_assignments(1);
  add_invar();

  if(only_propositional) return copy_CTL(spec);
  if(diameter_flag) return gen_diameter();
  if(check_invariant) return gen_invariant_checking_step();
  if(check_base) return gen_invariant_checking_base();
  if(alternative_translation) return translate_ECTL();
  if(spec -> is_propositional) return gen_prop_spec();
  if(spec -> tag == AG_Tag) return gen_AG_spec();
  if(spec -> tag == AF_Tag) return gen_AF_spec();

  fatal(POSITION, "unimplemented type of SPEC\n");
  return 0;
}

/*------------------------------------------------------------------------*/

static void _gen_indices_for_vars(CTL f, unsigned * idx_ptr)
{
  Variable v;
  Stack stack;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      f = (CTL) pop(stack);

      switch(f -> tag)
	{
	  case AND_Tag:
	  case OR_Tag:
	  case IMPLIES_Tag:
	  case EQUIV_Tag:

	    push(stack, f -> data.arg[1]);
	    push(stack, f -> data.arg[0]);
	    break;
	  
	  case EX_Tag:
	  case EF_Tag:
	  case EG_Tag:
	  case AX_Tag:
	  case AF_Tag:
	  case AG_Tag:
	  case NOT_Tag:

	    push(stack, f -> data.arg[0]);
	    break;
	  
	  case CURRENT_Tag:
	  case NEXT_Tag:

	    v = f -> data.base.variable;
	    if(!v -> cache.idx)
	      {
		v -> cache.idx = *idx_ptr;
		*idx_ptr += v -> size;
	      }

	    f -> cache.number = f -> data.base.idx + v -> cache.idx;
	    break;
	  
	  default:

	    fatal(POSITION, "non valid tag\n");
	    break;
	}
    }

  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

unsigned gen_indices_for_vars(CTL f)
{
  unsigned idx;

  idx = 1;
  _gen_indices_for_vars(f, &idx);

  return idx - 1;
}

/*------------------------------------------------------------------------*/

static void _gen_variables(CTL f, unsigned * idx_ptr)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
	{
	  g -> is_marked = 1;
	  /* Don't add anythin without adding it to _gen_defines_aux
	   */
	  if(g -> tag != CURRENT_Tag && g -> tag != NEXT_Tag &&
	     g -> tag != CONSTANT_Tag && g -> tag != NUMBER_Tag)
	    {
	      if(g -> is_flattened && g -> is_current)
	        {
		  g -> cache.variable = generate_Symbol();
		  *idx_ptr += 1;
		}

	      push_CTL(stack, g);
	    }
	}
    }

  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

unsigned gen_variables(CTL f)
{
  unsigned idx;

  idx = 1;
  _gen_variables(f, &idx);

  return idx - 1;
}

/*------------------------------------------------------------------------*/

static unsigned idx_for_functor;

/*------------------------------------------------------------------------*/

static void _gen_variables_assignment(Variable v, CTL formula, CTL * formulas)
{
  unsigned i;

  if(formula) _gen_variables(formula, &idx_for_functor);
  else
  if(formulas)
    {
      for(i = 0; i < v -> size; i++) 
	if(formulas[i]) _gen_variables(formulas[i], &idx_for_functor);
    }
}

/*------------------------------------------------------------------------*/

static void gen_variables_assignment(Assignment a)
{
  Variable v;

  v = a -> variable;
  _gen_variables_assignment(v, a -> init, a -> inits);
  _gen_variables_assignment(v, a -> next, a -> nexts);
  _gen_variables_assignment(v, a -> define, a -> defines);
}

/*------------------------------------------------------------------------*/

static unsigned gen_variables_all()
{
  idx_for_functor = 1;

  _gen_variables(init, &idx_for_functor);
  _gen_variables(trans, &idx_for_functor);
  _gen_variables(invar, &idx_for_functor);
  _gen_variables(spec, &idx_for_functor);
  forall_Assignment(gen_variables_assignment);

  return idx_for_functor - 1;
}

/*------------------------------------------------------------------------*/

static CTL extract_gen_var(CTL f)
{
  CTL res;
  Variable v;

  assert(f -> is_current);
  assert(f -> is_flattened);

  switch(f -> tag)
    {
      case CURRENT_Tag:
      case CONSTANT_Tag:
      case NUMBER_Tag:
      case NEXT_Tag:
        res = copy_CTL(f);
        break;
      
      default:
	v = f -> cache.variable;
	assert(v);
	res = current_CTL(v, 0);
	break;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static CTL gen_this_as_define(CTL f, int as_formula)
{
  CTL res, l, r, tmp;
  Variable v;

  assert(f -> is_flattened);
  assert(f -> is_current);

  switch(f -> tag)
    {
      case ADD_Tag:
      case AND_Tag:
      case CASE_Tag:
      case COLON_Tag:
      case EQUAL_Tag:
      case EQUIV_Tag:
      case IMPLIES_Tag:
      case OR_Tag:
      case SUB_Tag:
      case SWITCH_Tag:

	l = extract_gen_var(f -> data.arg[0]);
	r = extract_gen_var(f -> data.arg[1]);
	res = binary_CTL(f -> tag, l, r);
	break;

      case NOT_Tag:
      case INC_Tag:
      case DEC_Tag:
      case EX_Tag:
      case EF_Tag:
      case EG_Tag:
      case AX_Tag:
      case AF_Tag:
      case AG_Tag:
	l = extract_gen_var(f -> data.arg[0]);
	res = unary_CTL(f -> tag, l);
	break;
      
      case CURRENT_Tag:
      case NEXT_Tag:
      case NUMBER_Tag:
      case CONSTANT_Tag:
      default:
        res = 0;
	fatal(POSITION, "non valid tag\n");
	break;
    }
  
  v = f -> cache.variable;

  assert(v);

  if(as_formula)
    {
      tmp = current_CTL(v, 0);
      res = binary_CTL(EQUIV_Tag, tmp, res);
    }
  else
    {
      assert(!find_Assignment(v));
      insert_Assignment(DEFINE_Assignment, v, 0, res);
      res = 0;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/
/* This removes the marks introduced by _gen_variables.  As a `side effect'
 * (actually the main objective of this function) also all the nodes are
 * stored as defines.  The cache has to be cleared later.
 */

static CTL _gen_defines_constraints(CTL f, int as_formula)
{
  Stack stack;
  CTL g;
  CTL res, tmp;

  stack = new_Stack();
  push(stack, f);
  res = 0;

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);

      if(g -> is_marked)
        {
	  g -> is_marked = 0;
	  if(g -> tag != CURRENT_Tag && g -> tag != NEXT_Tag &&
	     g -> tag != CONSTANT_Tag && g -> tag != NUMBER_Tag)
	    {
	      if(g -> is_flattened && g -> is_current)
	        {
		  tmp = gen_this_as_define(g, as_formula);
	      
		  if(as_formula)
		    {
		      if(res) res = binary_CTL(AND_Tag, res, tmp);
		      else res = tmp;
		    }
		  else assert(!tmp);
		}
	      
	      push_CTL(stack, g);
	    }
	}
    }
  
  free_Stack(stack);

  if(as_formula && !res) res = true_CTL();

  return res;
}

/*------------------------------------------------------------------------*/
/* This function does not use the DAG structure of the formula f but
 * needs to see it as tree.  This can be a bottleneck.  But in general
 * it should only occur if there a lot of (non flattened or non current)
 * but shared formulas.
 */

static CTL _gen_defines_rest(CTL f)
{
  CTL res, l, r;

  if(f -> tag == CURRENT_Tag || f -> tag == NEXT_Tag ||
     f -> tag == CONSTANT_Tag || f -> tag == NUMBER_Tag)
    {
      res = copy_CTL(f);
    }
  else
  if(f -> is_flattened && f -> is_current)
    {
      res = current_CTL(f -> cache.variable, 0);
    }
  else
    {
      switch(f -> tag)
	{
	  case ADD_Tag:
	  case AND_Tag:
	  case CASE_Tag:
	  case COLON_Tag:
	  case EQUAL_Tag:
	  case EQUIV_Tag:
	  case IMPLIES_Tag:
	  case OR_Tag:
	  case SUB_Tag:
	  case SWITCH_Tag:

	    l = _gen_defines_rest(f -> data.arg[0]);
	    r = _gen_defines_rest(f -> data.arg[1]);
	    res = binary_CTL(f -> tag, l, r);
	    break;

	  case NOT_Tag:
	  case INC_Tag:
	  case DEC_Tag:
	  case EX_Tag:
	  case EF_Tag:
	  case EG_Tag:
	  case AX_Tag:
	  case AF_Tag:
	  case AG_Tag:
	    l = _gen_defines_rest(f -> data.arg[0]);
	    res = unary_CTL(f -> tag, l);
	    break;
	  
	  case CURRENT_Tag:
	  case NEXT_Tag:
	  case NUMBER_Tag:
	  case CONSTANT_Tag:
	  default:
	    res = 0;
	    fatal(POSITION, "non valid tag\n");
	    break;
	}
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

static CTL _gen_defines(CTL f, int as_formula)
{
  CTL res, tmp;

  res = _gen_defines_rest(f);
  tmp = _gen_defines_constraints(f, as_formula);

  if(as_formula) res = binary_CTL(AND_Tag, res, tmp);
  else assert(!tmp);

  return res;
}

/*------------------------------------------------------------------------*/

CTL gen_defines(CTL f)
{
  double old_size, old_nodes, new_size, new_nodes;
  unsigned tmp;
  CTL res;

  start_verbose("introducing new variables for sharing");
  old_size = f -> size;
  old_nodes = count_CTL(f);
  tmp = gen_variables(f);
  print_verbose("generated %u internal variables\n", tmp);
  res = _gen_defines(f, 1);
  new_size = res -> size;
  new_nodes = count_CTL(res);
  _end_changing("explicit shared", old_size, new_size, old_nodes, new_nodes);
  end_verbose("introduced variables for sharing");
  
  return res;
}

/*------------------------------------------------------------------------*/

static Stack reset_stack_for_functor = 0;

/*------------------------------------------------------------------------*/

static void _gen_defines_assignment(
  Variable v, CTL * formula_ptr, CTL * formulas)
{
  CTL tmp;
  unsigned i;

  if(*formula_ptr)
    {
      tmp = _gen_defines(*formula_ptr, 0);
      push(reset_stack_for_functor, *formula_ptr);
      *formula_ptr = tmp;
    }
  else
  if(formulas)
    {
      for(i = 0; i < v -> size; i++)
        {
	  if(formulas[i])
	    {
	      tmp = _gen_defines(formulas[i], 0);
	      push(reset_stack_for_functor, formulas[i]);
	      formulas[i] = tmp;
	    }
	}
    }
}

/*------------------------------------------------------------------------*/

static void gen_defines_assignment(Assignment a)
{
  Variable v;

  v = a -> variable;
  _gen_defines_assignment(v, &a -> init, a -> inits);
  _gen_defines_assignment(v, &a -> next, a -> nexts);
  _gen_defines_assignment(v, &a -> define, a -> defines);
}

/*------------------------------------------------------------------------*/

static void _gen_defines_all()
{
  CTL tmp, g;
  Stack reset_stack;
  unsigned i;

  reset_stack = new_Stack();

  reset_stack_for_functor = reset_stack;
  forall_Assignment(gen_defines_assignment);
  reset_stack_for_functor = 0;

  tmp = _gen_defines(init, 0);
  push(reset_stack, init);
  init = tmp;

  tmp = _gen_defines(trans, 0);
  push(reset_stack, trans);
  trans = tmp;

  tmp = _gen_defines(invar, 0);
  push(reset_stack, invar);
  invar = tmp;

  tmp = _gen_defines(spec, 0);
  push(reset_stack, spec);
  spec = tmp;

  for(i = 0; !is_mt_Stack(reset_stack); i++)
    {
      g = (CTL) pop(reset_stack);
      reset_variable_cache(g);
      free_CTL(g);
    }
  
  free_Stack(reset_stack);

  print_verbose("delayed deletion of %u formulas\n", i);
}

/*------------------------------------------------------------------------*/

void gen_defines_all()
{
  unsigned old, tmp;

  start_verbose("generating `defines' for complex formulas");
  start_changing();
  old = num_define_assignments;
  tmp = gen_variables_all();
  print_verbose("generated %u internal variables\n", tmp);
  _gen_defines_all();
  end_changing();
  end_verbose("generated %u `define' assignments", 
    num_define_assignments - old);
}

/*------------------------------------------------------------------------*/

static int bit(unsigned position, unsigned number)
{
  return (number >> position) & 1;
}

/*------------------------------------------------------------------------*/

static CTL extract_carry_bit(CTL f, CTL g, unsigned position)
{
  CTL l, r, c, lr, lc, cr, res;

  l = extract_bit(f, position, 0);
  r = extract_bit(g, position, 0);
  lr = binary_CTL(AND_Tag, l, r);

  if(position == 0) res = lr;
  else
    {
      c = extract_carry_bit(f, g, position - 1);
      lc = binary_CTL(AND_Tag, copy_CTL(l), copy_CTL(c));
      cr = binary_CTL(AND_Tag, c, copy_CTL(r));
      res = binary_CTL(OR_Tag, lc, lr);
      res = binary_CTL(OR_Tag, cr, res);
    }

  return res;
}

/*------------------------------------------------------------------------*/

static CTL extract_add_bit(CTL f, CTL g, unsigned position)
{
  CTL carry, l, r, res;

  l = extract_bit(f, position, 0);
  if(position == 0)
    {
      r = extract_bit(g, position, 1);
      res = binary_CTL(EQUIV_Tag, l, r);
    }
  else
    {
      r = extract_bit(g, position, 0);
      res = binary_CTL(EQUIV_Tag, l, r);
      carry = extract_carry_bit(f, g, position - 1);
      res = binary_CTL(EQUIV_Tag, res, carry);
    }

  return res;
}

/*------------------------------------------------------------------------*/

static CTL extract_sub_bit(CTL f, CTL g, unsigned position)
{
  CTL res;

  fatal(POSITION, "flattening of SUB not implemented yet\n");

  res = 0;
  return res;
}

/*------------------------------------------------------------------------*/

CTL extract_bit(CTL f, unsigned position, int sign)
{
  CTL res, tmp, g, cond, this_case, not_other_cases;
  unsigned i, idx;

  /* assert(width_CTL(f) > position); */

  switch(f -> tag)
    {
      case ADD_Tag:
        
	res = extract_add_bit(f -> data.arg[0], f -> data.arg[1], position);
	break;
      
      case SUB_Tag:
	res = extract_sub_bit(f -> data.arg[0], f -> data.arg[1], position);
	break;

      case INC_Tag:

	if(position)
	  {
	    /*    Let inc(f) = gn ... g0, with f = fn ... f0 then
	     *
	     *      gn <-> (fn & sum_i<n_(fi) < n | !fn & sum_i<n_(fi) = n)
	     *      =
	     *      gn <-> (fn <-> sum_i<n_(fi) < n)
	     *      =
	     *      gn <-> (fn <-> (!f0 | !f1 | ... | !fn-1))
	     */

	    g = f -> data.arg[0];
	    tmp = extract_bit(g, 0, 1);		/* !f0 */

	    for(i = 1; i < position; i++)
	      tmp = binary_CTL(OR_Tag, extract_bit(g, i, 1), tmp);

	    res = binary_CTL(EQUIV_Tag, extract_bit(g, position, 0), tmp);
	  }
	else res = extract_bit(f -> data.arg[0], 0, 1);

	break;
      
      case DEC_Tag:

	if(position)
	  {
	    /*    Let dec(f) = gn ... g0, with f = fn ... f0 then
	     *
	     *      gn <-> (fn & sum_i<n_(fi) > 0 | !fn & sum_i<n_(fi) = 0)
	     *      =
	     *      gn <-> (fn <-> sum_i<n_(fi) > 0)
	     *      =
	     *      gn <-> (fn <-> (f0 | f1 | ... | fn-1))
	     */

	    g = f -> data.arg[0];
	    tmp = extract_bit(g, 0, 0);		/* f0 */

	    for(i = 1; i < position; i++)
	      tmp = binary_CTL(OR_Tag, extract_bit(g, i, 0), tmp);

	    res = binary_CTL(EQUIV_Tag, extract_bit(g, position, 0), tmp);
	  }
	else res = extract_bit(f -> data.arg[0], 0, 1);

	break;

      case CURRENT_Tag:
	idx = is_vector_CTL(f) ? position : f -> data.base.idx;
	res = current_CTL(f -> data.base.variable, idx);
	break;

      case NEXT_Tag:
	idx = is_vector_CTL(f) ? position : f -> data.base.idx;
	res = next_CTL(f -> data.base.variable, idx);
	break;
      
      case SWITCH_Tag:
        
	/* Assume that the parser checked or generated coverage 
	 * for all the possible values.
	 */

	for(res = 0; f -> tag == SWITCH_Tag; f = f -> data.arg[1])
	  {
	    assert(f -> data.arg[0] -> tag == COLON_Tag);
	    assert(f -> data.arg[0] -> data.arg[0] -> tag == EQUAL_Tag);

	    cond = flatten_CTL(f -> data.arg[0] -> data.arg[0]);
	    tmp = extract_bit(f -> data.arg[0] -> data.arg[1], position, 0);
	    tmp = binary_CTL(IMPLIES_Tag, cond, tmp);

	    if(res) res = binary_CTL(AND_Tag, res, tmp);
	    else res = tmp;
	  }

	assert(f -> tag == COLON_Tag);
	assert(f -> data.arg[0] -> tag == EQUAL_Tag);

	cond = flatten_CTL(f -> data.arg[0]);
	tmp = extract_bit(f -> data.arg[1], position, 0);
	tmp = binary_CTL(IMPLIES_Tag, cond, tmp);

	assert(res);				/* since at least two cases */
	res = binary_CTL(AND_Tag, res, tmp);
	
	break;

      case CASE_Tag:

        res = 0;
	not_other_cases = 0;

	while(f -> tag == CASE_Tag)
	  {
	    assert(f -> data.arg[0] -> tag == COLON_Tag);

	    cond = flatten_CTL(f -> data.arg[0] -> data.arg[0]);
	    if(not_other_cases) 
	      {
	        this_case = binary_CTL(AND_Tag,
		  copy_CTL(not_other_cases), cond);

	        not_other_cases = binary_CTL(AND_Tag, not_other_cases,
		  unary_CTL(NOT_Tag, copy_CTL(cond)));
	      }
	    else
	      {
	        this_case = cond;
		not_other_cases = unary_CTL(NOT_Tag, copy_CTL(cond));
	      }

	    tmp = extract_bit(f -> data.arg[0] -> data.arg[1], position, 0);
	    tmp = binary_CTL(IMPLIES_Tag, this_case, tmp);

	    if(res) res = binary_CTL(AND_Tag, res, tmp);
	    else res = tmp;

	    f = f -> data.arg[1];
	  }

	assert(f -> tag == COLON_Tag);
	assert(is_true_CTL(f -> data.arg[0]));

	tmp = extract_bit(f -> data.arg[1], position, 0);
	tmp = binary_CTL(IMPLIES_Tag, not_other_cases, tmp);

	if(res) res = binary_CTL(AND_Tag, res, tmp);
	else res = tmp;
	
        break;
      
      case NUMBER_Tag:
	res = number_CTL(bit(position, f -> data.number));
        break;

      case CONSTANT_Tag:
	res = 0;
	assert(0);
	break;
      
      case AND_Tag:
      case OR_Tag:
      case EQUIV_Tag:
      case IMPLIES_Tag:
      case NOT_Tag:
        res = flatten_CTL(f);
	break;
      
      default:
	res = 0;
	fatal(POSITION, "non valid tag\n");
	break;
    }
  
  if(sign) res = unary_CTL(NOT_Tag, res);

  assert(res -> is_flattened);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL _flatten_CTL_equal(CTL a, CTL b)
{
  CTL res, l, r, tmp;
  unsigned w, a_width, b_width, i;

  a_width = width_CTL(a);
  b_width = width_CTL(b);

  w = a_width > b_width ? a_width : b_width;

  l = extract_bit(a, 0, 0);
  r = extract_bit(b, 0, 0);
  res = binary_CTL(EQUIV_Tag, l, r);

  for(i = 1; i < w; i++)
    {
      l = extract_bit(a, i, 0);
      r = extract_bit(b, i, 0);
      tmp = binary_CTL(EQUIV_Tag, l, r);
      res = binary_CTL(AND_Tag, tmp, res);
    }

  return res;
}

/*------------------------------------------------------------------------*/
/* Does the binary encoding.
 */

CTL flatten_CTL(CTL f)
{
  CTL res, f0, f1;

  if(f -> is_flattened) res = copy_CTL(f);
  else
    {
      switch(f -> tag)
	{
	  case EQUAL_Tag:
	    f0 = f -> data.arg[0];
	    f1 = f -> data.arg[1];
	    res = _flatten_CTL_equal(f0, f1);
	    break;

	  case AND_Tag:
	  case EQUIV_Tag:
	  case OR_Tag:
	  case IMPLIES_Tag:
	    f0 = flatten_CTL(f -> data.arg[0]);
	    f1 = flatten_CTL(f -> data.arg[1]);
	    res = binary_CTL(f -> tag, f0, f1);
	    break;
	  
	  case NOT_Tag:
	  case EX_Tag:
	  case EF_Tag:
	  case EG_Tag:
	  case AX_Tag:
	  case AF_Tag:
	  case AG_Tag:
	    f0 = flatten_CTL(f -> data.arg[0]);
	    res = unary_CTL(f -> tag, f0);
	    break;
	  
	  case CASE_Tag:
	    res = extract_bit(f, 0, 0);
	    break;
	  
	  default:
	    res = 0;
	    fatal(POSITION, "non valid tag\n");
	    break;
	}
    }

  assert(res -> is_flattened);

  return res;
}

/*------------------------------------------------------------------------*/

static Assignment_Type type_for_functor;

/*------------------------------------------------------------------------*/

static void flatten_assignment(Assignment a)
{
  CTL * formulas, * formula_ptr, old, tmp;
  Variable v;
  unsigned i, * counter_ptr;

  v = a -> variable;

  switch(type_for_functor)
    {
      case INIT_Assignment:
	formulas = a -> inits;
	formula_ptr = &a -> init;
	counter_ptr = &num_init_assignments;
        break;

      case NEXT_Assignment:
	formulas = a -> nexts;
	formula_ptr = &a -> next;
	counter_ptr = &num_next_assignments;
        break;

      case DEFINE_Assignment:
	formulas = a -> defines;
	formula_ptr = &a -> define;
	counter_ptr = &num_define_assignments;
        break;

      default:
        formulas = 0;
	formula_ptr = 0;
	counter_ptr = 0;
	fatal(POSITION, "non valid type");
	break;
    }

  if(*formula_ptr)
    {
      old = *formula_ptr;
      *formula_ptr = 0;
      (*counter_ptr)--;

      for(i = 0; i < v -> size; i++)
	{
	  tmp = extract_bit(old, i, 0);
	  insert_Assignment(type_for_functor, v, i, tmp);
	}

      free_CTL(old);
    }
  else
  if(formulas)
    {
      for(i = 0; i < v -> size; i++)
	{
	  if(formulas[i] && !formulas[i] -> is_flattened)
	    {
	      tmp = extract_bit(formulas[i], 0, 0);
	      free_CTL(formulas[i]);
	      formulas[i] = tmp;
	    }
	}
    }
}

/*------------------------------------------------------------------------*/

static void flatten_assignments(Assignment_Type type)
{
  type_for_functor = type;
  forall_Assignment(flatten_assignment);
}

/*------------------------------------------------------------------------*/

static void flatten_List(List l)
{
  List p;
  CTL tmp;

  for(p = l; p; p = cdr(p))
    {
      tmp = flatten_CTL((CTL) p -> head);
      free_CTL((CTL) p -> head);
      p -> head = tmp;
    }
}

/*------------------------------------------------------------------------*/

void flatten_all()
{
  CTL tmp;
  unsigned num_nodes, old_not_flattened;

  if(!is_true_CTL(init) && !init -> is_flattened)
    {
      start_verbose("flattening INIT");
      tmp = init;
      init = flatten_CTL(init);
      free_CTL(tmp);
      num_nodes = count_CTL(init);
      end_verbose("flat INIT: size %u (%u nodes = %0.f%%)",
        init -> size, num_nodes,
	100.0 * ((double)num_nodes) / ((double)init -> size));
    }
  
  if(num_init_assignments && 
     (old_not_flattened = num_not_flattened_assignments(INIT_Assignment)))
    {
      start_verbose("flattening `init' assignments");
      flatten_assignments(INIT_Assignment);
      assert(!num_not_flattened_assignments(INIT_Assignment));
      end_verbose("flattened %u `init' assignments", old_not_flattened);
    }

  if(!is_true_CTL(trans) && !trans -> is_flattened)
    {
      start_verbose("flattening TRANS");
      tmp = trans;
      trans = flatten_CTL(trans);
      free_CTL(tmp);
      num_nodes = count_CTL(trans);
      end_verbose("flat TRANS: size %u (%u nodes = %0.f%%)",
        trans -> size, num_nodes,
	100.0 * ((double)num_nodes) / ((double)trans -> size));
    }

  if(num_next_assignments &&
     (old_not_flattened = num_not_flattened_assignments(NEXT_Assignment)))
    {
      start_verbose("flattening `next' assignments");
      flatten_assignments(NEXT_Assignment);
      assert(!num_not_flattened_assignments(NEXT_Assignment));
      end_verbose("flattened %u `next' assignments", old_not_flattened);
    }

  if(!is_true_CTL(invar) && !invar -> is_flattened)
    {
      start_verbose("flattening INVAR");
      tmp = invar;
      invar = flatten_CTL(invar);
      free_CTL(tmp);
      num_nodes = count_CTL(invar);
      end_verbose("flat INVAR: size %u (%u nodes = %0.f%%)",
        invar -> size, num_nodes,
	100.0 * ((double)num_nodes) / ((double)invar -> size));
    }

  if(num_define_assignments && 
     (old_not_flattened = num_not_flattened_assignments(DEFINE_Assignment)))
    {
      start_verbose("flattening `define' assignments");
      flatten_assignments(DEFINE_Assignment);
      assert(!num_not_flattened_assignments(DEFINE_Assignment));
      end_verbose("flattened %u `define' assignments", old_not_flattened);
    }
  
  if(fairness)
    {
      start_verbose("flattening `FAIRNESS' constraints");
      flatten_List(fairness);
      end_verbose("flattened %u `FAIRNESS' constraints",
        length_List(fairness));
    }

  if(!is_true_CTL(spec) && !spec -> is_flattened)
    {
      start_verbose("flattening SPEC");
      tmp = spec;
      spec = flatten_CTL(spec);
      free_CTL(tmp);
      num_nodes = count_CTL(spec);
      end_verbose("flat SPEC: size %u (%u nodes = %.0f%%)",
        spec -> size, num_nodes,
	100.0 * ((double)num_nodes) / ((double)spec -> size));
    }
}

/*------------------------------------------------------------------------*/
#if 0
/*------------------------------------------------------------------------*/

void cnf_all()
{
  CTL tmp;

  flatten_all();

  if(!is_true_CTL(init) && !init -> is_cnf)
    {
      start_verbose("generating CNF for INIT");
      tmp = init;
      init = cnf(init);
      free_CTL(tmp);
      end_verbose("generated CNF for INIT");
    }

  if(!is_true_CTL(trans) && !trans -> is_cnf)
    {
      start_verbose("generating CNF for TRANS");
      tmp = trans;
      trans = cnf(trans);
      free_CTL(tmp);
      end_verbose("generated CNF for TRANS");
    }
  
  if(!is_true_CTL(spec) && spec -> is_propositional && !spec -> is_cnf)
    {
      start_verbose("generating CNF for SPEC");
      tmp = spec;
      spec = cnf(spec);
      free_CTL(tmp);
      end_verbose("generated CNF for SPEC");
    }
}

/*------------------------------------------------------------------------*/
#endif
/*------------------------------------------------------------------------*/

static int flattened_for_functor = 0;

/*------------------------------------------------------------------------*/

static void add_assignment(Assignment a)
{
  CTL tmp, l, r;
  Variable v;
  unsigned i;

  v = a -> variable;

  if(a -> init)
    {
      if(flattened_for_functor)
	{
	  for(i = 0; i < v -> size; i++)
	    {
	      l = current_CTL(v, i);
	      r = extract_bit(a -> init, i, 0);
	      tmp = binary_CTL(EQUIV_Tag, l, r);
	      init = binary_CTL(AND_Tag, init, tmp);
	    }
	}
      else
	{
	  l = current_CTL(v, v -> size);
	  r = copy_CTL(a -> init);
	  tmp = binary_CTL(EQUAL_Tag, l, r);
	  init = binary_CTL(AND_Tag, init, tmp);
	}
    }

  if(a -> inits)
    {
      for(i = 0; i < v -> size; i++)
	{
	  if(a -> inits[i])
	    {
	      l = current_CTL(v, i);
	      r = extract_bit(a -> inits[i], 0, 0);
	      tmp = binary_CTL(EQUIV_Tag, l, r);
	      init = binary_CTL(AND_Tag, init, tmp);
	    }
	}
    }

  if(a -> next)
    {
      if(flattened_for_functor)
	{
	  for(i = 0; i < v -> size; i++)
	    {
	      l = next_CTL(v, i);
	      r = extract_bit(a -> next, i, 0);
	      tmp = binary_CTL(EQUIV_Tag, l, r);
	      trans = binary_CTL(AND_Tag, trans, tmp);
	    }
	}
      else
	{
	  l = next_CTL(v, v -> size);
	  r = copy_CTL(a -> next);
	  tmp = binary_CTL(EQUAL_Tag, l, r);
	  trans = binary_CTL(AND_Tag, trans, tmp);
	}
    }

  if(a -> nexts)
    {
      for(i = 0; i < v -> size; i++)
	{
	  if(a -> nexts[i])
	    {
	      l = next_CTL(v, i);
	      r = extract_bit(a -> nexts[i], 0, 0);
	      tmp = binary_CTL(EQUIV_Tag, l, r);
	      trans = binary_CTL(AND_Tag, trans, tmp);
	    }
	}
    }

  if(a -> define)
    {
      if(flattened_for_functor)
	{
	  for(i = 0; i < v -> size; i++)
	    {
	      l = current_CTL(v, i);
	      r = extract_bit(a -> define, i, 0);
	      tmp = binary_CTL(EQUIV_Tag, l, r);
	      invar = binary_CTL(AND_Tag, invar, tmp);
	    }
	}
      else
	{
	  l = current_CTL(v, v -> size);
	  r = copy_CTL(a -> define);
	  tmp = binary_CTL(EQUAL_Tag, l, r);
	  invar = binary_CTL(AND_Tag, invar, tmp);
	}
    }

  if(a -> defines)
    {
      for(i = 0; i < v -> size; i++)
	{
	  if(a -> defines[i])
	    {
	      l = next_CTL(v, i);
	      r = extract_bit(a -> nexts[i], 0, 0);
	      tmp = binary_CTL(EQUIV_Tag, l, r);
	      invar = binary_CTL(AND_Tag, invar, tmp);
	    }
	}
    }

  free_Assignment(a);
}

/*------------------------------------------------------------------------*/

void add_assignments(int flattened)
{
  if(!num_init_assignments &&
     !num_next_assignments &&
     !num_define_assignments) return;

  start_verbose("adding assignments to TRANS");
  start_changing();
  flattened_for_functor = flattened;
  forall_Assignment(add_assignment);
  end_changing();
  end_verbose("added `init', `next', and `define' assignments");
}

/*------------------------------------------------------------------------*/

static CTL substitute_CURRENT_to_NEXT(CTL);

/*------------------------------------------------------------------------*/

void add_invar()
{
  if(is_true_CTL(invar)) return;

  start_verbose("adding INVAR section to TRANS and INIT");
  start_changing();
  trans = binary_CTL(AND_Tag, trans, copy_CTL(invar));
  trans = binary_CTL(AND_Tag, trans, substitute_CURRENT_to_NEXT(invar));
  init = binary_CTL(AND_Tag, init, copy_CTL(invar));
  free_CTL(invar);
  invar = true_CTL();
  end_changing();
  end_verbose("added INVAR section to TRANS and INIT");
}

/*------------------------------------------------------------------------*/

static CTL _substitute_CURRENT_to_NEXT(CTL f)
{
  CTL l, r, res;

  if(!f -> is_marked)
    {
      f -> is_marked = 1;

      assert(!f -> cache2.unary);

      switch(f -> tag)
        {
	  case ADD_Tag:
	  case AND_Tag:
	  case CASE_Tag:
	  case COLON_Tag:
	  case EQUAL_Tag:
	  case EQUIV_Tag:
	  case IMPLIES_Tag:
	  case OR_Tag:
	  case SUB_Tag:
	  case SWITCH_Tag:

	    l = _substitute_CURRENT_to_NEXT(f -> data.arg[0]);
	    r = _substitute_CURRENT_to_NEXT(f -> data.arg[1]);
	    res = binary_CTL(f -> tag, l, r);
	    break;

	  case NOT_Tag:
	  case INC_Tag:
	  case DEC_Tag:
	  case EX_Tag:
	  case EF_Tag:
	  case EG_Tag:
	  case AX_Tag:
	  case AF_Tag:
	  case AG_Tag:
	    l = _substitute_CURRENT_to_NEXT(f -> data.arg[0]);
	    res = unary_CTL(f -> tag, l);
	    break;
	  
	  case CURRENT_Tag:
	    res = next_CTL(f -> data.base.variable, f -> data.base.idx);
	    break;

	  case NUMBER_Tag:
	  case CONSTANT_Tag:
	    res = copy_CTL(f);
	    break;

	  case NEXT_Tag:
	  default:
	    res = 0;
	    fatal(POSITION, "non valid tag\n");
	    break;
	}
      
      f -> cache2.unary = res;
    }

  res = copy_CTL(f -> cache2.unary);

  assert(res);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL substitute_CURRENT_to_NEXT(CTL f)
{
  CTL res;

  res = _substitute_CURRENT_to_NEXT(f);
  unmark_CTL(f);
  reset_unary_cache2(f);

  return res;
}

/*------------------------------------------------------------------------*/

static void check_sub(Assignment a, CTL f)
{
  if(a)
    {
      assert(f -> tag == CURRENT_Tag || f -> tag == NEXT_Tag);
      assert(f -> data.base.variable == a -> variable);

      if(f -> data.base.idx == f -> data.base.variable -> size)
	{
	  fprintf(stderr,
	    "*** can not substitue array `%s' (use `-flatten')\n",
	    a -> variable -> name);
	  exit(1);
	}

      if(a -> define && a -> variable -> size > 1)
	{
	  fprintf(stderr,
	    "*** found array `define' for `%s' (use `-flatten')\n",
	    a -> variable -> name);
	  exit(1);
	}
    }
}

/*------------------------------------------------------------------------*/

static Stack clean_up_stack = 0;

/*------------------------------------------------------------------------*/

static CTL _substitute_defines_CTL(CTL f, int recursive)
{
  CTL res, l, r, tmp;
  Assignment a;
  Variable v;
  unsigned idx;

  if(f -> is_marked)
    {
      res = copy_CTL(f -> cache.unary);
      assert(res);
    }
  else
    {
      f -> is_marked = 1;

      switch(f -> tag)
	{
	  case ADD_Tag:
	  case AND_Tag:
	  case CASE_Tag:
	  case COLON_Tag:
	  case EQUAL_Tag:
	  case EQUIV_Tag:
	  case IMPLIES_Tag:
	  case OR_Tag:
	  case SUB_Tag:
	  case SWITCH_Tag:

	    l = _substitute_defines_CTL(f -> data.arg[0], recursive);
	    r = _substitute_defines_CTL(f -> data.arg[1], recursive);
	    res = binary_CTL(f -> tag, l, r);
	    break;

	  case NOT_Tag:
	  case INC_Tag:
	  case DEC_Tag:
	  case EX_Tag:
	  case EF_Tag:
	  case EG_Tag:
	  case AX_Tag:
	  case AF_Tag:
	  case AG_Tag:
	    l = _substitute_defines_CTL(f -> data.arg[0], recursive);
	    res = unary_CTL(f -> tag, l);
	    break;
	  
	  case CURRENT_Tag:

	    v = f -> data.base.variable;
	    idx = f -> data.base.idx;
	    a = find_Assignment(v);
	    check_sub(a, f);
	    if(a && a -> define)
	      {
		assert(v -> size == 1);
		if(recursive)
		  {
		    push(clean_up_stack, copy_CTL(a -> define));
		    tmp = _substitute_defines_CTL(a -> define, 1);
		    res = extract_bit(tmp, 0, 0);
		    free_CTL(tmp);
		  }
		else res = extract_bit(a -> define, 0, 0);
	      }
	    else
	    if(a && a -> defines && a -> defines[idx])
	      {
		if(recursive)
		  {
		    push(clean_up_stack, copy_CTL(a -> defines[idx]));
		    tmp = _substitute_defines_CTL(a -> defines[idx], 1);
		    res = extract_bit(tmp, 0, 0);
		    free_CTL(tmp);
		  }
		else res = extract_bit(a -> defines[idx], 0, 0);
	      }
	    else res = copy_CTL(f);
	    break;

	  case NEXT_Tag:
	    v = f -> data.base.variable;
	    idx = f -> data.base.idx;
	    a = find_Assignment(v);
	    check_sub(a, f);
	    if(a && a -> define)
	      {
		assert(v -> size == 1);
		if(recursive)
		  {
		    push(clean_up_stack, copy_CTL(a -> define));
		    tmp = _substitute_defines_CTL(a -> define, 1);
		    res = extract_bit(tmp, 0, 0);
		    free_CTL(tmp);
		    tmp = res;
		    res = substitute_CURRENT_to_NEXT(tmp);
		    free_CTL(tmp);
		  }
		else res = substitute_CURRENT_to_NEXT(a -> define);
	      }
	    else
	    if(a && a -> defines && a -> defines[idx])
	      {
		if(recursive)
		  {
		    push(clean_up_stack, copy_CTL(a -> defines[idx]));
		    tmp = _substitute_defines_CTL(a -> defines[idx], 1);
		    res = extract_bit(tmp, 0, 0);
		    free_CTL(tmp);
		    tmp = res;
		    res = substitute_CURRENT_to_NEXT(tmp);
		    free_CTL(tmp);
		  }
		else res = substitute_CURRENT_to_NEXT(a -> defines[idx]);
	      }
	    else res = copy_CTL(f);
	    break;

	  case NUMBER_Tag:
	  case CONSTANT_Tag:
	    res = copy_CTL(f);
	    break;

	  default:
	    res = 0;
	    fatal(POSITION, "non valid tag\n");
	    break;
	}

      f -> cache.unary = copy_CTL(res);
    }

  return res;
}

/*------------------------------------------------------------------------*/
#if 0
/*------------------------------------------------------------------------*/

static CTL substitute_defines_CTL(CTL f, int recursive)
{
  CTL res;

  res = _substitute_defines_CTL(f, recursive);
  unmark_CTL(f);
  reset_unary_cache(f);

  return res;
}

/*------------------------------------------------------------------------*/
#endif
/*------------------------------------------------------------------------*/

static void substitute_defines_in_section(CTL * formula_ptr)
{
  CTL tmp;

  tmp = _substitute_defines_CTL(*formula_ptr, 0);
  push(clean_up_stack, *formula_ptr);
  *formula_ptr = tmp;
}

/*------------------------------------------------------------------------*/

static void _substitute_define_Assignment(
  Variable v, CTL * formula_ptr, CTL * formulas, int recursive)
{
  CTL tmp;
  unsigned i;

  if(*formula_ptr)
    {
      tmp = _substitute_defines_CTL(*formula_ptr, recursive);
      push(clean_up_stack, *formula_ptr);
      *formula_ptr = tmp;
    }

  if(formulas)
    {
      for(i = 0; i < v -> size; i++)
	if(formulas[i])
	  {
	    tmp = _substitute_defines_CTL(formulas[i], recursive);
	    push(clean_up_stack, formulas[i]);
	    formulas[i] = tmp;
	  }
    }
}

/*------------------------------------------------------------------------*/

static void substitute_defines_only_Assignment(Assignment a)
{
  _substitute_define_Assignment(
    a -> variable, &a -> define, a -> defines, 1);
}

/*------------------------------------------------------------------------*/

static void substitute_defines_Assignment(Assignment a)
{
  Variable v;

  v = a -> variable;

  _substitute_define_Assignment(v, &a -> init, a -> inits, 0);
  _substitute_define_Assignment(v, &a -> next, a -> nexts, 0);
}

/*------------------------------------------------------------------------*/

static void remove_defines(Assignment a)
{
  unsigned i;
  Variable v;
  
  v = a -> variable;

  if(a -> define)
    {
      free_CTL(a -> define);
      a -> define = 0;
      num_define_assignments--;
    }

  if(a -> defines)
    {
      for(i = 0; i < v -> size; i++)
	if(a -> defines[i])
	  {
	    free_CTL(a -> defines[i]);
	    a -> defines[i] = 0;
	    num_define_assignments--;
	  }
    }

  remove_if_mt_Assignment(a);
}

/*------------------------------------------------------------------------*/

static void reset_marked_unary_cache(CTL f)
{
  CTL g;
  Stack stack;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(g -> is_marked)
	{
	  assert(g -> cache.unary);

	  g -> is_marked = 0;
	  free_CTL(g -> cache.unary);
	  g -> cache.unary = 0;
	  push_CTL(stack, g);
	}
    }

  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

void substitute_defines()
{
  CTL g;
  if(!num_define_assignments) return;

  start_verbose("substituting defines into model");
  start_changing();
  clean_up_stack = new_Stack();
  forall_Assignment(substitute_defines_only_Assignment);
  forall_Assignment(substitute_defines_Assignment);
  substitute_defines_in_section(&init);
  substitute_defines_in_section(&trans);
  substitute_defines_in_section(&invar);
  substitute_defines_in_section(&spec);
  while(!is_mt_Stack(clean_up_stack))
    {
      g = (CTL) pop(clean_up_stack);
      reset_marked_unary_cache(g);
      free_CTL(g);
    }
  free_Stack(clean_up_stack);
  clean_up_stack = 0;
  forall_Assignment(remove_defines);
  end_changing();
  end_verbose("substituted defines into model");
}
