#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include "assign.h"
#include "config.h"
#include "ctl.h"
#include "dep.h"
#include "dimacs.h"
#include "list.h"
#include "pp.h"
#include "prop.h"
#include "var.h"
#include "prove.h"
#include "smv.h"
#include "stack.h"
#include "util.h"

/*------------------------------------------------------------------------*/

extern int yyparse();
extern FILE * yyin;

/*------------------------------------------------------------------------*/
# ifdef DEBUG
/*------------------------------------------------------------------------*/
  
extern int yydebug;

/*------------------------------------------------------------------------*/
# endif
/*------------------------------------------------------------------------*/

CTL init = 0, trans = 0, spec = 0, invar = 0;
List fairness = 0;

unsigned bound = 0, max_on_line = MAX_ON_LINE_DEFAULT;
char * prefix = PREFIX_DEFAULT, * comment_string = "-- ";

int flatten_flag = 0, cnf_flag = 0, flatten_equivs_flag = 0;
int verbose = 0, no_multiargop_optimization = 0;
int renaming_simplification = 1, diameter_flag = 0;
int renaming_simplification_for_or = 0, check_invariant = 0, check_base = 0;
int only_propositional = 0, sharing = 1, sort_flag = 0, nodefines = 0;
int coi_flag = 1, extract_assignments = 1, use_varset = 0;
int extract_defines = 0, assume_property_on_the_way = 0, formula_flag = 0;
int exact_flag = 0, alternative_translation = 0;

/*------------------------------------------------------------------------*/
#ifdef DEBUG
/*------------------------------------------------------------------------*/

int debug = 0;

/*------------------------------------------------------------------------*/
#endif
/*------------------------------------------------------------------------*/

enum OutputFormat
{
  SMV_Format,
  PrettyPrinting_Format,
  Prove_Format,
  Dimacs_Format,
  NoOutput_Format
};

/*------------------------------------------------------------------------*/

static enum OutputFormat output_format = NoOutput_Format;

/*------------------------------------------------------------------------*/

static char * bmc_main_id =
"$Id: main.c,v 1.1.1.1.2.1 1999/02/02 18:31:56 armin Exp $";

/*------------------------------------------------------------------------*/

static int is_propositional_OutputFormat()
{
  switch(output_format)
    {
      case PrettyPrinting_Format:
      case SMV_Format:
        return 0;
      
      case Dimacs_Format:
      case Prove_Format:
        return 1;

      default:
        fatal(POSITION, "non valid output format\n");
	return 0;
	break;
    }
}

/*------------------------------------------------------------------------*/

static void version(FILE * file)
{
  fprintf(file, "%s\n%s\n", version_and_configuration(), bmc_main_id);
}

/*------------------------------------------------------------------------*/

static void banner(FILE * file)
{
  fprintf(file, "(C) 1998, Armin Biere, Carnegie Mellon University\n");
}

/*------------------------------------------------------------------------*/

static void options(FILE * file, char * argv0)
{
  char * program_name;

  program_name = rindex(argv0, '/');

  if(program_name) program_name++;
  else program_name = argv0;

  if(!*program_name) program_name = "bmc";

  fprintf(file,

"%s <options> [<file>]\n"
"\n"
"  -alt[ernative]     use new recursive translation scheme (buggy!!!)\n"
"  -base              check if a safety property is implied by all initial\n"
"                     states (assumes `SPEC AG <propositional>')\n"
"  -defines-to-INVAR  move all `define' assignments into INVAR section\n"
"  -diameter          check if the diameter of the system is at most\n"
"                     the bound specified by `-k' (must be > 0)\n"
"  -exact[ly]         only check if there is a counterexample of\n"
"  -extract[-defines] extract defines by introducing internal variables.\n"
"  -flatten           produce boolean encoding\n"
"  -invar[iant]       check if a safety property is an invariant of the\n"
"                     transition relation (assumes `SPEC AG <propositional>')\n"
"  -k <bound>         maximal length of counterexample (default %u)\n"
"  -max <size>        maximal size of term printed on one line (default %u)\n"
"  -nocoi             switch off cone of influence reduction\n"
"  -v                 enable verbose output\n"
"\n"
"  -<fmt>             output format where <fmt> is one of\n"
"\n"
"                       pp      pretty printing\n"
"                       smv     generate SMV model\n"
"                       prove   generate PROVER model\n"
"                       dimacs  generate CNF in Dimacs format\n"
"\n"
"  <file>             input file (if not specified stdin is used)\n"

/*------------------------------------------------------------------------*/
#ifdef DEBUG
/*------------------------------------------------------------------------*/

"  -d                 print debugging information\n"
"  -yydebug           print parser debugging information\n"

/*------------------------------------------------------------------------*/
#endif
/*------------------------------------------------------------------------*/

"\n"
"\n"
"  infrequently used <options>:\n"
"  ============================\n"
"\n"
"  -assume[-prop]     assume that there is no counterexample of length\n"
"                     less than `k'\n"
"  -cnf               produce boolean encoding in conjunctive normal form\n"
"                     exactly length <bound> (no `FAIRNESS')\n"
"  -flatten-equivs    remove `<->' operators before converting into CNF\n"
"                     (this should usually be kept switched off)\n"
"  -formula           print formula and do not introduce variables for\n"
"                     propositional formats (only `-prove' yet)\n"
"  -h                 help on command line options\n"
"  -nomultiargopt     switch off optimization for multi argument operators\n"
"                     when converting into CNF\n"
"  -norenamesimp      switch off the `renaming simplification'\n"
"  -nosharing         do not detect common subterms\n"
"  -orrenamesimp      switch on the `renaming simplification' for OR\n"
"  -prefix <str>      prefix used to generate new variables (default \"%s\")\n"
"  -sort              sort the clauses and remove duplicates\n"
"  -version           version information\n"
"\n"
"The command line options can be used in an arbitrary order.\n"

  ,program_name, 0, MAX_ON_LINE_DEFAULT, PREFIX_DEFAULT);
}

/*------------------------------------------------------------------------*/

static void clerror(const char * fmt, ...)
{
  va_list ap;

  fprintf(stderr, "*** [bmc] ");
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);

  fprintf(stderr, "\n");
  fprintf(stderr, 
    "*** [bmc] (use `bmc -h' to print a list of available options)\n");
  fflush(stderr);

  exit(1);
}

/*------------------------------------------------------------------------*/

static void check_cl_numarg(int * i_ptr, int argc, char ** argv, char * str)
{
  int i;

  i = *i_ptr + 1;

  if(i >= argc || argv[i][0] < '0' || argv[i][0] > '9') clerror(str);
  else (*i_ptr)++;
}

/*------------------------------------------------------------------------*/

static void parse_options(int argc, char ** argv)
{
  int i;
  char * p;

  yyin = stdin;				/* init input file */

  for(i=1; i<argc; i++)
    {
      if(strcmp(argv[i], "-h") == 0)
	{
	  banner(stdout);
	  printf("\n");
	  options(stdout, argv[0]);
	  fflush(stdout);
	  exit(0);
	}
      else
      if(strcmp(argv[i], "-version") == 0)
	{
	  version(stdout);
	  banner(stdout);
	  fflush(stdout);
	  exit(0);
	}
      else
      if(strcmp(argv[i], "-v") == 0)
	{
	  verbose = 1;
	}
      else
#ifdef DEBUG
      if(strcmp(argv[i], "-d") == 0)
	{
	  debug = 1;
	}
      else
      if(strcmp(argv[i], "-yydebug") == 0)
        {
	  yydebug = 1;
	}
      else
#endif
      if(strcmp(argv[i], "-pp") == 0)
	{
	  if(output_format != NoOutput_Format) 
	    clerror("multiple output formats specified");

	  comment_string = "-- ";
	  output_format = PrettyPrinting_Format;
	}
      else
      if(strcmp(argv[i], "-alternative") == 0 ||
         strcmp(argv[i], "-alt") == 0)
        {
	  alternative_translation = 1;
	}
      else
      if(strcmp(argv[i], "-smv") == 0)
	{
	  if(output_format != NoOutput_Format)
	    clerror("multiple output formats specified");

	  comment_string = "-- ";
	  output_format = SMV_Format;
	}
      else
      if(strcmp(argv[i], "-assume-prop") == 0 ||
         strcmp(argv[i], "-assume") == 0)
	{
	  assume_property_on_the_way = 1;
	}
      else
      if(strcmp(argv[i], "-exactly") == 0 ||
         strcmp(argv[i], "-exact") == 0)
	{
	  exact_flag = 1;
	}
      else
      if(strcmp(argv[i], "-prove") == 0)
	{
	  if(output_format != NoOutput_Format)
	    clerror("multiple output formats specified");

	  comment_string = "// ";
	  output_format = Prove_Format;
	}
      else
      if(strcmp(argv[i], "-dimacs") == 0)
	{
	  if(output_format != NoOutput_Format)
	    clerror("multiple output formats specified");

	  comment_string = "c ";
	  output_format = Dimacs_Format;
	}
      else
      if(strcmp(argv[i], "-cnf") == 0)
	{
	  cnf_flag = 1;
	}
      else
      if(strcmp(argv[i], "-flatten") == 0)
	{
	  flatten_flag = 1;
	}
      else
      if(strcmp(argv[i], "-nocoi") == 0)
	{
	  coi_flag = 0;
	}
      else
      if(strcmp(argv[i], "-flatten-equivs") == 0)
        {
	  flatten_equivs_flag = 1;
	}
      else
      if(strcmp(argv[i], "-sort") == 0)
        {
	  sort_flag = 1;
	}
      else
      if(strcmp(argv[i], "-formula") == 0)
        {
	  formula_flag = 1;
	}
      else
      if(strcmp(argv[i], "-nosharing") == 0)
	{
	  sharing = 0;
	}
      else
      if(strcmp(argv[i], "-nomultiargopt") == 0)
        {
	  no_multiargop_optimization = 1;
	}
      else
      if(strcmp(argv[i], "-norenamesimp") == 0)
        {
	  renaming_simplification = 0;
	}
      else
      if(strcmp(argv[i], "-orrenamesimp") == 0)
        {
	  renaming_simplification_for_or = 1;
	}
      else
      if(strcmp(argv[i], "-k") == 0)
	{
	  check_cl_numarg(&i, argc, argv, "argument to `-k' is missing");
	  bound = (unsigned) atoi(argv[i]);
	}
      else
      if(strcmp(argv[i], "-diameter") == 0)
        {
	  diameter_flag = 1;
	}
      else
      if(strcmp(argv[i], "-base") == 0)
	{
	  if(check_base)
	    clerror("only one of `-invariant' and `-base'  is allowed");

	  check_base = 1;
	}
      else
      if(strcmp(argv[i], "-defines-to-INVAR") == 0)
        {
	  nodefines = 1;
	}
      else
      if(strcmp(argv[i], "-extract-defines") == 0 ||
         strcmp(argv[i], "-extract") == 0)
        {
	  extract_defines = 1;
	}
      else
      if(strcmp(argv[i], "-invariant") == 0 ||
         strcmp(argv[i], "-invar") == 0)
        {
	  if(check_base)
	    clerror("only one of `-invariant' and `-base'  is allowed");

	  check_invariant = 1;
	}
      else
      if(strcmp(argv[i], "-max") == 0)
	{
	  check_cl_numarg(&i, argc, argv, "argument to `-max' is missing");
	  max_on_line = (unsigned) atoi(argv[i]);
	}
      else
      if(strcmp(argv[i], "-prefix") == 0)
	{
	  check_cl_numarg(&i, argc, argv, "argument to `-prefix' is missing");
	  prefix = argv[i];

	  if(!*prefix)
	    clerror("argument to `-prefix' is empty");

	  if(!(('a' <= *prefix && *prefix <= 'z') ||
	       ('A' <= *prefix && *prefix <= 'Z')))
	    {
	      clerror(
	       "argument `%s' to `-prefix' does not start with a letter",
	       prefix);
	    }

	  for(p = prefix; *p; p++)
	    if(!(('a' <= *prefix && *prefix <= 'z') ||
		 ('A' <= *prefix && *prefix <= 'Z') ||
		 ('0' <= *prefix && *prefix <= '9') ||
		 *prefix == '_'))
	      {
		clerror(
	  "argument `%s' to `-prefix' contains illegal character `%c'",
	          prefix, *prefix);
	      }
	}
      else
      if(argv[i][0] == '-')
	{
	  clerror("unknown command line argument `%s'", argv[i]);
	}
      else
        {
	  if(yyin != stdin) clerror("can not open more than one file");
	  yyin = fopen(argv[i], "r");
	  if(!yyin) clerror("could not open `%s'", argv[i]);
	}
    }
}

/*------------------------------------------------------------------------*/

int main(int argc, char ** argv)
{
  unsigned num_nodes;

  parse_options(argc, argv);

  start_verbose("%s", version_and_configuration());

  /* init other modules
   */
  init_CTL();
  init_Variable();
  init_Assignment();
  init_VarSet();

  /* setup parser result
   */
  init = true_CTL();
  trans = true_CTL();
  spec = true_CTL();
  invar = true_CTL();
  fairness = 0;

  start_verbose("parsing");
  yyparse();
  if(yyin != stdin) fclose(yyin);
  print_verbose("%u init assignments, %u next assignments, %u defines\n",
    num_init_assignments, num_next_assignments, num_define_assignments);
  
  if(!is_true_CTL(init))
    {
      num_nodes = count_CTL(init);
      print_verbose("INIT: size %u (%u nodes = %.0f%%)\n",
        init -> size, num_nodes,
	100.0 * ((double)num_nodes) / ((double)init -> size));
    }

  if(!is_true_CTL(trans))
    {
      num_nodes = count_CTL(trans);
      print_verbose("TRANS: size %u (%u nodes = %.0f%%)\n",
        trans -> size, num_nodes,
	100.0 * ((double)num_nodes) / ((double)trans -> size));
    }

  if(!is_true_CTL(invar))
    {
      num_nodes = count_CTL(invar);
      print_verbose("INVAR: size %u (%u nodes = %.0f%%)\n",
        invar -> size, num_nodes,
	100.0 * ((double)num_nodes) / ((double)invar -> size));
    }

  if(!is_true_CTL(spec))
    {
      num_nodes = count_CTL(spec);
      print_verbose("SPEC: size %u (%u nodes = %.0f%%)\n",
        spec -> size, num_nodes,
	100.0 * ((double)num_nodes) / ((double)spec -> size));
    }
  end_verbose("parsed");

  if(output_format == NoOutput_Format)
    {
      print_verbose("no output format specified\n");
    }
  else
    {
      /* semantic checking
       */
      start_verbose("semantic checking");
      if(is_propositional_OutputFormat())
        {
	  if(diameter_flag)
	    {
	      if(bound == 0)
	        {
		  clerror(
	    "`-diameter' requires `-k <bound>' with <bound> non zero");
		}
	    }

	  if(check_invariant || check_base)
	    {
	      if(spec -> tag != AG_Tag ||
		 !spec -> data.arg[0] -> is_propositional)
	        {
		  clerror("`-invar' needs `SPEC AG <propositional>");
		}

	      if(bound != 0)
		{
		  clerror(
	  "invariant checking can not be combined with `-k <bound>'");
		}
	    }
	  
	  if(fairness)
	    {
	      if(exact_flag)
		clerror("`-exactly' can not be used with FAIRNESS");

	      if(assume_property_on_the_way)
		clerror("`-assume-prop' can not be used with FAIRNESS");
	    }
	  
	  if(!alternative_translation)
	    {
	      if(spec -> is_propositional)
		{
		  if(bound > 0)
		    {
		      clerror(
			"SPEC is propositional but bound is greater than 0");
		    }
		}
	      else
	      if(spec -> tag == AG_Tag)
		{
		  if(!spec -> data.arg[0] -> is_propositional)
		    {
		      clerror(
			"only non nested safety properties can be handled");
		    }
		}
	      else
	      if(spec -> tag == AF_Tag)
		{
		  if(!spec -> data.arg[0] -> is_propositional)
		    {
		      clerror(
		    "only non nested liveness properties can be handled");
		    }
		  
		  if(bound == 0)
		    {
		      clerror(
      "specify `-k <bound>' with <bound> > 0 for liveness properties");
		    }
		}
	      else
		{
		  clerror(
  "only liveness and safety properties allowed for this output format");
		}
	    }
	}
      else
	{
	  if(check_invariant)
	    {
	      clerror(
	    "`-invar' can not be combined with this output format");
	    }

	  if(check_base)
	    {
	      clerror(
	    "`-check_base' can not be combined with this output format");
	    }

	  if(diameter_flag)
	    {
	      clerror(
	    "`-diameter' can not be combined with this output format");
	    }

	  if(!spec -> is_ACTL)
	    {
	      fatal(POSITION,
	        "ACTL should have been enforced by parser\n");
	    }

	  if(output_format == SMV_Format)
	    {
	      if(bitwise_assignment && !flatten_flag)
		{
		  clerror(
      "for bitwise assignments SMV output format requires `-flatten'");
		}
	    }
	}

      if(!init -> is_propositional)
	clerror("INIT predicate is not propositional");
      
      if(!trans -> is_propositional)
	clerror("TRANS predicate is not propositional");
      
      if(!invar -> is_propositional)
	clerror("INVAR predicate is not propositional");
      
      if(!invar -> is_current)
	clerror("INVAR predicate references next state variables");
      
      check_defines();
	  
      end_verbose("semantic checking");

      if(flatten_flag) flatten_all();
      gen_assignments();
      if(coi_flag) coi();
      if(extract_defines) gen_defines_all();
      if(nodefines) move_defines_to_INVAR();

      switch(output_format)
	{
	  case PrettyPrinting_Format:
	    print_pp();
	    break;
	  
	  case SMV_Format:
	    print_smv();
	    break;
	  
	  case Prove_Format:
	    print_prove();
	    break;
	  
	  case Dimacs_Format:
	    print_dimacs();
	    break;

	  case NoOutput_Format:
	  default:
	    break;
	}
    }

  start_verbose("cleaning up");

  free_CTL(init);
  free_CTL(trans);
  free_CTL(invar);
  free_CTL(spec);
  forall_List(fairness, (void(*)(void*)) free_CTL);
  free_List(fairness);

  exit_VarSet();
  exit_Assignment();
  exit_Variable();
  exit_CTL();
  exit_Stack();

  end_verbose("cleaned up");

  end_verbose("%s", version_and_configuration());

  exit(0);
  return 0;
}
