#include <stdio.h>
#include <stdlib.h>

#include "config.h"
#include "ctl.h"
#include "pp.h"
#include "prop.h"
#include "stack.h"
#include "util.h"
#include "var.h"

/*------------------------------------------------------------------------*/

static void _print_clause(CTL f)
{
  assert(f -> is_CLAUSE);

  switch(f -> tag)
    {
      case NOT_Tag:
	assert(f -> data.arg[0] -> tag == CURRENT_Tag ||
	       f -> data.arg[0] -> tag == NEXT_Tag);

	assert((unsigned) f -> data.arg[0] -> cache.number);

	putc('-', stdout);
	putu((unsigned) f -> data.arg[0] -> cache.number, stdout);
	putc(' ', stdout);
	break;
      
      case CURRENT_Tag:

	assert((unsigned) f -> cache.number);

	putu((unsigned) f -> cache.number, stdout);
	putc(' ', stdout);
	break;
      
	break;
      
      case OR_Tag:

	_print_clause(f -> data.arg[0]);
	_print_clause(f -> data.arg[1]);
	break;
      
      default:

	fatal(POSITION, "non valid tag\n");
	break;
    }
}

/*------------------------------------------------------------------------*/

static void print_clause(CTL f)
{
  assert(f -> is_CLAUSE);

  _print_clause(f);
  fputs("0\n", stdout);
}

/*------------------------------------------------------------------------*/

static void print_idx2names(CTL f)
{
  Variable v;
  unsigned i;
  Stack stack;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      f = (CTL) pop(stack);
      switch(f -> tag)
	{
	  case NOT_Tag:
	    push(stack, f -> data.arg[0]);
	    break;

	  case OR_Tag:
	  case AND_Tag:
	    push(stack, f -> data.arg[1]);
	    push(stack, f -> data.arg[0]);
	    break;

	  case CURRENT_Tag:
	    v = f -> data.base.variable;
	    if(v -> cache.idx)
	      {
		for(i = 0; i < v -> size; i++)
		  {
		    fputs("c ", stdout);
		    putu((unsigned) v -> cache.idx + i, stdout);
		    fputs(" = ", stdout);
		    fputs(v -> name, stdout);

		    putc('[', stdout);
		    putu(i, stdout);
		    putc(']', stdout);

		    putc('\n', stdout);
		  }

		v -> cache.idx = 0;
	      }
	    break;

	  default:
	    fatal(POSITION, "non valid tag\n");
	    break;
	}
     }

  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

static void _print_dimacs(CTL f)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);

      if(g -> is_CLAUSE) print_clause(g);
      else
        {
	  assert(g -> tag == AND_Tag);

	  push(stack, g -> data.arg[1]);
	  push(stack, g -> data.arg[0]);
	}
    }

  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

void print_dimacs()
{
  CTL prop, tmp;
  unsigned nv, nc;
  double prop_nodes, cnf_nodes;

  start_verbose("generating DIMACS");
  prop = gen_prop();
  prop_nodes = count_CTL(prop);
  tmp = cnf(prop);
  if(is_true_CTL(tmp))
    {
      fputs("c CONSTANT(1)\np cnf 1 1\n-1 1 0\n", stdout);
      cnf_nodes = 3;
      nv = nc = 0;
    }
  else 
  if(is_false_CTL(tmp))
    {
      fputs("c CONSTANT(0)\np cnf 1 2\n-1 0\n1 0\n", stdout);
      cnf_nodes = 3;
      nv = nc = 0;
    }
  else
    {
      cnf_nodes = count_CTL(tmp);
      nv = gen_indices_for_vars(tmp);
      print_idx2names(tmp);
      nc = num_clauses(tmp);
      printf("p cnf %u %u\n", nv, nc);
      _print_dimacs(tmp);
      reset_number_cache(tmp);
    }
  free_CTL(tmp);
  free_CTL(prop);
  end_verbose("DIMACS: %u variables, %u clauses (x %.2f)",
    nv, nc, cnf_nodes / prop_nodes);
}
