#include <stdlib.h>
#include <stdio.h>

#include "config.h"
#include "ctl.h"
#include "pp.h"
#include "stack.h"
#include "util.h"
#include "var.h"

/*------------------------------------------------------------------------*/

#ifdef DEBUG

extern int debug;

#endif

extern char * prefix, * comment_string;
extern int flatten_equivs_flag, no_multiargop_optimization;
extern int renaming_simplification, renaming_simplification_for_or;
extern int verbose, sharing, sort_flag;

/*------------------------------------------------------------------------*/

static unsigned num_formulas = 0, size_formulas = 0, max_formulas = 0;
static unsigned visited_formulas = 0, lookups_formulas = 0;
static CTL * formulas = 0;

/*------------------------------------------------------------------------*/

#define U(a) ((unsigned)(a))

/*------------------------------------------------------------------------*/

static unsigned hash_CTL(CTL_Tag tag, void * a, void * b)
{
  return ((U(a) * 99991 + U(b)) * 199999 + U(tag)) * 299993;
}

/*------------------------------------------------------------------------*/

static unsigned hash_CTL_from_CTL(CTL f)
{
  switch(f -> tag)
    {
      case CURRENT_Tag:
      case NEXT_Tag:
	return hash_CTL(
	  f -> tag, f -> data.base.variable, (void*) f -> data.base.idx);
	break;
      
      case CONSTANT_Tag:
      case NUMBER_Tag:
	return hash_CTL(f -> tag, (void*) f -> data.number, 0);
	break;
      
      default:
        /* This is the case corresponding to `unary_CTL' and `binary_CTL'
	 * (non valid flags will be detected by `hash_CTL')
	 */
	return hash_CTL(f -> tag, f -> data.arg[0], f -> data.arg[1]);
	break;
    }
}

/*------------------------------------------------------------------------*/

static int is_equal_CTL(CTL_Tag tag, void * a, void * b, CTL h)
{
  CTL f, g;
  Variable v;
  unsigned idx, number;

  if(h -> tag != tag) return 0;

  f = (CTL) a;
  g = (CTL) b;
  v = (Variable) a;
  idx = (unsigned) b;
  number = (unsigned) a;

  switch(tag)
    {
      case NUMBER_Tag:
      case CONSTANT_Tag:
	return h -> data.number == number;
	break;

      case CURRENT_Tag: 
      case NEXT_Tag: 
	return h -> data.base.variable == v && h -> data.base.idx == idx;
	break;
      
      case ADD_Tag:
      case AND_Tag:
      case CASE_Tag:
      case COLON_Tag:
      case EQUAL_Tag:
      case EQUIV_Tag:
      case IMPLIES_Tag:
      case OR_Tag:
      case SUB_Tag:
      case SWITCH_Tag:

	return h -> data.arg[0] == f && h -> data.arg[1] == g;
	break;

      case AF_Tag:
      case AG_Tag:
      case AX_Tag:
      case DEC_Tag:
      case EF_Tag:
      case EG_Tag:
      case EX_Tag:
      case INC_Tag:
      case NOT_Tag:

	return h -> data.arg[0] == f;
	break;
      
      default:
	fatal(POSITION, "non valid tag\n");
	return 0;
	break;
    }
}

/*------------------------------------------------------------------------*/

static void resize_formulas()
{
  CTL p, tmp, * old_formulas;
  unsigned i, h, old_size_formulas;

  old_size_formulas = size_formulas;
  old_formulas = formulas;

  size_formulas = next_size(num_formulas);
  formulas = (CTL*) calloc(size_formulas, sizeof(CTL));

  for(i = 0; i < old_size_formulas; i++)
    {
      for(p = old_formulas[i]; p; p = tmp)
        {
	  tmp = p -> next;
	  h = hash_CTL_from_CTL(p) % size_formulas;
	  p -> next = formulas[h];
	  formulas[h] = p;
	}
    }
  
  free(old_formulas);
}

/*------------------------------------------------------------------------*/

static void _reset_cache(CTL f)
{
  f -> cache.number = 0;
  f -> cache.cnf.variable = 0;
  f -> cache.cnf.constraint[0] = 0;
  f -> cache.cnf.constraint[1] = 0;
  f -> cache.unary = 0;
  f -> cache.binary[0] = 0;
  f -> cache.binary[1] = 0;
  f -> cache2.unary = 0;
}

/*------------------------------------------------------------------------*/

static CTL new_CTL(CTL_Tag tag, void * a, void * b)
{
  CTL f, g, res, * p;
  Variable v;
  unsigned idx, number, h;

  f = (CTL) a;
  g = (CTL) b;
  v = (Variable) a;
  idx = (unsigned) b;
  number = (unsigned) a;

  /* simplify by propagating constants
   */
  switch(tag)
    {
      case AND_Tag:

	if(f == g || is_true_CTL(f) || is_false_CTL(g))
	  {
	    free_CTL(f);
	    return g;
	  }
	else
	if(is_true_CTL(g) || is_false_CTL(f))
	  {
	    free_CTL(g);
	    return f;
	  }
	else
	if((f -> tag == NOT_Tag && f -> data.arg[0] == g) ||
	   (g -> tag == NOT_Tag && g -> data.arg[0] == f))
	  {
	    free_CTL(f);
	    free_CTL(g);
	    return false_CTL();
	  }
	break;
      
      case OR_Tag:

	if(f == g || is_true_CTL(g) || is_false_CTL(f))
	  {
	    free_CTL(f);
	    return g;
	  }
	else
	if(is_true_CTL(f) || is_false_CTL(g))
	  {
	    free_CTL(g);
	    return f;
	  }
	else
	if((f -> tag == NOT_Tag && f -> data.arg[0] == g) ||
	   (g -> tag == NOT_Tag && g -> data.arg[0] == f))
	  {
	    free_CTL(f);
	    free_CTL(g);
	    return true_CTL();
	  }
	break;
      
      case IMPLIES_Tag:

	if(f == g)
	  {
	    free_CTL(f);
	    free_CTL(g);
	    return true_CTL();
	  }
	else
        if(is_true_CTL(g) || is_true_CTL(f))
	  {
	    free_CTL(f);
	    return g;
	  }
	else
	if(is_false_CTL(f))
	  {
	    free_CTL(f);
	    free_CTL(g);
	    return true_CTL();
	  }
	else
	if(is_false_CTL(g))
	  {
	    free_CTL(g);
	    return unary_CTL(NOT_Tag, f);
	  }
	break;
      
      case EQUIV_Tag:

	if(f == g)
	  {
	    free_CTL(f);
	    free_CTL(g);
	    return true_CTL();
	  }
	else
        if(is_true_CTL(g))
	  {
	    free_CTL(g);
	    return f;
	  }
	else
        if(is_true_CTL(f))
	  {
	    free_CTL(f);
	    return g;
	  }
	else
	if(is_false_CTL(g))
	  {
	    free_CTL(g);
	    return unary_CTL(NOT_Tag, f);
	  }
	else
	if(is_false_CTL(f))
	  {
	    free_CTL(f);
	    return unary_CTL(NOT_Tag, g);
	  }
	break;
      
      case NOT_Tag:

        if(is_true_CTL(f))
	  {
	    free_CTL(f);
	    return false_CTL();
	  }
	else
	if(is_false_CTL(f))
	  {
	    free_CTL(f);
	    return true_CTL();
	  }
	else
	if(f -> tag == NOT_Tag)
	  {
	    res = copy_CTL(f -> data.arg[0]);
	    free_CTL(f);
	    return  res;
	  }
	break;
      
      default:	/* no other optimizations needed yet */
	break;
    };

  /* now look up in the unique table
   */
  if(sharing)
    {
      if(!formulas)
	{
	  size_formulas = next_size(100);
	  formulas = (CTL*) calloc(size_formulas, sizeof(CTL));
	  num_formulas = max_formulas = 0;
	}

      if(num_formulas >= 3 * size_formulas) resize_formulas();

      lookups_formulas++;
      h = hash_CTL(tag, a, b) % size_formulas;

      for(p = &formulas[h]; *p; p = &(*p) -> next)
	{
	  visited_formulas++;
	  if(is_equal_CTL(tag, a, b, *p)) break;
	}

      if(*p)		/* found equal node */
	{
	  switch(tag)	/* release references to `f' and `g' */
	    {
	      case AND_Tag:
	      case EQUIV_Tag:
	      case OR_Tag:
	      case IMPLIES_Tag:
	      case EQUAL_Tag:
	      case CASE_Tag:
	      case COLON_Tag:
	      case ADD_Tag:
	      case SUB_Tag:
	      case SWITCH_Tag:
		free_CTL(f);
		free_CTL(g);
		break;

	      case NOT_Tag:
	      case INC_Tag:
	      case DEC_Tag:
	      case EX_Tag:
	      case EF_Tag:
	      case EG_Tag:
	      case AX_Tag:
	      case AF_Tag:
	      case AG_Tag:
		free_CTL(f);
		break;
	      
	      case CURRENT_Tag:
	      case NEXT_Tag:
	      case NUMBER_Tag:
	      case CONSTANT_Tag:
		break;

	      default:
		fatal(POSITION, "non valid tag\n");
		break;
	    }

	  return copy_CTL(*p);
	}
    }
  else p = &res;	/* hack to avoid another check on `sharing' below */

  num_formulas++;
  if(num_formulas > max_formulas) max_formulas = num_formulas;

  /* allocate and do non tag specific initialization
   */
  res = (CTL) malloc(sizeof(struct CTL_));
  *p = res;

  res -> tag = tag;
  res -> ref = 1;
  res -> is_marked = 0;
  res -> next = 0;
  _reset_cache(res);

  /* tag specific initialization
   */
  switch(tag)
    {
      case CONSTANT_Tag:

	assert(0);
        assert(number == 0 || number == 1);

        res -> data.number = number;
	res -> size = 1;
	res -> is_propositional = 1;
	res -> is_ACTL = 1;
	res -> is_ECTL = 1;
	res -> is_CLAUSE = 0;
	res -> is_flattened = 1;
	res -> is_nnf = 1;
	res -> is_cnf = 1;
	res -> is_current = 1;
        break;

      case CURRENT_Tag: 
      case NEXT_Tag: 

	res -> data.base.variable = v;
	res -> data.base.idx = idx;
	res -> size = 1;
	res -> is_propositional = 1;
	res -> is_ACTL = 1;
	res -> is_ECTL = 1;
	res -> is_CLAUSE = 1;
	res -> is_flattened = (v -> size == 1 || v -> size > idx);
	res -> is_nnf = 1;
	res -> is_cnf = 1;
	res -> is_current = (tag == CURRENT_Tag);
	break;
      
      case NUMBER_Tag:

	res -> data.number = number;
	res -> size = 1;
	res -> is_propositional = 1;
	res -> is_ACTL = 1;
	res -> is_ECTL = 1;
	res -> is_CLAUSE = 0;
	res -> is_flattened = (number <= 1);
	res -> is_nnf = 0;
	res -> is_cnf = 0;
	res -> is_current = 1;
	break;
      
      case ADD_Tag:
      case AND_Tag: 
      case CASE_Tag:
      case COLON_Tag:
      case EQUAL_Tag:
      case EQUIV_Tag: 
      case IMPLIES_Tag: 
      case OR_Tag: 
      case SUB_Tag:
      case SWITCH_Tag:

	assert(f);
	assert(g);

	res -> data.arg[0] = f;
	res -> data.arg[1] = g;
	res -> size = f -> size + g -> size + 1;
	res -> is_propositional = 
	  f -> is_propositional && g -> is_propositional;
	res -> is_ACTL = f -> is_ACTL && g -> is_ACTL;
	res -> is_ECTL = f -> is_ECTL && g -> is_ECTL;

	if(tag == OR_Tag) res -> is_CLAUSE = f -> is_CLAUSE && g -> is_CLAUSE;
	else res -> is_CLAUSE = 0;

	if(tag == AND_Tag)
	  {
	    res -> is_cnf = (f -> is_CLAUSE || f -> is_cnf) &&
	                    (g -> is_CLAUSE || g -> is_cnf);
	  }
	else res -> is_cnf = 0;

	switch(tag)
	  {
	    case AND_Tag:
	    case OR_Tag:
	    case IMPLIES_Tag:
	    case EQUIV_Tag:
	      res -> is_flattened = f -> is_flattened && g -> is_flattened;
	      break;

	    case CASE_Tag:
	    case COLON_Tag:
	    case ADD_Tag:
	    case SUB_Tag:
	    case EQUAL_Tag:
	    case SWITCH_Tag:
	    default:
	      res -> is_flattened = 0;
	      break;
	  }

	switch(tag)
	  {
	    case AND_Tag: 
	    case OR_Tag: 
	      res -> is_nnf = f -> is_nnf && g -> is_nnf;
	      break;

	    case EQUIV_Tag: 
	      if(flatten_equivs_flag) res -> is_nnf = 0;
	      else res -> is_nnf = f -> is_nnf && g -> is_nnf;
	      break;

	    case ADD_Tag:
	    case SUB_Tag:
	    case CASE_Tag:
	    case COLON_Tag:
	    case EQUAL_Tag:
	    case IMPLIES_Tag: 
	    case SWITCH_Tag:
	    default:
	      res -> is_nnf = 0;
	      break;
	  }

	res -> is_current = (f -> is_current && g -> is_current);
	break;
      
      case AF_Tag:
      case AG_Tag:
      case AX_Tag:
      case DEC_Tag:
      case EF_Tag:
      case EG_Tag:
      case EX_Tag:
      case INC_Tag:
      case NOT_Tag:

	assert(f);
	assert(!g);

	res -> data.arg[0] = f;
	res -> data.arg[1] = 0;
	res -> size = f -> size + 1;
	res -> is_CLAUSE = 0;		/* but see case for NOT below */
	res -> is_nnf = 0;		/* but see case for NOT below */
	res -> is_cnf = 0;		/* but see case for NOT below */

	switch(tag)
	  {
	    case NOT_Tag:
	      res -> is_ACTL = f -> is_ACTL;
	      res -> is_ECTL = f -> is_ECTL;
	      res -> is_propositional = f -> is_propositional;
	      res -> is_CLAUSE =
		f -> tag == NEXT_Tag || f -> tag == CURRENT_Tag;
	      res -> is_flattened = f -> is_flattened;
	      res -> is_current = f -> is_current;
	      res -> is_nnf = f -> tag == CURRENT_Tag || f -> tag == NEXT_Tag;
	      res -> is_cnf = f -> tag == CURRENT_Tag || f -> tag == NEXT_Tag;
	      break;

	    case EX_Tag:
	    case EF_Tag:
	    case EG_Tag:
	      res -> is_propositional = 0;
	      res -> is_ECTL = f -> is_ECTL;
	      res -> is_ACTL = 0;
	      res -> is_flattened = f -> is_flattened;
	      res -> is_current = 0;
	      break;

	    case AX_Tag:
	    case AF_Tag:
	    case AG_Tag:
	      res -> is_propositional = 0;
	      res -> is_ECTL = 0;
	      res -> is_ACTL = f -> is_ACTL;
	      res -> is_flattened = f -> is_flattened;
	      res -> is_current = 0;
	      break;
	    
	    case INC_Tag:
	    case DEC_Tag:

	      res -> is_ACTL = f -> is_ACTL;
	      res -> is_ECTL = f -> is_ECTL;
	      res -> is_propositional = f -> is_propositional;
	      res -> is_CLAUSE = 0;
	      res -> is_flattened = 0;
	      res -> is_current = f -> is_current;
	      break;

	    default:
	      fatal(POSITION, "non valid tag\n");
	      break;
	  }
	break;
      
      default:
	fatal(POSITION, "non valid tag\n");
	break;
    }

  return res;
}

/*------------------------------------------------------------------------*/

CTL current_CTL(Variable v, unsigned idx)
{
  return new_CTL(CURRENT_Tag, v, (void*) idx);
}

/*------------------------------------------------------------------------*/

CTL next_CTL(Variable v, unsigned idx)
{
  return new_CTL(NEXT_Tag, v, (void*) idx);
}

/*------------------------------------------------------------------------*/

CTL true_CTL()
{
  return new_CTL(NUMBER_Tag, (void*) 1, 0);
}

/*------------------------------------------------------------------------*/

CTL false_CTL()
{
  return new_CTL(NUMBER_Tag, 0, 0);
}

/*------------------------------------------------------------------------*/

CTL number_CTL(unsigned number)
{
  return new_CTL(NUMBER_Tag, (void*) number, 0);
}

/*------------------------------------------------------------------------*/

CTL unary_CTL(CTL_Tag tag, CTL f)
{
  return new_CTL(tag, f, 0);
}

/*------------------------------------------------------------------------*/

CTL binary_CTL(CTL_Tag tag, CTL f, CTL g)
{
  return new_CTL(tag, f, g);
}

/*------------------------------------------------------------------------*/

static unsigned _count_CTL(CTL f)
{
  unsigned res;
  Stack stack;
  CTL g;

  res = 0;
  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;
	  res++;
	  switch(g -> tag)
	    {
	      case AND_Tag:
	      case EQUIV_Tag:
	      case OR_Tag:
	      case IMPLIES_Tag:
	      case EQUAL_Tag:
	      case CASE_Tag:
	      case COLON_Tag:
	      case ADD_Tag:
	      case SUB_Tag:
	      case SWITCH_Tag:
		push(stack, g -> data.arg[1]);
		push(stack, g -> data.arg[0]);
		break;

	      case NOT_Tag:
	      case INC_Tag:
	      case DEC_Tag:
	      case EX_Tag:
	      case EF_Tag:
	      case EG_Tag:
	      case AX_Tag:
	      case AF_Tag:
	      case AG_Tag:
		push(stack, g -> data.arg[0]);
		break;
	      
	      case CURRENT_Tag:
	      case NEXT_Tag:
	      case NUMBER_Tag:
	      case CONSTANT_Tag:
		break;

	      default:
		fatal(POSITION, "non valid tag\n");
		break;
	    }
	}
    }

  free_Stack(stack);
  
  return res;
}

/*------------------------------------------------------------------------*/

unsigned count_CTL(CTL f)
{
  unsigned res;

  res = _count_CTL(f);
  unmark_CTL(f);

  return res;
}

/*------------------------------------------------------------------------*/

static unsigned _num_literals(CTL f)
{
  Stack stack;
  CTL g;
  unsigned res;

  stack = new_Stack();
  push(stack, f);
  res = 0;

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);

      if(g -> tag != OR_Tag)
        {
	  assert(g -> tag == NOT_Tag ||
	         g -> tag == CURRENT_Tag || g -> tag == NEXT_Tag);
	  res++;
	}
      else
        {
	  push(stack, g -> data.arg[0]);
	  push(stack, g -> data.arg[1]);
	}
    }
  
  free_Stack(stack);

  return res;
}

/*------------------------------------------------------------------------*/

unsigned num_literals(CTL f)
{
  unsigned res;

  if(!f -> is_CLAUSE) fatal(POSITION, "argument not a CLAUSE\n");
  res = _num_literals(f);

  return res;
}

/*------------------------------------------------------------------------*/

static unsigned _num_clauses(CTL f)
{
  Stack stack;
  CTL g;
  unsigned res;

  stack = new_Stack();
  push(stack, f);
  res = 0;

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);

      if(g -> is_CLAUSE) res++;
      else
        {
	  assert(g -> tag == AND_Tag);

	  push(stack, g -> data.arg[0]);
	  push(stack, g -> data.arg[1]);
	}
    }
  
  free_Stack(stack);

  return res;
}

/*------------------------------------------------------------------------*/

unsigned num_clauses(CTL f)
{
  unsigned res;

  if(!f -> is_cnf) fatal(POSITION, "argument not in CNF\n");

  if(is_true_CTL(f) || is_false_CTL(f)) res = 0;
  else res = _num_clauses(f);

  return res;
}

/*------------------------------------------------------------------------*/

void push_CTL(Stack stack, CTL f)
{
  switch(f -> tag)
    {
      case AND_Tag:
      case EQUIV_Tag:
      case OR_Tag:
      case IMPLIES_Tag:
      case EQUAL_Tag:
      case CASE_Tag:
      case COLON_Tag:
      case ADD_Tag:
      case SUB_Tag:
      case SWITCH_Tag:
	push(stack, f -> data.arg[1]);
	push(stack, f -> data.arg[0]);
	break;

      case NOT_Tag:
      case INC_Tag:
      case DEC_Tag:
      case EX_Tag:
      case EF_Tag:
      case EG_Tag:
      case AX_Tag:
      case AF_Tag:
      case AG_Tag:
	push(stack, f -> data.arg[0]);
	break;
      
      case CURRENT_Tag:
      case NEXT_Tag:
      case NUMBER_Tag:
      case CONSTANT_Tag:
	break;

      default:
	fatal(POSITION, "non valid tag\n");
	break;
    }
}

/*------------------------------------------------------------------------*/

void unmark_CTL(CTL f)
{
  CTL g;
  Stack stack;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(g -> is_marked)
        {
	  g -> is_marked = 0;
	  push_CTL(stack, g);
	}
    }

  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

void free_CTL(CTL f)
{
  CTL * p, g;
  Stack stack;
  unsigned h;

  assert(f -> ref);

  if(f -> ref == 1)
    {
      stack = new_Stack();
      push(stack, f);

      while(!is_mt_Stack(stack))
	{
	  g = (CTL) pop(stack);
	  assert(g -> ref);
	  g -> ref--;
	  if(!g -> ref)
	    {
	      push_CTL(stack, g);

	      if(sharing)
	        {
		  assert(formulas);

		  h = hash_CTL_from_CTL(g) % size_formulas;
		  for(p = &formulas[h]; *p && *p != g; p = &(*p) -> next)
		    ;

		  if(!*p) fatal(POSITION, "could not find node\n");

		  *p = g -> next;
		}
	      
	      free(g);
	      num_formulas--;
	    }
	}

      free_Stack(stack);
    }
  else f -> ref--;
}

/*------------------------------------------------------------------------*/

unsigned priority_CTL(CTL f)
{
  switch(f -> tag)
    {
      case CASE_Tag:
      case SWITCH_Tag:
        return 50;
      
      case COLON_Tag:
        return 75;

      case EQUIV_Tag:
      case IMPLIES_Tag:
	return 100;

      case OR_Tag:
	return 200;

      case AND_Tag:
	return 300;

      case NOT_Tag:
      case EX_Tag:
      case EG_Tag:
      case EF_Tag:
      case AX_Tag:
      case AG_Tag:
      case AF_Tag:
	return 400;

      case EQUAL_Tag:
	return 450;

      case CURRENT_Tag:
      case NEXT_Tag:
      case NUMBER_Tag:
      case INC_Tag:
      case DEC_Tag:
      case ADD_Tag:
      case SUB_Tag:
      case CONSTANT_Tag:
	return 500;

      default:
	fatal(POSITION, "non valid tag\n");
	return 1000;
	break;
    }
}

/*------------------------------------------------------------------------*/

CTL copy_CTL(CTL f)
{
  CTL res, f0, f1;

  if(sharing)
    {
      assert(f -> ref);

      f -> ref++;
      res = f;
    }
  else
    {
      switch(f -> tag)
	{
	  case AND_Tag:
	  case EQUIV_Tag:
	  case OR_Tag:
	  case IMPLIES_Tag:
	  case EQUAL_Tag:
	  case CASE_Tag:
	  case COLON_Tag:
	  case ADD_Tag:
	  case SUB_Tag:
	  case SWITCH_Tag:
	    f0 = copy_CTL(f -> data.arg[0]);
	    f1 = copy_CTL(f -> data.arg[1]);
	    res = binary_CTL(f -> tag, f0, f1);
	    break;

	  case NOT_Tag:
	  case INC_Tag:
	  case DEC_Tag:
	  case EX_Tag:
	  case EF_Tag:
	  case EG_Tag:
	  case AX_Tag:
	  case AF_Tag:
	  case AG_Tag:
	    f0 = copy_CTL(f -> data.arg[0]);
	    res = unary_CTL(f -> tag, f0);
	    break;
	  
	  case CURRENT_Tag:
	    res = current_CTL(f -> data.base.variable, f -> data.base.idx);
	    break;

	  case NEXT_Tag:
	    res = next_CTL(f -> data.base.variable, f -> data.base.idx);
	    break;

	  case NUMBER_Tag:
	    res = number_CTL(f -> data.number);
	    break;

	  case CONSTANT_Tag:
	    res = 0;
	    assert(0);
	    break;

	  default:
	    fatal(POSITION, "non valid tag\n");
	    res = 0;
	    break;
	}
    }

  return res;
}

/*------------------------------------------------------------------------*/

CTL _nnf_CTL(CTL f, int sign, int eliminate_equivs)
{
  CTL res, f0, f1, g0, g1, l, r;

  assert(f -> is_flattened);
  assert(sign == 0 || sign == 1);

  if(!f -> cache.binary[sign])
    {
      if(f -> is_nnf && !sign) res = copy_CTL(f);
      else
        {
	  switch(f -> tag)
	    {
	      case AND_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], sign, eliminate_equivs);
		f1 = _nnf_CTL(f -> data.arg[1], sign, eliminate_equivs);
		if(sign) res = binary_CTL(OR_Tag, f0, f1);
		else res = binary_CTL(AND_Tag, f0, f1);
		break;

	      case OR_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], sign, eliminate_equivs);
		f1 = _nnf_CTL(f -> data.arg[1], sign, eliminate_equivs);
		if(sign) res = binary_CTL(AND_Tag, f0, f1);
		else res = binary_CTL(OR_Tag, f0, f1);
		break;

	      case EQUIV_Tag:
		if(eliminate_equivs)
		  {
		    /* We prefer (a <-> b) = (a | !b)(!a | b)
		     * and       (a <+> b) = (a | b)(!a | !b)
		     * because this is better if we want to
		     * generate CNF later.
		     */
		    if(sign)
		      {
			f0 = _nnf_CTL(f -> data.arg[0], 0, 1);
			f1 = _nnf_CTL(f -> data.arg[1], 0, 1);
			l = binary_CTL(OR_Tag, f0, f1);

			g0 = _nnf_CTL(f -> data.arg[0], 1, 1);
			g1 = _nnf_CTL(f -> data.arg[1], 1, 1);
			r = binary_CTL(OR_Tag, g0, g1);

			res = binary_CTL(AND_Tag, l, r);
		      }
		    else
		      {
			f0 = _nnf_CTL(f -> data.arg[0], 0, 1);
			f1 = _nnf_CTL(f -> data.arg[1], 1, 1);
			l = binary_CTL(OR_Tag, f0, f1);

			g0 = _nnf_CTL(f -> data.arg[0], 1, 1);
			g1 = _nnf_CTL(f -> data.arg[1], 0, 1);
			r = binary_CTL(OR_Tag, g0, g1);

			res = binary_CTL(AND_Tag, l, r);
		      }
		  }
		else
		  {
		    f0 = _nnf_CTL(f -> data.arg[0], 0, eliminate_equivs);
		    f1 = _nnf_CTL(f -> data.arg[1], sign, eliminate_equivs);
		    res = binary_CTL(EQUIV_Tag, f0, f1);
		  }
		break;

	      case IMPLIES_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], !sign, eliminate_equivs);
		f1 = _nnf_CTL(f -> data.arg[1], sign, eliminate_equivs);
		if(sign) res = binary_CTL(AND_Tag, f0, f1);
		else res = binary_CTL(OR_Tag, f0, f1);
		break;

	      case EX_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], sign, eliminate_equivs);
		if(sign) res = unary_CTL(AX_Tag, f0);
		else res = unary_CTL(EX_Tag, f0);
		break;

	      case EF_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], sign, eliminate_equivs);
		if(sign) res = unary_CTL(AG_Tag, f0);
		else res = unary_CTL(EF_Tag, f0);
		break;

	      case EG_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], sign, eliminate_equivs);
		if(sign) res = unary_CTL(AF_Tag, f0);
		else res = unary_CTL(EG_Tag, f0);
		break;

	      case AX_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], sign, eliminate_equivs);
		if(sign) res = unary_CTL(EX_Tag, f0);
		else res = unary_CTL(AX_Tag, f0);
		break;

	      case AF_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], sign, eliminate_equivs);
		if(sign) res = unary_CTL(EG_Tag, f0);
		else res = unary_CTL(AF_Tag, f0);
		break;

	      case AG_Tag:
		f0 = _nnf_CTL(f -> data.arg[0], sign, eliminate_equivs);
		if(sign) res = unary_CTL(EF_Tag, f0);
		else res = unary_CTL(AG_Tag, f0);
		break;

	      case NOT_Tag:
		res = _nnf_CTL(f -> data.arg[0], !sign, eliminate_equivs);
		break;

	      case CURRENT_Tag:
		f0 = current_CTL(f -> data.base.variable, f -> data.base.idx);
		if(sign) res = unary_CTL(NOT_Tag, f0);
		else res = f0;
		break;

	      case NEXT_Tag:
		f0 = next_CTL(f -> data.base.variable, f -> data.base.idx);
		if(sign) res = unary_CTL(NOT_Tag, f0);
		else res = f0;
		break;
	      
	      case CONSTANT_Tag:
		res = 0;
		assert(0);
		break;
	      
	      case NUMBER_Tag:
		res = number_CTL(sign != f -> data.number);
		break;
	      
	      default:
		fatal(POSITION, "non valid tag\n");
		res = 0;
		break;
	    }
	}

      f -> cache.binary[sign] = res;
    }

  res = copy_CTL(f -> cache.binary[sign]);

  assert(res);

  return res;
}

/*------------------------------------------------------------------------*/

static void reset_cnf_cache(CTL f)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;

	  g -> cache.cnf.variable = 0;

	  if(g -> cache.cnf.constraint[0])
	    {
	      free_CTL(g -> cache.cnf.constraint[0]);
	      g -> cache.cnf.constraint[0] = 0;
	    }

	  if(g -> cache.cnf.constraint[1])
	    {
	      free_CTL(g -> cache.cnf.constraint[1]);
	      g -> cache.cnf.constraint[1] = 0;
	    }

	  push_CTL(stack, g);
	}
    }
  
  free_Stack(stack);
  unmark_CTL(f);
}

/*------------------------------------------------------------------------*/

static void reset_binary_cache(CTL f)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;

	  if(g -> cache.binary[0])
	    {
	      free_CTL(g -> cache.binary[0]);
	      g -> cache.binary[0] = 0;
	    }

	  if(g -> cache.binary[1])
	    {
	      free_CTL(g -> cache.binary[1]);
	      g -> cache.binary[1] = 0;
	    }

	  push_CTL(stack, g);
	}
    }
  
  free_Stack(stack);
  unmark_CTL(f);
}

/*------------------------------------------------------------------------*/

void reset_number_cache(CTL f)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;
	  g -> cache.number = 0;
	  push_CTL(stack, g);
	}
    }
  
  free_Stack(stack);
  unmark_CTL(f);
}

/*------------------------------------------------------------------------*/

void reset_variable_cache(CTL f)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;
	  g -> cache.variable = 0;
	  push_CTL(stack, g);
	}
    }
  
  free_Stack(stack);
  unmark_CTL(f);
}

/*------------------------------------------------------------------------*/

void reset_unary_cache(CTL f)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;

	  if(g -> cache.unary)
	    {
	      free_CTL(g -> cache.unary);
	      g -> cache.unary = 0;
	    }

	  push_CTL(stack, g);
	}
    }
  
  free_Stack(stack);
  unmark_CTL(f);
}

/*------------------------------------------------------------------------*/

void reset_unary_cache2(CTL f)
{
  Stack stack;
  CTL g;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;

	  if(g -> cache2.unary)
	    {
	      free_CTL(g -> cache2.unary);
	      g -> cache2.unary = 0;
	    }

	  push_CTL(stack, g);
	}
    }
  
  free_Stack(stack);
  unmark_CTL(f);
}

/*------------------------------------------------------------------------*/

CTL nnf_CTL(CTL f, int eliminate_equivs)
{
  CTL res;

  res = _nnf_CTL(f, 0, eliminate_equivs);
  reset_binary_cache(f);

# ifdef DEBUG
    if(debug)
      {
	printf("%sbegin(NNF)\n", comment_string);
	_print_pp_CTL(comment_string, res, BMC_PP_Format, 0, 2, 0);
	printf("%send(NNF)\n", comment_string);
      }
# endif

  return res;
}

/*------------------------------------------------------------------------*/

static char * _sub_prop_CTL_name(CTL f, unsigned index)
{
  char * res, * old_name;

  old_name = f -> data.base.variable -> name;
  res = (char*) malloc(strlen(old_name) + 20);
  sprintf(res, "%s%u_%s", prefix, index, old_name);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL _sub_prop_CTL(CTL f, unsigned cidx, unsigned nidx)
{
  CTL res, f0, f1;
  char * name;
  Variable v;

  assert(f -> is_flattened);
  assert(f -> is_propositional);

  if(!f -> cache.unary)
    {
      switch(f -> tag)
	{
	  case NOT_Tag:
	    f0 = _sub_prop_CTL(f -> data.arg[0], cidx, nidx);
	    res = unary_CTL(NOT_Tag, f0);
	    break;

	  case OR_Tag:
	  case AND_Tag:
	  case IMPLIES_Tag:
	  case EQUIV_Tag:

	    f0 = _sub_prop_CTL(f -> data.arg[0], cidx, nidx);
	    f1 = _sub_prop_CTL(f -> data.arg[1], cidx, nidx);
	    res = binary_CTL(f -> tag, f0, f1);
	    break;
	  
	  case NUMBER_Tag:

	    res = copy_CTL(f);
	    break;

	  case CURRENT_Tag:
	  case NEXT_Tag:

	    name = _sub_prop_CTL_name(f, f -> tag == CURRENT_Tag ? cidx : nidx);
	    v = new_Variable(name);
	    v -> size = f -> data.base.variable -> size;
	    free(name);
	    res = current_CTL(v, f -> data.base.idx);
	    break;
	  
	  default:

	    fatal(POSITION, "non valid tag\n");
	    res = 0;
	    break;
	}
      
      f -> cache.unary = res;
    }

  res = copy_CTL(f -> cache.unary);

  return res;
}

/*------------------------------------------------------------------------*/

CTL sub_prop_CTL(CTL f, unsigned cidx, unsigned nidx)
{
  CTL res;

  res = _sub_prop_CTL(f, cidx, nidx);
  reset_unary_cache(f);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL AND(CTL f, CTL g)
{
  CTL res;

  if(f == 0) return g;
  if(g == 0) return f;
  res = binary_CTL(AND_Tag, f, g);

  return res;
}

/*------------------------------------------------------------------------*/
static CTL OR(CTL f, CTL g)
{
  CTL res;

  if(f == 0) return g;
  if(g == 0) return f;
  res = binary_CTL(OR_Tag, f, g);

  return res;
}

/*------------------------------------------------------------------------*/

static void gen_vars(CTL f)
{
  switch(f -> tag)
    {
      case NOT_Tag:
	assert(f -> data.arg[0] -> tag == NEXT_Tag ||
	       f -> data.arg[0] -> tag == CURRENT_Tag);
	
	/* FALL THROUGH */

      case NEXT_Tag:
      case CURRENT_Tag:
      case CONSTANT_Tag:
	f -> cache.cnf.variable = 0;
	break;
      
      case EQUIV_Tag:
      case AND_Tag:
      case OR_Tag:
	if(!f -> cache.cnf.variable)
	  f -> cache.cnf.variable = generate_Symbol();
	break;

      default:
	fatal(POSITION, "non valid tag\n");
	break;
    }
}

/*------------------------------------------------------------------------*/

static CTL literal(CTL f, int sign)
{
  CTL res;

  switch(f -> tag)
    {
      case NOT_Tag:

	assert(f -> data.arg[0] -> tag == NEXT_Tag ||
	       f -> data.arg[0] -> tag == CURRENT_Tag);

	res = copy_CTL(f -> data.arg[0]);
	if(!sign) res = unary_CTL(NOT_Tag, res);
	break;

      case CURRENT_Tag:
      case NEXT_Tag:

	res = copy_CTL(f);
	if(sign) res = unary_CTL(NOT_Tag, res);
	break;
      
      default:

	assert(f -> cache.cnf.variable);

	res = current_CTL(f -> cache.cnf.variable, 0);
	if(sign) res = unary_CTL(NOT_Tag, res);
	break;
    }

  return res;
}

/*------------------------------------------------------------------------*/

static CTL clause2(CTL a, int a_sign, CTL b, int b_sign)
{
  CTL res;

  assert(a || b);

  res = literal(b, b_sign);
  res = binary_CTL(OR_Tag, literal(a, a_sign), res);

  assert(res);
  assert(res -> is_CLAUSE);

  return res;
}

/*------------------------------------------------------------------------*/

static CTL clause3(CTL a, int a_sign, CTL b, int b_sign, CTL c, int c_sign)
{
  CTL res;

  assert(a || b || c);

  res = literal(c, c_sign);
  res = binary_CTL(OR_Tag, literal(b, b_sign), res);
  res = binary_CTL(OR_Tag, literal(a, a_sign), res);

  assert(res);
  assert(res -> is_CLAUSE);

  return res;
}

/*------------------------------------------------------------------------*/

static void gen_constraints(CTL toplevel_formula, int rsimp)
{
  CTL g, g0, g1, clause, h, tmp0, tmp1;
  Stack stack, multiop_stack;

  assert(rsimp == 0 || rsimp == 1);

  if(toplevel_formula -> tag == CURRENT_Tag
     ||
     toplevel_formula -> tag == NEXT_Tag
     ||
     toplevel_formula -> tag == NOT_Tag
     ||
     (rsimp && toplevel_formula -> cache.cnf.constraint[0])
     ||
     (!rsimp && toplevel_formula -> cache.cnf.constraint[0] &&
                toplevel_formula -> cache.cnf.constraint[1]))
    {
      /* Avoid allocating stack if not necessary.
       */
      return;
    }

  multiop_stack = new_Stack();
  stack = new_Stack();

  gen_vars(toplevel_formula);
  push(stack, toplevel_formula);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);

      switch(g -> tag)
	{
	  case NOT_Tag:

	    assert(g -> data.arg[0] -> tag == NEXT_Tag ||
		   g -> data.arg[0] -> tag == CURRENT_Tag);

	    /* FALL THROUGH */

	  case CURRENT_Tag:
	  case NEXT_Tag:
	    
	    /* nothing to be done here */

	    break;

	  case AND_Tag:

	    if(no_multiargop_optimization)
	      {
		g0 = g -> data.arg[0];
		g1 = g -> data.arg[1];

		if(!g -> cache.cnf.constraint[0])
		  {
		    /*
		    (g -> (g0 & g1)) <-> (~g # g0) & (~g # g1)

					 `---v---'   `---v---'
					     a           b
		    */

		    gen_vars(g0);
		    gen_vars(g1);

/* b */		    tmp0 = clause2(g, 1, g1, 0);
/* a */		    tmp0 = binary_CTL(AND_Tag, clause2(g, 1, g0, 0), tmp0);
		
		    g -> cache.cnf.constraint[0] = tmp0;
		  }
		
		if(!rsimp && !g -> cache.cnf.constraint[1])
		  {
		    /*

		    (g <-> (g0 & g1)) <-> (~g # g0)&(~g # g1)&(g # ~g0 # ~g1)

					  `---v---' `---v---' `-----v-------'
					      a         b           c        
		    */

/* c */		    tmp1 = clause3(g, 0, g0, 1, g1, 1);
		    tmp1 = binary_CTL(AND_Tag, 
		             copy_CTL(g -> cache.cnf.constraint[0]), tmp1);

		    g  -> cache.cnf.constraint[1] = tmp1;
		  }

		push(stack, g0);
		push(stack, g1);
	      }
	    else
	      {
	        /*

		e.g.

		(u <-> (a & b & c & d))
		<->
		(~u # a)&(~u # b)&(~u # c)&(~u # d)&(u # ~a # ~b # ~c # ~d)

		if `renaming simplification' is allowed then:

		(u -> (a & b & c & d)) <-> (~u # a)&(~u # b)&(~u # c)&(~u # d)
        
		*/

		if(rsimp || g -> cache.cnf.constraint[1]) clause = 0;
		else clause = literal(g, 0);

		assert(is_mt_Stack(multiop_stack));
		push(multiop_stack, g);

		tmp0 = 0;
		while(!is_mt_Stack(multiop_stack))
		  {
		    h = pop(multiop_stack);

		    if(h -> tag == AND_Tag
#ifdef SHARE_VARS
		       && 
		       (h == g || h -> ref == 1)
#endif
		      )
		      {
			push(multiop_stack, h -> data.arg[0]);
			push(multiop_stack, h -> data.arg[1]);
		      }
		    else
		      {
			if(!g -> cache.cnf.constraint[0])
			  {
			    if(h -> tag != CURRENT_Tag &&
			       h -> tag != NEXT_Tag &&
			       h -> tag != NOT_Tag)
			      {
				gen_vars(h);
				push(stack, h);
			      }

			    tmp0 = AND(clause2(g, 1, h, 0), tmp0);
			  }

			if(!rsimp && !g -> cache.cnf.constraint[1])
			  {
			    assert(clause);
			    clause = binary_CTL(OR_Tag, literal(h, 1), clause);
			  }
		      }
		  }

		if(!g -> cache.cnf.constraint[0])
		  {
		    assert(tmp0);
		    g -> cache.cnf.constraint[0] = tmp0;
		  }

		if(!rsimp && !g -> cache.cnf.constraint[1])
		  {
		    assert(clause);
		    g -> cache.cnf.constraint[1] = clause;
		  }
	      }
	    break;

	  case OR_Tag:

	    if(no_multiargop_optimization)
	      {
		/*
		
		(g <-> (g0 # g1)) <-> (g # ~g0) & (g # ~g1) & (~g # g0 # g1)
		                      `---v---'   `---v---'   `-----v------'
		                          a           b              c 
		
		or if `renaming simplification' is enabled:
		
		(g -> (g0 # g1)) <-> (~g # g0 # g1)
		                     `-----v------'
		                           c
		*/

		g0 = g -> data.arg[0];
		g1 = g -> data.arg[1];

		if(!g -> cache.cnf.constraint[0])
		  {
		    gen_vars(g0);
		    gen_vars(g1);

		    tmp0 = clause3(g, 1, g0, 0, g1, 0);			/* c */
		    g -> cache.cnf.constraint[0] = tmp0;
		  }

		if((!rsimp || !renaming_simplification_for_or) &&
		   !g -> cache.cnf.constraint[1])
		  {
		    tmp1 = clause2(g, 0, g1, 1);			/* b */
		    tmp1 = AND(clause2(g, 0, g0, 1), tmp1);		/* a */

		    tmp1 = AND(copy_CTL(g -> cache.cnf.constraint[0]), tmp1);
		    g -> cache.cnf.constraint[1] = tmp1;
		  }
                
		push(stack, g0);
		push(stack, g1);
	      }
	    else
	      {
	        /*

		e.g.

		(u <-> (a # b # c # d))
		<->
		(u # ~a)&(u # ~b)&(u # ~c)&(u # ~d)&(~u # a # b # c # d)

		or if `renaming simplification' is enabled:

		(u -> (a # b # c # d)) <-> (~u # a # b # c # d)
		*/

		if(g -> cache.cnf.constraint[0]) clause = 0;
		else clause = literal(g, 1);

		assert(is_mt_Stack(multiop_stack));
		push(multiop_stack, g);

		tmp1 = 0;
		while(!is_mt_Stack(multiop_stack))
		  {
		    h = pop(multiop_stack);

		    if(h -> tag == OR_Tag
#ifdef SHARE_VARS
		       &&
		       (h == g || h -> ref == 1)
#endif
		      )
		      {
			push(multiop_stack, h -> data.arg[0]);
			push(multiop_stack, h -> data.arg[1]);
		      }
		    else
		      {
			if(!g -> cache.cnf.constraint[0])
			  {
			    if(h -> tag != CURRENT_Tag &&
			       h -> tag != NEXT_Tag &&
			       h -> tag != NOT_Tag)
			      {
				gen_vars(h);
				push(stack, h);
			      }

			  clause = binary_CTL(OR_Tag, literal(h, 0), clause);
			}

			if((!rsimp || !renaming_simplification_for_or) &&
			   !g -> cache.cnf.constraint[1])
			  {
			    tmp1 = AND(clause2(g, 0, h, 1), tmp1);
			  }
		      }
		  }

		if(!g -> cache.cnf.constraint[0])
		  {
		    assert(clause);
		    g -> cache.cnf.constraint[0] = clause;
		  }

		if((!rsimp || !renaming_simplification_for_or) &&
		   !g -> cache.cnf.constraint[1])
		  {
		    assert(tmp1);
		    g -> cache.cnf.constraint[1] = tmp1;
		  }
	      }

	    break;

	  case EQUIV_Tag:
	    /*

	    (g <-> (g0 <-> g1))
	    <->
	    ((g0 # g1 # g)&(g0 # ~g1 # ~g)&(~g0 # g1 # ~g)&(~g0 # ~g1 # g))

	    `------v-----' `------v------' `------v------' `------v-------'
		   a              b               c               d
		   
	    */

	    g0 = g -> data.arg[0];
	    g1 = g -> data.arg[1];

	    if(rsimp)
	      {
	        /* Now negative constraints for the variables generated
		 * for the subformulas have to be considered as well
		 */
	        gen_constraints(g0, 0);
	        gen_constraints(g1, 0);
	      }

	    if(!g -> cache.cnf.constraint[0])
	      {
		gen_vars(g0);
		gen_vars(g1);

		tmp0 = clause3(g, 1, g0, 1, g1, 0);		/* d */
		tmp0 = AND(clause3(g, 1, g0, 0, g1, 1), tmp0);	/* c */
		tmp0 = AND(clause3(g, 0, g0, 1, g1, 1), tmp0);	/* b */
		tmp0 = AND(clause3(g, 0, g0, 0, g1, 0), tmp0);	/* a */

		g -> cache.cnf.constraint[0] = tmp0;
		g -> cache.cnf.constraint[1] = 0;

		push(stack, g0);
		push(stack, g1);
	      }

	    break;

	  default:
	    fatal(POSITION, "non valid tag\n");
	    break;
	}
    } 
  
  free_Stack(multiop_stack);
  free_Stack(stack);
}

/*------------------------------------------------------------------------*/

static void _cnf(CTL f, CTL * res_ptr)
{
  CTL g, g0, g1, l, r, not_l, not_r, tmp;
  Stack stack;

  stack = new_Stack();
  push(stack, f);

  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);

      switch(g -> tag)
	{
	  case EQUIV_Tag:

	    g0 = g -> data.arg[0];
	    g1 = g -> data.arg[1];

	    gen_constraints(g1, 0);
	    gen_constraints(g0, 0);

	    if(g0 -> cache.cnf.variable)
	      {
		l = current_CTL(g0 -> cache.cnf.variable, 0);
		not_l = unary_CTL(NOT_Tag, copy_CTL(l));
	      }
	    else
	      {
		l = copy_CTL(g0);

		if(g0 -> tag == NOT_Tag)
		  {
		    assert(g0 -> data.arg[0] -> tag == CURRENT_Tag ||
			   g0 -> data.arg[0] -> tag == NEXT_Tag);
		    
		    not_l = copy_CTL(g0 -> data.arg[0]);
		  }
		else
		  {
		    assert(g0 -> tag == CURRENT_Tag || g0 -> tag == NEXT_Tag);

		    not_l = unary_CTL(NOT_Tag, copy_CTL(l));
		  }
	      }

	    if(g1 -> cache.cnf.variable)
	      {
		r = current_CTL(g1 -> cache.cnf.variable, 0);
		not_r = unary_CTL(NOT_Tag, copy_CTL(r));
	      }
	    else
	      {
		r = copy_CTL(g1);

		if(g1 -> tag == NOT_Tag)
		  {
		    assert(g1 -> data.arg[0] -> tag == CURRENT_Tag ||
			   g1 -> data.arg[0] -> tag == NEXT_Tag);
		    
		    not_r = copy_CTL(g1 -> data.arg[0]);
		  }
		else
		  {
		    assert(g1 -> tag == CURRENT_Tag || g1 -> tag == NEXT_Tag);

		    not_r = unary_CTL(NOT_Tag, copy_CTL(r));
		  }
	      }
	    
	    tmp = binary_CTL(OR_Tag, l, not_r);
	    *res_ptr = AND(tmp, *res_ptr);

	    tmp = binary_CTL(OR_Tag, not_l, r);
	    *res_ptr = AND(tmp, *res_ptr);

	    break;

	  case EX_Tag:
	  case EF_Tag:
	  case EG_Tag:
	  case AX_Tag:
	  case AF_Tag:
	  case AG_Tag:
	    fatal(POSITION,
	      "can not generate CNF containing temporal operator\n");
	    break;

	  case AND_Tag:

	    push(stack, g -> data.arg[0]);
	    push(stack, g -> data.arg[1]);
	    break;
	  
	  case NEXT_Tag:
	  case CURRENT_Tag:

	    *res_ptr = AND(copy_CTL(g), *res_ptr);
	    break;
	  
	  case NOT_Tag:

	    if(g -> data.arg[0] -> tag != NEXT_Tag &&
	       g -> data.arg[0] -> tag != CURRENT_Tag)
	      fatal(POSITION,
		"can not handle arbitrary negated formulas\n");

	    *res_ptr = AND(copy_CTL(g), *res_ptr);
	    break;
	  
	  case OR_Tag:

	    if(g -> is_CLAUSE) *res_ptr = AND(copy_CTL(g), *res_ptr);
	    else
	      {
		g0 = g -> data.arg[0];
		g1 = g -> data.arg[1];

		gen_constraints(g1, renaming_simplification);
		gen_constraints(g0, renaming_simplification);
		
		if(g0 -> cache.cnf.variable)
		  {
		    l = current_CTL(g0 -> cache.cnf.variable, 0);
		  }
		else l = copy_CTL(g0);

		if(g1 -> cache.cnf.variable)
		  {
		    r = current_CTL(g1 -> cache.cnf.variable, 0);
		  }
		else r = copy_CTL(g1);

		tmp = binary_CTL(OR_Tag, l, r);
		*res_ptr = AND(tmp, *res_ptr);
	      }
	    break;
	  
	  case NUMBER_Tag:
	  case CONSTANT_Tag:

	    *res_ptr = AND(copy_CTL(g), *res_ptr);
	    break;
	  
	  default:
	    fatal(POSITION, "non valid tag\n");
	    break;
	}
    }
  
  push(stack, f);		/* now get all the generated constraints */
  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(!g -> is_marked)
        {
	  g -> is_marked = 1;

	  if(g -> cache.cnf.constraint[0])
	    *res_ptr = AND(copy_CTL(g -> cache.cnf.constraint[0]), *res_ptr);

	  if(g -> cache.cnf.constraint[1])
	    *res_ptr = AND(copy_CTL(g -> cache.cnf.constraint[1]), *res_ptr);

	  push_CTL(stack, g);
	}
    }

  unmark_CTL(f);
  free_Stack(stack);
  reset_cnf_cache(f);

  assert((*res_ptr) -> is_flattened);
}

/*------------------------------------------------------------------------*/

static int cmp_clauses(const void * a, const void * b)
{
  CTL f, g;
  unsigned nl_f, nl_g;

  f = *(CTL*) a;
  g = *(CTL*) b;

  if(f == g) return 0;

  nl_f = f -> cache.number;
  nl_g = g -> cache.number;

# ifdef VERYDEBUG
    {    
      assert(nl_f == num_literals(f));
      assert(nl_g == num_literals(g));
    }
# endif

  if(nl_f < nl_g) return -1;
  else 
  if(nl_f == nl_g) return f < g ? -1 : 1;
  else return 1;
}

/*------------------------------------------------------------------------*/

static int cmp_literals(const void * a, const void * b)
{
  CTL f, g;

  f = *(CTL*) a;
  g = *(CTL*) b;

  if(f < g) return -1;
  else if(a == b) return 0;
  else return 1;
}

/*------------------------------------------------------------------------*/

static CTL sort_clause(CTL f)
{
  Stack stack;
  CTL g, * a, * p, last, res;
  unsigned i, n, uniq_literals;

  n = num_literals(f);
  a = (CTL*) calloc(n, sizeof(CTL));

  stack = new_Stack();
  push(stack, f);

  p = a;
  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);

      if(g -> tag != OR_Tag)
        {
	  assert(g -> tag == NOT_Tag ||
	         g -> tag == CURRENT_Tag || g -> tag == NEXT_Tag);
	  
	  *p++ = g;
	}
      else
        {
	  push(stack, g -> data.arg[0]);
	  push(stack, g -> data.arg[1]);
	}
    }
  
  assert(p - a == n);

  qsort(a, n, sizeof(CTL), cmp_literals);

  for(uniq_literals = 0, res = 0, last = 0, i = 0; i < n; i++)
    {
      if(last != a[i])
	{
	  res = OR(copy_CTL(a[i]), res);
	  uniq_literals++;
	}
      last = a[i];
    }
  
  free(a);
  free_Stack(stack);

  res -> cache.number = uniq_literals;

  return res;
}

/*------------------------------------------------------------------------*/

static CTL sort_cnf(CTL f)
{
  Stack stack;
  CTL g, * a, * p, last, res;
  unsigned i, n;

  n = num_clauses(f);
  a = (CTL*) calloc(n, sizeof(CTL));

  stack = new_Stack();
  push(stack, f);

  p = a;
  while(!is_mt_Stack(stack))
    {
      g = (CTL) pop(stack);
      if(g -> is_CLAUSE) *p++ = sort_clause(g);
      else
        {
	  assert(g -> tag == AND_Tag);

	  push(stack, g -> data.arg[0]);
	  push(stack, g -> data.arg[1]);
	}
    }
  
  assert(p - a == n);

  qsort(a, n, sizeof(CTL), cmp_clauses);

  for(res = 0, last = 0, i = 0; i < n; i++)
    {
      assert(a[i] -> cache.number);

      if(last != a[i]) res = AND(copy_CTL(a[i]), res);
      last = a[i];
    }
  
  for(i = 0; i < n; i++)
    {
      a[i] -> cache.number = 0;
      free_CTL(a[i]);
    }

  free(a);
  free_Stack(stack);

  return res;
}

/*------------------------------------------------------------------------*/

CTL cnf(CTL f)
{
  CTL normalized, res, sorted;
  double f_size, normalized_size, res_size, sorted_size;
  double f_nodes, normalized_nodes, res_nodes, sorted_nodes;
  double new_nodes, new_size;

  start_verbose("generating CNF");

  f_size = f -> size;
  f_nodes = count_CTL(f);

  start_verbose("1. phase: generating NNF");
  normalized = nnf_CTL(f, flatten_equivs_flag);
  normalized_size = normalized -> size;
  normalized_nodes = count_CTL(normalized);
  end_verbose(
    "size %.0f (%.0f nodes = %.0f%%) x %.2f (%.2f)",
    normalized_size, normalized_nodes,
    100.0 * normalized_nodes / normalized_size,
    normalized_size / f_size, normalized_nodes / f_nodes);

  start_verbose("2. phase: introducing variables and constraints");
  res = 0;
  _cnf(normalized, &res);
  res_size = res -> size;
  res_nodes = count_CTL(res);

# ifdef DEBUG
    if(debug)
      {
	printf("%sbegin(CNF)\n", comment_string);
	_print_pp_CTL(comment_string, res, BMC_PP_Format, 0, 2, 0);
	printf("%send(CNF)\n", comment_string);
      }
# endif

  end_verbose("size %.0f (%.0f nodes = %.0f%%) x %.2f (%.2f)",
    res_size, res_nodes,
    100.0 * res_nodes / res_size,
    res_size / normalized_size, res_nodes / normalized_nodes);
  free_CTL(normalized);
  
  if(sort_flag)
    {
      start_verbose("sorting clauses in CNF");
      sorted = sort_cnf(res);
      sorted_size = sorted -> size;
      sorted_nodes = count_CTL(sorted);
      end_verbose("size %.0f (%.0f nodes = %.0f%%) x %.2f (%.2f)",
	sorted_size, sorted_nodes,
	100.0 * sorted_nodes / sorted_size,
	sorted_size / res_size, sorted_nodes / res_nodes);
      
      free_CTL(res);
      
      res = sorted;
    }
  
  new_size = res -> size;
  new_nodes = count_CTL(res);

  end_verbose("CNF: size %.0f (%.0f nodes = %.0f%%) x %.2f (%.2f)",
    new_size, new_nodes, 100.0 * new_nodes / new_size,
    new_size / f_size, new_nodes / f_nodes);
  
  return res;
}

/*------------------------------------------------------------------------*/

int check_width_CTL(unsigned max_width, CTL f)
{
  int res;
  unsigned idx;
  Variable v;
  CTL g, h;

  switch(f -> tag)
    {
      case INC_Tag:
      case DEC_Tag:
        res = check_width_CTL(max_width, f -> data.arg[0]);
	break;

      case ADD_Tag:
      case SUB_Tag:
        res = check_width_CTL(max_width, f -> data.arg[0]);
	if(res) res = check_width_CTL(max_width, f -> data.arg[1]);
	break;
      
      case CURRENT_Tag:
      case NEXT_Tag:
        v = f -> data.base.variable;
	idx = f -> data.base.idx;
	res = (max_width == ((v -> size == idx) ? idx : 1));
	break;
      
      case SWITCH_Tag:
      case CASE_Tag:
        
	for(res = 1, g = f; res && g -> tag == f -> tag; g = g -> data.arg[1])
	  {
	    h = g -> data.arg[0];
	    assert(h -> tag == COLON_Tag);
	    res = check_width_CTL(max_width, h -> data.arg[1]);
	  }
	
	if(res)
	  {
	    assert(g -> tag == COLON_Tag);
	    res = check_width_CTL(max_width, g -> data.arg[1]);
	  }
	break;
      
      case NUMBER_Tag:
	res = max_width >= 32 || (pow2(max_width) > f -> data.number);
        break;
      
      case AND_Tag:
      case OR_Tag:
      case EQUIV_Tag:
      case IMPLIES_Tag:
      case EQUAL_Tag:
        res = max_width == 1;
	if(res) res = check_width_CTL(1, f -> data.arg[0]);
	if(res) res = check_width_CTL(1, f -> data.arg[1]);
        break;
      
      case NOT_Tag:
        res = max_width == 1;
	if(res) res = check_width_CTL(1, f -> data.arg[0]);
        break;
      
      default:
	fatal(POSITION, "non valid tag\n");
        res = 0;
        break;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

unsigned contains_var_with_width(CTL f)
{
  unsigned res;
  CTL g;

  switch(f -> tag)
    {
      case CURRENT_Tag:
      case NEXT_Tag:
        if(f -> data.base.variable -> size == f -> data.base.idx)
	  {
	    res = f -> data.base.idx;
	  }
	else res = 1;
	break;
      
      case INC_Tag:
      case DEC_Tag:
        res = contains_var_with_width(f -> data.arg[0]);
	break;
      
      case ADD_Tag:
      case SUB_Tag:
        res = contains_var_with_width(f -> data.arg[0]);
	if(!res) res = contains_var_with_width(f -> data.arg[1]);
	break;
      
      case SWITCH_Tag:
      case CASE_Tag:

	for(res = 0, g = f; !res && g -> tag == f -> tag; g = g -> data.arg[1])
	  {
	    assert(g -> data.arg[0] -> tag == COLON_Tag);
	    res = contains_var_with_width(g -> data.arg[0] -> data.arg[1]);
	  }
	
	if(!res)
	  {
	    assert(g -> tag == COLON_Tag);
	    res = contains_var_with_width(g -> data.arg[1]);
	  }

	break;
      
      case NUMBER_Tag:
        res = 0;
	break;

      default:
        res = 0;
	fatal(POSITION, "non valid tag\n");
	break;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

unsigned width_CTL(CTL f)
{
  Variable v;
  unsigned idx, max, tmp, res;
  CTL g;

  switch(f -> tag)
    {
      case INC_Tag:
      case DEC_Tag:
	res = width_CTL(f -> data.arg[0]);
	break;

      case ADD_Tag:
      case SUB_Tag:
	res = width_CTL(f -> data.arg[0]);
	assert(res == width_CTL(f -> data.arg[1]));
	break;
      
      case CURRENT_Tag:
      case NEXT_Tag:
	v = f -> data.base.variable;
	idx = f -> data.base.idx;
        res = (v -> size == idx) ? idx : 1;
	break;
      
      case SWITCH_Tag:
      case CASE_Tag:		/* from `case' statement */

        for(max = 0, g = f; g -> tag == f -> tag; g = g -> data.arg[1])
	  {
	    assert(g -> data.arg[0] -> tag == COLON_Tag);

	    tmp = width_CTL(g -> data.arg[0] -> data.arg[1]);
	    if(tmp > max) max = tmp;
	  }

	assert(g -> tag == COLON_Tag);

	tmp = width_CTL(g -> data.arg[1]);
	if(tmp > max) max = tmp;
	res = max;
        break;
      
      case NUMBER_Tag:
	res = log2(f -> data.number);
        break;

      default:
	fatal(POSITION, "non valid tag\n");
	res = 0;
	break;
    }
  
  return res;
}

/*------------------------------------------------------------------------*/

void init_CTL()
{
  num_formulas = max_formulas = size_formulas = 0;
  visited_formulas = lookups_formulas = 0;
}

/*------------------------------------------------------------------------*/

void exit_CTL()
{
  double avg;
  CTL p, tmp;
  unsigned i;

  if(verbose)
    {
      if(sharing)
	{
	  if(lookups_formulas)
	    {
	      avg = ((double)visited_formulas) / ((double)lookups_formulas);
	      print_verbose("formulas: num %u, max %u, size %u, avg %.2f\n",
		num_formulas, max_formulas, size_formulas, avg);
	    }
	  else
	    {
	      print_verbose("formulas: num %u, max %u, size %u\n",
		num_formulas, max_formulas, size_formulas);
	    }
	}
      else 
	{
	  print_verbose("formulas: num %u, max %u\n",
	    num_formulas, max_formulas);
	}
    }

  if(formulas)
    {
      assert(sharing);
      
      for(i = 0; i < size_formulas; i++)
	{
	  for(p = formulas[i]; p; p = tmp)
	    {
	      tmp = p -> next;
	      free(p);
	    }
	}

      free(formulas);
      formulas = 0;
      size_formulas = num_formulas = 0;
    }
}
