/***************************************************************************
                          user.h  -  description
                             -------------------
    begin                : Wed Aug 8 2001
    copyright            : (C) 2001 by Yinglian Xie
    email                : ylxie@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
// user.h: interface for the CUserGroup class.
//
//////////////////////////////////////////////////////////////////////

#ifndef USER_H
#define USER_H

#include <unistd.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <string>
#include <list>
#include <functional>
#include <algorithm>
#include "utils.h"
using namespace std;

/*-----security setup files-------*/
#define USER_FNAME       ".userinfo"

/*---- SPD ID type -----*/
#define USER   0
#define GROUP  1
#define AFS    2

/*------- Mingle user structure -------- 
-----------used in request manager, qprocessor and security manager --------
*/
typedef struct{
	int    uid;   // if uid == -1, it is a remote user
	string mgid;  // if mgid.empty(), it is a local user
	string afsid; // if afsid.empty(), it is not a valid afs user
}Identity;

typedef struct{
	string keystring;
	time_t expire_tv;
}TRSAKey;

/*--------Class Mingle User ------------ 
*/
class Muser{
 public:
	string  m_mgid;    // mingle user id, cannot be empty
	TRSAKey m_pubkey;  // public key string  for "digital signature", 
	                   // empty if no key string is specified 
 public:
	Muser(string mgid);
	~Muser();

	void update_pubkey(string pubkey, time_t expire_tv);
};

class User_eq: public unary_function<Muser, int>{
	string s;
 public:
	explicit User_eq(string& ss): s(ss){}
	int operator() (const Muser& user) const{ return user.m_mgid == s;}
};

typedef list<Muser, allocator<Muser> > TUserSet;


/*--------class Arp: Access-right mapping -------------
*/
typedef struct{
	string aname;
	int    aright;
}TAfsRight;

typedef list<TAfsRight> TAfsRList;

class Arp{
 public:
	string    mgid;
	list<uid_t>  SPD_user;
	list<gid_t>  SPD_group;
	TStrList     SPD_afs;
 public:
	Arp(string id);
	~Arp();

	int    insert_SPD(uint nid, int mode);
	int    delete_SPD(uint nid, int mode);
	string list_SPD();
	void   flush_info(FILE *fp);
	void   setup_info(char* buf);
};

class Arp_eq: public unary_function<Arp, int>{
	string s;
 public:
	explicit Arp_eq(string& ss): s(ss){}
	int operator() (const Arp& arp) const{ return arp.mgid == s;}
};

typedef list<Arp> TArpList;


/*-------- class Mingle Owner ---------------
*/
class  Mowner{
 public:
	uid_t    m_uid;
	string   m_mgid;
	TRSAKey  m_prikey;  // private key file, empty if no private key

 private:
	TArpList m_arp;

 public:
	Mowner(uid_t uid, string mgid);
	~Mowner();

	void update_prikey(string prikey, time_t expire_tv);
	bool exist_map(string mgid);

	int  get_SPD(string mgid, list<uid_t>& uSPD, list<gid_t>& gSPD);
	int  clear_SPD(string mgid);

	int  insert_arp(string mgid, uint nid, int mode);
	int  delete_arp(string mgid, uint nid, int mode);
	string  list_arp(string mgid);

	void flush_info(FILE *fp);
	int  setup_arp(FILE *fp, int arpsize);
};

class Owner_eq: public unary_function<Mowner, int>{
        int u;
 public:
	explicit Owner_eq(int& uu): u(uu){}
	int operator() (const Mowner& owner) const{ return owner.m_uid == (uid_t)u;}
};

typedef list<Mowner, allocator<Mowner> > TOwnerSet;


/*------- class CUserGroup --------------
*/
class CUserGroup{
 private:
	TUserSet   m_users;      // local user record
	TOwnerSet  m_owners;     // local owner record

	int     m_securityMode;  // whether check security in running

	string  m_securitydir;   // dir where security info is stored
	string  m_userfile;      // local user information 

	string  m_prikey;        // mingle host private key
	int     m_prikey_ttl;     // private key cache ttl (in second)
	int     m_pubkey_ttl;     // public key cache ttl  (in second)  

 public:
	string  m_pubkey;        // mingle host public key  

 public:
	CUserGroup();
	~CUserGroup();

	/* config file set up functions */
	void set_security_mode(int mode);
	string setup_security(string dir, string pubkey, string prikey,
			      int pubkey_ttl, int prikey_ttl);

	/* rsa key caching functions */
	int  owner_exist(uid_t uid);
	void mingle_init(Identity& muser, string prikey);
	void update_pubkey(string mgid, string pubkey);
	int  add_pubkey_file(string mgid, string keyfile, string& errstr);

	/* access control functions */
	void display_userinfo(uid_t uid, string& output);
	int  add_arp(Identity& user, string mgid, string localID,
		     int mode, string& errstr);
	int  rm_arp(Identity& user, string mgid, string localID,
		    int mode, string& errstr);
	int  clear_SPD(Identity& user, string mgid, string& errstr);

	void ls_arp(Identity& user, string mgid, string& errstr);

	/* permission checking functions */
	bool  index_file_permitted(Identity& user, string file);
	bool  search_file_permitted(Identity& user, string file);

	/* authentication functions */
	void retrieve_identity(Identity& user);
	int  identity_confirmed(string mgid, string cmdline, 
				string signature);
	int  generate_signature(Identity& user, string input,
				string& signature);
	string encrypt_string(string pubkey, string cleartext);
	string decrypt_string(string ciphertext);

	inline string  get_mgid(uid_t uid);

 private:
	inline Mowner* get_owner_uid(uid_t uid);
	inline Mowner* get_owner_mgid(string mgid);
	inline Muser*  get_muser(string mgid);	
	inline TUserSet::iterator get_muser_iter(string mgid);

	/* security information setup and store */
	void  read_security();
	void  flush_security();
};


/***************************************************************
 *************** Inline function implementation ****************
 **************************************************************/

//////////////////////////////////////////////////////////////////////
//  class  CUserGroup inline functions
//////////////////////////////////////////////////////////////////////

/*
 * get_owner_uid: get owner record from uid
 *            return NULL if no such owner record
 */
inline Mowner* CUserGroup::get_owner_uid(uid_t uid)
{
	TOwnerSet::iterator i;
	for (i = m_owners.begin(); i != m_owners.end(); i ++){
		if (i->m_uid == uid)
			return &(*i);
	}
	return 0;
}

/*
 * get_owner_mgid: get owner record from mingle id
 *            return NULL if no such owner record
 */
inline Mowner* CUserGroup::get_owner_mgid(string mgid)
{
	TOwnerSet::iterator i;
	for (i = m_owners.begin(); i != m_owners.end(); i ++){
		if (i->m_mgid == mgid)
			return &(*i);
	}
	return 0;
}

/*
 * get_muser: get mingle user record from mgid
 *            return NULL if no such user record
 */
inline Muser* CUserGroup::get_muser(string mgid)
{
	TUserSet::iterator i;
	for (i = m_users.begin(); i != m_users.end(); i ++){
		if (i->m_mgid == mgid)
			return &(*i);
	}
	return 0;
}


/*
 * get_muser_iter: get mingle user list iterator from mgid
 *                 return end() if no such user record
 */
inline TUserSet::iterator CUserGroup::get_muser_iter(string mgid)
{
	TUserSet::iterator i;
	for (i = m_users.begin(); i != m_users.end(); i ++){
		if (i->m_mgid == mgid)
			break;
	}
	return i;
}

inline string CUserGroup::get_mgid(uid_t uid)
{
	string mgid;

	Mowner *owner = get_owner_uid(uid);
	if (owner) return owner->m_mgid;
	else return mgid;
}

#endif //USER_H
