/***************************************************************************
                          sender.cpp  -  description
                             -------------------
    begin                : Mon Sep 24 2001
    copyright            : (C) 2001 by Yinglian Xie
    email                : ylxie@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
// sender.cpp: implementation for the CSender class.
//
//////////////////////////////////////////////////////////////////////

#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <assert.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include "sender.h"
#include "request.h"
#include "fdmanager.h"
#include "utils.h"

#ifndef SHUT_WR
#define SHUT_WR 1
#endif	

//////////////////////////////////////////////////////////////////////
//  class  CSender implementation
//////////////////////////////////////////////////////////////////////

CSender::CSender(){
}

CSender::~CSender(){
}

/*
 * socket_connect: set up a TCP connection with a remote server
 */
int CSender::socket_connect(CRequest *req)
{
	int sockfd;
	struct sockaddr_in rmaddr;
	
	sockfd = socket(AF_INET, SOCK_STREAM, 0);
	if (sockfd < 0){
		
		mingle_debug2("\nCSender::socket_connect(): socket create error",
						errno);
		return -1;				
	}	
	
	bzero(&rmaddr, sizeof(rmaddr));
	rmaddr.sin_family = AF_INET;
	rmaddr.sin_port = htons(req->m_port);
	memcpy(&rmaddr.sin_addr, &req->m_host, 4);
	
	if (connect(sockfd, (struct sockaddr *)&rmaddr, sizeof(rmaddr)) == -1){
		mingle_debug2("\nCSender::socket_connect(): connect error", errno);
		close(sockfd);
		return -1;
	}
	
	req->m_sd = sockfd;
	
	return 0;
}

/* 
 * keep_writing: write until not allowed
 *               return: 0 writing finished, can clear sd after return
 *			-1 need more writing next time
 */				 			
int CSender::keep_writing(int sd)
{
	int num;
	
	CRequest *req = m_reqQ->find(sd);
	assert(req);
	
	while (1){
		
		num = write(sd, req->m_writePtr, req->m_nwsize);
		if (num == -1){
			// not ready to read
			if ((errno == EINTR) || (errno == EAGAIN)){
				return -1;
			}else{
				// read error
				mingle_debug2("\nCSender::keep_writing():write error",
					      errno);			
				m_reqQ->remove_request(req);
				close(sd);
				return 0;	
			}
		}
		
		req->m_nwsize -= num;
		req->m_twsize -= num;
		
		if (req->m_twsize == 0){
			// case 1: simply send back reply and we are done now
			if ((req->m_type == NORMAL)){
				close(sd);
				m_reqQ->remove_request(req);
				return 0;
			}	
			// case 2: we just forward a request, now expect the reply
			if ((req->m_type == CHILD) || (req->m_type == MASTER)){
				shutdown(sd, SHUT_WR);
				m_fdManager->set_rfd(sd);
				return 0;
			}
			// case 3: we just forward back a reply to the request initiator
			//    now check the dependency and remove useless data
			if (req->m_type == PARENT){
				//			req->m_status = PROCESS_DONE;
				if (req->m_outref == 0){
					close(sd);
					m_reqQ->remove_request(req);
				}else{
					req->delete_outdata();
				}				
				return 0;
			}
		}
	    	
		if (req->m_nwsize == 0){
			req->m_wbuf ++;
			req->m_writePtr = req->m_outdata[req->m_wbuf];
			req->m_nwsize = (req->m_twsize > BUFLEN) ? BUFLEN : req->m_twsize;        	
		}else{
			req->m_writePtr += num;
		}
	} // end while
	
	return -1;
}

/*
 * send_reply: send out a reply to a remote server (or local client)
 */
void CSender::send_reply(CRequest *req)
{
	assert((req != 0) && (req->m_twsize > 0));
	
	// set the fd to wfdSet of fdmanager for more writing
	req->m_wbuf = 0;
	req->m_writePtr = req->m_outdata[0];
	req->m_nwsize = (req->m_twsize > BUFLEN) ? BUFLEN : req->m_twsize;
	m_fdManager->set_wfd(req->m_sd);	
}

/*
 * send_request: send out (or forward) a request to a remote server
 */
int CSender::send_request(CRequest *req)
{
	assert((req != 0) && (req->m_twsize > 0));
	
	// set up connection to remote host
	if (socket_connect(req) == -1){
		return -1;		
	}
	
	// set the fd to wfdSet of fdmanager for more writing
	req->m_wbuf = 0;
	req->m_writePtr = req->m_outdata[0];
	req->m_nwsize = (req->m_twsize > BUFLEN) ? BUFLEN : req->m_twsize;
	m_fdManager->set_wfd(req->m_sd);	
	return 0;
}

