/***************************************************************************
                          security.cpp  -  description
                             -------------------
    begin                : Mon Sep 24 2001
    copyright            : (C) 2001 by Yinglian Xie
    email                : ylxie@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
// security.cpp: interface implementation for the security library
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "default.h"
#include "md5.h"
#include "cryptlib.h"
#include "files.h"
#include "filters.h"
#include "rng.h"
#include "hex.h"
#include "rsa.h"
#include "randpool.h"

#include <string>
#include <iostream>

USING_NAMESPACE(CryptoPP)
USING_NAMESPACE(std)

#define KEYLENGTH     512  // rsa key length 
#define SEED_SIZE_KEYGEN 16
#define MAXSTRSIZE    2048

/////////////////////////////////////////////////////////////////
//  RSA key util functions

/*
 * write_RSA_key: store a public key string to a file called keyfile
 */
int write_RSA_key(string& keystr, string& keyfile)
{
	FILE *fp = fopen(keyfile.c_str(), "w");
	if ((!fp) || (fputs(keystr.c_str(), fp) < 0)) 
		return -1;

	fclose(fp);
	return 0;
}

/*
 * read_RSA_key: read a public/private key string from a file called keyfile
 */
int read_RSA_key(string& keyfile, string& keystr)
{
	char buf[MAXSTRSIZE];
	FILE *fp = fopen(keyfile.c_str(), "r");

	if ((!fp) || (!fgets(buf, MAXSTRSIZE, fp)))
		return -1;

	int len = strlen(buf);
	if (buf[len-1] == '\n')
	    buf[len-1] = '\0';
	
	fclose(fp);
	keystr = buf;

	return 0;
}

void getSeed(byte *seed, int seedlength)
{
  int fd = open("/dev/random", O_RDONLY);
  
  if (fd < 0) {
    perror("security:llio.cc:getSeed:open");
    exit(5);
  }
  int got = 0, i;
  while (got < seedlength) {
    i = read(fd, seed + got, seedlength - got);
    if (i < 0) {
      if (errno != EINTR) {
	perror("security:llio.cc:getSeed:read");
	exit(5);
      }
    } else {
      got += i;
    }
  }
  close(fd);
}

void generate_RSA_key(const char* privFilename, const char* pubFilename)
{
	byte seed[SEED_SIZE_KEYGEN];
	getSeed(seed, SEED_SIZE_KEYGEN);
	RandomPool randPool;
	randPool.Put(seed, SEED_SIZE_KEYGEN);

	RSAES_OAEP_SHA_Decryptor priv(randPool, KEYLENGTH);
	HexEncoder privFile(new FileSink(privFilename));
	priv.DEREncode(privFile);
	privFile.MessageEnd();

	RSAES_OAEP_SHA_Encryptor pub(priv);
	HexEncoder pubFile(new FileSink(pubFilename));
	pub.DEREncode(pubFile);
	pubFile.MessageEnd();
}


/////////////////////////////////////////////////////////////////
//  RSA Encrypt/Decrypt functions

/*
 * RSAEncryptString: need to chop the cleartext into blocks, the blocks of
 *                   the ciphertext are concatenated by ":"
 */
void RSAEncryptString(string& pubkey, string& cleartext, string& ciphertext)
{
	int blocklen, finished = 0;

	/* get a random seed */
	byte seed[SEED_SIZE_KEYGEN];
	getSeed(seed, SEED_SIZE_KEYGEN);
	RandomPool randPool;
	randPool.Put(seed, SEED_SIZE_KEYGEN);

	StringSource pubString(pubkey, true, new HexDecoder);
	RSAES_OAEP_SHA_Encryptor pub(pubString);
	blocklen = pub.MaxPlainTextLength();
	char *outstr = new char[2*pub.CipherTextLength()+1];

	/* do encryption */
	int i = 0;
	while (!finished){
		unsigned int start = blocklen * i;
		unsigned int stop = start + blocklen;
		string remaintext = cleartext.substr(start, stop - start);
		if (stop > cleartext.length())
			finished = 1;
		else
			i ++;

		pub.Encrypt(randPool, (byte *)remaintext.c_str(), strlen(remaintext.c_str()), (byte *)outstr);

		HexEncoder hexEncoder;
		hexEncoder.Put((byte *)outstr, pub.CipherTextLength());
		hexEncoder.MessageEnd();
		hexEncoder.Get((byte *)outstr, 2*pub.CipherTextLength());

		outstr[2*pub.CipherTextLength()] = 0;
		if (i > 1)
			ciphertext += ":";
		ciphertext += outstr;
	}
	delete outstr;
}

void RSADecryptString(string& prikey, string& cleartext, string& ciphertext)
{
	int finished = 0;
	string::size_type pos = 0, pos1;

	StringSource privString(prikey, true, new HexDecoder);
	RSAES_OAEP_SHA_Decryptor priv(privString);
	char *outstr = new char[priv.MaxPlainTextLength()+1];

	while (!finished){
		/* get the next block */
		pos1 = ciphertext.find_first_of(':', pos);
		string remaintext = ciphertext.substr(pos, pos1-pos);
		if (pos1 > ciphertext.length())
			finished = 1;
		else
			pos = pos1 + 1;

		/* decrypt it */
		HexDecoder hexDecoder;
		hexDecoder.Put((byte *)remaintext.c_str(), strlen(remaintext.c_str()));
		hexDecoder.MessageEnd();
		SecByteBlock buf(priv.CipherTextLength());
		hexDecoder.Get(buf, priv.CipherTextLength());

		unsigned messageLength = priv.Decrypt(buf, (byte *)outstr);
		outstr[messageLength] = 0;

		/* put into cleartext */
		cleartext += outstr;
	}
	delete outstr;
}


/////////////////////////////////////////////////////////////////
//  RSA signature signing/verification functions


void RSASignString(string& prikey, string& messageText, string& signature)
{
	StringSource privString(prikey, true, new HexDecoder);
	RSASSA_PKCS1v15_SHA_Signer priv(privString);
	NullRNG rng;	// RSASSA_PKCS1v15_SHA_Signer ignores the rng. Use a real RNG for other signature schemes!
	StringSource f(messageText, true, new SignerFilter(rng, priv, new HexEncoder(new StringSink(signature))));
}

bool RSAVerifyString(string& pubkey, string& messageText, string& signatureText)
{
	StringSource pubString(pubkey, true, new HexDecoder);
	RSASSA_PKCS1v15_SHA_Verifier pub(pubString);

	StringSource signatureString(signatureText, true, new HexDecoder);
	if (signatureString.MaxRetrievable() != pub.SignatureLength())
		return false;
	SecByteBlock signature(pub.SignatureLength());
	signatureString.Get(signature, signature.size);

	VerifierFilter *verifierFilter = new VerifierFilter(pub);
	verifierFilter->PutSignature(signature);
	StringSource f(messageText, true, verifierFilter);

	byte result = 0;
	f.Get(result);
	return result == 1;
} 


/////////////////////////////////////////////////////////////////
//  MD5hash functions

void MD5hash(string passwd, string& hashpwd)
{
	char buf[MAXSTRSIZE];

	memset(buf, 0, MAXSTRSIZE);
	MD5 *m = new MD5();
	m->Update((const byte *)passwd.c_str(), passwd.size());
	m->Final((byte *)buf);

	hashpwd = buf;
}

