/***************************************************************************
                          request.h  -  description
                             -------------------
    begin                : Mon Sep 24 2001
    copyright            : (C) 2001 by Yinglian Xie
    email                : ylxie@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// request.h: interface for the CRequest, CReqQueue class.
//
//////////////////////////////////////////////////////////////////////

#ifndef REQUEST_H
#define REQUEST_H

#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <string>
#include <list>
#include <string>
#include "qprocessor.h"
#include "user.h"
#include "master.h"
#include "utils.h"
using namespace std;

// request status
#define  PROCESSING    0
#define  PROCESS_DONE  1

// request type
enum RequestType{
    NORMAL = 0, 
    CHILD, 
    PARENT, 
    MASTER,
    PUBKEY
};

// request location
#define  LOCAL  0 
#define  REMOTE 1

// data buffer
#define NUMBUF 100      // number of buffer allowed for each data packet
#define BUFLEN 1024     // each buffer length

class CSender;
class CFindex;
class CQprocessor;
class CUserGroup;
class CQresult;
class CMaster;


/*----------- class CRequest -------------- ------------------------
 * data structure for a request from a client ---------------------
 */
class CRequest {
public:
	int   m_sd;       // sock fd used for communication
	int   m_reqid;    // parent and children share the same reqid, created by local host
	uchar m_host[HOSTLEN];  // directly connected remote host
	short m_port;     // directly connected remote host port
	pid_t m_pid;      // process id if request from local machine
	Identity m_user;  // user identity, type defined in user.h

	short m_type;     // type of request: NORMAL, CHILD OR PARENT
	int   m_outref;   // > 0 if parent request
	int   m_status;   // current process status, valid for parent req
	int   m_location; // local request or remote request
	
	int   m_rbuf;     // index of current read buf
	int   m_wbuf;     // index of current write buf

	int   m_trsize;   // total size of read
	int   m_twsize;   // total size of write
	int   m_nrsize;   // size need to be read for this buffer
	int   m_nwsize;   // size need to be write for this buffer 

	char* m_readPtr;  // current ptr for reading
	char* m_writePtr; // current ptr for writing
	char* m_indata[NUMBUF]; // read buffer
	char* m_outdata[NUMBUF];// write buffer 

	string m_olddata; // some status to preserve for the request

#ifdef MINGLE_DEBUG
	struct timeval tv_start; // time receiving the complete request
	struct timeval tv_stop;  // time finish processing the request
#endif

public:
	CRequest(int id);
	~CRequest();
	
	void delete_outdata(); // delete space for write buffer
	void delete_indata(); // delete space for read buffer
};


/*-------------class CReqQueue-----------------------
 */
struct _request_list{
	CRequest  *req;
	struct _request_list* next;
	struct _request_list* prev;
};
typedef struct _request_list TReqList;

class CReqQueue{
 private:
	int       m_ismaster; // whether the server is a master server
	uchar     m_lo_ip[HOSTLEN];  // lo interface ip: 127.0.0.1

 public:
	TReqList *m_head;     // request queue head
	TReqList *m_tail;     // request queue tail
	int       m_num;      // number of requests in queue
	int       m_maxreqid; // max request id
	
	CSender  *m_sender;   // sender of the server
	CFindex  *m_index;    // index part of the server
	CQprocessor *m_qprocessor;  // query processor of the server
	CUserGroup *m_usergroup; // user management module of the server
	CMaster    *m_master;  // master server module

private:	
	/* user interface functions */
	void signon_request(CRequest *req, string& strcd);
	void passwd_request(CRequest *req, string& strcd);
	void init_request(CRequest *req, string& strcd);
	void display_request(CRequest *req);
	void addarp_request(CRequest *req, string& strcd);
	void rmarp_request(CRequest *req, string& strcd);
	void cleararp_request(CRequest *req, string& strcd);
	void lsarp_request(CRequest *req, string& strcd);
	void index_request(CRequest *req, string& strcd);
	void search_request(CRequest *req, string& strcd); 

	/* requests processed by master server */
	void join_request(CRequest *req, string& strcd);
	void register_request(CRequest *req, string& strcd);
	void chpasswd_request(CRequest *req, string& strcd);
	void host_request(CRequest *req);
	void getpubkey_request(CRequest *req, string& strcd);
	void getprikey_request(CRequest *req, string& strcd);

	/* replies from master server */
	void join_reply(CRequest *req, string& strcd);
	void register_reply(CRequest *req, string& strcd);
	void passwd_reply(CRequest *req, string& strcd);
	void host_reply(CRequest *req, string& strcd);
	void pubkey_reply(CRequest *req, string& strcd);
	void prikey_reply(CRequest *req, string& strcd);

	/* help functions to user interface functions */
	void process_master_reply(CRequest *req);
	int  parse_keyfile(string& strcd, string& option, string& keyfile,
			   string& errstr);
	int  parse_arp_request(string& strcd, string& mgid, string& localID,
			       int& mode);
	int  check_identity(CRequest *req, string& cmd, string::size_type pos);
	int  process_host_option(CRequest *req, string& cmd);
	int  parse_hosts(string& strhosts, THostList& hl, string& errstr);
	void qresult_to_string(CQuery& q, TResultList& qr, string& prostr); 
	void transform_request(CRequest *req, string& newdata);
	int  my_getuid(pid_t pid, int *uid); 
	int  valid_mgid(string& mgid);

	/* other help functions */
	void clear();  
	void delete_request(CRequest *req); 
	CRequest* find_parent(CRequest *creq);	
	void forward_request(CRequest *req, THostList& th); 
	void forward_reply(CRequest *req); 
	void request_for_master(CRequest *req, const char* rcontent); 
	uid_t getpeereuid(int sd);

public:
	CReqQueue();
	~CReqQueue();

	// configure master server true
	inline void set_master_on();  

	// configure master server false
	inline void set_master_off(); 

	// increment max request id
	inline void incre_reqid(); 
	
	// find a request based on socket descriptor
	CRequest* find(int sd); 

	// insert a new request
	void insert_request(CRequest *req); 

	// remove a request	
	void remove_request(CRequest *req); 
	
	// main function to process request
	void process_request(CRequest *req); 

	// generate reply back to client
	void generate_reply(CRequest *req, const char* outstring); 

	// send join request to the master server
	void join_cluster();
};


/***************************************************************
 *************** Inline function implementation ****************
 **************************************************************/

//////////////////////////////////////////////////////////////////////
//  class  CReqQueue inline functions
//////////////////////////////////////////////////////////////////////

inline void CReqQueue::set_master_on()
{
	m_ismaster = 1;
}

inline void CReqQueue::set_master_off()
{
	m_ismaster = 0;
}

inline void CReqQueue::incre_reqid()
{
	m_maxreqid ++;
	if (m_maxreqid > MAX_INT)
		m_maxreqid = 0;
}


#endif
