/***************************************************************************
                          request.cpp  -  description
			 -------------------
    begin                : Mon Sep 24 2001
    copyright            : (C) 2001 by Yinglian Xie
    email                : ylxie@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// request.cpp: interface for the CRequest, CReqQueue class.
//
//////////////////////////////////////////////////////////////////////

#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <list>
#include <string>
#include <algorithm>
#include "utils.h"
#include "request.h"
#include "findex.h"
#include "sender.h"
#include "qprocessor.h"
#include "user.h"
#include "master.h"
using namespace std;

//////////////////////////////////////////////////////////////////////
//  class  CRequest
//////////////////////////////////////////////////////////////////////

CRequest::CRequest(int id)
{
	m_reqid = id;
	m_type = NORMAL;     // by default, we have a normal request   
	m_pid = -1;          // we don't have a valid pid yet  
	m_location = LOCAL;  // by default, we have a local request

	m_user.uid = -1;     // by default, we don't have a valid uid

	m_trsize = 0;
	m_twsize = 0;
	m_nrsize = 0;
	m_nwsize = 0;
	m_outref = 0;

	m_rbuf = 0;
	m_wbuf = 0;

	m_readPtr = 0;
	m_writePtr = 0;

	for (int i = 0; i < NUMBUF; i ++){
	    m_indata[i] = 0;
	    m_outdata[i] = 0;
	}

}

CRequest::~CRequest()
{
	for (int i = 0; i < NUMBUF; i ++){
		if (m_indata[i])
			delete m_indata[i];
	}

	for (int i = 0; i < NUMBUF; i ++){
		if (m_outdata[i])
			delete m_outdata[i];
	}
}

void CRequest::delete_outdata()
{
	for (int i = 0; i < NUMBUF; i ++){
	  if (m_outdata[i]){
		  delete m_outdata[i];
		  m_outdata[i] = 0;
	  }	
	}

	m_writePtr = 0;
	m_wbuf = 0;
	m_twsize = m_nwsize = 0;	
}

void CRequest::delete_indata()
{
	for (int i = 0; i < NUMBUF; i ++){
	  if (m_indata[i]){
		  delete m_indata[i];
		  m_indata[i] = 0;
	  }	
	}

	m_readPtr = 0;
	m_rbuf = 0;
	m_trsize = m_nrsize = 0;
}

//////////////////////////////////////////////////////////////////////
//  class  CReqQueue
//////////////////////////////////////////////////////////////////////

CReqQueue::CReqQueue()
{
	m_maxreqid = 0;

	m_head = m_tail = 0;
	m_num = 0;
	
	// get lo_ip 
	string hoststr = "127.0.0.1";
	if (convert_host_to_ip(hoststr, m_lo_ip) == -1){
		mingle_debug1("\nCReqQueue::CReqQueue(): set lo ip error");
	}
}

CReqQueue::~CReqQueue()
{
	clear();
}

////////////////////////////////////
// public functions
//////////////////////////////////

/////////////////////////////////////////////////////////////////
//  insert/delete/retrieve request record in the request  queue

/* 
 *find: find the request record based on socket fd 
 */
CRequest* CReqQueue::find(int sd)
{
	TReqList *rl = m_head;
	while ((rl) && (rl->req) && (rl->req->m_sd != sd))
		rl = rl->next;

	if (rl){	
		assert(rl->req);
		return rl->req;
	}	
	else
		return 0;			
}

/*
 * insert_request: insert a new request into the request queue
 */
void CReqQueue::insert_request(CRequest *req)
{
	assert(req != 0);

	TReqList *rl = new TReqList;
	rl->prev = rl->next = 0;
	rl->req = req;

	if (m_tail){
		m_tail->next = rl;
		rl->prev = m_tail;
		m_tail = rl;
	}else{
		m_head = m_tail = rl;
	}

	m_num ++;
}

/* 
 * remove_request: remote a request from the request queue
 *                 difference between delete_request:
 *                    before delete, check dependency, 
 * 	              remove parent request if no dependency after deletion
 */	
void CReqQueue::remove_request(CRequest *req)
{
	if ((req->m_type == CHILD) || (req->m_type == MASTER)){
		CRequest *preq = find_parent(req);
		preq->m_outref --;
		if ((preq->m_outref == 0) && (preq->m_status == PROCESS_DONE)){
			close(preq->m_sd);
			delete_request(preq);
		}	
		for (int i = 0; i < NUMBUF; i ++){
			req->m_outdata[i] = 0;
		}
	}

	delete_request(req);
}

/* 
 * process_request: main function to parse command line and generate reply
 *                  local request:  mingle cmd-option arg
 *                  master request: mingle cmd-option arg | K(arg)
 *                  master reply:   mingle cmd-option arg | K-1(arg)
 *                  remote request: mingle search arg [-i mgid timestamp] [-d signature]
 *                                  require: exact one blank space after arg
 *                                  i.e. signature = sign(time cmd-option arg )
 */
void CReqQueue::process_request(CRequest *req)
{
	string  option, strparse, cmd;
	string::size_type pos = 0;

	// first check to see if the m_indata is a reply instead of a request
	if (req->m_type == CHILD){
		forward_reply(req);
		return;
	}else if (req->m_type == MASTER){
		process_master_reply(req);
		return;
	}

	// now we are sure it's a normal request, parse the command line
	// first merge several buffers to a single buffer into strcd	
	for (int i = 0; i <= req->m_rbuf; i ++){
		cmd += req->m_indata[i];
	}

	// find command line into "cmd"
	pos = cmd.find_first_not_of(" ");
	if (pos > cmd.length()){
		generate_reply(req, "Error: No command option!\n");
		return;
	}

	// user authentication 
	if (check_identity(req, cmd, pos) < 0)
		return;

	// decide whether to process the request or forward the request
	if (process_host_option(req, cmd) < 0)
		return;

	// now start parsing "cmd", we omit options we don't recognize
	pos = 0;
	pos = get_next_item(cmd, option, pos);
	if (option.empty()){ 
		generate_reply(req, "Error: No command option.\n"); 
		return; 
	} 

	// extract command arguments into "strparse"  
	if ((option != "host") && (option != "join") && (option != "display"))
		strparse = cmd.substr(pos, MAX_INT);  

	// local process: process request individually based on request type
	if (option == "signon"){
		signon_request(req, strparse);
	}else if (option == "init"){
		init_request(req, strparse);
	}else if (option == "passwd"){
		passwd_request(req, strparse);
	}else if (option == "display"){
		display_request(req);
	}else if (option == "index"){
		index_request(req, strparse);
	}else if (option == "search"){
		search_request(req, strparse);
	}else if (option == "addarp"){
		addarp_request(req, strparse);
	}else if (option == "rmarp"){
		rmarp_request(req, strparse);
	}else if (option == "cleararp"){
		cleararp_request(req, strparse);
	}else if (option == "lsarp"){
		lsarp_request(req, strparse);
	}else if (option == "register"){
		register_request(req, strparse);
	}else if (option == "chpasswd"){
		chpasswd_request(req, strparse);
	}else if (option == "join"){
		join_request(req, strparse);
	}else if (option == "host"){
		host_request(req);
	}else if (option == "getprikey"){
		getprikey_request(req, strparse);
	}else if (option == "getpubkey"){
		getpubkey_request(req, strparse);
	}else{ 
		// not recognized option
		req->m_type = NORMAL;
		generate_reply(req, "Error: Unrecognized command.\n");
	} 
}


/* 
 * generate_reply - generate outdata for the request 
 *                  by appending outstring to the outdata buffer
 */
void CReqQueue::generate_reply(CRequest *req, const char* outstring)
{
	char  hostname[BUFLEN];
	string outdata;
	char *destptr, *srcptr;

	// src ptr
	if (req->m_location == REMOTE){
		sprintf(hostname, "\nHost: %i.%i.%i.%i:\n",
			local_ip[0], local_ip[1], local_ip[2], local_ip[3]);
		outdata = hostname;
	}
	outdata += outstring;

	srcptr = (char*)outdata.c_str();

	// dst ptr
	int i = req->m_wbuf;
	int rlen = req->m_twsize % BUFLEN;
	if (rlen == 0){
		if (req->m_twsize > 0)
			rlen = BUFLEN - 1;
		else{
			assert(req->m_outdata[0] == 0);
			req->m_outdata[0] = new char[BUFLEN];
		}
	}else{
		rlen --;
		req->m_twsize --;
	}
	destptr = req->m_outdata[i] + rlen;

	// copy len
	int len = outdata.length() + 1;
	req->m_twsize += len;
	int buflen = (rlen > 0) ? (BUFLEN - rlen) : BUFLEN;
	int cplen = (len > buflen) ? buflen : len;

	/* copy first line of data */
	memcpy(destptr, srcptr, cplen);
	len -= cplen;

	/* copy other stuff, add padding to the last buffer */
	while (len > 0){
		i ++;
		assert(req->m_outdata[i] == 0);
		req->m_outdata[i] = new char[BUFLEN];
		destptr = req->m_outdata[i];
		srcptr += cplen;
		cplen = (len > BUFLEN) ? BUFLEN : len;
		memcpy(destptr, srcptr, cplen);
		if (cplen < BUFLEN)
			memset(destptr+cplen, 0, BUFLEN - cplen);
		len -= cplen;
	}

	// wrap up
	outdata.erase();
	req->m_wbuf = i;
	req->m_status = PROCESS_DONE;

	m_sender->send_reply(req);
}

/////////////////////////////////////////////////////////////////
//  request for master server's public key and join the cluster
//  only be executed when we are not master server

void CReqQueue::join_cluster()
{
	if (m_ismaster)
		return;
	
	// make a dummy request as the parent request
	CRequest *req = new CRequest(m_maxreqid);
	incre_reqid();
	req->m_type = NORMAL;
	req->m_location = REMOTE;
	insert_request(req);

	// send the request to the master server
	request_for_master(req, "join");
}

////////////////////////////////////
// private functions
//////////////////////////////////

/////////////////////////////////////////////////////////////////
//  user interface functions to process different types of requests

/* 
 * signon_request: mingle user sign on at master server for mingle id uniqueness
 *                 need to be followed by init request on mingle hosts where we
 *                 want to initiate a request
 *                 format: signon mgid password
 */
void CReqQueue::signon_request(CRequest *req, string& strcd)
{
	string::size_type pos = 0;
	string  password;

	/***** check request origin *******/
	// the request must come from local
	if ((req->m_location == REMOTE) || (req->m_user.uid < 0)){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}

	/***** start parsing *******/
	// the user must not have a mingle id already 
	if (m_usergroup->owner_exist(req->m_user.uid)){
		generate_reply(req, "Error: You have already signed on.\n");
		return;
	}

	// extract mgid and check mingle id validity
	pos = get_next_item(strcd, req->m_user.mgid, pos);
	if (req->m_user.mgid.empty()){
		generate_reply(req, 
			       "Error: No mingle id specified.\n");
		return;
	}
	if (!valid_mgid(req->m_user.mgid)){
		generate_reply(req,
			       "Error: Invalid mingle id.\n");
		return; 
	}

	// extract password
	pos = get_next_item(strcd, password, pos);

	/***** start doing real processing *******/
	if (!m_ismaster){
		// if we are not the master server, do the folloing
		// STEP 1: encrypt mgid and password
		string message = req->m_user.mgid + " " + password;
		string output = "register " + 
			m_usergroup->encrypt_string(m_master->m_pubkey, message);

		// STEP 2: send the request to the master server
		request_for_master(req, output.c_str());
		return;
	}else{
		// we are master, register the new user
		string strerr;
		if (m_master->register_mid(req->m_user.mgid, password, strerr) < 0){
			generate_reply(req, strerr.c_str());
			return;
		}
	}
	generate_reply(req, "Success: Sign-on successfully.\n");
	return;
}

/* 
 * init_request: init mingle owner
 *               check identity and get private key from the master server
 *               format: init mgid password
 */
void CReqQueue::init_request(CRequest *req, string& strcd)
{
	string::size_type pos = 0;
	string  passwd_mingle;

	// the request must come from local
	if ((req->m_location == REMOTE) || (req->m_user.uid < 0)){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}

	// extract mgid, afsid and passwords
	pos = get_next_item(strcd, req->m_user.mgid, pos);
	if (req->m_user.mgid.empty()){
		generate_reply(req, 
			       "Error: No mingle id specified.\n");
		return;
	}
	pos = get_next_item(strcd, passwd_mingle, pos);

	// check Mingle identity and get private key from master server
	if (!m_ismaster){
		// STEP 1: encrypt mgid and password, attach our public key
		string message = req->m_user.mgid + " " + passwd_mingle;
		string output = "getprikey " + m_usergroup->m_pubkey + " " +
			m_usergroup->encrypt_string(m_master->m_pubkey, message);
		
		// STEP 2: send the request to the master server
		request_for_master(req, output.c_str());
		return;
	}else{
		// we are master, do init directly
		string prikey;
		if (m_master->get_prikey(req->m_user.mgid, passwd_mingle, prikey) < 0){
			generate_reply(req, "Error: Permission denied.\n");
		}else{
			m_usergroup->mingle_init(req->m_user, prikey);
			generate_reply(req, "Success: Mingle init successfully.\n");
		}
	}

	return;
}

/* 
 * passwd_request: change password at master server 
 *                 format: passwd mgid old-password new-password
 */
void CReqQueue::passwd_request(CRequest *req, string& strcd)
{
	string::size_type pos = 0;
	string  oldpwd, newpwd;

	// extract fields: mgid, oldpwd, newpwd
	pos = get_next_item(strcd, req->m_user.mgid, pos);
	if (req->m_user.mgid.empty()){
		generate_reply(req, 
			       "Error: No mingle id specified.\n");
		return;
	}
	pos = get_next_item(strcd, oldpwd, pos);
	if (!oldpwd.empty())
		pos = get_next_item(strcd, newpwd, pos);

	/***** start doing real processing *******/
	if (!m_ismaster){
		// if we are not the master server, do the folloing
		// STEP 1: encrypt mgid and passwords
		string message = req->m_user.mgid + " " + oldpwd + " " + newpwd;
		string output = "passwd " + 
			m_usergroup->encrypt_string(m_master->m_pubkey, message);

		// STEP 2: send the request to the master server
		request_for_master(req, output.c_str());
		return;
	}else{
		// we are master, register the new user
		if (m_master->change_passwd(req->m_user.mgid, oldpwd, newpwd) < 0){
			generate_reply(req, "Error: Permission denied.\n");
			return;
		}
	}
	generate_reply(req, "Success: Password changed successfully.\n");
	return;
}

/* 
 * display_request: display mingle id
 *                  format: display
 */
void CReqQueue::display_request(CRequest *req)
{
	// the request must come from local
	if ((req->m_location == REMOTE) || (req->m_user.uid < 0)){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}
	
	string output;
	m_usergroup->display_userinfo(req->m_user.uid, output);
	generate_reply(req, output.c_str());
}


/*
 * addarp_request - add an access-right mapping 
 *                  format: addarp [-u/-g/-a] mgid localID/afsID
 */
void CReqQueue::addarp_request(CRequest *req, string& strcd)
{
	string mgid, mappedID, errstr;
	int mode = 0;
	
	if (parse_arp_request(strcd, mgid, mappedID, mode) < 0){
		generate_reply(req, "Error: Parse error.\n");
		return;
	}
	
	// add new arp item into arp list
	if (m_usergroup->add_arp(req->m_user,
				 mgid, mappedID, mode, errstr) < 0){
		generate_reply(req, errstr.c_str());
	}else{
		generate_reply(req, "Success: Access right mapping added sucessfully.\n");
	}
	
	return;
}

/*
 * rmarp_request: remove an access-right mapping
 *                format: rmarp [-u/-g/-a] mgid localID/afsID
 */
void CReqQueue::rmarp_request(CRequest *req, string& strcd)
{
	string mgid, mappedID, errstr;
	int mode = 0;
	
	if (parse_arp_request(strcd, mgid, mappedID, mode) < 0){
		generate_reply(req, "Error: Parse error.\n");
		return;
	}
	
	// remote arp from arp list
	if (m_usergroup->rm_arp(req->m_user,
				mgid, mappedID, mode, errstr) < 0){
		generate_reply(req, errstr.c_str());
	}else{
		generate_reply(req, "Success: Access right mapping removed sucessfully.\n");
	}
	
	return;
}

/*
 * cleararp_request: clear all access-right mapping for a user
 *                   format: cleararp mgid
 */
void CReqQueue::cleararp_request(CRequest *req, string& strcd)
{
	string mgid, errstr;
	string::size_type pos = 0;
	
	pos = get_next_item(strcd, mgid, pos);
	if (mgid.empty()){
		generate_reply(req, "Error: Parse error.\n");
		return;
	}
	
	// clear SPD mapped from mgid
	if (m_usergroup->clear_SPD(req->m_user, mgid, errstr) < 0){
		generate_reply(req, errstr.c_str());
	}else{
		generate_reply(req, "Success: Access right mapping cleared.\n");
	}
	
	return;
	
}

/*
 * lsarp_request: list access-right mappings
 *                format: lsarp [mgid]
 */
void CReqQueue::lsarp_request(CRequest *req, string& strcd)
{
	string mgid, errstr;
	string::size_type pos = 0;

	pos = get_next_item(strcd, mgid, pos);
	
	// list access-right mapping for the owner
	m_usergroup->ls_arp(req->m_user, mgid, errstr);
	generate_reply(req, errstr.c_str());
	
	return;
}

/*  
 * index_request: index a file or a directory
 *                format: index dir1 dir2 dir3 ...
 */
void CReqQueue::index_request(CRequest *req, string& strcd)
{
	string::size_type pos = 0, pos1;
	string  prostr, strarg, dir;
	
	int num = 0;
	while (pos < strcd.length()){
		pos1 = strcd.find_first_not_of(" ", pos);
		if ((strcd[pos1] == '-') || (pos1 >= strcd.length())){
			if (num == 0){
				prostr = "Error: Parse error.\n";
				break;
			}else
				break;
		}	
		pos = strcd.find_first_of(" \n", pos1);
		assert(pos < strcd.length());
		if (pos == pos1) break;
		strarg = strcd.substr(pos1, pos - pos1);
		prostr += "Directory: \"";
		
		/* turn dir into absolute path name */
		if (get_absolute_fname(strarg, dir) == -1){
			prostr += strarg;
			prostr += "\" - index error: system error\n";
			strarg.erase();
			dir.erase();
			num ++;
			continue;
		}

		/* check permission of index */
		if (!m_usergroup->index_file_permitted(req->m_user, dir)){
			prostr += dir;
			prostr += "\" - index permission denied\n";
		}else{
			/* start indexing now */
			if (m_index->start_index(dir.c_str(), req->m_user) < 0){
				prostr += dir;
				prostr += "\" - index error: ";
			}else{
				prostr += dir;
				prostr += "\" - index success\n";
			}
		}
		strarg.erase();
		dir.erase();
		num ++;
	}
	
	assert(!prostr.empty());
	generate_reply(req, prostr.c_str());
	return;
}


/* 
 * search_request: search keywords from the index table
 *                 format: search query [-ts][-a AND|OR][-n num][-r host]
 */
void CReqQueue::search_request(CRequest *req, string& strcd)
{
	string::size_type pos = 0, pos1;
	string prostr, strarg, stropt;
	CQuery q;
	char optc;
	
	// find search string
	pos1 = strcd.find_first_not_of(" ", pos);
	if (pos1 >= strcd.length()){
		generate_reply(req, "Command parsing error.\n");
		return;
	}	
	pos = strcd.find(" -", pos1);
	if (pos == string::npos){
	    pos = strcd.find('\n', pos1);	
	    strarg = strcd.substr(pos1, pos - pos1);
	}else{
	    strarg = strcd.substr(pos1, pos - pos1);
	    pos ++;
	}
	q.m_qStr = strarg;
	
	// find other options
	while (pos < strcd.length()-1){
		optc = strcd[++pos];
		pos ++;
		switch (optc){
		case 'n':
			pos1 = strcd.find_first_of(DIGITS, pos);
			if (pos1 > strcd.length())
				break;
			pos = strcd.find_first_not_of(DIGITS, pos1);
			stropt = strcd.substr(pos1, pos - pos1);
			pos1 = pos;
			q.m_qNumResult = atoi(stropt.c_str());
			break;				
			
		case 't':
			q.m_qPosition = TITLE;
			pos1 = pos;
			break;

		case 's':
			q.m_qDescription = SHOW_ALL;
			pos1 = pos;
			break;
		    
		case 'a':
			pos1 = strcd.find_first_not_of(" ", pos);
			if (pos1 > strcd.length())
				break;
			pos = strcd.find_first_of("\n ", pos1);
			stropt = strcd.substr(pos1, pos-pos1);
			pos1 = pos;
			if (stropt == "AND")
				q.m_qSearchOption = AND;
			if (stropt == "PHRASE")
				q.m_qSearchOption = PHRASE;
			if (stropt == "OR")
				q.m_qSearchOption = OR;
			break;							
			
		default: // unknown option
			pos1 = pos;
			break;
		} // end swtich
		pos = strcd.find_first_of("-", pos1);
	} // end while
	
	TResultList qr;

	m_qprocessor->lookup_local(q, qr, req->m_user);

	prostr += "String searched: \"";
	prostr += q.m_qStr;
	prostr += "\"  ";
	qresult_to_string(q, qr, prostr);
	
	req->m_status = PROCESS_DONE;
	generate_reply(req, prostr.c_str());
	
	return;
}

/////////////////////////////////////////////////////////////////
//  functions processed only be master server, not normal client requests

/*
 * join_request: mingle host join, received only by master server
 *               format: join
 */
void CReqQueue::join_request(CRequest *req, string& strcd)
{
	// the request must come from a remote server
	if (req->m_location != REMOTE){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}

	// we must be a master server to process this
	if (!m_ismaster){
		generate_reply(req, "--joinreply Error: Master server configuration error.\n");
		return;
	}

	string output = "--joinreply Success: " + m_usergroup->m_pubkey;
	generate_reply(req, output.c_str());
}


/* 
 * host_request: request for host list in the mingle cluster
 *               format: host
 */
void CReqQueue::host_request(CRequest *req)
{
	string prostr;
	
	// we have to be a master machine to be able to answer this
	if (!m_ismaster){
		generate_reply(req, "--hostreply Error: Master server configuration error.\n");
		return;
	}
	
	char buf[MAXSTRSIZE];
	THostList::iterator hiter;
	
	prostr = "--hostreply Success: ";
	for (hiter = m_master->m_chosts.begin(); hiter != m_master->m_chosts.end(); hiter ++){
		if (memcmp(hiter->ip, m_lo_ip, 4) == 0)
			sprintf(buf, "%i.%i.%i.%i %d\n", 
				local_ip[0], local_ip[1], local_ip[2], local_ip[3], hiter->port);
		else
			sprintf(buf, "%i.%i.%i.%i %d\n", 
				hiter->ip[0], hiter->ip[1], hiter->ip[2], hiter->ip[3], hiter->port);
		prostr += buf;
	}
	generate_reply(req, prostr.c_str());
	return;
}


/* 
 * register_request: mingle user register, received only by master server
 *                   format: register K-1(mgid password)
 */
void CReqQueue::register_request(CRequest *req, string& strcd)
{
	string ciphertext, cleartext;
	string mgid, passwd;
	string::size_type pos = 0;

	// the request must come from a remote server
	if (req->m_location != REMOTE){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}

	// we must be a master server to process this
	if (!m_ismaster){
		generate_reply(req, "--registerreply Error: Master server configuration error.\n");
		return;
	}

	// extract mgid and password
	pos = get_next_item(strcd, ciphertext, pos);
	if (ciphertext.empty()){
		generate_reply(req, 
			       "--registerreply Error: Empty message.\n");
		return;
	}
	cleartext = m_usergroup->decrypt_string(ciphertext);
	pos = get_next_item(cleartext, mgid, 0);
	if (mgid.empty()){
		generate_reply(req,
			       "--registerreply Error: Empty mingle id.\n");
		return;
	}
	pos = get_next_item(cleartext, passwd, pos);
	if (mgid.empty()){
		generate_reply(req,
			       "--registerreply Error: Password cannot be empty.\n");
		return;
	}

	// do user registration and reply
	string strerr;
	if (m_master->register_mid(mgid, passwd, strerr) < 0){
		string output = "--registerreply " + strerr;
		generate_reply(req, output.c_str());
	}else{
		generate_reply(req, "--registerreply Success: \n");
	}
	return;
}

/* 
 * chpasswd_request: change mingle password, received only by master server
 *                   format: chpasswd K-1(mgid oldpwd newpwd)
 */
void CReqQueue::chpasswd_request(CRequest *req, string& strcd)
{
	string ciphertext, cleartext;
	string mgid, oldpwd, newpwd;
	string::size_type pos = 0;

	// the request must come from a remote server
	if (req->m_location != REMOTE){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}

	// we must be a master server to process this
	if (!m_ismaster){
		generate_reply(req, "--passwdreply Error: Master server configuration error.\n");
		return;
	}

	// extract mgid and passwords
	pos = get_next_item(strcd, ciphertext, pos);
	if (ciphertext.empty()){
		generate_reply(req, 
			       "--passwdreply Error: Empty message.\n");
		return;
	}
	cleartext = m_usergroup->decrypt_string(ciphertext);
	pos = get_next_item(cleartext, mgid, 0);
	if (mgid.empty()){
		generate_reply(req,
			       "--passwdreply Error: Empty mingle id.\n");
		return;
	}
	pos = get_next_item(cleartext, oldpwd, pos);
	if (!oldpwd.empty())
	    pos = get_next_item(cleartext, newpwd, pos);

	// change user password
	if (m_master->change_passwd(mgid, oldpwd, newpwd) < 0)
		generate_reply(req, "--passwdreply Error: Permission denied.\n");
	else
		generate_reply(req, "--passwdreply Success: \n");
	return;
}


/*
 * getprikey: request for private key of a mingle user from the master server
 *            format: getprikey pubkey K-1(mgid password)
 */
void CReqQueue::getprikey_request(CRequest *req, string& strcd)
{
	string ciphertext, cleartext;
	string mgid, passwd, prikey, pubkey;
	string::size_type pos = 0;

	// the request must come from a remote server
	if (req->m_location != REMOTE){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}

	// we must be a master server to process this
	if (!m_ismaster){
		generate_reply(req, "--prikeyreply Error: Master server configuration error.\n");
		return;
	}

	// extract client server's public key
	pos = get_next_item(strcd, pubkey, pos);
	if (pubkey.empty()){
		generate_reply(req, 
			       "--prikeyreply Error: Empty message.\n");
		return;
	}

	// extract mgid and password
	pos = get_next_item(strcd, ciphertext, pos);
	if (ciphertext.empty()){
		generate_reply(req, 
			       "--prikeyreply Error: Empty message.\n");
		return;
	}
	cleartext = m_usergroup->decrypt_string(ciphertext);
	pos = get_next_item(cleartext, mgid, 0);
	pos = get_next_item(cleartext, passwd, pos);

	// check user identity and return key
	if (m_master->get_prikey(mgid, passwd, prikey) < 0){
		generate_reply(req, "--prikeyreply Error: Permission denied.\n");
	}else{
		/* encrypt the private key */
		string output = "--prikeyreply Success: " + 
			m_usergroup->encrypt_string(pubkey, prikey);
		generate_reply(req, output.c_str());
	}
	
	return;
}

/*
 * getpubkey: request for public key of a mingle user from the master server
 *            format: getpubkey mgid
 */
void CReqQueue::getpubkey_request(CRequest *req, string& strcd)
{
	string mgid, pubkey;
	string::size_type pos = 0;

	// the request must come from a remote server
	if (req->m_location != REMOTE){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}

	// we must be a master server to process this
	if (!m_ismaster){
		generate_reply(req, "--pubkeyreply Error: Master server configuration error.\n");
		return;
	}

	// extract mgid
	pos = get_next_item(strcd, mgid, pos);
	if (mgid.empty()){
		generate_reply(req, 
			       "--pubkeyreply Error: Empty message.\n");
		return;
	}

	// check user identity and return key
	if (m_master->get_pubkey(mgid, pubkey) < 0){
		generate_reply(req, "--pubkeyreply Error: The specified Mingle id does not exist.\n");
	}else{
		string output = "--pubkeyreply Success: " + pubkey;
		generate_reply(req, output.c_str());
	}
	
	return;
}

/////////////////////////////////////////////////////////////////
// replies from master server, not normal client requests

/*
 * join_reply: reply message from the master server about join request
 *             format: --joinreply pubkey
 */


void CReqQueue::join_reply(CRequest *req, string& strcd)
{
	string pubkey, result;
	string::size_type pos = 0;

	// extract result description "Error:" or "Success:"
	pos = get_next_item(strcd, result, pos);
	if (result != "Success:"){
		mingle_debug1("\nCReqQueue::join_reply(): ");
		mingle_debug1(strcd.c_str());
	}else{
		// extract public key
		pos = get_next_item(strcd, pubkey, pos);
		if (!pubkey.empty())
			// update user record and check user identity again
			m_master->setup_pubkey(pubkey);
		else
			mingle_debug1("\nCReqQueue::join_reply(): master server public key empty.");
	}

	// delete this request
	remove_request(req);
}

/*
 * register_reply: reply from the master server about mingle id sign-on
 *                 format: --registerreply [Error:/Success:  info]
 */
void CReqQueue::register_reply(CRequest *req, string& strcd)
{
	string::size_type pos = 0;
	string result;

	// extract result description "Error:" or "Sucess:"
	pos = get_next_item(strcd, result, pos);
	if ((result != "Error:") && (result != "Success:")){
		generate_reply(req, 
			       "Error: No valid answer from master server.\n");
		return;
	}

	/* if master does not approve sign-on, 
	   forward the reply from master server */
	if (result == "Error:")
		generate_reply(req, strcd.c_str());
	else
		// signon is approved
		generate_reply(req, "Success: Sign-on successfully.\n");
	return;
}

/*
 * passwd_reply: reply from the master server about changing password
 *               format: --passwdreply [Error:/Success:  info]
 */
void CReqQueue::passwd_reply(CRequest *req, string& strcd)
{
	string::size_type pos = 0;
	string result;

	// extract result description "Error:" or "Sucess:"
	pos = get_next_item(strcd, result, pos);
	if ((result != "Error:") && (result != "Success:")){
		generate_reply(req, 
			       "Error: No valid answer from master server.\n");
		return;
	}

	/* if master does not approve it
	   forward the reply from master server */
	if (result == "Error:")
		generate_reply(req, strcd.c_str());
	else
		// password changed successfully
		generate_reply(req, "Success: Password changed successfully.\n");
	return;
}


/*
 * prikey_reply: private key from the Master server
 *               format: --prikeyreply K(prikey)
 */
void CReqQueue::prikey_reply(CRequest *req, string& strcd)
{
	string ciphertext, prikey, result;
	string::size_type pos = 0;

	// extract result description "Error:" or "Success:"
	pos = get_next_item(strcd, result, pos);
	if ((result != "Error:") && (result != "Success:")){
		generate_reply(req, "Error: Invalid answer from master server.\n");
		return;
	}

	if (result == "Error:"){
		generate_reply(req, strcd.c_str());
		return;
	}

	// extract signed private key 
	pos = get_next_item(strcd, ciphertext, pos);
	if (ciphertext.empty()){
		generate_reply(req, 
			       "Error: Empty message from master server.\n");
		return;
	}

	//decrypt and get the private key
	prikey = m_usergroup->decrypt_string(ciphertext);
	m_usergroup->mingle_init(req->m_user, prikey);
	generate_reply(req, "Success: Mingle init successfully.\n");
	return;
}

/*
 * pubkey_reply: public key from the Master server
 *               format: --pubkeyreply pubkey
 */
void CReqQueue::pubkey_reply(CRequest *req, string& strcd)
{
	string pubkey, result;
	string::size_type pos = 0;

	// extract result description "Error:" or "Success:"
	// for all error cases, we cannot determine the indentity of
	// the request originator
	pos = get_next_item(strcd, result, pos);
	if (result == "Success:"){
		// extract public key
		pos = get_next_item(strcd, pubkey, pos);
		if (!pubkey.empty())
			// update user record and check user identity again
			m_usergroup->update_pubkey(req->m_user.mgid, pubkey);
	}

	req->m_user.mgid.erase();
	process_request(req);
}

/*
 * host_reply: reply from the master server about cluster host list
 *             format: --hostreply [Success: ip1 port1\n ...][Error: info]
 */
void CReqQueue::host_reply(CRequest *req, string& strcd)
{
	string::size_type pos = 0, pos1;
	string result;
	THostList hl;

	// get result description "Error:" or "Success:"
	pos = get_next_item(strcd, result, pos);
	if (result != "Success:"){
		generate_reply(req, "Error: Master server configuration error.\n");
		return;
	}

	while (pos < strcd.length()){
		pos1 = strcd.find_first_of("0123456789", pos);
		if (pos1 > strcd.length())
			break;
		
		pos = strcd.find_first_of("\n", pos1);
		string hoststr = strcd.substr(pos1, pos-pos1);
		Thost nh;
		string::size_type p = hoststr.find_first_of(" ");
		if (p > hoststr.length())
			continue;
		
		// getip address
		string ip = hoststr.substr(0, p);
		inet_pton(AF_INET, ip.c_str(), nh.ip);
		
		// get port number
		p = hoststr.find_first_of("0123456789", p);
		if ((nh.port = atoi(hoststr.substr(p).c_str())) <= 0)
			continue;
		
		// skip if local server
		if (((memcmp(nh.ip, local_ip, 4) == 0) || 
		     ( memcmp(nh.ip, m_lo_ip, 4) == 0)) &&
		     (nh.port == local_port) )
			continue;
		
		// push the new host into hostlist;
		hl.push_back(nh);
		req->m_outref ++;
	}
	
	
	if (hl.size() > 0){
		forward_request(req, hl);
	}else{
		if (req->m_outref == 0){
			close(req->m_sd);
			delete_request(req);
		}
	}
}

/////////////////////////////////////////////////////////////////
//  help functions to request processing

/*
 * process_master_reply: request is a reply from a master server
 *                       types of request: registerreply
 *                                         hostreply
 *                                         prikeyreply
 *                                         pubkeyreply
 */
void CReqQueue::process_master_reply(CRequest *req)
{
	string strcd, option, strparse;
	string::size_type pos;

	// the request must come from a remote server
	if (req->m_location != REMOTE){
		generate_reply(req, "Error: Permission denied.\n");
		return;
	}

	// find the corresponding parent request
	assert(req->m_type == MASTER);
	CRequest *preq = find_parent(req);
	assert(preq);

	// extract reply message
	for (int i = 0; i <= req->m_rbuf; i ++){
		strcd += req->m_indata[i];
	}

	// remove the MASTER reply
	preq->m_status = PROCESSING;	 	
	remove_request(req);
	preq->m_status = PROCESS_DONE;

	// find first "--" sign
	pos = strcd.find("--");
	if (pos < strcd.length()){

		// now start parsing, we omit options we don't recognize
		pos += 2;
		pos = get_next_item(strcd, option, pos);
		strparse = strcd.substr(pos, MAX_INT);
		
		if (option == "registerreply"){
			register_reply(preq, strparse);
			return;
		}else if (option == "passwdreply"){
			passwd_reply(preq, strparse);
			return;
		}else if (option == "joinreply"){
			join_reply(preq, strparse);
			return;
		}else if (option == "hostreply"){
			host_reply(preq, strparse);
			return;
		}else if (option == "prikeyreply"){
			prikey_reply(preq, strparse);
			return;
		}else if (option == "pubkeyreply"){
			pubkey_reply(preq, strparse);
			return;
		}
	}

	// parse error
	generate_reply(preq, "Error: Master server error.\n");
	return;
}


/*
 * parse_arp_request - help function to addarp and rmarp requests 
 *                     after parsing, extract: mgid, mappedID, mode
 */
int  CReqQueue::parse_arp_request(string& strcd, string& mgid, string& mappedID,
				  int& mode)
{
	string::size_type pos = 0;
	string  option;
	
	// find out option from [-u/-g]
	pos = get_next_item(strcd, option, pos);
	if (((int)pos == -1) || (pos > strcd.length()))
		return -1;

	if (option == "-u") 
		mode = USER;
	else if (option == "-g")
		mode = GROUP;
	else
		return -1;

	// find mingle user id
	pos = get_next_item(strcd, mgid, pos);
	if (((int)pos == -1) || (pos > strcd.length()))
		return -1;
	
	// extract local id
	pos = get_next_item(strcd, mappedID, pos);
	if ((int)pos == -1)
		return -1;

	return 0;
}

/*
 * check_identity: user authentication, put user identity into request record
 *                 return value:  0: success
 *                               -1: error in parsing user identity
 */
int  CReqQueue::check_identity(CRequest *req, string& cmd, string::size_type pos)
{
	string strcd = cmd;
	string::size_type pos1;

	pos1 = strcd.find("-i");
	cmd.erase();
	cmd = strcd.substr(pos, pos1-pos); 
	
	// local request, find out from uid
	if (req->m_location == LOCAL){
		// if a local request, find out from process id
		if (my_getuid(req->m_pid, &(req->m_user.uid)) < 0){
			generate_reply(req, "Get user id from process id error\n");
			return -1;
		}else{
			m_usergroup->retrieve_identity(req->m_user);
		}
		return 0;
	}


	if (pos1 > strcd.length())
		return 0;

	// if a remote request, find out from digital signature 
	// extract mingle user id
	string::size_type pos_s = pos1+2, pos_e;
	string tmpmgid, timestamp;
	pos_e = get_next_item(strcd, tmpmgid, pos_s);
	if ((pos_e < 0) || (pos_e >= (strcd.length()-1))){
		generate_reply(req, "Error: Mingle id or timestamp parse error.\n");
		return -1;
	}
	
	// extract and check timestamp
	pos_e = get_next_item(strcd, timestamp, pos_e);
	if (pos_e < 0){
		generate_reply(req, "Error: No timestamp specified.\n");
		return -1;
	}
	// !!!!! not actually checking time yet
	
	// extract signature
	string signature;
	pos1 = strcd.find("-d");
	if (pos1 < strcd.length()){
		get_next_item(strcd, signature, pos1+2);

		// check identity info is true
		string input = timestamp + cmd;
	tryagain:
		int ret = m_usergroup->identity_confirmed(tmpmgid, input, signature);
		if (ret == 1){
			// identity found
			req->m_user.mgid = tmpmgid;
			m_usergroup->retrieve_identity(req->m_user);
		}else if (ret < 0){
			// fetch public key first
			// put tmpmgid in the m_user record now
			if (m_ismaster){
				string pubkey;
				if (m_master->get_pubkey(tmpmgid, pubkey) < 0)
					return 0;
				else{
					m_usergroup->update_pubkey(tmpmgid, pubkey);
					goto tryagain;
					
				}
			}else{
				req->m_user.mgid = tmpmgid;
				string output = "getpubkey " + tmpmgid;
				request_for_master(req, output.c_str());
				return -1;
			}
		}
	}
		
	/* if no signature is attached, we cannot determine identity
	   we do nothing in such case */
	return 0;
}

/*
 * process_host_option: process -r option in the request line
 *                      return value: 0 -- need local processing
 *                                   -1 -- do not need local processing
 */
int CReqQueue::process_host_option(CRequest *req, string& cmd)
{
	string::size_type pos1;
	string newcmd;
	int   keysigned = 1;

	pos1 = cmd.find("-r", 0);
	if (pos1 < cmd.length()){
		THostList hl;
		string strout;
		
		string strhosts = cmd.substr(pos1+2, MAX_INT);
		int ret = parse_hosts(strhosts, hl, strout);

		if (ret < 0){
			// fail to extract hosts 
			generate_reply(req, strout.c_str());
			return -1;
		}else{
			newcmd = cmd.substr(0, pos1);
			cmd.erase();
			
			if (ret == 3)
				// -r any option
				cmd = newcmd + "-r " + newcmd;
			else
				// -r hosts options
				cmd = newcmd;

		}

		// need to forward requests 
		// STEP1: check the validity of private key
		string timestamp, signature, input;

		char buf[1024];
		sprintf(buf, "%lu", time(NULL));
		timestamp = buf;
		input = timestamp + cmd;

		if (m_usergroup->generate_signature(req->m_user, input, signature) < 0){
			strout = cmd;
			strout += "\n";			
			keysigned = 0;
		}else{
			// STEP2: attach signature to the original request
			strout.erase();
			req->m_user.mgid = m_usergroup->get_mgid((uid_t)req->m_user.uid);
			strout = cmd + "-i " + req->m_user.mgid + " " + timestamp
				+ " " + "-d " + signature; 
		}

		// if -r any option, return 
		if (ret == 3){
			cmd.erase();
			cmd = strout;
			return 0;
		}

		
		// if -r hosts option, transform request indata 
		transform_request(req, strout);
		
		// we need to fetch the host list from master machine first
		if (ret == 2){
			if (m_ismaster){
				hl = m_master->m_chosts;
				req->m_outref = m_master->m_chosts.size();
			}else{
				request_for_master(req, "host");
			}
		}else{
			req->m_outref = hl.size();
		}
		
		// how we have the individual host specification in the hl
		if (req->m_outref > 0){
			int needforward = 0;

			if (req->m_type != PARENT){
				req->m_type = PARENT;
				needforward = 1;
			}

			if (!keysigned)
				generate_reply(req, "Warning: No valid private key. The request will be sent out unsigned!\nUse \"mingle init\" to authenticate to the Master server for a signed request.\n\n");

			if (needforward)
				forward_request(req, hl);
		}
		if (ret < 1){
			req->m_status = PROCESS_DONE;
			return -1;
		}
	}
	return 0;
}

/*
 * parse_hosts: help function to parst out host list from string form :
 *                         -r host1 -p port1 host2 -p port2...
 * return value: 2: if -r all
 *               0: individual host specification and no local server
 *               1: individual host specification and with local server
 *              -1: failed in finding host
 */
int CReqQueue::parse_hosts(string& strhosts, THostList& hl, string& errstr)
{
	string::size_type pos, pos1;
	string stropt;
	Thost  nh;
	int needLocal = 0;

	pos1 = strhosts.find_first_not_of(" ", 0);	    
	while (pos1 < strhosts.length()){
		pos = strhosts.find_first_of("\n ", pos1);
		stropt = strhosts.substr(pos1, pos - pos1);
		if (pos < strhosts.length())
			pos1 = strhosts.find_first_not_of("\n ", pos);
		else
			pos1 = pos;
		
		// 'all' option
		if (stropt == "all"){
			// we need to get the host list from a master machine
			return 2;
		}
		
		// individual host specification, ip (may have port numbers)
		if (convert_host_to_ip(stropt, nh.ip) == -1){
			errstr += "Error: get IP error for ";
			errstr += stropt;
			errstr += "\n";
			return -1;
		}
		stropt.erase();
		nh.port = DEFAULT_PORT;
		
		// we also have a port number
		if ((pos1 < strhosts.length()) && (strhosts[pos1] == '-')){
			pos1 ++;
			if ((strhosts[pos1] != 'p') || (pos1 == (strhosts.length()-1))){
				errstr = "Error: Parse error.\n";
				return -1;
			}
			pos1 ++;
			pos1 = strhosts.find_first_not_of(" ", pos1);
			if (pos1 < strhosts.length()){
				pos =  strhosts.find_first_of("\n ", pos1);
				stropt = strhosts.substr(pos1, pos-pos1);
				nh.port = atoi(stropt.c_str());
				if (pos < strhosts.length())
					pos1 = strhosts.find_first_not_of("\n ", pos);
				else
					pos1 = pos;
				stropt.erase();
			}else{
				errstr = "Error: Parse erro.r\n";
				return -1;
			}
		}
		if (nh.port <= 0){
			errstr = "Error: Parse (port number) error.\n";
			return -1;
		}
		
		if (((memcmp(nh.ip, local_ip, 4) == 0) ||
		     (memcmp(nh.ip, m_lo_ip, 4) == 0)) &&
		    (nh.port == local_port)){
			needLocal = 1;
		}else{
			THostList::iterator hiter = find_if(hl.begin(),hl.end(),
						    Host_eq(nh));
			// a new host
			if (hiter == hl.end()){
				hl.push_back(nh);
			}
		}
	}	

	if (needLocal) return 1;
	else           return 0;
}

/*
 * qresult_to_string: help function: turn query result to string 
 */
void CReqQueue::qresult_to_string(CQuery& q, TResultList& qr, string& prostr)
{
	char  str[BUFLEN];

	sprintf(str, "Number of results found: %d\n\n", qr.size());
	prostr += str;

	TResultList::iterator riter;
	unsigned int i = 1;
	for (riter = qr.begin(); riter != qr.end(); riter ++){
		if (i > q.m_qNumResult)
			break;

		struct tm *curtime = localtime(&(riter->fLmtime));
		sprintf(str, "%d:  %s (fsize=%d  last-modified=%d/%d:%d:%d:%d)\n",
				i, riter->fName.c_str(), riter->fSize,
				curtime->tm_mon+1, curtime->tm_mday,
				curtime->tm_hour, curtime->tm_min, curtime->tm_sec);
		prostr += str;		
		if ((q.m_qPosition == CONTENT) || 
		    (q.m_qPosition == BOTH)){
			prostr += riter->fDescription;
		}
		prostr += "\n\n";				
		i ++;
	}
}

/*
 * transform_request : help function: remove signature from the original 
 *                     request after user identity is confirmed
 */
void CReqQueue::transform_request(CRequest *req, string& newdata)
{
	char* srcptr = (char*)newdata.c_str();
	char* destptr;

	/* delete indata */
	req->delete_indata();

	// dst ptr
	int i = req->m_rbuf;
	int rlen = req->m_trsize % BUFLEN;
	if (rlen == 0){
		if (req->m_trsize > 0)
			rlen = BUFLEN - 1;
		else{
			assert(req->m_indata[0] == 0);
			req->m_indata[0] = new char[BUFLEN];
		}
	}else{
		rlen --;
	}
	destptr = req->m_indata[i] + rlen;

	// copy len
	int len = newdata.length() + 1;
	req->m_trsize += len;
	int buflen = (rlen > 0) ? (BUFLEN - rlen) : BUFLEN;
	int cplen = (len > buflen) ? buflen : len;

	/* copy first line of data */
	memcpy(destptr, srcptr, cplen);
	len -= cplen;

	/* copy other stuff */
	while (len > 0){
		i ++;
		assert(req->m_indata[i] == 0);
		req->m_indata[i] = new char[BUFLEN];
		destptr = req->m_indata[i];
		srcptr += cplen;
		cplen = (len > BUFLEN) ? BUFLEN : len;
		memcpy(destptr, srcptr, cplen);
		len -= cplen;
	}
}


/*
 * getuid: help function: get uid of a process 
 */
int CReqQueue::my_getuid(pid_t pid, int *uid)
{
	struct stat pst;
	
	char sbuf[1024];
	sprintf(sbuf, "/proc/%d", pid);
	if (stat(sbuf, &pst) == -1){
		return -1;
	}
	
	*uid = pst.st_uid;
	
	return 0;
}

/*
 * valid_mgid: help function, check if a mingle user id is in an allowed format
 */
int CReqQueue::valid_mgid(string& mgid)
{
	return 1;
}


/////////////////////////////////////////////////////////////////
//  other help functions

/* 
 * forward_reply:  forward a reply received by a child request to 
 *                 parent request by copying indata from child request 
 *                 to outdata of parent request
 */
void CReqQueue::forward_reply(CRequest *req)
{
	CRequest *preq = find_parent(req);
	assert(preq && preq->m_outref);
	
	// copy in_packet from child's in_packet to parent's out_packet
	int pb = 0;

	if (preq->m_twsize > 0){
		pb = preq->m_twsize / BUFLEN;
		if (preq->m_twsize % BUFLEN > 0)
			pb ++;
	}

	for (int i = 0; i <= req->m_rbuf; i ++){
		preq->m_outdata[pb+i] = req->m_indata[i];
		req->m_indata[i] = 0;
	}

	preq->m_twsize = pb * BUFLEN + req->m_trsize;
	//	preq->m_twsize = req->m_trsize;
	preq->m_status = PROCESSING;	 	


	// set out_data to 0 since we just move it from the parent 
	for (int i = 0; i <= req->m_wbuf; i ++){
		req->m_outdata[i] = 0;
	}
	
	// delete child request first
	remove_request(req);

	// then go on forward back reply using parent		
	if (preq->m_outref == 0){
	    preq->m_status = PROCESS_DONE;
	}
	m_sender->send_reply(preq);
}

/*
 * forward_request - forward the request to all the hosts in the host list 
 */
void CReqQueue::forward_request(CRequest *req, THostList& th)
{
	while (th.size() > 0){
		Thost& nh = th.front();
		th.pop_front();

		// discard the host if local
		if (((memcmp(nh.ip, local_ip, 4) == 0) ||
		     (memcmp(nh.ip, m_lo_ip, 4) == 0)) && 
		    (nh.port == local_port)){
			req->m_outref --;
			continue;
		}

		// make a new request
		CRequest *nreq = new CRequest(req->m_reqid);
		
		// child type of request
		nreq->m_type =  CHILD;
		nreq->m_location = REMOTE;
		memcpy(nreq->m_host, nh.ip, 4);
		nreq->m_port = nh.port;
		
		nreq->m_twsize = req->m_trsize;
		for (int i = 0; i <= req->m_rbuf; i ++){
			assert (nreq->m_outdata[i] == 0);
			nreq->m_outdata[i] = req->m_indata[i];
		}
		
		insert_request(nreq);
		
		if (m_sender->send_request(nreq) == -1){
			char buf[BUFLEN];
			sprintf(buf, "Connection error to host %i.%i.%i.%i port %d.\n\n",
				nreq->m_host[0], nreq->m_host[1], nreq->m_host[2], nreq->m_host[3],
				nreq->m_port);
			string strbuf = buf;
			generate_reply(req, strbuf.c_str());
			req->m_status = PROCESSING;
			/* before removing the request, set out_data to 0
			   since we just move this from the parent 
			   note: we use req->m_rbuf instead of nreq->m_wbuf */
			for (int i = 0; i <= req->m_rbuf; i ++){
				nreq->m_outdata[i] = 0;
			}
			remove_request(nreq);
			req->m_status = PROCESS_DONE;
		}
	}	
}

/* 
 * request_for_master: create a child request that sends a request to master server
 *  
 */
void CReqQueue::request_for_master(CRequest *req, const char* rcontent)
{	
	assert(!m_ismaster);
	
	// check whether we've set the master success
	if (m_master->m_port == 0){
		req->m_outref --;
		generate_reply(req, "Error: Master machine was not correctly configured.\n");
		return;
	}
	
	// make a new request
	CRequest *nreq = new CRequest(req->m_reqid);
	req->m_outref ++;
	req->m_type = PARENT;
	
	// child type of request
	nreq->m_type = MASTER;
	nreq->m_location = REMOTE;
	memcpy(nreq->m_host, m_master->m_ip, 4);
	nreq->m_port = m_master->m_port;
	
	nreq->m_outdata[0] = new char[BUFLEN];
	strcpy(nreq->m_outdata[0], rcontent);
	nreq->m_twsize = strlen(nreq->m_outdata[0]) + 1;
	
	insert_request(nreq);
	if (m_sender->send_request(nreq) == -1){
		char buf[BUFLEN];
		sprintf(buf, "Connection error to master server %i.%i.%i.%i port %d.\n",
			nreq->m_host[0], nreq->m_host[1], nreq->m_host[2], nreq->m_host[3],
			nreq->m_port);

		if (strcmp(rcontent, "join") == 0){
			printf("%s", buf);
			remove_request(req);
		}else{
			string strbuf = buf;
			generate_reply(req, strbuf.c_str());
			req->m_status = PROCESSING;
			remove_request(nreq);
			req->m_status = PROCESS_DONE;
		}
	}
}


/* 
 * find_parent: find the corresponding parent request for a child type one 
 */
CRequest* CReqQueue::find_parent(CRequest *creq)
{
	TReqList *rl = m_head;

	while (rl){
		assert(rl->req);
		if ((rl->req->m_reqid == creq->m_reqid) &&
			(rl->req->m_outref > 0))
			return rl->req;
		rl = rl->next;	
	}

	return 0;
}

/* 
 * delete_request: delete req from reqQueue, do not check dependencies
 */ 
void CReqQueue::delete_request(CRequest *req)
{
	assert(req != 0);

	TReqList *rl = m_head;
	while ((rl) && (rl->req) && (rl->req->m_sd != req->m_sd))
		rl = rl->next;	

	assert(rl && (rl->req));		

	if (rl->prev)
		rl->prev->next = rl->next;	
	else
		m_head = rl->next;

	if (rl->next)
		rl->next->prev = rl->prev;

	else
		m_tail = rl->prev;

	delete rl->req;				
	delete rl;			
	m_num --;
}

/*
 * clear:  clear the request queue
 */
void CReqQueue::clear()
{
	TReqList *rl = m_head;
	while (m_head){
		rl = m_head;
		m_head = m_head->next;
		if (rl->req)
			delete rl->req;
			
		delete rl;		
	}
	
	m_head = m_tail = 0;
	m_num = 0;
}


#ifdef undef
/*
 * getpeereuid : get user credential information from the other end of the socket
 *               supported on some platforms
 */
uid_t CReqQueue::getpeereuid(int sd)
{
	struct ucred cred;
	int len = sizeof (cred);

	if (getsockopt(sd,SOL_SOCKET,SO_PEERCRED,&cred,&len))
	    return -1;

	return cred.uid;
}
#endif // undef
