/***************************************************************************
                          user.cpp  -  description
                             -------------------
    begin                : Wed Aug 8 2001
    copyright            : (C) 2001 by Yinglian Xie
    email                : ylxie@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
// user.cpp: implementation for the CUserGroup class.
//
//////////////////////////////////////////////////////////////////////

#include <unistd.h>
#include <string>
#include <list>
#include <algorithm>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <assert.h>
#include <errno.h>
#include <time.h>
#include "security.h"
#include "user.h"
#include "utils.h"

using namespace std;


//////////////////////////////////////////////////////////////////////
//  class  Muser implementation
//////////////////////////////////////////////////////////////////////

Muser::Muser(string mgid)
{
	m_mgid = mgid;
}

Muser::~Muser()
{
}

void Muser::update_pubkey(string pubkey, time_t expire_tv)
{
	m_pubkey.keystring.erase();
	m_pubkey.keystring = pubkey;
	m_pubkey.expire_tv = expire_tv;
}


//////////////////////////////////////////////////////////////////////
//  class Arp implementation
//////////////////////////////////////////////////////////////////////

Arp::Arp(string id)
{
	mgid = id;
}

Arp::~Arp()
{
}

int Arp::insert_SPD(uint nid, int mode)
{
	assert((mode == USER) || (mode == GROUP));

	if (mode == USER){
		list<uid_t>::iterator i;
		for (i = SPD_user.begin(); i != SPD_user.end(); i ++){
			if (*i == nid)
				return 0;
		}
		SPD_user.push_back(nid);
	}else{
		list<gid_t>::iterator i;
		for (i = SPD_group.begin(); i != SPD_group.end(); i ++){
			if (*i == nid)
				return 0;
		}
		SPD_group.push_back(nid);
	}
	return 0;
}

int Arp::delete_SPD(uint nid, int mode)
{
	assert((mode == USER) || (mode == GROUP));

	if (mode == USER){
		list<uid_t>::iterator i = find(SPD_user.begin(), SPD_user.end(),
					       nid);
		if (i != SPD_user.end())
			SPD_user.erase(i);
		else
			return -1;
	}else{
		list<gid_t>::iterator i = find(SPD_group.begin(), SPD_group.end(),
					       nid);
		if (i != SPD_group.end())
			SPD_group.erase(i);
		else
			return -1;
	}
	return 0;
}

string Arp::list_SPD()
{
	string output = mgid + "\tuser< ";
	
	list<uid_t>::iterator i;
	for (i = SPD_user.begin(); i != SPD_user.end(); i ++){
		output += get_ustring(*i);
		output += " ";
	}

	output += ">\tgroup< ";
	list<gid_t>::iterator j;
	for (j = SPD_group.begin(); j != SPD_group.end(); j ++){
		output += get_gstring(*j);
		output += " ";
	}
	output += ">\n";

	return output;
}

void Arp::flush_info(FILE *fp)
{
	fprintf(fp, "%s %d %d:", mgid.c_str(), SPD_user.size(), SPD_group.size());

	list<uid_t>::iterator ui;
	for (ui = SPD_user.begin(); ui != SPD_user.end(); ui ++){
		fprintf(fp, " %u", *ui);
	}

	list<gid_t>::iterator gi;
	for (gi = SPD_group.begin(); gi != SPD_group.end(); gi ++){
		fprintf(fp, " %u", *gi);
	}
}

void Arp::setup_info(char* buf)
{
	int us, gs;
	char *ptr, buf1[MAXSTRSIZE];
	uint  tempid;
	uid_t uid;
	gid_t gid;

	memset(buf1, 0, MAXSTRSIZE);
	sscanf(buf, "%s %d %d:", buf1, &us, &gs);
	mgid = buf1;
	ptr = strchr(buf, ':');
	ptr ++;
	for (int i = 0; i < us; i ++){
		sscanf(ptr, " %u", &tempid);
		uid = tempid;
		SPD_user.push_back(uid);
		ptr ++;
		ptr = strchr(ptr, ' ');
	}
	for (int i = 0; i < gs; i ++){
		sscanf(ptr, " %u", &tempid);
		gid = tempid;
		SPD_group.push_back(gid);
		ptr ++;
		ptr = strchr(ptr, ' ');
	}
}

//////////////////////////////////////////////////////////////////////
//  class Mowner implementation
//////////////////////////////////////////////////////////////////////

Mowner::Mowner(uid_t uid, string mgid)
{
	m_uid = uid;
	m_mgid = mgid;
}

Mowner::~Mowner()
{
}

int Mowner::get_SPD(string mgid, list<uid_t>& uSPD, list<gid_t>& gSPD)
{
	TArpList::iterator aiter = find_if(m_arp.begin(), 
					   m_arp.end(), Arp_eq(mgid));

	if (aiter == m_arp.end())
		return -1;

	uSPD = aiter->SPD_user;
	gSPD = aiter->SPD_group;

	return 0;
}

int Mowner::clear_SPD(string mgid)
{
	TArpList::iterator aiter = find_if(m_arp.begin(), 
					   m_arp.end(), Arp_eq(mgid));

	if (aiter == m_arp.end())
		return -1;

	m_arp.erase(aiter);
	return 0;
}

/*
 * insert_arp: insert an access-right mapping for mgid
 *             return: 0 if mgid does not exist before
 *                     1 if mgid exists before
 */
int Mowner::insert_arp(string mgid, uint nid, int mode)
{
	/* check if the mgid already exists in ARP list */
	TArpList::iterator aiter = find_if(m_arp.begin(), m_arp.end(), Arp_eq(mgid));
	if (aiter == m_arp.end()){
		/* need to create a new ARP list entry */
		Arp newarp(mgid);
		newarp.insert_SPD(nid, mode);
		m_arp.push_back(newarp);

		return 0;
	}else{
		/* update the old ARP list entry */
		aiter->insert_SPD(nid, mode);
	}

	return 1;
}

/*
 * delete_arp: delete an access-right mapping for mgid
 *             return: 0 if operation success
 *                    -1 if operation fails
 */
int Mowner::delete_arp(string mgid, uint nid, int mode)
{
	/* check if the mgid already exists in ARP list */
	TArpList::iterator aiter = find_if(m_arp.begin(), 
					   m_arp.end(), Arp_eq(mgid));
	if (aiter == m_arp.end())
		return -1;
	
	/* remote arp from the found SPD */
	if (aiter->delete_SPD(nid, mode) < 0)
		return -1;

	return 0;
}

/*
 * list-arp: list access-right information for a user
 *           note: if mgid is empty, list all arp info
 */
string Mowner::list_arp(string mgid)
{
	TArpList::iterator aiter;
	string output;

	if (mgid.empty()){
		for (aiter = m_arp.begin(); aiter != m_arp.end(); aiter ++){
			string newout = aiter->list_SPD();
			output += newout;
		}
		return output;
	}

	/* check if the mgid already exists in ARP list */
	aiter = find_if(m_arp.begin(), m_arp.end(), Arp_eq(mgid));
	if (aiter == m_arp.end())
		output = "Error: The specified mgid does not exist in mappings.\n";
	else
		output = aiter->list_SPD();
	return output;
}

void Mowner::update_prikey(string keystr, time_t expire_tv)
{
	m_prikey.keystring.erase();
	m_prikey.keystring = keystr;
	m_prikey.expire_tv = expire_tv;
}


/*
 * exist_map: check if mgid exist in the arp list
 */
bool Mowner::exist_map(string mgid)
{
	TArpList::iterator i = find_if(m_arp.begin(), m_arp.end(), Arp_eq(mgid));
	if (i == m_arp.end())
		return 0;
	else
		return 1;
}

void Mowner::flush_info(FILE *fp)
{
	/* uid, mgid, private_key, arp_size */
	fprintf(fp, "%u %s %d\n", m_uid, m_mgid.c_str(), m_arp.size());
	
	/* arp info */
	TArpList::iterator i;
	for (i = m_arp.begin(); i != m_arp.end(); i ++){
		i->flush_info(fp);
		fprintf(fp, "\n");
	}
}

int Mowner::setup_arp(FILE *fp, int arpsize)
{
	char buf[MAXSTRSIZE];
	string temp;

	for (int i = 0; i < arpsize; i ++){
		if ((!fgets(buf, MAXSTRSIZE, fp)) || (strcmp(buf, "\n") == 0))
			return -1;
		Arp arp(temp);
		arp.setup_info(buf);
		m_arp.push_back(arp);
	}
	return 0;
}


//////////////////////////////////////////////////////////////////////
//  class CUse  roup implementation
//////////////////////////////////////////////////////////////////////

CUserGroup::CUserGroup()
{
}

CUserGroup::~CUserGroup()
{
}

////////////////////////////////////
// public functions
//////////////////////////////////

/////////////////////////////////////////////////////////////////
//  functions to set up parameters, directory based on config file

/*
 * set_security_mode: if mode == 1, we need security checking
 */
void CUserGroup::set_security_mode(int mode)
{
	m_securityMode = mode;
}

/*
 * setup_security: security setup, include user information 
 */
string CUserGroup::setup_security(string dir, string pubkey, string prikey,
				  int pubkey_ttl, int prikey_ttl)
{
	struct stat fst;

	get_absolute_fname(dir, m_securitydir);
	m_userfile = m_securitydir + "/" + USER_FNAME;

	/* dir already exists */
	if (stat(m_securitydir.c_str(), &fst) == 0){
	  if (S_ISDIR(fst.st_mode)){
	    read_security();
	  }else{
	    printf("Security dir setup error! exit now...\n");
	    exit(0);
	  }
	}else{
		/* we need to create a new dir */
		if ((errno != ENOENT) ||
		    (mkdir(m_securitydir.c_str(), S_IRWXU | S_IRGRP | S_IROTH) == -1)){
			printf("Security dir set up error! (%s)  exit now...\n", strerror(errno));
			exit(0);
		}
	}
	
	/* setup public/private key string and ttl 
	   if public/private key does not exist, create one */
	if (stat(pubkey.c_str(), &fst) != 0){
		if (errno == ENOENT){
			generate_RSA_key(prikey.c_str(), pubkey.c_str());
		}else{
			printf("Public key read error! (%s)  exit now...\n", strerror(errno));
			exit(0);
		}
	}
	if (stat(prikey.c_str(), &fst) != 0){
		if (errno == ENOENT){
			generate_RSA_key(prikey.c_str(), pubkey.c_str());
		}else{
			printf("Private key read error! (%s)  exit now...\n", strerror(errno));
			exit(0);
		}
	}

	if (read_RSA_key(pubkey, m_pubkey) < 0){
		printf("Read public key error! exit now...\n");
		exit(0);
	}
	if (read_RSA_key(prikey, m_prikey) < 0){
		printf("Read private key error! exit now...\n");
		exit(0);
	}

	m_pubkey_ttl = pubkey_ttl * 60 * 60;
	m_prikey_ttl = prikey_ttl * 60 * 60;

	return m_securitydir;
}

/*
 * read_security: read security info (user info)
 */
void CUserGroup::read_security()
{
	FILE *fp;
	char buf[MAXSTRSIZE], buf1[MAXSTRSIZE];
	
	/* set up user info based on user file */
	if (!(fp = fopen(m_userfile.c_str(), "r"))){
		return;
	}

	/* read info now */
	while ((fgets(buf, MAXSTRSIZE, fp)) && 
	       (buf[0] != '\n') && (buf[0] != '\r')){
		
		uid_t  uid;
		uint   tempid;
		string mgid;
		int  arpsize;
		
		/* uid and mgid */
		memset(buf1, 0, MAXSTRSIZE);
		sscanf(buf, "%u %s %d\n", &(tempid), buf1, &arpsize);
		uid = tempid;
		mgid = buf1;
		Mowner owner(uid, mgid);
		
		if (owner.setup_arp(fp, arpsize) < 0){
				mingle_debug1("\nUserGroup::set_security_file(): security file format error.");
				exit(0);
		}
		m_owners.push_back(owner);
	}

	/* close user info file */
	fclose(fp);
}
 

/////////////////////////////////////////////////////////////////
//  RSA key cache functions

/*
 * mingle_init: Mingle init, update private key
 *              if owner does not exist: create an owner entry first
 */
void CUserGroup::mingle_init(Identity& muser, string prikey)
{
	/* set private key expiration time */
	time_t expire_tv = time(NULL) + m_prikey_ttl;

	/* the owner already exists */
	TOwnerSet::iterator i;
	for (i = m_owners.begin(); i != m_owners.end(); i ++){
		if ((unsigned int)muser.uid == i->m_uid){
			i->update_prikey(prikey, expire_tv);
			return;
		}
	}
	
	/* the owner does not exist */
	Mowner newowner(muser.uid, muser.mgid);
	newowner.update_prikey(prikey, expire_tv);
	m_owners.push_back(newowner);

	flush_security();
	return;
}

/*
 * display_userinfo: Display Mingle owner information
 *                   include : mingle id, uid, afs id
 */
void CUserGroup::display_userinfo(uid_t uid, string& output)
{
	/* check if the owner already exists */
	Mowner *owner = get_owner_uid(uid);
	if (!owner){
		output = "Error: Permission denied.\n";
		return;
	}

	output = "Mingle ID     = ";
	output += owner->m_mgid;
	output +="\nLocal user ID = ";
	output += get_ustring(uid);
	output += "\n";
}

/*
 * update_pubkey: reset ttl for the public key
 *                if user does not exist: create a user entry first
 *                however, do not need to flush this info to disk
 */
void CUserGroup::update_pubkey(string mgid, string pubkey)
{
	/* set public key expiration time */
	time_t expire_tv = time(NULL) + m_pubkey_ttl;

	Muser *mu = get_muser(mgid);
	if (!mu){
		Muser newu(mgid);
		newu.update_pubkey(pubkey, expire_tv);
		m_users.push_back(newu);
	}else{
		mu->update_pubkey(pubkey, expire_tv);
	}
}

/*
 * owner_exist: check if a user has already signed on
 */
int CUserGroup::owner_exist(uid_t uid)
{
	Mowner* owner = get_owner_uid(uid);
	if (owner)
		return 1;
	else
		return 0;
}


/////////////////////////////////////////////////////////////////
//  access control functions

/*
 * add_arp: add one access-right mapping for owner 
 *          note: the same mapping can be added twice( it does not hurt :))
 *                create a new entry in user list if no such user record
 */
int CUserGroup::add_arp(Identity& user, string mgid, string localID,
			int mode, string& errstr)
{
	Mowner *owner;
	int  nid;

	/* check if the owner exists */
	if ((owner = get_owner_mgid(user.mgid)) == 0){
		errstr = "Error: Permission denied.\n";
		return -1;
	}

	/* check if the localID exists */
	if (mode == USER)
		nid = get_uid(localID);
	else
		nid = get_gid(localID);

	if (nid < 0){
		errstr = "Error: The specified local ID does not exist.\n";
		return -1;
	}

	/* insert mapping for owner */
	owner->insert_arp(mgid, (uint)nid, mode);
	flush_security();

	return 0;
}


/*
 * rm_arp: add one access-right mapping from owner's arp list
 *         note: cannot remove sth that does not exist
 *               do not remove entries from user list even SPDs are empty
 */
int CUserGroup::rm_arp(Identity& user, string mgid, string localID,
			int mode, string& errstr)
{
	Mowner *owner;
	int  nid;

	/* check if the owner exists */
	if ((owner = get_owner_mgid(user.mgid)) ==  0){
		errstr = "Error: Permission denied.\n";
		return -1;
	}

	/* check if the localID exists */
	if (mode == USER)
		nid = get_uid(localID);
	else
		nid = get_gid(localID);

	if (nid < 0){
		errstr = "Error: The specified local ID does not exist.\n";
		return -1;
	}

	/* remove the arp */
	if (owner->delete_arp(mgid, (uint)nid, mode) < 0){
		errstr = "Error: The mapping does not exist.\n";
		return -1;
	}

	flush_security();
	return 0;
}

/*
 * clear_SPD: clear SPD_user(mgid)
 */
int CUserGroup::clear_SPD(Identity& user, string mgid, string& errstr)
{
	Mowner *owner;

	/* check if the owner exists */
	if ((owner = get_owner_mgid(user.mgid)) == 0){
		errstr = "Error: Permission denied.\n";
		return -1;
	}
	
	/* clear SPD now */
	if (owner->clear_SPD(mgid) < 0){
		errstr = "Error: The specified mingle id does not exist.\n";
		return -1;
	}

	flush_security();
	return 0;
}

/*
 * ls_arp: list SPDs for mgid
 *         note: if mgid is an empty string, list all arps in the arp list
 */
void  CUserGroup::ls_arp(Identity& user, string mgid, string& errstr)
{
	Mowner *owner;

	/* check if the owner exists */
	if ((owner = get_owner_mgid(user.mgid)) == 0){
		errstr = "Error: Permission denied.\n";
		return;
	}

	/* put arp list into the errstr */
	errstr = owner->list_arp(mgid);

}


/////////////////////////////////////////////////////////////////
//   permission checking functions



/*
 * index_file_permitted: check if user is permitted to index a file
 *                       if system error, return permission denied (0)
 */
bool CUserGroup::index_file_permitted(Identity& user, string file)
{
	/* if security mode is turned off , always return true */
	if (m_securityMode == 0)
		return 1;

	/* if not some owner of Mingle, return false */
	Mowner *owner = get_owner_mgid(user.mgid);
	if (!owner)
		return 0;

	/* get owner of the file, only owner can index */
	struct stat fst;
	if (stat(file.c_str(), &fst) == -1){
		return 0;
	}
	if ((owner->m_uid == fst.st_uid) && (S_IRUSR & fst.st_mode))
		return 1;
	else
		return 0;

}

/*
 * search_file_permitted: check if user is permitted to search a file
 *                        check based on file system access control
 *                        and access-right mapping of the file owner
 */
bool CUserGroup::search_file_permitted(Identity& user, string file)
{
	/* if security mode is turned off , always return true */
	if (m_securityMode == 0)
		return 1;

	/* get file owner */
	struct stat fst;
	if (stat(file.c_str(), &fst) == -1){
		return 0;
	}

	Mowner *owner = get_owner_uid(fst.st_uid);
	if (!owner)
		return 0;
	if ((owner->m_mgid == user.mgid) || 
	    ((user.uid >= 0) && (owner->m_uid == (uid_t)user.uid)))
		return 1;

	/* get user's SPD */
	list<uid_t> uSPD;
	list<gid_t> gSPD;
	if (user.uid >= 0)
		uSPD.push_back((uid_t)user.uid);
	owner->get_SPD(user.mgid, uSPD, gSPD);

	/* check if any user in the SPDs can read the file */
	list<uid_t>::iterator i;
	for (i = uSPD.begin(); i != uSPD.end(); i ++){
		if ((*i == fst.st_uid) && (S_IRUSR & fst.st_mode))
			return 1;
		else{
			int gid = get_gid(*i);
			if ((gid >= 0) && ((gid_t)gid != fst.st_gid) && 
			    (S_IROTH & fst.st_mode))
			return 1;
		}
	}

	list<gid_t>::iterator j;
	for (j = gSPD.begin(); j != gSPD.end(); j ++){
		if ((*j == fst.st_gid) && (S_IRGRP & fst.st_mode))
			return 1;
	}

	return 0;
}


/////////////////////////////////////////////////////////////////
//   authentication functions

/*
 * retrieve_identity: fill the blank fields of a user identity
 *                    either uid or mingle id has to be know prior
 */
void CUserGroup::retrieve_identity(Identity& user)
{
	Mowner* o;

	if (user.uid >= 0){
		o = get_owner_uid(user.uid);
		if (o)
			user.mgid = o->m_mgid;

	}else if (!user.mgid.empty()){
		o = get_owner_mgid(user.mgid);
		if (o)
			user.uid = o->m_uid;
	}
}

/*
 * identity_confirmed: confirm the identity of claimed mgid by checking 
 *                     signature(an entrypted version of cmdline)
 *                     return value:  0  false identity
 *                                   -1  public key expired or no public key
 *                                    1  identity confirmed
 */
int CUserGroup::identity_confirmed(string mgid, string cmdline, string signature)
{
	/* get the public key */
	Muser* u = get_muser(mgid);
	if ((!u) || (u->m_pubkey.keystring.empty()) ||
	    (u->m_pubkey.expire_tv < time(NULL)))
		return -1;

	return RSAVerifyString(u->m_pubkey.keystring, cmdline, 
			       signature);
}

/*
 * generate_signature: generate signature for a request
 *                     need to check if private key has expired
 *                     return value: 0 input signed
 *                                  -1 generate signature failure
 */
int CUserGroup::generate_signature(Identity& user, string input, string& signature)
{
	// check if private key is valid
	Mowner* o = get_owner_uid(user.uid);
	if ((!o) || (o->m_prikey.keystring.empty()) ||
	    (o->m_prikey.expire_tv < time(NULL)))
		return -1;

	/* generate signature */
	RSASignString(o->m_prikey.keystring, input, signature);
	
	/* return */
	return 0;
}

string CUserGroup::encrypt_string(string pubkey, string cleartext)
{
	string ciphertext;
	RSAEncryptString(pubkey, cleartext, ciphertext);
	return ciphertext;
}

string CUserGroup::decrypt_string(string ciphertext)
{
	string cleartext;
	RSADecryptString(m_prikey, cleartext, ciphertext);
	return cleartext;
}


////////////////////////////////////
// private functions
//////////////////////////////////

/////////////////////////////////////////////////////////////////
//  flush security info into disk

void CUserGroup::flush_security()
{
	FILE *fp;

	/* open user file for rewriting */
	if (!(fp = fopen(m_userfile.c_str(), "w"))){
		mingle_debug2("\nUserGroup::open user info file error", errno);
		return;
	}

	/* flush owner info */
	TOwnerSet::iterator i;
	for (i = m_owners.begin(); i != m_owners.end(); i ++){
		i->flush_info(fp);
	}

	/* close user info file */
	fclose(fp);
}

