/***************************************************************************
                          receiver.cpp  -  description
                             -------------------
    begin                : Mon Sep 24 2001
    copyright            : (C) 2001 by Yinglian Xie
    email                : ylxie@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
// receiver.cpp: implementation for the Creceiver class.
//
//////////////////////////////////////////////////////////////////////

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "receiver.h"
#include "request.h"
#include "mingle.h"
#include "utils.h"

#ifndef SHUT_RD
#define SHUT_RD 0
#endif

//////////////////////////////////////////////////////////////////////
//  class  CReceiver implementation
//////////////////////////////////////////////////////////////////////

CReceiver::CReceiver(){
}

CReceiver::~CReceiver(){
}

void CReceiver::set_mingle_port(short port)
{
	m_minglePort = port;
}

void CReceiver::set_mingle_path(const char* path)
{
	m_unixPath = path;
}

void CReceiver::set_user_mode(int mode)
{
	if (mode == SINGLEUSER)
		m_multiuser = 0;
	else if (mode == MULTIUSER)
		m_multiuser = 1;
	else{
		printf("UserMode in config file not valid! Switch to single user mode\n");
		m_multiuser = 0;
	}
}

int CReceiver::net_socket_listen()
{
	int option = 1;
	struct sockaddr_in addr;

	/* Create a socket, then bind() and listen() it */
	m_netsd = socket(AF_INET, SOCK_STREAM, 0);
	if (m_netsd == -1)
	{
		mingle_debug2("\nCReceiver::net_socket_listen():unable to create a socket",
						errno);
		return -1;
	}

//	setsockopt(m_sd, SOL_SOCKET, SO_KEEPALIVE, (void *) &option, sizeof(option));
	setsockopt(m_netsd, SOL_SOCKET, SO_REUSEADDR, (void *) &option, sizeof(option));

	fcntl(m_netsd, F_SETFL, O_NONBLOCK);	/* Set the file descriptor non blocking */

	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = htons(m_minglePort);

	/* bind() the socket */
	if (bind(m_netsd, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		mingle_debug2("\nCReceiver::net_socket_listen():unable to bind the socket",
						errno);
		return -1;
	}

	/* listen() the socket */
	if (listen(m_netsd, 1) == -1)
	{
		mingle_debug2("\nCReceiver::net_socket_listen():unable to listen the socket",
						errno);
		return -1;
	}	
	
	return 0;
}

int CReceiver::unix_socket_listen()
{
	struct sockaddr_un addr;
	
	/* create a socket */
	m_unixsd = socket(AF_LOCAL, SOCK_STREAM, 0);
	if (m_unixsd == -1)
	{
		mingle_debug2("\nCReceiver::unix_socket_listen():unable to create a socket",
						errno);
		return -1;
	}	

	fcntl(m_unixsd, F_SETFL, O_NONBLOCK);	/* Set the file descriptor non blocking */

	unlink(m_unixPath);
	bzero((char*)&addr, sizeof(addr));
	addr.sun_family = AF_LOCAL;
	strcpy(addr.sun_path, m_unixPath);
	
	/* bind() and then listen() */
	if (bind(m_unixsd, (struct sockaddr *) &addr, sizeof(addr)) == -1)
	{
		mingle_debug2("\nCReceiver::unix_socket_listen():unable to bind the socket",
						errno);
		return -1;
	}
	
	if (listen(m_unixsd, 1) == -1)
	{
		mingle_debug2("\nCReceiver::unix_socket_listen():unable to listen the socket",
						errno);
		return -1;
	}			
		
	/* modify the access permission of m_unixsd 
	   so that other users can access */
	if (m_multiuser){
		if (chmod(m_unixPath, 0777) < 0){
			mingle_debug2("\nCReceiver::unix_socket_list():chmod() error ", errno);
		}
	}

	return 0;
}

/*
 * request_accept: accept a request from a listening socket
 *                 need to set up request record and hand over to 
 *                 the request manager
 */
int CReceiver::request_accept(int listenfd)
{
	int newsock;
	struct sockaddr_in  sinaddr;
	struct sockaddr_un  sunaddr;
	socklen_t len;
	
	if (listenfd == m_netsd){
		len = sizeof(sinaddr);
		newsock = accept(listenfd, (struct sockaddr *)&sinaddr, &len);
	}	
	else{
		len = sizeof(sunaddr);
		newsock = accept(listenfd, (struct sockaddr *)&sunaddr, &len);
	}		
			
	if (newsock < 0) {
		mingle_debug2("\nCReceiver::request_accept(): accept error",
						errno);
		return -1;				
	}

	// set the socket to be non-blocking
	fcntl(newsock, F_SETFL, O_NONBLOCK);	
	
	// create a new request structure
	CRequest *req = new CRequest(m_reqQ->m_maxreqid);
	m_reqQ->incre_reqid();

	req->m_sd = newsock;
	req->m_status = PROCESSING;
	if (listenfd == m_netsd){
		// request from network
		memcpy(req->m_host, &sinaddr.sin_addr, HOSTLEN);
		req->m_port = ntohs(sinaddr.sin_port);
		req->m_location = REMOTE;
	}else{
		// request from local machine
		req->m_location = LOCAL;
	}

	m_reqQ->insert_request(req);
	
	return newsock;	
}	

/*
 * keep_reading: read from a socket until not allowed
 *               return 0: process finished (or error)
 *                     -1: need to continue reading next time
 */				
int CReceiver::keep_reading(int sd)
{
	int num, i;
	
	CRequest *req = m_reqQ->find(sd);	
	assert(req);

	while (1){
	    // read things into readbuf
	    i = req->m_rbuf;
	    if (!req->m_indata[i]){
		    req->m_indata[i] = new char[BUFLEN];
		    req->m_nrsize = BUFLEN - 1;
		    req->m_readPtr = req->m_indata[i];
	    }

	    num = read(sd, req->m_readPtr, req->m_nrsize);
	    if (num == -1){
		    // not ready to read
		    if ((errno == EINTR) || (errno == EAGAIN)){
			    return -1;
		    }else{
			    // read error
			    mingle_debug2("\nCReceiver::keep_reading():read error",
					  errno);			
			    m_reqQ->remove_request(req);
			    close(sd);
			    return 0;	
		    }
	    }
	    
	    // EOF: we are done with reading, pad end of data with '\n'
	    if (num == 0){
		    i = req->m_rbuf;
		    if (!req->m_indata[i]){
			    req->m_indata[i] = new char[BUFLEN];
			    req->m_readPtr = req->m_indata[i];
		    }
		    *(req->m_readPtr) = '\0';
		    req->m_trsize ++;
		
		    break;
	    }
	    
	    // read success
	    req->m_nrsize -= num;
	    req->m_readPtr += num;
	    req->m_trsize += num;
	    if (req->m_nrsize == 0){
		    req->m_indata[req->m_rbuf][BUFLEN-1] = '\0';
		    req->m_rbuf ++;
		    if (req->m_rbuf >= NUMBUF){
			    mingle_debug1("\nCReceiver::keep_reading(): data size too large");
			    req->m_rbuf = BUFLEN;
			    req->m_indata[NUMBUF-1][BUFLEN-1] = '\0';

			    break;
		    }
	    }
	} // end while

	if ((req->m_type == CHILD) || (req->m_type == MASTER))
		close(sd);
	else
		shutdown(sd, SHUT_RD);
	m_reqQ->process_request(req);
	return 0;
}

