/***************************************************************************
                          mingle.cpp  -  description
                             -------------------
    begin                : Wed Aug 8 2001
    copyright            : (C) 2001 by Yinglian Xie
    email                : ylxie@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// mingle.cpp: implementation of the CMingle class.
//
//////////////////////////////////////////////////////////////////////

#include <sys/socket.h>
#include <unistd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <netdb.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <string>
#include "utils.h"
#include "mingle.h"
#include "findex.h"
#include "qprocessor.h"
#include "sender.h"
#include "receiver.h"
#include "fdmanager.h"
#include "request.h"
#include "user.h"
using namespace std;

#define MAXSTRLEN 8096

uchar local_ip[HOSTLEN];  
short local_port;

//////////////////////////////////////////////////////////////////////
//  class  CMingle
//////////////////////////////////////////////////////////////////////

CMingle::CMingle(const char* name, const char* configFile)
{
	char* iface = 0;

	init_config();

	// read from configuration file
	if (configFile)
	    read_config_file(configFile);

	// init all mingle modules
	init_module();	

	// print name and version
	printf("%s\n", name);

	// init the network 
	memset(local_ip, 0, 4);
	if (init_net(iface) == -1){
		printf("The network is not up, working in local mode only\n");
		m_fdManager->m_netUp = 0;
	}	
	else{
		local_port = atoi(m_optvalue[MinglePort].c_str());
		printf("Local ip: %u.%u.%u.%u  local port: %d\n",
		       local_ip[0], local_ip[1], local_ip[2], local_ip[3], 
		       local_port);
		m_fdManager->set_net_up();
	}

	// set up the configuration options
	setup_config();
	fflush(stdout);
}

CMingle::~CMingle()
{
	delete m_index;
	delete m_qprocessor;
	delete m_sender;
	delete m_receiver;
	delete m_reqQueue;
	delete m_fdManager;
	delete m_usergroup;

	delete m_master;
}

const string CMingle::m_optname[MAXOPTNUM] = 
{
	"MinglePort", "MinglePath", "CacheSize", 
	"IndexMemory", "MaxWordLen", "MinWordLen",
	"IndexDir",  "WordChar", "StopWords",
	"IsMaster", "MasterHost", "MasterPort",
	"ClusterHosts", "SecurityMode", "SecurityDir",
	"IndexInterval", "IndexException", "DelimitChar",
	"UserMode", "PublicKey", "PrivateKey", 
	"PublicTTL", "PrivateTTL"
};

/*
 * init_config: set up default configure options without config file
 */
void CMingle::init_config()
{
	// by default, no cluster host set up, and no stopwords setup
	m_optvalue[MinglePort] = "7018";
	m_optvalue[MinglePath] = "/tmp/mingle.str";
	m_optvalue[CacheSize] = "1000";     // 1M
	m_optvalue[IndexMemory] = "30";     // 10M
	m_optvalue[MaxWordLen] = "40";
	m_optvalue[MinWordLen] = "2";
	m_optvalue[IndexDir] = "../index";
	m_optvalue[WordChar] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890_@";
	m_optvalue[IsMaster] = "1";
	m_optvalue[MasterHost] = "LOCAL";
	m_optvalue[MasterPort] = "7018";
	m_optvalue[SecurityMode] = "1";
	m_optvalue[SecurityDir] = "../security";
	m_optvalue[IndexInterval] = "24"; // by default, update index per day
	m_optvalue[DelimitChar] = ",.;:\"\'!?()[]{}<>@";
	m_optvalue[UserMode] = "0";
	m_optvalue[PublicKey] = "../security/rsa-public";
	m_optvalue[PrivateKey] = "../security/rsa-private";
}

/*
 * read_config_file: change configure options based on config file
 */
void CMingle::read_config_file(const char* cfname)
{
	char buf[MAXSTRLEN];
	string optname, optvalue, line;

	FILE *cfp = fopen(cfname, "r");
	if (!cfp){
		printf("Cannot open configuration file: %s... Abort!!\n", cfname);
		exit(-1);
	}

	while (fgets(buf, MAXSTRLEN, cfp) != 0){
	  if ((buf[0] == '#') || (buf[0] == '\n') || (buf[0] == '\r'))
	    continue;

		line = buf;
		string::size_type pos_s = line.find_first_not_of(" ");
		if (pos_s > line.length())
		  continue;
		string::size_type pos_e = line.find_first_of(" =", pos_s);
		if (pos_e > line.length())
		  continue;
		optname = line.substr(pos_s, pos_e-pos_s);
		
		pos_s = line.find_first_not_of(" =", pos_e);
		if (pos_s > line.length())
		  continue;

		if ((optname == "StopWords") || (optname == "ClusterHosts") ||
		    (optname == "IndexException"))
		    pos_e = line.find_first_of("\r\n", pos_s);
		else
		    pos_e = line.find_first_of(" \r\n", pos_s);
		optvalue = line.substr(pos_s, pos_e-pos_s);

		int i = get_opt_index(optname);
		if (i >= 0){
			m_optvalue[i].erase();
			m_optvalue[i] = optvalue;
		}

		optname.erase();
		optvalue.erase();
		line.erase();
	}

	fclose(cfp);
}

/*
 * setup_config: based on configure options, set up modules
 */
void CMingle::setup_config( )
{
	string sdir;

	m_receiver->set_mingle_path(m_optvalue[MinglePath].c_str());
	m_receiver->set_mingle_port(local_port);
	m_receiver->set_user_mode(atoi(m_optvalue[UserMode].c_str()));

	m_usergroup->set_security_mode(atoi(m_optvalue[SecurityMode].c_str()));
	sdir = m_usergroup->setup_security(m_optvalue[SecurityDir],
					   m_optvalue[PublicKey], m_optvalue[PrivateKey],
					   atoi(m_optvalue[PublicTTL].c_str()),
					   atoi(m_optvalue[PrivateTTL].c_str()));

	if (atoi(m_optvalue[IsMaster].c_str()) == 1){
		m_reqQueue->set_master_on();
		m_master->setup_signon(sdir);
		if (!m_optvalue[ClusterHosts].empty())
			m_master->setup_cluster(m_optvalue[ClusterHosts]);
	}
	else{
		m_reqQueue->set_master_off();
		m_master->set_master_host(m_optvalue[MasterHost],
					    atoi(m_optvalue[MasterPort].c_str()));
	} 

	m_index->set_max_wordlen(atoi(m_optvalue[MaxWordLen].c_str()));
	m_index->set_min_wordlen(atoi(m_optvalue[MinWordLen].c_str()));
	m_index->set_indexdir(m_optvalue[IndexDir]);
	m_index->set_cache_size(atoi(m_optvalue[CacheSize].c_str()));
	m_index->set_wordchar(m_optvalue[WordChar]);
	m_index->set_delimitchar(m_optvalue[DelimitChar]);
	m_index->set_index_memory(atoi(m_optvalue[IndexMemory].c_str()));
	if (!m_optvalue[StopWords].empty())
		m_index->set_stopwords(m_optvalue[StopWords]);
	if (!m_optvalue[IndexException].empty())
		m_index->set_exception(m_optvalue[IndexException]);

	m_qprocessor->set_delimitchar(m_optvalue[DelimitChar]);

	set_index_interval(atoi(m_optvalue[IndexInterval].c_str()));
}

int CMingle::get_opt_index(string optname)
{
	int i;
	for (i = 0; i < MAXOPTNUM; i ++){
		if (optname == m_optname[i])
			return i;
	}
	return -1;
}

/* 
 * init_module: init all the mingle server modules 
 */
void CMingle::init_module()
{
	// creat all the modules
	m_index = new CFindex;
	m_qprocessor = new CQprocessor(m_index);
	m_usergroup = new CUserGroup;
	m_master = new CMaster;
	
	m_receiver = new CReceiver;
	m_sender = new CSender;
	m_fdManager = new CFdManager;
	
	m_reqQueue = new CReqQueue;
	
	// initialize variables of the modules
	m_index->m_usergroup = m_usergroup;
	m_receiver->m_reqQ = m_reqQueue;
	
	m_sender->m_reqQ = m_reqQueue;
	m_sender->m_fdManager = m_fdManager;
	
	m_reqQueue->m_index = m_index;
	m_reqQueue->m_qprocessor = m_qprocessor;
	m_reqQueue->m_usergroup = m_usergroup;
	m_reqQueue->m_sender = m_sender;
	m_reqQueue->m_master = m_master;
	
	m_fdManager->m_receiver = m_receiver;
	m_fdManager->m_sender = m_sender;
}


void CMingle::mingle_stop()
{
	return;
}


/*
 * set_index_interval: start a timer so that index table 
 *                     will be updated regularly
 */
void CMingle::set_index_interval(int hour)
{
        struct itimerval value;
 
	value.it_interval.tv_sec = hour * 60 * 60;
	value.it_value.tv_sec = hour * 60 * 60;
        value.it_interval.tv_usec = 0;
        value.it_value.tv_usec = 0;

        if (setitimer(ITIMER_REAL, &value, NULL) < 0) {
		mingle_debug1("\nCFindex::set_index_interval(): setitmer failed!");
        }
}

/*
 * update_index(): called by SIGALRM delivered from the system
 */
void CMingle::update_index()
{
	m_index->update_index_table();
}

/*
 * exec: mingle server execution, go into the main loop
 */
int CMingle::exec()
{
	/* start networking */
	if (m_receiver->unix_socket_listen() < 0){
		m_fdManager->set_unix_down();
		printf("Socket error! Will not accept local requests! See mingle.debug for details\n");
	}else{
		m_fdManager->m_unixlfd = m_receiver->m_unixsd;
		m_fdManager->set_unix_up();
	}

	if (m_fdManager->m_netUp){
		if (m_receiver->net_socket_listen() < 0){
			m_fdManager->set_net_down();
			printf("Socket error! Will not accept remote requests! See mingle.debug for details\n");
		}else{
			m_fdManager->m_netlfd = m_receiver->m_netsd;
		}
	}

	if ((!m_fdManager->m_netUp) && (!m_fdManager->m_unixUp)){
		printf("Quit now...\n");
		exit(0);
	}

	/* join the cluster */
	m_reqQueue->join_cluster();

	/* we've finished initing mingle */
	printf("Start accepting requests now...\n");
	fflush(stdout);

	m_fdManager->loop_select();

	mingle_stop();

	return 1;
}

/*
 * get_interfaces: get network interface
 */
int CMingle::get_interfaces(int sockfd, struct ifconf *ifc)
{
	int   len, lastlen;
	char *buf;

	/* Grab a list of interfaces */
	lastlen = 0;
	len = 100 * sizeof(struct ifreq);  /* initial buffer size guess */
	for ( ; ; ) {
    	buf = new char[len];
		ifc->ifc_len = len;
		ifc->ifc_buf = buf;
	    if (ioctl(sockfd, SIOCGIFCONF, ifc) < 0) {
        	if (errno != EINVAL || lastlen != 0) {
        		mingle_debug1("\nCMingle::get_interface():ioctl() error");
				return -1;
		}
	    } else {
	 		if (ifc->ifc_len == lastlen)
				break;    /* success, len has not changed */
	 		lastlen = ifc->ifc_len;
	    }
	    len += 10 * sizeof(struct ifreq);  /* increment */
	    delete buf;
	}
	return 0;
}

/*
 * get_if_addr: get host ip address from an up interface
 */
int CMingle::get_if_addr(char* iface)
{
	int family = AF_INET;
	int sockfd, len, flags;
	char      *ptr;
	struct ifconf    ifc;
	struct ifreq    *ifr = 0, ifrcopy;
	struct sockaddr_in  *sinptr = 0;

	if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
		mingle_debug2("\nCMingle::get_if_addr():cannot create socket", errno);
		return -1;
	}	

	ifc.ifc_buf = 0;
	if (get_interfaces(sockfd, &ifc) == -1){
		if (ifc.ifc_buf)
			delete ifc.ifc_buf;
		close(sockfd);	
		return -1;
	}	

	/* Walk the list of interfaces and locate the first one which is up */
	for (ptr = ifc.ifc_buf; ptr < (ifc.ifc_buf + ifc.ifc_len); ) {
		ifr = (struct ifreq *) ptr;
		
		len = sizeof(struct ifreq);
		ptr += len;  /* for next one in buffer */
		
		if (ifr->ifr_addr.sa_family != family)
			continue;  /* ignore if not desired address family */
		
		ifrcopy = *ifr;
		if (ioctl(sockfd, SIOCGIFFLAGS, &ifrcopy) < 0) {
			mingle_debug1("\nCMingle::get_if_addr():ioctl() error");
		}
		
		flags = ifrcopy.ifr_flags;
		if ((flags & IFF_UP) == 0)
			continue;  /* ignore if interface not up */
		
		if (strncmp("lo", ifr->ifr_name, 2)==0)
			continue;  /* ignore lo interface */
		
		if (iface) {
			if (!strncmp(iface, ifr->ifr_name, IFNAMSIZ) == 0)
				continue; /* Not the interface we're looking for */
		}
		
		switch (ifr->ifr_addr.sa_family) {
		case AF_INET:
			sinptr = (struct sockaddr_in *) &ifr->ifr_addr;
			memcpy(local_ip, &(sinptr->sin_addr), HOSTLEN);
			break;
			
		default:
			mingle_debug1("\nCMingle::get_if_addr(): Protocol family not supported");
		}
		if (local_ip[0] != '\0')
		  break;
	}
	/* release space from ifc and close sockfd */
	if (ifc.ifc_buf)
		delete ifc.ifc_buf;
	close(sockfd);

	/* no interface is up or the specified interface does not exist */
	if (local_ip[0] == '\0'){
		if (iface)
			mingle_debug1("\nCMingle::get_if_addr(): No such interface");
		else
			mingle_debug1("\nCMingle::get_if_addr(): No interface is up now");
		return -1;	
	}	
	
	return 0;
}

/*
 * get_ip_from_host: get host ip address from the host home
 */
int CMingle::get_ip_from_host()
{
	char host[MAXSTRLEN];
	struct hostent *he;
	
	if (gethostname(host,sizeof(host)) == -1){
		mingle_debug2("\nCMingle::get_ip_from_host(): gethostname error", errno);
		return -1;		
	}
	
	if (!(he = gethostbyname(host))){
		mingle_debug2("\nCMingle::get_ip_from_host(): gethostbyname error", h_errno);
		return -1;		
	}
	
	memcpy(local_ip, he->h_addr_list[0], HOSTLEN);
	
	return 0;
}

/* 
 * init_net: get interface and local ip address 
 */
int CMingle::init_net(char *iface)
{
	if (iface) {
		if (inet_aton(iface, (struct in_addr*)local_ip)) {
			return 0;
		}
	}
	
	if ((get_if_addr(iface)) == -1) {
	    if (get_ip_from_host() == -1)
	    	mingle_debug1("\nCMingle::init_net(): Couldn't get host ip");
	    else{
	
	    	char buf[MAXSTRLEN];
	    	sprintf(buf, "\nCMingle::init_net(): Local ip: %i.%i.%i.%i. The network is not up",
	    			local_ip[0], local_ip[1], local_ip[2], local_ip[3]);	
	    	mingle_debug1(buf);		
	    }			
	    return -1;			
	}
	
	return 0;	
}



