/* word.c */

#include "io.h"
#include "word.h"
#include "glo.h"
#include "types.h"   /* K_FIRST, etc */
#include <stdio.h>   /* FILE * */
#include <string.h>  /* strcmp, strcpy */
#include <stdlib.h>  /* malloc */
#include <ctype.h>
#include <time.h>

/* 
   Words are
   - english term (maybe several words)
   - cathegory in parantheses (v, n, adj, adv, ...)
   - a dash
   - alternate definitions separated by semicolons
   - romanian translation in brackets
   */

/* keep this structure and enum Cathegory synchronized */
PRIVATE char * parse_category[] = { 
  /* strings corresponding to cathegories */
  "none", "n", "v", "adj", "adv", "prep", "conj", "interj", "pron"
};

/********* word manipulation functions **********/

PUBLIC Word * 
newword(void)
     /* allocate a new word */
{
  Word * p;
  p = (Word *) malloc (sizeof(Word));
  if (p == NULL) return p;

  p->category = NONE;
  p->word = p->definition = p->translation = NULL;
  return p;
}

PUBLIC int 
cmpword(Word * e1, Word * e2)
     /* compares two entry pointers according to their word,
        and next to their category */
{
  int c;
  c = strcmp(e1->word, e2->word); 
  if ( c != 0 ) return c;
  if (e1->category == NONE || e2->category == NONE) return 0;
       /* ERROR means dummy category - matches anything */
  if (e1->category < e2->category) return -1;
  else if (e1->category == e2->category) return 0;
  else return 1;
}

PRIVATE int 
readcatheg(FILE * fd, char stop)
     /* read one category name and parse it. It ends at character
	`stop'; return category code or NONE on error */
{
  char * text;
  int i;

  text = readto(fd, stop);
  if (text == NULL) return NONE;
  for (i = 0; i < SIZE(parse_category); i++) 
    if (! strcmp(parse_category[i], text)) {
      free(text);
      return i;
    }
  /* no category has fit */
  return NONE;
}

PUBLIC int 
readword(FILE * fd, Word ** w)
     /* read one entry from the file; if done allocate one struct and
	all arrays and put its addres in w.  Return error code */

{
  char * text;

  text = readto(fd, '(');
  if (text == NULL) {
    *w = NULL;
    skipto(fd, '\n');
    return NOWORD;  /* error or no text */
  }
  *w = newword();
  if (*w == NULL) {
    skipto(fd, '\n');
    return OUTMEM;
  }
  (*w)->word = text;
  (*w)->category = readcatheg(fd, ')');  /* read up to ')' */
  if ((*w)->category == NONE) {
    skipto(fd, '\n');
    return NOCATHEG;
  }
  skipto(fd, '-');  /* skip to dash */
  text = readto(fd, '[');       /* error here gives void definition */
  (*w)->definition = text;
  text = readto(fd, ']');       /* error here gives void translation */
  (*w)->translation = text;
  skipto(fd, '\n');
  if (unknown(*w)) return UNKNOWN;
  else if (incomplete(*w)) return INCOMPLETE;
  else return OK;
}

PUBLIC Word *  
printword(FILE * f, Word * crt)
     /* print it, return it */
{
  if (crt == NULL || f == NULL) return crt;
  fprintf(f, "%s (%s) -", crt->word, parse_category[crt->category]);
  if (crt->definition) fprintf(f, " %s", crt->definition);
  if (crt->translation) fprintf(f, " [%s]", crt->translation);
  fprintf(f, "\n");
  return crt;
}

PUBLIC bool 
incomplete(Word * e)
     /* YES if entry is incomplete (no transl. or def.) */
{
  return (e->translation == NULL || e->definition == NULL);
}

PUBLIC bool 
unknown(Word * e)
     /* YES if entry is unknown (no transl. AND def.) */
{
  return ((e->translation == NULL) && (e->definition == NULL));
}

PUBLIC Word *
killword(Word * e)
     /* release it; return NULL */
{
  if (e == NULL) return NULL;
  if (e->translation != NULL) free(e->translation);
  if (e->definition != NULL) free(e->definition);
  if (e->word != NULL) free (e->word);
  free(e);
  return NULL;
}

PRIVATE char *
twotoone(char * s1, char * s2)
     /* make them one string (allocated); 
	if both non-void insert "; " between */
{
  char * tmp, * tmp1;
  
  if (s1 == NULL) return catenate("", s2);  /* to allocate it */
  if (s2 == NULL) return catenate(s1, "");
  /* both are non-null.  Insert "; " */

  tmp = catenate(s1, "; ");
  tmp1 = catenate(tmp, s2);
  free(tmp);
  return tmp1;
}

PRIVATE int 
mystrcmp(char * s1, char * s2)
     /* compare two strings, which may be NULL */
{
  if (s1 == NULL) {
    if (s2 == NULL) return 0;
    else return -1;
  }
  else {
    if (s2 == NULL) return 1;
    else return strcmp(s1, s2);
  }
}

PRIVATE struct menu blend_menu[] ={
  0, "Keep first",
  1, "Keep second",
  2, "Concatenate",
  3, "Edit"
};

PUBLIC Word *
blendwords(Word * w1, Word * w2)
     /* mix the two definitions and translations; 
	return a new one; free necessary space */
{
  int op;
  char * s;
  extern bool global_var_DIRTY;	/* stupid hack to say that blend happened*/

  global_var_DIRTY = YES;
  if ((mystrcmp(w1->definition, w2->definition) == 0)
      && (mystrcmp(w1->translation, w2->translation) == 0))
    /* the two words are identical! */
    return secondword(w1, w2);
    
  fprintf(stdout, "Duplicate definitions:\n");
  printword(stdout, w1);
  printword(stdout, w2);
  do 
    op = menu(blend_menu, SIZE(blend_menu));
  while (op < 0 || op > 3);
  switch(op) {
    /* see the case label meanings a bit earlier */
  case 0:
    killword(w2);
    return printword(stdout, w1);
  case 1:
    killword(w1);
    return printword(stdout, w2);
  case 2:
    s = twotoone(getdefinition(w1), getdefinition(w2));
    if (s != NULL) 
      changedefinition(w1, s);
    s = twotoone(gettranslation(w1), gettranslation(w2));
    if (s != NULL) 
      changetranslation(w1, s);
    killword(w2);          /* done with it */
    return printword(stdout, w1);
  case 3:                  /* edit by hand */
    killword(w1);
    printf("%s (%s):\n", getword(w2), parse_category[w2->category]);
    printf("Give definition:");
    changedefinition(w2, readto(stdin, '\n'));
    printf("Give translation:");
    changetranslation(w2, readto(stdin, '\n'));
    return printword(stdout, w2);
  }
  return NULL;
}

PUBLIC Word *
secondword(Word * w1, Word * w2)
     /* return the second word */
{
  killword(w1);
  return w2;
}

PUBLIC bool
illegalword(Word * w)
     /* YES if the word is illegal */
{
  if (w == NULL) return YES;
  if (w->word == NULL) return YES;
  return (! isalpha(w->word[0]));
}

PUBLIC Word * 
copyword(Word * w)
     /* make a copy of the word */
{
  Word * p;

  if (w == NULL) return NULL;
  p = newword();
  if (p == NULL) return NULL;
  if (w->word != NULL) {
    p->word = malloc(strlen(w->word) + 1);
    if (p->word != NULL) strcpy(p->word, w->word);
    else return killword(p);
  }

  p->category = w->category;

  if (w->definition != NULL) {
    p->definition = malloc(strlen(w->definition) + 1);
    if (p->definition != NULL) strcpy(p->definition, w->definition);
    else return killword(p);
  }

  if (w->translation != NULL) {
    p->translation = malloc(strlen(w->translation) + 1);
    if (p->translation != NULL) strcpy(p->translation, w->translation);
    else return killword(p);
  }
  return p;
}

PUBLIC char * 
getword(Word * w)
     /* return the base word of the structure */
{
  return w->word;
}

PUBLIC Word * 
changeword(Word * w, char * s)
     /* change the basic word */
{
  char * nw;

  if (s != NULL) {
    nw = malloc(strlen(s) + 1);
    if (nw == NULL) {
      error("Can't change base words");
      return w;
    }
    strcpy(nw, s);
  }
  else nw = NULL;
  if (w->word != NULL) free(w->word);
  w->word = nw;
  return w;
}

PUBLIC char * 
getcathstring(Word * w)
     /* return the category string */
{
  if (w->category < SIZE(parse_category))
    return parse_category[w->category];
  else return NULL;
}

PUBLIC Word * 
changedefinition(Word * w, char * newdef)
     /* change the definition; copy the string */
{
  char * nd;

  if (newdef != NULL) {
    nd = malloc(strlen(newdef) + 1);
    if (nd == NULL) {
      error("Can't change def");
      return w;
    }
    strcpy(nd, newdef);
  }
  else nd = NULL;
  if (w->definition != NULL) free(w->definition);
  w->definition = nd;
  return w;
}

PUBLIC Word * 
changetranslation(Word * w, char * newtransl)
     /* change the translation */
{
  char * nd;

  if (newtransl != NULL) {
    nd = malloc(strlen(newtransl) + 1);
    if (nd == NULL) {
      error("Can't change translation");
      return w;
    }
    strcpy(nd, newtransl);
  }
  else nd = NULL;
  if (w->translation != NULL) free(w->translation);
  w->translation = nd;
  return w;
}

PUBLIC char * 
getdefinition(Word * w)
     /* get it */
{
  return w->definition;
}

PUBLIC char * 
gettranslation(Word * w)
     /* just get it */
{
  return w->translation;
}
