/*
 * tester.c - GRE verbal enhancer
 * (c) 1995 by Mihai Budiu
 *
 * This program and the accompanying dictionary are
 * subject to the GNU Public License
 * They have no warranty of any kind, although I tried to make them good.
 */

char version[] = "version 1.63, (c) 1995, 1996 by Mihai Budiu\n\
This program is subject to the GNU Public License\n\
It comes with the dictionaries but with no warranty.\n";

#if MAC
#include <console.h>
#endif
#include "glo.h"
#include "io.h"
#include "word.h"
#include "dict.h"
#include <stdio.h>   /* getchar, fprintf */
#include <stdlib.h>  /* free */
#include "getopt.h"
#include <ctype.h>   /* tolower */
#include "params.h"  /* user changeable parameters influencing grev */

PRIVATE bool allowchanges;      /* the dictionary can be changed only 
				   if it is read entirely */

PUBLIC bool global_var_DIRTY;	/* I hate to do this, but there's no
				   better way to solve this problem
				   without closures.  This flag is
				   modified (by the function
				   blendwords) when the dictionary
				   changes */

/********************************************************************/

PRIVATE int
timesknown(Word * w)
     /* how many times this (log-file) word was known.
	This is the definition field.
	*/
{
  char * s;

  if (w == NULL) return error("bad word");
  if ((s = getdefinition(w)) == NULL) return 0;
  return atoi(s);
}

PRIVATE void
updatedate(Word * w, bool known)
     /* replace the 
	- translation by a string denoting today's date 
	- definition by a string showing if the word was known 
	*/
{
  time_t now;
  char buf[100];     /* temporary buffer */
  char * s;          /* the time representation */
  char sg[2];

  time(&now);
  s = ctime(&now);
  s[strlen(s) - 1] = '\0';  /* drop the '\n' */
  sprintf(buf, "%ld = %s", now, s);
  changetranslation(w, buf);
  if (known) 
    sg[0] = '0' + timesknown(w) + 1;
  else sg[0] = '0';  /* not known - forget all known about the word */
  sg[1] = '\0';
  changedefinition(w, sg);
}

PRIVATE bool 
known(Word * w)
     /* YES if the definition is known */
{
  return (timesknown(w) > 0);
}

PRIVATE bool 
older(Word * w, time_t distance)
     /* yes if w older with `distance' or more than current time */
{
  time_t tw;
  char * stw;

  stw = gettranslation(w);
  if (stw == NULL) return YES;
  tw = (time_t)atol(stw);
  return ( (time(NULL) - tw ) > distance);
}

PRIVATE void
maintenance(Dictionary * d, bool * changes, bool allowchg)
     /* maintenance function;
	report if any change */
{
  Word * w, * w1;
  int err;
  bool done;

  done = NO;
  while(! done) {
    fprintf(stdout, "Type a word:");
    err = readword(stdin, &w);
    switch(err) {
    case UNKNOWN:
    case NOCATHEG:
    case NOWORD:
      if (illegalword(w)) done = YES;
      else {
	w1 = findentry(d, w);
	if (w1 != NULL) printword(stdout, w1);
	else fprintf(stderr, "not found!\n");
      }
      killword(w);        /* used just for I/O */
      break;
    case OUTMEM:
      killword(w);
      error("can't do - out of memory");
      break;
    case INCOMPLETE:
    case OK:
      if (allowchg) {
	fprintf(stdout,"adding to dictionary!\n");
	w1 = addentry(d, w);
	if (w1 == NULL) {
	  error("Could not add word");
	  killword(w);
	}
	else *changes = YES;
      }
      else {
	error("Can't modify dictionary.\n");
	killword(w);
      }
    }
  }
}

PRIVATE Word * 
selectword(Dictionary * d, Dictionary * log)
     /* select a word in the d for testing the user;
	it must be 
	- not in log or
	- unknown but in log older than MINDISTANCE
	- known but less than NO_KNOWN times
	- known and older in log that TESTDIFF 
	*/
{
  Word * w, * lw;
  int retry;

  for (retry = 0; retry < 200; retry ++) {
    w = randomentry(d);
    lw = findentry(log, w);
    if (lw == NULL) return w;     /* not in log -> new word */
    if (! known(lw)) 
      if (older(lw, MINDISTANCE)) return w;
    /* here: it is known */
    if (timesknown(lw) >= NO_KNOWN) continue;  
      /* do NOT retest it more than NO_KNOWN */
    if (older(lw, TESTDIFF)) return w;
  }
  return NULL;   /* could not generate a good word; maybe all are known */
}

PRIVATE int 
testword(Dictionary * d, Word * w, Dictionary * log, 
	 bool * known, bool * changed, int howmany)
     /* test the word and write the result in the log; return 1 if
	user gave meaningless answer */
{
  Word * tmp, * logw;
  char * yn;
  
  printf("\n[%d] <CR> to see `%s (%s)'", 
         howmany, getword(w), getcathstring(w));
  yn = readto(stdin, '\n');
  if (yn != NULL) free(yn);
  tmp = copyword(w);          /* w may be modified! */
 again:
  printword(stdout, tmp);
  printf("You knew it? (y/n/?/other) ");
  yn = readto(stdin, '\n');
  if (yn == NULL) return 1;  /* done testing */
  switch( tolower(yn[0])) {
  case 'y': 
    *known = YES;
    break;
  case 'n':
    *known = NO;
    break;
  case '?':
    maintenance(d, changed, allowchanges);
    printf("\nprevious word >> ");
    goto again;
    break;
  default:
    free(yn);
    return 1;
  }
  free(yn);                           /* it's not NULL */
  logw = copyword(findentry(log, tmp)); /* modify the log */
  if (logw == NULL) 
    logw = copyword(tmp);             /* use this one as basis */
  updatedate(logw, *known);           /* update it */
  addentry(log, logw);                /* back with it */
  killword(tmp);
  return 0;
}

PRIVATE void
saveagain(Dictionary * d, char * outputfile)
     /* try to save a dictionary in a file.  
	If cannot, ask for another one */
{
  FILE * f;
  bool free_it = NO;

  while (1) {
    if (savedictionary(d, outputfile)==OK) {       /* save result */
      if (free_it) free(outputfile);
      return;
    }
    else {
      while (1) {
	printf("Indicate a new file to save in (<CR> to abort save):");
	if (free_it) free(outputfile);
	outputfile = readto(stdin, '\n');
	free_it = YES;		/* this was dynamically allocated */
	if (outputfile == NULL) {
	  error("Modified dictionary unsaved!"); 
	  return;
	}
	else if ((f = fopen(outputfile, "r")) == NULL) 
	  break;		/* try to save again */
	else {		
	  error("This file exists!");
	  fclose(f);		/* ask a new name */
	}
      }
    }
  }
}

PRIVATE void 
iask(Dictionary * d, char * logfile, bool * changed)
     /* testing the user */
{
  Dictionary * userlog;
  Word * w, *f, *l;
  int err;
  long selected, known;
  bool yn;

  f = newword();
  l = newword();
  if (f == NULL || l == NULL) {
    error("Can't initialize");
    return;
  }
  f = changeword(f, "aa");       /* the log is always read entirely! */
  l = changeword(l, "zz");
  selected = known = 0l;      /* trite statistics for this run */
  userlog = newdictionary((Compare) cmpword, (Kill) killword, (Read) readword,
			  (Write) printword, (Itis) secondword);
  err = readdictionary(userlog, logfile, f, l);
  if (err != OK) return;      /* cannot read user log */

  while (1) {
    w = selectword(d, userlog);
    if (w == NULL) {
      error("Cannot select any word now to test you.  Retry a bit later!");
      break;
    }
    if (testword(d, w, userlog, &yn, changed, selected+1)) break;
    selected++;
    known += (yn == YES);
  }
  if (selected != 0l) 
    printf("From %ld words you knew %ld, that is %2.2f%%\n",
	   selected, known, (double)known*100/(double)selected);
  saveagain(userlog, logfile);
  destroydictionary(userlog);
}

PRIVATE int 
tester(char * inputfile, char * logfile, char * firstword, char * lastword)
     /* main function of the program; returns program exit status */
{
  char *outputfile;
  FILE * f;
  bool changes;
  Dictionary * d;
  int err;
  Word * first, * last;

  first = newword();
  last = newword();
  if (first == NULL || last == NULL) {
    error("cannot initialize!");
    return 1;
  }
  changeword(first, firstword);
  changeword(last, lastword);

  global_var_DIRTY = NO;	/* will be modified by blendword */
  d = newdictionary((Compare)cmpword, (Kill)killword, (Read)readword, 
		    (Write)printword, (Itis)blendwords);
  err = readdictionary(d, inputfile, first, last);  
				/* read the dictionary */
  if (err != OK) return err;
  changes = global_var_DIRTY;
  if ((! allowchanges) && changes) {    /* the dictionary has been loaded,
					  modifications have been made then, 
					  but it won't be saved! */
    error("Dictionary modified, but I won't be able to write it back!");
    /* continue still; give the user a chance */
  }
  if (dictionarysize(d) == 0l) {
    error("dictionary contains no valid words; aborting");
    return 1;
  }
  if (err != OK) return 1;             /* cannot read dictionary */
  
  if (logfile == NULL) 
    maintenance(d, &changes, allowchanges);
  else 
    iask(d, logfile, &changes);

  if (changes && allowchanges) 
    saveagain(d, inputfile);
  destroydictionary(d);
  return 0;
}

PRIVATE int
usage(char * name)
     /* usage information */
{
  printf("usage: %s [-vh] [-f word] [-l word] input-dictionary [log-file]\n", name);
  return 1;
}

PRIVATE void 
help(void)

{
  printf("\nVocabulary enhancer (designed for the verbal GRE).\n");
  printf("With one argument it maintains a dictionary.\n");
  printf("With two arguments it tests you from that dictionary.\n");
  printf("-f specifies the first word to be loaded from dictionary.\n");
  printf("-l specifies the last word to be loaded.\n");
  printf("-v prints version information and exits.\n");
  printf("-h prints this help and exits.\n");
  printf("You can change the dictionary only when you load it fully.\n");
}

PRIVATE void
warn(void)
{
  fprintf(stderr, 
	  "WARNING: you can't modify the partially loaded dictionary\n");
  fprintf(stderr, 
	  "WARNING: neither can you look-up for words not loaded\n");
}

int 
main(int argc, char *argv[])
     /* parse options, set up and call function */
{
  char * log_file;
  char * dict_file;
  char * f = "aa", * l = "zz";                /* default first & last word */
  int c;

#if MAC
  argc = ccommand(&argv)
#endif

  opterr = 0;                                 /* for getopt */
  allowchanges = YES;                         /* default behaviour */
  while ((c = getopt(argc, argv, "f:l:vh")) != -1) {
    switch (c) {
    case 'v':          /* version */
      printf("%s - %s", argv[0], version);
      return 0;
      break;
    case 'h':
      usage(argv[0]);
      help();
      return 0;
    case 'f':          /* first word is indicated */
      if (allowchanges) 
	warn();
      allowchanges = NO;
      f = optarg;
      break;
    case 'l':          /* last word */
      if (allowchanges) 
	warn();
      allowchanges = NO;
      l = optarg;
      break;
    case '?':          /* unknown option */
      if (isprint (optopt))
	fprintf (stderr, "Unknown option `-%c'.\n", optopt);
      else
	fprintf (stderr, "Unknown option character `\\x%x'.\n", optopt);
      return usage(argv[0]);
    }
  }
  if (optind < argc) 
    dict_file = argv[optind++];
  else {
    error("Too few arguments");
    return usage(argv[0]);
  }
  if (optind < argc)
    log_file = argv[optind++];
  else
    log_file = NULL;
  if (optind < argc) {
    error("Too many arguments");
    return usage(argv[0]);
  }
  return tester(dict_file, log_file, f, l);   /* that's the main function */
}
