/*
 * dict.c - dictionary manipulation
 * (c) 1995 by Mihai Budiu
 */

#include "io.h"
#include "glo.h"
#include "word.h"
#include "dict.h"
#include "btree.h"
#include <stdlib.h>  /* malloc */
#include <stdio.h>   /* printf */

/* works with a dictionary - a binary sort tree */

/****************** functions **************************/

PUBLIC Dictionary * 
newdictionary(Compare c, Kill k, Read r, Write w, Itis i)
     /* build a new empty dictionary - constructor */
{
  Dictionary * d;
  
  d = (Dictionary *) malloc(sizeof(Dictionary));
  if (d == NULL) return d;
  d->root = newtree();
  d->items = 0l;		
  d->compare = c;
  d->kill = k;
  d->read = r;
  d->write = w;
  d->itis = i;
  d->first = "aa";
  d->last = "zz";
  return d;
}

PUBLIC Dictionary * 
destroydictionary(Dictionary * d)
     /* dictionary destructor */
{
  if (d == NULL) return NULL;
  d->root = killtree(d->root, d->kill);
  free(d);
  return NULL;
}

PUBLIC void *
findentry(Dictionary * d, void * incompleteentry)
     /* finds an (incompletely specified) entry in the dictionary */
{
  if (d->compare(incompleteentry, d->first) < 0 ||
      d->compare(incompleteentry, d->last) > 0) {
    printf("Dictionary contains only words between\n");
    d->write(stdout, d->first);
    d->write(stdout, d->last);
    return NULL;
  }
  return find(d->root, incompleteentry, d->compare);
}

PUBLIC void * 
addentry(Dictionary * d, void * entry)
     /* add a new entry to the dictionary;
        return pointer to it or null */
{
  Tree t;

  t = addcell(d->root, entry, d->compare, d->itis);  /* try to add cell */
  if (t == NULL) return NULL;   /* error ! */
  d->root = t;
  d->items++;
  return entry;
}
  
PUBLIC int 
readdictionary(Dictionary * d, char * dictfile, void * first, void * last)
     /* read in the dictionary */
{
  void * crt;
  FILE * file;
  int err = OK;

  if (dictfile == NULL) file = stdin;
  else {
    file = fopen(dictfile, "r");
    fprintf(stdout, "Reading %s...", dictfile);
    fflush(stdout);     /* show it */
  }
  if (file == NULL) 
    return error("can't open dictionary");
  
  d->first = first;
  d->last = last;
  while (err != NOCATHEG && err != NOWORD && err != OUTMEM) {
    
    err = d->read(file, &crt);   /* read one dict entry entry */
    switch(err) {
    case NOCATHEG:
      d->kill(crt);
      error("Wrong cathegory");
      break;
    case NOWORD:
      break;
    case OUTMEM:
      d->kill(crt);
      error("Out of memory");
      break;
    default:
      if ((d->compare(crt, d->first) >= 0) &&
	  (d->compare(crt, d->last) <= 0)) {
	crt = addentry(d, crt);
	if (crt == NULL) error("Could not add word to dictionary");
      }
      else 
	d->kill(crt);        /* not added! */
      break;
    }
  }
  fprintf(stdout, "done.  %ld entries read (search tree depth %ld).\n", 
	  d->items, depth(d->root));
  if (! feof(file)) {
    fprintf(stderr,"File not read completely.  Syntax error after entry\n");
    d->write(stderr, crt);
  }
  if (dictfile != NULL) fclose(file);
  return OK;
}

PUBLIC int
savedictionary(Dictionary * d, char * file)
     /* save the new dictionary in the indicated file */
{
  FILE * f;

  if (file == NULL) f = stdout;
  else {
    printf("Writing %s...", file);
    fflush(stdout);
    f = fopen(file, "w");
    if (f == NULL) {
      return error("Can't write");
    }
  }
  writetree(d->root, f, d->write);
  if (file != NULL) fclose(f);
  printf("done!\n");
  return OK;
}

PUBLIC void * 
randomentry(Dictionary * d)
     /* pick an entry at random */
{
  return randomnodeentry(d->root);
}

PUBLIC long
dictionarysize(Dictionary * d)
     /* how many words inside */
{
  return d->items;
}
